# !usr/bin/env python2
# -*- coding: utf-8 -*-
#
# Licensed under a 3-clause BSD license.
#
# @Author: Brian Cherinka
# @Date:   2017-01-31 13:34:09
# @Last modified by:   Brian Cherinka
# @Last Modified time: 2017-01-31 16:40:28

from __future__ import print_function, division, absolute_import

import os
import pkg_resources


def _version_split(version):
    """
    Split a version string into major, minor, and bugfix numbers.  If any of
    those numbers are missing the default is zero.  Any pre/post release
    modifiers are ignored.
    Examples
    ========
    >>> _version_split('1.2.3')
    (1, 2, 3)
    >>> _version_split('1.2')
    (1, 2, 0)
    >>> _version_split('1.2rc1')
    (1, 2, 0)
    >>> _version_split('1')
    (1, 0, 0)
    >>> _version_split('')
    (0, 0, 0)
    """

    parsed_version = pkg_resources.parse_version(version)

    if hasattr(parsed_version, 'base_version'):
        # New version parsing for setuptools >= 8.0
        if parsed_version.base_version:
            parts = [int(part)
                     for part in parsed_version.base_version.split('.')]
        else:
            parts = []
    else:
        parts = []
        for part in parsed_version:
            if part.startswith('*'):
                # Ignore any .dev, a, b, rc, etc.
                break
            parts.append(int(part))

    if len(parts) < 3:
        parts += [0] * (3 - len(parts))

    # In principle a version could have more parts (like 1.2.3.4) but we only
    # support <major>.<minor>.<micro>
    return tuple(parts[:3])


def _get_version_py_str(packagename, version, release):
    import datetime
    timestamp = datetime.datetime.now()
    major, minor, bugfix = _version_split(version)

    if packagename.lower() == 'sdss-marvin':
        packagetitle = 'Marvin'

    version_template = ("""
# Autogenerated by {packagetitle}'s setup.py on {timestamp!s}
import datetime
from pkg_resources import get_distribution, DistributionNotFound
major = {major}
minor = {minor}
bugfix = {bugfix}
release = {rel}
timestamp = {timestamp!r}
version = '{version}'


def get_version():
    try:
        dist = get_distribution('sdss-marvin')
    except DistributionNotFound:
        return version
    else:
        return dist.version
    """)

    return version_template.format(packagetitle=packagetitle, timestamp=timestamp, version=version,
                                   major=major, minor=minor, bugfix=bugfix, rel=release)


def get_pkg_version_module(packagename, fromlist=None):
    ''' '''
    if not fromlist:
        return __import__(packagename + '.version', fromlist=[''])
    else:
        mod = __import__(packagename + '.version', fromlist=fromlist)
        return tuple(getattr(mod, member) for member in fromlist)


def generate_version_py(packagename, version, release=None):
    ''' generates the version.py used in Marvin '''
    import sys
    import imp
    try:
        version_module = get_pkg_version_module(packagename)

        try:
            last_generated_version = version_module._last_generated_version
        except AttributeError:
            last_generated_version = version_module.version

        current_release = version_module.release
    except ImportError:
        version_module = None
        last_generated_version = None
        current_release = None

    if release is None:
        # Keep whatever the current value is, if it exists
        release = bool(current_release)

    pypackage = packagename.split('-')[1]
    version_py = os.path.join('python/{0}'.format(pypackage), 'version.py')

    if (last_generated_version != version or current_release != release):
        print('Freezing version number to {0}'.format(version_py))
        with open(version_py, 'w') as f:
            # This overwrites the actual version.py
            f.write(_get_version_py_str(packagename, version, release))

        if version_module:
            imp.reload(version_module)
