# Code for: Roost selection by male northern long-eared bats     #
#           (Myotis septentrionalis) in a managed fire-adapted   #
#           forest                                               #
#                                                                #
# Author: Jesse Alston                                           #
# Last revised: 5/10/19                                          #
##################################################################

# Packages
library(car)
library(MuMIn)
library(pROC)
library(MKmisc)
library(plotrix)

# Data
roosts <- read.csv("C:\\Users\\jmals\\Dropbox\\e\\Proj_NPS_JECA_Bats\\Data\\combined\\myse_roost_data_v3.csv") # read data here

# Specify Global Model
M0 <- glm(UsedAvail ~ DBH + Ht + DecayClass + Slope + Eastness + Southness + Snag +
          PercentBark + TreeDens + SnagDens + CC_total + Slope:Eastness + Slope:Southness,
          data = roosts, family = binomial)
summary(M0)

# Calculate VIFs (don't include interactions--they're collinear by definition)
Mvif <- glm(UsedAvail ~ DBH + Ht + DecayClass + Slope + Eastness + Southness + Snag +
              PercentBark + TreeDens + SnagDens + CC_total,
            data = roosts, family = binomial)

vif(Mvif)

# Remove "Snag" because it is less useful than "DecayClass" and repeat
Mvif <- glm(UsedAvail ~ DBH + Ht + DecayClass + Slope + Eastness + Southness + 
              PercentBark + TreeDens + SnagDens + CC_total,
            data = roosts, family = binomial)

vif(Mvif)

# Correct global model
M0 <- glm(UsedAvail ~ DBH + Ht + DecayClass + Slope + Eastness + Southness + 
            PercentBark + TreeDens + SnagDens + CC_total + Slope:Eastness + Slope:Southness,
          data = roosts, family = binomial)
summary(M0)

# Goodness of Fit (le Cessie-van Houwelingen-Copas-Hosmer GOF test)
HLgof.test(fitted(M0), roosts$UsedAvail, 
           X=model.matrix(UsedAvail ~ DBH + Ht + DecayClass + Slope + Eastness + Southness 
                          + PercentBark + TreeDens + SnagDens + CC_total 
                          + Slope:Eastness + Slope:Southness, data=roosts))
# global model fits well

# Model Selection
options(na.action = "na.fail")
m_list <- dredge(M0, rank="AICc", m.max=8)
head(m_list, 20)

M_avg_list <- get.models(m_list, subset=cumsum(weight) <= .95)
M_avg_list

M_avg <- model.avg(M_avg_list)
summary(M_avg)

confint(M_avg, full=TRUE)


# Model Validation with AUC
ROC_model <- roc(roosts$UsedAvail, predict(M_avg, backTransform=TRUE), plot=T)
ROC_model


##################################################################
# Calculate odds ratios for Tables 2 and 3
# Calculate Unscaled Odds Ratios (Coefficients)
exp(0.09481204) # DBH
exp(0.7464572) # Decay Class
exp(0.08161863) # Canopy Cover

# Calculate Scaled Odds Ratios (Coefficients)
(exp(0.09481204)^5) # DBH
(exp(0.08161863)^10) # Canopy Cover

# Calculate Unscaled Odds Ratios (LCL)
exp(0.0381997) # DBH
exp(0.3834759) # Decay Class
exp(0.0437700) # Canopy Cover

# Calculate Scaled Odds Ratios (LCL)
(exp(0.0381997)^5) # DBH
(exp(0.0437700)^10) # Canopy Cover

# Calculate Unscaled Odds Ratios (UCL)
exp(0.15142438) # DBH
exp(1.10943857) # Decay Class
exp(0.11946726) # Canopy Cover

# Calculate Scaled Odds Ratios (UCL)
(exp(0.15142438)^5) # DBH
(exp(0.11946726)^10) # Canopy Cover


#################################################################
# Calculate means and standard errors for Table 4
# Height #
mean(roosts$Ht[roosts$UsedAvail==1])
mean(roosts$Ht[roosts$UsedAvail==0])
std.error(roosts$Ht[roosts$UsedAvail==1])
std.error(roosts$Ht[roosts$UsedAvail==0])

# DBH #
mean(roosts$DBH[roosts$UsedAvail==1])
mean(roosts$DBH[roosts$UsedAvail==0])
std.error(roosts$DBH[roosts$UsedAvail==1])
std.error(roosts$DBH[roosts$UsedAvail==0])

# Decay Class #
mean(roosts$DecayClass[roosts$UsedAvail==1]) + 3
mean(roosts$DecayClass[roosts$UsedAvail==0]) + 3
std.error(roosts$DecayClass[roosts$UsedAvail==1])
std.error(roosts$DecayClass[roosts$UsedAvail==0])

# Bark Remaining #
mean(roosts$PercentBark[roosts$UsedAvail==1])
mean(roosts$PercentBark[roosts$UsedAvail==0])
std.error(roosts$PercentBark[roosts$UsedAvail==1])
std.error(roosts$PercentBark[roosts$UsedAvail==0])

# Snag Density #
mean(roosts$SnagDens[roosts$UsedAvail==1])
mean(roosts$SnagDens[roosts$UsedAvail==0])
std.error(roosts$SnagDens[roosts$UsedAvail==1])
std.error(roosts$SnagDens[roosts$UsedAvail==0])

# Tree Density #
mean(roosts$TreeDens[roosts$UsedAvail==1])
mean(roosts$TreeDens[roosts$UsedAvail==0])
std.error(roosts$TreeDens[roosts$UsedAvail==1])
std.error(roosts$TreeDens[roosts$UsedAvail==0])

# Canopy Cover #
mean(roosts$CC_total[roosts$UsedAvail==1])
mean(roosts$CC_total[roosts$UsedAvail==0])
std.error(roosts$CC_total[roosts$UsedAvail==1])
std.error(roosts$CC_total[roosts$UsedAvail==0])

# Slope #
mean(roosts$Slope[roosts$UsedAvail==1])
mean(roosts$Slope[roosts$UsedAvail==0])
std.error(roosts$Slope[roosts$UsedAvail==1])
std.error(roosts$Slope[roosts$UsedAvail==0])

# Eastness #
mean(roosts$Eastness[roosts$UsedAvail==1])
mean(roosts$Eastness[roosts$UsedAvail==0])
std.error(roosts$Eastness[roosts$UsedAvail==1])
std.error(roosts$Eastness[roosts$UsedAvail==0])

# Southness #
mean(roosts$Southness[roosts$UsedAvail==1])
mean(roosts$Southness[roosts$UsedAvail==0])
std.error(roosts$Southness[roosts$UsedAvail==1])
std.error(roosts$Southness[roosts$UsedAvail==0])