(* --------------------------------------------------------------------------
   Occurrences of the following types in type definitions have been changed
   to use the types indicated in parentheses so that the visitor generation
   code creates calls to the correct method.

        Location.t      (Location_visitors.location_t)
        Longident.t     (Longident_visitors.longident_t)

   All local types (apart from toplevel_phrase and directive_argument) have
   been prefixed with "pt_" as a namespace; this is to avoid clashses in
   generating names for methods visiting the types in the Types and Typedtree
   modules.
   ------------------------------------------------------------------------- *)
open Compiler

open Asttypes

(* Parsetree has no implementation, so we cannot declare a module alias
     module Base = Parsetree     *)

(* The basic datatype is taken from the 4.04.0 compiler. Conditional cppo
   directives are based on the assumption that this is the minimum version that
   the code is compiled against. *)

type pt_constant = Parsetree.constant =
    Pconst_integer of string * char option
  | Pconst_char of char
  | Pconst_string of string * string option
  | Pconst_float of string * char option
and pt_attribute = string loc * pt_payload
and pt_extension = string loc * pt_payload
and pt_attributes = pt_attribute list
and pt_payload = Parsetree.payload =
  | PStr of pt_structure
  | PSig of pt_signature (* : SIG *)
  | PTyp of pt_core_type  (* : T *)
  | PPat of pt_pattern * pt_expression option  (* ? P  or  ? P when E *)
and pt_core_type = Parsetree.core_type =
    {
     ptyp_desc: pt_core_type_desc;
     ptyp_loc: Location_visitors.location_t;
     ptyp_attributes: pt_attributes; (* ... [@id1] [@id2] *)
    }
and pt_core_type_desc = Parsetree.core_type_desc =
  | Ptyp_any
  | Ptyp_var of string
  | Ptyp_arrow of arg_label * pt_core_type * pt_core_type
  | Ptyp_tuple of pt_core_type list
  | Ptyp_constr of Longident_visitors.longident_t loc * pt_core_type list

#if   OCAML_MINOR = 4
  | Ptyp_object of (string * pt_attributes * pt_core_type) list * closed_flag
#elif OCAML_MINOR = 5
  | Ptyp_object of (string loc * pt_attributes * pt_core_type) list * closed_flag
#elif OCAML_MINOR >= 6
  | Ptyp_object of pt_object_field list * closed_flag
#endif

  | Ptyp_class of Longident_visitors.longident_t loc * pt_core_type list
  | Ptyp_alias of pt_core_type * string
  | Ptyp_variant of pt_row_field list * closed_flag * label list option

#if OCAML_MINOR = 4
  | Ptyp_poly of string list * pt_core_type
#else
  | Ptyp_poly of string loc list * pt_core_type
#endif

  | Ptyp_package of pt_package_type
  | Ptyp_extension of pt_extension
and pt_package_type = Longident_visitors.longident_t loc * (Longident_visitors.longident_t loc * pt_core_type) list
and pt_row_field = Parsetree.row_field =

#if OCAML_MINOR < 6
  | Rtag of label * pt_attributes * bool * pt_core_type list
#else
  | Rtag of label loc * pt_attributes * bool * pt_core_type list
#endif

  | Rinherit of pt_core_type

#if OCAML_MINOR >= 6
and pt_object_field = Parsetree.object_field =
  | Otag of label loc * pt_attributes * pt_core_type
  | Oinherit of pt_core_type
#endif

and pt_pattern = Parsetree.pattern =
    {
     ppat_desc: pt_pattern_desc;
     ppat_loc: Location_visitors.location_t;
     ppat_attributes: pt_attributes; (* ... [@id1] [@id2] *)
    }
and pt_pattern_desc = Parsetree.pattern_desc =
  | Ppat_any
  | Ppat_var of string loc
  | Ppat_alias of pt_pattern * string loc
  | Ppat_constant of pt_constant
  | Ppat_interval of pt_constant * pt_constant
  | Ppat_tuple of pt_pattern list
  | Ppat_construct of Longident_visitors.longident_t loc * pt_pattern option
  | Ppat_variant of label * pt_pattern option
  | Ppat_record of (Longident_visitors.longident_t loc * pt_pattern) list * closed_flag
  | Ppat_array of pt_pattern list
  | Ppat_or of pt_pattern * pt_pattern
  | Ppat_constraint of pt_pattern * pt_core_type
  | Ppat_type of Longident_visitors.longident_t loc
  | Ppat_lazy of pt_pattern
  | Ppat_unpack of string loc
  | Ppat_exception of pt_pattern
  | Ppat_extension of pt_extension
  | Ppat_open of Longident_visitors.longident_t loc * pt_pattern
and pt_expression = Parsetree.expression =
    {
     pexp_desc: pt_expression_desc;
     pexp_loc: Location_visitors.location_t;
     pexp_attributes: pt_attributes; (* ... [@id1] [@id2] *)
    }
and pt_expression_desc = Parsetree.expression_desc =
  | Pexp_ident of Longident_visitors.longident_t loc
  | Pexp_constant of pt_constant
  | Pexp_let of rec_flag * pt_value_binding list * pt_expression
  | Pexp_function of pt_case list
  | Pexp_fun of arg_label * pt_expression option * pt_pattern * pt_expression
  | Pexp_apply of pt_expression * (arg_label * pt_expression) list
  | Pexp_match of pt_expression * pt_case list
  | Pexp_try of pt_expression * pt_case list
  | Pexp_tuple of pt_expression list
  | Pexp_construct of Longident_visitors.longident_t loc * pt_expression option
  | Pexp_variant of label * pt_expression option
  | Pexp_record of (Longident_visitors.longident_t loc * pt_expression) list * pt_expression option
  | Pexp_field of pt_expression * Longident_visitors.longident_t loc
  | Pexp_setfield of pt_expression * Longident_visitors.longident_t loc * pt_expression
  | Pexp_array of pt_expression list
  | Pexp_ifthenelse of pt_expression * pt_expression * pt_expression option
  | Pexp_sequence of pt_expression * pt_expression
  | Pexp_while of pt_expression * pt_expression
  | Pexp_for of
      pt_pattern *  pt_expression * pt_expression * direction_flag * pt_expression
  | Pexp_constraint of pt_expression * pt_core_type
  | Pexp_coerce of pt_expression * pt_core_type option * pt_core_type

#if   OCAML_MINOR = 4
  | Pexp_send of pt_expression * string
#elif OCAML_MINOR = 5
  | Pexp_send of pt_expression * string loc
#elif OCAML_MINOR >= 6
  | Pexp_send of pt_expression * label loc
#endif

  | Pexp_new of Longident_visitors.longident_t loc

#if   OCAML_MINOR <= 5
  | Pexp_setinstvar of string loc * pt_expression
  | Pexp_override of (string loc * pt_expression) list
#else
  | Pexp_setinstvar of label loc * pt_expression
  | Pexp_override of (label loc * pt_expression) list
#endif

  | Pexp_letmodule of string loc * pt_module_expr * pt_expression
  | Pexp_letexception of pt_extension_constructor * pt_expression
  | Pexp_assert of pt_expression
  | Pexp_lazy of pt_expression
  | Pexp_poly of pt_expression * pt_core_type option
  | Pexp_object of pt_class_structure

#if OCAML_MINOR = 4
  | Pexp_newtype of string * pt_expression
#else
  | Pexp_newtype of string loc * pt_expression
#endif

  | Pexp_pack of pt_module_expr
  | Pexp_open of override_flag * Longident_visitors.longident_t loc * pt_expression
  | Pexp_extension of pt_extension
  | Pexp_unreachable
and pt_case = Parsetree.case =
    {
     pc_lhs: pt_pattern;
     pc_guard: pt_expression option;
     pc_rhs: pt_expression;
    }
and pt_value_description = Parsetree.value_description =
    {
     pval_name: string loc;
     pval_type: pt_core_type;
     pval_prim: string list;
     pval_attributes: pt_attributes;  (* ... [@@id1] [@@id2] *)
     pval_loc: Location_visitors.location_t;
    }
and pt_type_declaration = Parsetree.type_declaration =
    {
     ptype_name: string loc;
     ptype_params: (pt_core_type * variance) list;
     ptype_cstrs: (pt_core_type * pt_core_type * Location_visitors.location_t) list;
     ptype_kind: pt_type_kind;
     ptype_private: private_flag;   (* = private ... *)
     ptype_manifest: pt_core_type option;  (* = T *)
     ptype_attributes: pt_attributes;   (* ... [@@id1] [@@id2] *)
     ptype_loc: Location_visitors.location_t;
    }
and pt_type_kind = Parsetree.type_kind =
  | Ptype_abstract
  | Ptype_variant of pt_constructor_declaration list
  | Ptype_record of pt_label_declaration list
  | Ptype_open
and pt_label_declaration = Parsetree.label_declaration =
    {
     pld_name: string loc;
     pld_mutable: mutable_flag;
     pld_type: pt_core_type;
     pld_loc: Location_visitors.location_t;
     pld_attributes: pt_attributes; (* l [@id1] [@id2] : T *)
    }
and pt_constructor_declaration = Parsetree.constructor_declaration =
    {
     pcd_name: string loc;
     pcd_args: pt_constructor_arguments;
     pcd_res: pt_core_type option;
     pcd_loc: Location_visitors.location_t;
     pcd_attributes: pt_attributes; (* C [@id1] [@id2] of ... *)
    }
and pt_constructor_arguments = Parsetree.constructor_arguments =
  | Pcstr_tuple of pt_core_type list
  | Pcstr_record of pt_label_declaration list
and pt_type_extension = Parsetree.type_extension =
    {
     ptyext_path: Longident_visitors.longident_t loc;
     ptyext_params: (pt_core_type * variance) list;
     ptyext_constructors: pt_extension_constructor list;
     ptyext_private: private_flag;
     ptyext_attributes: pt_attributes;   (* ... [@@id1] [@@id2] *)
    }
and pt_extension_constructor = Parsetree.extension_constructor =
    {
     pext_name: string loc;
     pext_kind : pt_extension_constructor_kind;
     pext_loc : Location_visitors.location_t;
     pext_attributes: pt_attributes; (* C [@id1] [@id2] of ... *)
    }
and pt_extension_constructor_kind = Parsetree.extension_constructor_kind =
    Pext_decl of pt_constructor_arguments * pt_core_type option
  | Pext_rebind of Longident_visitors.longident_t loc
and pt_class_type = Parsetree.class_type =
    {
     pcty_desc: pt_class_type_desc;
     pcty_loc: Location_visitors.location_t;
     pcty_attributes: pt_attributes; (* ... [@id1] [@id2] *)
    }
and pt_class_type_desc = Parsetree.class_type_desc =
  | Pcty_constr of Longident_visitors.longident_t loc * pt_core_type list
  | Pcty_signature of pt_class_signature
  | Pcty_arrow of arg_label * pt_core_type * pt_class_type
  | Pcty_extension of pt_extension

#if OCAML_MINOR >= 6
  | Pcty_open of override_flag * Longident_visitors.longident_t loc * pt_class_type
          (* let open M in CT *)
#endif

and pt_class_signature = Parsetree.class_signature =
    {
     pcsig_self: pt_core_type;
     pcsig_fields: pt_class_type_field list;
    }
and pt_class_type_field = Parsetree.class_type_field =
    {
     pctf_desc: pt_class_type_field_desc;
     pctf_loc: Location_visitors.location_t;
     pctf_attributes: pt_attributes; (* ... [@@id1] [@@id2] *)
    }
and pt_class_type_field_desc = Parsetree.class_type_field_desc =
  | Pctf_inherit of pt_class_type

#if   OCAML_MINOR = 4
  | Pctf_val of (string * mutable_flag * virtual_flag * pt_core_type)
#elif OCAML_MINOR = 5
  | Pctf_val of (string loc * mutable_flag * virtual_flag * pt_core_type)
#else
  | Pctf_val of (label loc * mutable_flag * virtual_flag * pt_core_type)
#endif

#if   OCAML_MINOR = 4
  | Pctf_method  of (string * private_flag * virtual_flag * pt_core_type)
#elif OCAML_MINOR = 5
  | Pctf_method  of (string loc * private_flag * virtual_flag * pt_core_type)
#else
  | Pctf_method  of (label loc * private_flag * virtual_flag * pt_core_type)
#endif

  | Pctf_constraint  of (pt_core_type * pt_core_type)
  | Pctf_attribute of pt_attribute
  | Pctf_extension of pt_extension
and 'a pt_class_infos = 'a Parsetree.class_infos =
    {
     pci_virt: virtual_flag;
     pci_params: (pt_core_type * variance) list;
     pci_name: string loc;
     pci_expr: 'a;
     pci_loc: Location_visitors.location_t;
     pci_attributes: pt_attributes;  (* ... [@@id1] [@@id2] *)
    }
and pt_class_description = pt_class_type pt_class_infos
and pt_class_type_declaration = pt_class_type pt_class_infos
and pt_class_expr = Parsetree.class_expr =
    {
     pcl_desc: pt_class_expr_desc;
     pcl_loc: Location_visitors.location_t;
     pcl_attributes: pt_attributes; (* ... [@id1] [@id2] *)
    }
and pt_class_expr_desc = Parsetree.class_expr_desc =
  | Pcl_constr of Longident_visitors.longident_t loc * pt_core_type list
  | Pcl_structure of pt_class_structure
  | Pcl_fun of arg_label * pt_expression option * pt_pattern * pt_class_expr
  | Pcl_apply of pt_class_expr * (arg_label * pt_expression) list
  | Pcl_let of rec_flag * pt_value_binding list * pt_class_expr
  | Pcl_constraint of pt_class_expr * pt_class_type
  | Pcl_extension of pt_extension

#if OCAML_MINOR >= 6
  | Pcl_open of override_flag * Longident_visitors.longident_t loc * pt_class_expr
    (* let open M in CE *)
#endif

and pt_class_structure = Parsetree.class_structure =
    {
     pcstr_self: pt_pattern;
     pcstr_fields: pt_class_field list;
    }
and pt_class_field = Parsetree.class_field =
    {
     pcf_desc: pt_class_field_desc;
     pcf_loc: Location_visitors.location_t;
     pcf_attributes: pt_attributes; (* ... [@@id1] [@@id2] *)
    }
and pt_class_field_desc = Parsetree.class_field_desc =

#if OCAML_MINOR = 4
  | Pcf_inherit of override_flag * pt_class_expr * string option
#else
  | Pcf_inherit of override_flag * pt_class_expr * string loc option
#endif

#if OCAML_MINOR < 6
  | Pcf_val of (string loc * mutable_flag * pt_class_field_kind)
  | Pcf_method of (string loc * private_flag * pt_class_field_kind)
#else
  | Pcf_val of (label loc * mutable_flag * pt_class_field_kind)
  | Pcf_method of (label loc * private_flag * pt_class_field_kind)
#endif

  | Pcf_constraint of (pt_core_type * pt_core_type)
  | Pcf_initializer of pt_expression
  | Pcf_attribute of pt_attribute
  | Pcf_extension of pt_extension
and pt_class_field_kind = Parsetree.class_field_kind =
  | Cfk_virtual of pt_core_type
  | Cfk_concrete of override_flag * pt_expression
and pt_class_declaration = pt_class_expr pt_class_infos
and pt_module_type = Parsetree.module_type =
    {
     pmty_desc: pt_module_type_desc;
     pmty_loc: Location_visitors.location_t;
     pmty_attributes: pt_attributes; (* ... [@id1] [@id2] *)
    }
and pt_module_type_desc = Parsetree.module_type_desc =
  | Pmty_ident of Longident_visitors.longident_t loc
  | Pmty_signature of pt_signature
  | Pmty_functor of string loc * pt_module_type option * pt_module_type
  | Pmty_with of pt_module_type * pt_with_constraint list
  | Pmty_typeof of pt_module_expr
  | Pmty_extension of pt_extension
  | Pmty_alias of Longident_visitors.longident_t loc
and pt_signature = pt_signature_item list
and pt_signature_item = Parsetree.signature_item =
    {
     psig_desc: pt_signature_item_desc;
     psig_loc: Location_visitors.location_t;
    }
and pt_signature_item_desc = Parsetree.signature_item_desc =
  | Psig_value of pt_value_description
  | Psig_type of rec_flag * pt_type_declaration list
  | Psig_typext of pt_type_extension
  | Psig_exception of pt_extension_constructor
  | Psig_module of pt_module_declaration
  | Psig_recmodule of pt_module_declaration list
  | Psig_modtype of pt_module_type_declaration
  | Psig_open of pt_open_description
  | Psig_include of pt_include_description
  | Psig_class of pt_class_description list
  | Psig_class_type of pt_class_type_declaration list
  | Psig_attribute of pt_attribute
  | Psig_extension of pt_extension * pt_attributes
and pt_module_declaration = Parsetree.module_declaration =
    {
     pmd_name: string loc;
     pmd_type: pt_module_type;
     pmd_attributes: pt_attributes; (* ... [@@id1] [@@id2] *)
     pmd_loc: Location_visitors.location_t;
    }
and pt_module_type_declaration = Parsetree.module_type_declaration =
    {
     pmtd_name: string loc;
     pmtd_type: pt_module_type option;
     pmtd_attributes: pt_attributes; (* ... [@@id1] [@@id2] *)
     pmtd_loc: Location_visitors.location_t;
    }
and pt_open_description = Parsetree.open_description =
    {
     popen_lid: Longident_visitors.longident_t loc;
     popen_override: override_flag;
     popen_loc: Location_visitors.location_t;
     popen_attributes: pt_attributes;
    }
and 'a pt_include_infos = 'a Parsetree.include_infos =
    {
     pincl_mod: 'a;
     pincl_loc: Location_visitors.location_t;
     pincl_attributes: pt_attributes;
    }
and pt_include_description = pt_module_type pt_include_infos
and pt_include_declaration = pt_module_expr pt_include_infos
and pt_with_constraint = Parsetree.with_constraint =
  | Pwith_type of Longident_visitors.longident_t loc * pt_type_declaration
  | Pwith_module of Longident_visitors.longident_t loc * Longident_visitors.longident_t loc

#if OCAML_MINOR < 6
  | Pwith_typesubst of pt_type_declaration
  | Pwith_modsubst of string loc * Longident_visitors.longident_t loc
#else
  | Pwith_typesubst of Longident_visitors.longident_t loc * pt_type_declaration
  | Pwith_modsubst of Longident_visitors.longident_t loc * Longident_visitors.longident_t loc
#endif

and pt_module_expr = Parsetree.module_expr =
    {
     pmod_desc: pt_module_expr_desc;
     pmod_loc: Location_visitors.location_t;
     pmod_attributes: pt_attributes; (* ... [@id1] [@id2] *)
    }
and pt_module_expr_desc = Parsetree.module_expr_desc =
  | Pmod_ident of Longident_visitors.longident_t loc
  | Pmod_structure of pt_structure
  | Pmod_functor of string loc * pt_module_type option * pt_module_expr
  | Pmod_apply of pt_module_expr * pt_module_expr
  | Pmod_constraint of pt_module_expr * pt_module_type
  | Pmod_unpack of pt_expression
  | Pmod_extension of pt_extension
and pt_structure = pt_structure_item list
and pt_structure_item = Parsetree.structure_item =
    {
     pstr_desc: pt_structure_item_desc;
     pstr_loc: Location_visitors.location_t;
    }
and pt_structure_item_desc = Parsetree.structure_item_desc =
  | Pstr_eval of pt_expression * pt_attributes
  | Pstr_value of rec_flag * pt_value_binding list
  | Pstr_primitive of pt_value_description
  | Pstr_type of rec_flag * pt_type_declaration list
  | Pstr_typext of pt_type_extension
  | Pstr_exception of pt_extension_constructor
  | Pstr_module of pt_module_binding
  | Pstr_recmodule of pt_module_binding list
  | Pstr_modtype of pt_module_type_declaration
  | Pstr_open of pt_open_description
  | Pstr_class of pt_class_declaration list
  | Pstr_class_type of pt_class_type_declaration list
  | Pstr_include of pt_include_declaration
  | Pstr_attribute of pt_attribute
  | Pstr_extension of pt_extension * pt_attributes
and pt_value_binding = Parsetree.value_binding =
  {
    pvb_pat: pt_pattern;
    pvb_expr: pt_expression;
    pvb_attributes: pt_attributes;
    pvb_loc: Location_visitors.location_t;
  }
and pt_module_binding = Parsetree.module_binding =
    {
     pmb_name: string loc;
     pmb_expr: pt_module_expr;
     pmb_attributes: pt_attributes;
     pmb_loc: Location_visitors.location_t;
    }

[@@deriving
    visitors {
        variety = "iter"
      ; polymorphic = true; monomorphic = ["'env"]
      ; ancestors = [
            "Asttype_visitors.iter"
          ; "Longident_visitors.iter"
        ]
    }
  , visitors {
        variety = "map"
      ; polymorphic = true; monomorphic = ["'env"]
      ; ancestors = [
            "Asttype_visitors.map"
          ; "Longident_visitors.map"
        ]
    }
  , visitors {
        variety = "reduce"
      ; polymorphic = true; monomorphic = ["'env"]
      ; ancestors = [
            "Asttype_visitors.reduce"
          ; "Longident_visitors.reduce"
        ]
    }

  , visitors {
        variety = "iter2"
      ; polymorphic = true; monomorphic = ["'env"]
      ; concrete = true
      ; ancestors = [
            "Asttype_visitors.iter2"
          ; "Longident_visitors.iter2"
        ]
    }
]
