# ROTOR: A Reliable OCaml Tool for OCaml Refactoring - Trustworthy Refactoring for OCaml

Authors: Reuben N. S. Rowe, Hugo Férée, Simon Thompson, and Scott Owens

## ARTIFACT EVALUATION README

This README is written in valid Markdown.

### Getting Started Guide

The archive constitutes the artifact associated with the following paper:

- Characterising Renaming within OCaml's Module System: Theory and Implementation.
  Reuben N. S. Rowe, Hugo Férée, Simon Thompson, and Scott Owens.
  <https://doi.org/10.1145/3314221.3314600>

It contains the following items.

- `README.md`:
    This README file.
- `refactorer`:
    A directory containing the source code of the ROTOR tool. This is the snapshot labelled with the `PLDI'19-artifact` tag in the repository found at <https://gitlab.com/trustworthy-refactoring/refactorer>.
- `Dockerfile`:
    A configuration file for building a Docker image of the executable form of the artifact.
- `proofs`:
    A directory containing the Coq formalisation of the theoretical framework in the paper.
- `scripts`:
    A directory containing scripts that can be invoked inside the artifact's Docker container in order to run the testbeds.
- `testbed-configs`:
    A directory containing configuration scripts needed by the executable form of the artifact to run the testbeds.
- `jane-street-test-results`:
    A directory containing the expected results of running the full Jane Street testbed.
- `ocaml-compiler-test-results`:
    A directory containing the expected results of running the full OCaml compiler testbed.

To run the artifact you will need to install the Docker container manager, either via your system's package manager, or from <https://www.docker.com/>.

Then build the docker image as specified in the Dockerfile, as follows:

    > docker build . -t ocaml-rotor

This process may take a while, as it has to download and install a number of packages.
Once it has finished, you should see following acknowledgement that the container has built successfully.

    Successfully tagged ocaml-rotor:latest

You can then run the image with the following command.

    > docker run -ti ocaml-rotor

Note that the container should run with at least 4GB of memory, otherwise you may observe processes within the container being killed by Docker. If you are running Docker on a Mac, you may have to increase the memory available to the application using the Desktop application settings, as described here: <https://docs.docker.com/docker-for-mac/#advanced>.

The home directory of the image should contain the following directories:

- `refactorer`:
    the source code of the ROTOR tool described in the paper.
- `coq-formalisation`:
    the Coq formalisation of the theoretical framework described in the paper.
- `testbeds`:
    the testbeds mentioned in the paper, i.e. a snapshot of the Jane Street standard library overlay, and the OCaml compiler (version 4.04.0).
- `scripts`:
    contains shell scripts for running the Jane Street and OCaml compiler testbeds (see [Step-by-Step Instructions], below)

#### Coq Formalisation

1. Contents

   The `coq-formalisation` directory contains the following Coq (`*.v`) files, formalising the renaming semantics and theory in the paper:

   - `Modules.v`:
       Definition of the module system and basic syntactic properties (Section 2).
   - `Semantics.v`:
       Semantic rules and their properties (Section 3), including
       - determinism as well as a derivation function
       - monotonicity properties
       - weakening lemmas
   - `Default_renaming.v`:
       Defines and proves the correctness of the renaming from Proposition 5
   - `Renaming_operations.v`:
       Definitions and basic properties of renamings (Section 2.1)
   - `Characterising_renaming.v`:
       Proves properties from Section 4:
       - Lower bound for footprints
       - Factorisation
   - `Examples.v`:
       - Tools to write examples
       - Running example from the paper

2. Compiling

   Within the `coq-formalisation` directory, run (it takes about two minutes):

       > make

   The compilation of the example file will output a warning related to extraction, which is harmless.

   The build artifacts can be removed by running:

       > make clean

3. Documentation

   The following command generates HTML documentation in the `coq-formalisation/html` folder:

       > make gallinahtml

   The simplest way to view the documentation is to open `coq-formalisation/index.html`.

   This is perhaps best done by copying this file and the `html` subdirectory from the docker container onto your own filesystem, where you can open the HTML files in a browser. You can do this by running the following commands **on your local system** (n.b. not within the docker container itself):

       > docker cp [container-id]:/home/rotor/coq-formalisation/index.html .
       > docker cp [container-id]:/home/rotor/coq-formalisation/html .

   where `[container-id]` is the identifier of the docker container. You can obtain this identifier by running the following command, which lists all of the containers present on your system:

       > docker ps -a

   The `index.html` contains a dependency graph for the proof files, where each node is a link to the correponding documentation page. Note that many technical lemmas and definitions are not displayed in the documentation, leaving almost only those stated in the article and the appendix. The table of contents can also be found in `html/toc.html`.

4. Extraction

   The compilation of `Example.v` will produce several `*.ml` and `*.mli` resulting from the extraction of Coq functions and proofs as OCaml programs.

   They may be moved to the examples directory with:

       > make examples

   This will also compile a small example (`main.ml`), producing an executable `main.native` in the `examples` directory. It can be run by calling it directly:

       > ./examples/main.native

   It provides an example of verified renaming (short of trusting Coq's extraction module), using the article's running example (Figure 2).

   It will:
     - display the original program and the renamed program;
     - print the list of renamed locations and a description of the extension kernel as a list of pairs of locations;
     - derive the semantics for both programs and check if they are semantically equivalent.

#### ROTOR

1. Compiling

   ROTOR is already compiled and installed in the image via OPAM. A usage string for invoking ROTOR can be displayed using the `-help` flag:

       > rotor -help

   ROTOR's source code can be (re-)compiled using the make tool. Within the
`refactorer` directory, run:

       > make clean && make

2. Quick Setup Tests

   There is a small suite of quick, stand-alone tests that can be run to check the basic setup. From within the `refactorer` directory, run:

       > make tests.internal.all

   The patch and log file produced by ROTOR for each of these tests can be found in the numbered sub-directories in the following location:

       test/internal/rename

   There is also a small selection of tests that can be run on the Jane Street testbed, again using make:

       > make tests.jane-street.all

   These tests take a little longer (between 10-30 seconds per test). The patch and log file produced by ROTOR for each test can be found in the following sub-directory.

       test/jane-street/rename

   The functions renamed by these tests can be found in the file

       test/jane-street/rename/tests

   For example, the first line

       base.Array0:fold foo

   describes a test to rename the `fold` function in the `Array0` module of the `base` library to `foo`.

3. Display of Detailed Output when Executing ROTOR

   When computing a renaming, ROTOR can output detailed information about what it is doing, including listing all of the dependent functions it is renaming, and displaying progress indicators. To demonstrate this on the Jane Street testbed, carry out the following steps.

   First, we load some flags that ROTOR needs into some shell variables. These flags will tell ROTOR where to find the source files, and where to find representations of the ASTS for these source files produced by the compiler.

       > cd ~/testbeds/.configs/jane-street
       > source load_params.sh

   This loads the flags into the shell variables `INPUT_DIRS` and `INCLUDE_PARAMS`. Their contents can be viewed as follows:

       > echo $INPUT_DIRS
       > echo $INCLUDE_PARAMS

   Now, change into the Jane Street testbed directory:

       > cd ~/testbeds/jane-street

   Next, we call ROTOR, asking it to rename the fold function in the Array0 module of the base library to "foo", and displaying detailed output using the `-show-progress flag`.

       > rotor $INPUT_DIRS $INCLUDE_PARAMS -show-progress -r rename base.Array0:fold foo

   ROTOR should indicate that it is doing the following things:

     - computing the dependencies of each file on the others in the codebase;
     - renaming a number of different of different functions;
     - its progress processing each such different function, indicated by a line of dots.

   ROTOR outputs the computed diff patch to the standard output.

### Step-by-Step Instructions

#### Running the Test Suites

Both the Jane Street test suite and the OCaml compiler test suite contain around 3000 test cases each, and each take up to 48 hours to run our OpenStack instance.

We have included the output of running the full testbeds in the following files in the artifact archive:

- `jane-street-test-results.tar`
- `ocaml-compiler-test-results.tar`

Each of these archives contains the following:

- `output.txt`:
    A file giving the result of each test case (`REFACTORING_FAILED`, `BUILD_FAILED`, or `SUCCEEDED`)

- `[test_case].log`,
  `[test_case].deps`,
  `[test_case].error` or `[test_case].patch`,
  `[test_case].failed` if ROTOR produced a `[test_case].patch` file but recompilation failed:

  These are the outputs of ROTOR for each test case.

  - A log file containing very detailed information about the execution of ROTOR on this test case.
  - A file containing all the renaming dependencies found by ROTOR for this test case; for each dependency, the dependencies that it generates are also given, along with a reason (e.g. `ModuleInclude`, `ModuleIsAliased`) and the point in the codebase that caused the dependency to be generated from the initial one.
  - A `.patch` file if ROTOR succeeded in computing a renaming, and otherwise a `.error` file containing the error output.
  - A `.failed` file if the testbed failed to recompile successfully after applying the patch generated by ROTOR; this contains the compiler's build log.

- `stats/error-list.csv`:
    A file containing a list of the test cases for which ROTOR produced an error, along with the error produced.

    These are all cases which require renaming a function outside of the codebase (e.g. in the standard library, or in an automatically generated file).

- `stats/failure-cases-stats.csv`,
  `stats/success-cases-stats.csv`:

    These files contain a list of the test cases that failed and succeeded to recompile, respectively.

    For each test case, the number of files modified, the number of hunks in the patch, and the number of renaming dependencies are given.

- `stats/failure-cases.ods`,
  `stats/success-cases.ods`:

    These are spreadsheets that we have created by hand by importing the `.csv` files mentioned above, and then aggregating the metrics to give the mean, median and mode values for each one.

We have provided scripts that will allow you to run both the full test suites for each testbed, and also a pared down test suite for each testbed containing 10 representative cases.

To run the test suites, you must first create a directory in which to store the resulting files produced by ROTOR and the testbed framework, e.g.

    > mkdir /home/rotor/renaming-test-results

You can then run one of the test suite initialiser scripts:

    > /home/rotor/scripts/jane-street-rename-short.sh /home/rotor/renaming-test-results
    > /home/rotor/scripts/jane-street-rename-full.sh /home/rotor/renaming-test-results
    > /home/rotor/scripts/ocaml-rename-short.sh /home/rotor/renaming-test-results
    > /home/rotor/scripts/ocaml-rename-full.sh /home/rotor/renaming-test-results

The short test suites should take between 5 and 10 minutes to run to completion.

You may more easily compare the results of running the test suites with our provided results by copying them from the docker container to your own filesystem, e.g.:

    > docker cp [container-id]:/home/rotor/renaming-test-results <dest-on-local-filesystem>

#### Running ROTOR on Your Own Examples

This section describes how you can run ROTOR on your own simple (multi-file) OCaml programs. These instructions apply when all the modules in your program are contained within a single directory and do not reference any external functions apart from those found in OCaml's standard library `Pervasives` module.

The `vi` editor is included in the Docker image, and you can use this to create your OCaml source files within the Docker container itself. Alternatively, you may create them on your local filesystem and copy them to a directory `<dir-name>` in the Docker container using:

    > docker cp <local-path-to-src-dir>/ [container-name]:/home/rotor/<dir-name>/

Once the OCaml source files are created, they must be compiled and corresponding `cmt` and `cmti` files created before ROTOR can be run over them. This can be done **in the Docker container** as follows, from within the directory containing the source files:

    > ocamlc -c -bin-annot *.ml *.mli

You can then invoke ROTOR by running:

    > rotor -d . -r rename <identifier> <new-name>

or

    > rotor -d . -show-progress -r rename <identifier> <new-name>

to output progress information while ROTOR is running. The `-d` flag tells ROTOR to look for source files in the current directory.

As mentioned in the paper, ROTOR uses an extended syntax for OCaml identifiers. OCaml programs have a hierarchical structure, in which both modules and module types can be nested within one another. OCaml uses 'dot notation' for identifiers, in which the infix operator dot (`.`) indicates this hierarchical nesting. ROTOR generalises OCaml's identifier notation in two ways. Firstly, instead of treating the dot as an infix operator, it uses it as a prefix operator on names to indicate an element of a particular sort and introduces new prefix operators to express other sorts (e.g. module, module type, value). Secondly, the hierarchical structure is now represented by the sequencing of prefixed names. ROTOR currently uses the operators `.`, `#`, `%`, `*`, and `:` to indicate structures, functors, structure types (i.e. signatures), functor types, and values, respectively. ROTOR also uses an indexer element of the form `[i]`, to stand for the `i`th parameter of a functor or functor type.

Specifically, ROTOR uses the following syntax for identifiers where the nonterminal `<name>` denotes a standard OCaml (short) identifier, and `<number>` denotes a positive integer literal.

    <signifier>  ::= '.' | '#' | '%' | '*' | ':'
    <id_link>    ::= <signifier> <name> | '[' <index> ']'
    <identifier> ::= <id_link> | <id_link> <identifier>

So, for example, to specify a function `foo` nested within a number of (sub)modules, you could use the identifier `.A.B.Bar.Baz:foo`.

To give a more complex example, `.Set%S:add` would refer to the `add` value declaration within the `S` module type within the `Set` module.

Similarly, `.Set#Make[1]:compare` refers to the declaration of the `compare` value in the first parameter of the `Make` functor within the `Set` module.

Note that when specifying the new name in the invocation of ROTOR

    > rotor -d . -r rename <identifier> <new-name>

you should give simply a short identifier (e.g. `foo`), i.e. you do not need to specify a full path; indeed doing so will cause ROTOR to raise an error.

## Claims (Not) Supported by the Artifact

The following are a list of claims in the paper that are supported by the artifact.

- For the abstract renaming theory:

  - Proposition 1 (Section 2) in `coq-formalisation/Renaming_operations.v`, on line 656.
  - Proposition 2 (Section 3) in `coq-formalisation/Semantics.v` on line 4220.
  - Proposition 3 (Section 3) in `coq-formalisation/Semantics.v` on line 5347.
  - Proposition 4, parts (i)-(iii) (Section 4) in `coq-formalisation/Renaming_operations.v` on lines 131, 172, and 208 (for renamings in general) and `coq-formalisation/Semantics.v` on lines 5365, 5368, and 5375 (for valid renamings).
  - Conjectures 1, 2, 3 and 4 (Section 4) now proved in `coq-formalisation/Characterising_renaming.v` on lines 247, 301, 314, and 359 respectively. Conjecture 4 is slightly weakened to include the assumption that at least one of the new names is fresh.
  - Proposition 5 (Section 4) in `coq-formalisation/Default_renaming.v` on line 3592.

- For ROTOR:

  - [line 1197 of the paper] We evaluated Rotor on two substantial, real-world code-bases.
  
    The codebases are included in the Docker image in the `jane-street` and `ocaml` subdirectories of the `testbeds` directory. The results of this evaluation are given in the archives `jane-street-test-results.tar` and `ocaml-compiler-test-results.tar`. The results can be reproduced using the scripts included in the Docker image described in the step-by-step instructions above.

  - [line 1209 of the paper] Setting aside the cases that we do not handle, and the cases which fail because they generate a requirement to rename an (external) library function, at the point of writing more than 70% of the tests pass.

    This should actually say "almost 70%" (it is a typo).

    This can be verified by looking at the files in the `stats` subdirectory of each of the Jane Street and OCaml compiler testbed results, found in the archive files `jane-street-test-results.tar` and `ocaml-compiler-test-results.tar`.

    - For the OCaml Compiler test suite, 1721 cases successfully rebuild, whereas recompilation fails for 901 cases; in 27 cases ROTORS raises an error because the source file corresponding to the module containing the declaration of the value to be renamed is automatically generated during the build process. This corresponds to a success rate of 65% overall, or 68% when discounting the 27 cases that fail because no corresponding source file exists.

    - For the Jane Street test suite, the situation is a little more complicated. This is because the Jane Street code makes heavy use of OCaml's PPX preprocessing system to generate code at compile time. This means that many test cases actually rely on renaming a function declaration that is atuomatically generated. This is problematic since, in order to rename these functions, ROTOR would have to modify the preprocessor code to generate different names. This might involve, e.g., changing a name inside a string literal. In general, it requires reasoning on the meta-level and is currently beyond ROTOR's capabilities.

      The test results show that 244 test cases fail because they involve having to rename an external dependency (i.e. library function) or carry out renaming in a source file that is automatically generated by the build system. These cases are given in the file `error-cases.csv` in the `stats` subdirectory of the test results.

      There are 1215 test cases for which recompilation is successful. These can be found in the `success-cases-stats.csv` file in the `stats` subdirectory of the test results.

      There are 1615 test cases that fail to recompile. These are given in the `failure-cases-stats.csv` file in the `stats` subdirectory. However, we have identified that 814 of these fail due to the need to rename functions that are declared by automatically PPX generated code.

      Thus, discounting the cases failing due to external dependencies and PPX generated code, the success rate in the Jane Street test suite is 60%.

      Aggregated statistics can be found in the spreadsheets `success-cases-stats.ods` and `failure-cases-stats.ods` in the `stats` subdirectory of the test results. The spreadsheet detailing the failing cases also contains a worksheet listing those test cases that we have identified as failing due to complications involving PPX preprocessing.

      We have analysed the build logs of those test cases failing to recompile but not due to complications arising from PPX. Detailed notes of our analysis can be found in the file `build-error-notes.txt` in the `stats` subdirectory of the test results archive. This analysis shows a number of different bugs in the implementation of ROTOR, which we are currently working to resolve.

  - [line 1215 of the paper] Among the refactorings for the OCaml compiler, more than thirty generate sets of dependencies of size at least 24, and over a hundred have non-trivial sets of dependencies. These more complex refactorings typically span multiple files, and generate multiple patches.

    This claim is made at the top of the first column of page 12.

    It can be verified by examining the file `success-cases-stats.ods` in the `stats` subdirectory of the OCaml compiler test suite results, in the `ocaml-compiler-test-results.tar` archive. The rows may be sorted by column D, which contains the number of renaming dependencies for each test case.

    The table of statistics for the results of the OCaml compiler test suite found near the top of the first column of page 12 is also reproduced in this file.

The following are a list of claims in the paper that are **NOT** supported by the artifact.

- Proposition 6 (Adequacy).
