import numpy as np
import os,sys

import modules_mcmc_zipf as zipf


def powerlaw_anticorrelated_make_sequence(Ntokens, Ntypes, xmin, alpha, B, ):
    '''
    Generate a sequence of N symbols $x_i \in [xmin,xmin+Ntypes]$ such that the autocorrelation at lag 1 is negtive and 
        $$N \rightarrow \infty \Rightarrow p(x) \sim x^{-\alpha}$$


    I) split the interval $[xmin,xmin+Ntypes]$ in four groups:

        g1. $[xmin, xmin+n]$ such that the sum of $p(x)$ for these items $p(g1) \approx 1/4$ 

        g2. $[xmin+n+1, \bar{x}$], where $\bar{x}$ is the largest integer smaller than the average of $x$ 

        g3. $[\bar{x}+1, \bar{x}+m]$ such that $p(g3) \approx p(g1)$

        g4. $[\bar{x}+m, xmin+Ntypes]$


    II) Sequence is generated by repeating the following step:
            1. With probability p(g1)+p(g3) draw 2B numbers according to rule 1
            2. With the complementary probability draw 2B number acccording to rule 2
            
        Rule 1: - draw a number in g1 according to p(x) and immediately after draw a number in g3 according to p(x)

        Rule 2: - decide group g, either g2 or g4, respecting probability   p(g2) and p(g4);
                - draw a number in the selected group according to p(x)
    
    
    '''

    ## true distribution
    truepi= zipf.p_power_law(Ntypes,alpha,xmin)
    truepia=np.array([])
    for i in sorted(truepi.keys()):
        truepia=np.append(truepia,truepi[i])
    #################################################################
    ## determine the 4 different intervals

    i0 = 0
    x0 = xmin

    #first gropu
    i=0
    while sum(truepia[:i])<0.25:
        i=i+1
    i1=i-1
    x1=xmin+i1-1
    pg1=sum(truepia[:i1])

    #Second group
    #average
    avg=0
    for i in truepi:
        avg=avg+i*truepia[i-xmin]
    xbar=int(avg)
    i2 = xbar-xmin+1
    x2 = xmin+i2-1
    pg2=sum(truepia[i1:i2])
    if(1-pg2-pg1 < pg1):
        print("Problem, sum of probabilities does not allow to construct")
    # third group
    i=i2
    while sum(truepia[i2:i])<pg1:
        #print(i,sum(truepia[xbar-xmin+1:i]))
        i=i+1
    i3=i-1
    x3 = xmin+i3-1
    pg3=sum(truepia[i2:i3])
    # fourth group
    i4 = Ntypes
    x4 = xmin+i4-1

    pg4=sum(truepia[i3:i4])


    ##########################################################
    ## generate the sequence
    #precomputed random values for each of the four groups:

    group1 = zipf.draw_power_binary(int(Ntokens/2),x0,x1,alpha) 
    # pre-computed list of random numbers from power-law, drawn in advance for efficiency reasons
    group2 = zipf.draw_power_binary(int(Ntokens),x1+1,x2,alpha) 
    # pre-computed list of random numbers from power-law, drawn in advance for efficiency reasons
    group3 = zipf.draw_power_binary(int(Ntokens/2),x2+1,x3,alpha) 
    # pre-computed list of random numbers from power-law, drawn in advance for efficiency reasons
    group4 = zipf.draw_power_binary(int(Ntokens),x3+1,x4,alpha) 
    # pre-computed list of random numbers from power-law, drawn in advance for efficiency reasons

    x=[]
    while len(x)< Ntokens:
        if np.random.rand() < pg1+pg3: # use rule 1
            if np.random.rand()<0.5:
                for i in range(B): #Add 2B numbers to sequence x
                
                
                    x.append(group1[0])
                    group1=group1[1:]
                    x.append(group3[0])
                    group3=group3[1:]
                    ## if we have N datapoints already --> stop the loop
                    if len(x)==Ntokens:
                        break
            else:
                for i in range(B): #Add 2B numbers to sequence x

                    x.append(group3[0])
                    group3=group3[1:]
                    x.append(group1[0])
                    group1=group1[1:]
                    ## if we have N datapoints already --> stop the loop
                    if len(x)==Ntokens:
                        break
        else: # use rule 2
            for i in range(2*B): #Add 2B number following rule 2
                if np.random.rand() <= pg2/(pg2+pg4):
                    x.append(group2[0])
                    group2=group2[1:]
                else:
                    x.append(group4[0])
                    group4=group4[1:]     
                    ## if we have N datapoints already --> stop the loop
                if len(x)==Ntokens:
                    break           

    x_data = np.array(x)

    return x_data



