! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      
      implicit none
      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 1 ! minutes

      
      contains
      
      
      subroutine extras_controls(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine extras_controls
      
      
      integer function extras_startup(id, restart, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_startup = 0
         call system_clock(time0,clock_rate)
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(id, id_extra, ierr)
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         double precision :: dt
         character (len=strlen) :: test
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         call GET_ENVIRONMENT_VARIABLE( &
            "MESA_TEST_SUITE_CHECK_RUNTIME", test, status=ierr, trim_name=.true.)
         if (ierr == 0 .and. trim(test) == 'true' .and. dt > 1.5*expected_runtime) then
            write(*,'(/,a70,2f12.1,99i10/)') &
               'failed: EXCESSIVE runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         else
            write(*,'(/,a50,2f12.1,99i10/)') 'runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
         ierr = 0
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_history_columns(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_history_columns = 0
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(id, id_extra, n, names, vals, ierr)
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(id, id_extra)
         use star_def, only: star_info
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(id, id_extra, n, nz, names, vals, ierr)
         use star_def, only: star_info, maxlen_profile_column_name
         use const_def, only: dp
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: k
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s

         integer :: i, k, k_at_maxdq_constant_abundance
         integer :: kA, kB,k_check,i_lnT, j00, jp1
         integer :: k_at_maxdq_minus_10,k_looper
         real (dp) :: dt_this_shell, h1_surface_abundance, &
         max_dq_at_const_abundance, delta_m, old_xmstar, new_xmstar
         logical :: flag,dbg
         real(dp) :: lnTlim_A, lnTlim_B, sumdq, sumdq1, sumdq2, sumdq3, &
            min_xq_const_mass, min_q_for_kB, frac, lnTmax, lnT_A, lnT_B, &
            qA, qB_old, qB_new, qfrac, dt_previous, potential_dt, &
            eps_grav_time_deriv_separation
         integer :: max_k_const_abund

         ierr = 0
         call star_ptr(id, s, ierr)
         ! default return is terminate
         if (ierr /= 0) return

         eps_grav_time_deriv_separation = s% eps_grav_time_deriv_separation
     
         flag = .false.
         dbg = .false.
         extras_finish_step = keep_going
         
         if (s% mstar_dot <= 0) return
         
         ! below here implements
         ! timestep constraint for accretion problems

         ! Find the zone with the largest dq in the region with
         ! constant abundance starting from the surface
         ! keep its index and the max dq
         max_dq_at_const_abundance = 0
         max_k_const_abund = 1
         do k = 2, s% nz
            ! constant abundance is defined not too strictly
            ! generall dq will decrease as we approach abundance transition
            ! so the selection of maxdq will keep L-H transition out of abundance transition
            if ( abs( s% xa(1,k) - s% xa(1,k-1) ) > 0.001) then
               exit
            else
               max_k_const_abund = k
               if(s% dq(k) > max_dq_at_const_abundance) then
                  k_at_maxdq_constant_abundance = k
                  max_dq_at_const_abundance = s% dq(k)
               end if
               cycle
            end if
         end do
         if(k_at_maxdq_constant_abundance - 11 > 0) then
            k_at_maxdq_minus_10 = k_at_maxdq_constant_abundance - 11
         else
            k_at_maxdq_minus_10 = 1
         end if

         if (dbg) then
            write(*,*) 'max_k_const_abund=', max_k_const_abund
            write(*,*) 'k_at_maxdq_constant_abundance=', k_at_maxdq_constant_abundance
            write(*,*) 'k_at_maxdq_minus_10=', k_at_maxdq_minus_10
         end if

         ! starting from surface, for each shell in the mesh do:
         ! 1. estimate dt constraint based on local dq and q
         ! 2. compute mass that would be added if that dt was used
         ! 3. find location (kB) of lagrangian/homologous boundary for that
         !    mass added.
         ! repeat those steps until kB > k_at_maxdq_minus 10, i.e. until the
         ! L-H boundary is inferred to be no deeper than 10 cells outside the 
         ! largest shell in the constant abundance region

         ! Use the last tried dt that wasn't too close as the dtnext

         ! constraint from just first zone
         dt_previous = (s% xmstar * s% dq(1) * & 
             eps_grav_time_deriv_separation)/(s% mstar_dot * s%  q(1))
         if (dbg) write(*,*) 'inital dt_previous=', dt_previous

         do k = 1, s% nz
            ! estimated dt constraint
            dt_this_shell = (s% xmstar * s% dq(k) * &
                        eps_grav_time_deriv_separation)/(s% mstar_dot * s% &
                                q(k))
                
            ! mass accreted for that
            delta_m = dt_this_shell * s% mstar_dot

            ! now reproduce logic to determine kB from adjust_mass.f90
            old_xmstar = s% xmstar
            new_xmstar = old_xmstar + delta_m
            lnTlim_A = ln10*s% max_logT_for_k_below_const_q

            lnTlim_B = ln10*s% max_logT_for_k_const_mass

            frac = old_xmstar / new_xmstar
            s% max_q_for_k_below_const_q = frac
            i_lnT = s% i_lnT

            lnTmax = maxval(s% xh(i_lnT,1:s% nz))

            lnT_A = min(lnTmax, lnTlim_A)
            kA = 5
            do k_looper = 5, s% nz-1
               kA = k_looper
               if (s% q(k_looper) > s% max_q_for_k_below_const_q) cycle
                  if (s% q(k_looper) <= frac) then
                     if (.not. flag) then
                        k_check = k_looper
                           flag = .true.
                     end if
                     if( (k_looper > 5+k_check) .and. ((frac - s% q(k_looper)) > 5*abs((1.0- &
                                frac))) .and. (s% q(k_looper)*( dt_this_shell* s% mstar_dot)/(new_xmstar*s% dq(k_looper))) < &
                                                  eps_grav_time_deriv_separation) then
                        exit
                     else
                        cycle
                     end if
                  end if
               if (s% xh(i_lnT,k_looper) >= lnT_A .or. &
                  s% q(k_looper) <= s% min_q_for_k_below_const_q) exit
            end do


            qA = s% q(kA)
            lnT_B = min(lnTmax, lnTlim_B)
            kB = kA+1
            do k_looper = kB, s% nz
               kB = k_looper
               if ((dt_this_shell * s% mstar_dot)/s% dm(k_looper) >  &
                        eps_grav_time_deriv_separation) then
                   cycle
               else
                   exit
               end if
               !if( s% q(k) > s% max_q_for_k_const_mass) cycle
               if (s% xh(i_lnT,k_looper) >= lnT_B .or. &
                  s% q(k_looper) <= s% min_q_for_k_const_mass) exit
            end do
            qB_old = s% q(kB)

            qB_new = qB_old * frac ! in order to keep m(kB) constant
            !write(*,2) 'frac', kB, frac

            do ! make sure qfrac is not too far from 1 by moving kB inward
               qfrac = (qA - qB_new) / max(1d-99,qA - qB_old)
               if (kB == s% nz) exit
               if (kB-kA > 10) then
                  if (qfrac > 0.67d0 .and. qfrac < 1.5d0) then
                     exit
                  end if
                  if (qfrac > 0.50d0 .and. qfrac < 2.0d0) then
                     j00 = maxloc(s% xa(:,kB),dim=1) ! most abundant species at kB
                     jp1 = maxloc(s% xa(:,kB+1),dim=1) ! most abundant species at kB+1
                     if (j00 /= jp1) then ! change in composition.
                        if (dbg) write(*,*) 'change in composition.  back up kB.'
                        kB = max(1,kB-5)
                        exit
                     end if
                  end if
               end if
               kB = kB+1
               qB_old = s% q(kB)
               qB_new = qB_old * frac
            end do


            if(kB > k_at_maxdq_minus_10) then
                potential_dt = dt_previous
                exit ! loop over trial dt values
            else
                dt_previous = dt_this_shell
                cycle
            end if
         end do

         if (dbg) then
            write(*,*) 'last kB=', kB
            write(*,*) 'nz=', s% nz
            write(*,*) 'potential_dt=', potential_dt/3.15e7
         end if

         if (potential_dt < s% dt_next) then
            s% dt_next = potential_dt
         end if
                
         
         call store_extra_info(s)
      end function extras_finish_step
 







      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            double precision :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      end module run_star_extras
      
