! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      
      implicit none
      
      integer :: time0, time1, clock_rate
      real(dp), parameter :: expected_runtime = 0.2 ! minutes

      real(dp) :: target_a3, target_b3, target_b1, target_c
      real(dp), parameter :: tolerance = 1d-6
      
      ! these routines are called by the standard run_star check_model
      contains
      
      
      subroutine extras_controls(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         ! this is the place to set any procedure pointers you want to change
         ! e.g., other_wind, other_mixing, other_energy  (see star_data.inc)
         
         
      end subroutine extras_controls


      ! subroutine extras_controls(id, ierr)
      ! 	use adipls_support, only: &
      ! 	   astero_other_procs, use_other_adipls_mode_info
      !    integer, intent(in) :: id
      !    integer, intent(out) :: ierr
      !    type (star_info), pointer :: s
      !    ierr = 0
      !    call star_ptr(id, s, ierr)
      !    if (ierr /= 0) return
      !    write(*,*) 'extras_controls: set use_other_adipls_mode_info'
      ! 	use_other_adipls_mode_info = .true.
      ! 	astero_other_procs% other_adipls_mode_info => my_other_adipls_mode_info
      ! end subroutine extras_controls


      ! subroutine my_other_adipls_mode_info( &
      !       l, order, freq, inertia, x, y, aa, data, nn, iy, iaa, ispcpr, ierr)
      !    integer, intent(in) :: l, order
      !    real(dp), intent(in) :: freq, inertia
      !    real(dp), intent(in) :: x(1:nn), y(1:iy,1:nn), aa(1:iaa,1:nn), data(8)
      !    integer, intent(in) :: nn, iy, iaa, ispcpr
      !    integer, intent(out) :: ierr
      !    ierr = 0
      !    write(*,*) 'astero called my_other_adipls_mode_info'
      ! end subroutine my_other_adipls_mode_info
      
      
      integer function extras_startup(id, restart, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_startup = 0
         call system_clock(time0,clock_rate)
         ! bp: changed these by tiny amounts
         ! after set include_dmu_dt_in_eps_grav = .true.          
         target_a3 = -3.2993015647783552d-19
         target_b3 = -2.9999998484444680d-19          
         target_b1 = -3.0000001315228224d-6          
         target_c = -1.8462781272406559d-16 

         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(id, id_extra, ierr)
         use astero_data
         use astero_support, only: get_one_el_info, &
              get_kjeldsen_radial_freq_corr, &
              get_cubic_all_freq_corr, &
              get_combined_all_freq_corr
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         
         real(dp) :: dt, expected_freq, freq
         logical :: store_model, okay
         character (len=strlen) :: test

         integer :: i
         real(dp) :: a3, b3, b1, c

         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return

         store_model = .true.

         call get_one_el_info(s, 0, &
               nu_lower_factor*l0_obs(1), &
               nu_upper_factor*l0_obs(nl0), &
               iscan_factor_l0*nl0, 1, nl0, store_model, &
               oscillation_code, ierr)
         !write(*,*) 'done adipls_get_one_el_info'
         if (ierr /= 0) then
            write(*,*) 'failed to find l=0 modes'
            stop 1
         end if

         call get_one_el_info(s, 1, &
               nu_lower_factor*l1_obs(1), &
               nu_upper_factor*l1_obs(nl1), &
               iscan_factor_l1*nl1, 1, nl1, store_model, &
               oscillation_code, ierr)
         !write(*,*) 'done adipls_get_one_el_info'
         if (ierr /= 0) then
            write(*,*) 'failed to find l=1 modes'
            stop 1
         end if

         call get_one_el_info(s, 2, &
               nu_lower_factor*l2_obs(1), &
               nu_upper_factor*l2_obs(nl2), &
               iscan_factor_l2*nl2, 1, nl2, store_model, &
               oscillation_code, ierr)
         !write(*,*) 'done adipls_get_one_el_info'
         if (ierr /= 0) then
            write(*,*) 'failed to find l=2 modes'
            stop 1
         end if

         call get_one_el_info(s, 3, &
               nu_lower_factor*l3_obs(1), &
               nu_upper_factor*l3_obs(nl3), &
               iscan_factor_l3*nl3, 1, nl3, store_model, &
               oscillation_code, ierr)
         !write(*,*) 'done adipls_get_one_el_info'
         if (ierr /= 0) then
            write(*,*) 'failed to find l=3 modes'
            stop 1
         end if

         ! do i = 1, nl0
         !    write(*,*) l0_obs(i), l0_obs_sigma(i), l0_freq(i), l0_inertia(i)
         ! end do

         ! do i = 1, nl1
         !    write(*,*) l1_obs(i), l1_obs_sigma(i), l1_freq(i), l1_inertia(i)
         ! end do

         ! do i = 1, nl2
         !    write(*,*) l2_obs(i), l2_obs_sigma(i), l2_freq(i), l2_inertia(i)
         ! end do

         ! do i = 1, nl3
         !    write(*,*) l3_obs(i), l3_obs_sigma(i), l3_freq(i), l3_inertia(i)
         ! end do


         call get_kjeldsen_radial_freq_corr( &
            a_div_r, correction_b, nu_max, correction_factor, .false., &
            nl0, l0_obs, l0_freq, l0_freq_corr, l0_inertia)

         c = (l0_freq_corr(nl0)-l0_freq(nl0))/l0_freq(nl0)**correction_b

         call get_cubic_all_freq_corr(a3, .false., &
            nl0, l0_obs, l0_obs_sigma, l0_freq, l0_freq_corr, l0_inertia, &
            nl1, l1_obs, l1_obs_sigma, l1_freq, l1_freq_corr, l1_inertia, &
            nl2, l2_obs, l2_obs_sigma, l2_freq, l2_freq_corr, l2_inertia, &
            nl3, l3_obs, l3_obs_sigma, l3_freq, l3_freq_corr, l3_inertia)

         call get_combined_all_freq_corr(b3, b1, .false., &
            nl0, l0_obs, l0_obs_sigma, l0_freq, l0_freq_corr, l0_inertia, &
            nl1, l1_obs, l1_obs_sigma, l1_freq, l1_freq_corr, l1_inertia, &
            nl2, l2_obs, l2_obs_sigma, l2_freq, l2_freq_corr, l2_inertia, &
            nl3, l3_obs, l3_obs_sigma, l3_freq, l3_freq_corr, l3_inertia)

         write(*,*) 'a3', a3, target_a3
         write(*,*) 'b3', b3, target_b3
         write(*,*) 'b1', b1, target_b1
         write(*,*) 'c', c, target_c
         
         okay = .true.
         if (abs(a3/target_a3-1d0) > tolerance) then 
            write(*,*) 'FAILED -- cubic correction a_3 too large'
            okay = .false.
         else
            write(*,*) 'cubic correction a_3 within tolerance'
         end if
         
         if (abs(b3/target_b3-1d0) > tolerance) then 
            write(*,*) 'FAILED -- combined correction b_3 too large'
            okay = .false.
         else
            write(*,*) 'combined correction b_3 within tolerance'
         end if
         
         if (abs(b1/target_b1-1d0) > tolerance) then 
            write(*,*) 'FAILED -- combined correction b_{-1} too large'
            okay = .false.
         else
            write(*,*) 'combined correction b_{-1} within tolerance'
         end if
         
         if (abs(c/target_c-1d0) > tolerance) then 
            write(*,*) 'FAILED -- Kjeldsen et al. correction c too large'
            okay = .false.
         else
            write(*,*) 'Kjeldsen et al. correction c within tolerance'
         end if
         
         
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         call GET_ENVIRONMENT_VARIABLE( &
            "MESA_TEST_SUITE_CHECK_RUNTIME", test, status=ierr, trim_name=.true.)
         if (ierr == 0 .and. trim(test) == 'true' .and. dt > 1.5*expected_runtime) then
            write(*,'(/,a70,2f12.1,99i10/)') &
               'failed: EXCESSIVE runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         else
            if (okay) write(*,*) 'All tests within tolerance'
            write(*,'(/,a50,2f12.1,99i10/)') 'runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
         ierr = 0
         
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_check_model = keep_going         
         if (.false. .and. s% star_mass_h1 < 0.35d0) then
            ! stop when star hydrogen mass drops to specified level
            extras_check_model = terminate
            write(*, *) 'have reached desired hydrogen mass'
            return
         end if


         ! if you want to check multiple conditions, it can be useful
         ! to set a different termination code depending on which
         ! condition was triggered.  MESA provides 9 customizeable
         ! termination codes, named t_xtra1 .. t_xtra9.  You can
         ! customize the messages that will be printed upon exit by
         ! setting the corresponding termination_code_str value.
         ! termination_code_str(t_xtra1) = 'my termination condition'

         ! by default, indicate where (in the code) MESA terminated
         if (extras_check_model == terminate) s% termination_code = t_extras_check_model
      end function extras_check_model
      
      
      integer function how_many_extra_history_columns(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_history_columns = 0
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(id, id_extra, n, names, vals, ierr)
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(id, id_extra)
         use star_def, only: star_info
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(id, id_extra, n, nz, names, vals, ierr)
         use star_def, only: star_info, maxlen_profile_column_name
         use const_def, only: dp
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: k
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_finish_step = keep_going
         call store_extra_info(s)
      end function extras_finish_step
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            real(dp) :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      end module run_star_extras
      
