! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use run_star_support
      
      implicit none
      
      character (len=256) :: masses_filename
      
      namelist /job/ &
         masses_filename
         
      double precision, pointer :: masses(:)
      integer :: nmasses, which_mass

      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 1.1 ! minutes


      contains
      

      subroutine do_run
         use run_star_support
         
         type (star_info), pointer :: s
         integer :: id, ierr, i, j, result
         logical :: first_try, restart
         double precision :: dt
         character (len=strlen) :: test

         logical, parameter :: dbg = .true., do_alloc_star = .false., &
            do_free_star = .false., okay_to_restart = .true.
         
         include 'formats'
         
         ierr = 0
         call system_clock(time0,clock_rate)

         call do_read_star_job('inlist', ierr)
         if (ierr /= 0) return

         id = id_from_read_star_job
         id_from_read_star_job = 0

         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call read_controls('inlist_job',ierr)
         if (ierr /= 0) return
      
         nullify(masses)
         call read_masses(masses_filename, masses, nmasses, ierr)
         if (ierr /= 0) return
         
         do i=1, nmasses
         
            which_mass = i ! for use by multistar_controls
            
            call run1_star( &
               do_alloc_star, do_free_star, okay_to_restart, &
               id, restart, &
               multistar_controls, &
               extras_startup, &
               extras_check_model, &
               how_many_extra_history_columns, &
               data_for_extra_history_columns, &
               how_many_extra_profile_columns, &
               data_for_extra_profile_columns, &
               extras_finish_step, &
               extras_after_evolve, &
               ierr)
            if (ierr /= 0) then
               write(*,*) 'failed'
               return
            end if
            
            do j = 1, 10
               write(*,*)
            end do
                        
         end do
         
         write(*,*)

         write (*, *)
         write (*, *)
         write(*, *)
         write(*, '(a)') 'finished'
         write(*, *)
         

         call starlib_shutdown
                  
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         call GET_ENVIRONMENT_VARIABLE( &
            "MESA_TEST_SUITE_CHECK_RUNTIME", test, status=ierr, trim_name=.true.)
         if (ierr == 0 .and. trim(test) == 'true' .and. dt > 1.5*expected_runtime) then
            write(*,'(/,a70,2f12.1,99i10/)') &
               'failed: EXCESSIVE runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         else
            write(*,'(/,a50,2f12.1,99i10/)') 'runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
         ierr = 0

      end subroutine do_run
      
      
      subroutine multistar_controls(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         
         integer :: i
         character (len=256) :: history_name, profiles_index_name, profile_data_prefix
         
         include 'formats'

         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         i = which_mass
         s% initial_mass = masses(i)
         
         if (i < 10) then
            write(history_name, '(a,i1,a)') 'i00', i, '_history.data'
            write(profiles_index_name, '(a,i1,a)') 'i00', i, '_profiles.index'
            write(profile_data_prefix, '(a,i1,a)') 'i00', i, '_profile'
         else if (i < 100) then
            write(history_name, '(a,i2,a)') 'i0', i, '_history.data'
            write(profiles_index_name, '(a,i2,a)') 'i0', i, '_profiles.index'
            write(profile_data_prefix, '(a,i2,a)') 'i0', i, '_profile'
         else
            write(history_name, '(a,i3,a)') 'i', i, '_history.data'
            write(profiles_index_name, '(a,i3,a)') 'i', i, '_profiles.index'
            write(profile_data_prefix, '(a,i3,a)') 'i', i, '_profile'
         end if 
         s% star_history_name = trim(history_name)
         s% profiles_index_name = trim(profiles_index_name)
         s% profile_data_prefix = trim(profile_data_prefix)
         
         write(*,*)
         write(*,2) 'mass', i, s% initial_mass
         write(*,2) 'star_history_name: ' // trim(s% star_history_name)
         write(*,2) 'profiles_index_name: ' // trim(s% profiles_index_name)
         write(*,2) 'profile_data_prefix: ' // trim(s% profile_data_prefix)
         write(*,*)
         
      end subroutine multistar_controls      

      
      subroutine read_controls(filename,ierr)
         use utils_lib
         character (len=*) :: filename
         integer, intent(out) :: ierr

         
         character (len=256) :: message
         integer :: unit
         
         ! set defaults
         masses_filename = 'mass.list'
         
         unit=alloc_iounit(ierr)
         if (ierr /= 0) return

         open(unit=unit, file=trim(filename), action='read', delim='quote', iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'Failed to open control namelist file ', trim(filename)
         else
            read(unit, nml=job, iostat=ierr)  
            close(unit)
            if (ierr /= 0) then
               write(*, *) 'Failed while trying to read control namelist file ', trim(filename)
               write(*, '(a)') &
                  'The following runtime error message might help you find the problem'
               write(*, *) 
               open(unit=unit, file=trim(filename), action='read', &
                  delim='quote', status='old', iostat=ierr)
               read(unit, nml=job)
               close(unit)
            end if  
         end if
         call free_iounit(unit)

      end subroutine read_controls
      
      
      subroutine extras_controls(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         ! this is the place to set any procedure pointers you want to change
         ! e.g., other_wind, other_mixing, other_energy  (see star_data.inc)
         
         
      end subroutine extras_controls
      
      
      integer function extras_startup(id, restart, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_startup = 0
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_history_columns(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_history_columns = 0
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(id, id_extra, n, names, vals, ierr)
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(id, id_extra)
         use star_def, only: star_info
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(id, id_extra, n, nz, names, vals, ierr)
         use star_def, only: star_info, maxlen_profile_column_name
         use const_def, only: dp
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: k
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      ! note: cannot request retry or backup; extras_check_model can do that.
      integer function extras_finish_step(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         real(dp) :: new_surface_rotation_v, new_omega
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         include 'formats'
         
         extras_finish_step = keep_going
         call store_extra_info(s)
         
         if (s% model_number == 10) then
            write(*,*) 'turn on velocities'
            call star_set_v_flag(id, .true., ierr)
            if (ierr /= 0) then
               write(*,*) 'failed'
               extras_finish_step = terminate
               return
            end if
         end if

         if (s% model_number == 30) then
            write(*,*) 'turn on rotation'
            call star_set_rotation_flag(id, .true., ierr)
            if (ierr /= 0) then
               write(*,*) 'failed'
               extras_finish_step = terminate
               return
            end if
            new_surface_rotation_v = 45
            new_omega = new_surface_rotation_v*1d5/s% r(1)
            write(*,1) 'new_surface_rotation_v =', new_surface_rotation_v, new_omega
            call star_set_uniform_omega(id, new_omega, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed'
               extras_finish_step = terminate
               return
            end if
         end if
         
         if (s% model_number == 50) then
            write(*,*) 'allow large timesteps'
            s% max_years_for_timestep = 1d99
         end if
            
      end function extras_finish_step
      
      
      subroutine extras_after_evolve(id, id_extra, ierr)
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         ierr = 0
         if (s% termination_code /= t_xa_central_lower_limit) then
            write(*,*) 'failed to get expected termination code'
            ierr = -1
         end if
      end subroutine extras_after_evolve
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            double precision :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      end module run_star_extras
      
