! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use crlibm_lib
      
      implicit none

      logical :: done1
      
      
      integer :: time0, time1, clock_rate
      double precision, parameter :: expected_runtime = 2.6 ! minutes
      
      


      contains

      
      subroutine extras_controls(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine extras_controls
      
      
      integer function extras_startup(id, restart, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_startup = 0
         call system_clock(time0,clock_rate)
         done1 = .false.
      end function extras_startup
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_history_columns(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_history_columns = 0
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(id, id_extra, n, names, vals, ierr)
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(id, id_extra)
         use star_def, only: star_info
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(id, id_extra, n, nz, names, vals, ierr)
         use star_def, only: star_info, maxlen_profile_column_name
         use const_def, only: dp
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: k
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_finish_step(id, id_extra)
         use chem_def
         integer, intent(in) :: id, id_extra
         integer :: c13, k_max_c13, k, ierr
         double precision :: max_c13, mass_max_c13, pocket_mass_c13
         type (star_info), pointer :: s
         include 'formats'
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_finish_step = keep_going
         c13 = s% net_iso(ic13)
         if (c13 == 0) then
            write(*,*) 'failed to find c13'
            stop 'extras_finish_step'
         end if
         
         if ((.not. done1) .and. abs(s% surface_c12 - 1.104e-2) < 1d-5) then
            write(*,1) 's% he_core_mass', s% he_core_mass
            write(*,1) 's% surface_c12', s% surface_c12
            if (abs(s% he_core_mass - 0.5877d0) > 1d-3) then
               write(*,1) 'bad value for s% he_core_mass', &
                  s% he_core_mass - 0.5877d0, s% he_core_mass, 0.5877d0
               stop 1
            end if
            k_max_c13 = maxloc(s% xa(c13,1:s% nz),dim=1)
            max_c13 = s% xa(c13,k_max_c13)
            if (max_c13 > 1d-4) then
               write(*,1) 'max_c13 too large too soon', max_c13
               stop 1
            end if
            done1 = .true.
            return
         end if
         
         k_max_c13 = maxloc(s% xa(c13,1:s% nz),dim=1)
         max_c13 = s% xa(c13,k_max_c13)
         if (max_c13 < 0.065) return
         
         mass_max_c13 = s% mstar*(s% q(k_max_c13) + s% dq(k_max_c13)/2)/Msun
         pocket_mass_c13 = 0
         do k = 1, s% nz
            if (s% xa(c13,k) > 1d-2) then
               pocket_mass_c13 = pocket_mass_c13 + s% dq(k)
               !write(*,2) 'pocket_mass_c13', k, s% dq(k)*s% star_mass, pocket_mass_c13*s% star_mass
            end if
         end do
         pocket_mass_c13 = pocket_mass_c13*s% star_mass ! mass in Msun units
         write(*,1) 'max_c13', max_c13
         write(*,1) 'mass_max_c13', mass_max_c13
         write(*,1) 'pocket_mass_c13', pocket_mass_c13
         if (abs(mass_max_c13 - 0.5877) > 5d-3) then
            write(*,*) 'bad value for mass_max_c13'
            write(*,1) 'mass_max_c13', mass_max_c13
            write(*,1) 'expected', 0.5877
            write(*,1) 'mass_max_c13-expected', mass_max_c13-0.5877
            stop 1
         end if
         if (pocket_mass_c13 < 5d-6) then
            write(*,*) 'bad value for pocket_mass_c13'
            write(*,1) 'pocket_mass_c13', pocket_mass_c13
            write(*,1) 'expected', 1.2d-5
            write(*,1) 'pocket_mass_c13-expected', pocket_mass_c13-1.2d-5
            stop 1
         end if
         write(*,'(a)') 'all values are within tolerance'
         extras_finish_step = terminate

      end function extras_finish_step
      
      
      subroutine extras_after_evolve(id, id_extra, ierr)
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         double precision :: dt
         character (len=strlen) :: test
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         call GET_ENVIRONMENT_VARIABLE( &
            "MESA_TEST_SUITE_CHECK_RUNTIME", test, status=ierr, trim_name=.true.)
         if (ierr == 0 .and. trim(test) == 'true' .and. dt > 1.5*expected_runtime) then
            write(*,'(/,a70,2f12.1,99i10/)') &
               'failed: EXCESSIVE runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         else
            write(*,'(/,a50,2f12.1,99i10/)') 'runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
         ierr = 0
      end subroutine extras_after_evolve
      

      end module run_star_extras
      
