! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module run_star_extras

      use star_lib
      use star_def
      use const_def
      use crlibm_lib
      
      implicit none
      
      integer :: time0, time1, clock_rate
      real(dp), parameter :: expected_runtime = 1.3 ! minutes

      real(dp), parameter :: axion_g10 = 0.7d0
      
      !namelist /bsm_physics/ axion_g10
      
            
      ! these routines are called by the standard run_star check_model
      contains
      
      
      subroutine extras_controls(id, ierr)
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         s% other_neu => other_neu
      end subroutine extras_controls
      
      
      !Friedland-Giannotti-Wise
      subroutine other_neu(  &
            id, k, T, log10_T, Rho, log10_Rho, abar, zbar, z2bar, log10_Tlim, flags, &
            loss, sources, ierr)
         use neu_lib, only: neu_get
         use neu_def
         integer, intent(in) :: id ! id for star         
         integer, intent(in) :: k ! cell number or 0 if not for a particular cell         
         real(dp), intent(in) :: T ! temperature
         real(dp), intent(in) :: log10_T ! log10 of temperature
         real(dp), intent(in) :: Rho ! density
         real(dp), intent(in) :: log10_Rho ! log10 of density
         real(dp), intent(in) :: abar ! mean atomic weight
         real(dp), intent(in) :: zbar ! mean charge
         real(dp), intent(in) :: z2bar ! mean charge squared
         real(dp), intent(in) :: log10_Tlim 
         logical, intent(inout) :: flags(num_neu_types) ! true if should include the type of loss
         real(dp), intent(out) :: loss(num_neu_rvs) ! total from all sources
         real(dp), intent(out) :: sources(num_neu_types, num_neu_rvs)
         integer, intent(out) :: ierr
         
         real(dp) :: ye, axionz2ye, axioncsi, faxioncsi, &
            sprimakoff, d_sprimakoff_dT, d_sprimakoff_dRho
         type (star_info), pointer :: s
         
         include 'formats'

         ierr = 0         
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call neu_get(  &
            T, log10_T, Rho, log10_Rho, abar, zbar, z2bar, log10_Tlim, flags, &
            loss, sources, ierr)
         if (ierr /= 0) return

         !..   Add axion production and energy loss through the  Primakoff process.
         !..   Fitting formula by Alexander Friedland and Maurizio Giannotti
         
         ye = zbar/abar
         axionz2ye=z2bar+ye

         axioncsi=  1.658d20*axionz2ye*Rho/(T*T*T)
         !.. coefficient = 4*pi*alpha/(4*(1 Kelvin)^3) * N_A * cm^(-3) 
         !..    pi*(1/137.0.35)/(1/11604.5)^3 * (6.022*10^23) * (197.326*10^6*10^(-13))^3

         faxioncsi = (46.6754 + 0.843043*axioncsi) &
            /(44.44 + (8.7822 + 0.116255*axioncsi)*axioncsi) &
            *log_cr(3.85 + 3.99/axioncsi)

         sprimakoff    =  4.66d-31*axionz2ye*faxioncsi*axion_g10**2*T*T*T*T
         !.. this is the loss rate per unit mass per unit time
         !.. the formula is (g/2*pi)^2 * T^4 * pi* alpha/m_P * faxioncsi*axionz2ye
         !.. in natural units, for g10 =1 and T = 1K, one gets
         !.. 10^-20/10^18/(4*pi^2) * (1/11604.5)^4 pi*1/137.035 1/(938.272*10^6) 
         !..     = 3.4129*10^-67 eV
         !.. Also, erg/s/g = 
         !.. (6.2415*10^11)/((2.99792458*10^8)/(197.326*10^6*10^-15)*5.609*10^32)
         !.. = 7.32432*10^-37 eV
         ! Therefore, the coefficient is 3.4129*10^-67/ 7.32432*10^-37 = 4.66*10^-31
         
         ! MIGHT WANT TO FIX THESE, BUT THEY DON'T SEEM TO BE REQUIRED NOW
         d_sprimakoff_dT = 0
         d_sprimakoff_dRho = 0
         
         loss(ineu) = loss(ineu) + sprimakoff 
         loss(idneu_dT) = loss(idneu_dT) + d_sprimakoff_dT
         loss(idneu_dRho) = loss(idneu_dRho) + d_sprimakoff_dRho
         
         !if (k == s% nz) write(*,3) 'axioncsi', s% model_number, k, axioncsi
         !if (k == s% nz) write(*,3) 'faxioncsi', s% model_number, k, faxioncsi
         !if (k == s% nz) write(*,3) 'sprimakoff', s% model_number, k, sprimakoff
         
         
            
            
      end subroutine other_neu
      
      
      
      
      integer function extras_startup(id, restart, ierr)
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_startup = 0
         call system_clock(time0,clock_rate)
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function extras_startup
      
      
      subroutine extras_after_evolve(id, id_extra, ierr)
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         real(dp) :: dt
         character (len=strlen) :: test
         logical :: okay
         include 'formats'
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         okay = .true.
         write(*,1) 's% star_age', s% star_age
         if (s% star_age > 6e5) then
            write(*,1) 'star_age too large'
            okay = .false.
         end if
         write(*,1) 's% power_neutrinos', s% power_neutrinos
         if (s% power_neutrinos < 1e3) then
            write(*,1) 'power_neutrinos too small'
            okay = .false.
         end if
         write(*,1) 's% non_nuc_neu(s% nz)', s% non_nuc_neu(s% nz)
         if (s% non_nuc_neu(s% nz) < 2d3) then
            write(*,1) 'center non_nuc_neu too small'
            okay = .false.
         end if
         call system_clock(time1,clock_rate)
         dt = dble(time1 - time0) / clock_rate / 60
         call GET_ENVIRONMENT_VARIABLE( &
            "MESA_TEST_SUITE_CHECK_RUNTIME", test, status=ierr, trim_name=.true.)
         if (ierr == 0 .and. trim(test) == 'true' .and. dt > 1.5*expected_runtime) then
            write(*,'(/,a70,2f12.1,99i10/)') &
               'failed: EXCESSIVE runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         else
            write(*,'(/,a50,2f12.1,99i10/)') 'runtime, prev time, retries, backups, steps', &
               dt, expected_runtime, s% num_retries, s% num_backups, s% model_number
         end if
         ierr = 0
         if (okay) write(*,*) 'all tests okay'
      end subroutine extras_after_evolve
      

      ! returns either keep_going, retry, backup, or terminate.
      integer function extras_check_model(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_check_model = keep_going         
      end function extras_check_model


      integer function how_many_extra_history_columns(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_history_columns = 0
      end function how_many_extra_history_columns
      
      
      subroutine data_for_extra_history_columns(id, id_extra, n, names, vals, ierr)
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_history_columns

      
      integer function how_many_extra_profile_columns(id, id_extra)
         use star_def, only: star_info
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         how_many_extra_profile_columns = 0
      end function how_many_extra_profile_columns
      
      
      subroutine data_for_extra_profile_columns(id, id_extra, n, nz, names, vals, ierr)
         use star_def, only: star_info, maxlen_profile_column_name
         use const_def, only: dp
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         integer :: k
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
      end subroutine data_for_extra_profile_columns
      

      ! returns either keep_going or terminate.
      integer function extras_finish_step(id, id_extra)
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         extras_finish_step = keep_going
         call store_extra_info(s)
      end function extras_finish_step
      
      
      ! routines for saving and restoring extra data so can do restarts
         
         ! put these defs at the top and delete from the following routines
         !integer, parameter :: extra_info_alloc = 1
         !integer, parameter :: extra_info_get = 2
         !integer, parameter :: extra_info_put = 3
      
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg    
         num_ints = i
         
         i = 0
         ! call move_dbl       
         
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            real(dp) :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info

      end module run_star_extras
      
