# data_plots.rb

class ProfilePlots

    include Math
    include Tioga
    include FigureConstants
    
    
        
    def plot1(xs, ys, title, xlabel, ylable, ymin, ymax, 
                show_zero = nil, skip_mass_locs = nil, dots = nil)
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        do_plot1(
            'xlist' => [xs],
            'ylist' => [ys],
            'color_list' => [BrickRed],
            'legend_list' => [nil],
            'line_type_list' => [Line_Type_Solid],
            'title' => title,
            'xlabel' => xlabel, 
            'ylable' => ylable, 
            'xleft' => xleft, 
            'xright' => xright, 
            'ymin' => ymin, 
            'ymax' => ymax, 
            'show_zero' => show_zero, 
            'skip_mass_locs' => skip_mass_locs, 
            'dots' => dots
            )
    end
    
    
    def show_rms(xs,ys,ys_legend)
        rms = sqrt(ys.dot(ys)/ys.length)
        #puts "#{ys_legend} rms #{rms}"
        # wrt radius
        sumy2 = 0
        sumx = 0
        ys.length.times {|i|
            if i == 0
                dx = xs[1] - xs[0]
            else
                dx = xs[i] - xs[i-1]
            end
            dx = -dx if dx < 0
            
            dx = 1
            
            sumx = sumx + dx
            sumy2 = sumy2 + dx*ys[i]*ys[i]
        }
        rms = sqrt(sumy2/sumx)
        puts "#{ys_legend} rms #{rms}"
    end
    
        
    def do_plot1(dict)
        xlist = dict['xlist']
        if xlist == nil
            xs = dict['xs']
            xlist = [xs] unless xs == nil
        end
        ylist = dict['ylist']
        if ylist == nil
            ys = dict['ys']
            ylist = [ys] unless ys == nil
        end
        color_list = dict['color_list'] # BrickRed, Teal, RoyalPurple
        color_list = [] if color_list == nil
        legend_list = dict['legend_list']
        legend_list = [] if legend_list == nil
        line_type_list = dict['line_type_list'] # Line_Type_Solid, Line_Type_Dash, Line_Type_Dot
        line_type_list = [] if line_type_list == nil
        title = dict['title']
        xlabel = dict['xlabel']
        ylable = dict['ylable']
        xleft = dict['xleft']
        xright = dict['xright']
        ymin = dict['ymin']
        ymax = dict['ymax']
        ymin = @force_ymin if ymin == nil
        ymax = @force_ymax if ymax == nil
        dy = dict['dy']
        show_zero = dict['show_zero']
        calc_rms = dict['calc_rms']
        skip_mass_locs = dict['skip_mass_locs']
        dots = dict['dots']
        evalstr = dict['evalstr']
        title = get_star_age_title(title) unless title == nil
        title = nil if @no_title
        show_box_labels(title, xlabel, ylable)
        n = xlist.length
        have_ymin = ymin != nil
        have_ymax = ymax != nil
        if calc_rms || (!have_ymax) || (!have_ymin)
            n.times { |i|
               xs = xlist[i]
               ys = ylist[i]
               ys_legend = legend_list[i] if legend_list.length > i
               xs_to_plot = xs[@grid_min..@grid_max]
               ys_to_plot = ys[@grid_min..@grid_max]
               ymn = ys_to_plot.min
               ymx = ys_to_plot.max
               unless have_ymin
                  ymin = ymn if ymin == nil || ymin > ymn
               end
               unless have_ymax
                  ymax = ymx if ymax == nil || ymax < ymx
               end
               show_rms(xs_to_plot,ys_to_plot,ys_legend) if calc_rms
            }
        end
        dy = ymax - ymin if dy == nil
        dy = 0.1 if dy <= 0
        ymin = ymin - 0.05*dy
        ymax = ymax + 0.05*dy
        t.show_plot('left_boundary' => xleft, 'right_boundary' => xright,
            'top_boundary' => ymax, 'bottom_boundary' => ymin) do
            n.times { |i|
               xs = xlist[i]
               ys = ylist[i]
               ys_legend = legend_list[i] if legend_list.length > i
               show_mesh(xs)
               show_y0(xleft, xright) if show_zero == true
               ycolor = color_list[i] if color_list.length > i
               line_type = line_type_list[i] if line_type_list.length > i
               t.show_polyline(xs, ys, ycolor, ys_legend, line_type)
               show_mass_locs(xs, ymin) unless xs == d.mass || skip_mass_locs == true
            }
            eval(evalstr) if evalstr != nil
        end
    end    
        
    def sound_speed(xs, title, xlabel, xleft, xright, ymin = -20, ymax = 20)
        plot1(xs, d.csound, title, 'Sound Speed --- Age %s', 
            xlabel, 'speed (cm/second)', xleft, xright, ymin, ymax)
    end
    
    def read_BP98_solar_sound_speed(path = 'star_data')
        # see Bahcall, Basu, & Pinsonneault, Phys Letters B 433 (1998) 1-8
        path = '' if path == nil
        path = path + '/' if path.length > 0 && path[-1..-1] != '/'
        @bp98_solar_cs = [
            @radius = Dvector.new,
            @csound = Dvector.new ]
        Dvector.read(path + 'bp98soundspeeds.dat', @bp98_solar_cs, 7) # data starts on line 7
    end
    
    def read_CESAM_solar_sound_speed(path = 'star_data')
        path = '' if path == nil
        path = path + '/' if path.length > 0 && path[-1..-1] != '/'
        @cesam_solar_cs = [
            Dvector.new,
            Dvector.new,
            Dvector.new,
            Dvector.new,
            Dvector.new,
            Dvector.new,
            Dvector.new ]
        Dvector.read(path + 'sun_cesam.txt', @cesam_solar_cs, 2) # data starts on line 7
    end

    
    def read_Serenelli_solar_sound_speed(path = 'star_data')
        # www.mpa-garching.mpg.de/~aldos
        path = '' if path == nil
        path = path + '/' if path.length > 0 && path[-1..-1] != '/'
        @aldo_solar_cs = [
            @radius = Dvector.new,
            @csound = Dvector.new ]
        Dvector.read(path + 'aldo_soundspeeds.data', @aldo_solar_cs, 6) # data starts on line 6
        @aldo_solar_cs[1].mul!(1e-5)
    end



    
    def read_solar_sound_speed_data(path = 'star_data')
        path = '' if path == nil
        path = path + '/' if path.length > 0 && path[-1..-1] != '/'
        @solar_cs_data = [
            Dvector.new, # R/Rsun
            Dvector.new, 
                # sound speed of the standard solar model S (Christensen-Dalsgaard, et al, 1996)
            Dvector.new
                # Sound-speed inversion results from the mean-multiplet frequencies
            ]
        Dvector.read(path + 'csound.data', @solar_cs_data, 12) # data starts on line 12
    end

    
    def read_solar_new_sound_speed_data(path = 'star_data')
        path = '' if path == nil
        path = path + '/' if path.length > 0 && path[-1..-1] != '/'
        @solar_new_cs_data = [
            Dvector.new, # R/Rsun
            Dvector.new # cs^2
                # sound speed of the standard solar model S (Christensen-Dalsgaard, et al, 1996)
            ]
        Dvector.read(path + 'c2-modelA2.data', @solar_new_cs_data, 2) # data starts on line 2
        @solar_new_cs_data[1].sqrt!
    end
    
    
    def solar_sound_speed_comparison(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('Sound Speed vs. Solar Data --- Age %s')
        show_box_labels(title, xlabel, 'relative diff: (mesa - data) / data')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        read_solar_sound_speed_data
        ys = Dvector.new(xs.length)
        mesa_ys = d.cs_at_cell_bdy.mul(1e-5)
        ys.length.times {|i| ys[i] = 
            Dvector.linear_interpolate(xs[i], @solar_cs_data[0], @solar_cs_data[2]) }
        ys.mul!(1e-5)
        do_plot1(
            'xs' => xs,
            'ys' => mesa_ys.sub(ys).div(ys),
            'xleft' => xleft, 
            'xright' => xright, 
            'title' => title, 
            'ymin' => ymin, 
            'ymax' => ymax
            )
            
    end
        
    
    def get_solar_data_interpolant(pm_flag)
        return get_data_interpolant(@solar_cs_data[0],@solar_cs_data[2],pm_flag)
    end
        
    
    def get_data_interpolant(xs,ys,pm_flag)
        if pm_flag then
            data_interpolant = Dvector.create_pm_cubic_interpolant(xs, ys)
        else
            start_clamped = false
            start_slope = 0
            end_clamped = false
            end_slope = 0
            data_interpolant = Dvector.create_spline_interpolant(
                xs, ys, start_clamped, start_slope, end_clamped, end_slope)
        end
        return data_interpolant
    end
    
    
    def get_sound_speed_ys(xs, new_data = false)
        pm_flag = true
        if new_data
           read_solar_new_sound_speed_data
           csdata_interpolant = 
               get_data_interpolant(@solar_new_cs_data[0],@solar_new_cs_data[1],pm_flag)
        else
           read_solar_sound_speed_data
           csdata_interpolant = 
               get_data_interpolant(@solar_cs_data[0],@solar_cs_data[2],pm_flag)
        end
        ys = xs.dup
        ys.length.times {|i| ys[i] = 
                    Dvector.pm_cubic_interpolate(xs[i], csdata_interpolant) }
        ys = ys.mul(1e-5)
        return ys
    end
    
    
    def solar_comparison(xs, xlabel, new_data = false, ymin = nil, ymax = nil)
        title = nil #get_star_age_title('Sound Speed Comparison --- Age %s')
        t.rescale(1.4)
        show_box_labels(title, xlabel, 'Sound Speed (Model - Sun) / Sun')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        if d.cs_at_cell_bdy == nil
            5.times { puts " " }
            puts "must have cs_at_cell_bdy in profile data for solar comparison"
            5.times { puts " " }
            return
        end
        mesa_ys = d.cs_at_cell_bdy.mul(1e-5)
        #mesa_ys = d.csound.mul(1e-5)

        ys = get_sound_speed_ys(xs, new_data)
        
        read_BP98_solar_sound_speed
        csdata_interpolant = 
            get_data_interpolant(@bp98_solar_cs[0], @bp98_solar_cs[1],true)
        ys2 = Dvector.new(xs.length)
        ys2.length.times {|i| ys2[i] = 
                    Dvector.pm_cubic_interpolate(xs[i], csdata_interpolant) }            
            
        read_Serenelli_solar_sound_speed
        csdata_interpolant = 
            get_data_interpolant(@aldo_solar_cs[0], @aldo_solar_cs[1],true)
        ys3 = Dvector.new(xs.length)
        ys3.length.times {|i| ys3[i] = 
                    Dvector.pm_cubic_interpolate(xs[i], csdata_interpolant) }            
        
        read_CESAM_solar_sound_speed
        rsun = 6.9598e10
        csdata_interpolant = 
            get_data_interpolant(@cesam_solar_cs[0].div(rsun).reverse, 
                  @cesam_solar_cs[6].sqrt.div(1e5).reverse,true)
        ys4 = Dvector.new(xs.length)
        ys4.length.times {|i| ys4[i] = 
                    Dvector.pm_cubic_interpolate(xs[i], csdata_interpolant) }            

        do_plot1(
            'xlist' => [xs, xs, xs, xs],
            'ylist' => [mesa_ys.sub(ys).div(ys), ys2.sub(ys).div(ys), 
                  ys3.sub(ys).div(ys), ys4.sub(ys).div(ys)],
            'color_list' => [BrickRed, Teal, RoyalPurple, Coral],
            'legend_list' => ['MESA', 'BBP98', 'S09', 'Ana using CESAM'],
            'line_type_list' => [Line_Type_Solid, Line_Type_Dash, 
               Line_Type_Dot, Line_Type_Dot_Dash],
            'show_zero' => true,
            'calc_rms' => true,
            'xleft' => xleft, 
            'xright' => xright, 
            'title' => nil, 
            'skip_mass_locs' => true,
            'ymin' => ymin, 
            'ymax' => ymax
            )
            
    end
    
    
    def brunt_N_comparison(xs, xlabel)
        #t.rescale(1.2)
        t.set_aspect_ratio(0.85)
        show_box_labels('', xlabel, 'N / (3GM/R^3)^{1/2}')
        xleft = 0.05
        xright = 1.02
        ymin = 0
        ymax = 8.4
        title = nil
        do_plot1(
            'xlist' => [xs],
            'ylist' => [d.brunt_N_dimensionless],
            'xleft' => xleft, 
            'xright' => xright, 
            'title' => nil, 
            'skip_mass_locs' => true,
            'evalstr' => 'brunt_N_decorations',
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def brunt_N_decorations
        xs = [0.1, 0.109, 0.130, 0.15, 0.2, 0.25, 0.3, 0.35, 0.4, 0.45, 
                0.5, 0.55, 0.6, 0.65, 0.7, 0.75, 0.8, 0.85, 0.9, 0.95, 0.96, 0.97, 0.98]
        ys = [0, 8, 7.05, 1.8, 2.6, 3, 3, 2.9, 2.6, 2.35, 
                2.1, 1.9, 1.6, 1.5, 1.38, 1.50, 1.63, 1.75, 1.88, 2.0, 1.7, 2.3, 3.6]
        t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
            'color' => Teal, 'scale' => 0.64)
        scale = 0.7
        x = 0.5
        y = 8; dy = 0.64*t.default_text_height_dy
        t.show_label('text' => 'M = 4.0 $\mathrm{M_\odot}$', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'Z = 0.02', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'X_c = 0.37', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'log L/$\mathrm{L_\odot}$ = 2.50', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'log $\mathrm{T_{eff}}$ = 4.125', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
    end
    
    
    def show_brunt_N_comparison
        #t.rescale(0.8)
        xs = d.r_div_R
        xlabel = @r_div_R_xlabel
        t.top_edge_type = AXIS_LINE_ONLY
        t.rescale(@legend_scale)
        t.legend_text_dy = 1.1
        t.show_plot_with_legend(
            'legend_top_margin' => 0.70,
            'legend_left_margin' => 0.05,
            'plot_scale' => 1,
            'legend_scale' => 1.3,
            'plot_right_margin' => 0) { 
                brunt_N_comparison(xs, xlabel) }
    end

    
    
    def brunt_frequency_comparison(xs, xlabel)
        #t.rescale(1.2)
        t.set_aspect_ratio(0.85)
        show_box_labels('', xlabel, '$\mathrm{\nu_{BV}}$ (cycles day^{-1})')
        xleft = 0.05
        xright = 1.02
        ymin = 0
        ymax = 11
        title = nil
        do_plot1(
            'xlist' => [xs],
            'ylist' => [d.brunt_frequency],
            'xleft' => xleft, 
            'xright' => xright, 
            'title' => nil, 
            'skip_mass_locs' => true,
            'evalstr' => 'brunt_frequency_decorations',
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def brunt_frequency_decorations
        xs = [0.117, 0.17, 0.2, 0.25, 0.3, 0.35, 0.4, 0.45, 0.5, 
              0.55, 0.6, 0.65, 0.7, 0.75, 0.8, 0.85, 0.9, 0.95, 0.965, 0.995]
        ys = [0  , 8   , 4.5, 6.25, 7,   7,    6.6, 6.25, 5.75, 
              5.25, 4.75, 4, 3.8, 4,  4.5, 5.15, 6, 6.25, 0, 10.6]
        t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
            'color' => Teal, 'scale' => 0.6)
        scale = 0.6
        x = 0.5
        y = 10.4; dy = 0.55*t.default_text_height_dy
        t.show_label('text' => 'M = 9.858 $\mathrm{M_\odot}$', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'Z = 0.015', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'X_c = 0.2414', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'log L/$\mathrm{L_\odot}$ = 3.966', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'log $\mathrm{T_{eff}}$ = 4.358', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
    end
    
    
    def show_brunt_frequency_comparison
        xs = d.r_div_R
        xlabel = @r_div_R_xlabel
        t.top_edge_type = AXIS_LINE_ONLY
        #t.rescale(0.8)
        t.legend_text_dy = 1.1
        t.show_plot_with_legend(
            'legend_top_margin' => 0.70,
            'legend_left_margin' => 0.05,
            'plot_scale' => 1,
            'legend_scale' => 1.3,
            'plot_right_margin' => 0) { 
                brunt_frequency_comparison(xs, xlabel) }
    end

    
    
    def log_brunt_N2_comparison(xs, xlabel)
        t.rescale(1.2)
        t.set_aspect_ratio(0.85)
        show_box_labels('', xlabel, 'log N^2')
        xleft = 0.0
        xright = -13.5
        ymin = -4.2
        ymax = -1.4
        title = nil
        do_plot1(
            'xlist' => [xs],
            'ylist' => [d.log_brunt_N2],
            'xleft' => xleft, 
            'xright' => xright, 
            'title' => nil, 
            'skip_mass_locs' => true,
            'evalstr' => 'log_brunt_N2_decoration',
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def log_brunt_N2_decoration
        xs = [0.1, 0.109, 0.130, 0.15, 0.2, 0.25, 0.3, 0.35, 0.4, 0.45, 
                0.5, 0.55, 0.6, 0.65, 0.7, 0.75, 0.8, 0.85, 0.9, 0.95, 0.96, 0.97, 0.98]
        ys = [0, 8, 7.05, 1.8, 2.6, 3, 3, 2.9, 2.6, 2.35, 
                2.1, 1.9, 1.6, 1.5, 1.38, 1.50, 1.63, 1.75, 1.88, 2.0, 1.7, 2.3, 3.6]
        t.show_marker('xs' => xs, 'ys' => ys, 'marker' => Bullet,
            'color' => Teal, 'scale' => 0.7)
        scale = 0.9
        x = 0.7
        y = 8; dy = 0.6
        t.show_label('text' => 'M = 4.0 $\mathrm{M_\odot}$', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'X_c = 0.37', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'log L/$\mathrm{L_\odot}$ = 2.50', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y-dy
        t.show_label('text' => 'log $\mathrm{T_{eff}}$ = 4.125', 'x' => x, 'y' => y, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
    end
    
    
    def show_log_brunt_N2_comparison
        t.top_edge_type = AXIS_LINE_ONLY
        t.rescale(@legend_scale)
        t.legend_text_dy = 1.1
        t.show_plot_with_legend(
            'legend_top_margin' => 0.70,
            'legend_left_margin' => 0.05,
            'plot_scale' => 1,
            'legend_scale' => 1.3,
            'plot_right_margin' => 0) { 
                log_brunt_N2_comparison(d.logxm, @logxm_xlabel) }
    end

    
    
    def show_solar_comparison
        xs = d.radius
        xlabel = @radius_xlabel
        t.top_edge_type = AXIS_LINE_ONLY
        t.rescale(@legend_scale)
        t.legend_text_dy = 1.1
        t.show_plot_with_legend(
            'legend_top_margin' => 0.60,
            'legend_left_margin' => 0.05,
            'plot_scale' => 1,
            'legend_scale' => 1.6,
            'plot_right_margin' => 0) { 
                background
                solar_comparison(xs, xlabel,false) }
    end

    
    
    def show_solar_new_comparison
        xs = d.radius
        xlabel = @radius_xlabel
        t.top_edge_type = AXIS_LINE_ONLY
        t.rescale(@legend_scale)
        t.legend_text_dy = 1.1
        t.show_plot_with_legend(
            'legend_top_margin' => 0.70,
            'legend_left_margin' => 0.05,
            'plot_scale' => 1,
            'legend_scale' => 1.6,
            'plot_right_margin' => 0) { 
                background
                solar_comparison(xs, xlabel, true) }
    end
    
    
    def solar_sound_speed2_comparison(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('Sound Speed vs. Solar Data --- Age %s')
        show_box_labels(title, xlabel, 'relative diff: (mesa \; cs^2 - data \; cs^2) / data \; cs^2')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        read_solar_sound_speed_data
        mesa_ys2 = d.cs_at_cell_bdy.mul(1e-5).pow(2)
        pm_flag = true
        csdata_interpolant = get_solar_data_interpolant(pm_flag)
        ys = xs.dup
        if pm_flag then
            ys.length.times {|i| ys[i] = 
                    Dvector.pm_cubic_interpolate(xs[i], csdata_interpolant) }
        else
            ys.length.times {|i| ys[i] = 
                    Dvector.spline_interpolate(xs[i], csdata_interpolant) }
        end
        ys.mul!(1e-5)
        ys2 = ys.pow(2)
        ymin = -0.015 if ymin == nil
        ymax = 0.015 if ymax == nil  

        ys_plot = mesa_ys2.sub(ys2).div(ys2)
        ymin = ys_plot.min
        ymax = ys_plot.max      
        puts "title #{title}"
        puts "xleft #{xleft}"
        puts "xright #{xright}"
        puts "ymin #{ymin}"
        puts "ymax #{ymax}"
        puts "ys_plot.length #{ys_plot.length}"
        puts "ys_plot.min #{ys_plot.min}"
        puts "ys_plot.max #{ys_plot.max}"
        do_plot1(
            'xs' => xs,
            'ys' => ys_plot,
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    
    def sound_speed_data(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('Solar Sound Speed Data --- Age %s')
        show_box_labels(title, xlabel, 'km/sec')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        ys = get_sound_speed_ys(xs)
        do_plot1(
            'xs' => xs,
            'ys' => ys,
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end 

    
    def bp98_solar_sound_speed_comparison(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('Sound Speed vs. Solar Model BP98 --- Age %s')
        show_box_labels(title, xlabel, 'relative diff: (mesa - BP98) / BP98')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        read_BP98_solar_sound_speed
        ys = Dvector.new(xs.length)
        ys.length.times {|i| ys[i] = 
            Dvector.linear_interpolate(xs[i], @bp98_solar_cs[0], @bp98_solar_cs[1]) }
        mesa_ys = d.cs_at_cell_bdy.mul(1e-5)
        do_plot1(
            'xs' => xs,
            'ys' => mesa_ys.sub(ys).div(ys),
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def heat_capacity(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('heat capacity --- Age %s')
        show_box_labels(title, xlabel, 'heat capacity')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        ys1 = d.cp
        ys2 = d.cv
        do_plot1(
            'xlist' => [xs,xs],
            'ylist' => [ys1,ys2],
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def x_eps_grav_comparison(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('eps grav and env eps grav --- Age %s')
        show_box_labels(title, xlabel, 'eps grav and env eps grav')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        ys1 = d.eps_grav
        ys2 = d.env_eps_grav
        do_plot1(
            'xlist' => [xs,xs],
            'ylist' => [ys1,ys2],
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def eps_grav_comparison(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('eps grav minus env eps grav --- Age %s')
        show_box_labels(title, xlabel, 'eps grav minus env eps grav')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        ys1 = d.eps_grav
        ys2 = d.env_eps_grav
        do_plot1(
            'xlist' => [xs],
            'ylist' => [ys1.sub(ys2)],
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def eps_grav_comparison_relative(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('(eps grav - env eps grav)/eps grav --- Age %s')
        show_box_labels(title, xlabel, '(eps grav - env eps grav)/eps grav')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        ys1 = d.eps_grav
        ys2 = d.env_eps_grav
        do_plot1(
            'xlist' => [xs],
            'ylist' => [ys1.sub(ys2).div(ys1)],
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def velocity_km_per_sec(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('Velocity --- Age %s')
        show_box_labels(title, xlabel, 'v (km s^{-1})')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        ys = d.velocity_km_per_sec
        do_plot1(
            'xlist' => [xs],
            'ylist' => [ys],
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def r_div_v(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('radius/speed of infall --- Age %s')
        show_box_labels(title, xlabel, '$r/|v|$ (s)')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        ys = d.v_div_r.map { |vr| (vr >= 0)? 1e20 : -1.0/vr }
        puts "ys.max #{ys.max}"
        puts "ys.min #{ys.min}"
        ymin = 0.5
        ymax = 7
        do_plot1(
            'xlist' => [xs],
            'ylist' => [ys],
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def ye(xs, xlabel, ymin = nil, ymax = nil)
        title = get_star_age_title('Ye --- Age %s')
        show_box_labels(title, xlabel, 'Ye')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        ys = d.ye
        puts "ys.max #{ys.max}"
        puts "ys.min #{ys.min}"
        ymin = 0.44
        ymax = 0.501
        do_plot1(
            'xlist' => [xs],
            'ylist' => [ys],
            'title' => title,
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'ymin' => ymin, 
            'ymax' => ymax
            )
    end
    
    
    def edvs(xs, xlabel, ymin = nil, ymax = nil)
        if d.edv_h1 == nil || @xaxis_by != 'radius'
            puts "edvs plot only when xaxis == radius and have diffusion velocities in log"
            return
        end
        t.set_aspect_ratio(0.85)
        title = '' #'Element Diffusion Speeds'
        t.rescale(1.0)
        show_box_labels(title, xlabel, '$|w|$ ($\mathrm{R_\odot}/\tau_\theta$)')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        do_plot1(
            'xlist' => [xs, xs, xs, xs],
            'ylist' => [d.edv_h1, d.edv_o16.neg, d.edv_mg24.neg, d.edv_he4.neg],
            'title' => title,
            'color_list' => [Black, Black, Blue, Red],
            'line_type_list' => [Line_Type_Solid, Line_Type_Dash, Line_Type_Dot_Dash, Line_Type_Long_Dash],
            'evalstr' => 'edvs_decorations',
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'dy' => 1e-6, 
            'ymin' => -1, 
            'ymax' => 190
            )
    end

    def edvs_decorations
            
        x = 0.1; dx = 0.07; dy = -10
        y = 180; scale = 1
        t.stroke_color = Black
        t.line_type = Line_Type_Solid
        t.stroke_line(x,y,x+0.7*dx,y)
        t.show_label('text' => 'H', 'x' => x+dx, 'y' => y-3, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y+dy
        
        t.stroke_color = Black
        t.line_type = Line_Type_Dash
        t.stroke_line(x,y,x+0.7*dx,y)
        t.show_label('text' => 'O', 'x' => x+dx, 'y' => y-3, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y+dy
        
        t.stroke_color = Tan
        t.line_type = Line_Type_Solid
        t.stroke_line(x,y,x+0.7*dx,y)
        t.show_label('text' => 'Fe (Z=21)', 'x' => x+dx, 'y' => y-3, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y+dy
        
        t.stroke_color = Blue
        t.line_type = Line_Type_Dot_Dash
        t.stroke_line(x,y,x+0.7*dx,y)
        t.show_label('text' => 'Fe', 'x' => x+dx, 'y' => y-3, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y+dy
        
        t.stroke_color = Red
        t.line_type = Line_Type_Long_Dash
        t.stroke_line(x,y,x+0.7*dx,y)
        t.show_label('text' => 'He', 'x' => x+dx, 'y' => y-3, 
            'color' => Black, 'scale' => scale, 'justification' => LEFT_JUSTIFIED)
        y = y+dy
        
        edv_data = read_edv_data('edv.data')
        t.show_polyline(edv_data[0], edv_data[7].neg, Tan, nil, Line_Type_Solid)
        
        x = 0.35
        #t.show_label('text' => 'fe', 'x' => x, 'y' => 73, 'color' => Black, 'scale' => 0.8)
        #t.show_label('text' => 'o', 'x' => x, 'y' => 45, 'color' => Black, 'scale' => 0.8)
        #t.show_label('text' => 'h', 'x' => x, 'y' => 18, 'color' => Black, 'scale' => 0.8)
        
        xs = [0.1, 0.2, 0.3, 0.4, 0.5, 0.6]
        t.show_marker('xs' => xs, 'ys' => [14.5, 17, 22, 30, 41.5, 56], 'marker' => Bullet,
            'color' => Teal, 'scale' => 0.5)
        t.show_marker('xs' => xs, 'ys' => [20, 32.5, 44.5, 60, 80, 110], 'marker' => Bullet,
            'color' => Teal, 'scale' => 0.5)
        t.show_marker('xs' => xs, 'ys' => [23, 39, 50, 66, 87, 118], 'marker' => Bullet,
            'color' => Teal, 'scale' => 0.5)
    end

    
    def read_edv_data(fname, path = 'star_data')
        path = '' if path == nil
        path = path + '/' if path.length > 0 && path[-1..-1] != '/'
        data = [
            Dvector.new, # R/Rsun
            Dvector.new, # mass 
            Dvector.new, # rho
            Dvector.new, # T
            Dvector.new, # edv_h
            Dvector.new, # edv_he
            Dvector.new, # edv_o
            Dvector.new  # edv_fe
            ]
        Dvector.read(path + fname, data, 3) # data starts on line 3
        return data
    end

    
    
    def edvs_fe(xs, xlabel, ymin = nil, ymax = nil)
        if d.edv_h1 == nil || @xaxis_by != 'radius'
            puts "edvs_fe plot only when xaxis == radius and have diffusion velocities in log"
            return
        end
        title = '' #'Element Diffusion Speeds'
        t.rescale(1.1)
        show_box_labels(title, xlabel, '$|w|$ ($\mathrm{R_\odot}/\tau_\theta$)')
        xleft = xs[@grid_max]
        xright = xs[@grid_min]
        do_plot1(
            'xlist' => [xs],
            'ylist' => [d.edv_mg24.neg],
            'title' => title,
            'color_list' => [Black, Black, Blue, Red],
            'line_type_list' => [Line_Type_Solid],
            'evalstr' => 'edvs_fe_decorations',
            'xleft' => xleft, 
            'xright' => xright, 
            'skip_mass_locs' => true, 
            'ycolor' => Black,
            'dy' => 1e-6, 
            'ymin' => -1, 
            'ymax' => 200
            )
    end

    def edvs_fe_decorations
            
        # Russian Fe points
        t.show_marker(
            'xs' => [0.2, 0.3, 0.4, 0.5, 0.6], 
            'ys' => [40.7, 51.3, 71, 95.1, 128.3], 'marker' => Square,
            'color' => Tan, 'scale' => 0.5)
        
        t.show_marker(
            'xs' => [0.2, 0.3, 0.4, 0.5, 0.6], 
            'ys' => [31.7, 42.3, 54.3, 73.9, 96.6], 'marker' => Square,
            'color' => Tan, 'scale' => 0.5)
            
        return
            
        x = 0.1; dx = 0.07; dy = -10
        y = 180; scale = 1
        
        edv_data = read_edv_data('edv.data')
        edv2_data = read_edv_data('edv2.data')
        t.show_polyline(edv_data[0], edv_data[7].neg, Red, nil, Line_Type_Dot)
        t.show_polyline(edv2_data[0], edv2_data[7].neg, Red, nil, Line_Type_Dash)

        xs = [0.1, 0.2, 0.3, 0.4, 0.5, 0.6]
        t.show_marker('xs' => xs, 'ys' => [23, 39, 50, 66, 87, 118], 'marker' => Bullet,
            'color' => Teal, 'scale' => 0.5)

    end


end
