from Tkinter import *
from tkFileDialog import *
from tkMessageBox import *

elements = ["None", "H1", "He4", "C12", "Mg24"]

NSPEC = 5

def ffloat(literal):
    from string import replace
    try:
        return float(literal)
    except:
        return replace(literal, "d", "e", 1)
        

def match_element(name):
    i = 0
    for e in elements:
        if name == "'"+e+"'":
            return i
        i = i+1
    return 0

def add_scalar_option(frame, label, variable):
        frame1 = Frame(frame)
        frame1.pack(side=TOP, fill=X)

        L = Label(frame1, text=label)
        L.pack(side=LEFT)
        e = Entry(frame1, textvariable=variable)
        e.pack(side=LEFT)

def add_xa_option(frame, label, start_species, variable):
        frame1 = Frame(frame)
        frame1.pack(side=TOP, fill=X)

        cll_scrollbar = Scrollbar(frame1, orient=VERTICAL)
        L = Label(frame1, text="Central Lower Limit ")
        L.pack(side=LEFT)
        cll_listbox = Listbox(frame1, selectmode=BROWSE,
                                   height=1,
                                   yscrollcommand=cll_scrollbar.set)
        cll_scrollbar.config(command=cll_listbox.yview)
        cll_scrollbar.pack(side=LEFT, fill=Y)
        cll_listbox.pack(side=LEFT)
        
        for item in elements:
            cll_listbox.insert(END, item)

        cll_listbox.see(start_species)
        e = Entry(frame1, textvariable=variable)
        e.pack(side=LEFT)
        return cll_scrollbar

def add_checkbox(frame, label, variable):
        frame1 = Frame(frame)
        frame1.pack(side=TOP, fill=X)

        c = Checkbutton(frame1, text=label, variable=variable)
        c.pack(side=LEFT)

def add_input_file_select(frame, label, variable):
        frame1 = Frame(frame)
        frame1.pack(side=TOP, fill=X)

        L = Label(frame1, text=label)
        L.pack(side=LEFT)
        e = Entry(frame1)
        e.pack(side=LEFT)
        e.delete(0, END)
        e.insert(END, variable.get())

        def file_select():
            file = askopenfilename()
            if (file != ()):
                variable.set(file)
            e.delete(0, END)
            e.insert(END, file)
            
        b = Button(frame1, text="Browse...", command=file_select)
        b.pack(side=LEFT)

def add_output_file_select(frame, label, variable):
        frame1 = Frame(frame)
        frame1.pack(side=TOP, fill=X)

        L = Label(frame1, text=label)
        L.pack(side=LEFT)
        e = Entry(frame1)
        e.pack(side=LEFT)
        e.delete(0, END)
        e.insert(END, variable.get())

        def file_select():
            file = asksaveasfilename()
            if (file != ()):
                variable.set(file)
            e.delete(0, END)
            e.insert(END, file)
            
        b = Button(frame1, text="Browse...", command=file_select)
        b.pack(side=LEFT)

def add_directory_select(frame, label, variable):
        frame1 = Frame(frame)
        frame1.pack(side=TOP, fill=X)

        L = Label(frame1, text=label)
        L.pack(side=LEFT)
        e = Entry(frame1)
        e.pack(side=LEFT)
        e.delete(0, END)
        e.insert(END, variable.get())

        def file_select():
            file = askdirectory()
            if (file != ()):
                variable.set(file)
            e.delete(0, END)
            e.insert(END, file)
            
        b = Button(frame1, text="Browse...", command=file_select)
        b.pack(side=LEFT)




class CompletionDialog(Toplevel):
    def __init__(self, basic_dialog):
        Toplevel.__init__(self, basic_dialog.master)
        self.title("MESA Completion")
        self.basic_dialog = basic_dialog
        frame = Frame(self)
        frame.pack(side=TOP, fill=X)

        add_scalar_option(frame, "Maximum Age ",
                          self.basic_dialog.max_age)
        
        add_scalar_option(frame,
                          "Maximum Model Number ",
                          self.basic_dialog.max_model_number)
        
        self.cll_scrollbar = add_xa_option(frame,
                                           "Central Lower Limit ",
                                           basic_dialog.xa_central_lower_limit_species,
                                           self.basic_dialog.xa_central_lower_limit)
        
        self.sul_scrollbar = add_xa_option(frame,
                                           "Surface Upper Limit ",
                                           basic_dialog.xa_surface_upper_limit_species,
                                           self.basic_dialog.xa_surface_upper_limit)

        add_scalar_option(frame, "H Power Limit ",
                          self.basic_dialog.power_h_burn_upper_limit)
        
        add_scalar_option(frame, "He Power Limit ",
                          self.basic_dialog.power_he_burn_upper_limit)

        add_checkbox(frame, "Stop on thermal pulse", self.basic_dialog.stop_at_TP)

        frame = Frame(self)
        frame.pack(side=TOP, fill=X)
        
        done_button = Button(frame, text="Done",
                              command = self.quit)
        done_button.pack(side=LEFT)

    def quit(self):
        self.basic_dialog.xa_central_lower_limit_species = int((float(self.cll_scrollbar.get()[0])*NSPEC+0.5))
        self.basic_dialog.xa_surface_upper_limit_species = int((float(self.sul_scrollbar.get()[0])*NSPEC+0.5))
        self.destroy()

#########

standard_windows = (("MAIN", "main"),
                    ("TRHo_Profile", "T-rho profile"),
                    ("Profile", "profile"),
                    ("CONV", "convection"),
                    ("Surf_Hist", "surface history"),
                    ("Power", "power"),
                    ("Abundance", "abundance"),
                    ("dynamo", "dynamo"))


class PgstarDialog(Toplevel):
    def __init__(self, basic_dialog):
        Toplevel.__init__(self, basic_dialog.master)
        self.title("MESA PGSTAR")
        self.basic_dialog = basic_dialog
        frame = Frame(self)
        frame.pack(side=TOP, fill=X)

        i = 0
        for window in standard_windows:
            c = Checkbutton(frame, text="Show "+window[1]+" window",
                            variable=self.basic_dialog.win_flag[i])
            c.grid(sticky=W, row=i)
            i = i+1

        frame = Frame(self)
        frame.pack(side=TOP, fill=X)
        
        done_button = Button(frame, text="Done",
                              command = self.quit)
        done_button.pack(side=LEFT)

    def quit(self):
        self.destroy()

#########

class OutputDialog(Toplevel):
    def __init__(self, basic_dialog):
        Toplevel.__init__(self, basic_dialog.master)
        self.basic_dialog = basic_dialog
        self.title("MESA Output")
        frame = Frame(self)
        frame.pack(side=TOP, fill=X)

        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_scalar_option(subframe, "Save model number ",
                          basic_dialog.save_model_number);
        add_output_file_select(subframe,
                               "Save model file name ",
                               basic_dialog.save_model_filename)

        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_output_file_select(subframe,
                               "Log columns file name ",
                               basic_dialog.log_columns_file)
        add_directory_select(subframe,
                             "Log directory ",
                             basic_dialog.log_directory)
        add_scalar_option(subframe, "Log entry interval ",
                          basic_dialog.log_cnt);
      
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_output_file_select(subframe,
                               "Profile columns file name ",
                               basic_dialog.profile_columns_file)
        add_scalar_option(subframe, "Profile interval ",
                          basic_dialog.profile_interval);
        
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_directory_select(subframe,
                             "Photo directory ",
                             basic_dialog.photo_directory)
        add_scalar_option(subframe, "Photo interval ",
                          basic_dialog.photostep);
        
        add_checkbox(subframe,
                     "Report hydro time step information ",
                     basic_dialog.report_hydro_dt_info)
        add_scalar_option(subframe, "Terminal output interval ",
                          basic_dialog.terminal_cnt);

        frame = Frame(self)
        frame.pack(side=TOP, fill=X)
        
        done_button = Button(frame, text="Done",
                              command = self.quit)
        done_button.pack(side=LEFT)

        self.transient(basic_dialog.master)

    def quit(self):
        self.destroy()

#########

class InitializationDialog(Toplevel):
    def __init__(self, basic_dialog):
        Toplevel.__init__(self, basic_dialog.master)
        self.title("MESA Initialization")
        self.basic_dialog = basic_dialog
        frame = Frame(self)
        frame.pack(side=TOP, fill=X)

        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_checkbox(subframe,
                     "Create pre main sequence model ",
                     basic_dialog.create_pre_main_sequence_model)
        add_scalar_option(subframe, "  Pre main sequence central T  ",
                          basic_dialog.pre_ms_T_c);
        
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_checkbox(subframe,
                     "Load saved model",
                     basic_dialog.load_saved_model)
        add_input_file_select(subframe,
                               "  Saved model name ",
                              basic_dialog.saved_model_name)
        
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_checkbox(subframe,
                     "Relax mass scale ",
                     basic_dialog.relax_mass_scale)
        add_scalar_option(subframe, "  Target mass  ",
                          basic_dialog.new_mass);
        add_scalar_option(subframe, "  Log change in mass per step  ",
                          basic_dialog.dlgm_per_step);
        
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_checkbox(subframe,
                     "Change reaction network ",
                     basic_dialog.change_net)
        add_scalar_option(subframe, "  New net name  ",
                          basic_dialog.new_net_name);

        add_directory_select(frame,
                             "Log directory ",
                             basic_dialog.log_directory)
        add_directory_select(frame,
                             "Photo directory ",
                             basic_dialog.photo_directory)
        
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        add_checkbox(subframe,
                     "Change hydro flag ",
                     basic_dialog.change_v_flag)
        add_checkbox(subframe,
                     "  Hydro active ",
                     basic_dialog.new_v_flag)
                
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        ssf = Frame(subframe)
        ssf.pack(side=TOP, fill=X)
        L = Label(ssf, text="Initial composition")
        L.pack(side=LEFT, fill=X)
        add_scalar_option(subframe, "  Initial H2  ",
                          basic_dialog.initial_h2);
        add_scalar_option(subframe, "  Initial He3  ",
                          basic_dialog.initial_he3);

        add_checkbox(frame,
                     "Force uniform initial composition  ",
                     basic_dialog.set_uniform_initial_composition)

        frame = Frame(self)
        frame.pack(side=TOP, fill=X)

        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        ssf = Frame(subframe)
        ssf.pack(side=TOP, fill=X)
        L = Label(ssf, text="Z fractions")
        L.pack(side=LEFT, fill=X)
        Radiobutton(subframe, text="AG89", variable=basic_dialog.initial_zfracs, value=1).pack(anchor=W)
        Radiobutton(subframe, text="GN93", variable=basic_dialog.initial_zfracs, value=2).pack(anchor=W)
        Radiobutton(subframe, text="GS98", variable=basic_dialog.initial_zfracs, value=3).pack(anchor=W)
        Radiobutton(subframe, text="L03", variable=basic_dialog.initial_zfracs, value=4).pack(anchor=W)
        Radiobutton(subframe, text="AGS04", variable=basic_dialog.initial_zfracs, value=5).pack(anchor=W)
        Radiobutton(subframe, text="AGSS09", variable=basic_dialog.initial_zfracs, value=6).pack(anchor=W)

        done_button = Button(frame, text="Done",
                              command = self.quit)
        done_button.pack(side=LEFT)

        self.transient(basic_dialog.master)

    def quit(self):
        self.destroy()

#########

class MicrophysicsDialog(Toplevel):
    def __init__(self, basic_dialog):
        Toplevel.__init__(self, basic_dialog.master)
        self.basic_dialog = basic_dialog
        self.title("MESA Microphysics")
        frame = Frame(self)
        frame.pack(side=TOP, fill=X)

        add_scalar_option(frame, "EOS file prefix ",
                          basic_dialog.eos_file_prefix);
        add_scalar_option(frame, "Opacity file prefix ",
                          basic_dialog.kappa_file_prefix);

        frame = Frame(self)
        frame.pack(side=TOP, fill=X)
        
        done_button = Button(frame, text="Done",
                              command = self.quit)
        done_button.pack(side=LEFT)

        self.transient(basic_dialog.master)

    def quit(self):
        self.destroy()

#########

class MacrophysicsDialog(Toplevel):
    def __init__(self, basic_dialog):
        Toplevel.__init__(self, basic_dialog.master)
        self.basic_dialog = basic_dialog
        self.title("MESA Macrophysics")
        frame = Frame(self)
        frame.pack(side=TOP, fill=X)

        add_scalar_option(frame, "Central Ye at which to activate hydro ",
                          basic_dialog.center_ye_limit_for_v_flag);
        add_checkbox(frame,
                     "  Use artificial viscosity  ",
                     basic_dialog.use_artificial_viscosity)
        add_scalar_option(frame, "  Linear coefficient ",
                          basic_dialog.l1_coef);
        add_scalar_option(frame, "  Quadratic coefficient ",
                          basic_dialog.l2_coef);

        
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        ssf = Frame(subframe)
        ssf.pack(side=TOP, fill=X)
        L = Label(ssf, text="Convection")
        L.pack(side=LEFT, fill=X)
        add_scalar_option(subframe, "  Mixing length alpha ",
                          basic_dialog.mixing_length_alpha);
        add_scalar_option(subframe, "  Overshoot above nonburn ",
                          basic_dialog.overshoot_f_above_nonburn);
        add_scalar_option(subframe, "  Overshoot below nonburn ",
                          basic_dialog.overshoot_f_below_nonburn);
        add_scalar_option(subframe, "  Overshoot above H burn ",
                          basic_dialog.overshoot_f_above_burn_h);
        add_scalar_option(subframe, "  Overshoot below H burn ",
                          basic_dialog.overshoot_f_below_burn_h);
        add_scalar_option(subframe, "  Overshoot above He burn ",
                          basic_dialog.overshoot_f_above_burn_he);
        add_scalar_option(subframe, "  Overshoot below He burn ",
                          basic_dialog.overshoot_f_below_burn_he);
        add_checkbox(subframe,
                     "  Use Henyey mixing length theory  ",
                     basic_dialog.use_Henyey_MLT)
       
        subframe = Frame(frame, bd=1, relief=SUNKEN)
        subframe.pack(side=TOP, fill=X)
        ssf = Frame(subframe)
        ssf.pack(side=TOP, fill=X)
        L = Label(ssf, text="Mass loss")
        L.pack(side=LEFT, fill=X)
        add_scalar_option(subframe, "  RGB to AGB wind switch ",
                          basic_dialog.RGB_to_AGB_wind_switch);
        self.add_mass_loss_options(subframe,
                                   "  Red giant branch mass loss formulation",
                                   basic_dialog.RGB_wind_scheme,
                                   basic_dialog)
        self.add_mass_loss_options(subframe,
                                   "  Asymptotic giant branch mass loss formulation",
                                   basic_dialog.AGB_wind_scheme,
                                   basic_dialog)
        add_scalar_option(subframe, "  Reimers wind eta ",
                          basic_dialog.Reimers_wind_eta);
        add_scalar_option(subframe, "  Blocker wind eta ",
                          basic_dialog.Blocker_wind_eta);
        add_scalar_option(subframe, "  Super-Eddington wind eta ",
                          basic_dialog.super_eddington_wind_eta);

        frame = Frame(self)
        frame.pack(side=TOP, fill=X)
        done_button = Button(frame, text="Done",
                              command = self.quit)
        done_button.pack(side=LEFT)

        self.transient(basic_dialog.master)

    def add_mass_loss_options(self, subframe, label, scheme_variable, basic_dialog):
        ssf = Frame(subframe)
        ssf.pack(side=TOP, fill=X)
        L = Label(ssf, text=label)
        L.pack(side=LEFT, fill=X)
        Radiobutton(subframe, text="None",
                    variable=scheme_variable,
                    value="").pack(anchor=W)
        Radiobutton(subframe, text="Dutch",
                    variable=scheme_variable,
                    value="Dutch").pack(anchor=W)
        Radiobutton(subframe, text="Reimers",
                    variable=scheme_variable,
                    value="Reimers").pack(anchor=W)
        Radiobutton(subframe, text="Blocker",
                    variable=scheme_variable,
                    value="Blocker").pack(anchor=W)
        Radiobutton(subframe, text="de Jager",
                    variable=scheme_variable,
                    value="de Jager").pack(anchor=W)
        Radiobutton(subframe, text="other",
                    variable=scheme_variable,
                    value="other").pack(anchor=W)


    def quit(self):
        self.destroy()

#########

class InlistDialog:

    def __init__(self, master):
        # Inlist simple edit frame
        self.master = master
        frame = Frame(master)
        frame.pack(side=TOP, fill=X)

        self.pgstar_flag = BooleanVar(value=1);
        add_checkbox(frame, "Display pgstar window", self.pgstar_flag)

        self.pause = BooleanVar(value=0);
        add_checkbox(frame, "Pause on each time step", self.pause)

        self.initial_mass = DoubleVar(value=1.0)
        add_scalar_option(frame, "M ", self.initial_mass)
        
        frame1 = Frame(frame)
        frame1.pack(side=TOP, fill=X)

        self.Y = DoubleVar(value=0.2485)
        add_scalar_option(frame, "Y ", self.Y)

        self.Z = DoubleVar(value=0.0169)
        add_scalar_option(frame, "Z ", self.Z)

        # Inlist file load/save frame
        frame = Frame(master)
        frame.pack(side=TOP)
        
        load_ilist_button = Button(frame, text="Load inlist template...",
                              command = self.load_inlist_template)
        load_ilist_button.pack(side=LEFT)
        save_ilist_button = Button(frame, text="Save inlist template...",
                              command = self.save_inlist_template)
        save_ilist_button.pack(side=LEFT)

        # Subdialog quantities

        # Completion dialog

        self.create_pre_main_sequence_model = BooleanVar(value=1)
        self.max_age = DoubleVar(value="5e9")
        self.max_model_number = IntVar(value=120000)
        self.xa_central_lower_limit = DoubleVar()
        self.xa_central_lower_limit_species = 0
        self.xa_surface_upper_limit = DoubleVar()
        self.xa_surface_upper_limit_species = 0
        self.power_h_burn_upper_limit = DoubleVar(value=1e99)
        self.power_he_burn_upper_limit = DoubleVar(value=1e99)
        self.stop_at_TP = BooleanVar(value=0);

        self.win_flag = []
        for window in standard_windows:
            self.win_flag.append(BooleanVar(value=0))
        
        self.win_flag[0].set(1)
        self.save_model_number = IntVar(value=0)
        self.save_model_filename = StringVar()
        self.log_columns_file = StringVar()
        self.profile_columns_file = StringVar()
        self.log_directory = StringVar()
        self.photo_directory = StringVar()
        self.load_saved_model = BooleanVar(value=0);
        self.saved_model_name = StringVar()
        self.pre_ms_T_c = DoubleVar(value=3e5)
        self.relax_mass_scale = BooleanVar(value=0);
        self.new_mass = DoubleVar()
        self.dlgm_per_step = DoubleVar(value=1e-3)
        self.change_net = BooleanVar(value=0);
        self.new_net_name = StringVar(value="")
        self.change_v_flag = BooleanVar(value=0);
        self.new_v_flag = IntVar(value=0);
        self.set_uniform_initial_composition = BooleanVar(value=0);
        self.initial_h2 = DoubleVar(value=3e-5)
        self.initial_he3 = DoubleVar(value=2e-5)
        self.initial_zfracs = IntVar(value=3);
        self.eos_file_prefix = StringVar(value="mesa")
        self.kappa_file_prefix = StringVar(value="gs98")
        self.center_ye_limit_for_v_flag = DoubleVar(value=0.45)
        self.mixing_length_alpha = DoubleVar(value=1.89)
        self.use_Henyey_MLT = BooleanVar(value=0);
        self.photostep = IntVar(value=10);
        self.profile_interval = IntVar(value=10);
        self.overshoot_f_above_nonburn = DoubleVar(value=0.016)
        self.overshoot_f_below_nonburn = DoubleVar(value=0.016)
        self.overshoot_f_above_burn_h = DoubleVar(value=0.016)
        self.overshoot_f_below_burn_h = DoubleVar(value=0.016)
        self.overshoot_f_above_burn_he = DoubleVar(value=0.016)
        self.overshoot_f_below_burn_he = DoubleVar(value=0.016)
        self.RGB_wind_scheme = StringVar(value="")
        self.AGB_wind_scheme = StringVar(value="")
        self.Reimers_wind_eta = DoubleVar(value="0.4")
        self.Blocker_wind_eta = DoubleVar(value="0.1")
        self.super_eddington_wind_eta = DoubleVar(value="0.5")
        self.RGB_to_AGB_wind_switch = DoubleVar(value="1e-4")
        self.report_hydro_dt_info = BooleanVar(value=0)
        self.use_artificial_viscosity = BooleanVar(value=1)
        self.l1_coef = DoubleVar(value=0.1)
        self.l2_coef = DoubleVar(value=2.0)
        self.log_cnt = IntVar(value=1)
        self.terminal_cnt = IntVar(value=1)

    def create_advanced_menu(self, menu):
        menu.add_command(label="Initialization", command=self.open_initialization)
        menu.add_command(label="Completion", command=self.open_completion)
        menu.add_command(label="Macrophysics", command=self.open_macrophysics)
        menu.add_command(label="Microphysics", command=self.open_microphysics)
        menu.add_command(label="PGSTAR", command=self.open_pgstar)
        menu.add_command(label="Output", command=self.open_output)

    def open_initialization(self):
        d = InitializationDialog(self)
        d.mainloop()

    def open_completion(self):
        advanced_completion = CompletionDialog(self)
        advanced_completion.mainloop()

    def open_macrophysics(self):
        d = MacrophysicsDialog(self)
        d.mainloop()

    def open_microphysics(self):
        d = MicrophysicsDialog(self)
        d.mainloop()

    def open_pgstar(self):
        advanced_pgstar = PgstarDialog(self)
        advanced_pgstar.mainloop()

    def open_output(self):
        advanced_output = OutputDialog(self)
        advanced_output.mainloop()

    def save_inlist_template(self):
        inlist = asksaveasfilename()
        if len(inlist)==0:
            return
        self.write_inlist_generic(inlist)        
        
    def write_inlist(self):
        self.write_inlist_generic("inlist")

    def load_inlist_template(self):
        inlist = askopenfilename()
        if len(inlist)==0:
            return
        f = open(inlist, "r")
        for line in f:
            for i in range(len(line)):
                if line[i]=='&':
                    break
            if i+1 < len(line):  # found an inlist starter
                for j in range(i+1, len(line)):
                    if line[j].isalnum()==0 and line[j]!='_':
                        break
                namelist = line[i+1:j].lower()
                if namelist=="star_job":
                    self.load_star_job(f)
                elif namelist == "controls":
                    self.load_controls(f)
                elif namelist == "pgstar":
                    self.load_pgstar(f)

    def write_inlist_generic(self, filename):
        f = open(filename, 'w')
        self.write_star_job(f)
        self.write_controls(f)
        self.write_pgstar(f)
        f.close()

    def read_value(self, line):
        for i in range(len(line)):
          if line[i]=='=':
              break
        for j in range(i+1,len(line)):
            if line[j].isspace()==0:
                break;
        if line[j]=='.': # boolean
            if line[j+1]=='t':
                return 1
            else:
                return 0
        else: # copy everything to whitespace or !
            for k in range(j+1,len(line)):
                if line[k].isspace()==1 or line[k]=='!' or line[k]=='\n':
                    break;
            return line[j:k]
        

    def read_keyword(self, f):
         for line in f:
            for i in range(len(line)):
                if line[i].isspace()==0:
                    break
            if line[i] == '!':
                continue
            elif line[i] == '/':
                return None
            elif line[i] == '\n':
                continue
            elif line[i].isalpha()==0:
                print "read failed: unexpected character in inlist file:"
                print line[i]
                print '\n'
                raise "inlist template load failed"
            elif i < len(line):  # found a keyword
                for j in range(i, len(line)):
                    if line[j].isalnum()==0 and line[j]!='_':
                        break
                keyword = line[i:j].lower()
                return (keyword, self.read_value(line[j:]))

    def write_star_job(self, f):
        f.write("&star_job\n")
        if self.save_model_number.get() != 0:
            f.write("  save_model_number = " +
                    str(self.save_model_number.get()) + "\n")
        if self.save_model_filename.get() != "":
            f.write("  save_model_filename = " +
                    self.save_model_filename.get() + "\n")
        if self.log_columns_file.get() != "":
            f.write("  log_columns_file = " +
                    self.log_columns_file.get() + "\n")
        if self.profile_columns_file.get() != "":
            f.write("  profile_columns_file = " +
                    self.profile_columns_file.get() + "\n")
        if self.create_pre_main_sequence_model.get():
            f.write("  create_pre_main_sequence_model = .true.\n")
        else:
            f.write("  create_pre_main_sequence_model = .false.\n")
        if self.pre_ms_T_c.get() != 0:
            f.write("  pre_ms_T_c = " + str(self.pre_ms_T_c.get()) + "\n")
        if self.load_saved_model.get():
            f.write("  load_saved_model = .true.\n")
        else:
            f.write("  load_saved_model = .false.\n")
        if self.saved_model_name.get() != "":
            f.write("  saved_model_name = " +
                    self.saved_model_name.get() + "\n")
        f.write("  initial_he4 = " + str(self.Y.get()) + "\n")
        if self.pgstar_flag.get():
            f.write("  pgstar_flag = .true.\n")
        else:
            f.write("  pgstar_flag = .false.\n")
        if self.relax_mass_scale.get():
            f.write("  relax_mass_scale = .true.\n")
        else:
            f.write("  relax_mass_scale = .false.\n")
        if self.new_mass.get() != 0.0:
            f.write("  new_mass = " + str(self.new_mass.get()) + "\n")
        if self.dlgm_per_step.get() != 0:
            f.write("  dlgm_per_step = " + str(self.dlgm_per_step.get()) + "\n")
            
        if self.change_net.get():
            f.write("  change_net = .true.\n")
        else:
            f.write("  change_net = .false.\n")
        if self.new_net_name.get() != "":
            f.write("  new_net_name = " + self.new_net_name.get() + "\n")
             
        if self.change_v_flag.get():
            f.write("  change_v_flag = .true.\n")
        else:
            f.write("  change_v_flag = .false.\n")
        if self.new_v_flag.get():
            f.write("  new_v_flag = .true.\n")
        else:
            f.write("  new_v_flag = .false.\n")
             
        if self.set_uniform_initial_composition.get():
            f.write("  set_uniform_initial_composition = .true.\n")
        else:
            f.write("  set_uniform_initial_composition = .false.\n")

        f.write("  initial_h2 = " + str(self.initial_h2.get()) + "\n")
        f.write("  initial_he3 = " + str(self.initial_he3.get()) + "\n")
        f.write("  initial_zfracs = " + str(self.initial_zfracs.get()) + "\n")
        f.write("  eos_file_prefix = '" + str(self.eos_file_prefix.get()) + "'\n")
        f.write("  kappa_file_prefix = '" + str(self.kappa_file_prefix.get()) + "'\n")
        f.write("  center_ye_limit_for_v_flag = "
                + str(self.center_ye_limit_for_v_flag.get()) + "\n")

        f.write("/ ! end of star_job namelist\n\n")
 
    def load_star_job(self, f):
        key = self.read_keyword(f)
        while key != None:
            value = key[1]
            key = key[0]

#            print value
#            print key
            
            if key=="create_pre_main_sequence_model":
                self.create_pre_main_sequence_model.set(int(value))
            elif key=="initial_he4":
                self.Y.set(ffloat(value))
            elif key=="pgstar_flag":
                self.pgstar_flag.set(int(value))
            elif key=="save_model_number":
                self.save_model_number.set(int(value))
            elif key=="save_model_filename":
                self.save_model_filename.set(value)
            elif key=="log_columns_file":
                self.log_columns_file.set(value)
            elif key=="profile_columns_file":
                self.profile_columns_file.set(value)
            elif key=="load_saved_model":
                self.load_saved_model.set(int(value))
            elif key=="saved_model_name":
                self.saved_model_name.set(value)
            elif key=="pre_ms_t_c":
                self.pre_ms_T_c.set(ffloat(value))
            elif key=="dlgm_per_step":
                self.dlgm_per_step.set(ffloat(value))
            elif key=="change_net":
                self.change_net.set(int(value))
            elif key=="new_net_name":
                self.new_net_name.set(value)
            elif key=="change_v_flag":
                self.change_v_flag.set(int(value))
            elif key=="new_v_flag":
                self.new_v_flag.set(int(value))
            elif key=="set_uniform_initial_composition":
                self.set_uniform_initial_composition.set(int(value))
            elif key=="initial_h2":
                self.initial_h2.set(ffloat(value))
            elif key=="initial_he3":
                self.initial_he3.set(ffloat(value))
            elif key=="relax_mass_scale":
                self.relax_mass_scale.set(int(value))
            elif key=="new_mass":
                self.new_mass.set(ffloat(value))
            elif key=="initial_zfracs":
                self.new_mass.set(int(value))
            elif key=="eos_file_prefix":
                self.eos_file_prefix.set(value.strip("'"))
            elif key=="kappa_file_prefix":
                self.kappa_file_prefix.set(value.strip("'"))
            elif key=="center_ye_limit_for_v_flag":
                self.center_ye_limit_for_v_flag.set(value)
            elif key=="initial_h1":
                # Ignore for now. Should be defined by initial_he and
                # initial_z
                pass
            else:
                print "warning: unrecognized job keyword: " + key

            key = self.read_keyword(f)

    def write_controls(self, f):
        f.write("&controls\n")
        f.write("  initial_mass = " + str(self.initial_mass.get()) + "\n")
        f.write("  initial_z = " + str(self.Z.get()) + "\n")
        f.write("  max_age = " + str(self.max_age.get()) + "\n")
        f.write("  max_model_number = " + str(self.max_model_number.get())
                + "\n")
        if (self.xa_central_lower_limit_species>0):
            f.write("  xa_central_lower_limit_species(1) = '" +
                    elements[self.xa_central_lower_limit_species] +
                    "'\n")
            f.write("  xa_central_lower_limit(1) = " +
                    str(self.xa_central_lower_limit.get()) + "\n")
        if (self.xa_surface_upper_limit_species>0):
            f.write("  xa_surface_upper_limit_species(1) = '" +
                    elements[self.xa_surface_upper_limit_species] +
                    "'\n")
            f.write("  xa_surface_upper_limit(1) = " +
                    str(self.xa_surface_upper_limit.get()) + "\n")
        if (self.power_h_burn_upper_limit.get()<1e99):
            f.write("  power_h_burn_upper_limit = " +
                    str(self.power_h_burn_upper_limit.get()) +
                    "\n")
        if (self.power_he_burn_upper_limit.get()<1e99):
            f.write("  power_he_burn_upper_limit = " +
                    str(self.power_he_burn_upper_limit.get()) +
                    "\n")
        if self.stop_at_TP.get():
            f.write("  stop_at_TP = .true.\n")
        if self.log_directory.get() != "":
            f.write("  log_directory = " +
                    self.log_directory.get() + "\n")
        if self.photo_directory.get() != "":
            f.write("  photo_directory = " +
                    self.photo_directory.get() + "\n")

        f.write("  mixing_length_alpha = " + str(self.mixing_length_alpha.get()) + "\n")
        if self.use_Henyey_MLT.get() == 1:
            f.write("  use_Henyey_MLT = .true.\n")
        else:
            f.write("  use_Henyey_MLT = .false.\n")
        f.write("  overshoot_f_above_nonburn = " +
                str(self.overshoot_f_above_nonburn.get()) + "\n")
        f.write("  overshoot_f_below_nonburn = " +
                str(self.overshoot_f_below_nonburn.get()) + "\n")
        f.write("  overshoot_f_above_burn_h = " +
                str(self.overshoot_f_above_burn_h.get()) + "\n")
        f.write("  overshoot_f_below_burn_h = " +
                str(self.overshoot_f_below_burn_h.get()) + "\n")
        f.write("  overshoot_f_above_burn_he = " +
                str(self.overshoot_f_above_burn_he.get()) + "\n")
        f.write("  overshoot_f_below_burn_he = " +
                str(self.overshoot_f_below_burn_he.get()) + "\n")

        f.write("  photostep = " + str(self.photostep.get()) + "\n")
        f.write("  profile_interval = " +
                str(self.profile_interval.get()) + "\n")

        f.write("  RGB_to_AGB_wind_switch = " +
                str(self.RGB_to_AGB_wind_switch.get())
                + "\n")
        f.write("  RGB_wind_scheme = '" + self.RGB_wind_scheme.get() +
                "'\n")
        f.write("  AGB_wind_scheme = '" + self.AGB_wind_scheme.get() +
                "'\n")
        f.write("  Reimers_wind_eta = " +
                str(self.Reimers_wind_eta.get()) + "\n")
        f.write("  Blocker_wind_eta = " +
                str(self.Blocker_wind_eta.get()) + "\n")
        f.write("  super_eddington_wind_eta = " +
                str(self.super_eddington_wind_eta.get()) + "\n")

        if self.report_hydro_dt_info.get()==0:
            f.write("  report_hydro_dt_info = .false.\n")
        else:
            f.write("  report_hydro_dt_info = .true.\n")

        if self.use_artificial_viscosity.get()==0:
            f.write("  use_artificial_viscosity = .false.\n")
        else:
            f.write("  use_artificial_viscosity = .true.\n")
        f.write("  l1_coef = " + str(self.l1_coef.get()) + "\n")
        f.write("  l2_coef = " + str(self.l2_coef.get()) + "\n")

        f.write("  log_cnt = " + str(self.log_cnt.get()) + "\n")
        f.write("  terminal_cnt = " + str(self.terminal_cnt.get()) + "\n")

        f.write("/ ! end of controls namelist\n\n")
 
    def load_controls(self, f):
        key = self.read_keyword(f)
        while key != None:
            value = key[1]
            key = key[0]
            
            if key=="initial_mass":
                self.initial_mass.set(ffloat(value))
            elif key=="initial_z":
                self.Z.set(ffloat(value))
            elif key=="max_age":
                self.max_age.set(value)
            elif key=="max_model_number":
                self.max_model_number.set(int(value))
            elif key=="xa_central_lower_limit_species":
                self.xa_central_lower_limit_species = match_element(value)
            elif key=="xa_central_lower_limit":
                self.xa_central_lower_limit.set(value)
            elif key=="xa_surface_upper_limit_species":
                self.xa_surface_upper_limit_species = match_element(value)
            elif key=="xa_surface_upper_limit":
                self.xa_surface_upper_limit.set(value)
            elif key=="power_h_burn_upper_limit":
                self.power_h_burn_upper_limit.set(ffloat(value))
            elif key=="power_he_burn_upper_limit":
                self.power_he_burn_upper_limit.set(ffloat(value))
            elif key=="stop_at_tp":
                self.stop_at_TP.set(int(value))
            elif key=="log_directory":
                self.log_directory.set(value)
            elif key=="photo_directory":
                self.photo_directory.set(value)
            elif key=="mixing_length_alpha":
                self.mixing_length_alpha.set(ffloat(value))
            elif key=="use_henyey_mlt":
                self.use_Henyey_MLT.set(int(value))
            elif key=="photostep":
                self.photostep.set(int(value))
            elif key=="profile_interval":
                self.profile_interval.set(int(value))
            elif key=="overshoot_f_above_nonburn":
                self.overshoot_f_above_nonburn.set(ffloat(value))
            elif key=="overshoot_f_below_nonburn":
                self.overshoot_f_below_nonburn.set(ffloat(value))
            elif key=="overshoot_f_above_burn_h":
                self.overshoot_f_above_burn_h.set(ffloat(value))
            elif key=="overshoot_f_below_burn_h":
                self.overshoot_f_below_burn_h.set(ffloat(value))
            elif key=="overshoot_f_above_burn_he":
                self.overshoot_f_above_burn_he.set(ffloat(value))
            elif key=="overshoot_f_below_burn_he":
                self.overshoot_f_below_burn_he.set(ffloat(value))
            elif key=="rgb_wind_scheme":
                self.RGB_wind_scheme.set(value.strip("'"))
            elif key=="agb_wind_scheme":
                self.AGB_wind_scheme.set(value.strip("'"))
            elif key=="reimers_wind_eta":
                self.Reimers_wind_eta.set(ffloat(value))
            elif key=="blocker_wind_eta":
                self.Blocker_wind_eta.set(ffloat(value))
            elif key=="super_eddington_wind_eta":
                self.super_eddington_wind_eta.set(ffloat(value))
            elif key=="rgb_to_agb_wind_switch":
                self.RGB_to_AGB_wind_switch.set(ffloat(value))
            elif key=="report_hydro_dt_info":
                self.report_hydro_dt_info.set(int(value))
            elif key=="use_artificial_viscosity":
                self.use_artificial_viscosity.set(int(value))
            elif key=="l1_coef":
                self.l1_coef.set(ffloat(value))
            elif key=="l2_coef":
                self.l2_coef.set(ffloat(value))
            elif key=="log_cnt":
                self.log_cnt.set(int(value))
            elif key=="terminal_cnt":
                self.terminal_cnt.set(int(value))
            else:
                print "warning: unrecognized controls keyword: " + key

            key = self.read_keyword(f)

    def write_pgstar(self, f):
        f.write("&pgstar\n")
        if self.pause.get():
            f.write("  pause = .true.\n")

        i=0
        for window in standard_windows:
            if self.win_flag[i].get():
                f.write("  "+window[0]+"_win_flag = .true.\n")
            else:
                f.write("  "+window[0]+"_win_flag = .false.\n")
            i = i+1

        f.write("/ ! end of pgstar namelist\n")
 
    def load_pgstar(self, f):
        key = self.read_keyword(f)
        while key != None:
            value = key[1]
            key = key[0].lower()
            
            i=0
            for window in standard_windows:
                window = window[0].lower() + "_win_flag"
                if key==window:
                    self.win_flag[i].set(value)
                    break
                i = i+1
                
            key = self.read_keyword(f)
