! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module turbulent_diffusion

      use const_def
      use num_lib
      use utils_lib
      use star_private_def
      use star_utils, only: zero_D_mix_partials

      implicit none

      
      contains

      
      ! Morel, P., and Thevenin, F.,
      ! Atomic diffusion in stellar models of type earlier than G.,
      ! A&A, 390:611-620 (2002)
      
      subroutine add_radiation_turbulence(s, ierr)
         use mlt_def
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: nz, k
         real(dp) :: coeff, alfa, beta, T_face, rho_face, opacity_face, D, &
            P_face, r_face, q_face, cdc, Hp, vc
         logical :: dbg
         
         include 'formats'
         
         dbg = .false.
         
         ierr = 0
         coeff = s% radiation_turbulence_coeff
         if (coeff <= 0) return
         nz = s% nz
         
         do k = 2, nz
            if (s% mixing_type(k) /= no_mixing) cycle
            alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
            beta = 1 - alfa
            T_face = alfa*s% T(k) + beta*s% T(k-1)
            rho_face = alfa*s% rho(k) + beta*s% rho(k-1)
            opacity_face = alfa*s% opacity(k) + beta*s% opacity(k-1)         
            D = coeff*4*crad*T_face*T_face*T_face*T_face/(15*clight*opacity_face*rho_face*rho_face)
            if (D < s% D_mix_ov_limit) exit
            P_face = alfa*s% P(k) + beta*s% P(k-1)
            r_face = s% r(k)
            q_face = s% q(k)
            cdc = pow2(pi4*s% r(k)*s% r(k)*rho_face)*D ! gm^2/sec
            Hp = P_face/(rho_face*s% cgrav(k)*(s% M_center + s% xmstar*q_face)/(r_face*r_face))
            vc = 3*D/(s% mixing_length_alpha*Hp)
            s% cdc(k) = cdc
            s% D_mix(k) = D
            call zero_D_mix_partials(s,k)
            s% conv_vel(k) = vc
            s% mixing_type(k) = anonymous_mixing
         end do
         
         
      end subroutine add_radiation_turbulence

      
      ! Proffitt, C.R., and Michaud, G.,
      ! GRAVITATIONAL SETTLING IN SOLAR MODELS,
      ! ApJ, 380:238-290, 1991.
      
      ! turbulent mixing below envelope convection zone; similar to overshooting.
      
      subroutine add_turbulent_diffusion(s, ierr)
         use mlt_def
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: i, loc, n_conv_bdy, nz
         logical :: dbg
         
         include 'formats'
         
         dbg = .false.
         
         ierr = 0
         if (s% turbulent_diffusion_D0 <= 0) return
              
         nz = s% nz
         n_conv_bdy = s% num_conv_boundaries
         
         do i=1, n_conv_bdy
            
            loc = s% conv_bdy_loc(i)
            if (loc < 1 .or. loc > nz) then
               write(*,*) 'bad s% conv_bdy_loc(i)', i, s% conv_bdy_loc(i)
               ierr = -1
               return
            end if
            
            if (s% top_conv_bdy(i)) cycle
            
            if (s% mixing_type(loc-1) == anonymous_mixing) then
               call do_turbulent_diff(ierr)
               if (ierr /= 0) return
            end if

            if (dbg) write(*,*)
            
         end do
                           
         
         contains
         
         
         subroutine do_turbulent_diff(ierr)
            integer, intent(out) :: ierr
            
            real(dp) :: rho_cz_base, alfa, beta, rho_face, Hp, D, cdc, vc, &
               P_face, r_face, q_face
            integer :: k
            logical :: dbg
            include 'formats'


            dbg = .false. ! r_edge < 0.7*Rsun .and. r_edge > 0.6*Rsun

            ierr = 0
            call get_rho_edge(loc-1, rho_cz_base, ierr)
            if (ierr /= 0) return
            
            if (rho_cz_base > s% turbulent_diffusion_rho_max) return
            
            do k=loc, nz
               
               alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
               beta = 1 - alfa
               rho_face = alfa*s% rho(k) + beta*s% rho(k-1)
               D = s% turbulent_diffusion_D0*pow3(rho_cz_base/rho_face)
               
               if (D < s% D_mix_ov_limit) exit
               
               P_face = alfa*s% P(k) + beta*s% P(k-1)
               r_face = s% r(k)
               q_face = s% q(k)
               cdc = pow2(pi4*s% r(k)*s% r(k)*rho_face)*D ! gm^2/sec
               Hp = P_face/(rho_face*s% cgrav(k)*(s% M_center + s% xmstar*q_face)/(r_face*r_face))
               vc = 3*D/(s% mixing_length_alpha*Hp)
               
               if (s% mixing_type(k) == convective_mixing) then 
                  ! leave it convective type, but change mixing coeff
                  s% cdc(k) = cdc
                  s% D_mix(k) = D
                  s% conv_vel(k) = vc
               else if (s% mixing_type(k) == no_mixing) then ! change to overshooting
                  s% cdc(k) = cdc
                  s% D_mix(k) = D
                  s% conv_vel(k) = vc
                  s% mixing_type(k) = overshoot_mixing
               else if (s% mixing_type(k) == overshoot_mixing) then
                  ! have encountered overshooting from another cz; add effects
                  s% cdc(k) = s% cdc(k) + cdc
                  s% D_mix(k) = s% D_mix(k) + D
                  s% conv_vel(k) = s% conv_vel(k) + vc
               else ! have run into a different mixing region
                  exit
               end if
               call zero_D_mix_partials(s,k)
               
            end do

         end subroutine do_turbulent_diff
         
         
         subroutine get_rho_edge(k00, rho_edge, ierr)
            integer, intent(in) :: k00 
            real(dp), intent(out) :: rho_edge
            integer, intent(out) :: ierr         
            real(dp) :: dq, ddq, dqfrac, x, x0, x1, x2, a0, a1, a2, lnd_edge       
            integer :: km1, kp1, kk         
            include 'formats'
            ierr = 0
            km1 = k00-1
            kp1 = k00+1         
            if (kp1 > nz) return
            dq = s% dq(k00)
            ddq = s% cz_bdy_dq(k00)
            dqfrac = ddq/dq ! fractional distance from k00 to kp1 for edge of convection         
            x0 = s% dq(km1)/2
            x1 = s% dq(km1) + s% dq(k00)/2
            x2 = s% dq(km1) + s% dq(k00) + s% dq(kp1)/2
            x = s% dq(km1) + ddq
            call two_piece_linear_coeffs(x, x0, x1, x2, a0, a1, a2, ierr)
            if (ierr /= 0) return        
            lnd_edge = a0*s% lnd(km1) + a1*s% lnd(k00) + a2*s% lnd(kp1)
            rho_edge = exp_cr(lnd_edge)                     
         end subroutine get_rho_edge

         
      end subroutine add_turbulent_diffusion
         
         


      
      
      end module turbulent_diffusion
