! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module timestep
      
      use star_private_def
      use const_def
      use utils_lib, only:is_bad_num

      implicit none

      logical, parameter :: dbg_timestep = .false.
      real(dp) :: max_dt
      integer :: why_Tlim

      contains


      integer function timestep_controller(s, max_timestep)
         ! if don't return keep_going, then set result_reason to say why.
         type (star_info), pointer :: s
         real(dp), intent(in) :: max_timestep
         
         include 'formats'
                  
         timestep_controller = do_timestep_limits(s, s% dt)
         if (timestep_controller /= keep_going) s% result_reason = timestep_limits
         
         max_dt = max_timestep
         if (max_dt <= 0) max_dt = 1d99
         if (s% dt_next > max_dt) then
            s% dt_next = max_dt
            s% why_Tlim = Tlim_max_timestep
            if (s% ebdf_order > s% min_ebdf_order) then
               write(*,2) 'reduce ebdf_order since at max timestep'
               s% ebdf_order = s% ebdf_order - 1
               s% ebdf_hold = s% model_number + &
                  max(1, s% steps_before_try_higher_ebdf_order)
               s% startup_increment_ebdf_order = .false.
            end if
         end if
         
         ! don't reduce dt by more than would for backup
         s% dt_next = max(s% dt_next, s% dt*s% timestep_factor_for_backups)
         
         if (s% timestep_hold >= s% model_number .and. s% dt_next > s% dt) then
            s% dt_next = s% dt
            s% why_Tlim = Tlim_timestep_hold
            if (s% report_why_dt_limits .and. timestep_controller == keep_going) &
               write(*,3) 'timestep_hold >= model_number, so no timestep increase', &
                  s% timestep_hold, s% model_number
         end if
         
         if (is_bad_num(s% dt_next)) then
            write(*, *) 'timestep_controller: dt_next', s% dt_next
            timestep_controller = terminate
            s% termination_code = t_timestep_controller
            return
         end if
         
         if (dbg_timestep) then
            write(*,*) 'final result from timestep_controller for model_number', &
               timestep_controller, s% model_number
            write(*,1) 'lg dt/secyer', log10_cr(s% dt/secyer)
            write(*,1) 'lg dt_next/secyer', log10_cr(s% dt_next/secyer)
            write(*,1) 'dt_next/dt', s% dt_next/s% dt
            write(*,*)
            write(*,*)
            stop 'timestep_controller'
         end if
         
      end function timestep_controller
      
      
      integer function do_timestep_limits(s, dt)
         use rates_def, only: i_rate
         use chem_def, only: chem_isos, ipp, icno, category_name
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt ! timestep just completed

         real(dp) :: dt_limit_ratio(numTlim), dt_limit, limit, order
         integer :: i_limit, nz, ierr
         logical :: skip_hard_limit
         integer :: num_mix_boundaries ! boundaries of regions with mixing_type /= no_mixing
         real(dp), pointer :: mix_bdy_q(:) ! (num_mix_boundaries)
         integer, pointer :: mix_bdy_loc(:) ! (num_mix_boundaries)

         
         include 'formats'
                  
         skip_hard_limit = ((s% relax_hard_limits_after_backup) .and. &
                           (s% timestep_hold >= s% model_number)) .or. &
                           ((s% relax_hard_limits_after_retry) .and. &
                           (s% model_number_for_last_retry == s% model_number)) 
         !skip_hard_limit = .false.
         
         nz = s% nz
         
         ! NOTE: when we get here, complete_model has called the report routine, 
         ! so we can use information that it has calculated

         ierr = 0
         
         num_mix_boundaries = s% num_mix_boundaries
         mix_bdy_q => s% mix_bdy_q
         mix_bdy_loc => s% mix_bdy_loc
                  
         dt_limit_ratio(:) = 0d0
                  
         if (s% ebdf_order > 0 .and. s% use_truncation_ratio_limit) then
            do_timestep_limits = check_truncation_ratio( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_truncation_ratio))
            if (return_now(Tlim_truncation_ratio)) return
         else
            do_timestep_limits = check_varcontrol_limit( &
               s, dt_limit_ratio(Tlim_struc))
            if (return_now(Tlim_struc)) return
         end if

         if (.not. s% doing_first_model_of_run) then

            do_timestep_limits = check_newton_iterations_limit( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_num_newton_iterations))
            if (return_now(Tlim_num_newton_iterations)) return
         
            do_timestep_limits = check_rotation_steps_limit( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_num_rotation_steps))
            if (return_now(Tlim_num_rotation_steps)) return
         
            do_timestep_limits = check_diffusion_steps_limit( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_num_diff_solver_steps))
            if (return_now(Tlim_num_diff_solver_steps)) return
         
            do_timestep_limits = check_diffusion_iters_limit( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_num_diff_solver_iters))
            if (return_now(Tlim_num_diff_solver_iters)) return
         
            do_timestep_limits = check_v_div_v_crit( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_v_div_v_crit))
            if (return_now(Tlim_v_div_v_crit)) return
         
            do_timestep_limits = check_max_fixup_for_mix( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_max_mix_fixup))
            if (return_now(Tlim_max_mix_fixup)) return
         
            do_timestep_limits = check_burn_max_iters_limit( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_burn_max_num_iters))
            if (return_now(Tlim_burn_max_num_iters)) return
         
            do_timestep_limits = check_burn_max_substeps_limit( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_burn_max_num_substeps))
            if (return_now(Tlim_burn_max_num_substeps)) return
         
            do_timestep_limits = check_dX(s, 0, skip_hard_limit, dt, &
               num_mix_boundaries, mix_bdy_loc, mix_bdy_q, &
               dt_limit_ratio(Tlim_dH), dt_limit_ratio(Tlim_dH_div_H))
            if (return_now(0)) return
         
            do_timestep_limits = check_dX(s, 1, skip_hard_limit, dt, &
               num_mix_boundaries, mix_bdy_loc, mix_bdy_q, &
               dt_limit_ratio(Tlim_dHe), dt_limit_ratio(Tlim_dHe_div_He))
            if (return_now(0)) return
         
            do_timestep_limits = check_dX(s, 2, skip_hard_limit, dt, &
               num_mix_boundaries, mix_bdy_loc, mix_bdy_q, &
               dt_limit_ratio(Tlim_dHe3), dt_limit_ratio(Tlim_dHe3_div_He3))
            if (return_now(0)) return
         
            do_timestep_limits = check_dX(s, -1, skip_hard_limit, dt, &
               num_mix_boundaries, mix_bdy_loc, mix_bdy_q, &
               dt_limit_ratio(Tlim_dX), dt_limit_ratio(Tlim_dX_div_X))
            if (return_now(0)) return

            do_timestep_limits = check_dL_div_L( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dL_div_L))
            if (return_now(Tlim_dL_div_L)) return
         
            do_timestep_limits = check_dlgP_change( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_dlgP))
            if (return_now(Tlim_dlgP)) return
         
            do_timestep_limits = check_dlgRho_change( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_dlgRho))
            if (return_now(Tlim_dlgRho)) return
         
            do_timestep_limits = check_dlgT_change( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_dlgT))
            if (return_now(Tlim_dlgT)) return
         
            do_timestep_limits = check_dlgE_change( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_dlgE))
            if (return_now(Tlim_dlgE)) return
         
            do_timestep_limits = check_dlgR_change( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_dlgR))
            if (return_now(Tlim_dlgR)) return
         
            do_timestep_limits = check_d_deltaR_shrink( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_d_deltaR_shrink))
            if (return_now(Tlim_d_deltaR_shrink)) return
         
            do_timestep_limits = check_d_deltaR_grow( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_d_deltaR_grow))
            if (return_now(Tlim_d_deltaR_grow)) return
         
            do_timestep_limits = check_lgL_nuc_cat_change( &
               s, num_mix_boundaries, mix_bdy_q, &
               skip_hard_limit, dt_limit_ratio(Tlim_dlgL_nuc_cat))
            if (return_now(Tlim_dlgL_nuc_cat)) return

            do_timestep_limits = check_lgL_H_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgL_H))
            if (return_now(Tlim_dlgL_H)) return
         
            do_timestep_limits = check_lgL_He_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgL_He))
            if (return_now(Tlim_dlgL_He)) return
         
            do_timestep_limits = check_lgL_z_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgL_z))
            if (return_now(Tlim_dlgL_z)) return
         
            do_timestep_limits = check_lgL_photo_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgL_photo))
            if (return_now(Tlim_dlgL_photo)) return

            do_timestep_limits = check_lgL_nuc_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgL_nuc))
            if (return_now(Tlim_dlgL_nuc)) return
         
            do_timestep_limits = check_dlgTeff_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgTeff))
            if (return_now(Tlim_dlgTeff)) return
         
            do_timestep_limits = check_dlgRho_cntr_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgRho_cntr))
            if (return_now(Tlim_dlgRho_cntr)) return
         
            do_timestep_limits = check_dlgT_cntr_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgT_cntr))
            if (return_now(Tlim_dlgT_cntr)) return  
         
            do_timestep_limits = check_dlgT_max_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgT_max))
            if (return_now(Tlim_dlgT_max)) return  
         
            do_timestep_limits = check_dlgRho_max_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlgRho_max))
            if (return_now(Tlim_dlgRho_max)) return  
         
            do_timestep_limits = check_dlog_eps_nuc_cntr_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlog_eps_nuc_cntr))
            if (return_now(Tlim_dlog_eps_nuc_cntr)) return   

            do_timestep_limits = check_dYe_change( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_delta_Ye)) 
            if (return_now(Tlim_delta_Ye)) return

            do_timestep_limits = check_dYe_highT_change( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_delta_Ye_highT)) 
            if (return_now(Tlim_delta_Ye_highT)) return
         
            do_timestep_limits = check_dlog_eps_nuc_change( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dlog_eps_nuc))
            if (return_now(Tlim_dlog_eps_nuc)) return                  
                  
            do_timestep_limits = check_lg_XH_cntr( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_lg_XH_cntr))
            if (return_now(Tlim_lg_XH_cntr)) return
         
            do_timestep_limits = check_lg_XHe_cntr( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_lg_XHe_cntr))
            if (return_now(Tlim_lg_XHe_cntr)) return
         
            do_timestep_limits = check_lg_XC_cntr( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_lg_XC_cntr))
            if (return_now(Tlim_lg_XC_cntr)) return
         
            do_timestep_limits = check_lg_XNe_cntr( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_lg_XNe_cntr))
            if (return_now(Tlim_lg_XNe_cntr)) return
         
            do_timestep_limits = check_lg_XO_cntr( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_lg_XO_cntr))
            if (return_now(Tlim_lg_XO_cntr)) return
         
            do_timestep_limits = check_lg_XSi_cntr( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_lg_XSi_cntr))
            if (return_now(Tlim_lg_XSi_cntr)) return

            do_timestep_limits = check_delta_mstar( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dmstar))
            if (return_now(Tlim_dmstar)) return

            do_timestep_limits = check_delta_mdot( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_del_mdot))
            if (return_now(Tlim_del_mdot)) return

            do_timestep_limits = check_delta_ang_momentum( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_deltaJ))
            if (return_now(Tlim_deltaJ)) return

            do_timestep_limits = check_CpT_absMdot_div_L( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_CpT_absMdot_div_L))
            if (return_now(Tlim_CpT_absMdot_div_L)) return
         
            do_timestep_limits = check_delta_lgL( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_lgL))
            if (return_now(Tlim_lgL)) return
         
            do_timestep_limits = check_dt_div_dt_cell_collapse( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dt_div_dt_cell_collapse))
            if (return_now(Tlim_dt_div_dt_cell_collapse)) return
         
            do_timestep_limits = check_dt_div_dt_Courant( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dt_div_dt_Courant))
            if (return_now(Tlim_dt_div_dt_Courant)) return
         
            do_timestep_limits = check_dt_div_dt_thermal( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dt_div_dt_thermal))
            if (return_now(Tlim_dt_div_dt_thermal)) return
         
            do_timestep_limits = check_dt_div_dt_dynamic( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dt_div_dt_dynamic))
            if (return_now(Tlim_dt_div_dt_dynamic)) return
         
            do_timestep_limits = check_dt_div_dt_acoustic( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dt_div_dt_acoustic))
            if (return_now(Tlim_dt_div_dt_acoustic)) return
         
            do_timestep_limits = check_dt_div_dt_mass_loss( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dt_div_dt_mass_loss))
            if (return_now(Tlim_dt_div_dt_mass_loss)) return
         
            do_timestep_limits = check_delta_HR( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_delta_HR))
            if (return_now(Tlim_delta_HR)) return
         
            do_timestep_limits = check_rel_error_in_energy( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_error_in_energy_conservation))
            if (return_now(Tlim_error_in_energy_conservation)) return
         
            do_timestep_limits = check_rel_rate_in_energy( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_error_rate_energy_conservation))
            if (return_now(Tlim_error_rate_energy_conservation)) return
         
            do_timestep_limits = check_avg_E_residual( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_avg_E_residual))
            if (return_now(Tlim_avg_E_residual)) return
         
            do_timestep_limits = check_max_E_residual( &
               s, skip_hard_limit, dt_limit_ratio(Tlim_max_E_residual))
            if (return_now(Tlim_max_E_residual)) return
         
            do_timestep_limits = check_dX_nuc_drop( &
               s, skip_hard_limit, dt, dt_limit_ratio(Tlim_dX_nuc_drop))
            if (return_now(Tlim_dX_nuc_drop)) return
         
         end if

         i_limit = maxloc(dt_limit_ratio(1:numTlim), dim=1)

         if (s% ebdf_order > s% min_ebdf_order .and. &
             s% ebdf_order > 0 .and. &
               i_limit /= Tlim_truncation_ratio) then
            write(*,2) 'reduce ebdf_order since other limit on timestep'
            s% ebdf_order = s% ebdf_order - 1
            s% ebdf_hold = s% model_number + &
               max(1, s% steps_before_try_higher_ebdf_order)
            s% startup_increment_ebdf_order = .false.
         end if

         order = max(1,s% ebdf_order)
         call filter_dt_next(s, order, dt_limit_ratio(i_limit)) ! sets s% dt_next

         if (s% max_timestep_factor > 0 .and. s% dt_next > s% max_timestep_factor*s% dt) then
            s% dt_next = s% max_timestep_factor*s% dt
            if (i_limit == Tlim_struc) i_limit = Tlim_max_timestep_factor
         end if

         if (s% min_timestep_factor > 0 .and. s% dt_next < s% min_timestep_factor*s% dt) then
            s% dt_next = s% min_timestep_factor*s% dt
            if (i_limit == Tlim_struc) i_limit = Tlim_min_timestep_factor
         end if
         
         s% why_Tlim = i_limit
         
         if (s% dt_next < s% dt .and. s% report_why_dt_limits) call report_why
         
         if (i_limit == Tlim_dX_nuc_drop .and. s% report_dX_nuc_drop_dt_limits) then
            limit = s% dX_nuc_drop_limit
            if (s% lnT(nz) >= ln10*9.45d0 .and. s% dX_nuc_drop_limit_at_high_T > 0) &
               limit = s% dX_nuc_drop_limit_at_high_T
            write(*,'(a70,a15,i8,99f12.6)') &
               'dt dX_nuc_drop iso, k, dX/dX_limit, dX, dX_limit, m(k)/Msun', &
                  trim(chem_isos% name(s% chem_id(s% dX_nuc_drop_max_j))), &
               s% dX_nuc_drop_max_k, dt_limit_ratio(Tlim_dX_nuc_drop), &
               s% dX_nuc_drop_max_drop, limit, s% m(s% dX_nuc_drop_max_k)/Msun
         end if
         
         
         contains
         
         
         subroutine report_why
            integer :: k
            real(dp) :: dt_next, max_dln
            dt_next = s% dt_next
            if (i_limit == Tlim_dX) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt: large dX: iso, dX, model, mass loc  logdt/yr' // &
                  trim(chem_isos% name(s% chem_id(s% Tlim_dX_species))), &
                  s% model_number, s% Tlim_dX_cell, s% star_mass*s% q(s% Tlim_dX_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dlgRho) then
               call get_dlgRho_info(s, k, max_dln)
               if (k > 0) write(*, '(a, i8, i5, 99f16.9)') &
                  'reduce dt because of dlgRho', &
                  s% model_number, k, max_dln/ln10, s% lnd(k)/ln10, s% m(k)/Msun
            else if (i_limit == Tlim_dlgT) then
               call get_dlgT_info(s, k, max_dln)
               if (k > 0) write(*, '(a, i8, i5, 99f16.9)') &
                  'reduce dt because of dlgT', &
                  s% model_number, k, max_dln/ln10, s% lnT(k)/ln10, s% m(k)/Msun
            else if (i_limit == Tlim_dlgE) then
               call get_dlgE_info(s, k, max_dln)
               if (k > 0) write(*, '(a, i8, i5, 99f16.9)') &
                  'reduce dt because of dlgE', &
                  s% model_number, k, max_dln/ln10, s% lnE(k)/ln10, s% m(k)/Msun
            else if (i_limit == Tlim_dX_nuc_drop) then
               write(*, '(a, i8, i5, 99f16.9)') &
                  'reduce dt because of dX_nuc_drop  ' // &
                  trim(chem_isos% name(s% chem_id(s% Tlim_dXnuc_drop_species))), &
                  s% model_number, s% Tlim_dXnuc_drop_cell, &
                  s% xa(s% Tlim_dXnuc_drop_species, s% Tlim_dXnuc_drop_cell) - &
                     s% xa_compare(s% Tlim_dXnuc_drop_species, s% Tlim_dXnuc_drop_cell), &
                  s% xa(s% Tlim_dXnuc_drop_species, s% Tlim_dXnuc_drop_cell), &
                  s% xa_compare(s% Tlim_dXnuc_drop_species, s% Tlim_dXnuc_drop_cell), &
                  s% star_mass*s% q(s% Tlim_dXnuc_drop_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dX_div_X) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dX_div_X  ' // &
                  trim(chem_isos% name(s% chem_id(s% Tlim_dX_div_X_species))), &
                  s% model_number, s% Tlim_dX_div_X_cell, s% star_mass*s% q(s% Tlim_dX_div_X_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dH) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dH', &
                  s% model_number, s% Tlim_dX_cell, s% star_mass*s% q(s% Tlim_dX_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dH_div_H) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dH_div_H', &
                  s% model_number, s% Tlim_dX_div_X_cell, s% star_mass*s% q(s% Tlim_dX_div_X_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dHe) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dHe', &
                  s% model_number, s% Tlim_dX_cell, s% star_mass*s% q(s% Tlim_dX_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dHe_div_He) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dHe_div_He', &
                  s% model_number, s% Tlim_dX_div_X_cell, s% star_mass*s% q(s% Tlim_dX_div_X_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dHe3) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dHe3', &
                  s% model_number, s% Tlim_dX_cell, s% star_mass*s% q(s% Tlim_dX_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dHe3_div_He3) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dHe3_div_He3', &
                  s% model_number, s% Tlim_dX_div_X_cell, s% star_mass*s% q(s% Tlim_dX_div_X_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dlgL_nuc_cat) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dlgL ' // &
                  trim(category_name(s% Tlim_dlgL_nuc_category)), &
                  s% model_number, s% Tlim_dlgL_nuc_cell, s% star_mass*s% q(s% Tlim_dlgL_nuc_cell), &
                  log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_neg_X) then
               write(*, '(a, i8, f16.9)') &
                  'reduce dt because of negative mass fraction', &
                  s% model_number, log10_cr(dt_next/secyer)
            else if (i_limit == Tlim_dt_div_dt_Courant) then
               write(*, '(a, i8, i5, 2f16.9)') &
                  'reduce dt because of dt/dt_Courant limit', &
                  s% model_number, s% Tlim_dt_div_dt_Courant_cell, &
                  s% star_mass*s% q(s% Tlim_dt_div_dt_Courant_cell), &
                  log10_cr(dt_next)
            else if (i_limit == Tlim_bad_Xsum) then
               write(*, '(a, i8, f16.9)') &
                  'reduce dt because of bad sum of mass fractions', &
                  s% model_number, log10_cr(dt_next/secyer)
            else
               write(*, '(a, i8)') 'reduce dt because of ' // &
                  trim(dt_why_str(i_limit)), s% model_number
            end if
         end subroutine report_why
         
         
         logical function return_now(i_limit)
            integer, intent(in) :: i_limit
            integer :: k
            real(dp) :: max_dln
            
            if (do_timestep_limits == keep_going) then
               return_now = .false.
               return
            end if
            
            if (i_limit == Tlim_dX) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of dX  ' // &
                  trim(chem_isos% name(s% chem_id(s% Tlim_dX_species))), &
                  s% model_number, s% Tlim_dX_cell
            else if (i_limit == Tlim_dlgRho) then
               call get_dlgRho_info(s, k, max_dln)
               if (k > 0) write(s% retry_message, '(a, i8, i5, 99f16.9)') &
                  'retry because of dlgRho', &
                  s% model_number, k, max_dln/ln10, s% lnd(k)/ln10, s% m(k)/Msun
            else if (i_limit == Tlim_dlgT) then
               call get_dlgT_info(s, k, max_dln)
               if (k > 0) write(s% retry_message, '(a, i8, i5, 99f16.9)') &
                  'retry because of dlgT', &
                  s% model_number, k, max_dln/ln10, s% lnT(k)/ln10, s% m(k)/Msun
            else if (i_limit == Tlim_dlgE) then
               call get_dlgE_info(s, k, max_dln)
               if (k > 0) write(s% retry_message, '(a, i8, i5, 99f16.9)') &
                  'retry because of dlgE', &
                  s% model_number, k, max_dln/ln10, s% lnE(k)/ln10, s% m(k)/Msun
            else if (i_limit == Tlim_dX_div_X) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of dX_div_X  ' // &
                  trim(chem_isos% name(s% chem_id(s% Tlim_dX_div_X_species))), &
                  s% model_number, s% Tlim_dX_div_X_cell
            else if (i_limit == Tlim_dH) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of dH', &
                  s% model_number, s% Tlim_dX_cell
            else if (i_limit == Tlim_dH_div_H) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of dH_div_H', &
                  s% model_number, s% Tlim_dX_div_X_cell, s% model_number
            else if (i_limit == Tlim_dHe) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of dHe', &
                  s% model_number, s% Tlim_dX_cell, s% model_number
            else if (i_limit == Tlim_dHe_div_He) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of dHe_div_He', &
                  s% model_number, s% Tlim_dX_div_X_cell, s% model_number
            else if (i_limit == Tlim_dHe3) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of dHe3', &
                  s% model_number, s% Tlim_dX_cell, s% model_number
            else if (i_limit == Tlim_dHe3_div_He3) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of dHe3_div_He3', &
                  s% model_number, s% Tlim_dX_div_X_cell, s% model_number
            else if (i_limit == Tlim_dlgL_nuc_cat) then
               write(s% retry_message, '(a, i8, i5)') &
                  'retry because of change in lgL ' // &
                  trim(category_name(s% Tlim_dlgL_nuc_category)), &
                  s% model_number, s% Tlim_dlgL_nuc_cell, s% model_number
            else
               write(s% retry_message, '(a, i8)') 'retry because of timestep limit ' // &
                  trim(dt_why_str(i_limit)), s% model_number
            end if
            
            if (i_limit > 0) s% why_Tlim = i_limit
            return_now = .true.
            
         end function return_now
         
         
      end function do_timestep_limits
      
      
      integer function check_integer_limit( &
            s, limit, hard_limit, value, msg, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         integer, intent(in) :: limit, hard_limit, value
         character (len=*), intent(in) :: msg
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         if (value > hard_limit .and. hard_limit > 0 .and. (.not. skip_hard_limit)) then
            if (s% report_all_dt_limits) write(*,*) trim(msg) // ' hard limit', hard_limit, value
            check_integer_limit = retry
            return
         end if
         check_integer_limit = keep_going
         if (value <= 0 .or. limit <= 0) return
         dt_limit_ratio = dble(value)/dble(limit) - 0.05d0
            ! subtract a bit so that allow dt to grow even if value == limit
         !dt_limit_ratio = 2d0*dble(value)/dble(limit + value) 
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*,'(a30,f20.10,99i10)') &
               trim(msg) // ' dt_limit_ratio', dt_limit_ratio, value, limit
         end if
      end function check_integer_limit
         
            
      integer function check_rotation_steps_limit(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         if (.not. s% rotation_flag) then
            check_rotation_steps_limit = keep_going; return
         end if
         check_rotation_steps_limit = check_integer_limit( &
           s, s% rotation_steps_limit, s% rotation_steps_hard_limit, &
           s% num_rotation_solver_steps,  &
           'num_rotation_solver_steps', skip_hard_limit, dt, dt_limit_ratio)
      end function check_rotation_steps_limit
      
      
      integer function check_newton_iterations_limit(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         integer :: iters
         iters = s% num_newton_iterations
         if (s% used_extra_iter_in_newton_for_accretion) iters = iters - 1
         check_newton_iterations_limit = check_integer_limit( &
           s, s% newton_iterations_limit, s% newton_iterations_hard_limit, &
           iters, 'num_newton_iterations', skip_hard_limit, dt, dt_limit_ratio)
      end function check_newton_iterations_limit
         
            
      integer function check_diffusion_steps_limit(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         if (.not. s% do_element_diffusion) then
            check_diffusion_steps_limit = keep_going; return
         end if
         check_diffusion_steps_limit = check_integer_limit( &
           s, s% diffusion_steps_limit, s% diffusion_steps_hard_limit, &
           s% num_diffusion_solver_steps,  &
           'num_diffusion_solver_steps', skip_hard_limit, dt, dt_limit_ratio)
      end function check_diffusion_steps_limit
         
            
      integer function check_diffusion_iters_limit(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         if (.not. s% do_element_diffusion) then
            check_diffusion_iters_limit = keep_going; return
         end if
         check_diffusion_iters_limit = check_integer_limit( &
           s, s% diffusion_iters_limit, s% diffusion_iters_hard_limit, &
           s% num_diffusion_solver_iters,  &
           'num_diffusion_solver_iters', skip_hard_limit, dt, dt_limit_ratio)
      end function check_diffusion_iters_limit


      integer function check_burn_max_iters_limit( &
            s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         include 'formats'
         if (s% split_mixing_choice >= 0 .or. &
             s% split_mix_do_burn) then
            check_burn_max_iters_limit = keep_going; return
         end if
         check_burn_max_iters_limit = check_integer_limit( &
           s, s% burn_max_iters_limit, s% burn_max_iters_hard_limit, &
           maxval(s% burn_num_iters(1:s% nz)),  &
           'burn_max_num_iters', skip_hard_limit, dt, dt_limit_ratio)
      end function check_burn_max_iters_limit


      integer function check_burn_max_substeps_limit( &
            s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         include 'formats'
         if (s% split_mixing_choice >= 0 .or. &
             s% split_mix_do_burn) then
            check_burn_max_substeps_limit = keep_going; return
         end if
         check_burn_max_substeps_limit = check_integer_limit( &
           s, s% burn_max_substeps_limit, s% burn_max_substeps_hard_limit, &
           maxval(s% burn_num_substeps(1:s% nz)),  &
           'burn_max_num_substeps', skip_hard_limit, dt, dt_limit_ratio)
      end function check_burn_max_substeps_limit
         

      integer function check_dX(s, which, skip_hard_limit, dt, &
            n_mix_bdy, mix_bdy_loc, mix_bdy_q, &
            dX_dt_limit_ratio, dX_div_X_dt_limit_ratio)
         use chem_def
         use num_lib, only: binary_search
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         integer, intent(in) :: which, n_mix_bdy, mix_bdy_loc(:)
         real(dp), intent(in) :: dt
         real(dp), intent(in), pointer :: mix_bdy_q(:)
         real(dp), intent(inout) :: dX_dt_limit_ratio, dX_div_X_dt_limit_ratio
         
         real(dp) :: X, X_old, delta_dX, delta_dX_div_X, max_dX, max_dX_div_X, &
            bdy_dist_dm, max_dX_bdy_dist_dm, max_dX_div_X_bdy_dist_dm, cz_dist_limit
         integer :: j, k, cid, bdy, max_dX_j, max_dX_k, max_dX_div_X_j, max_dX_div_X_k
         real(dp) :: D_mix_cutoff, dX_limit_min_X, dX_limit, dX_hard_limit
         real(dp) :: dX_div_X_limit_min_X, dX_div_X_limit, dX_div_X_hard_limit
         logical :: decreases_only
         
         include 'formats'
         
         check_dX = keep_going
         
         if (which == 0) then ! hydrogen
            dX_limit_min_X = s% dH_limit_min_H
            dX_limit = s% dH_limit
            dX_hard_limit = s% dH_hard_limit
            dX_div_X_limit_min_X = s% dH_div_H_limit_min_H
            dX_div_X_limit = s% dH_div_H_limit
            dX_div_X_hard_limit = s% dH_div_H_hard_limit
            decreases_only = s% dH_decreases_only
         else if (which == 1) then ! helium
            dX_limit_min_X = s% dHe_limit_min_He
            dX_limit = s% dHe_limit
            dX_hard_limit = s% dHe_hard_limit
            dX_div_X_limit_min_X = s% dHe_div_He_limit_min_He
            dX_div_X_limit = s% dHe_div_He_limit
            dX_div_X_hard_limit = s% dHe_div_He_hard_limit
            decreases_only = s% dHe_decreases_only
         else if (which == 2) then ! He3
            dX_limit_min_X = s% dHe3_limit_min_He3
            dX_limit = s% dHe3_limit
            dX_hard_limit = s% dHe3_hard_limit
            dX_div_X_limit_min_X = s% dHe3_div_He3_limit_min_He3
            dX_div_X_limit = s% dHe3_div_He3_limit
            dX_div_X_hard_limit = s% dHe3_div_He3_hard_limit
            decreases_only = s% dHe3_decreases_only
         else ! metals
            dX_limit_min_X = s% dX_limit_min_X
            dX_limit = s% dX_limit
            dX_hard_limit = s% dX_hard_limit
            dX_div_X_limit_min_X = s% dX_div_X_limit_min_X
            dX_div_X_limit = s% dX_div_X_limit
            dX_div_X_hard_limit = s% dX_div_X_hard_limit
            decreases_only = s% dX_decreases_only
         end if
         
         if (  dX_limit_min_X >= 1 .and. &
               dX_limit >= 1 .and. &
               dX_hard_limit >= 1 .and. &
               dX_div_X_limit_min_X >= 1 .and. &
               dX_div_X_limit >= 1 .and. &
               dX_div_X_hard_limit >= 1) then
            if (.false. .and. which == 2) then ! He3
               write(*,1) 'dX_limit', dX_limit
               write(*,1) 'dX_div_X_hard_limit', dX_div_X_hard_limit
               write(*,1) 'dX_limit_min_X', dX_limit_min_X
               write(*,1) 'dX_div_X_limit_min_X', dX_div_X_limit_min_X
            end if
            return
         end if
         
         max_dX = -1; max_dX_j = -1; max_dX_k = -1
         max_dX_div_X = -1; max_dX_div_X_j = -1; max_dX_div_X_k = -1
         bdy = 0
         max_dX_bdy_dist_dm = 0
         max_dX_div_X_bdy_dist_dm = 0
         cz_dist_limit = s% dX_mix_dist_limit*Msun

         if (s% set_min_D_mix .and. s% ye(s% nz) >= s% min_center_Ye_for_min_D_mix) then
            D_mix_cutoff = s% min_D_mix
         else
            D_mix_cutoff = 0
         end if
         
         do k = 1, s% nz
         
            if (s% D_mix(k) > D_mix_cutoff) then
               cycle
            end if
            if (k < s% nz) then
               if (s% D_mix(k+1) > D_mix_cutoff) then
                  cycle
               end if
            end if
            if (s% newly_nonconvective(k)) then
               cycle
            end if
            
            ! find the nearest mixing boundary
            bdy = binary_search(n_mix_bdy, mix_bdy_q, bdy, s% q(k))
            ! don't check cells near a mixing boundary
            if (bdy > 0 .and. bdy < n_mix_bdy) then
               bdy_dist_dm = s% xmstar*abs(s% q(k) - mix_bdy_q(bdy))
               if (bdy_dist_dm < cz_dist_limit) cycle
            else
               bdy_dist_dm = 0
            end if

            do j = 1, s% species
               
               cid = s% chem_id(j)
               if (which == 0) then ! hydrogen
                  if (cid /= ih1) cycle
               else if (which == 1) then ! helium
                  if (cid /= ihe4) cycle
               else if (which == 2) then ! he3
                  if (cid /= ihe3) cycle
               else ! other
                  if (chem_isos% Z(cid) <= 2) cycle
               end if
               
               X = s% xa(j,k)
               X_old = s% xa_compare(j,k)
               delta_dX = X_old - X ! decrease in abundance

               if ((.not. decreases_only) .and. delta_dX < 0) delta_dX = -delta_dX
               
               if (X >= dX_limit_min_X) then
                  if ((.not. skip_hard_limit) .and. delta_dX > dX_hard_limit) then
                     check_dX = retry
                     s% why_Tlim = Tlim_dX
                     s% Tlim_dX_species = j
                     s% Tlim_dX_cell = k
                     if (s% report_why_dt_limits) then
                        write(*, '(a30, i5, 99(/,a30,e20.10))') &
                           'dX ' // trim(chem_isos% name(s% chem_id(j))), &
                           k, 'delta_dX', delta_dX, 'dX_hard_limit', dX_hard_limit
                     end if
                     return
                  end if
                  if (delta_dX > max_dX) then
                     max_dX = delta_dX
                     max_dX_j = j
                     max_dX_k = k
                     max_dX_bdy_dist_dm = bdy_dist_dm
                  end if
               end if
               if (X >= dX_div_X_limit_min_X) then
                  delta_dX_div_X = delta_dX/X
                  if ((.not. skip_hard_limit) .and. delta_dX_div_X > dX_div_X_hard_limit) then
                     check_dX = retry
                     s% why_Tlim = Tlim_dX_div_X
                     s% Tlim_dX_div_X_species = j
                     s% Tlim_dX_div_X_cell = k
                     if (s% report_why_dt_limits) then
                        write(*, '(a30, i5, 99(/,a30,e20.10))') &
                           'delta_dX_div_X ' // trim(chem_isos% name(s% chem_id(j))), &
                           k, 'delta_dX_div_X', delta_dX_div_X, 'dX_div_X_hard_limit', dX_div_X_hard_limit
                     end if
                     return
                  end if
                  if (delta_dX_div_X > max_dX_div_X) then
                     max_dX_div_X = delta_dX_div_X
                     max_dX_div_X_j = j
                     max_dX_div_X_k = k
                     max_dX_div_X_bdy_dist_dm = bdy_dist_dm
                  end if
               end if
            end do
         end do
         
         if (dX_limit > 0) then
            dX_dt_limit_ratio = max_dX/dX_limit
            if (dX_dt_limit_ratio <= s% dt_limit_ratio_target) then
               dX_dt_limit_ratio = 0
            else
               j = max_dX_j
               k = max_dX_k
               s% Tlim_dX_species = j
               s% Tlim_dX_cell = k
               write(*, '(a30, i5, 99e20.10)') &
                  'dX ' // trim(chem_isos% name(s% chem_id(j))), &
                  k, max_dX, dX_limit, (s% M_center + s% xmstar*(s% q(k) - s% dq(k)/2))/Msun, &
                  max_dX_bdy_dist_dm/Msun
               if (s% report_all_dt_limits) then
                  write(*, 2) 's% D_mix(k)', k, s% D_mix(k)
                  X_old = s% xa_compare(j,k)
                  write(*, 2) 'X_old', k, X_old
                  write(*, 2) 's% xa(j,k)', k, s% xa(j,k)
                  write(*, 2) 'dX', k, s% xa(j,k) - X_old
                  do j=k-5,k+5
                     write(*,2) 'D_mix', j, s% D_mix(j), s% q(j) - s% q(k)
                  end do
                  write(*,*)
                  !stop 'check_dX'
               end if
            end if
            
         end if
         
         if (dX_div_X_limit > 0) then         
            dX_div_X_dt_limit_ratio = max_dX_div_X/dX_div_X_limit
            if (dX_div_X_dt_limit_ratio <= s% dt_limit_ratio_target) then
               dX_div_X_dt_limit_ratio = 0
            else
               s% Tlim_dX_div_X_species = max_dX_div_X_j
               s% Tlim_dX_div_X_cell = max_dX_div_X_k
               if (which > 2) write(*, '(a30, i5, 99e20.10)') &
                  'limit dt because of large dX_div_X ' // &
                     trim(chem_isos% name(s% chem_id(max_dX_div_X_j))) // &
                     ' k, max, lim, m ', &
                  max_dX_div_X_k, max_dX_div_X, dX_div_X_limit, &
                  max_dX_div_X_bdy_dist_dm/Msun
               if (s% report_all_dt_limits) then
                  write(*,*)
                  k = max_dX_div_X_k
                  j = max_dX_div_X_j
                  X = s% xa(j,k)
                  X_old = s% xa_compare(j,k)
                  write(*,'(a)') &
                     '                                                X, X_old, X-X_old, (X-X_old)/X'
                  do j = 1, s% species
                     write(*,2) trim(chem_isos% name(s% chem_id(j))), k, X, X_old, X - X_old, (X - X_old)/X
                  end do
                  write(*,2) 'which', which
                  write(*,*) 's% report_all_dt_limits', s% report_all_dt_limits
                  bdy = binary_search(n_mix_bdy, mix_bdy_q, 0, s% q(k))
                  if (bdy > 0) then
                     write(*,2) 'q', k, s% q(k)
                     write(*,2) 'nearest cz bdy', mix_bdy_loc(bdy), mix_bdy_q(bdy)
                     write(*,2) 'cz_dq', mix_bdy_loc(bdy), s% q(k) - mix_bdy_q(bdy)
                     write(*,*)
                  end if
                  write(*,*)
               end if
            end if
         end if
         
      end function check_dX


      integer function check_dL_div_L(s, skip_hard_limit, dt, dL_div_L_dt_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dL_div_L_dt_ratio
         
         real(dp) :: L, abs_dL, abs_dL_div_L, max_dL_div_L
         integer :: k, max_dL_div_L_k
         real(dp) :: dL_div_L_limit_min_L, dL_div_L_limit, dL_div_L_hard_limit
         
         check_dL_div_L = keep_going
         
         dL_div_L_limit_min_L = Lsun*s% dL_div_L_limit_min_L
         dL_div_L_limit = s% dL_div_L_limit
         dL_div_L_hard_limit = s% dL_div_L_hard_limit
         
         if (dL_div_L_limit_min_L <= 0) return
         if (dL_div_L_limit <= 0 .and. dL_div_L_hard_limit <= 0) return
         
         max_dL_div_L = -1
         max_dL_div_L_k = -1
         abs_dL_div_L = 0; L=0 ! to quiet gfortran
         
         do k = 1, s% nz
            L = s% L(k)
            abs_dL = abs(L - s% L_start(k))
            if (L >= dL_div_L_limit_min_L) then
               abs_dL_div_L = abs_dL/L
               if (dL_div_L_hard_limit > 0 .and. (.not. skip_hard_limit) &
                     .and. abs_dL_div_L > dL_div_L_hard_limit) then
                  check_dL_div_L= retry
                  if (s% report_all_dt_limits) write(*, '(a30, i5, 99e20.10)') &
                     'dL_div_L too large at', k, L, s% L_start(k), abs_dL_div_L, dL_div_L_limit
                  return
               end if
               if (abs_dL_div_L > max_dL_div_L) then
                  max_dL_div_L = abs_dL_div_L
                  max_dL_div_L_k = k
               end if
            end if
         end do

         if (dL_div_L_limit <= 0) return
         dL_div_L_dt_ratio = max_dL_div_L/dL_div_L_limit
         if (dL_div_L_dt_ratio > s% dt_limit_ratio_target) then
            if (s% report_all_dt_limits) write(*, '(a30, i5, 99e20.10)') &
               'dL_div_L too large at', max_dL_div_L_k, &
               dL_div_L_dt_ratio, max_dL_div_L, dL_div_L_limit
         end if
         
      end function check_dL_div_L
      
      
      integer function check_change( &
            s, delta_value, lim, hard_lim, i, msg, &
            skip_hard_limit, dt_limit_ratio, relative_excess)
         use const_def, only:ln10
         type (star_info), pointer :: s
         real(dp), intent(in) :: delta_value, lim, hard_lim
         integer, intent(in) :: i
         character (len=*), intent(in) :: msg
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp), intent(out) :: relative_excess
         real(dp) :: abs_change
         check_change = keep_going
         abs_change = abs(delta_value)
         if (hard_lim > 0 .and. abs_change > hard_lim .and. (.not. skip_hard_limit)) then
            if (s% report_all_dt_limits) &
               write(*, '(a30, i5, 99e20.10)') trim(msg) // ' hard limit', &
                  i, delta_value, hard_lim
            check_change= retry
            return
         end if
         if (lim <= 0) return
         relative_excess = (abs_change - lim) / lim
         dt_limit_ratio = abs_change/lim ! 1d0/(s% timestep_dt_factor**relative_excess)
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*, '(a30, f20.10, i5, 99e20.10)') trim(msg), &
               dt_limit_ratio, i, delta_value, relative_excess, abs_change, lim
         end if
      end function check_change
      
      
      subroutine get_dlgP_info(s, i, max_dlnP)
         use const_def, only:ln10
         type (star_info), pointer :: s
         integer, intent(out) :: i
         real(dp), intent(out) :: max_dlnP
         real(dp) :: lim, dlnP
         integer :: k
         include 'formats'
         lim = ln10*s% delta_lgP_limit_min_lgP
         i = 0
         max_dlnP = 0
         do k=1,s% nz
            if (s% lnP(k) < lim) cycle
            dlnP = abs(s% lnP(k) - s% lnP_start(k))
            if (dlnP > max_dlnP) then
               max_dlnP = dlnP
               i = k
            end if
         end do      
      end subroutine get_dlgP_info
      
      
      integer function check_dlgP_change(s, skip_hard_limit, dt_limit_ratio)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, max_dlnP
         integer :: i
         include 'formats'
         call get_dlgP_info(s, i, max_dlnP)
         if (i == 0) then
            check_dlgP_change = keep_going
            return
         end if
         check_dlgP_change = check_change(s, max_dlnP/ln10, &
            s% delta_lgP_limit, s% delta_lgP_hard_limit, &
            i, 'check_dlgP_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgP_change
      
      
      subroutine get_dlgRho_info(s, i, max_dlnRho)
         use const_def, only:ln10
         type (star_info), pointer :: s
         integer, intent(out) :: i
         real(dp), intent(out) :: max_dlnRho
         real(dp) :: lim, dlnRho
         integer :: k
         include 'formats'
         lim = ln10*s% delta_lgRho_limit_min_lgRho
         i = 0
         max_dlnRho = 0
         do k=1,s% nz
            if (s% lnd(k) < lim) cycle
            dlnRho = abs(s% lnd(k) - s% lnd_start(k))
            if (dlnRho > max_dlnRho) then
               max_dlnRho = dlnRho
               i = k
            end if
         end do      
      end subroutine get_dlgRho_info
      
      
      integer function check_dlgRho_change(s, skip_hard_limit, dt_limit_ratio) 
         ! check max change in log10(density)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, max_dlnd
         integer :: i
         include 'formats'
         call get_dlgRho_info(s, i, max_dlnd)
         if (i == 0) then
            check_dlgRho_change = keep_going
            return
         end if
         check_dlgRho_change = check_change(s, max_dlnd/ln10, &
            s% delta_lgRho_limit, s% delta_lgRho_hard_limit, &
            i, 'check_dlgRho_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgRho_change
      
      
      subroutine get_dlgT_info(s, i, max_dlnT)
         use const_def, only:ln10
         type (star_info), pointer :: s
         integer, intent(out) :: i
         real(dp), intent(out) :: max_dlnT
         real(dp) :: lim, dlnT
         integer :: k
         include 'formats'
         lim = ln10*s% delta_lgT_limit_min_lgT
         i = 0
         max_dlnT = 0
         do k=1,s% nz
            if (s% lnT(k) < lim) cycle
            dlnT = abs(s% lnT(k) - s% lnT_start(k))
            if (dlnT > max_dlnT) then
               max_dlnT = dlnT
               i = k
            end if
         end do      
      end subroutine get_dlgT_info
      
      
      integer function check_dlgT_change(s, skip_hard_limit, dt_limit_ratio) 
         ! check max change in log10(temperature)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, max_dlnT
         integer :: i
         include 'formats'
         call get_dlgT_info(s, i, max_dlnT)
         if (i == 0) then
            check_dlgT_change = keep_going
            return
         end if
         check_dlgT_change = check_change(s, max_dlnT/ln10, &
            s% delta_lgT_limit, s% delta_lgT_hard_limit, &
            i, 'check_dlgT_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgT_change
      
      
      subroutine get_dlgE_info(s, i, max_dlnE)
         use const_def, only:ln10
         type (star_info), pointer :: s
         integer, intent(out) :: i
         real(dp), intent(out) :: max_dlnE
         real(dp) :: lim, dlnE
         integer :: k
         include 'formats'
         lim = ln10*s% delta_lgE_limit_min_lgE
         i = 0
         max_dlnE = 0
         do k=1,s% nz
            if (s% lnE(k) < lim) cycle
            dlnE = abs(s% energy(k) - s% energy_start(k))/s% energy(k)
            if (dlnE > max_dlnE) then
               max_dlnE = dlnE
               i = k
            end if
         end do      
         !write(*,2) 'max_dlgE', i, max_dlnE/ln10
      end subroutine get_dlgE_info
      
      
      integer function check_dlgE_change(s, skip_hard_limit, dt_limit_ratio) 
         ! check max change in log10(internal energy)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, max_dlnE
         integer :: i
         include 'formats'
         call get_dlgE_info(s, i, max_dlnE)
         if (i == 0) then
            check_dlgE_change = keep_going
            return
         end if
         check_dlgE_change = check_change(s, max_dlnE/ln10, &
            s% delta_lgE_limit, s% delta_lgE_hard_limit, &
            i, 'check_dlgE_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgE_change
      
      
      subroutine get_dlgR_info(s, i, max_dlnR)
         use const_def, only:ln10
         type (star_info), pointer :: s
         integer, intent(out) :: i
         real(dp), intent(out) :: max_dlnR
         real(dp) :: lim, dlnR
         integer :: k
         include 'formats'
         lim = ln10*s% delta_lgR_limit_min_lgR
         i = 0
         max_dlnR = 0
         do k=1,s% nz
            if (s% lnR(k) < lim) cycle
            dlnR = abs(s% lnR(k) - s% lnR_start(k))
            if (dlnR > max_dlnR) then
               max_dlnR = dlnR
               i = k
            end if
         end do      
      end subroutine get_dlgR_info
      
      
      integer function check_dlgR_change(s, skip_hard_limit, dt_limit_ratio)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, max_dlnR
         integer :: i
         include 'formats'
         call get_dlgR_info(s, i, max_dlnR)
         if (i == 0) then
            check_dlgR_change = keep_going
            return
         end if
         check_dlgR_change = check_change(s, max_dlnR/ln10, &
            s% delta_lgR_limit, s% delta_lgR_hard_limit, &
            i, 'check_dlgR_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgR_change
      
      
      integer function check_d_deltaR_shrink(s, skip_hard_limit, dt_limit_ratio)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: Rm1, R00, dR_pre, dR, diff, relative_excess, max_diff, &
            max_Rm1, max_R00
         integer :: k, kmax
         include 'formats'
         R00 = s% r_start(1)
         max_diff = 0; max_R00 = 0; max_Rm1 = 0
         do k=2, s% nz
            Rm1 = R00
            R00 = s% r_start(k)
            dR_pre = Rm1 - R00
            dR = s% r(k-1) - s% r(k)
            diff = max(0d0,dR_pre - dR) / max(1d-99,dR_pre) ! only consider reductions in thickness
            if (diff > max_diff) then
               max_Rm1 = Rm1; max_R00 = R00; max_diff = diff; kmax = k
            end if
         end do
         
         if (s% report_all_dt_limits .and. max_diff > s% d_deltaR_shrink_limit &
               .and. s% d_deltaR_shrink_limit > 0) then
            k = kmax
            dR = s% r(k-1) - s% r(k)
            dR_pre = max_Rm1 - max_R00
            write(*,2) 'max_diff', k, max_diff
            write(*,2) 'dR', k, dR
            write(*,2) 'dR_pre', k, dR_pre
            write(*,2) 'dR - dR_pre', k, dR - dR_pre
            write(*,2) 's% d_deltaR_shrink_limit', k, s% d_deltaR_shrink_limit
            write(*,*)
         end if
         
         check_d_deltaR_shrink = check_change(s, max_diff, &
            s% d_deltaR_shrink_limit, s% d_deltaR_shrink_hard_limit, &
            kmax, 'check_d_deltaR_shrink', skip_hard_limit, dt_limit_ratio, relative_excess)
            
      end function check_d_deltaR_shrink
      
      
      integer function check_d_deltaR_grow(s, skip_hard_limit, dt_limit_ratio)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: Rm1, R00, dR_pre, dR, diff, relative_excess, max_diff, &
            max_Rm1, max_R00
         integer :: k, kmax
         include 'formats'
         R00 = s% r_start(1)
         max_diff = 0; max_R00 = 0; max_Rm1 = 0
         do k=2, s% nz
            Rm1 = R00
            R00 = s% r_start(k)
            dR_pre = Rm1 - R00
            dR = s% r(k-1) - s% r(k)
            diff = max(0d0,dR - dR_pre) / max(1d-99,dR_pre) ! only consider increases in thickness
            if (diff > max_diff) then
               max_Rm1 = Rm1; max_R00 = R00; max_diff = diff; kmax = k
            end if
         end do
         
         if (s% report_all_dt_limits .and. max_diff > s% d_deltaR_grow_limit &
               .and. s% d_deltaR_grow_limit > 0) then
            k = kmax
            dR = s% r(k-1) - s% r(k)
            dR_pre = max_Rm1 - max_R00
            write(*,2) 'max_diff', k, max_diff
            write(*,2) 'dR', k, dR
            write(*,2) 'dR_pre', k, dR_pre
            write(*,2) 'dR - dR_pre', k, dR - dR_pre
            write(*,2) 's% d_deltaR_grow_limit', k, s% d_deltaR_grow_limit
            write(*,*)
         end if
         
         check_d_deltaR_grow = check_change(s, max_diff, &
            s% d_deltaR_grow_limit, s% d_deltaR_grow_hard_limit, &
            kmax, 'check_d_deltaR_grow', skip_hard_limit, dt_limit_ratio, relative_excess)
            
      end function check_d_deltaR_grow
      
      
      integer function check_lgL( &
            s, iso, msg, skip_hard_limit, dt, dt_limit_ratio)
         use chem_def
         type (star_info), pointer :: s         
         integer, intent(in) :: iso
         character (len=*), intent(in) :: msg
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio

         real(dp) :: &
            sum_LH, sum_LHe, sum_Lz, &
            new_L, max_other_L, old_L, lim, hard_lim, lgL_min, &
            drop_factor, relative_limit, lgL, max_other_lgL, lgL_old, &
            abs_change, relative_excess
         logical, parameter :: dbg = .false.
         include 'formats'
         check_lgL = keep_going

         sum_LH = s% L_by_category(ipp) + s% L_by_category(icno)
         sum_LHe = s% L_by_category(i3alf)
         sum_Lz = sum(s% L_by_category) - &
            (s% L_by_category(ipp) + s% L_by_category(icno) &
               + s% L_by_category(i3alf) + s% L_by_category(iphoto))

         if (iso == ih1) then
            new_L = sum_LH
            max_other_L = max(sum_LHe, sum_Lz)
            old_L = s% L_by_category_old(ipp) + s% L_by_category_old(icno)
            lim = s% delta_lgL_H_limit
            hard_lim = s% delta_lgL_H_hard_limit
            lgL_min = s% lgL_H_burn_min
            drop_factor = s% lgL_H_drop_factor
            relative_limit = s% lgL_H_burn_relative_limit
         else if (iso == ihe4) then
            new_L = sum_LHe
            max_other_L = max(sum_LH, sum_Lz)
            old_L = s% L_by_category_old(i3alf)
            lim = s% delta_lgL_He_limit
            hard_lim = s% delta_lgL_He_hard_limit
            lgL_min = s% lgL_He_burn_min
            drop_factor = s% lgL_He_drop_factor
            relative_limit = s% lgL_He_burn_relative_limit
         else if (iso == isi28) then
            new_L = sum_Lz
            max_other_L = max(sum_LH, sum_LHe)
            old_L = sum(s% L_by_category_old) - &
               (s% L_by_category_old(ipp) + s% L_by_category_old(icno) &
                  + s% L_by_category_old(i3alf) + s% L_by_category_old(iphoto))
            lim = s% delta_lgL_z_limit
            hard_lim = s% delta_lgL_z_hard_limit
            lgL_min = s% lgL_z_burn_min
            drop_factor = s% lgL_z_drop_factor
            relative_limit = s% lgL_z_burn_relative_limit
         else if (iso == iprot) then ! check_lgL_photo_change
            new_L = abs(s% L_by_category(iphoto))
            max_other_L = 0d0
            old_L = abs(s% L_by_category_old(iphoto))
            lim = s% delta_lgL_photo_limit
            hard_lim = s% delta_lgL_photo_hard_limit
            lgL_min = s% lgL_photo_burn_min
            drop_factor = s% lgL_photo_drop_factor 
            relative_limit = 0d0
         else if (iso == ineut) then ! check_lgL_nuc_change
            new_L = sum(s% L_by_category) - s% L_by_category(iphoto)
            max_other_L = 0d0
            old_L = sum(s% L_by_category_old) - s% L_by_category_old(iphoto)
            lim = s% delta_lgL_nuc_limit
            hard_lim = s% delta_lgL_nuc_hard_limit
            lgL_min = s% lgL_nuc_burn_min
            drop_factor = s% lgL_nuc_drop_factor 
            relative_limit = 0d0
         else
            stop 'bad iso arg for check_lgL'
         end if
         
         if (new_L < old_L) then
            lim = lim*drop_factor
            hard_lim = hard_lim*drop_factor
         end if

         if (dbg) write(*,*)
         if (dbg) write(*,1) trim(msg) // ' new_L', new_L
         if (dbg) write(*,1) 'old_L', old_L
         if (new_L <= 0 .or. old_L <= 0) return
         
         lgL = safe_log10_cr(new_L)
         if (dbg) write(*,1) 'lgL', lgL
         if (lgL < lgL_min) return
         
         if (max_other_L > 0) then
            max_other_lgL = safe_log10_cr(max_other_L)
            if (dbg) write(*,1) 'max_other_lgL', max_other_lgL
            if (max_other_lgL - relative_limit > lgL) return
         end if
         
         lgL_old = safe_log10_cr(old_L)
         if (dbg) write(*,1) 'lgL_old', lgL_old
         abs_change = abs(lgL - lgL_old)
         if (dbg) write(*,1) 'abs_change', abs_change
         if (dbg) write(*,1) 'hard_lim', hard_lim
         if (hard_lim > 0 .and. abs_change > hard_lim .and. (.not. skip_hard_limit)) then
            if (s% report_all_dt_limits) write(*, '(a30, f20.10, 99e20.10)') trim(msg), &
               lgL - lgL_old, hard_lim, lgL, lgL_old
            check_lgL = retry
            return
         end if
         
         if (dbg) write(*,1) 'lim', lim
         if (lim <= 0) return
         
         relative_excess = (abs_change - lim) / lim
         if (dbg) write(*,1) 'relative_excess', relative_excess
         dt_limit_ratio = 1d0/pow_cr(s% timestep_dt_factor,relative_excess)
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*, '(a30, f20.10, 99e20.10)') trim(msg), &
               dt_limit_ratio, lgL - lgL_old, lim, lgL, lgL_old, relative_excess
         end if

      end function check_lgL

      
      integer function check_lgL_H_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use chem_def, only: ih1
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         check_lgL_H_change = check_lgL( &
            s, ih1, 'check_lgL_H_change', skip_hard_limit, dt, dt_limit_ratio)
      end function check_lgL_H_change
      
      
      integer function check_lgL_He_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use chem_def, only: ihe4
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         check_lgL_He_change = check_lgL( &
            s, ihe4, 'check_lgL_H_change', skip_hard_limit, dt, dt_limit_ratio)
      end function check_lgL_He_change

            
      integer function check_lgL_z_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use chem_def, only: isi28
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         check_lgL_z_change = check_lgL( &
            s, isi28, 'check_lgL_H_change', skip_hard_limit, dt, dt_limit_ratio)
      end function check_lgL_z_change

            
      integer function check_lgL_photo_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use chem_def, only: iprot
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         check_lgL_photo_change = check_lgL( &
            s, iprot, 'check_lgL_photo_change', skip_hard_limit, dt, dt_limit_ratio)
      end function check_lgL_photo_change

      
      integer function check_lgL_nuc_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use chem_def, only: ineut
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         check_lgL_nuc_change = check_lgL( &
            s, ineut, 'check_lgL_nuc_change', skip_hard_limit, dt, dt_limit_ratio)
      end function check_lgL_nuc_change
      
      
      integer function check_lgL_nuc_cat_change( &
            s, n_mix_bdy, mix_bdy_q, skip_hard_limit, dt_limit_ratio)
         use rates_def
         use chem_def
         use num_lib, only: binary_search
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         integer, intent(in) :: n_mix_bdy
         real(dp), intent(in), pointer :: mix_bdy_q(:)
         real(dp), intent(inout) :: dt_limit_ratio
         
         integer :: k, max_j, max_k, bdy
         real(dp) :: max_lgL_diff, relative_excess, max_diff, cat_burn_min
         
         include 'formats'

         check_lgL_nuc_cat_change = keep_going
         
         if (s% delta_lgL_nuc_cat_limit <= 0 .and. s% delta_lgL_nuc_cat_hard_limit <= 0) return

         cat_burn_min = exp10_cr(s% lgL_nuc_cat_burn_min)
         max_diff = 0
         max_j = -1
         max_k = -1
         bdy = -1
         
         do k = 1, s% nz
            
            ! find the nearest mixing boundary
            bdy = binary_search(n_mix_bdy, mix_bdy_q, bdy, s% q(k))
            ! don't check cells near a mixing boundary
            if (bdy > 0) then
               if (abs(s% q(k) - mix_bdy_q(bdy)) < s% lgL_nuc_mix_dist_limit) cycle
            end if

            if (s% check_delta_lgL_pp) call do1_category(ipp,k)
            if (s% check_delta_lgL_cno) call do1_category(icno,k)
            if (s% check_delta_lgL_3alf) call do1_category(i3alf,k)
            
            if (s% check_delta_lgL_burn_c) call do1_category(i_burn_c,k)
            if (s% check_delta_lgL_burn_n) call do1_category(i_burn_n,k)
            if (s% check_delta_lgL_burn_o) call do1_category(i_burn_o,k)
            if (s% check_delta_lgL_burn_ne) call do1_category(i_burn_ne,k)
            if (s% check_delta_lgL_burn_na) call do1_category(i_burn_na,k)
            if (s% check_delta_lgL_burn_mg) call do1_category(i_burn_mg,k)
            if (s% check_delta_lgL_burn_si) call do1_category(i_burn_si,k)
            if (s% check_delta_lgL_burn_s) call do1_category(i_burn_s,k)
            if (s% check_delta_lgL_burn_ar) call do1_category(i_burn_ar,k)
            if (s% check_delta_lgL_burn_ca) call do1_category(i_burn_ca,k)
            if (s% check_delta_lgL_burn_ti) call do1_category(i_burn_ti,k)
            if (s% check_delta_lgL_burn_cr) call do1_category(i_burn_cr,k)
            if (s% check_delta_lgL_burn_fe) call do1_category(i_burn_fe,k)
            
            if (s% check_delta_lgL_cc) call do1_category(icc,k)
            if (s% check_delta_lgL_co) call do1_category(ico,k)
            if (s% check_delta_lgL_oo) call do1_category(ioo,k)
            
         end do
         

         if (max_diff <= 0) return
         
         max_lgL_diff = log10_cr(max_diff/Lsun)
         s% Tlim_dlgL_nuc_category = max_j
         s% Tlim_dlgL_nuc_cell = max_k
         
         check_lgL_nuc_cat_change = check_change(s, max_lgL_diff, &
            s% delta_lgL_nuc_cat_limit, s% delta_lgL_nuc_cat_hard_limit, &
            max_j, 'check_lgL_nuc_cat_change', skip_hard_limit, dt_limit_ratio, relative_excess)
         
         contains         
         
         subroutine do1_category(j, k)
            integer, intent(in) :: j, k
            real(dp) :: diff, abs_diff
            if (s% luminosity_by_category(j,k) < cat_burn_min) return
            if (s% luminosity_by_category_start(j,k) < cat_burn_min) return
            diff = s% luminosity_by_category(j,k) - s% luminosity_by_category_start(j,k)
            abs_diff = abs(diff)
            if (abs_diff <= max_diff) return
            max_diff = abs_diff
            max_j = j
            max_k = k
         end subroutine do1_category
         
      end function check_lgL_nuc_cat_change

         
      integer function check_delta_ang_momentum(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: delta_lg_J, relative_excess
         check_delta_ang_momentum = keep_going
         if (s% total_angular_momentum_old <= 0 .or. s% total_angular_momentum <= 0) return
         delta_lg_J = log10_cr(s% total_angular_momentum/s% total_angular_momentum_old)
         check_delta_ang_momentum = check_change(s, delta_lg_J, &
            s% delta_lg_total_J_limit, s% delta_lg_total_J_hard_limit, &
            1, 'check_delta_ang_momentum', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_delta_ang_momentum
      
      
      integer function check_dlgTeff_change(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess
         include 'formats'
         if (s% doing_relax .or. s% Teff_old <= 0 .or. s% Teff <= 0)  then
            check_dlgTeff_change = keep_going
            return
         end if
         check_dlgTeff_change = check_change(s, safe_log10_cr(s% Teff/s% Teff_old), &
            s% delta_lgTeff_limit, s% delta_lgTeff_hard_limit, &
            1, 'check_dlgTeff_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgTeff_change
      
      
      integer function check_dYe_change(s, skip_hard_limit, dt_limit_ratio) ! check max change in Ye
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, max_diff
         integer :: i, nz
         nz = s% nz
         include 'formats'
         i = maxloc(abs(s% ye(1:nz) - s% ye_start(1:nz)), dim=1)
         max_diff = s% ye(i) - s% ye_start(i)
         
         !write(*,2) 'dYe', i, max_diff, s% ye_start(i), s% ye(i), &
         !   s% delta_Ye_limit, s% delta_Ye_hard_limit
         !write(*,*) 'skip_hard_limit', skip_hard_limit
         !write(*,*)
         
         check_dYe_change = check_change(s, max_diff, &
            s% delta_Ye_limit, s% delta_Ye_hard_limit, &
            i, 'check_dYe_change', .false., dt_limit_ratio, relative_excess)
         !   i, 'check_dYe_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dYe_change
      
      
      integer function check_dYe_highT_change(s, skip_hard_limit, dt_limit_ratio) ! check max change in Ye
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, max_diff, ye_diff, T_limit
         integer :: i, k
         include 'formats'
         i = 0
         max_diff = 0
         T_limit = s% minT_for_highT_Ye_limit
         do k=1, s% nz
            if (s% T(k) < T_limit) cycle
            ye_diff = abs(s% ye(k) - s% ye_start(k))
            if (ye_diff <= max_diff) cycle
            max_diff = ye_diff
            i = k
         end do
         !if (max_diff > 0d0) write(*,2) 'check_dYe_highT_change max_diff', i, max_diff
         check_dYe_highT_change = check_change(s, max_diff, &
            s% delta_Ye_highT_limit, s% delta_Ye_highT_hard_limit, &
            i, 'check_dYe_highT_change', .false., dt_limit_ratio, relative_excess)
      end function check_dYe_highT_change


      integer function check_dlgT_cntr_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, change
         if (s% doing_relax) then
            check_dlgT_cntr_change = keep_going
            return
         end if
         change = (s% lnT(s% nz) - s% lnT_start(s% nz))/ln10
         check_dlgT_cntr_change = check_change(s, change, &
            s% delta_lgT_cntr_limit, s% delta_lgT_cntr_hard_limit, &
            s% nz, 'check_dlgT_cntr_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgT_cntr_change


      integer function check_dlgRho_cntr_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, dlgRho_cntr
         integer :: nz
         if (s% doing_relax) then
            check_dlgRho_cntr_change = keep_going
            return
         end if
         nz = s% nz
         dlgRho_cntr = (s% lnd(nz) - s% lnd_start(nz))/ln10
         check_dlgRho_cntr_change = check_change(s, dlgRho_cntr, &
            s% delta_lgRho_cntr_limit, s% delta_lgRho_cntr_hard_limit, &
            nz, 'check_dlgRho_cntr_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgRho_cntr_change


      integer function check_dlgT_max_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, dlgT_max
         if (s% doing_relax) then
            check_dlgT_max_change = keep_going
            return
         end if
         dlgT_max = maxval(s% lnT(1:s% nz) - s% lnT_start(1:s% nz))/ln10
         check_dlgT_max_change = check_change(s, dlgT_max, &
            s% delta_lgT_max_limit, s% delta_lgT_max_hard_limit, &
            s% nz, 'check_dlgT_max_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgT_max_change


      integer function check_dlgRho_max_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use const_def, only:ln10
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, dlgRho_max
         integer :: nz
         if (s% doing_relax) then
            check_dlgRho_max_change = keep_going
            return
         end if
         nz = s% nz
         dlgRho_max = maxval(s% lnd(1:nz) - s% lnd_start(1:nz))/ln10
         check_dlgRho_max_change = check_change(s, dlgRho_max, &
            s% delta_lgRho_max_limit, s% delta_lgRho_max_hard_limit, &
            s% nz, 'check_dlgRho_max_change', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlgRho_max_change


      integer function check_dlog_eps_nuc_cntr_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use mlt_def, only: convective_mixing
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, delta
         delta = 0
         if (s% mixing_type(s% nz) /= convective_mixing .and. &
             s% center_eps_nuc > s% center_eps_nuc_old .and. &
             s% center_eps_nuc_old > 1) then
            delta = log10_cr(s% center_eps_nuc / s% center_eps_nuc_old)
         end if
         check_dlog_eps_nuc_cntr_change = check_change(s, delta, &
            s% delta_log_eps_nuc_cntr_limit, s% delta_log_eps_nuc_cntr_hard_limit, &
            s% nz, 'check_dlog_eps_nuc_cntr_change', &
            skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlog_eps_nuc_cntr_change


      integer function check_dlog_eps_nuc_change(s, skip_hard_limit, dt, dt_limit_ratio)
         use mlt_def, only: convective_mixing
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, max_ratio, ratio, delta, &
            limit_ratio, delta_r
         integer :: k, j, nz, k_max
         include 'formats'
         nz = s% nz
         limit_ratio = exp10_cr(s% delta_log_eps_nuc_limit)
         max_ratio = limit_ratio
         k_max = 0
         zoneloop: do k=1,nz
            if (s% eps_nuc_start(k) < 1) cycle zoneloop
            ratio = s% eps_nuc(k)/s% eps_nuc_start(k)
            if (s% mixing_type(k) /= convective_mixing .and. &
                s% mixing_type(min(nz,k+1)) /= convective_mixing .and. &
                ratio > max_ratio) then
               do j = 1, s% num_conv_boundaries
                  delta_r = abs(s% r(s% conv_bdy_loc(j)) - s% r(k))
                  if (delta_r <= s% scale_height(k)) then
                     cycle zoneloop ! skip ones that are too close to convection zone
                  end if
               end do
               max_ratio = ratio
               k_max = k
            end if
         end do zoneloop
         if (k_max > 0) then
            delta = log10_cr(max_ratio)
            if (s% report_all_dt_limits) write(*,4) 'log eps nuc ratio', &
               k_max, nz, s% model_number, &
               delta/s% delta_log_eps_nuc_limit, delta, &
               s% eps_nuc(k_max), s% eps_nuc_start(k_max)
         else
            delta = 0
         end if
         check_dlog_eps_nuc_change = check_change(s, delta, &
            s% delta_log_eps_nuc_limit, s% delta_log_eps_nuc_hard_limit, &
            nz, 'check_dlog_eps_nuc_change', &
            skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dlog_eps_nuc_change

      
      integer function check_lg_XH_cntr(s, skip_hard_limit, dt_limit_ratio)
         use chem_def, only: ih1
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, lg_XH_cntr, lg_XH_cntr_old
         integer :: h1, nz
         include 'formats'
         check_lg_XH_cntr = keep_going
         h1 = s% net_iso(ih1)
         if (h1 == 0) return
         nz = s% nz
         if (s% xa(h1,nz) < 1d-10) return
         lg_XH_cntr = log10_cr(s% xa(h1,nz))
         if (lg_XH_cntr > s% delta_lg_XH_cntr_max) return
         if (lg_XH_cntr < s% delta_lg_XH_cntr_min) return
         lg_XH_cntr_old = safe_log10_cr(s% xa_compare(h1,nz))
         check_lg_XH_cntr = check_change(s, lg_XH_cntr - lg_XH_cntr_old, &
            s% delta_lg_XH_cntr_limit, s% delta_lg_XH_cntr_hard_limit, &
            1, 'check_lg_XH_cntr', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_lg_XH_cntr

      
      integer function check_lg_XHe_cntr(s, skip_hard_limit, dt_limit_ratio) 
         use chem_def, only: ih1, ihe4
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, lg_XHe_cntr, lg_XHe_cntr_old
         integer :: h1, he4, nz
         real(dp) :: xh1, xhe4
         include 'formats'
         check_lg_XHe_cntr = keep_going
         h1 = s% net_iso(ih1)
         he4 = s% net_iso(ihe4)
         if (h1 == 0 .or. he4 == 0) return
         nz = s% nz
         xh1 = s% xa(h1,nz)
         xhe4 = s% xa(he4,nz)
         if (xhe4 < max(xh1, 1d-10)) return
         lg_XHe_cntr = log10_cr(xhe4)
         if (lg_XHe_cntr > s% delta_lg_XHe_cntr_max) return
         if (lg_XHe_cntr < s% delta_lg_XHe_cntr_min) return
         lg_XHe_cntr_old = safe_log10_cr(s% xa_compare(he4,nz))
         check_lg_XHe_cntr = check_change(s, lg_XHe_cntr - lg_XHe_cntr_old, &
            s% delta_lg_XHe_cntr_limit, s% delta_lg_XHe_cntr_hard_limit, &
            1, 'check_lg_XHe_cntr', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_lg_XHe_cntr

      
      integer function check_lg_XC_cntr(s, skip_hard_limit, dt_limit_ratio) 
         use chem_def, only: ih1, ihe4, ic12
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, lg_XC_cntr, lg_XC_cntr_old
         integer :: h1, he4, c12, nz
         real(dp) :: xh1, xhe4, xc12
         include 'formats'
         check_lg_XC_cntr = keep_going
         h1 = s% net_iso(ih1)
         he4 = s% net_iso(ihe4)
         c12 = s% net_iso(ic12)
         if (h1 == 0 .or. he4 == 0 .or. c12 == 0) return
         nz = s% nz
         xh1 = s% xa(h1,nz)
         xhe4 = s% xa(he4,nz)
         xc12 = s% xa(c12,nz)
         if (xc12 < max(xh1, xhe4, 1d-10)) return
         if (s% xa(c12,nz) < 1d-10) return
         lg_XC_cntr = log10_cr(xc12)
         if (lg_XC_cntr > s% delta_lg_XC_cntr_max) return
         if (lg_XC_cntr < s% delta_lg_XC_cntr_min) return
         lg_XC_cntr_old = safe_log10_cr(s% xa_compare(c12,nz))
         check_lg_XC_cntr = check_change(s, lg_XC_cntr - lg_XC_cntr_old, &
            s% delta_lg_XC_cntr_limit, s% delta_lg_XC_cntr_hard_limit, &
            1, 'check_lg_XC_cntr', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_lg_XC_cntr

      
      integer function check_lg_XNe_cntr(s, skip_hard_limit, dt_limit_ratio) 
         use chem_def, only: ih1, ihe4, ic12, ine20
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, lg_XNe_cntr, lg_XNe_cntr_old
         integer :: h1, he4, c12, ne20, nz
         real(dp) :: xh1, xhe4, xc12, xne20
         include 'formats'
         h1 = s% net_iso(ih1)
         he4 = s% net_iso(ihe4)
         c12 = s% net_iso(ic12)
         ne20 = s% net_iso(ine20)
         check_lg_XNe_cntr = keep_going
         if (h1 == 0 .or. he4 == 0 .or. c12 == 0 .or. ne20 == 0) return
         nz = s% nz
         xh1 = s% xa(h1,nz)
         xhe4 = s% xa(he4,nz)
         xc12 = s% xa(c12,nz)
         xne20 = s% xa(ne20,nz)
         if (xne20 < max(xh1, xhe4, xc12, 1d-10)) return
         lg_XNe_cntr = log10_cr(xne20)
         if (lg_XNe_cntr > s% delta_lg_XNe_cntr_max) return
         if (lg_XNe_cntr < s% delta_lg_XNe_cntr_min) return
         lg_XNe_cntr_old = safe_log10_cr(s% xa_compare(ne20,nz))
         check_lg_XNe_cntr = check_change(s, lg_XNe_cntr - lg_XNe_cntr_old, &
            s% delta_lg_XNe_cntr_limit, s% delta_lg_XNe_cntr_hard_limit, &
            1, 'check_lg_XNe_cntr', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_lg_XNe_cntr

      
      integer function check_lg_XO_cntr(s, skip_hard_limit, dt_limit_ratio) 
         use chem_def, only: ih1, ihe4, ic12, io16
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, lg_XO_cntr, lg_XO_cntr_old
         integer :: h1, he4, c12, o16, nz
         real(dp) :: xh1, xhe4, xc12, xo16
         include 'formats'
         check_lg_XO_cntr = keep_going
         h1 = s% net_iso(ih1)
         he4 = s% net_iso(ihe4)
         c12 = s% net_iso(ic12)
         o16 = s% net_iso(io16)
         if (h1 == 0 .or. he4 == 0 .or. c12 == 0 .or. o16 == 0) return
         nz = s% nz
         xh1 = s% xa(h1,nz)
         xhe4 = s% xa(he4,nz)
         xc12 = s% xa(c12,nz)
         xo16 = s% xa(o16,nz)
         if (xo16 < max(xh1, xhe4, xc12, 1d-10)) return
         lg_XO_cntr = log10_cr(xo16)
         if (lg_XO_cntr > s% delta_lg_XO_cntr_max) return
         if (lg_XO_cntr < s% delta_lg_XO_cntr_min) return
         lg_XO_cntr_old = safe_log10_cr(s% xa_compare(o16,nz))
         check_lg_XO_cntr = check_change(s, lg_XO_cntr - lg_XO_cntr_old, &
            s% delta_lg_XO_cntr_limit, s% delta_lg_XO_cntr_hard_limit, &
            1, 'check_lg_XO_cntr', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_lg_XO_cntr

      
      integer function check_lg_XSi_cntr(s, skip_hard_limit, dt_limit_ratio) 
         use chem_def, only: ih1, ihe4, ic12, io16, isi28
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: relative_excess, lg_XSi_cntr, lg_XSi_cntr_old
         integer :: h1, he4, c12, o16, si28, nz
         real(dp) :: xh1, xhe4, xc12, xo16, xsi28
         include 'formats'
         check_lg_XSi_cntr = keep_going
         h1 = s% net_iso(ih1)
         he4 = s% net_iso(ihe4)
         c12 = s% net_iso(ic12)
         o16 = s% net_iso(io16)
         si28 = s% net_iso(isi28)
         if (h1 == 0 .or. he4 == 0 .or. c12 == 0 .or. o16 == 0 .or. si28 == 0) return
         nz = s% nz
         xh1 = s% xa(h1,nz)
         xhe4 = s% xa(he4,nz)
         xc12 = s% xa(c12,nz)
         xo16 = s% xa(o16,nz)
         xsi28 = s% xa(si28,nz)
         if (xo16 < max(xh1, xhe4, xc12, xo16, 1d-10)) return
         lg_XSi_cntr = log10_cr(xsi28)
         if (lg_XSi_cntr > s% delta_lg_XSi_cntr_max) return
         if (lg_XSi_cntr < s% delta_lg_XSi_cntr_min) return
         lg_XSi_cntr_old = safe_log10_cr(s% xa_compare(si28,nz))
         check_lg_XSi_cntr = check_change(s, lg_XSi_cntr - lg_XSi_cntr_old, &
            s% delta_lg_XSi_cntr_limit, s% delta_lg_XSi_cntr_hard_limit, &
            1, 'check_lg_XSi_cntr', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_lg_XSi_cntr
         
            
      integer function check_truncation_ratio(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         check_truncation_ratio = keep_going
         ! assume that have already set s% truncation_ratio before calling this.
         !if ((.not. skip_hard_limit) .and. s% truncation_ratio_hard_limit > 0 .and. &
         if (s% truncation_ratio_hard_limit > 0 .and. &
               s% truncation_ratio > s% truncation_ratio_hard_limit) then
            check_truncation_ratio = retry
            if (s% report_all_dt_limits) &
               write(*, '(a30, 99e20.10)') 'truncation_ratio hard limit', &
                  s% truncation_ratio, s% truncation_ratio_hard_limit
            return
         end if
         if (s% truncation_ratio_limit <= 0) return
         dt_limit_ratio = s% truncation_ratio/s% truncation_ratio_limit
         if (s% report_all_dt_limits .and. &
                  dt_limit_ratio > s% dt_limit_ratio_target) then
            write(*, '(a30, 99e20.10)') &
               'truncation_ratio: log dt, truncation_ratio, limit', &
               dt_limit_ratio, s% truncation_ratio, s% truncation_ratio_limit
         end if
      end function check_truncation_ratio
         
            
      integer function check_v_div_v_crit(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: v_div_v_crit
         check_v_div_v_crit = keep_going
         v_div_v_crit = s% v_div_v_crit_avg_surf
         if ((.not. skip_hard_limit) .and. s% v_div_v_crit_hard_limit > 0 .and. &
               v_div_v_crit > s% v_div_v_crit_hard_limit) then
            check_v_div_v_crit = retry
            if (s% report_all_dt_limits) &
               write(*, '(a30, 99e20.10)') 'v_div_v_crit hard limit', &
                  v_div_v_crit, s% v_div_v_crit_hard_limit
            return
         end if
         if (s% v_div_v_crit_limit <= 0) return
         dt_limit_ratio = v_div_v_crit/s% v_div_v_crit_limit
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*, '(a30, 99e20.10)') &
               'v_div_v_crit: log dt, v_div_v_crit, limit', &
               dt_limit_ratio, v_div_v_crit, s% v_div_v_crit_limit
         end if
      end function check_v_div_v_crit
         
            
      integer function check_max_fixup_for_mix(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: max_fixup_for_mix
         check_max_fixup_for_mix = keep_going
         max_fixup_for_mix = s% max_fixup_for_mix
         if ((.not. skip_hard_limit) .and. s% max_fixup_for_mix_hard_limit > 0 .and. &
               max_fixup_for_mix > s% max_fixup_for_mix_hard_limit) then
            check_max_fixup_for_mix = retry
            if (s% report_all_dt_limits) &
               write(*, '(a30, 99e20.10)') 'max_fixup_for_mix hard limit', &
                  max_fixup_for_mix, s% max_fixup_for_mix_hard_limit
            return
         end if
         if (s% max_fixup_for_mix_limit <= 0) return
         dt_limit_ratio = max_fixup_for_mix/s% max_fixup_for_mix_limit
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*, '(a30, 99e20.10)') &
               'max_fixup_for_mix: log dt, max_fixup_for_mix, limit', &
               dt_limit_ratio, max_fixup_for_mix, s% max_fixup_for_mix_limit
         end if
      end function check_max_fixup_for_mix


      integer function check_delta_mdot(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: mdot, mdot_old, delta_mdot
         check_delta_mdot = keep_going
         mdot = s% mstar_dot
         mdot_old = s% mstar_dot_old
         delta_mdot = abs(mdot - mdot_old)/ &
            (s% delta_mdot_atol*Msun/secyer + &
               s% delta_mdot_rtol*max(abs(mdot),abs(mdot_old)))
         if (delta_mdot == 0) return
         if (s% delta_mdot_hard_limit > 0 .and. (.not. skip_hard_limit)) then
            if (delta_mdot > s% delta_mdot_hard_limit) then
               if (s% report_all_dt_limits) &
                  write(*, '(a30, f20.10, 99e20.10)') 'delta_mdot_hard_limit', &
                     delta_mdot, s% delta_mdot_hard_limit
                  check_delta_mdot= retry
               return
            end if
         end if
         if (s% delta_mdot_limit <= 0) return
         dt_limit_ratio = delta_mdot/s% delta_mdot_limit
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*, '(a30, f20.10, 99e20.10)') 'delta_mdot_limit', &
               dt_limit_ratio, delta_mdot, s% delta_mdot_limit
         end if
      end function check_delta_mdot

         
      integer function check_delta_mstar(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: delta_lg_star_mass
         delta_lg_star_mass = abs(log10_cr(s% mstar/s% mstar_old))
         if (s% delta_lg_star_mass_hard_limit > 0 .and. (.not. skip_hard_limit)) then
            if (delta_lg_star_mass > s% delta_lg_star_mass_hard_limit) then
               if (s% report_all_dt_limits) &
                  write(*, '(a30, f20.10, 99e20.10)') 'delta_lg_star_mass_hard_limit', &
                     delta_lg_star_mass, s% delta_lg_star_mass_hard_limit
                  check_delta_mstar= retry
               return
            end if
         end if
         check_delta_mstar = keep_going
         if (s% delta_lg_star_mass_limit <= 0) return
         dt_limit_ratio = delta_lg_star_mass/s% delta_lg_star_mass_limit
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*, '(a30, f20.10, 99e20.10)') 'delta_lg_star_mass_limit', &
               dt_limit_ratio, delta_lg_star_mass, s% delta_lg_star_mass_limit
         end if
      end function check_delta_mstar

         
      integer function check_CpT_absMdot_div_L( &
            s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         
         real(dp) :: ratio, L
         integer :: k
         include 'formats'
         
         check_CpT_absMdot_div_L = keep_going
         
         k = s% k_for_test_CpT_absMdot_div_L
         L = max(1d-99,s% L(k))
         ratio = s% Cp(k)*s% T(k)*abs(s% mstar_dot)/L
         !write(*,1) 'ratio for CpT_absMdot_div_L', ratio

         if (s% CpT_absMdot_div_L_hard_limit > 0 .and. &
               (.not. skip_hard_limit) .and. &
               ratio > s% CpT_absMdot_div_L_hard_limit) then
            if (s% report_all_dt_limits) &
               write(*, '(a30, 99e20.10)') 'CpT_absMdot_div_L_hard_limit', &
                  ratio, s% CpT_absMdot_div_L_hard_limit
               check_CpT_absMdot_div_L = retry
            return
         end if
         
         if (s% CpT_absMdot_div_L_limit <= 0) return                  
         dt_limit_ratio = ratio/s% CpT_absMdot_div_L_limit
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*, '(a30, 99e20.10)') 'CpT_absMdot_div_L_limit', &
               dt_limit_ratio, ratio, s% CpT_absMdot_div_L_limit
         end if
         
      end function check_CpT_absMdot_div_L

         
      integer function check_delta_lgL(s, skip_hard_limit, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: dlgL, relative_excess
         if (s% doing_relax) then
            check_delta_lgL = keep_going
            return
         end if
         if (s% L_phot < s% delta_lgL_limit_L_min) then
            dlgL = 0
         else
            dlgL = log10_cr(s% L_phot/s% L_phot_old)
         end if
         check_delta_lgL = check_change(s, dlgL, &
            s% delta_lgL_limit, s% delta_lgL_hard_limit, &
            1, 'check_delta_lgL', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_delta_lgL

         
      integer function check_delta_HR(s, skip_hard_limit, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: dlgL, dlgTeff, dHR, relative_excess
         check_delta_HR = keep_going
         if (s% L_phot_old <= 0 .or. s% Teff_old <= 0) return
         dlgL = log10_cr(s% L_phot/s% L_phot_old)
         dlgTeff = log10_cr(s% Teff/s% Teff_old)
         dHR = sqrt(pow2(s% delta_HR_ds_L*dlgL) + pow2(s% delta_HR_ds_Teff*dlgTeff))
         check_delta_HR = check_change(s, dHR, &
            s% delta_HR_limit, s% delta_HR_hard_limit, &
            1, 'check_delta_HR', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_delta_HR

         
      integer function check_avg_E_residual(s, skip_hard_limit, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: val, relative_excess
         include 'formats'
         check_avg_E_residual = keep_going
         val = abs(dot_product(s% dq(1:s% nz),s% E_residual(1:s% nz)))
         check_avg_E_residual = check_change(s, val, &
            s% limit_for_avg_E_residual, &
            s% hard_limit_for_avg_E_residual, &
            1, 'check_avg_E_residual', &
            .false., dt_limit_ratio, relative_excess)
      end function check_avg_E_residual

         
      integer function check_max_E_residual(s, skip_hard_limit, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: val, relative_excess
         include 'formats'
         check_max_E_residual = keep_going
         val = maxval(abs(s% E_residual(1:s% nz)))
         check_max_E_residual = check_change(s, val, &
            s% limit_for_max_E_residual, &
            s% hard_limit_for_max_E_residual, &
            1, 'check_max_E_residual', &
            .false., dt_limit_ratio, relative_excess)
      end function check_max_E_residual

         
      integer function check_rel_rate_in_energy(s, skip_hard_limit, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: rel_rate, relative_excess
         include 'formats'
         check_rel_rate_in_energy = keep_going
         if (s% mstar /= s% mstar_old) then
            return
         end if
         if (s% use_piston .and. .not. s% done_with_piston) return
         rel_rate = abs(s% error_in_energy_conservation/s% total_energy)/s% dt
         check_rel_rate_in_energy = check_change(s, rel_rate, &
            s% limit_for_rel_rate_in_energy_conservation, &
            s% hard_limit_for_rel_rate_in_energy_conservation, &
            1, 'check_rel_rate_in_energy', &
            .false., dt_limit_ratio, relative_excess)
      end function check_rel_rate_in_energy

         
      integer function check_rel_error_in_energy(s, skip_hard_limit, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: rel_error, relative_excess
         include 'formats'
         check_rel_error_in_energy = keep_going
         if (s% mstar /= s% mstar_old) then
            return
         end if
         if (s% use_piston .and. .not. s% done_with_piston) return
         rel_error = abs(s% error_in_energy_conservation/s% total_energy)
         check_rel_error_in_energy = check_change(s, rel_error, &
            s% limit_for_rel_error_in_energy_conservation, &
            s% hard_limit_for_rel_error_in_energy_conservation, &
            1, 'check_rel_error_in_energy', &
            .false., dt_limit_ratio, relative_excess)
      end function check_rel_error_in_energy

         
      integer function check_dt_div_dt_Courant(s, skip_hard_limit, dt, dt_limit_ratio)
         use star_utils, only: dt_Courant
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: ratio, dt_C, relative_excess
         if (s% doing_relax) then
            check_dt_div_dt_Courant = keep_going
            return
         end if
         dt_C = dt_Courant(s, s% Tlim_dt_div_dt_Courant_cell)
         ratio = dt/dt_C
         check_dt_div_dt_Courant = check_change(s, ratio, &
            s% dt_div_dt_Courant_limit, s% dt_div_dt_Courant_hard_limit, &
            1, 'check_dt_div_dt_Courant', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dt_div_dt_Courant

         
      integer function check_dt_div_dt_cell_collapse(s, skip_hard_limit, dt, dt_limit_ratio)
         use star_utils, only: eval_min_cell_collapse_time
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: ratio, dt_timescale, relative_excess
         integer :: min_collapse_k, ierr
         include 'formats'
         check_dt_div_dt_cell_collapse = keep_going
         if (s% doing_relax) return
         dt_timescale = eval_min_cell_collapse_time( &
            s, 2, s% nz, min_collapse_k, ierr)
         if (ierr /= 0) return
         ratio = dt/dt_timescale
         !write(*,*) 'skip_hard_limit', skip_hard_limit
         !write(*,2) 'min collapse', min_collapse_k, dt_timescale, ratio, &
         !   s% dt_div_dt_cell_collapse_limit, s% dt_div_dt_cell_collapse_hard_limit
         check_dt_div_dt_cell_collapse = check_change(s, ratio, &
            s% dt_div_dt_cell_collapse_limit, s% dt_div_dt_cell_collapse_hard_limit, &
            1, 'check_dt_div_dt_cell_collapse', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dt_div_dt_cell_collapse

         
      integer function check_dt_div_dt_thermal(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: ratio, dt_timescale, relative_excess
         if (s% doing_relax) then
            check_dt_div_dt_thermal = keep_going
            return
         end if
         dt_timescale = s% kh_timescale*secyer
         ratio = dt/dt_timescale
         check_dt_div_dt_thermal = check_change(s, ratio, &
            s% dt_div_dt_thermal_limit, s% dt_div_dt_thermal_hard_limit, &
            1, 'check_dt_div_dt_thermal', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dt_div_dt_thermal

         
      integer function check_dt_div_dt_dynamic(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: ratio, dt_timescale, relative_excess
         if (s% doing_relax) then
            check_dt_div_dt_dynamic = keep_going
            return
         end if
         dt_timescale = s% dynamic_timescale
         ratio = dt/dt_timescale
         check_dt_div_dt_dynamic = check_change(s, ratio, &
            s% dt_div_dt_dynamic_limit, s% dt_div_dt_dynamic_hard_limit, &
            1, 'check_dt_div_dt_dynamic', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dt_div_dt_dynamic

         
      integer function check_dt_div_dt_acoustic(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: ratio, dt_timescale, relative_excess
         if (s% doing_relax) then
            check_dt_div_dt_acoustic = keep_going
            return
         end if
         dt_timescale = s% photosphere_acoustic_r
         ratio = dt/dt_timescale
         check_dt_div_dt_acoustic = check_change(s, ratio, &
            s% dt_div_dt_acoustic_limit, s% dt_div_dt_acoustic_hard_limit, &
            1, 'check_dt_div_dt_acoustic', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dt_div_dt_acoustic

         
      integer function check_dt_div_dt_mass_loss(s, skip_hard_limit, dt, dt_limit_ratio)
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         real(dp) :: ratio, dt_timescale, relative_excess
         if (s% doing_relax .or. s% mstar_dot >= 0d0) then
            check_dt_div_dt_mass_loss = keep_going
            return
         end if
         dt_timescale = -s% mstar/s% mstar_dot
         ratio = dt/dt_timescale
         check_dt_div_dt_mass_loss = check_change(s, ratio, &
            s% dt_div_dt_mass_loss_limit, s% dt_div_dt_mass_loss_hard_limit, &
            1, 'check_dt_div_dt_mass_loss', skip_hard_limit, dt_limit_ratio, relative_excess)
      end function check_dt_div_dt_mass_loss
            
      
      integer function check_dX_nuc_drop(s, skip_hard_limit, dt, dt_limit_ratio)
         use chem_def
         use rates_def, only: i_rate
         type (star_info), pointer :: s
         logical, intent(in) :: skip_hard_limit
         real(dp), intent(in) :: dt
         real(dp), intent(inout) :: dt_limit_ratio
         
         integer :: k, nz, max_k, max_j
         real(dp), pointer, dimension(:) :: sig
         real(dp) :: max_dx_nuc_drop, X_limit, A_limit, min_dt, limit
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         check_dX_nuc_drop = keep_going
         
         X_limit = s% dX_nuc_drop_min_X_limit
         A_limit = s% dX_nuc_drop_max_A_limit
         nz = s% nz
         sig => s% sig
         
         max_dx_nuc_drop = 0
         max_k = 0
         max_j = 0
         do k = 1, nz
            call do1(k)
         end do
         
         s% Tlim_dXnuc_drop_cell = max_k
         s% Tlim_dXnuc_drop_species = max_j

         if (s% dX_nuc_drop_hard_limit > 0 .and. (.not. skip_hard_limit) .and. &
               max_dx_nuc_drop > s% dX_nuc_drop_hard_limit) then
            if (s% report_all_dt_limits) then
               write(*, '(a30, i6, f20.10, 99e20.10)') &
                  'dX_nuc_drop_hard_limit ' // trim(chem_isos% name(s% chem_id(max_j))), &
                  max_k, max_dx_nuc_drop, s% dX_nuc_drop_hard_limit
            end if
            check_dX_nuc_drop= retry
            return
         end if
         limit = s% dX_nuc_drop_limit
         if (s% lnT(nz) >= ln10*9.45d0 .and. s% dX_nuc_drop_limit_at_high_T > 0) &
            limit = s% dX_nuc_drop_limit_at_high_T
         if (limit <= 0 .or. max_dx_nuc_drop <= 0) return
         
         if (dt < secyer*s% dX_nuc_drop_min_yrs_for_dt) return
         
         min_dt = secyer*s% dX_nuc_drop_min_yrs_for_dt
         dt_limit_ratio = min( &
            max_dx_nuc_drop/limit, &
            s% dt_limit_ratio_target*dt/min_dt)
         if (dt_limit_ratio <= s% dt_limit_ratio_target) then
            dt_limit_ratio = 0
         else if (s% report_all_dt_limits) then
            write(*,2) 'dt limit: dX_nuc_drop ' // trim(chem_isos% name(s% chem_id(max_j))), &
               max_k, dt_limit_ratio, max_dx_nuc_drop, limit, s% m(max_k)/Msun
         end if
         
         s% dX_nuc_drop_max_j = max_j
         s% dX_nuc_drop_max_k = max_k
         s% dX_nuc_drop_max_drop = max_dx_nuc_drop

         contains
         
         subroutine write_max(io)
            integer, intent(in) :: io
            if (.not. s% report_dX_nuc_drop_dt_limits) return
            write(io,'(a,i6,a20,i6,99(1x,a,1x,1pe10.3))') &
               'dt limit dX_nuc_drop:  model ', s% model_number, &
               '  ' // trim(chem_isos% name(s% chem_id(max_j))) // '  at cell', &
               max_k, '  drop', max_dx_nuc_drop, &
               '  before', s% xa(max_j,max_k)+max_dx_nuc_drop, &
               '  after', s% xa(max_j,max_k), &
               '  m', s% m(max_k)/Msun
         end subroutine write_max
         

         subroutine do1(k)
            integer, intent(in) :: k

            integer :: j, jj, ii
            real(dp) :: dx, dx_drop, dm, dt_dm, dx_burning, dx_inflow, dxdt_nuc
            real(dp) ::dx00, dxp1, sig00, sigp1, flux00, fluxp1

            include 'formats'
            
            dm = s% dm(k)  
            dt_dm = dt/dm

            do j=1, s% species

               if (chem_isos% W(s% chem_id(j)) > A_limit) then
                  if (dbg .and. k == 1387) &
                     write(*,2) 'dX_nuc_max_A_limit ' // trim(chem_isos% name(s% chem_id(j))), &
                        k, s% xa(j,k), chem_isos% W(s% chem_id(j)), A_limit
                  cycle
               end if

               if (chem_isos% Z(s% chem_id(j)) <= 2) then
                  cycle ! skip the little guys
               end if
            
               if (s% xa(j,k) < X_limit) then
                  if (dbg .and. k == 1387) &
                     write(*,2) &
                        'dX_nuc_drop_min_X_limit ' // trim(chem_isos% name(s% chem_id(j))), &
                        k, s% xa(j,k), X_limit
                  cycle
               end if
            
               dxdt_nuc = s% dxdt_nuc(j,k)
               if (dxdt_nuc >= 0) cycle
               
               dx_burning = dxdt_nuc*dt
               sig00 = sig(k)
         
               if (k < s% nz) then
                  sigp1 = sig(k+1)
               else
                  sigp1 = 0
               end if
         
               if (k > 1) then
                  dx00 = s% xa(j,k-1) - s% xa(j,k)
                  flux00 = -sig00*dx00
               else
                  flux00 = 0
               end if
         
               if (k < s% nz) then
                  dxp1 = s% xa(j,k) - s% xa(j,k+1)
                  fluxp1 = -sigp1*dxp1
               else
                  fluxp1 = 0
               end if

               dx_inflow = max(0d0, fluxp1, -flux00)*dt_dm
               dx_drop = -(dx_burning + dx_inflow) ! dx_burning < 0 for drop
               
               dx = s% xa_compare(j,k) - s% xa(j,k) ! the actual drop
               if (dx < dx_drop) dx_drop = dx
               
               if (dx_drop > max_dx_nuc_drop) then
                  max_dx_nuc_drop = dx_drop
                  max_k = k
                  max_j = j
               end if

            end do
         

         end subroutine do1
         
                       
      end function check_dX_nuc_drop


      integer function check_varcontrol_limit(s, dt_limit_ratio)
         type (star_info), pointer :: s
         real(dp), intent(inout) :: dt_limit_ratio
         
         real(dp) :: varcontrol, vc_target
         integer :: ierr
         
         include 'formats'
                  
         ierr = 0
         check_varcontrol_limit = keep_going
         
         varcontrol = eval_varcontrol(s, ierr)
         if (ierr /= 0) then
            check_varcontrol_limit = retry
            if (s% report_ierr) write(*, *) 'check_varcontrol_limit: eval_varcontrol ierr', ierr
            s% result_reason = nonzero_ierr
            return
         end if

         vc_target = max(1d-99,s% varcontrol_target)
         if (s% DUP_varcontrol_factor > 0 .and. s% TP_state == 2) &
            vc_target = vc_target * s% DUP_varcontrol_factor
            
         dt_limit_ratio = varcontrol/vc_target
 
         if (dt_limit_ratio > s% dt_limit_ratio_target .and. s% report_all_dt_limits) &
            write(*, '(a30, f20.10, 99e20.10)') 'varcontrol', &
               dt_limit_ratio, varcontrol, vc_target

         if (s% report_hydro_dt_info) then
            write(*, 1) 's% varcontrol_target', s% varcontrol_target
            write(*, 1) 's% DUP_varcontrol_factor', s% DUP_varcontrol_factor
            write(*, 2) 's% TP_state', s% TP_state
            write(*, 1) 'vc_target', vc_target
            write(*, 1) 'varcontrol', varcontrol
            write(*, 1) 'dt_limit_ratio', dt_limit_ratio
            write(*, *)
         end if
         
         if (dt_limit_ratio > s% varcontrol_dt_limit_ratio_hard_max) then
            write(*, '(a50, f20.10, 99e20.10)') 'varcontrol_dt_limit_ratio too large', &
               dt_limit_ratio, varcontrol, vc_target
            check_varcontrol_limit = retry
            return
         end if
         
      end function check_varcontrol_limit


      real(dp) function eval_varcontrol(s, ierr) result(varcontrol)
         use utils_lib, only: is_bad_num, alloc_iounit, free_iounit
         use chem_def
         use star_utils, only: std_write_internals_to_file
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: j, nterms, nvar_hydro, nz, k, kk, &
            skip1, skip2, i_lnPgas, i_lnd, i_E, iounit
         real(dp) :: sumj, sumvar, sumscales, sumterm(s% nvar)
         real(dp), pointer :: vc_data(:,:)
         logical :: dbg
         real(dp), parameter :: xscale_min = 1

         vc_data => null()
         
         include 'formats'
         
         ierr = 0
         varcontrol = 1d99
         dbg = (s% write_varcontrol_internals >= 0)
         nvar_hydro = s% nvar_hydro
         nz = s% nz
         
         if (dbg) then
            allocate(vc_data(nvar_hydro,nz),stat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in alloca in eval_varcontrol'
               return
            end if
         end if
         
         if (s% include_L_in_error_est) then
            skip1 = 0
         else
            skip1 = s% i_lum
         end if
         
         if (s% include_v_in_error_est) then
            skip2 = 0
         else
            skip2 = s% i_v
         end if
         
         i_lnd = s% i_lnd
         i_lnPgas = s% i_lnPgas
         i_E = s% i_E
         
         nterms = 0
         sumvar = 0
         sumscales = 0
         sumterm(:) = 0
         
         ! use differences in smoothed old and new to filter out high frequency noise.
         do j = 1, nvar_hydro
                  
            if (j == skip1 .or. &
                j == skip2 .or. &
                j == i_E) cycle
            
            nterms = nterms + nz
            do k = 3, nz-2
               sumj = abs(sum(s% xh(j,k-2:k+2)) - sum(s% xh_compare(j,k-2:k+2)))/5
               sumterm(j) = sumterm(j) + sumj
            end do
            sumterm(j) = sumterm(j) + &
               abs((2*s% xh(j,1) + s% xh(j,2)) - (2*s% xh_compare(j,1) + s% xh_compare(j,2)))/3 + &
               abs((2*s% xh(j,nz) + s% xh(j,nz-1)) - (2*s% xh_compare(j,nz) + s% xh_compare(j,nz-1)))/3
            k = 2
            sumj = abs(sum(s% xh(j,k-1:k+1)) - sum(s% xh_compare(j,k-1:k+1)))/3
            sumterm(j) = sumterm(j) + sumj
            k = nz-1
            sumj = abs(sum(s% xh(j,k-1:k+1)) - sum(s% xh_compare(j,k-1:k+1)))/3
               
            if (j == i_lnd) then 
               sumterm(j) = sumterm(j)/3
            end if
               
            if (j == i_lnPgas) then
               sumterm(j) = sumterm(j)/5
            end if
            
            sumvar = sumvar + sumterm(j)
            sumscales = sumscales + max(xscale_min, abs(s% xh_compare(j,1)))
            
         end do

         sumterm(:) = sumterm(:)/sumscales
         sumvar = sumvar/sumscales

         if (dbg) then
            call show_info            
            if (s% write_varcontrol_internals > 0) &
               call std_write_internals_to_file(s% id, s% write_varcontrol_internals)            
            call std_write_internals_to_file(s% id, 0)
            
            deallocate(vc_data)
            stop 'debug: timestep'
         end if
         
         varcontrol = sumvar/nterms
         
         contains
         
         subroutine show_info
            character (len=64) :: filename
            real(dp) :: newterm
            write(*,*)
            write(*, *) 'sumvar', sumvar
            write(*, *) 'nterms', nterms
            write(*, *) 'sumvar/nterms', sumvar/nterms
            write(*,*)
            do j=1, nvar_hydro
               if (j == skip1 .or. &
                   j == skip2) cycle
               write(*, '(a40, d26.16)') &
                  'varcontrol fraction for ' // trim(s% nameofvar(j)) // ' = ', sumterm(j)/sumvar
            end do
            write(*,*)
            do j=1, s% species
               if (sumterm(nvar_hydro + j) /= 0) &
                  write(*, '(a40, d26.16)') 'varcontrol fraction for ' // &
                        trim(chem_isos% name(s% chem_id(j))) // ' = ', &
                        sumterm(nvar_hydro + j)/sumvar
            end do
            if (s% write_varcontrol_internals >= 0) then
               iounit = alloc_iounit(ierr); if (ierr /= 0) return
               filename = 'plot_data/varcontrol.data'
               write(*,*) 'write ' // trim(filename)
               open(iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed to open file for varcontrol info'
                  call free_iounit(iounit)
                  return
               end if
               write(iounit, fmt='(a26)', advance='no') 'k'
               do j=1, nvar_hydro
               if (j == skip1 .or. &
                   j == skip2) cycle
                  write(iounit, fmt='(a26)', advance='no') trim(s% nameofvar(j))
               end do
               write(iounit,*)
               do k=1,nz
                  write(iounit, fmt='(i26)', advance='no') k
                  do j=1, nvar_hydro
                     if (j == skip1 .or. &
                         j == skip2) cycle
                     if (k <= 2) then
                        newterm = &
                           abs((2*s% xh(j,1) + s% xh(j,2)) - (2*s% xh_compare(j,1) + s% xh_compare(j,2)))/3
                     else if (k >= nz-1) then
                        newterm = &
                           abs((2*s% xh(j,nz) + s% xh(j,nz-1)) - (2*s% xh_compare(j,nz) + s% xh_compare(j,nz-1)))/3
                     else
                        newterm = abs(sum(s% xh(j,k-2:k+2)) - sum(s% xh_compare(j,k-2:k+2)))/5
                     end if
                     write(iounit, fmt='(d26.16)', advance='no') newterm
                  end do
                  write(iounit,*)
               end do
               close(iounit)
               call free_iounit(iounit)
            end if
         end subroutine show_info
         
         
      end function eval_varcontrol


      real(dp) function eval_truncation_ratio(s, ierr) result(error)
         use utils_lib, only: is_bad_num
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: i, j, nvar_hydro, nz, k, species, nterms, skip1, skip2
         real(dp) :: sum_xa_term, xh_terms(s% nvar_hydro), atol, rtol, &
            xa_terms(s% species), err_chem, err_hydro, &
            v, v_compare, v_scale, err1
         logical :: do_chem, do_max, trace
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         ierr = 0
         
         if (.not. associated(s% xh_compare)) return
         if (.not. associated(s% xa_compare)) return
         
         trace = s% trace_truncation_ratio
         do_max = s% eval_truncation_ratio_do_max
         error = 1d99
         nz = s% nz    
         species = s% species 
         nvar_hydro = s% nvar_hydro    
         xh_terms(:) = 0
         xa_terms(:) = 0
         atol = s% truncation_ratio_xh_atol
         rtol = s% truncation_ratio_xh_rtol
         
         if (s% include_L_in_error_est) then
            skip1 = 0
         else
            skip1 = s% i_lum
         end if
         
         if (s% include_v_in_error_est) then
            skip2 = 0
         else
            skip2 = s% i_v
         end if

         nterms = 0
         do j = 1, nvar_hydro
            if (j == skip1 .or. j == skip2) cycle
            if (j == s% i_lum) then
               v_scale = maxval(abs(s% xh(j,1:nz)))
               do k = 1, nz
                  v = s% xh(j,k)/v_scale
                  v_compare = s% xh_compare(j,k)/v_scale
                  err1 = abs(v - v_compare)/(atol + rtol*max(abs(v), abs(v_compare)))
                  if (do_max) then
                     if (err1 > xh_terms(j)) xh_terms(j) = err1
                  else
                     xh_terms(j) = xh_terms(j) + err1
                  end if
               end do
            else if (j == s% i_v) then
               v_scale = max(1d0,maxval(abs(s% xh(j,1:nz))))
               do k = 1, nz
                  v = s% xh(j,k)/v_scale
                  v_compare = s% xh_compare(j,k)/v_scale
                  err1 = abs(v - v_compare)/(atol + rtol*max(abs(v), abs(v_compare)))
                  if (do_max) then
                     if (err1 > xh_terms(j)) xh_terms(j) = err1
                  else
                     xh_terms(j) = xh_terms(j) + err1
                  end if
               end do
            else if (j == s% i_lnd) then
               do k = 1, nz
                  v = s% xh(j,k) + 14*ln10
                  v_compare = s% xh_compare(j,k) + 14*ln10
                  err1 = abs(v - v_compare)/(atol + rtol*max(abs(v), abs(v_compare)))
                  if (do_max) then
                     if (err1 > xh_terms(j)) xh_terms(j) = err1
                  else
                     xh_terms(j) = xh_terms(j) + err1
                  end if
               end do
            else
               do k = 1, nz
                  v = s% xh(j,k)
                  v_compare = s% xh_compare(j,k)
                  err1 = abs(v - v_compare)/(atol + rtol*max(abs(v), abs(v_compare)))
                  if (do_max) then
                     if (err1 > xh_terms(j)) xh_terms(j) = err1
                  else
                     xh_terms(j) = xh_terms(j) + err1
                  end if
               end do
            end if
            nterms = nterms + nz
         end do
         
         if (do_max) then
            err_hydro = maxval(xh_terms(1:nvar_hydro))
         else
            err_hydro = sum(xh_terms(1:nvar_hydro))/nterms
         end if
         
         do_chem = (s% do_burn .or. s% do_mix)
         if (.not. do_chem) then
            err_chem = 0
         else
            atol = s% truncation_ratio_xa_atol
            rtol = s% truncation_ratio_xa_rtol
            do k = 1, nz
               do j = 1, species
                  err1 = abs(s% xa(j,k) - s% xa_compare(j,k))/ &
                        (atol + rtol*max(abs(s% xa(j,k)), abs(s% xa_compare(j,k))))
                  if (do_max) then
                     if (err1 > xa_terms(j)) xa_terms(j) = err1
                  else
                     xa_terms(j) = xa_terms(j) + err1
                  end if
               end do
            end do
            if (do_max) then
               err_chem = maxval(xa_terms(1:species))
            else
               err_chem = sum(xa_terms(1:species))/(species*nz)
            end if
         end if

         !error = (err_hydro + err_chem)/2
         error = max(err_hydro, err_chem)

         if (trace .or. dbg) then
            write(*,*)
            do j=1,nvar_hydro
               if (j == skip1 .or. j == skip2) cycle
               if (do_max) then
                  err1 = xh_terms(j)
                  write(*, '(a40, i6, f26.16)') &
                     'max error for ' // trim(s% nameofvar(j)), &
                     s% model_number, err1
               else
                  err1 = xh_terms(j)/nz
                  write(*, '(a40, i6, f26.16)') &
                     'avg error for ' // trim(s% nameofvar(j)), &
                     s% model_number, err1
               end if
            end do
            write(*,*)
            if (do_chem) then
               do i=1,4
                  j = maxloc(xa_terms(1:species),dim=1)
                  if (do_max) then
                     err1 = xa_terms(j)
                     write(*, '(a40, i6, f26.16)') 'max error for ' // &
                        trim(s% nameofvar(nvar_hydro+j)), &
                        s% model_number, err1
                  else
                     err1 = xa_terms(j)/nz
                     write(*, '(a40, i6, f26.16)') 'avg error for ' // &
                        trim(s% nameofvar(nvar_hydro+j)), &
                        s% model_number, err1
                  end if
                  xa_terms(j) = -1
               end do
               write(*,*)
               write(*, '(a40, i6, f26.16)') 'error for structure', s% model_number, err_hydro
               write(*, '(a40, i6, f26.16)') 'error for abundances', s% model_number, err_chem
            end if
            write(*, '(a40, i6, f26.16)') 'estimated local truncation error', s% model_number, error
            write(*,*)
         end if
         
      end function eval_truncation_ratio


      subroutine filter_dt_next(s, order, dt_limit_ratio)
         ! H211b "low pass" controller.
         ! Soderlind & Wang, J of Computational and Applied Math 185 (2006) 225 – 243.
         use num_def
         use utils_lib, only:is_bad_num
         type (star_info), pointer :: s
         real(dp), intent(in) :: order, dt_limit_ratio
         
         real(dp) :: ratio, ratio_prev, limtr, dt_limit_ratio_target, &
            beta1, beta2, alpha2
         
         include 'formats'

         beta1 = 0.25d0/order
         beta2 = 0.25d0/order
         alpha2 = 0.25d0

         s% dt_limit_ratio = dt_limit_ratio
         dt_limit_ratio_target = s% dt_limit_ratio_target
         
         if (s% use_dt_low_pass_controller .and. &
               s% dt_limit_ratio_old > 0 .and. s% dt_old > 0 .and. &
               s% generations > 2) then ! use 2 values to do "low pass" controller
            ratio = limiter(dt_limit_ratio_target/dt_limit_ratio)
            ratio_prev = limiter(dt_limit_ratio_target/s% dt_limit_ratio_old)    
            limtr = limiter( &
               pow_cr(ratio,beta1) * pow_cr(ratio_prev,beta2) * pow_cr(s% dt/s% dt_old,-alpha2))
            s% dt_next = s% dt*limtr
            
            if (s% report_hydro_dt_info) then
               write(*,2) 'dt_limit_ratio_target', s% model_number, dt_limit_ratio_target
               write(*,2) 'dt_limit_ratio', s% model_number, dt_limit_ratio
               write(*,2) 's% dt_limit_ratio_old', s% model_number, s% dt_limit_ratio_old
               write(*,2) 'order', s% model_number, order
               write(*,2) 'ratio', s% model_number, ratio
               write(*,2) 'ratio_prev', s% model_number, ratio_prev
               write(*,2) 'limtr', s% model_number, limtr
               write(*,2) 's% dt_next', s% model_number, s% dt_next
               write(*,*)
            end if

         else ! no history available, so fall back to the 1st order controller
            s% dt_next = s% dt*dt_limit_ratio_target/dt_limit_ratio
         end if
         
         
         contains
         
         
         real(dp) function limiter(x)
            real(dp), intent(in) :: x
            real(dp), parameter :: kappa = 10 ! 2
            ! for x >= 0 and kappa = 2, limiter value is between 0.07 and 4.14
            ! for x >= 0 and kappa = 10, limiter value is between 0.003 and 16.7
            ! for x = 1, limiter = 1
            limiter = 1 + kappa*atan_cr((x-1)/kappa)
         end function limiter
         
         
      end subroutine filter_dt_next
         

      end module timestep


