! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module struct_burn_mix

      use star_private_def
      use const_def
      use dirk

      implicit none
      
      integer, parameter :: ode_nvar = 3, &
         ode_i_r = 1, ode_i_v = 2, ode_i_e = 3

      contains


      integer function do_struct_burn_mix( &
            s, skip_global_corr_coeff_limit, dt) 
         use mix_info, only: set_mixing_info, get_convection_sigmas
         use solve_hydro, only: &
            set_surf_info, set_tol_correction, do_hydro_converge
         use solve_mix, only: do_solve_mix
         use utils_lib, only:is_bad_num
         use rates_def, only: num_rvs
         use chem_def, only: num_categories, ih1, isi28, ine20
         
         type (star_info), pointer :: s
         logical, intent(in) :: skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt
         
         integer :: nz, nvar, species, ierr, j, k, ne20, k_bad, max_order
         logical :: do_chem
         real(dp) :: dt_1st_mix, dt_2nd_mix, &
             tol_correction_norm, tol_max_correction

         include 'formats'
         
         ierr = 0
         nz = s% nz
         species = s% species
         dt_1st_mix = 0d0
         dt_2nd_mix = 0d0

         if (s% use_other_before_struct_burn_mix) then
            call s% other_before_struct_burn_mix(s% id, dt, do_struct_burn_mix)
            if (do_struct_burn_mix /= keep_going) return
         end if
         
         do_struct_burn_mix = retry
         
         s% do_struct_hydro = .true.
         s% do_struct_thermo = .true.

         s% use_avg_burn_dxdt = .false.
         s% use_avg_mix_dxdt = .false.
         
         ne20 = s% net_iso(ine20)
         
         if (s% split_mixing_choice > 0) then
            stop 'split_mixing_choice > 0'
         end if
         
         if (s% split_mixing_choice == 0) then

            s% do_burn = (s% dxdt_nuc_factor > 0d0)
            s% do_mix = (s% mix_factor > 0d0)
            call save_pre_values(s, ierr)
            if (ierr /= 0) return               
            
         else
         
            s% do_burn = s% split_mix_do_burn
            s% do_mix = .false.     
            s% max_fixup_for_mix = 0
                
            select case (s% split_mixing_choice)
            case (-1) ! mix first
               dt_1st_mix = dt
               dt_2nd_mix = 0d0
            case (-2) ! mix last
               dt_1st_mix = 0d0
               dt_2nd_mix = dt
            case (-3) ! mix 1/2 first and 1/2 last
               dt_1st_mix = 0.5d0*dt
               dt_2nd_mix = dt_1st_mix
            case (-4) ! iterate mix+burn to get avg_mix_dxdt.
               ! then do_hydro_converge using avg_mix_dxdt to get avg_burn_dxdt
               dt_1st_mix = 0d0
               dt_2nd_mix = 0d0
            case (-5) ! iterate mix+burn to get avg_mix_dxdt.
               ! then do_hydro_converge using avg_mix_dxdt to get avg_burn_dxdt
               ! finally do mix using avg_burn_dxdt
               dt_1st_mix = 0d0
               dt_2nd_mix = dt
            case (-6)
               dt_1st_mix = 0d0
               dt_2nd_mix = 0d0
            case default
               write(*,2) 'invalid value for split_mixing_choice', &
                  s% split_mixing_choice
               stop 'struct_burn_mix'
            end select
                        
            if (dt_1st_mix > 0d0) then
              
               call get_convection_sigmas(s, dt, ierr)
               if (ierr /= 0) then
                  if (s% report_ierr) write(*,*) 'get_convection_sigmas failed'
                  return
               end if
                           
               do k=1,nz
                  do j=1,species
                     s% xa_pre(j,k) = s% xa(j,k)
                  end do
               end do  

               do_struct_burn_mix = do_solve_mix(s, dt_1st_mix, s% species, 1, 1)
               if (do_struct_burn_mix /= keep_going) return

               if (s% trace_k > 0 .and. s% trace_k <= s% nz) then
                  do j=1,s% species
                     write(*,4) 'after 1st do_solve_mix xa(j)', &
                        s% model_number, s% trace_k, j, s% xa(j,s% trace_k)
                  end do
               end if
               
            end if
               
            call save_pre_values(s, ierr)
            if (ierr /= 0) return
            
            if (s% split_mixing_choice <= -4) then
               stop 'split_mixing_choice <= -4'
            else if (.not. s% split_mix_do_burn) then
               stop '.not. s% split_mix_do_burn'
            end if
                        
         end if
               
         if (s% doing_first_model_of_run) then
            if (s% i_lum /= 0) then
               s% L_phot_old = s% xh(s% i_lum,1)/Lsun
            else
               s% L_phot_old = 0
            end if
         end if
         
         if (s% do_mix) then
            call get_convection_sigmas(s, dt, ierr)
            if (ierr /= 0) then
               if (s% report_ierr) write(*,*) 'get_convection_sigmas failed'
               return
            end if
         end if
         
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! no chem => just do structure
            nvar = s% nvar_hydro
            !write(*,2) 'no chem => just do structure nvar', nvar
         end if
         
         s% xh_compare => s% xh_old
         s% xa_compare => s% xa_old
         
         s% ebdf_order = &
            max(s% min_ebdf_order, min(s% max_ebdf_order, s% ebdf_order))
         
         if (s% ebdf_order >= 1) then
            if (s% use_sdirk_instead_of_ebdf) then
               select case (s% ebdf_order)
               case (3)
                  do_struct_burn_mix = do_sdirk32( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (5)
                  do_struct_burn_mix = do_sdirk54( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case default
                  write(*,2) 'bad s% ebdf_order for sdirk', s% ebdf_order
                  stop 1
               end select
            else if (s% use_symplectic_instead_of_ebdf) then
               select case (s% ebdf_order)
               case (1)
                  do_struct_burn_mix = do_symplectic1( &
                     s, s% newton_itermin, nvar, .true., do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (2)
                  do_struct_burn_mix = do_symplectic2( &
                     s, s% newton_itermin, nvar, .true., do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (3)
                  do_struct_burn_mix = do_symplectic_3stage( &
                     s, s% newton_itermin, nvar, .true., do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (4)
                  do_struct_burn_mix = do_symplectic4( &
                     s, s% newton_itermin, nvar, .true., do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case default
                  write(*,2) 'bad s% ebdf_order for symplectic', s% ebdf_order
                  stop 1
               end select
            else 
               select case (s% ebdf_order)
               case (1)
                  do_struct_burn_mix = do_bdf1( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (2)
                  do_struct_burn_mix = do_ebdf2( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (3)
                  do_struct_burn_mix = do_ebdf3( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (4)
                  do_struct_burn_mix = do_ebdf4( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (5)
                  do_struct_burn_mix = do_ebdf5( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (6)
                  do_struct_burn_mix = do_ebdf6( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case (7)
                  do_struct_burn_mix = do_ebdf7( &
                     s, s% newton_itermin, nvar, do_chem, &
                     skip_global_corr_coeff_limit, dt, max_order)
               case default
                  write(*,2) 'bad s% ebdf_order', s% ebdf_order
                  stop 1
               end select
            end if
            s% ebdf_stage = -1
            if (max_order < s% ebdf_order .and. max_order >= s% min_ebdf_order) then
               write(*,4) 'reduce edbf order', max_order, s% ebdf_order, s% model_number
               s% ebdf_order = max_order
               s% ebdf_hold = s% model_number + &
                  max(1, s% steps_before_try_higher_ebdf_order)
               s% startup_increment_ebdf_order = .false.
            end if
         else
            call set_tol_correction(s, s% T(s% nz), &
               tol_correction_norm, tol_max_correction)
            call set_surf_info(s)
            do_struct_burn_mix = &
               do_hydro_converge( &
                  s, s% newton_itermin, nvar, skip_global_corr_coeff_limit, &
                  tol_correction_norm, tol_max_correction, dt)
            call store_results_by_stage(s,1)
            call store_bdf_average_results(s,1)
         end if
               
         if (do_struct_burn_mix /= keep_going) return

         if (s% trace_k > 0 .and. s% trace_k <= s% nz) then
            do j=1,s% species
               write(*,4) 'after do_hydro_converge xa(j)', &
                  s% model_number, s% trace_k, j, s% xa(j,s% trace_k)
            end do
         end if

         if (dt_2nd_mix > 0d0) then
         
            if (s% split_mixing_choice > -4) then
               ! set the abundances from the end of struct+burn
               do k=1,nz
                  do j=1,species
                     s% xa_pre(j,k) = s% xa(j,k)
                  end do
               end do
            end if
            
            if (s% reset_mixing_info_before_final_mix) then
               call set_mixing_info(s, ierr)
               if (ierr /= 0) then
                  if (s% report_ierr) write(*,*) 'set_mixing_info failed'
                  return
               end if
               call get_convection_sigmas(s, dt, ierr)
               if (ierr /= 0) then
                  if (s% report_ierr) write(*,*) 'get_convection_sigmas failed'
                  return
               end if
            end if
            
            do_struct_burn_mix = do_solve_mix(s, dt_2nd_mix, s% species, 1, 1)
            if (do_struct_burn_mix /= keep_going) return

            if (s% trace_k > 0 .and. s% trace_k <= s% nz) then
               do j=1,s% species
                  write(*,4) 'after 2nd do_solve_mix xa(j)', &
                     s% model_number, s% trace_k, j, s% xa(j,s% trace_k)
               end do
            end if

         end if
         
         do_struct_burn_mix = do_smooth_conv_bdy_and_mix_omega(s,dt)

         if (s% trace_k > 0 .and. s% trace_k <= s% nz) then
            do j=1,s% species
               write(*,4) 'after do_smooth_conv_bdy_and_mix_omega xa(j)', &
                  s% model_number, s% trace_k, j, s% xa(j,s% trace_k)
            end do
         end if

         if (s% use_other_after_struct_burn_mix) &
            call s% other_after_struct_burn_mix(s% id, dt, do_struct_burn_mix)
         
         
         contains
         
         
         subroutine do_alloc(ierr)
            use alloc, only: non_crit_do1_alloc_if_necessary
            integer, intent(out) :: ierr
            integer :: sz
            sz = nz*species
            ierr = 0
            call non_crit_do1_alloc_if_necessary( &
               s, s% avg_burn_dxdt1, sz, 'do_struct_burn_mix', ierr)
            if (ierr /= 0) return            
            s% avg_burn_dxdt(1:species,1:nz) => s% avg_burn_dxdt1(1:sz)
            call non_crit_do1_alloc_if_necessary( &
               s, s% avg_mix_dxdt1, sz, 'do_struct_burn_mix', ierr)
            if (ierr /= 0) return            
            s% avg_mix_dxdt(1:species,1:nz) => s% avg_mix_dxdt1(1:sz)
         end subroutine do_alloc


      end function do_struct_burn_mix

      
      subroutine save_pre_values(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr         
         integer :: k, j
         include 'formats'    
         ierr = 0    
         do k=1,s% nz
            do j=1,s% nvar_hydro
               s% xh_pre(j,k) = s% xh(j,k)
            end do
         end do    
         do k=1,s% nz
            do j=1,s% species
               s% xa_pre(j,k) = s% xa(j,k)
            end do
         end do    
      end subroutine save_pre_values
      
      
      integer function do_smooth_conv_bdy_and_mix_omega(s, dt)
         use solve_omega_mix, only: do_solve_omega_mix
         use mix_smoothing, only: smooth_newly_non_conv

         type (star_info), pointer :: s
         real(dp), intent(in) :: dt
         
         integer :: ierr
      
         do_smooth_conv_bdy_and_mix_omega = keep_going
         
         if (s% smooth_convective_bdy) then
            ierr = 0
            call smooth_newly_non_conv(s, ierr) ! do this after do_struct_burn_mix
            if (ierr /= 0) then
               if (s% report_ierr) then
                  write(*, *) 'smooth_newly_non_conv ierr: retry', s% model_number
               end if
               do_smooth_conv_bdy_and_mix_omega = retry
               s% result_reason = nonzero_ierr
               return
            end if
         end if
      
         if (s% rotation_flag) then
            if (s% premix_omega) then
               do_smooth_conv_bdy_and_mix_omega = do_solve_omega_mix(s, 0.5d0*dt)
            else
               do_smooth_conv_bdy_and_mix_omega = do_solve_omega_mix(s, dt)
            end if
            if (do_smooth_conv_bdy_and_mix_omega /= keep_going) return
         end if      
      
      end function do_smooth_conv_bdy_and_mix_omega
      
      
      subroutine eval_order_terms(s, &
            xhm0, xam0, xhm1, xam1, xhm2, xam2, xhm3, xam3, &
            xhm4, xam4, xhm5, xam5, xhm6, xam6, xhm7, xam7, &
            term1, term2, term3, term4, term5, term6, term7, &
            max_order)
         ! xhm0, xam0 are for last bdf stage of step
         ! xhm1, xam1 are for stage just before xhm0, xam0
         ! xhm2, xam2 are for stage just before xhm1, xam1
         ! etc
         ! term1 is for 1st difference, using xm0 xm1
         ! term2 is for 2nd difference, using xm0 xm1 xm2
         ! term3 is for 3rd difference, using xm0 xm1 xm2 xm3
         ! etc
         type (star_info), pointer :: s
         real(dp), pointer, dimension(:,:) :: &
            xhm0, xam0, xhm1, xam1, xhm2, xam2, xhm3, xam3, &
            xhm4, xam4, xhm5, xam5, xhm6, xam6, xhm7, xam7
         real(dp), intent(out) :: &
            term1, term2, term3, term4, term5, term6, term7
         integer, intent(out) :: max_order
                 
         integer :: j, k, nz, skip1, skip2
         logical :: do2, do3, do4, do5, do6, do7
         
         include 'formats'
         
         nz = s% nz
         do2 = associated(xhm2)
         do3 = associated(xhm3)
         do4 = associated(xhm4)
         do5 = associated(xhm5)
         do6 = associated(xhm6)
         do7 = associated(xhm7)
         
         if (s% include_L_in_error_est) then
            skip1 = 0
         else
            skip1 = s% i_lum
         end if         
         if (s% include_v_in_error_est) then
            skip2 = 0
         else
            skip2 = s% i_v
         end if     
             
         term1 = 0
         term2 = 0
         term3 = 0
         term4 = 0
         term5 = 0
         term6 = 0
         term7 = 0
         
         do j = 1, s% nvar_hydro
            if (j == skip1 .or. j == skip2) cycle
            do k = 1, nz
               term1 = term1 + abs( &
                  xhm0(j,k) - xhm1(j,k))
               if (.not. do2) cycle
               term2 = term2 + abs( &
                  xhm0(j,k) - 2d0*xhm1(j,k) + xhm2(j,k))
               if (.not. do3) cycle
               term3 = term3 + abs( &
                  xhm0(j,k) - 3d0*xhm1(j,k) + 3d0*xhm2(j,k) - xhm3(j,k))
               if (.not. do4) cycle
               term4 = term4 + abs( &
                  xhm0(j,k) - 4d0*xhm1(j,k) + 6d0*xhm2(j,k) - 4d0*xhm3(j,k) + xhm4(j,k))
               if (.not. do5) cycle
               term5 = term5 + abs( &
                  xhm0(j,k) - 5d0*xhm1(j,k) + 10d0*xhm2(j,k) - 10d0*xhm3(j,k) &
                     + 5d0*xhm4(j,k) - xhm5(j,k))
               if (.not. do6) cycle
               term6 = term6 + abs( &
                  xhm0(j,k) - 6d0*xhm1(j,k) + 15d0*xhm2(j,k) - 20d0*xhm3(j,k) &
                     + 15d0*xhm4(j,k) - 6d0*xhm5(j,k) + xhm6(j,k))
               if (.not. do7) cycle
               term7 = term7 + abs( &
                  xhm0(j,k) - 7d0*xhm1(j,k) + 21d0*xhm2(j,k) - 35d0*xhm3(j,k) &
                     + 35d0*xhm4(j,k) - 21d0*xhm5(j,k) + 7d0*xhm6(j,k) - xhm7(j,k))
            end do
         end do        

         if (s% do_burn .or. s% do_mix) then
            do k = 1, nz
               do j = 1, s% species
                  term1 = term1 + abs( &
                     xam0(j,k) - xam1(j,k))
                  if (.not. do2) cycle
                  term2 = term2 + abs( &
                     xam0(j,k) - 2d0*xam1(j,k) + xam2(j,k))
                  if (.not. do3) cycle
                  term3 = term3 + abs( &
                     xam0(j,k) - 3d0*xam1(j,k) + 3d0*xam2(j,k) - xam3(j,k))
                  if (.not. do4) cycle
                  term4 = term4 + abs( &
                     xam0(j,k) - 4d0*xam1(j,k) + 6d0*xam2(j,k) &
                        - 4d0*xam3(j,k) + xam4(j,k))
                  if (.not. do5) cycle
                  term5 = term5 + abs( &
                     xam0(j,k) - 5d0*xam1(j,k) + 10d0*xam2(j,k) - 10d0*xam3(j,k) &
                        + 5d0*xam4(j,k) - xam5(j,k))
                  if (.not. do6) cycle
                  term6 = term6 + abs( &
                     xam0(j,k) - 6d0*xam1(j,k) + 15d0*xam2(j,k) - 20d0*xam3(j,k) &
                        + 15d0*xam4(j,k) - 6d0*xam5(j,k) + xam6(j,k))
                  if (.not. do7) cycle
                  term7 = term7 + abs( &
                     xam0(j,k) - 7d0*xam1(j,k) + 21d0*xam2(j,k) - 35d0*xam3(j,k) &
                        + 35d0*xam4(j,k) - 21d0*xam5(j,k) + 7d0*xam6(j,k) - xam7(j,k))
               end do
            end do
         end if
         
         if (term2 > term1*s% factor_for_ebdf_order_tests) then
            max_order = 1
         else if (term3 > term2*s% factor_for_ebdf_order_tests) then
            max_order = 2
         else if (term4 > term3*s% factor_for_ebdf_order_tests) then
            max_order = 3
         else if (term5 > term4*s% factor_for_ebdf_order_tests) then
            max_order = 4
         else if (term6 > term5*s% factor_for_ebdf_order_tests) then
            max_order = 5
         else if (term7 > term6*s% factor_for_ebdf_order_tests) then
            max_order = 6
         else
            max_order = 7
         end if
         
         max_order = min(s% ebdf_order, max(s% min_ebdf_order, max_order))
         
         if (.not. s% trace_truncation_ratio) return
         
         write(*,2) 'eval_order_terms: max_order, factor_for_ebdf_order_tests', &
            max_order, s% factor_for_ebdf_order_tests
         if (term2 /= 0) write(*,2) 'term1/term2', s% model_number, term1/term2
         if (term3 /= 0) write(*,2) 'term2/term3', s% model_number, term2/term3
         if (term4 /= 0) write(*,2) 'term3/term4', s% model_number, term3/term4
         if (term5 /= 0) write(*,2) 'term4/term5', s% model_number, term4/term5
         if (term6 /= 0) write(*,2) 'term5/term6', s% model_number, term5/term6
         if (term7 /= 0) write(*,2) 'term6/term7', s% model_number, term6/term7

      end subroutine eval_order_terms
         

      integer function do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs, &
            h, tol_correction_norm, tol_max_correction, &
            xh0, xa0, xh1, xa1, xh1_1, xa1_1, num_iters)
         use solve_hydro, only: do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar, nz, nvar_hydro, species
         logical, intent(in) :: &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs
         real(dp), intent(in) :: h, tol_correction_norm, tol_max_correction
         real(dp), pointer, dimension(:,:) :: xh0, xa0, xh1, xa1
         real(dp), pointer, dimension(:) :: xh1_1, xa1_1
         integer, intent(out) :: num_iters    
         integer :: k, j, ierr
         real(dp) :: dt_stage      
         include 'formats'  
         ierr = 0
         do_stage_bdf2 = terminate
         if (redo_mix_coeffs) then
            call ebdf_redo_mix_coeffs(s,ierr)
            if (ierr /= 0) return
         end if
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            xh1, xa1, xh1_1, xa1_1, ierr)
         if (ierr /= 0) return         
         do k=1,nz
            do j=1,nvar_hydro
               xh1(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = (4d0*xh1(j,k) - xh0(j,k))/3d0
            end do
            if (.not. do_chem) cycle
            do j=1,species
               xa1(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = (4d0*xa1(j,k) - xa0(j,k))/3d0
            end do
         end do
         dt_stage = (2d0/3d0)*h
         s% dVARdot_dVAR = 1/dt_stage         
         do_stage_bdf2 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_stage_bdf2 /= keep_going) return   
         num_iters = s% num_newton_iterations
      end function do_stage_bdf2
         

      integer function do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs, &
            h, tol_correction_norm, tol_max_correction, &
            xh0, xa0, xh1, xa1, xh2, xa2, xh2_1, xa2_1, num_iters)
         use solve_hydro, only: do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar, nz, nvar_hydro, species
         logical, intent(in) :: &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs
         real(dp), intent(in) :: h, tol_correction_norm, tol_max_correction
         real(dp), pointer, dimension(:,:) :: xh0, xa0, xh1, xa1, xh2, xa2
         real(dp), pointer, dimension(:) :: xh2_1, xa2_1
         integer, intent(out) :: num_iters    
         integer :: k, j, ierr
         real(dp) :: dt_stage      
         include 'formats'  
         ierr = 0
         do_stage_bdf3 = terminate
         if (redo_mix_coeffs) then
            call ebdf_redo_mix_coeffs(s,ierr)
            if (ierr /= 0) return
         end if
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            xh2, xa2, xh2_1, xa2_1, ierr)
         if (ierr /= 0) return         
         do k=1,nz
            do j=1,nvar_hydro
               xh2(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  (18d0*xh2(j,k) - 9d0*xh1(j,k) + 2d0*xh0(j,k))/11d0
            end do
            if (.not. do_chem) cycle
            do j=1,species
               xa2(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  (18d0*xa2(j,k) - 9d0*xa1(j,k) + 2d0*xa0(j,k))/11d0
            end do
         end do
         dt_stage = (6d0/11d0)*h
         s% dVARdot_dVAR = 1/dt_stage         
         do_stage_bdf3 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_stage_bdf3 /= keep_going) return   
         num_iters = s% num_newton_iterations
      end function do_stage_bdf3
      
      
      integer function do_stage_bdf4( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs, &
            h, tol_correction_norm, tol_max_correction, &
            xh0, xa0, xh1, xa1, xh2, xa2, xh3, xa3, xh3_1, xa3_1, num_iters)
         use solve_hydro, only: do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar, nz, nvar_hydro, species
         logical, intent(in) :: &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs
         real(dp), intent(in) :: h, tol_correction_norm, tol_max_correction
         real(dp), pointer, dimension(:,:) :: &
            xh0, xa0, xh1, xa1, xh2, xa2, xh3, xa3
         real(dp), pointer, dimension(:) :: xh3_1, xa3_1
         integer, intent(out) :: num_iters    
         integer :: k, j, ierr
         real(dp) :: dt_stage      
         include 'formats'  
         ierr = 0
         do_stage_bdf4 = terminate
         if (redo_mix_coeffs) then
            call ebdf_redo_mix_coeffs(s,ierr)
            if (ierr /= 0) return
         end if
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            xh3, xa3, xh3_1, xa3_1, ierr)
         if (ierr /= 0) return         
         do k=1,nz
            do j=1,nvar_hydro
               xh3(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  (48d0*xh3(j,k) - 36d0*xh2(j,k) + &
                        16d0*xh1(j,k) - 3d0*xh0(j,k))/25d0
            end do
            if (.not. do_chem) cycle
            do j=1,species
               xa3(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  (48d0*xa3(j,k) - 36d0*xa2(j,k) + &
                        16d0*xa1(j,k) - 3d0*xa0(j,k))/25d0
            end do
         end do
         dt_stage = (12d0/25d0)*h
         s% dVARdot_dVAR = 1/dt_stage         
         do_stage_bdf4 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_stage_bdf4 /= keep_going) return   
         num_iters = s% num_newton_iterations
      end function do_stage_bdf4
      
      
      integer function do_stage_bdf5( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs, &
            h, tol_correction_norm, tol_max_correction, &
            xh0, xa0, xh1, xa1, xh2, xa2, xh3, xa3, &
            xh4, xa4, xh4_1, xa4_1, num_iters)
         use solve_hydro, only: do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar, nz, nvar_hydro, species
         logical, intent(in) :: &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs
         real(dp), intent(in) :: h, tol_correction_norm, tol_max_correction
         real(dp), pointer, dimension(:,:) :: &
            xh0, xa0, xh1, xa1, xh2, xa2, xh3, xa3, xh4, xa4
         real(dp), pointer, dimension(:) :: xh4_1, xa4_1
         integer, intent(out) :: num_iters    
         integer :: k, j, ierr
         real(dp) :: dt_stage      
         include 'formats'  
         ierr = 0
         do_stage_bdf5 = terminate
         if (redo_mix_coeffs) then
            call ebdf_redo_mix_coeffs(s,ierr)
            if (ierr /= 0) return
         end if
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            xh4, xa4, xh4_1, xa4_1, ierr)
         if (ierr /= 0) return         
         do k=1,nz
            do j=1,nvar_hydro
               xh4(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  (300d0*xh4(j,k) - 300d0*xh3(j,k) + &
                        200d0*xh2(j,k) - 75d0*xh1(j,k) + 12d0*xh0(j,k))/137d0
            end do
            if (.not. do_chem) cycle
            do j=1,species
               xa4(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  (300d0*xa4(j,k) - 300d0*xa3(j,k) + &
                        200d0*xa2(j,k) - 75d0*xa1(j,k) + 12d0*xa0(j,k))/137d0
            end do
         end do
         dt_stage = (60d0/137d0)*h
         s% dVARdot_dVAR = 1/dt_stage         
         do_stage_bdf5 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_stage_bdf5 /= keep_going) return   
         num_iters = s% num_newton_iterations
      end function do_stage_bdf5
      
      
      integer function do_stage_bdf6( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs, &
            h, tol_correction_norm, tol_max_correction, &
            xh0, xa0, xh1, xa1, xh2, xa2, xh3, xa3, &
            xh4, xa4, xh5, xa5, xh5_1, xa5_1, num_iters)
         use solve_hydro, only: do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar, nz, nvar_hydro, species
         logical, intent(in) :: &
            do_chem, skip_global_corr_coeff_limit, redo_mix_coeffs
         real(dp), intent(in) :: h, tol_correction_norm, tol_max_correction
         real(dp), pointer, dimension(:,:) :: &
            xh0, xa0, xh1, xa1, xh2, xa2, xh3, xa3, xh4, xa4, xh5, xa5
         real(dp), pointer, dimension(:) :: xh5_1, xa5_1
         integer, intent(out) :: num_iters    
         integer :: k, j, ierr
         real(dp) :: dt_stage      
         include 'formats'  
         ierr = 0
         do_stage_bdf6 = terminate
         if (redo_mix_coeffs) then
            call ebdf_redo_mix_coeffs(s,ierr)
            if (ierr /= 0) return
         end if
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            xh5, xa5, xh5_1, xa5_1, ierr)
         if (ierr /= 0) return         
         do k=1,nz
            do j=1,nvar_hydro
               xh5(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  (360d0*xh5(j,k) - 450d0*xh4(j,k) + 400d0*xh3(j,k) - &
                        225d0*xh2(j,k) + 72d0*xh1(j,k) - 10d0*xh0(j,k))/147d0
            end do
            if (.not. do_chem) cycle
            do j=1,species
               xa5(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  (360d0*xa5(j,k) - 450d0*xa4(j,k) + 400d0*xa3(j,k) - &
                        225d0*xa2(j,k) + 72d0*xa1(j,k) - 10d0*xa0(j,k))/147d0
            end do
         end do
         dt_stage = (60d0/147d0)*h
         s% dVARdot_dVAR = 1/dt_stage         
         do_stage_bdf6 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_stage_bdf6 /= keep_going) return   
         num_iters = s% num_newton_iterations
      end function do_stage_bdf6


      integer function do_bdf1( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            do_hydro_converge, set_tol_correction, set_surf_info
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical,intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         
         include 'formats'
         
         do_bdf1 = terminate
         max_order = 1
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         h = dt
         
         s% bdf_stage = 1
         do_bdf1 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_bdf1 /= keep_going) return   
         call store_results_by_stage(s,1)
         call store_bdf_average_results(s,1)
         
         ! for local truncation error estimates, compare to x0
         s% xh_compare => s% xh0
         if (do_chem) then
            s% xa_compare => s% xa0
         end if
                  
      end function do_bdf1


      integer function do_bdf2( &
            s, itermin, nvar, do_chem, &
            skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            do_hydro_converge, set_tol_correction, set_surf_info
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical,intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         do_bdf2 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         ! order 2
         ! ss_bfd2: 2 stage bdf1, bdf2. h = dt/2
         
         h = dt/2
         
         ! stage 1
         s% bdf_stage = 1
         do_bdf2 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_bdf2 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2
         s% bdf_stage = 2
         do_bdf2 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_final_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_bdf2 /= keep_going) return   
         call store_results_by_stage(s,2)
         call store_bdf_average_results(s,2)
         
         if (s% min_ebdf_order == 2 .and. .not. s% trace_truncation_ratio) then
            max_order = 2
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh1, s% xa1, &
               s% xh0, s% xa0, null(), null(), &
               null(), null(), null(), null(), &
               null(), null(), null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if
         
         ! for local truncation error estimates, compare to extrapolated solution, 2*x1 - x0
         do k = 1,nz
            do j=1, nvar_hydro
               s% xh1(j,k) = 2d0*s% xh1(j,k) - s% xh0(j,k)
            end do
         end do
         s% xh_compare => s% xh1
         if (do_chem) then
            do k = 1,nz
               do j=1, species
                  s% xa1(j,k) = 2d0*s% xa1(j,k) - s% xa0(j,k)
               end do
            end do
            s% xa_compare => s% xa1
         end if
                  
      end function do_bdf2


      integer function do_bdf3( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            do_hydro_converge, set_tol_correction, set_surf_info
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical,intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         do_bdf3 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         ! order 3
         ! ss_bfd3: 3 stage bdf1, bdf2, bdf3. h = dt/3
         
         h = dt/3
         
         ! stage 1
         s% bdf_stage = 1
         do_bdf3 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_bdf3 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2
         s% bdf_stage = 2
         do_bdf3 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_bdf3 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf3)
         s% bdf_stage = 3
         do_bdf3 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_final_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_bdf3 /= keep_going) return   
         call store_results_by_stage(s,3)
         call store_bdf_average_results(s,3)
         
         if (s% min_ebdf_order == 3 .and. .not. s% trace_truncation_ratio) then
            max_order = 3
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh2, s% xa2, &
               s% xh1, s% xa1, null(), null(), &
               null(), null(), null(), null(), &
               null(), null(), null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if
         
         ! for local truncation error estimates,
         ! compare to extrapolated solution, 3*x2 - 3*x1 + x0
         do k = 1,nz
            do j=1, nvar_hydro
               s% xh2(j,k) = 3d0*(s% xh2(j,k) - s% xh1(j,k)) + s% xh0(j,k)
            end do
         end do
         s% xh_compare => s% xh2
         if (do_chem) then
            do k = 1,nz
               do j=1, species
                  s% xa2(j,k) = 3d0*(s% xa2(j,k) - s% xa1(j,k)) + s% xa0(j,k)
               end do
            end do
            s% xa_compare => s% xa2
         end if
                  
      end function do_bdf3


      integer function do_bdf4( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            do_hydro_converge, set_tol_correction, set_surf_info
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical,intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         do_bdf4 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         ! order 4
         ! ss_bfd3: 4 stage bdf1, bdf2, bdf3, bdf4. h = dt/4
         
         h = dt/4
         
         ! stage 1
         s% bdf_stage = 1
         do_bdf4 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_bdf4 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2
         s% bdf_stage = 2
         do_bdf4 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_bdf4 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf3)
         s% bdf_stage = 3
         do_bdf4 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_bdf4 /= keep_going) return   
         call store_results_by_stage(s,3)
         
         ! stage 4 (bdf4)
         s% bdf_stage = 4
         do_bdf4 = do_stage_bdf4( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_final_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh3, s% xa3, &
            s% xh3_1, s% xa3_1, s% num_newton_iters_stage4)
         if (do_bdf4 /= keep_going) return   
         call store_results_by_stage(s,4)
         call store_bdf_average_results(s,4)
         
         if (s% min_ebdf_order == 4 .and. .not. s% trace_truncation_ratio) then
            max_order = 4
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh3, s% xa3, &
               s% xh2, s% xa2, s% xh1, s% xa1, &
               s% xh0, s% xa0, null(), null(), &
               null(), null(), null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if
         
         ! for local truncation error estimates, compare to extrapolated solution, 
         ! 4*x3 - 6*x2 + 4*x1 - x0
         do k = 1,nz
            do j=1, nvar_hydro
               s% xh3(j,k) = &
                  4d0*s% xh3(j,k) - 6d0*s% xh2(j,k) + 4d0*s% xh1(j,k) - s% xh0(j,k)
            end do
         end do
         s% xh_compare => s% xh3
         if (do_chem) then
            do k = 1,nz
               do j=1, species
                  s% xa3(j,k) = &
                     4d0*s% xa3(j,k) - 6d0*s% xa2(j,k) + 4d0*s% xa1(j,k) - s% xa0(j,k)
               end do
            end do
            s% xa_compare => s% xa3
         end if
                  
      end function do_bdf4


      integer function do_bdf5( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            do_hydro_converge, set_tol_correction, set_surf_info
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical,intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         do_bdf5 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         ! order 4
         ! ss_bfd3: 4 stage bdf1, bdf2, bdf3, bdf4. h = dt/4
         
         h = dt/4
         
         ! stage 1
         s% ebdf_stage = 1
         do_bdf5 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_bdf5 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2
         s% ebdf_stage = 2
         do_bdf5 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_bdf5 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf3)
         s% ebdf_stage = 3
         do_bdf5 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_bdf5 /= keep_going) return   
         call store_results_by_stage(s,3)
         
         ! stage 4 (bdf4)
         s% ebdf_stage = 4
         do_bdf5 = do_stage_bdf4( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh3_1, s% xa3_1, &
            s% num_newton_iters_stage4)
         if (do_bdf5 /= keep_going) return   
         call store_results_by_stage(s,4)
         
         ! stage 5 (bdf5)
         s% ebdf_stage = 5
         do_bdf5 = do_stage_bdf5( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_final_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh4, s% xa4, s% xh4_1, s% xa4_1, &
            s% num_newton_iters_stage5)
         if (do_bdf5 /= keep_going) return   
         call store_results_by_stage(s,5)
         call store_bdf_average_results(s,5)
         
         if (s% min_ebdf_order == 5 .and. .not. s% trace_truncation_ratio) then
            max_order = 5
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh4, s% xa4, &
               s% xh3, s% xa3, s% xh2, s% xa2, &
               s% xh1, s% xa1, s% xh0, s% xa0, &
               null(), null(), null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if
         
         ! for local truncation error estimates, compare to extrapolated solution, 
         ! x0 - 5 x1 + 10 x2 - 10 x3 + 5 x4
         do k = 1,nz
            do j=1, nvar_hydro
               s% xh4(j,k) = &
                  5d0*s% xh4(j,k) - 10d0*s% xh3(j,k) + &
                     10d0*s% xh2(j,k) - 5d0*s% xh1(j,k) + s% xh0(j,k)
            end do
         end do
         s% xh_compare => s% xh4
         if (do_chem) then
            do k = 1,nz
               do j=1, species
                  s% xa4(j,k) = &
                     5d0*s% xa4(j,k) - 10d0*s% xa3(j,k) + &
                        10d0*s% xa2(j,k) - 5d0*s% xa1(j,k) + s% xa0(j,k)
               end do
            end do
            s% xa_compare => s% xa4
         end if
                  
      end function do_bdf5


      integer function do_bdf6( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            do_hydro_converge, set_tol_correction, set_surf_info
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical,intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         do_bdf6 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         ! order 4
         ! ss_bfd3: 4 stage bdf1, bdf2, bdf3, bdf4. h = dt/4
         
         h = dt/4
         
         ! stage 1
         s% ebdf_stage = 1
         do_bdf6 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_bdf6 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2
         s% ebdf_stage = 2
         do_bdf6 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_bdf6 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf3)
         s% ebdf_stage = 3
         do_bdf6 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_bdf6 /= keep_going) return   
         call store_results_by_stage(s,3)
         
         ! stage 4 (bdf4)
         s% ebdf_stage = 4
         do_bdf6 = do_stage_bdf4( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh3_1, s% xa3_1, &
            s% num_newton_iters_stage4)
         if (do_bdf6 /= keep_going) return   
         call store_results_by_stage(s,4)
         
         ! stage 5 (bdf5)
         s% ebdf_stage = 5
         do_bdf6 = do_stage_bdf5( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_final_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh4, s% xa4, s% xh4_1, s% xa4_1, &
            s% num_newton_iters_stage5)
         if (do_bdf6 /= keep_going) return   
         call store_results_by_stage(s,5)
         
         ! stage 6 (bdf6)
         s% ebdf_stage = 6
         do_bdf6 = do_stage_bdf6( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_final_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh4, s% xa4, s% xh5, s% xa5, s% xh5_1, s% xa5_1, &
            s% num_newton_iters_stage6)
         if (do_bdf6 /= keep_going) return   
         call store_results_by_stage(s,6)
         call store_bdf_average_results(s,6)
         
         if (s% min_ebdf_order == 6 .and. .not. s% trace_truncation_ratio) then
            max_order = 6
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh5, s% xa5, &
               s% xh4, s% xa4, s% xh3, s% xa3, &
               s% xh2, s% xa2, s% xh1, s% xa1, &
               s% xh0, s% xa0, null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if
         
         ! for local truncation error estimates, compare to extrapolated solution, 
         ! -y0 + 6 y1 - 15 y2 + 20 y3 - 15 y4 + 6 y5
         do k = 1,nz
            do j=1, nvar_hydro
               s% xh5(j,k) = &
                  6d0*s% xh5(j,k) - 15d0*s% xh4(j,k) + 20d0*s% xh3(j,k) - &
                     15d0*s% xh2(j,k) + 6d0*s% xh1(j,k) - s% xh0(j,k)
            end do
         end do
         s% xh_compare => s% xh5
         if (do_chem) then
            do k = 1,nz
               do j=1, species
                  s% xa5(j,k) = &
                     6d0*s% xa5(j,k) - 15d0*s% xa4(j,k) + 20d0*s% xa3(j,k) - &
                        15d0*s% xa2(j,k) + 6d0*s% xa1(j,k) - s% xa0(j,k)
               end do
            end do
            s% xa_compare => s% xa5
         end if
                  
      end function do_bdf6


      integer function do_ebdf2( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            do_hydro_converge, set_tol_correction, set_surf_info
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical,intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         if (s% use_bdf_instead_of_ebdf) then
            do_ebdf2 = do_bdf2( &
               s, s% newton_itermin, nvar, do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            return
         end if
         
         do_ebdf2 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         ! order 2
         ! ss_ebfd2: 3 stage bdf1, bdf1, ebdf2. h = dt
         
         h = dt
         
         ! stage 1
         s% ebdf_stage = 1
         do_ebdf2 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_ebdf2 /= keep_going) return   
         call store_results_by_stage(s,1)

         ! stage 2
         s% ebdf_stage = 2
         if (s% ebdf_redo_mix_coeffs_other_stages) then
            call ebdf_redo_mix_coeffs(s,ierr)
            if (ierr /= 0) return
         end if
         do k=1,nz
            do j=1,nvar_hydro
               s% xh_pre(j,k) = s% xh(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa_pre(j,k) = s% xa(j,k)
            end do
         end do
         do_ebdf2 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh1, s% xa1, s% xh0_1, s% xa0_1, s% num_newton_iters_stage2)
         if (do_ebdf2 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         if (s% min_ebdf_order == 2 .and. .not. s% trace_truncation_ratio) then
            max_order = 2
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh1, s% xa1, &
               s% xh0, s% xa0, null(), null(), &
               null(), null(), null(), null(), &
               null(), null(), null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if

         ! 3rd stage (ebdf2)
         s% ebdf_stage = 3
         do k=1,nz
            do j=1,nvar_hydro
               s% xh_pre(j,k) = s% xh0(j,k) - 0.5d0*(s% xh(j,k) - s% xh1(j,k))
               s% xh(j,k) = s% xh0(j,k) ! double backup gives better convergence
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa_pre(j,k) = s% xa0(j,k) - 0.5d0*(s% xa(j,k) - s% xa1(j,k))
               s% xa(j,k) = s% xa0(j,k) ! double backup gives better convergence
            end do
         end do

         dt_stage = 1.5d0*h
         s% dVARdot_dVAR = 1/dt_stage
         
         call ebdf_set_vars( &
            s, s% ebdf_redo_mix_for_final_stage, dt_stage, ierr)
         if (ierr /= 0) return

         do_ebdf2 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_ebdf2 /= keep_going) return   
         s% num_newton_iters_stage3 = s% num_newton_iterations
         call store_results_by_stage(s,3)
         call store_ebdf_average_results(s,2)
         
         ! for local truncation error estimates, compare to stage 1 solution
         s% xh_compare => s% xh1
         if (do_chem) s% xa_compare => s% xa1
         
      end function do_ebdf2


      integer function do_ebdf3( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         if (s% use_bdf_instead_of_ebdf) then
            do_ebdf3 = do_bdf3( &
               s, s% newton_itermin, nvar, do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            return
         end if
         
         do_ebdf3 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         ! ss_ebfd3: 4 stage bdf1, bdf2, bdf2, ebdf3. h = dt/2
         h = dt/2d0
         
         ! stage 1 (bdf1)
         s% ebdf_stage = 1
         do_ebdf3 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_ebdf3 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2 (bdf2)
         s% ebdf_stage = 2
         do_ebdf3 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_ebdf3 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf2)
         s% ebdf_stage = 3
         do_ebdf3 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_ebdf3 /= keep_going) return   
         call store_results_by_stage(s,3)
         
         if (s% min_ebdf_order == 3 .and. .not. s% trace_truncation_ratio) then
            max_order = 3
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh2, s% xa2, &
               s% xh1, s% xa1, s% xh0, s% xa0, &
               null(), null(), null(), null(), &
               null(), null(), null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if

         ! 4th stage (ebdf3)
         ! 4) f(y1) = (y1 - y_pre)/(22/23 h)
         !    y_pre = (26 z1 - 5 y0 - 6 z3 + 8 z2)/23
         ! use z2 as initial guess for y1, and y2-z1 as estimate of local truncation error
         s% ebdf_stage = 4
         
         do k=1,nz
            do j=1,nvar_hydro
               s% xh_pre(j,k) = &
                  (26d0*s% xh1(j,k) - 5d0*s% xh0(j,k) - 6d0*s% xh(j,k) + 8d0*s% xh2(j,k))/23d0
               s% xh(j,k) = s% xh2(j,k) ! double backup gives better convergence
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa_pre(j,k) = &
                  (26d0*s% xa1(j,k) - 5d0*s% xa0(j,k) - 6d0*s% xa(j,k) + 8d0*s% xa2(j,k))/23d0
               s% xa(j,k) = s% xa2(j,k) ! double backup gives better convergence
            end do
         end do

         dt_stage = (22d0/23d0)*h
         s% dVARdot_dVAR = 1/dt_stage
         
         call ebdf_set_vars( &
            s, s% ebdf_redo_mix_for_final_stage, dt_stage, ierr)
         if (ierr /= 0) return

         do_ebdf3 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_ebdf3 /= keep_going) return   
         s% num_newton_iters_stage4 = s% num_newton_iterations
         call store_results_by_stage(s,4)
         call store_ebdf_average_results(s,3)
         
         ! for local truncation error estimates, compare to stage 2 solution
         s% xh_compare => s% xh2
         if (do_chem) s% xa_compare => s% xa2
                  
      end function do_ebdf3


      integer function do_ebdf4( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         if (s% use_bdf_instead_of_ebdf) then
            do_ebdf4 = do_bdf4( &
               s, s% newton_itermin, nvar, do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            return
         end if
         
         do_ebdf4 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
                  
         ! ss_ebfd4: 5 stage bdf1, bdf2, bdf3, bdf3, ebdf4
         h = dt/3
         
         ! stage 1 (bdf1)
         s% ebdf_stage = 1
         do_ebdf4 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_ebdf4 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2 (bdf2)
         s% ebdf_stage = 2
         do_ebdf4 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_ebdf4 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf3)
         s% ebdf_stage = 3
         do_ebdf4 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_ebdf4 /= keep_going) return   
         call store_results_by_stage(s,3)
         
         ! stage 4 (bdf3)
         s% ebdf_stage = 4
         do_ebdf4 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh1, s% xa1, s% xh2, s% xa2, s% xh3, s% xa3, s% xh3_1, s% xa3_1, &
            s% num_newton_iters_stage4)
         if (do_ebdf4 /= keep_going) return   
         call store_results_by_stage(s,4)
         
         if (s% min_ebdf_order == 4 .and. .not. s% trace_truncation_ratio) then
            max_order = 4
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh3, s% xa3, &
               s% xh2, s% xa2, s% xh1, s% xa1, &
               s% xh0, s% xa0, null(), null(), &
               null(), null(), null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if

         ! 5th stage (ebdf4)
         ! 5) f(y1) = (y1 - y_pre)/(150/197 h)
         !    y_pre = (252 z2 - 93 z1 + 17 y0 - 33 z4 + 54 z3)/197
         s% ebdf_stage = 5
         
         do k=1,nz
            do j=1,nvar_hydro
               s% xh_pre(j,k) = &
                  (252d0*s% xh2(j,k) - 93d0*s% xh1(j,k) + &
                     17d0*s% xh0(j,k) - 33d0*s% xh(j,k) + 54d0*s% xh3(j,k))/197d0
               !s% xh(j,k) = s% xh2(j,k) ! double backup gives better convergence
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa_pre(j,k) = &
                  (252d0*s% xa2(j,k) - 93d0*s% xa1(j,k) + &
                     17d0*s% xa0(j,k) - 33d0*s% xa(j,k) + 54d0*s% xa3(j,k))/197d0
               !s% xa(j,k) = s% xa2(j,k) ! double backup gives better convergence
            end do
         end do

         dt_stage = (150d0/197d0)*h
         s% dVARdot_dVAR = 1/dt_stage
         
         call ebdf_set_vars( &
            s, s% ebdf_redo_mix_for_final_stage, dt_stage, ierr)
         if (ierr /= 0) return

         do_ebdf4 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_ebdf4 /= keep_going) return   
         s% num_newton_iters_stage5 = s% num_newton_iterations
         call store_results_by_stage(s,5)
         call store_ebdf_average_results(s,4)
          
         ! for local truncation error estimates, compare to stage 3 solution
         s% xh_compare => s% xh3
         if (do_chem) s% xa_compare => s% xa3
         
      end function do_ebdf4


      integer function do_ebdf5( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         if (s% use_bdf_instead_of_ebdf) then
            do_ebdf5 = do_bdf5( &
               s, s% newton_itermin, nvar, do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            return
         end if
         
         do_ebdf5 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
                  
         ! ss_ebfd5: 6 stage bdf1, bdf2, bdf3, bdf4, bdf4, ebdf5
         h = dt/4
         
         ! stage 1 (bdf1)
         s% ebdf_stage = 1
         do_ebdf5 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_ebdf5 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2 (bdf2)
         s% ebdf_stage = 2
         do_ebdf5 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_ebdf5 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf3)
         s% ebdf_stage = 3
         do_ebdf5 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_ebdf5 /= keep_going) return   
         call store_results_by_stage(s,3)
         
         ! stage 4 (bdf4)
         s% ebdf_stage = 4
         do_ebdf5 = do_stage_bdf4( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh3_1, s% xa3_1, &
            s% num_newton_iters_stage4)
         if (do_ebdf5 /= keep_going) return   
         call store_results_by_stage(s,4)
         
         ! stage 5 (bdf4)
         s% ebdf_stage = 5
         do_ebdf5 = do_stage_bdf4( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh1, s% xa1, s% xh2, s% xa2, s% xh3, s% xa3, &
            s% xh4, s% xa4, s% xh4_1, s% xa4_1, &
            s% num_newton_iters_stage5)
         if (do_ebdf5 /= keep_going) return   
         call store_results_by_stage(s,5)
         
         if (s% min_ebdf_order == 5 .and. .not. s% trace_truncation_ratio) then
            max_order = 5
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh4, s% xa4, &
               s% xh3, s% xa3, s% xh2, s% xa2, &
               s% xh1, s% xa1, s% xh0, s% xa0, &
               null(), null(),null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if

         ! 6th stage (ebdf5)
         ! 6) f(y1) = (y1 - y_pre)/(1644/2501 h)
         !    y_pre = (4440 z3 - 2316 z2 + 764 z1 - 111 y0 + 300 z5 - 576 z4)/2501
         s% ebdf_stage = 6
         
         do k=1,nz
            do j=1,nvar_hydro
               s% xh_pre(j,k) = &
                  (4440d0*s% xh3(j,k) - 2316d0*s% xh2(j,k) + 764d0*s% xh1(j,k) &
                     - 111d0*s% xh0(j,k) + 300d0*s% xh(j,k) - 576d0*s% xh4(j,k))/2501d0
               s% xh(j,k) = s% xh3(j,k) ! double backup gives better convergence
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa_pre(j,k) = &
                  (4440d0*s% xa3(j,k) - 2316d0*s% xa2(j,k) + 764d0*s% xa1(j,k) &
                     - 111d0*s% xa0(j,k) + 300d0*s% xa(j,k) - 576d0*s% xa4(j,k))/2501d0
               s% xa(j,k) = s% xa3(j,k) ! double backup gives better convergence
            end do
         end do

         dt_stage = (1644d0/2501d0)*h
         s% dVARdot_dVAR = 1/dt_stage
         
         call ebdf_set_vars( &
            s, s% ebdf_redo_mix_for_final_stage, dt_stage, ierr)
         if (ierr /= 0) return

         do_ebdf5 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_ebdf5 /= keep_going) return   
         s% num_newton_iters_stage6 = s% num_newton_iterations
         call store_results_by_stage(s,6)
         call store_ebdf_average_results(s,5)
         
         ! for local truncation error estimates, compare to stage 4 solution
         s% xh_compare => s% xh4
         if (do_chem) s% xa_compare => s% xa4
         
      end function do_ebdf5
      

      integer function do_ebdf6( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         if (s% use_bdf_instead_of_ebdf) then
            do_ebdf6 = do_bdf6( &
               s, s% newton_itermin, nvar, do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            return
         end if
         
         do_ebdf6 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
                  
         ! ss_ebfd6: 7 stage bdf1, bdf2, bdf3, bdf4, bdf5, bdf5, ebdf6
         h = dt/5
         
         ! stage 1 (bdf1)
         s% ebdf_stage = 1
         do_ebdf6 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_ebdf6 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2 (bdf2)
         s% ebdf_stage = 2
         do_ebdf6 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_ebdf6 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf3)
         s% ebdf_stage = 3
         do_ebdf6 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_ebdf6 /= keep_going) return   
         call store_results_by_stage(s,3)
         
         ! stage 4 (bdf4)
         s% ebdf_stage = 4
         do_ebdf6 = do_stage_bdf4( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh3_1, s% xa3_1, &
            s% num_newton_iters_stage4)
         if (do_ebdf6 /= keep_going) return   
         call store_results_by_stage(s,4)
         
         ! stage 5 (bdf5)
         s% ebdf_stage = 5
         do_ebdf6 = do_stage_bdf5( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh4, s% xa4, s% xh4_1, s% xa4_1, &
            s% num_newton_iters_stage5)
         if (do_ebdf6 /= keep_going) return   
         call store_results_by_stage(s,5)
         
         ! stage 6 (bdf5)
         s% ebdf_stage = 6
         do_ebdf6 = do_stage_bdf5( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh1, s% xa1, s% xh2, s% xa2, s% xh3, s% xa3, &
            s% xh4, s% xa4, s% xh5, s% xa5, s% xh5_1, s% xa5_1, &
            s% num_newton_iters_stage6)
         if (do_ebdf6 /= keep_going) return   
         call store_results_by_stage(s,6)
         
         if (s% min_ebdf_order == 6 .and. .not. s% trace_truncation_ratio) then
            max_order = 6
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh5, s% xa5, &
               s% xh4, s% xa4, s% xh3, s% xa3, &
               s% xh2, s% xa2, s% xh1, s% xa1, &
               s% xh0, s% xa0, null(), null(), &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if

         ! 7th stage (ebdf6)
         ! 7) f(y1) = (y1 - y_pre)/(8820/14919 h)
         !   y_pre = (29550 z4 - 20700 z3 + 10350 z2 - 3045 z1 + 394 y0 + 1370 z6 - 3000 z5)/14919
         s% ebdf_stage = 7
         
         do k=1,nz
            do j=1,nvar_hydro
               s% xh_pre(j,k) = &
                  (29550d0*s% xh4(j,k) - 20700d0*s% xh3(j,k) + 10350d0*s% xh2(j,k) - &
                  3045d0*s% xh1(j,k) + 394d0*s% xh0(j,k) + &
                  1370d0*s% xh(j,k) - 3000d0*s% xh5(j,k))/14919d0
               s% xh(j,k) = s% xh4(j,k) ! double backup gives better convergence
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa_pre(j,k) = &
                  (29550d0*s% xa4(j,k) - 20700d0*s% xa3(j,k) + 10350d0*s% xa2(j,k) - &
                  3045d0*s% xa1(j,k) + 394d0*s% xa0(j,k) + &
                  1370d0*s% xa(j,k) - 3000d0*s% xa5(j,k))/14919d0
               s% xa(j,k) = s% xa4(j,k) ! double backup gives better convergence
            end do
         end do

         dt_stage = (8820d0/14919d0)*h
         s% dVARdot_dVAR = 1/dt_stage
         
         call ebdf_set_vars( &
            s, s% ebdf_redo_mix_for_final_stage, dt_stage, ierr)
         if (ierr /= 0) return

         do_ebdf6 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_ebdf6 /= keep_going) return   
         s% num_newton_iters_stage7 = s% num_newton_iterations
         call store_results_by_stage(s,7)
         call store_ebdf_average_results(s,6)
         
         ! for local truncation error estimates, compare to stage 5 solution
         s% xh_compare => s% xh5
         if (do_chem) s% xa_compare => s% xa5
         
      end function do_ebdf6
      

      integer function do_ebdf7( &
            s, itermin, nvar, do_chem, skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp) :: tol_correction_norm, tol_max_correction, h, dt_stage, &
            term1, term2, term3, term4, term5, term6, term7
         integer :: ierr, i, j, k, nz, nvar_hydro, species
         real(dp), pointer :: p2(:,:)
         
         include 'formats'
         
         if (s% use_bdf_instead_of_ebdf) then
            s% ebdf_order = 6
            do_ebdf7 = do_bdf6( &
               s, s% newton_itermin, nvar, do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            return
         end if
         
         do_ebdf7 = terminate
         ierr = 0

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
                  
         ! ss_ebfd7: 8 stage bdf1, bdf2, bdf3, bdf4, bdf5, bdf6, bdf6, ebdf7
         h = dt/6
         
         ! stage 1 (bdf1)
         s% ebdf_stage = 1
         do_ebdf7 = do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
         if (do_ebdf7 /= keep_going) return   
         call store_results_by_stage(s,1)
         
         ! stage 2 (bdf2)
         s% ebdf_stage = 2
         do_ebdf7 = do_stage_bdf2( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh1_1, s% xa1_1, &
            s% num_newton_iters_stage2)
         if (do_ebdf7 /= keep_going) return   
         call store_results_by_stage(s,2)
         
         ! stage 3 (bdf3)
         s% ebdf_stage = 3
         do_ebdf7 = do_stage_bdf3( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, s% xh2_1, s% xa2_1, &
            s% num_newton_iters_stage3)
         if (do_ebdf7 /= keep_going) return   
         call store_results_by_stage(s,3)
         
         ! stage 4 (bdf4)
         s% ebdf_stage = 4
         do_ebdf7 = do_stage_bdf4( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh3_1, s% xa3_1, &
            s% num_newton_iters_stage4)
         if (do_ebdf7 /= keep_going) return   
         call store_results_by_stage(s,4)
         
         ! stage 5 (bdf5)
         s% ebdf_stage = 5
         do_ebdf7 = do_stage_bdf5( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_for_comparison_stage, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh4, s% xa4, s% xh4_1, s% xa4_1, &
            s% num_newton_iters_stage5)
         if (do_ebdf7 /= keep_going) return   
         call store_results_by_stage(s,5)
         
         ! stage 6 (bdf6)
         s% ebdf_stage = 6
         do_ebdf7 = do_stage_bdf6( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh0, s% xa0, s% xh1, s% xa1, s% xh2, s% xa2, &
            s% xh3, s% xa3, s% xh4, s% xa4, s% xh5, s% xa5, s% xh5_1, s% xa5_1, &
            s% num_newton_iters_stage6)
         if (do_ebdf7 /= keep_going) return   
         call store_results_by_stage(s,6)
         
         ! stage 7 (bdf6)
         s% ebdf_stage = 6
         do_ebdf7 = do_stage_bdf6( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            s% ebdf_redo_mix_coeffs_other_stages, &
            h, tol_correction_norm, tol_max_correction, &
            s% xh1, s% xa1, s% xh2, s% xa2, s% xh3, s% xa3, &
            s% xh4, s% xa4, s% xh5, s% xa5, s% xh6, s% xa6, s% xh6_1, s% xa6_1, &
            s% num_newton_iters_stage6)
         if (do_ebdf7 /= keep_going) return   
         call store_results_by_stage(s,7)
         
         if (s% min_ebdf_order == 7 .and. .not. s% trace_truncation_ratio) then
            max_order = 7
         else
            call eval_order_terms(s, &
               s% xh, s% xa, s% xh6, s% xa6, &
               s% xh5, s% xa5, s% xh4, s% xa4, &
               s% xh3, s% xa3, s% xh2, s% xa2, &
               s% xh1, s% xa1, s% xh0, s% xa0, &
               term1, term2, term3, term4, term5, term6, term7, &
               max_order)
         end if

         ! 8th stage (ebdf7)
         ! 8) f(y1) = (y1 - y_pre)/(21780/39981 h)
         !    y_pre = (86940 z5 - 76450 z4 + 51300 z3 - 22815 z2 + 
         !              5956 z1 - 690 y0 + 2940 z7 - 7200 z6)/39981
         s% ebdf_stage = 8

         do k=1,nz
            do j=1,nvar_hydro
               s% xh_pre(j,k) = &
                  (86940d0*s% xh5(j,k) - 76450d0*s% xh4(j,k) + 51300d0*s% xh3(j,k) - &
                  22815d0*s% xh2(j,k) + 5956d0*s% xh1(j,k) - 690d0*s% xh0(j,k) + &
                  2940d0*s% xh(j,k) - 7200d0*s% xh6(j,k))/39981d0
               s% xh(j,k) = s% xh5(j,k) ! double backup gives better convergence
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa_pre(j,k) = &
                  (86940d0*s% xa5(j,k) - 76450d0*s% xa4(j,k) + 51300d0*s% xa3(j,k) - &
                  22815d0*s% xa2(j,k) + 5956d0*s% xa1(j,k) - 690d0*s% xa0(j,k) + &
                  2940d0*s% xa(j,k) - 7200d0*s% xh6(j,k))/39981d0
               s% xa(j,k) = s% xa5(j,k) ! double backup gives better convergence
            end do
         end do

         dt_stage = (21780d0/39981d0)*h
         s% dVARdot_dVAR = 1/dt_stage
         
         call ebdf_set_vars( &
            s, s% ebdf_redo_mix_for_final_stage, dt_stage, ierr)
         if (ierr /= 0) return

         do_ebdf7 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_ebdf7 /= keep_going) return   
         s% num_newton_iters_stage8 = s% num_newton_iterations
         call store_results_by_stage(s,8)
         call store_ebdf_average_results(s,7)
         
         ! for local truncation error estimates, compare to stage 6 solution
         s% xh_compare => s% xh6
         if (do_chem) s% xa_compare => s% xa6
         
      end function do_ebdf7


      end module struct_burn_mix


