! ***********************************************************************
!
!   Copyright (C) 2013  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************



      module star_sparse
      
      use star_private_def
      use const_def
      use num_def
      use utils_lib, only: fill_with_NaNs, fill_quad_with_NaNs
      use mtx_lib, only: num_klu_ipar_decsol, num_klu_rpar_decsol, &
         klu_dble_decsols_nrhs_0_based, klu_quad_decsols_nrhs_0_based, &
         dense_to_col_with_diag_0_based, dense_to_col_with_diag_0_based_qp
         
      implicit none
      
      
      
      logical, parameter :: use_quad = .false. ! currently all-or-none for qp
      !logical, parameter :: use_quad = .true. ! currently all-or-none for qp
      
      
      logical, parameter :: do_fill_with_NaNs = .false.
      

      integer, parameter :: lid = num_klu_ipar_decsol
      integer, parameter :: lrd = num_klu_rpar_decsol
      

      
      logical :: dbg = .false.
      
      logical :: keep_sprs_statistics = .false.
      
      integer :: sprs_num_alloc_klu_storage = 0
      integer :: sprs_num_clear_klu_storage = 0
      integer :: sprs_num_analyze = 0
      integer :: sprs_num_factor = 0
      integer :: sprs_num_refactor = 0
      integer :: sprs_num_solve = 0
      integer :: sprs_num_free_numeric = 0
      integer :: sprs_num_free_symbolic = 0
      integer :: sprs_num_alloc_klu_factors = 0
      integer :: sprs_num_free_klu_factors = 0

      
      contains


      subroutine star_alloc_klu_storage(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         integer :: nz, k, old_sz, new_sz
         type(sparse_info), pointer :: ks(:)
         
         include 'formats'
         if (dbg) write(*,*) 'star_alloc_klu_storage'
         
         nz = s% nz
         if (associated(s% bcyclic_sprs_storage)) then
            ks => s% bcyclic_sprs_storage
            old_sz = size(s% bcyclic_sprs_storage)
            if (old_sz >= nz) then
               return
            end if
         else  
            old_sz = 0
         end if
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_alloc_klu_storage = sprs_num_alloc_klu_storage + 1
!$omp end critical (sprs_statistics)
         end if
         
         new_sz = nz*2 + 1000
         allocate(s% bcyclic_sprs_storage(new_sz), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'alloc failed for klu_storage in star_sparse'
            return
         end if
         
         if (old_sz > 0) then
            do k = 1, old_sz
               s% bcyclic_sprs_storage(k) = ks(k)
            end do
         end if
         
         do k = old_sz+1, new_sz
            s% bcyclic_sprs_storage(k)% sprs_nonzeros = -1
            s% bcyclic_sprs_storage(k)% ia => null()
            s% bcyclic_sprs_storage(k)% ja => null()
            s% bcyclic_sprs_storage(k)% values => null()
            s% bcyclic_sprs_storage(k)% values_qp => null()
            s% bcyclic_sprs_storage(k)% rpar_decsol = 0
            s% bcyclic_sprs_storage(k)% ipar8_decsol = 0
            s% bcyclic_sprs_storage(k)% have_Numeric = .false.
         end do
         
         if (old_sz > 0) then
            write(*,*) 'free old klu_storage'
            deallocate(ks)
         end if
            
      end subroutine star_alloc_klu_storage
      
      
      subroutine star_sparse_setup_shared(s, k, k_shared, nvar, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, k_shared, nvar
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)

         include 'formats'
         
         ierr = 0
         
         if (dbg) write(*,*) 'star_sparse_setup_shared'
         
         if (s% bcyclic_shared_sprs_nonzeros <= 0) then
            write(*,3) 's% bcyclic_shared_sprs_nonzeros', &
               s% bcyclic_shared_sprs_nonzeros, s% model_number
            stop 'sparse_setup_shared: burn1_BE'
         end if
      
         ks => s% bcyclic_sprs_storage
         
         if (ks(k)% have_Numeric) return ! already setup from last time
         
         if (use_quad) then ! check private values_qp array
            if (associated(ks(k)% values_qp)) then
               if (size(ks(k)% values_qp,dim=1) < s% bcyclic_shared_sprs_nonzeros) then
                  deallocate(ks(k)% values_qp)
                  allocate(ks(k)% values_qp(s% bcyclic_shared_sprs_nonzeros + 100), stat=ierr)
               end if
            else 
               allocate(ks(k)% values_qp(s% bcyclic_shared_sprs_nonzeros + 100), stat=ierr)
            end if
            if (ierr /= 0) then
               write(*,2) 'burn1_BE: allocate sparse matrix values_qp failed', &
                  s% bcyclic_shared_sprs_nonzeros
               stop
               return
            end if
            if (do_fill_with_NaNs) call fill_quad_with_NaNs(ks(k)% values_qp)            
         else ! check private values array
            if (associated(ks(k)% values)) then
               if (size(ks(k)% values,dim=1) < s% bcyclic_shared_sprs_nonzeros) then
                  deallocate(ks(k)% values)
                  allocate(ks(k)% values(s% bcyclic_shared_sprs_nonzeros + 100), stat=ierr)
               end if
            else 
               allocate(ks(k)% values(s% bcyclic_shared_sprs_nonzeros + 100), stat=ierr)
            end if
            if (ierr /= 0) then
               write(*,2) 'burn1_BE: allocate sparse matrix values failed', &
                  s% bcyclic_shared_sprs_nonzeros
               stop
               return
            end if
            if (do_fill_with_NaNs) call fill_with_NaNs(ks(k)% values)
         end if
         
         if (k == k_shared) return
         ! copy shared stuff
         
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         
         ks(k)% sprs_nonzeros = s% bcyclic_shared_sprs_nonzeros
         ks(k)% ia => s% bcyclic_sprs_shared_ia
         ks(k)% ja => s% bcyclic_sprs_shared_ja
         ks(k)% ipar8_decsol = s% bcyclic_sprs_shared_ipar8_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if
         
         if (use_quad) then
            b_qp => b_qp_ary
            if (dbg) write(*,*) 'star_sparse_setup_shared call klu_quad'
            call klu_quad_decsols_nrhs_0_based( &  ! copy klu factors array
               -6, 0, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)
         else
            if (dbg) write(*,*) 'star_sparse_setup_shared call klu_dble'
            call klu_dble_decsols_nrhs_0_based( &  ! copy klu factors array
               -6, 0, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
         if (ierr /= 0) return
      
      end subroutine star_sparse_setup_shared


      subroutine star_sparse_matrix_info( &
            s, k, nvar, species, net_lwork, mtx, sprs_nonzeros, ierr)   

         use eos_def, only : i_eta
         use net_def, only: Net_Info
         use net_lib, only: net_get_symbolic_d_dxdt_dx
         use rates_def, only: std_reaction_Qs, std_reaction_neuQs, num_rvs
         
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar, species, net_lwork
         real(dp), pointer, intent(inout) :: mtx(:,:) ! (nvar,nvar)
         integer, intent(out) :: ierr, sprs_nonzeros
      
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: values(:), b(:)
         real(qp), pointer :: values_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         integer :: nonzero_cnt, j, i, nvar_hydro, i1, i2
         
         real(dp) :: d_epsnuc_dx(species) 
         real(dp) :: dxdt_nuc(species), dxdt_dRho(species), dxdt_dT(species)
         real(dp) :: d_dxdt_dRho(species)
         real(dp) :: d_dxdt_dT(species)
         real(dp) :: d_dxdt_dx(species, species)            
         real(dp) :: reaction_eps_nuc(num_rvs, s% num_reactions)
         real(dp) :: eps_nuc_categories(num_categories)
         real(dp) :: eps_nuc_neu_total, eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT
         real(dp), dimension(s% num_reactions) :: &
            rate_raw, rate_raw_dT, rate_raw_dRho, &
            rate_screened, rate_screened_dT, rate_screened_dRho
         
         type (Net_Info), target :: net_info_target
         type (Net_Info), pointer :: netinfo
         real(dp), target :: net_work_ary(net_lwork)
         real(dp), pointer :: net_work(:)
         integer, parameter :: screening_mode = 0
          
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_matrix_info', k
         
         ierr = 0
         net_work => net_work_ary
         netinfo => net_info_target
         
         call net_get_symbolic_d_dxdt_dx( &
            s% net_handle, netinfo, species, s% num_reactions, s% xa(1:species,k), &
            s% T(k), s% lnT(k)/ln10, s% rho(k), s% lnd(k)/ln10, &
            s% abar(k), s% zbar(k), s% z2bar(k), s% ye(k), &
            s% eta(k), s% d_eos_dlnd(i_eta,k), s% d_eos_dlnT(i_eta,k), &
            s% rate_factors, s% weak_rate_factor, &
            std_reaction_Qs, std_reaction_neuQs, &
            eps_nuc, d_eps_nuc_dRho, d_eps_nuc_dT, d_epsnuc_dx, & 
            dxdt_nuc, dxdt_dRho, dxdt_dT, d_dxdt_dx, &
            screening_mode, s% theta_e(k), &
            eps_nuc_categories, eps_nuc_neu_total, &
            net_lwork, net_work, ierr)
         if (ierr /= 0) then
            write(*,2) 'net_get_symbolic_d_dxdt_dx failed', k
            stop
            return
         end if
         
         mtx(:,:) = 0
         nvar_hydro = nvar - species
         do j=1,species
            do i=1,species
               mtx(i+nvar_hydro,j+nvar_hydro) = d_dxdt_dx(i,j)
            end do
            mtx(j+nvar_hydro,j+nvar_hydro) = 10 ! always include diagonal
         end do
         
         do j=1,nvar_hydro
            do i=1,nvar_hydro
               mtx(i,j) = 1
            end do
            mtx(j,j) = 10
         end do
         
         if (.false.) then
            i1 = 0
            do j=1,nvar
               do i=1,nvar
                  if (mtx(i,j) == 0d0) cycle
                  i1 = i1+1
               end do
            end do
            write(*,*) nvar, i1
            write(*,*) 'i j mtx(i,j)'
            do j=1,nvar
               do i=1,nvar
                  if (mtx(i,j) == 0d0) cycle
                  write(*,'(2i5,5x,1pd26.16)') i, j, mtx(i,j)
               end do
            end do
            write(*,*) 'rhs'
            do j=1,nvar
               write(*,'(i5,5x,1pd26.16)') j, 1d0
            end do
            write(*,*) 'soln'
            do j=1,nvar
               write(*,'(i5,5x,1pd26.16)') j, 1d0
            end do
            stop
         end if
         
         i1 = max(s% i_lnd, s% i_lnPgas) ! only one is nonzero
         i2 = s% i_lnT
         do j=nvar_hydro+1,nvar
            mtx(s% i_dE_dt,j) = 1 ! energy eqn depends on abundances
            mtx(j,i1) = 1 ! abundance eqns depend on density or pressure
            mtx(j,i2) = 1 ! and on temperature
         end do
         
         ks => s% bcyclic_sprs_storage
      
         nonzero_cnt = 0
         do j=1,nvar
            do i=1,nvar
               if (mtx(i,j) == 0) cycle
               nonzero_cnt = nonzero_cnt + 1
            end do
         end do
         
         write(*,3) 'nonzero, nvar**2, frac', nonzero_cnt, nvar*nvar, &
            dble(nonzero_cnt)/dble(nvar*nvar)
         
         if (associated(ks(k)% ia)) then
            if (size(ks(k)% ia,dim=1) < nvar+1) then
               deallocate(ks(k)% ia)
               allocate(ks(k)% ia(nvar + 10), stat=ierr)
            end if
         else 
            allocate(ks(k)% ia(nvar + 10), stat=ierr)
         end if
         if (do_fill_with_NaNs) ks(k)% ia = -9999999
         if (ierr /= 0) then
            write(*,2) 'allocate failed for ia', nvar+1
            stop
            return
         end if
         
         if (associated(ks(k)% ja)) then
            if (size(ks(k)% ja,dim=1) < nonzero_cnt) then
               deallocate(ks(k)% ja)
               allocate(ks(k)% ja(nonzero_cnt + 100), stat=ierr)
            end if
         else 
            allocate(ks(k)% ja(nonzero_cnt + 100), stat=ierr)
         end if
         if (do_fill_with_NaNs) ks(k)% ja = -9999999
         if (ierr /= 0) then
            write(*,2) 'allocate failed for ja', nonzero_cnt
            stop
            return
         end if

         if (use_quad) then
            if (associated(ks(k)% values_qp)) then
               if (size(ks(k)% values_qp,dim=1) < nonzero_cnt) then
                  deallocate(ks(k)% values_qp)
                  allocate(ks(k)% values_qp(nonzero_cnt + 100), stat=ierr)
               end if
            else 
               allocate(ks(k)% values_qp(nonzero_cnt + 100), stat=ierr)
            end if
            if (ierr /= 0) then
               write(*,2) 'allocate failed for values', nonzero_cnt
               stop
               return
            end if
            values_qp => ks(k)% values_qp      
            if (do_fill_with_NaNs) call fill_quad_with_NaNs(values_qp)
            ! compressed_format is compressed_col_sparse_0_based for KLU
            if (dbg) write(*,2) 'call dense_to_col_with_diag_0_based_qp', k
            call dense_to_col_with_diag_0_based_qp( &
               nvar, nvar, mtx, nonzero_cnt, sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, values_qp, ierr)
         else         
            if (associated(ks(k)% values)) then
               if (size(ks(k)% values,dim=1) < nonzero_cnt) then
                  deallocate(ks(k)% values)
                  allocate(ks(k)% values(nonzero_cnt + 100), stat=ierr)
               end if
            else 
               allocate(ks(k)% values(nonzero_cnt + 100), stat=ierr)
            end if
            if (ierr /= 0) then
               write(*,2) 'allocate failed for values', nonzero_cnt
               stop
               return
            end if
            values => ks(k)% values      
            if (do_fill_with_NaNs) call fill_with_NaNs(values)
            ! compressed_format is compressed_col_sparse_0_based for KLU
            if (dbg) write(*,2) 'call dense_to_col_with_diag_0_based', k
            call dense_to_col_with_diag_0_based( &
               nvar, nvar, mtx, nonzero_cnt, sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, values, ierr)
         end if

         if (ierr /= 0 .or. sprs_nonzeros /= nonzero_cnt) then
            write(*,*) &
               'solve burn sparse failed in converting dense to sparse'
            ierr = -1
            return
         end if
         ks(k)% sprs_nonzeros = sprs_nonzeros
      
      end subroutine star_sparse_matrix_info
      
         
      subroutine star_sparse_store_new_values(s, k, nvar, mtx, dbg, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         logical, intent(in) :: dbg
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: values(:)
         real(qp), pointer :: values_qp(:)
         integer, pointer :: colptr(:) ! (n+1)
         integer, pointer :: rowind(:) ! (nz)
         integer :: j, i, nz, c0, c1, c, sprs_nonzeros
         
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_store_new_values', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         values => ks(k)% values
         values_qp => ks(k)% values_qp
         colptr => ks(k)% ia
         rowind => ks(k)% ja
         sprs_nonzeros = ks(k)% sprs_nonzeros
         nz = 0
         do j=1,nvar ! column index
            c0 = colptr(j)+1 ! 1st rowind
            c1 = colptr(j+1) ! last rowind
            do c=c0,c1
               i = rowind(c)+1 ! row index
               nz = nz+1
               if (nz > sprs_nonzeros) then
                  write(*,5) 'bad value for nz', nz, k, i, j
                  stop 'sparse_store_new_values'
                  ierr = -1
                  return
               end if
               if (i < 1 .or. i > nvar) then
                  write(*,5) 'bad value for i', nz, k, i, j
                  stop 'sparse_store_new_values'
                  ierr = -1
                  return
               end if
               if (dbg) write(*,5) 'values(nz) = mtx(i,j)', nz, i, j, k, mtx(i,j)
               if (use_quad) then
                  values_qp(nz) = mtx(i,j)
               else
                  values(nz) = mtx(i,j)
               end if
            end do
         end do
         if (nz /= sprs_nonzeros) then
            ierr = -1
            write(*,4) 'nz /= ks(k)% sprs_nonzeros', &
               k, nz, ks(k)% sprs_nonzeros
            stop 'sparse_store_new_values'
            return
         end if

      end subroutine star_sparse_store_new_values
         
         
      subroutine star_sparse_analyze_and_factor(s, k, nvar, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_analyze_and_factor', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         b_qp => b_qp_ary
         
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if

         call star_sparse_free_numeric(s, k, nvar, ierr)
         if (ierr /= 0) return
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_analyze = sprs_num_analyze + 1
            sprs_num_factor = sprs_num_factor + 1
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if
         
         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_analyze_and_factor call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & ! klu_analyze and klu_factor
               0, 1, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)
         else
            if (dbg) write(*,*) 'star_sparse_analyze_and_factor call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & ! klu_analyze and klu_factor
               0, 1, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
            
         ks(k)% have_Numeric = .true.
         
         if (ierr /= 0) then
            write(*,2) 'do_klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
      
      end subroutine star_sparse_analyze_and_factor
         
         
      subroutine star_sparse_analyze(s, k, nvar, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         integer :: i
         
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_analyze', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         b_qp => b_qp_ary

         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_analyze = sprs_num_analyze + 1
            sprs_num_alloc_klu_factors = sprs_num_alloc_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         if (use_quad) then
            if (dbg) write(*,2) 'size(ks(k)% values_qp,dim=1)', size(ks(k)% values_qp,dim=1)
            if (dbg) write(*,2) 'size(ks(k)% ia,dim=1)', size(ks(k)% ia,dim=1)
            if (dbg) write(*,2) 'size(ks(k)% ja,dim=1)', size(ks(k)% ja,dim=1)
            if (dbg) write(*,2) 'size(b_qp,dim=1)', size(b_qp,dim=1)
            if (dbg) write(*,2) 'ks(k)% sprs_nonzeros', ks(k)% sprs_nonzeros
            if (dbg) write(*,2) 'star_sparse_analyze call klu_quad', k
            if (.false.) then
               do i=1,nvar+1
                  write(*,3) 'ia', i, ks(k)% ia(i)
               end do
               do i=1,ks(k)% sprs_nonzeros
                  write(*,3) 'ja', i, ks(k)% ja(i)
               end do
               do i=1,ks(k)% sprs_nonzeros
                  write(*,2) 'values', i, ks(k)% values_qp(i)
               end do
            end if
            call klu_quad_decsols_nrhs_0_based( & ! klu_analyze
               -2, 1, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)
            if (dbg) write(*,2) 'done klu_quad_decsols_nrhs_0_based', k
         else
            if (dbg) write(*,2) 'size(ks(k)% values,dim=1)', size(ks(k)% values,dim=1)
            if (dbg) write(*,2) 'size(ks(k)% ia,dim=1)', size(ks(k)% ia,dim=1)
            if (dbg) write(*,2) 'size(ks(k)% ja,dim=1)', size(ks(k)% ja,dim=1)
            if (dbg) write(*,2) 'size(b,dim=1)', size(b,dim=1)
            if (dbg) write(*,2) 'ks(k)% sprs_nonzeros', ks(k)% sprs_nonzeros
            if (dbg) write(*,2) 'star_sparse_analyze call klu_dble', k
            if (.false.) then
               do i=1,nvar+1
                  write(*,3) 'ia', i, ks(k)% ia(i)
               end do
               do i=1,ks(k)% sprs_nonzeros
                  write(*,3) 'ja', i, ks(k)% ja(i)
               end do
               do i=1,ks(k)% sprs_nonzeros
                  write(*,2) 'values', i, ks(k)% values(i)
               end do
            end if
            call klu_dble_decsols_nrhs_0_based( & ! klu_analyze
               -2, 1, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
         !stop 'star_sparse_analyze'
            
         if (ierr /= 0) then
            write(*,2) 'klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if

         if (dbg) write(*,2) 'done star_sparse_analyze', k
      
      end subroutine star_sparse_analyze
         
         
      subroutine star_sparse_factor(s, k, nvar, mtx, ierr)            
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_factor', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         b_qp => b_qp_ary
 
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         
         call star_sparse_free_numeric(s, k, nvar, ierr)
         if (ierr /= 0) return
         
         
         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_factor = sprs_num_factor + 1
!$omp end critical (sprs_statistics)
         end if
         
         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_factor call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & ! klu_factor
               -1, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)
         else
            if (dbg) write(*,*) 'star_sparse_factor call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & ! klu_factor
               -1, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
            
         ks(k)% have_Numeric = .true.
            
         if (ierr /= 0) then
            write(*,2) 'klu_dble_decsols_nrhs_0_based failed factor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
      
      end subroutine star_sparse_factor
         
         
      subroutine star_sparse_refactor(s, k, nvar, mtx, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         real(dp), pointer, intent(inout) :: mtx(:,:)
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_refactor', k
         
         ierr = 0
         ks => s% bcyclic_sprs_storage
         if (.not. ks(k)% have_Numeric) then
            write(*,3) 'called refactor without Numeric', k
            ierr = -1
            return
         end if
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         b_qp => b_qp_ary
 
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_refactor = sprs_num_refactor + 1
!$omp end critical (sprs_statistics)
         end if

         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_refactor call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & ! refactor
               3, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)
         else
            if (dbg) write(*,*) 'star_sparse_refactor call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & ! refactor
               3, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
            
      end subroutine star_sparse_refactor
         
         
      subroutine star_sparse_no_pivot(s, k, nvar, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_no_pivot', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         b_qp => b_qp_ary
 
         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_no_pivot call klu_quad'
            call klu_quad_decsols_nrhs_0_based( &
               -7, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)  
         else
            if (dbg) write(*,*) 'star_sparse_no_pivot call klu_dble'
            call klu_dble_decsols_nrhs_0_based( &
               -7, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)  
         end if

      end subroutine star_sparse_no_pivot
         
         
      real(dp) function star_sparse_rcond(s, k, nvar, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_rcond', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         b_qp => b_qp_ary
 
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if

         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_rcond call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & ! klu_rcond
               4, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)  
         else
            if (dbg) write(*,*) 'star_sparse_rcond call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & ! klu_rcond
               4, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)  
         end if
            
         star_sparse_rcond = rpar(1)    
      end function star_sparse_rcond
         
         
      real(dp) function star_sparse_rgrowth(s, k, nvar, ierr)   
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_rgrowth', k
         
         ierr = 0
         ks => s% bcyclic_sprs_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         b_qp => b_qp_ary
 
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         
         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_rgrowth call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & ! klu_rgrowth
               5, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)  
         else
            if (dbg) write(*,*) 'star_sparse_rgrowth call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & ! klu_rgrowth
               5, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)  
         end if
            
         star_sparse_rgrowth = rpar(1)    
      end function star_sparse_rgrowth
         
         
      real(dp) function star_sparse_condest(s, k, nvar, ierr)  
         ! accurate condition number estimate,
         ! but more expensive to compute than rgrowth
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         
         type(sparse_info), pointer :: ks(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_condest', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol
         b => b_ary
         b_qp => b_qp_ary
 
         if (do_fill_with_NaNs) then
            ipar(:) = -9999999
            call fill_with_NaNs(rpar)
         end if
         
         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_condest call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & ! klu_condest
               6, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)  
         else
            if (dbg) write(*,*) 'star_sparse_condest call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & ! klu_condest
               6, nvar, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)  
         end if
            
         star_sparse_condest = rpar(1)    
      end function star_sparse_condest


      subroutine star_sparse_solve(s, k, nrhs, nvar, b, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nrhs, nvar
         real(dp), pointer :: b(:)
         integer, intent(out) :: ierr
         real(dp), pointer :: rpar(:)
         type(sparse_info), pointer :: ks(:)
         integer, pointer :: ipar(:)
         real(dp), pointer :: b1(:)
         integer :: i, shift
         include 'formats'

         if (dbg) write(*,2) 'star_sparse_solve', k
         
         ierr = 0
         ks => s% bcyclic_sprs_storage
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_solve = sprs_num_solve + 1
!$omp end critical (sprs_statistics)
         end if

         if (use_quad) then
            do i=1,nrhs
               shift = (i-1)*nvar
               b1(1:nvar) => b(shift+1:shift+nvar)
               call solve1_column
            end do
         else if (.true.) then
            do i=1,nrhs
               shift = (i-1)*nvar
               b1(1:nvar) => b(shift+1:shift+nvar)
               call solve1_column
            end do
         else
            if (dbg) write(*,*) 'star_sparse_solve call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & ! klu_solve
               1, nrhs, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
            
         if (ierr /= 0) then
            write(*,2) 'do_klu_dble_decsols_nrhs_0_based failed refactor', ierr
            write(*,2) 'k', k
            write(*,2) 'sprs_nonzeros', ks(k)% sprs_nonzeros
            write(*,*)
         end if
         
         
         contains
         
         
         subroutine solve1_column
            real(qp), target :: b_qp_ary(nvar)
            real(qp), pointer :: b_qp(:)
            integer :: i
            b_qp => b_qp_ary
            do i=1,nvar
               b_qp(i) = b1(i)
            end do
            if (use_quad) then
               if (dbg) write(*,*) 'solve1_column call klu_quad'
               call klu_quad_decsols_nrhs_0_based( & ! klu_solve
                  1, 1, nvar, ks(k)% sprs_nonzeros, &
                  ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
                  lrd, rpar, lid, ipar, ierr)
               do i=1,nvar
                  b1(i) = b_qp(i)
               end do
            else
               if (dbg) write(*,*) 'solve1_column call klu_dble'
               call klu_dble_decsols_nrhs_0_based( & ! klu_solve
                  1, 1, nvar, ks(k)% sprs_nonzeros, &
                  ks(k)% ia, ks(k)% ja, ks(k)% values, b1, &
                  lrd, rpar, lid, ipar, ierr)
            end if
         end subroutine solve1_column
         

      end subroutine star_sparse_solve


      subroutine star_sparse_free_all(s, nvar, k_min, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: nvar, k_min
         integer, intent(out) :: ierr
         
         integer :: k
         type(sparse_info), pointer :: ks(:)
         
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_free_all', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         do k = k_min, size(ks)
            if (ks(k)% sprs_nonzeros <= 0) cycle
            call star_sparse_free_numeric(s, k, nvar, ierr)
            if (ierr /= 0) exit
            call star_sparse_free_klu_factors(s, k, nvar, ierr)
            if (ierr /= 0) exit
            ks(k)% sprs_nonzeros = -1
         end do
      end subroutine star_sparse_free_all
      
      
      subroutine star_sparse_free_klu_factors(s, k, nvar, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_free_klu_factors', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         b => b_ary
         b_qp => b_qp_ary
 
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol


         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_klu_factors = sprs_num_free_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         ! free klu factors array (this is also done by klu_free_symbolic)
         
         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_free_klu_factors call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & 
               -5, 0, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)
         else
            if (dbg) write(*,*) 'star_sparse_free_klu_factors call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & 
               -5, 0, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
               
      end subroutine star_sparse_free_klu_factors
      
      
      subroutine star_sparse_free_symbolic(s, k, nvar, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)
         
         include 'formats'
         if (dbg) write(*,2) 'star_sparse_free_symbolic', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         b => b_ary
         b_qp => b_qp_ary

         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_symbolic = sprs_num_free_symbolic + 1
            sprs_num_free_klu_factors = sprs_num_free_klu_factors + 1
!$omp end critical (sprs_statistics)
         end if

         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_free_symbolic call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & 
               -4, 0, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)
         else
            if (dbg) write(*,*) 'star_sparse_free_symbolic call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & 
               -4, 0, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
               
      end subroutine star_sparse_free_symbolic
      
      
      subroutine star_sparse_free_numeric(s, k, nvar, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: k, nvar
         integer, intent(out) :: ierr
         type(sparse_info), pointer :: ks(:)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         real(dp), target :: b_ary(1)
         real(dp), pointer :: b(:)
         real(qp), target :: b_qp_ary(1)
         real(qp), pointer :: b_qp(:)

         include 'formats'
         if (dbg) write(*,2) 'star_sparse_free_numeric', k

         ierr = 0
         ks => s% bcyclic_sprs_storage
         if (.not. ks(k)% have_Numeric) return
         b => b_ary
         b_qp => b_qp_ary
 
         ipar => ks(k)% ipar8_decsol
         rpar => ks(k)% rpar_decsol

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_free_numeric = sprs_num_free_numeric + 1
!$omp end critical (sprs_statistics)
         end if         
         
         if (use_quad) then
            if (dbg) write(*,*) 'star_sparse_free_numeric call klu_quad'
            call klu_quad_decsols_nrhs_0_based( & 
               -3, 0, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values_qp, b_qp, &
               lrd, rpar, lid, ipar, ierr)
         else
            if (dbg) write(*,*) 'star_sparse_free_numeric call klu_dble'
            call klu_dble_decsols_nrhs_0_based( & 
               -3, 0, nvar, ks(k)% sprs_nonzeros, &
               ks(k)% ia, ks(k)% ja, ks(k)% values, b, &
               lrd, rpar, lid, ipar, ierr)
         end if
            
         ks(k)% have_Numeric = .false.
         
      end subroutine star_sparse_free_numeric
      
      
      subroutine star_clear_klu_storage(s, max_k_for_free_shared)
         type (star_info), pointer :: s
         integer, intent(in) :: max_k_for_free_shared
         integer :: k
         
         include 'formats'

         if (dbg) write(*,2) 'star_clear_klu_storage', k

         if (keep_sprs_statistics) then
!$omp critical (sprs_statistics)
            sprs_num_clear_klu_storage = sprs_num_clear_klu_storage + 1
!$omp end critical (sprs_statistics)
         end if

         do k = 1, size(s% bcyclic_sprs_storage)
            if (k <= max_k_for_free_shared) then
               if (associated(s% bcyclic_sprs_storage(k)% ia)) &
                  deallocate(s% bcyclic_sprs_storage(k)% ia)
               if (associated(s% bcyclic_sprs_storage(k)% ja)) &
                  deallocate(s% bcyclic_sprs_storage(k)% ja)
            end if
            if (associated(s% bcyclic_sprs_storage(k)% values)) &
               deallocate(s% bcyclic_sprs_storage(k)% values)
         end do
         deallocate(s% bcyclic_sprs_storage)
         nullify(s% bcyclic_sprs_storage)
         
      end subroutine star_clear_klu_storage
      

      end module star_sparse



