! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module overshoot

      use const_def
      use num_lib
      use utils_lib
      use star_private_def
      use star_utils, only: zero_D_mix_partials

      implicit none

      
      contains

      
      ! exponential diffusive overshoot as described in the paper by Falk Herwig, 
      ! "The evolution of AGB stars with convective overshoot", A&A, 360, 952-968 (2000)
      subroutine add_overshooting(s, ierr)
         use mlt_def
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         real(dp) :: D0, vc0, Hp, r0, D_ov, cdc_ov, vc_ov, q_edge, r_edge, &
            f_above, f_below, step_f_above, step_f_below, f0_above, f0_below, &
            f2_above, f2_below, D2_above, D2_below, &
            ov_limit, D_ov_limit, h1_czb_mass, h1_czb_dm, &
            h1_czb_mass_old, top_r, bot_r, zone_dr
         integer :: i, k, loc, k0, n_conv_bdy, nz, new_mix_type
         real(dp), pointer, dimension(:) :: grada, gradr

         logical :: dbg, have_h1_czb
         
         include 'formats'
         
         dbg = .false.
         
         ierr = 0         
         nz = s% nz
         grada => s% grada_at_face
         gradr => s% gradr
         n_conv_bdy = s% num_conv_boundaries
         
         if (dbg) write(*,3) 'add_overshooting model n_conv_bdy', s% model_number, n_conv_bdy

         s% core_overshoot_Hp = 0
         s% core_overshoot_r0 = 0
         
         do i=1, n_conv_bdy ! from center to surface

            if (s% conv_bdy_q(i) < s% min_overshoot_q) then
               if (dbg) write(*,*) 'skip since s% conv_bdy_q(i) < min_overshoot_q', i
               cycle
            end if
            
            call set_f(i)
            
            have_h1_czb = .false.
            
            if (f_above == 0 .and. f_below == 0 .and. &
                  step_f_above == 0 .and. step_f_below == 0) cycle
            
            if (f0_above <= 0 .and. (f_above > 0 .or. step_f_above > 0)) then
               write(*,*) 'ERROR: when using overshooting above, must set f0_above > 0'
               write(*,*) 'see description of overshooting in star/defaults/control.defaults'
               if (f_above > 0) write(*,*) 'f_above', f_above
               if (step_f_above > 0) write(*,*) 'step_f_above', step_f_above
               write(*,3) 'i s% conv_bdy_loc(i)', i, s% conv_bdy_loc(i)
               write(*,*) 'burn_h_conv_region', s% burn_h_conv_region(i)
               write(*,*) 'burn_he_conv_region', s% burn_he_conv_region(i)
               write(*,*) 'burn_z_conv_region', s% burn_z_conv_region(i)
               ierr = -1
               return
            end if
            
            if (f0_below <= 0 .and. (f_below > 0 .or. step_f_below > 0)) then
               write(*,*) 'ERROR: when using overshooting below, must set f0_below > 0'
               write(*,*) 'see description of overshooting in star/defaults/control.defaults'
               if (f_below > 0) write(*,*) 'f_below', f_below
               if (step_f_below > 0) write(*,*) 'step_f_below', step_f_below
               write(*,3) 'i s% conv_bdy_loc(i)', i, s% conv_bdy_loc(i)
               write(*,*) 'burn_h_conv_region', s% burn_h_conv_region(i)
               write(*,*) 'burn_he_conv_region', s% burn_he_conv_region(i)
               write(*,*) 'burn_z_conv_region', s% burn_z_conv_region(i)
               ierr = -1
               return
            end if
            
            loc = s% conv_bdy_loc(i)
            if (loc < 1 .or. loc > nz) then
               write(*,*) 'bad s% conv_bdy_loc(i)', i, s% conv_bdy_loc(i)
               ierr = -1
               return
            end if
            
            if (s% top_conv_bdy(i)) then ! overshoot toward surface
               if (f_above + step_f_above > 0 .and. &
                     s% mixing_type(loc+1) == convective_mixing .and. loc > 1) then
                  if (i <= 1) then
                     bot_r = s% R_center
                  else
                     if (s% top_conv_bdy(i-1)) cycle
                     bot_r = s% r(s% conv_bdy_loc(i-1))
                  end if
                  top_r = s% r(loc)
                  zone_dr = top_r - bot_r
                  if (dbg) write(*,2) 'toward surface zone_dr', loc, zone_dr, top_r
                  call overshoot_toward_surface((i == 1), ierr)
                  if (ierr /= 0) return
               end if
            else ! overshoot toward center
               if (f_below + step_f_below > 0 .and. &
                     s% mixing_type(loc-1) == convective_mixing) then
                  if (i >= n_conv_bdy) then
                     top_r = s% r(1)
                  else
                     if (.not. s% top_conv_bdy(i+1)) then
                        write(*,*) 's% top_conv_bdy(i+1)', s% top_conv_bdy(i+1)
                        cycle
                     end if
                     top_r = s% r(s% conv_bdy_loc(i+1))
                  end if
                  bot_r = s% r(loc)
                  zone_dr = top_r - bot_r
                  if (dbg) write(*,1) 'toward center zone_dr', &
                     zone_dr, s% m(loc)/Msun, bot_r/Rsun
                  call overshoot_toward_center( &
                     s% burn_h_conv_region(i), s% burn_he_conv_region(i), &
                     s% burn_z_conv_region(i), ierr)
                  if (ierr /= 0) return
               else if (.false.) then
                  write(*,3) 'f_below', s% mixing_type(loc+2), loc+2, f_below
                  write(*,3) 'f_below', s% mixing_type(loc+1), loc+1, f_below
                  write(*,3) 'f_below', s% mixing_type(loc), loc, f_below
                  write(*,3) 'f_below', s% mixing_type(loc-1), loc-1, f_below
                  write(*,3) 'f_below', s% mixing_type(loc-2), loc-2, f_below
                  !stop
               end if
            end if
            if (dbg) write(*,*)

         end do
         
         !if (n_conv_bdy > 0) stop 'overshoot'
         
         
         contains
         
         
         subroutine overshoot_toward_surface(is_core_zone, ierr)
            logical, intent(in) :: is_core_zone
            integer, intent(out) :: ierr
            real(dp) :: h, hstep, h2, dr, dr2, dr_limit, r_limit
            logical :: dbg
            integer :: kk
            include 'formats'

            dbg = .false.
            ierr = 0
            if (dbg) write(*,1) 'overshoot_toward_surface call get_r0_vc0'
            ! top  of region is between loc+1 and loc, i.e. in cell k=loc
            call get_r0_vc0( &
               s, loc, .false., zone_dr, f0_above, k0, r0, vc0, D0, Hp, q_edge, r_edge, dbg, ierr)
            if (ierr /= 0) then
               write(*,*) 'overshoot toward surface failure'
               return
            end if
            if (k0 <= 0) then
               if (dbg) then
                  write(*,2) 'skip overshoot_toward_surface: k0', k0
                  do k=loc+10,loc-10,-1
                     write(*,3) 'mixing_type', k, s% mixing_type(k)
                  end do
                  stop 'overshoot_toward_surface'
                  end if
               return
            end if
            
            if (is_core_zone) then
               s% core_overshoot_Hp = Hp
               s% core_overshoot_r0 = r0
               s% core_overshoot_f0 = f0_above
               s% core_overshoot_f = f_above
               s% core_overshoot_hstep = hstep
            end if

            h = f_above*Hp    
            hstep = step_f_above*Hp
            if (D2_above > 0 .and. f2_above > 0 .and. D2_above < D0) then
               dr2 = h*log_cr(D0/D2_above)/2
               h2 = f2_above*Hp
            else
               dr2 = 1d99
            end if

            do k = k0, 1, -1
               if (s% r(k) < r0) cycle
               dr = s% r(k) - r0
               if (k == k0) then
                  ov_limit = 0
               else
                  ov_limit = D_ov_limit
               end if
               if (dr < dr2) then
                  if (dbg .and. D0 < 0) write(*,2) 'bogus D0', k, D0
                  call get_D_ov(s, k, dr, vc0, D0, h, hstep, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
               else
                  call get_D_ov(s, k, dr-dr2, vc0, D2_above, h2, 0d0, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
               end if
               if (dbg) write(*,*) 'k (D_ov > ov_limit)', k, D_ov > ov_limit, D_ov, ov_limit
               if (cdc_ov == 0 .or. (s% max_brunt_B_for_overshoot > 0d0 .and. &
                     s% unsmoothed_brunt_B(k) > s% max_brunt_B_for_overshoot)) then
                  if (dbg) write(*,2) 'finished extending upward: cdc_ov == 0', k
                  if (cdc_ov == 0 .and. k < k0 .and. s% D_mix(k+1) > ov_limit .and. &
                        (D_ov-ov_limit)*(s% D_mix(k+1)-ov_limit) < 0d0) then
                     ! set s% cz_bdy_dq(k) to sub-cell edge of overshooting
                     s% cz_bdy_dq(k) = find0(0d0,D_ov-ov_limit,s% dq(k),s% D_mix(k+1)-ov_limit)
                     if (s% cz_bdy_dq(k) < 0d0 .or. s% cz_bdy_dq(k) > s% dq(k)) then
                        ierr = -1
                        return
                        write(*,2) 'bad cz_bdy_dq', k, s% cz_bdy_dq(k), s% dq(k)
                        write(*,2) 'D_ov', k, D_ov
                        write(*,2) 's% D_mix(k+1)', k+1, s% D_mix(k+1)
                        write(*,2) 's% D_mix(k+2)', k+2, s% D_mix(k+2)
                        write(*,2) 'ov_limit', k, ov_limit
                        write(*,2) 'D_ov-ov_limit', k, D_ov-ov_limit
                        write(*,2) 's% D_mix(k+1)-ov_limit', k, s% D_mix(k+1)-ov_limit
                        write(*,2) 'k0', k0
                        stop 'overshoot upward'
                     end if
                  end if
                  if (k > loc) then ! undershooting
                     do kk = k, loc, -1 ! change to radiative
                        s% cdc(k) = 0
                        s% D_mix(k) = 0
                        call zero_D_mix_partials(s,k)
                        s% conv_vel(k) = 0
                        s% mixing_type(k) = no_mixing
                     end do
                  end if

                  exit
               end if
               if (dbg) then
                  write(*,*) 'overshoot upward m/Msun: s% r(k) <= r_edge', s% r(k) <= r_edge
                  write(*,2) 's% m(k)/Msun', k, s% m(k)/Msun
                  write(*,2) '(s% r(k)-r0)/Rsun', k, (s% r(k)-r0)/Rsun
                  write(*,2) 's% r(k)/Rsun', k, s% r(k)/Rsun
                  write(*,2) 'r_edge/Rsun', k, r_edge/Rsun
                  write(*,2) 'Hp/Rsun', k, Hp/Rsun
                  write(*,2) 'D_ov', k, D_ov
                  write(*,2) 's% D_mix(k)', k, s% D_mix(k)
                  write(*,2) 's% r(k) - r_edge', k, s% r(k) - r_edge
               end if
               if (s% r(k) <= r_edge) then ! inside conv zone
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  call zero_D_mix_partials(s,k)
                  s% conv_vel(k) = vc_ov
                  if (new_mix_type == softened_convective_mixing) then
                     s% mixing_type(k) = softened_convective_mixing
                  else
                     s% mixing_type(k) = convective_mixing
                  end if
               else if (D_ov > s% D_mix(k)) then ! increase mixing
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  call zero_D_mix_partials(s,k)
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = new_mix_type
               end if
            end do
         end subroutine overshoot_toward_surface
         
         
         subroutine overshoot_toward_center(burn_h, burn_he, burn_z, ierr)
            logical, intent(in) :: burn_h, burn_he, burn_z
            integer, intent(out) :: ierr
            
            real(dp) :: h, hstep, h2, dr, dr2, dr_limit, r_limit
            logical :: dbg
            integer :: kk
            include 'formats'

            dbg = .false.

            if (dbg) write(*,*) 'overshoot_toward_center'
            ierr = 0
            ! bottom of region is between loc-1 and loc, i.e., in cell loc-1
            call get_r0_vc0( &
               s, loc-1, .true., zone_dr, f0_below, k0, r0, vc0, D0, Hp, q_edge, r_edge, dbg, ierr)
            if (ierr /= 0) return
            if (k0 <= 0) return       
            
            if (dbg) then
               write(*,2) 'loc-1', loc-1
               write(*,2) 'k0', k0
               write(*,1) 'f0_below', f0_below
               write(*,1) 'r0', r0
               write(*,1) 'Hp', Hp
            end if     
            
            if ((.not. (burn_h .or. burn_he .or. burn_z)) .and. (.not. have_h1_czb)) then
               have_h1_czb = .true.
               h1_czb_mass = s% M_center + s% xmstar*q_edge
               s% h1_czb_mass = h1_czb_mass/Msun
               if (.false.) then
                  write(*,2) 's% h1_czb_mass', k0, s% h1_czb_mass
                  write(*,2) 'q_edge', k0, q_edge
                  write(*,2) 'r_edge/Rsun', k0, r_edge/Rsun
                  write(*,2) 'Hp/Rsun', k0, Hp/Rsun
                  write(*,*)
               end if
               call check_TP
               if (dbg) write(*,1) 'f_below', f_below
            end if

            h = f_below*Hp
            hstep = step_f_below*Hp
            if (D2_below > 0 .and. f2_below > 0 .and. D2_below < D0) then
               dr2 = h*log_cr(D0/D2_below)/2
               h2 = f2_below*Hp
               if (dbg) then
                  write(*,2) 'dr2', k0, dr2
                  if (r0 > dr2) write(*,1) 'log10((r0-dr2)/Rsun)', log10_cr((r0-dr2)/Rsun)
                  write(*,1) 'log10(r0/Rsun)', log10_cr(r0/Rsun)
                  write(*,1) 'D2_below', D2_below
                  write(*,1) 'f2_below', f2_below
                  write(*,1) 'f_below', f_below
                  write(*,1) 'Hp', Hp
                  write(*,1) 'h', h
                  write(*,1) 'D0', D0
                  write(*,*)
               end if
            else
               dr2 = 1d99
               h2 = 0
            end if
            
            do k=k0+1, nz
               if (s% r(k) > r0) cycle
               dr = r0 - s% r(k)
               
               if (dbg) write(*,1) 'dr/(f_below*Hp)', dr/(f_below*Hp)
               
               if (k == k0+1) then
                  ov_limit = 0
               else
                  ov_limit = D_ov_limit
               end if
               if (dr < dr2) then
                  call get_D_ov(s, k, dr, vc0, D0, h, hstep, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
               else
                  call get_D_ov(s, k, dr-dr2, vc0, D2_below, h2, 0d0, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
               end if
               
               if (dbg) write(*,1) 'D_ov', D_ov
               if (dbg) write(*,1) 'D_ov/D0', D_ov/D0
               
               if (cdc_ov == 0 .or. (s% max_brunt_B_for_overshoot > 0d0 .and. &
                     s% unsmoothed_brunt_B(k+1) > s% max_brunt_B_for_overshoot)) then
                  if (dbg) write(*,2) 'finished extending downward', k
                  ! set s% cz_bdy_dq(k) to sub-cell edge of overshooting
                  if (s% D_mix(k-1) > ov_limit) then
                     s% cz_bdy_dq(k-1) = find0(0d0,D_ov-ov_limit,s% dq(k-1),s% D_mix(k-1)-ov_limit)                  
                     if (s% cz_bdy_dq(k-1) < 0d0 .or. s% cz_bdy_dq(k-1) > s% dq(k-1)) then
                        write(*,2) 'bad cz_bdy_dq', k-1, s% cz_bdy_dq(k-1), s% dq(k-1)
                        stop 'overshoot downward'
                        ierr = -1
                        return
                     end if
                  end if
                  if (k < loc) then ! undershooting
                     do kk = k, loc ! change to radiative
                        s% cdc(k) = 0
                        s% D_mix(k) = 0
                        call zero_D_mix_partials(s,k)
                        s% conv_vel(k) = 0
                        s% mixing_type(k) = no_mixing
                     end do
                  end if
                  exit
               end if
               if (dbg) then
                  write(*,2) 'overshoot downward', k, s% star_mass*s% q(k), s% r(k)/Rsun
                  write(*,*)
               end if
               if (s% r(k) >= r_edge) then ! leave it convective type, but change mixing coeff
                  if (dbg) write(*,2) 'leave it convective', k
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  call zero_D_mix_partials(s,k)
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = convective_mixing

               else if (D_ov > s% D_mix(k)) then
                  if (dbg) write(*,2) 'change to overshooting', k
                  s% cdc(k) = cdc_ov
                  s% D_mix(k) = D_ov
                  call zero_D_mix_partials(s,k)
                  s% conv_vel(k) = vc_ov
                  s% mixing_type(k) = new_mix_type
               end if
            end do
            if (dbg) write(*,*) 'overshoot_toward_center'
            
         end subroutine overshoot_toward_center

         
         subroutine set_f(i)
            integer, intent(in) :: i
            real(dp) :: frac
            
            f_above = 0
            f_below = 0
            step_f_above = 0
            step_f_below = 0
            f0_above = 0
            f0_below = 0
            D2_above = 0
            D2_below = 0
            f2_above = 0
            f2_below = 0
            D_ov_limit = s% D_mix_ov_limit
            new_mix_type = overshoot_mixing

            if (s% burn_h_conv_region(i)) then
            
               if (dbg) write(*,*) 'burn_h_conv_region'
               
               if (s% star_mass > s% mass_for_overshoot_full_off .or. i > 1) then
                  
                  if (i == 1 .and. s% R_center == 0d0 .and. s% top_conv_bdy(i)) then 
                     ! this is top of core convection zone                  
                     f_above = s% overshoot_f_above_burn_h_core
                     f_below = 0
                     step_f_above = s% step_overshoot_f_above_burn_h_core
                     step_f_below = 0
                     f0_above = s% overshoot_f0_above_burn_h_core
                     f0_below = 0                  
                  else ! shell convection zone
                     f_above = s% overshoot_f_above_burn_h_shell
                     f_below = s% overshoot_f_below_burn_h_shell
                     step_f_above = s% step_overshoot_f_above_burn_h_shell
                     step_f_below = s% step_overshoot_f_below_burn_h_shell
                     f0_above = s% overshoot_f0_above_burn_h_shell
                     f0_below = s% overshoot_f0_below_burn_h_shell                  
                  end if
                     
                  D2_above = s% overshoot_D2_above_burn_h
                  D2_below = s% overshoot_D2_below_burn_h
                  f2_above = s% overshoot_f2_above_burn_h
                  f2_below = s% overshoot_f2_below_burn_h

                  ! mass limits only apply to top of core H burn region
                  ! that only happens when i == 1
                  ! so for i > 1, ignore the mass limits.
                  if (s% star_mass <= s% mass_for_overshoot_full_on .and. i == 1) then
                     frac = (s% star_mass - s% mass_for_overshoot_full_off) / &
                            (s% mass_for_overshoot_full_on - s% mass_for_overshoot_full_off)
                     frac = 0.5d0*(1 - cospi_cr(frac))
                     f_above = f_above*frac
                     f0_above = f0_above*frac
                     step_f_above = step_f_above*frac
                  end if

               end if
               
            else if (s% burn_he_conv_region(i)) then 
            
               if (dbg) write(*,*) 'burn_he_conv_region'
                       
               if (i == 1 .and. s% R_center == 0d0 .and. s% top_conv_bdy(i)) then 
                  ! this is top of core convection zone                  
                  f_above = s% overshoot_f_above_burn_he_core
                  f_below = 0
                  step_f_above = s% step_overshoot_f_above_burn_he_core
                  step_f_below = 0
                  f0_above = s% overshoot_f0_above_burn_he_core
                  f0_below = 0
               else ! shell convection zone
                  f_above = s% overshoot_f_above_burn_he_shell
                  f_below = s% overshoot_f_below_burn_he_shell
                  step_f_above = s% step_overshoot_f_above_burn_he_shell
                  step_f_below = s% step_overshoot_f_below_burn_he_shell
                  f0_above = s% overshoot_f0_above_burn_he_shell
                  f0_below = s% overshoot_f0_below_burn_he_shell
               end if
               
               D2_above = s% overshoot_D2_above_burn_he
               D2_below = s% overshoot_D2_below_burn_he
               f2_above = s% overshoot_f2_above_burn_he
               f2_below = s% overshoot_f2_below_burn_he
               
               if (s% have_done_TP) then
                  f_below = f_below*s% ovr_below_burn_he_shell_factor
                  f0_below = f0_below*s% ovr_below_burn_he_shell_factor
                  f2_below = f2_below*s% ovr_below_burn_he_shell_factor
               end if               
               
            else if (s% burn_z_conv_region(i)) then    
            
               if (dbg) write(*,*) 'burn_z_conv_region'
                   
               if (i == 1 .and. s% R_center == 0d0 .and. s% top_conv_bdy(i)) then 
                  ! this is top of core convection zone                  
                  f_above = s% overshoot_f_above_burn_z_core
                  f_below = 0
                  step_f_above = s% step_overshoot_f_above_burn_z_core
                  step_f_below = 0
                  f0_above = s% overshoot_f0_above_burn_z_core
                  f0_below = 0
               else ! shell convection zone
                  f_above = s% overshoot_f_above_burn_z_shell
                  f_below = s% overshoot_f_below_burn_z_shell
                  step_f_above = s% step_overshoot_f_above_burn_z_shell
                  step_f_below = s% step_overshoot_f_below_burn_z_shell
                  f0_above = s% overshoot_f0_above_burn_z_shell
                  f0_below = s% overshoot_f0_below_burn_z_shell
               end if

               D2_above = s% overshoot_D2_above_burn_z
               D2_below = s% overshoot_D2_below_burn_z
               f2_above = s% overshoot_f2_above_burn_z
               f2_below = s% overshoot_f2_below_burn_z
               
            else  
            
               if (dbg) write(*,*) 'nonburn_conv_region'
                      
               if (i == 1 .and. s% R_center == 0d0 .and. s% top_conv_bdy(i)) then 
                  ! this is top of core convection zone                  
                  f_above = s% overshoot_f_above_nonburn_core
                  f_below = 0
                  step_f_above = s% step_overshoot_f_above_nonburn_core
                  step_f_below = 0
                  f0_above = s% overshoot_f0_above_nonburn_core
                  f0_below = 0
               else ! shell convection zone
                  f_above = s% overshoot_f_above_nonburn_shell
                  f_below = s% overshoot_f_below_nonburn_shell
                  step_f_above = s% step_overshoot_f_above_nonburn_shell
                  step_f_below = s% step_overshoot_f_below_nonburn_shell
                  f0_above = s% overshoot_f0_above_nonburn_shell
                  f0_below = s% overshoot_f0_below_nonburn_shell
               end if
                             
               D2_above = s% overshoot_D2_above_nonburn
               D2_below = s% overshoot_D2_below_nonburn
               f2_above = s% overshoot_f2_above_nonburn
               f2_below = s% overshoot_f2_below_nonburn
               
            end if
            
            if (f_above <= 0 .and. f_below <= 0 .and. &
                step_f_above <= 0 .and. step_f_below <= 0) then
               new_mix_type = softened_convective_mixing
               D_ov_limit = s% conv_bdy_mix_softening_min_D_mix
               f_above = s% conv_bdy_mix_softening_f
               f_below = f_above
               f0_above = s% conv_bdy_mix_softening_f0
               f0_below = f0_above
               step_f_above = 0
               step_f_below = 0
            end if 
            
         end subroutine set_f
         
         
         subroutine check_TP ! check for AGB thermal pulse dredge up
            logical, parameter :: dbg = .false.
            include 'formats'
            if (s% power_h_burn + s% power_he_burn < 0.5d0*s% power_nuc_burn) then
               if (dbg .and. s% TP_state /= 0) then
                  write(*,*) 'advanced burning: change to TP_state = 0'
                  write(*,*)
               end if
               s% TP_state = 0
            else if (s% he_core_mass > s% c_core_mass + 0.1d0) then
               if (dbg .and. s% TP_state /= 0) then
                  write(*,*) 'not yet: change to TP_state = 0'
                  write(*,1) 's% he_core_mass', s% he_core_mass
                  write(*,1) 's% c_core_mass', s% c_core_mass
                  write(*,1) 'h1 - c_core_mass', s% he_core_mass - s% c_core_mass
                  write(*,*)
               end if
               s% TP_state = 0
            else if (s% TP_state == 0) then ! not in TP
               if (s% power_he_burn > s% power_h_burn) then ! starting TP
                  s% TP_state = 1
                  s% TP_M_H_on = h1_czb_mass
                  s% TP_M_H_min = h1_czb_mass
                  if (dbg) then
                     write(*,*) 'change to TP_state = 1'
                     write(*,1) 's% TP_M_H_on', s% TP_M_H_on
                     write(*,*)
                  end if
               end if
            else
               h1_czb_dm = h1_czb_mass*1d-6
               if (s% TP_state == 1) then ! early part of TP
                  if (h1_czb_mass < s% TP_M_H_on - h1_czb_dm) then
                     if (dbg) then
                        write(*,*) 'change to TP_state = 2'
                        write(*,1) 's% TP_M_H_on', s% TP_M_H_on
                        write(*,1) 'h1_czb_dm', h1_czb_dm
                        write(*,1) 's% TP_M_H_on - h1_czb_dm', s% TP_M_H_on - h1_czb_dm
                        write(*,1) 'h1_czb_mass', h1_czb_mass
                        write(*,1) '(s% TP_M_H_on - h1_czb_dm) - h1_czb_mass', &
                           (s% TP_M_H_on - h1_czb_dm) - h1_czb_mass
                        write(*,*)
                     end if
                     s% TP_state = 2
                     s% TP_count = 0
                     s% TP_M_H_min = h1_czb_mass
                     s% have_done_TP = .true.
                  else if (s% power_h_burn > s% power_he_burn) then ! no dredge up
                     if (dbg) write(*,*) 'change to TP_state = 0: no dredge up this time'
                     s% TP_state = 0
                  end if
               else if (s% TP_state == 2) then ! dredge up part of TP
                  if (h1_czb_mass < s% TP_M_H_min) s% TP_M_H_min = h1_czb_mass
                  h1_czb_mass_old = s% h1_czb_mass_old*Msun
                  if (dbg) then
                     write(*,1) '(h1_czb_mass - h1_czb_mass_old)/Msun', &
                        (h1_czb_mass - h1_czb_mass_old)/Msun
                     write(*,1) 'h1_czb_dm/Msun', h1_czb_dm/Msun
                     write(*,*)
                  end if
                  if (h1_czb_mass > h1_czb_mass_old + h1_czb_dm .or. &
                        s% power_h_burn > s% power_he_burn) then
                     s% TP_count = s% TP_count + 1
                     if (dbg) write(*,2) '++ TP_count', s% TP_count, &
                           h1_czb_mass - (h1_czb_mass_old + h1_czb_dm), &
                           h1_czb_mass,  h1_czb_mass_old + h1_czb_dm, &
                           s% h1_czb_mass_old, h1_czb_dm
                  else if (h1_czb_mass < h1_czb_mass_old - h1_czb_dm) then
                     s% TP_count = max(0, s% TP_count - 1)
                     if (dbg) write(*,3) '-- TP_count', s% TP_count, s% TP_state
                  end if
                  if (s% TP_count > s% max_DUP_counter) then
                     if (dbg) write(*,*) 'change to TP_state = 0'
                     s% TP_state = 0
                  end if
               end if
               if (s% TP_state == 2) then
                  f_below = f_below*s% overshoot_below_noburn_shell_factor
                  f0_below = f0_below*s% overshoot_below_noburn_shell_factor
               end  if
            end if
         end subroutine check_TP

         
      end subroutine add_overshooting

      
      
      subroutine get_r0_vc0( &
            s, k00, overshoot_toward_center, zone_dr, f0, k0, r0, &
            vc0, D0, Hp, q_edge, r_edge, dbg, ierr)
         type (star_info), pointer :: s         
         integer, intent(in) :: k00 
         ! expect edge of convection somewhere in cell k00
         ! if s% allow_semiconvective_mixing, then edge is where gradL == gradr
         ! else, else is where grada_at_face == gradr
         logical, intent(in) :: overshoot_toward_center
         real(dp), intent(in) :: zone_dr, f0
         integer, intent(out) :: k0
         real(dp), intent(out) :: r0, vc0, D0, Hp, q_edge, r_edge
         logical, intent(in) :: dbg
         ! r(k0) >= r0 >= r(k0+1)
         ! r0 is radius where start overshooting
         ! vc0 is convection velocity at r0
         ! Hp is scale height at edge of convection
         integer, intent(out) :: ierr
         
         real(dp) :: dq, ddq, dqfrac, x, x0, x1, x2, a0, a1, a2, lambda, &
            lnP_edge, P_edge, lnd_edge, rho_edge, frac, dg00, dgp1, grada_at_face         
         integer :: km1, kp1, kk, nz
         
         include 'formats'
         
         k0 = -1
         r0 = 0
         vc0 = 0
         D0 = 0
         Hp = 0
         ierr = 0

         km1 = k00-1
         kp1 = k00+1
         nz = s% nz
         
         if (kp1 > nz) then
            if (.not. overshoot_toward_center) return
            k0 = nz
            r0 = s% r(nz)
            vc0 = s% conv_vel(nz)
            D0 = s% D_mix(nz)
            Hp = r0
            return
         end if

         dq = s% dq(k00)
         ddq = s% cz_bdy_dq(k00)
         dqfrac = ddq/dq ! fractional distance from k00 to kp1 for edge of convection
         if (dqfrac > 1d0 .or. dqfrac < 0d0) then
            write(*,2) 'bad dqfrac', k00, dqfrac, ddq, dq
            stop 'get_r0_vc0'
         end if
         q_edge = s% q(k00) - ddq
         r_edge = pow_cr(dqfrac*s% r(kp1)*s% r(kp1)*s% r(kp1) + &
                        (1-dqfrac)*s% r(k00)*s% r(k00)*s% r(k00),1d0/3d0)
         
         x0 = s% dq(km1)/2
         x1 = s% dq(km1) + s% dq(k00)/2
         x2 = s% dq(km1) + s% dq(k00) + s% dq(kp1)/2
         x = s% dq(km1) + ddq
         call two_piece_linear_coeffs(x, x0, x1, x2, a0, a1, a2, ierr)
         if (ierr /= 0) return
         
         lnP_edge = a0*s% lnP(km1) + a1*s% lnP(k00) + a2*s% lnP(kp1)
         P_edge = exp_cr(lnP_edge)
         lnd_edge = a0*s% lnd(km1) + a1*s% lnd(k00) + a2*s% lnd(kp1)
         rho_edge = exp_cr(lnd_edge)
         Hp = P_edge/(rho_edge*s% cgrav(k00)* &
               (s% M_center + s% xmstar*q_edge)/(r_edge*r_edge))
         if (s% overshoot_alpha > 0d0) then
            if (s% overshoot_alpha*Hp > zone_dr) Hp = zone_dr/s% overshoot_alpha
         else
            if (s% mixing_length_alpha*Hp > zone_dr) Hp = zone_dr/s% mixing_length_alpha
         end if
         if (dbg) then
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,1) 'r_edge+f0*Hp', (r_edge+f0*Hp)/Rsun
            write(*,2) 'r(km1)', k00-1, exp_cr(s% lnR(k00-1))/Rsun
            write(*,2) 'r(k00)', k00, exp_cr(s% lnR(k00))/Rsun
            write(*,1) 'r_edge', r_edge/Rsun
            write(*,2) 'r(kp1)', kp1, exp_cr(s% lnR(kp1))/Rsun
            write(*,2) 'r(kp2)', kp1+1, exp_cr(s% lnR(kp1+1))/Rsun
            write(*,1) 'r_edge-f0*Hp', (r_edge-f0*Hp)/Rsun
            write(*,*)
         end if
         
         if (overshoot_toward_center) then
            ! r0 is toward surface from r_edge
            
            !r0 = min(s% r(1), max(s% r(k00), r_edge + f0*Hp))
            r0 = min(s% r(1), r_edge + f0*Hp)
            
            if (dbg) then
               write(*,*) 'get_r0_vc0: overshoot toward center'
               write(*,2) 'r_edge/Rsun', k00, r_edge/Rsun
               write(*,2) 'Hp/Rsun', k00, Hp/Rsun
               write(*,2) 'f0/Rsun', k00, f0
               write(*,2) 'f0*Hp/Rsun', k00, f0*Hp/Rsun
               write(*,2) '(r_edge + f0*Hp)//Rsun', k00, (r_edge + f0*Hp)/Rsun
               write(*,2) 'initial k0 for toward center: k00, r0', k00, r0/Rsun
            end if
            if (r0 == s% r(1)) then
               k0 = 1
            else
               k0 = k00
               do kk = k00, 1, -1
                  if (s% r(kk) > r0) then
                     k0 = kk; exit
                  end if
               end do
            end if
            if (dbg) write(*,2) 'final k0, r(k0)/Rsun', k0, s% r(k0)/Rsun, s% r(k0-1)/Rsun
            if (dbg) write(*,*)
            
         else ! overshoot toward surface
            ! r0 is toward center from r_edge
            
            !r0 = max(s% r(nz), min(s% r(k00+1), r_edge - f0*Hp))
            r0 = max(s% r(nz), r_edge - f0*Hp)
            
            if (dbg) then
               write(*,*) 'get_r0_vc0: overshoot toward surface'
               write(*,2) 'r_edge', k00, r_edge/Rsun
               write(*,2) 'Hp', k00, Hp/Rsun
               write(*,2) 'f0', k00, f0
               write(*,2) 'f0*Hp', k00, f0*Hp/Rsun
               write(*,2) 'r_edge - f0*Hp', k00, (r_edge - f0*Hp)/Rsun
               write(*,2) 'initial k0 for toward surface: k00, r0', k00, r0/Rsun
            end if
            k0 = kp1
            if (s% r(k0) <= r0) then
               k0 = k00
            else
               do kk = k0, nz-1
                  if (dbg) write(*,2) 's% r(kk+1)', kk+1, s% r(kk+1)
                  if (s% r(kk+1) <= r0) then
                     k0 = kk; exit
                  end if
               end do
            end if
            if (dbg) write(*,2) 'final k0', k0
            if (dbg) write(*,*)
         end if

         if (dbg) write(*,*) 's% r(k0+1) > r0', s% r(k0+1) > r0
         if (dbg) write(*,*) 'r0 > s% r(k0)', r0 > s% r(k0)
         if (s% r(k0+1) > r0 .or. r0 > s% r(k0)) then
            k0 = -1
            return
            
            write(*,*) 'overshoot_toward_center', overshoot_toward_center
            write(*,2) 's% r(k0) - r0', k0, s% r(k0) - r0
            write(*,2) 'r0 - s% r(k0+1)', k0, r0 - s% r(k0+1)
            write(*,*)
            write(*,2) 's% r(k00) - r0', k00, s% r(k00) - r0
            write(*,2) 'r0 - s% r(k00+1)', k00, r0 - s% r(k00+1)
            write(*,*)
            write(*,2) 's% r(k00)', k00, s% r(k00)
            write(*,2) 'r0', k0, r0
            write(*,2) 's% r(k00+1)', k00+1, s% r(k00+1)
            write(*,*)
            write(*,3) 'mixing_type', k00-1, s% mixing_type(k00-1)
            write(*,3) 'mixing_type', k00, s% mixing_type(k00)
            write(*,3) 'mixing_type', k00+1, s% mixing_type(k00+1)
            write(*,3) 'mixing_type', k00+2, s% mixing_type(k00+2)
            write(*,*)
            write(*,2) 'D0', k0, D0
            write(*,2) 'vc0', k0, vc0
            write(*,2) 'frac', k0, frac
            stop 'get_r0_vc0'
         end if
         
         frac = (s% r(k0)*s% r(k0)*s% r(k0) - r0*r0*r0)/&
                  (s% r(k0)*s% r(k0)*s% r(k0) - s% r(k0+1)*s% r(k0+1)*s% r(k0+1))
         lambda = (1-frac)*s% mlt_mixing_length(k0) + frac*s% mlt_mixing_length(k0+1)
         if (s% conv_vel(k0) /= 0 .and. s% conv_vel(k0+1) /= 0) then ! AOK
            vc0 = (1-frac)*s% conv_vel(k0) + frac*s% conv_vel(k0+1)
            if (is_bad_num(vc0)) then
               write(*,2) 'vc0', k0, vc0
               write(*,2) 'lambda', k0, lambda
               write(*,2) 's% conv_vel(k0)', k0, s% conv_vel(k0)
               write(*,2) 's% conv_vel(k0+1)', k0+1, s% conv_vel(k0+1)
               write(*,2) 'frac', k0, frac
               write(*,2) 's% r(k0)', k0, s% r(k0)
               write(*,2) 'r0', k0, r0
               write(*,2) 's% r(k0+1)', k0, s% r(k0+1)
               stop 'get_r0_vc0'
            end if
         else if (s% conv_vel(k0) /= 0 .and. s% conv_vel(k0+1) == 0 .and. r_edge < r0) then
            ! use conv_vel = 0 at r_edge; use r_edge in place of r(k0+1)
            frac = (s% r(k0)*s% r(k0)*s% r(k0) - r0*r0*r0)/&
                     (s% r(k0)*s% r(k0)*s% r(k0) - r_edge*r_edge*r_edge)
            vc0 = (1-frac)*s% conv_vel(k0)
            if (is_bad_num(vc0)) then
               write(*,2) 'vc0', k0, vc0
               write(*,2) 'lambda', k0, lambda
               write(*,2) 's% conv_vel(k0)', k0, s% conv_vel(k0)
               write(*,2) 'frac', k0, frac
               write(*,2) 's% r(k0)', k0, s% r(k0)
               write(*,2) 'r0', k0, r0
               write(*,2) 'r_edge', k0, r_edge
               stop 'get_r0_vc0'
            end if
         else if (s% conv_vel(k0) == 0 .and. s% conv_vel(k0+1) /= 0 .and. r_edge > r0) then
            ! use conv_vel = 0 at r_edge; use r_edge in place of r(k0)
            frac = (r_edge*r_edge*r_edge - r0*r0*r0)/&
                     (r_edge*r_edge*r_edge - s% r(k0+1)*s% r(k0+1)*s% r(k0+1))
            vc0 = frac*s% conv_vel(k0+1)
            if (is_bad_num(vc0)) then
               write(*,2) 'vc0', k0, vc0
               write(*,2) 'lambda', k0, lambda
               write(*,2) 's% conv_vel(k0)', k0, s% conv_vel(k0)
               write(*,2) 's% conv_vel(k0+1)', k0+1, s% conv_vel(k0+1)
               write(*,2) 'frac', k0, frac
               write(*,2) 'r_edge', k0, r_edge
               write(*,2) 'r0', k0, r0
               write(*,2) 's% r(k0+1)', k0, s% r(k0+1)
               stop 'get_r0_vc0'
            end if
         else
            k0 = -1
            r0 = 0
            vc0 = 0
            D0 = 0
            Hp = 0
            return
         end if
         
         D0 = vc0*lambda/3 ! diffusion coefficient [cm^2/sec]
         
         if (is_bad_num(D0)) then
            write(*,2) 'D0', k0, D0
            write(*,2) 'lambda', k0, lambda
            write(*,2) 'vc0', k0, vc0
            write(*,2) 'frac', k0, frac
            write(*,2) 's% r(k0)', k0, s% r(k0)
            write(*,2) 'r0', k0, r0
            write(*,2) 's% r(k0+1)', k0, s% r(k0+1)
            stop 'get_r0_vc0'
         endif
         if (D0 < 0) then
            ierr = -1
            if (dbg) then
               write(*,2) 'D0', k0, D0
               write(*,2) 'vc0', k0, vc0
               write(*,2) 'frac', k0, frac
               write(*,2) 's% r(k0)', k0, s% r(k0)
               write(*,2) 'r0', k0, r0
               write(*,2) 's% r(k0+1)', k0, s% r(k0+1)
               stop 'get_r0_vc0'
            end if
         end if
                     
      end subroutine get_r0_vc0
      
      
      subroutine get_D_ov(s, k, &
            z, vc0, D0, ov_scale, hstep, D_ov, ov_limit, cdc_ov, vc_ov, dbg)
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         real(dp), intent(in) :: vc0, D0, ov_scale, hstep, ov_limit
         real(dp), intent(out) :: D_ov, cdc_ov, vc_ov
         logical, intent(in) :: dbg
         
         real(dp) :: rho, r, z, fctr
         include 'formats'
         fctr = exp_cr(-2*z/ov_scale)
         D_ov = D0*fctr
         if (hstep >= z) &
            D_ov = max(D_ov, s% step_overshoot_D + D0*s% step_overshoot_D0_coeff)
         if (D_ov < ov_limit) then
            cdc_ov = 0
            vc_ov = 0
            if (dbg) then
               write(*,2) 'get_D_ov: D_ov < ov_limit', &
                  k, D_ov, ov_limit
               write(*,2) 'D0', k, D0
               write(*,2) 'fctr', k, fctr
               write(*,2) 'hstep', k, hstep
               write(*,2) 'z', k, z
               write(*,1) 's% step_overshoot_D', s% step_overshoot_D
               write(*,1) 's% step_overshoot_D0_coeff', s% step_overshoot_D0_coeff
               write(*,*)
            end if
            return
         end if
         rho = s% rho(k)
         r = s% r(k)
         cdc_ov = (pi4*r*r*rho)*(pi4*r*r*rho)*D_ov ! gm^2/sec
         vc_ov = vc0*fctr
      end subroutine get_D_ov

      
      
      end module overshoot
