! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module mlt_info

      use const_def
      use num_lib
      use utils_lib
      use star_private_def
      use mlt_def

      implicit none

      logical, parameter :: dbg = .false.
      integer, parameter :: kdbg = -1
      
      contains


      subroutine set_mlt_vars(s, nzlo, nzhi, ierr)
         use star_utils, only: start_time, update_time
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         integer, intent(out) :: ierr
         integer :: k, op_err, time0
         real(dp) :: total, opacity, gamma1, Cv, chiRho, chiT, Cp, &
            grada, P, xh, gradL_composition_term
         include 'formats'
         if (dbg) write(*, *) 'doing set_mlt_vars'
         if (s% doing_timing) call start_time(s, time0, total)
         gradL_composition_term = -1
         opacity = -1
         gamma1 = -1
         Cv = -1
         chiRho = -1
         chiT = -1
         Cp = -1
         grada = -1
         P = -1
         xh = -1
!$OMP PARALLEL DO PRIVATE(k,op_err)
         do k = nzlo, nzhi
            op_err = 0
            call do1_mlt(s, k, s% mixing_length_alpha, gradL_composition_term, &
               opacity, gamma1, Cv, chiRho, chiT, Cp, grada, P, xh, &
               op_err)
            if (op_err /= 0) ierr = op_err
         end do
!$OMP END PARALLEL DO
         if (s% doing_timing) call update_time(s, time0, total, s% time_mlt)         
      end subroutine set_mlt_vars


      subroutine do1_mlt(s, k, mixing_length_alpha, gradL_composition_term_in, &
            opacity_face_in, gamma1_face_in, Cv_face_in, chiRho_face_in, &
            chiT_face_in, Cp_face_in, grada_face_in, P_face_in, xh_face_in, &
            ierr) 
         ! get convection info for point k
         use mlt_def
         use mlt_lib
         use eos_def
         use chem_def, only: ih1
         use utils_lib, only: is_bad_num
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         real(dp), intent(in) :: mixing_length_alpha, &
            gradL_composition_term_in, opacity_face_in, &
            gamma1_face_in, Cv_face_in, chiRho_face_in, chiT_face_in, &
            Cp_face_in, grada_face_in, P_face_in, xh_face_in
         integer, intent(out) :: ierr

         real(dp) :: m, L, r, dlnm_dlnq, v0, thc, asc, max_convective_velocity, &
            a, b, Pgas_div_P_limit, da_dlnd, da_dlnT, db_dlnd, db_dlnT, &
            max_q_for_Pgas_div_P_limit, min_q_for_Pgas_div_P_limit, &
            mlt_basics(num_mlt_results), prev_conv_vel, max_conv_vel, dt, &
            alfa, beta, T_face, rho_face, P_face, Cv_face, gamma1_face, &
            chiRho_face, chiT_face, Cp_face, opacity_face, grada_face, v, &
            cv_old, gradr_factor, f, xh_face, tau_face, &
            d_grada_at_face_dlnd00, d_grada_at_face_dlnT00, &
            d_grada_at_face_dlndm1, d_grada_at_face_dlnTm1, &
            gradL_composition_term
         real(dp), target :: mlt_partials1_ary(num_mlt_partials*num_mlt_results)
         real(dp), pointer :: mlt_partials1(:), mlt_partials(:,:)
         integer :: mixing_type, h1
         real(dp), parameter :: conv_vel_mach_limit = 0.9d0
         logical :: Schwarzschild_stable, Ledoux_stable, MLT_dbg

         include 'formats'
         
         ierr = 0
         mlt_partials1 => mlt_partials1_ary
         mlt_partials(1:num_mlt_partials,1:num_mlt_results) => &
            mlt_partials1(1:num_mlt_partials*num_mlt_results)
         
         alfa = 0
                  
         m = s% m_grav(k)
         
         if (m < 0) then
            write(*,2) 'nz', s% nz
            write(*,2) 's% q(k)', k, s% q(k)
            write(*,2) 's% m(k)', k, s% m(k)
            write(*,2) 's% m_grav(k)', k, s% m_grav(k)
            stop 'mlt'
         end if
         
         dlnm_dlnq = 1
         r = s% r(k)
         L = s% L(k)
         
         if (is_bad_num(L)) then
            write(*,2) 'do1_mlt L', k, L
            stop 1
         end if

         ! alfa is the fraction coming from k; (1-alfa) from k-1.
         if (k == 1) then
            alfa = 1d0
         else
            alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
         end if
         beta = 1d0 - alfa
         h1 = s% net_iso(ih1)
         
         opacity_face = opacity_face_in
         gamma1_face = gamma1_face_in
         Cv_face = Cv_face_in
         chiRho_face = chiRho_face_in
         chiT_face = chiT_face_in
         Cp_face = Cp_face_in
         grada_face = grada_face_in
         P_face = P_face_in
         xh_face = xh_face_in
         gradL_composition_term = gradL_composition_term_in
         
         tau_face = 1 ! this is only used for atmosphere integrations where tau < 2/3
         s% actual_gradT(k) = 0
         rho_face = s% rho_face(k)
         
         if (alfa == 1d0) then
         
            T_face = s% T(k)
            if (P_face < 0) P_face = s% P(k)
            if (gamma1_face < 0) gamma1_face = s% gamma1(k)
            if (Cv_face < 0) Cv_face = s% Cv(k)
            if (chiRho_face < 0) chiRho_face = s% chiRho(k)         
            if (chiT_face < 0) chiT_face = s% chiT(k)         
            if (Cp_face < 0) Cp_face = s% Cp(k)
            if (opacity_face < 0) opacity_face = s% opacity(k)         
            if (grada_face < 0) grada_face = s% grada(k)
            if (h1 /= 0 .and. xh_face < 0) xh_face = s% xa(h1, k)
            
         else
         
            T_face = alfa*s% T(k) + beta*s% T(k-1)
            if (P_face < 0) P_face = alfa*s% P(k) + beta*s% P(k-1)
            if (gamma1_face < 0) gamma1_face = alfa*s% gamma1(k) + beta*s% gamma1(k-1)
            if (Cv_face < 0) Cv_face = alfa*s% Cv(k) + beta*s% Cv(k-1)
            if (chiRho_face < 0) chiRho_face = alfa*s% chiRho(k) + beta*s% chiRho(k-1)         
            if (chiT_face < 0) chiT_face = alfa*s% chiT(k) + beta*s% chiT(k-1)         
            if (Cp_face < 0) Cp_face = alfa*s% Cp(k) + beta*s% Cp(k-1)         
            if (opacity_face < 0) opacity_face = alfa*s% opacity(k) + beta*s% opacity(k-1)         
            if (grada_face < 0) grada_face = alfa*s% grada(k) + beta*s% grada(k-1)
            if (h1 /= 0 .and. xh_face < 0) xh_face = alfa*s% xa(h1,k) + beta*s% xa(h1,k-1)
            s% actual_gradT(k) = &
               ((s% T(k-1) - s% T(k))/T_face) / ((s% P(k-1) - s% P(k))/P_face)
            if (is_bad_num(s% actual_gradT(k))) s% actual_gradT(k) = 0
         end if
         
         if (s% rotation_flag .and. s% mlt_use_rotation_correction) then
            gradr_factor = s% ft_rot(k)/s% fp_rot(k)
         else
            gradr_factor = 1d0
         end if
         if (is_bad_num(gradr_factor)) then
            ierr = -1
            if (s% report_ierr) then
               if (.false.) call show_stuff
               write(*,2) 'do1_mlt_eval gradr_factor', k, gradr_factor
            end if
            return
         end if
         
         thc = s% thermohaline_coeff
         asc = s% alpha_semiconvection
         if (s% center_h1 > s% semiconvection_upper_limit_center_h1) asc = 0
         
         if (T_face >= s% min_T_for_acceleration_limited_conv_velocity .and. &
               s% have_previous_conv_vel) then
            if (s% generations >= 2) then
               if (is_bad_num(s% conv_vel_old(k))) then
                  cv_old = -1
               else
                  cv_old = s% conv_vel_old(k)
               end if
            else if (s% use_previous_conv_vel_from_file) then
               if (is_bad_num(s% prev_conv_vel_from_file(k))) then
                  cv_old = -1
               else
                  cv_old = s% prev_conv_vel_from_file(k)
               end if
            else
               cv_old = -1
            end if
         else
            cv_old = -1
         end if
         if (cv_old >= 0) then
            prev_conv_vel = cv_old
            !write(*,2) 'prev_conv_vel', k, prev_conv_vel
            dt = s% dt
         else
            prev_conv_vel = -1
            dt = -1
         end if
         
         max_conv_vel = s% csound_at_face(k)*s% max_conv_vel_div_csound
         if (prev_conv_vel >= 0) then
            if (must_limit_conv_vel(s,k)) then
               max_conv_vel = prev_conv_vel
            end if
         end if

         !if (s% m(k) < 0.3*Msun) write(*,2) 'max_conv_vel', s% model_number, max_conv_vel
         
         MLT_dbg = .false.
         
         if (s% use_Ledoux_criterion .and. gradL_composition_term < 0) then
            gradL_composition_term = s% gradL_composition_term(k)
         else
            gradL_composition_term = 0d0
         end if

         call do1_mlt_eval( &
            s, k, s% cgrav(k), m, r, T_face, rho_face, L, P_face, chiRho_face, chiT_face, & 
            Cp_face, Cv_face, s% csound_at_face(k), &
            xh_face, opacity_face, grada_face, gradr_factor, &
            gradL_composition_term, &
            asc, s% semiconvection_option, thc, s% thermohaline_option, &
            s% dominant_iso_for_thermohaline(k), &
            mixing_length_alpha, s% alt_scale_height_flag, s% remove_small_D_limit, & 
            s% MLT_option, s% Henyey_MLT_y_param, s% Henyey_MLT_nu_param, &
            prev_conv_vel, max_conv_vel, s% mlt_accel_g_theta, dt, tau_face, MLT_dbg, & 
            mixing_type, mlt_basics, mlt_partials1, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) then
!$OMP critical
               if (.false.) call show_stuff
               write(*,*) 'ierr in do1_mlt_eval for k', k
!$OMP end critical
               
               !stop 'mlt'
               
            end if
            return
         end if
         
         s% mlt_mixing_type(k) = mixing_type
         s% mlt_mixing_length(k) = mlt_basics(mlt_Lambda)
         s% mlt_vc(k) = mlt_basics(mlt_convection_velocity)
         
         if (mixing_type == 0 .and. s% mlt_vc(k) /= 0d0) then
            write(*,2) 'mixing_type mlt_vc', mixing_type, s% mlt_vc(k)
            stop 'MLT_dbg'
         end if
         
         if (MLT_dbg) then
            write(*,2) 'mixing_type', mixing_type
            stop 'MLT_dbg'
         end if

         s% mlt_Gamma(k) = mlt_basics(mlt_Gamma)
         s% mlt_cdc(k) = mlt_basics(mlt_D)*pow2(pi4*r*r*rho_face)
         s% grada_at_face(k) = grada_face
         s% gradr(k) = mlt_basics(mlt_gradr)
         s% scale_height(k) = mlt_basics(mlt_scale_height)
         
         !if (s% model_number == 3005 .and. k == 925 .and. s% mlt_mixing_type(k) == 0) then
         !if (k==930) then
         if (.false.) then
!$OMP critical
               write(*,2) 's% mlt_cdc(k)', k, s% mlt_cdc(k)
               write(*,3) 's% mlt_mixing_type(k)', k, s% mlt_mixing_type(k)
!$OMP end critical
            !stop 'mlt info'
         end if

         if (is_bad_num(s% mlt_cdc(k)) .or. s% mlt_cdc(k) < 0 .or. &
				 is_bad_num(s% mlt_vc(k)) .or. s% mlt_vc(k) < 0) then
            if (s% report_ierr) then
               write(*,2) 's% mlt_cdc(k)', k, s% mlt_cdc(k)
               if (k > 1) write(*,2) 's% r(k-1)', k-1, s% r(k-1)
               write(*,2) 's% r(k)', k, s% r(k)
               if (k < s% nz) write(*,2) 's% r(k+1)', k+1, s% r(k+1)
               if (k > 1) write(*,2) 's% rho(k-1)', k-1, s% rho(k-1)
               write(*,2) 'rho_face', k, rho_face
               write(*,2) 's% rho(k)', k, s% rho(k)
               if (k > 1) write(*,2) 's% dq(k-1)', k-1, s% dq(k-1)
               write(*,2) 's% dq(k)', k, s% dq(k)
               write(*,2) 's% lnS(k)/ln10', k, s% lnS(k)/ln10
               write(*,2) 's% lnS_pre(k)/ln10', k, s% lnS_pre(k)/ln10
               write(*,2) '(lnS - lnS_pre)/ln10', k, (s% lnS(k) - s% lnS_pre(k))/ln10
               write(*,2) 's% T(k)', k, s% T(k)
               write(*,2) 'dt', k, s% dt
               write(*,2) 's% eps_grav(k)', k, s% eps_grav(k)
               write(*,2) 's% dlnT_dt(k)', k, s% dlnT_dt(k)
               write(*,2) 's% dlnd_dt(k)', k, s% dlnd_dt(k)
               if (.true.) then
!$OMP critical
                  call show_stuff
                  write(*,*)
                  !stop 'mlt info'
!$OMP end critical
               end if
            end if
            ierr = -1
            return
            !stop 'mlt info'
         end if

         s% gradL(k) = mlt_basics(mlt_gradL)

         Schwarzschild_stable = (s% gradr(k) < grada_face)
         Ledoux_stable = (s% gradr(k) < s% gradL(k))
         
         s% d_gradr_dlnR(k) = mlt_partials(mlt_dlnR, mlt_gradr)
         s% d_gradr_dL(k) = mlt_partials(mlt_dL, mlt_gradr)
         
         s% d_gradr_dlnd00(k) = alfa*( &
            mlt_partials(mlt_dlnd, mlt_gradr) + &
            mlt_partials(mlt_dP, mlt_gradr)*s% P(k)*s% chiRho(k) + &
            mlt_partials(mlt_dchiRho, mlt_gradr)*s% d_eos_dlnd(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_gradr)*s% d_eos_dlnd(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_gradr)*s% d_eos_dlnd(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_gradr)*s% d_opacity_dlnd(k) + &
            mlt_partials(mlt_dgrada, mlt_gradr)*s% d_eos_dlnd(i_grad_ad, k))
         
         s% d_gradr_dlnT00(k) = alfa*( &
            mlt_partials(mlt_dlnT, mlt_gradr) + &
            mlt_partials(mlt_dP, mlt_gradr)*s% P(k)*s% chiT(k) + &
            mlt_partials(mlt_dchiRho, mlt_gradr)*s% d_eos_dlnT(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_gradr)*s% d_eos_dlnT(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_gradr)*s% d_eos_dlnT(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_gradr)*s% d_opacity_dlnT(k) + &
            mlt_partials(mlt_dgrada, mlt_gradr)*s% d_eos_dlnT(i_grad_ad, k))

         if (k == 1) then
         
            s% d_gradr_dlndm1(k) = 0d0
            s% d_gradr_dlnTm1(k) = 0d0
            
         else

            s% d_gradr_dlndm1(k) = beta*( &
               mlt_partials(mlt_dlnd, mlt_gradr) + &
               mlt_partials(mlt_dP, mlt_gradr)*s% P(k-1)*s% chiRho(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_gradr)*s% d_eos_dlnd(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_gradr)*s% d_eos_dlnd(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_gradr)*s% d_eos_dlnd(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_gradr)*s% d_opacity_dlnd(k-1) + &
               mlt_partials(mlt_dgrada, mlt_gradr)*s% d_eos_dlnd(i_grad_ad, k-1))
            
            s% d_gradr_dlnTm1(k) = beta*( &
               mlt_partials(mlt_dlnT, mlt_gradr) + &
               mlt_partials(mlt_dP, mlt_gradr)*s% P(k-1)*s% chiT(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_gradr)*s% d_eos_dlnT(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_gradr)*s% d_eos_dlnT(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_gradr)*s% d_eos_dlnT(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_gradr)*s% d_opacity_dlnT(k-1) + &
               mlt_partials(mlt_dgrada, mlt_gradr)*s% d_eos_dlnT(i_grad_ad, k-1))
            
         end if
         
         s% conv_dP_term(k) = mlt_basics(mlt_conv_dP_term)
         s% d_conv_dP_term_dlnR(k) = mlt_partials(mlt_dlnR, mlt_conv_dP_term)
         s% d_conv_dP_term_dL(k) = mlt_partials(mlt_dL, mlt_conv_dP_term)
         
         s% d_conv_dP_term_dlnd00(k) = alfa*( &
            mlt_partials(mlt_dlnd, mlt_conv_dP_term) + &
            mlt_partials(mlt_dP, mlt_conv_dP_term)*s% P(k)*s% chiRho(k) + &
            mlt_partials(mlt_dchiRho, mlt_conv_dP_term)*s% d_eos_dlnd(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_conv_dP_term)*s% d_eos_dlnd(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_conv_dP_term)*s% d_eos_dlnd(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_conv_dP_term)*s% d_opacity_dlnd(k) + &
            mlt_partials(mlt_dgrada, mlt_conv_dP_term)*s% d_eos_dlnd(i_grad_ad, k))
         
         s% d_conv_dP_term_dlnT00(k) = alfa*( &
            mlt_partials(mlt_dlnT, mlt_conv_dP_term) + &
            mlt_partials(mlt_dP, mlt_conv_dP_term)*s% P(k)*s% chiT(k) + &
            mlt_partials(mlt_dchiRho, mlt_conv_dP_term)*s% d_eos_dlnT(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_conv_dP_term)*s% d_eos_dlnT(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_conv_dP_term)*s% d_eos_dlnT(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_conv_dP_term)*s% d_opacity_dlnT(k) + &
            mlt_partials(mlt_dgrada, mlt_conv_dP_term)*s% d_eos_dlnT(i_grad_ad, k))

         if (k == 1) then
         
            s% d_conv_dP_term_dlndm1(k) = 0d0
            s% d_conv_dP_term_dlnTm1(k) = 0d0
         
         else

            s% d_conv_dP_term_dlndm1(k) = beta*( &
               mlt_partials(mlt_dlnd, mlt_conv_dP_term) + &
               mlt_partials(mlt_dP, mlt_conv_dP_term)*s% P(k-1)*s% chiRho(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_conv_dP_term)*s% d_eos_dlnd(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_conv_dP_term)*s% d_eos_dlnd(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_conv_dP_term)*s% d_eos_dlnd(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_conv_dP_term)*s% d_opacity_dlnd(k-1) + &
               mlt_partials(mlt_dgrada, mlt_conv_dP_term)*s% d_eos_dlnd(i_grad_ad, k-1))
            
            s% d_conv_dP_term_dlnTm1(k) = beta*( &
               mlt_partials(mlt_dlnT, mlt_conv_dP_term) + &
               mlt_partials(mlt_dP, mlt_conv_dP_term)*s% P(k-1)*s% chiT(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_conv_dP_term)*s% d_eos_dlnT(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_conv_dP_term)*s% d_eos_dlnT(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_conv_dP_term)*s% d_eos_dlnT(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_conv_dP_term)*s% d_opacity_dlnT(k-1) + &
               mlt_partials(mlt_dgrada, mlt_conv_dP_term)*s% d_eos_dlnT(i_grad_ad, k-1))
            
         end if

         s% gradT(k) = mlt_basics(mlt_gradT)
         s% d_gradT_dlnR(k) = mlt_partials(mlt_dlnR, mlt_gradT)
         s% d_gradT_dL(k) = mlt_partials(mlt_dL, mlt_gradT)
                         
         s% d_gradT_dlnd00(k) = alfa*( &
            mlt_partials(mlt_dlnd, mlt_gradT) + &
            mlt_partials(mlt_dP, mlt_gradT)*s% P(k)*s% chiRho(k) + &
            mlt_partials(mlt_dchiRho, mlt_gradT)*s% d_eos_dlnd(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_gradT)*s% d_eos_dlnd(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_gradT)*s% d_eos_dlnd(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_gradT)*s% d_opacity_dlnd(k) + &
            mlt_partials(mlt_dgrada, mlt_gradT)*s% d_eos_dlnd(i_grad_ad, k))
   
         s% d_gradT_dlnT00(k) = alfa*( &
            mlt_partials(mlt_dlnT, mlt_gradT) + &
            mlt_partials(mlt_dP, mlt_gradT)*s% P(k)*s% chiT(k) + &
            mlt_partials(mlt_dchiRho, mlt_gradT)*s% d_eos_dlnT(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_gradT)*s% d_eos_dlnT(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_gradT)*s% d_eos_dlnT(i_Cp, k) + &
            mlt_partials(mlt_dopacity, mlt_gradT)*s% d_opacity_dlnT(k) + &
            mlt_partials(mlt_dgrada, mlt_gradT)*s% d_eos_dlnT(i_grad_ad, k))
         
         if (k == 1) then
         
            s% d_gradT_dlndm1(k) = 0d0
            s% d_gradT_dlnTm1(k) = 0d0
         
         else

            s% d_gradT_dlndm1(k) = beta*( &
               mlt_partials(mlt_dlnd, mlt_gradT) + &
               mlt_partials(mlt_dP, mlt_gradT)*s% P(k-1)*s% chiRho(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_gradT)*s% d_eos_dlnd(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_gradT)*s% d_eos_dlnd(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_gradT)*s% d_eos_dlnd(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_gradT)*s% d_opacity_dlnd(k-1) + &
               mlt_partials(mlt_dgrada, mlt_gradT)*s% d_eos_dlnd(i_grad_ad, k-1))
      
            s% d_gradT_dlnTm1(k) = beta*( &
               mlt_partials(mlt_dlnT, mlt_gradT) + &
               mlt_partials(mlt_dP, mlt_gradT)*s% P(k-1)*s% chiT(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_gradT)*s% d_eos_dlnT(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_gradT)*s% d_eos_dlnT(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_gradT)*s% d_eos_dlnT(i_Cp, k-1) + &
               mlt_partials(mlt_dopacity, mlt_gradT)*s% d_opacity_dlnT(k-1) + &
               mlt_partials(mlt_dgrada, mlt_gradT)*s% d_eos_dlnT(i_grad_ad, k-1))

         end if
         
         s% mlt_D(k) = mlt_basics(mlt_D)

         s% d_mlt_D_dlnR(k) = mlt_partials(mlt_dlnR, mlt_D)
         s% d_mlt_D_dL(k) = mlt_partials(mlt_dL, mlt_D)
         
         s% d_mlt_D_dlnd00(k) = alfa*( &
            mlt_partials(mlt_dlnd, mlt_D) + &
            mlt_partials(mlt_dP, mlt_D)*s% P(k)*s% chiRho(k) + &
            mlt_partials(mlt_dchiRho, mlt_D)*s% d_eos_dlnd(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_D)*s% d_eos_dlnd(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_D)*s% d_eos_dlnd(i_Cp, k) + &
            mlt_partials(mlt_dgrada, mlt_D)*s% d_eos_dlnd(i_grad_ad, k) + &
            mlt_partials(mlt_dopacity, mlt_D)*s% d_opacity_dlnd(k))
         
         s% d_mlt_D_dlnT00(k) = alfa*( &
            mlt_partials(mlt_dlnT, mlt_D) + &
            mlt_partials(mlt_dP, mlt_D)*s% P(k)*s% chiT(k) + &
            mlt_partials(mlt_dchiRho, mlt_D)*s% d_eos_dlnT(i_chiRho, k) + &
            mlt_partials(mlt_dchiT, mlt_D)*s% d_eos_dlnT(i_chiT, k) + &
            mlt_partials(mlt_dCp, mlt_D)*s% d_eos_dlnT(i_Cp, k) + &
            mlt_partials(mlt_dgrada, mlt_D)*s% d_eos_dlnT(i_grad_ad, k) + &
            mlt_partials(mlt_dopacity, mlt_D)*s% d_opacity_dlnT(k))

         if (k == 1) then
         
            s% d_mlt_D_dlndm1(k) = 0d0
            s% d_mlt_D_dlnTm1(k) = 0d0
         
         else

            s% d_mlt_D_dlndm1(k) = beta*( &
               mlt_partials(mlt_dlnd, mlt_D) + &
               mlt_partials(mlt_dP, mlt_D)*s% P(k-1)*s% chiRho(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_D)*s% d_eos_dlnd(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_D)*s% d_eos_dlnd(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_D)*s% d_eos_dlnd(i_Cp, k-1) + &
               mlt_partials(mlt_dgrada, mlt_D)*s% d_eos_dlnd(i_grad_ad, k-1) + &
               mlt_partials(mlt_dopacity, mlt_D)*s% d_opacity_dlnd(k-1))
            
            s% d_mlt_D_dlnTm1(k) = beta*( &
               mlt_partials(mlt_dlnT, mlt_D) + &
               mlt_partials(mlt_dP, mlt_D)*s% P(k-1)*s% chiT(k-1) + &
               mlt_partials(mlt_dchiRho, mlt_D)*s% d_eos_dlnT(i_chiRho, k-1) + &
               mlt_partials(mlt_dchiT, mlt_D)*s% d_eos_dlnT(i_chiT, k-1) + &
               mlt_partials(mlt_dCp, mlt_D)*s% d_eos_dlnT(i_Cp, k-1) + &
               mlt_partials(mlt_dgrada, mlt_D)*s% d_eos_dlnT(i_grad_ad, k-1) + &
               mlt_partials(mlt_dopacity, mlt_D)*s% d_opacity_dlnT(k-1))
            
         end if

         if (.false. .and. k == 920 .and. s% mlt_D(k) > 0 .and. &
             s% d_mlt_D_dlnd00(k) > 1000*s% mlt_D(k)) then
         
            write(*,2) 'd_ln_mlt_D_dlnd00(k)', k, &
               s% d_mlt_D_dlnd00(k)/s% mlt_D(k), s% d_mlt_D_dlnd00(k), s% mlt_D(k)
            write(*,2) 'alfa', k, alfa
            
            alfa = mlt_partials(mlt_dlnd, mlt_D)
            write(*,2) 'mlt_dlnd', k, alfa/s% mlt_D(k), alfa, s% mlt_D(k)
            
            alfa = mlt_partials(mlt_dP, mlt_D)*s% P(k)*s% chiRho(k)
            write(*,2) 'mlt_dP', k, alfa/s% mlt_D(k), alfa, s% mlt_D(k)
            
            alfa = mlt_partials(mlt_dchiRho, mlt_D)*s% d_eos_dlnd(i_chiRho, k)
            write(*,2) 'mlt_dchiRho', k, alfa/s% mlt_D(k), alfa, s% mlt_D(k)
            
            alfa = mlt_partials(mlt_dchiT, mlt_D)*s% d_eos_dlnd(i_chiT, k)
            write(*,2) 'mlt_dchiT', k, alfa/s% mlt_D(k), alfa, s% mlt_D(k)
            
            alfa = mlt_partials(mlt_dCp, mlt_D)*s% d_eos_dlnd(i_Cp, k)
            write(*,2) 'mlt_dCp', k, alfa/s% mlt_D(k), alfa, s% mlt_D(k)
            
            alfa = mlt_partials(mlt_dgrada, mlt_D)*s% d_eos_dlnd(i_grad_ad, k)
            write(*,2) 'mlt_dgrada', k, alfa/s% mlt_D(k), alfa, s% mlt_D(k)
            
            alfa = mlt_partials(mlt_dopacity, mlt_D)*s% d_opacity_dlnd(k)
            write(*,2) 'mlt_dopacity', k, alfa/s% mlt_D(k), alfa, s% mlt_D(k)
            
            call show_stuff
            stop

         end if
         
         if (s% q(k) <= s% qmax_zero_non_radiative_luminosity) then
            f = 0d0
         else
            f = s% adjust_mlt_gradT_fraction(k)
         end if
         call adjust_gradT_fraction(s, k, f) 
         
         if (s% gradr(k) <= 0) then
            s% L_conv_div_L(k) = 0
         else
            s% L_conv_div_L(k) = 1d0 - s% gradT(k)/s% gradr(k) ! C&G 14.109
         end if

         if (is_bad_num(s% gradT(k))) then
            if (s% report_ierr) then
               write(*,2) 's% gradT(k)', k, s% gradT(k)
               if (.false.) then
!$OMP critical
                  call show_stuff
!$OMP end critical
               end if
               return
               !stop 'mlt info'
            end if
            ierr = -1
            return
            !stop 'mlt info'
         end if
         
         !if (s% model_number == 3005 .and. k == 925 .and. s% mlt_mixing_type(k) == 0) then
         !if (k==930) then
         if (.false.) then
!$OMP critical
                  call show_stuff
!$OMP end critical
            stop 'mlt info'
         end if

         if (k == s% trace_k) then
            write(*,5) 'mlt_info gradT', &
               k, s% newton_iter, s% model_number, s% newton_adjust_iter, s% gradT(k)
            write(*,1) 'T =', T_face
            write(*,1) 'rho =', rho_face
            write(*,1) 'L =', L
            write(*,1) 'P =', P_face
            write(*,1) 'chiRho =', chiRho_face         
            write(*,1) 'chiT =', chiT_face
            write(*,1) 'Cp =', Cp_face
            write(*,1) 'Cv =', Cv_face
            write(*,1) 'X =', xh_face
            write(*,1) 'opacity =', opacity_face
            write(*,1) 'grada =', grada_face
         end if
         
         
         contains


         subroutine show_stuff
            real(dp) :: vsem, Lambda, D, Dsem, Drad, radiative_conductivity
            include 'formats'
            write(*,*)
            write(*,*) 'do1_mlt info for k, nz', k, s% nz
            write(*,2) 's% model_number', s% model_number
            write(*,2) 's% newton_iter', s% newton_iter
            write(*,*)
            write(*,1) 'cgrav =', s% cgrav(k)
            write(*,1) 'm =', m
            write(*,1) 'r =', r
            write(*,1) 'T =', T_face
            write(*,1) 'rho =', rho_face
            write(*,1) 'L =', L
            write(*,1) 'P =', P_face
            write(*,1) 'chiRho =', chiRho_face         
            write(*,1) 'chiT =', chiT_face
            write(*,1) 'Cp =', Cp_face
            write(*,1) 'Cv =', Cv_face
            write(*,1) 'csound =', s% csound_at_face(k)
            write(*,1) 'X =', xh_face
            write(*,1) 'opacity =', opacity_face
            write(*,1) 'grada =', grada_face
            write(*,*)
            write(*,1) 'gradr_factor =', gradr_factor
            write(*,1) 'gradL_composition_term =', s% gradL_composition_term(k)
            write(*,*) 
            write(*,1) 'alpha_semiconvection =', asc
            write(*,1) 'semiconvection_option = "' // trim(s% semiconvection_option) // '" '
            write(*,*) 
            write(*,1) 'thermohaline_coeff =', thc
            write(*,1) 'thermohaline_option = "' // trim(s% thermohaline_option) // '"'
            write(*,2) 'dominant_iso_for_thermohaline =', s% dominant_iso_for_thermohaline(k)
            write(*,*) 
            write(*,1) 'mixing_length_alpha =', mixing_length_alpha
            if (s% alt_scale_height_flag) then
               write(*,'(a50)') '         alt_scale_height = .true.'
            else
               write(*,'(a50)') '         alt_scale_height = .false.'
            end if
            write(*,*)
            write(*,1) 'Henyey_y_param =', s% Henyey_MLT_y_param
            write(*,1) 'Henyey_nu_param =', s% Henyey_MLT_nu_param
            write(*,*) 
            write(*,1) "MLT_option = '" // trim(s% MLT_option) // "'"
            write(*,*)
            write(*,1) 'prev_conv_vel =', prev_conv_vel
            write(*,1) 'max_conv_vel =', max_conv_vel
            write(*,1) 'dt =', dt
            write(*,1) 'tau_face =', tau_face
            write(*,*)
            write(*,*) '--------------------------------------'
            write(*,*)
            write(*,*)
            write(*,*)

            write(*,1) 'logRho =', s% lnd(k)/ln10
            write(*,1) 'logT =', s% lnT(k)/ln10
            write(*,*)
            write(*,*)
            write(*,3) 'k, nz', k, s% nz
            write(*,*)
            write(*,*)
            if (k > 1) then
               write(*,2) 's% opacity(k)', k, s% opacity(k)
               write(*,2) 's% opacity(k-1)', k-1, s% opacity(k-1)
               write(*,1) 'alfa', alfa
               write(*,1) 'beta', beta
               write(*,1) 'alfa', alfa*s% opacity(k)
               write(*,1) 'beta', beta*s% opacity(k-1)
               write(*,1) 'opacity_face', opacity_face
            end if
            
            if (ierr /= 0) return
            write(*,1) 's% gradr(k)', s% gradr(k)
            write(*,1) 's% gradT(k)', s% gradT(k)
            write(*,1) 's% gradL(k)', s% gradL(k)
            write(*,1) 's% gradL(k) - grada_at_face', s% gradL(k) - grada_face
            write(*,*)
            write(*,1) 's% mlt_D(k)', s% mlt_D(k)
            write(*,1) 's% mlt_vc(k)', s% mlt_vc(k)
            write(*,2) 's% mlt_mixing_type(k)', s% mlt_mixing_type(k)
            write(*,1) 's% mlt_mixing_length(k)', s% mlt_mixing_length(k)
            write(*,1) 's% d_gradT_dlnd00(k)', s% d_gradT_dlnd00(k)
            write(*,1) 's% d_gradT_dlnT00(k)', s% d_gradT_dlnT00(k)
            write(*,1) 's% d_gradT_dlndm1(k)', s% d_gradT_dlndm1(k)
            write(*,1) 's% d_gradT_dlnTm1(k)', s% d_gradT_dlnTm1(k)
            write(*,1) 's% d_gradT_dlnR(k)', s% d_gradT_dlnR(k)
            write(*,1) 's% d_gradT_dL(k)', s% d_gradT_dL(k)
            write(*,*)
            
            write(*,*) 'Schwarzschild_stable', Schwarzschild_stable
            write(*,*) 'Ledoux_stable', Ledoux_stable
            write(*,*)
            
            !if (s% T(k) < 1d-20) stop 'bogus T: do1_mlt'
            
            !stop 'do1_mlt'
            
         end subroutine show_stuff

      end subroutine do1_mlt
      
      
      logical function must_limit_conv_vel(s,k0)
         type (star_info), pointer :: s 
         integer, intent(in) :: k0
         real(dp) :: alfa, beta, one_m_f
         integer :: k, wdth
         include 'formats'
         must_limit_conv_vel = .false.
         if (s% q(k0) <= s% max_q_for_limit_conv_vel .or. &
             s% m(k0) <= s% max_mass_in_gm_for_limit_conv_vel .or. &
             s% r(k0) <= s% max_r_in_cm_for_limit_conv_vel) then
            must_limit_conv_vel = .true.
            return
         end if
         wdth = s% width_for_limit_conv_vel
         if (wdth < 0) return
         do k = max(2,k0-wdth),min(s% nz,k0+wdth)
            if (s% csound(k) < s% v(k) .and. s% v(k) <= s% csound(k-1)) then
               must_limit_conv_vel = .true.
               return
            end if
         end do      
      end function must_limit_conv_vel

         
      subroutine adjust_gradT_fraction(s,k,f)
         ! replace gradT by combo of grada_at_face and gradr
         ! then check excess
         use eos_def
         type (star_info), pointer :: s 
         real(dp), intent(in) :: f
         integer, intent(in) :: k        
         
         real(dp) :: alfa, beta, one_m_f
         
         include 'formats'
         
         if (f >= 0 .and. f <= 1) then

            ! alfa is the fraction coming from k; (1-alfa) from k-1.
            if (k == 1) then
               alfa = 1
            else
               alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
            end if
            beta = 1 - alfa
            
            if (f == 0d0) then
               s% gradT(k) = s% gradr(k)         
               s% d_gradT_dlnR(k) = s% d_gradr_dlnR(k)
               s% d_gradT_dL(k) = s% d_gradr_dL(k)
               s% d_gradT_dlnd00(k) = s% d_gradr_dlnd00(k)
               s% d_gradT_dlnT00(k) = s% d_gradr_dlnT00(k)
               if (k > 1) then               
                  s% d_gradT_dlndm1(k) = s% d_gradr_dlndm1(k)
                  s% d_gradT_dlnTm1(k) = s% d_gradr_dlndm1(k)               
               end if         
            else ! mix
               one_m_f = 1d0 - f         
               s% gradT(k) = f*s% grada_at_face(k) + one_m_f*s% gradr(k)         
               s% d_gradT_dlnR(k) = one_m_f*s% d_gradr_dlnR(k)
               s% d_gradT_dL(k) = one_m_f*s% d_gradr_dL(k)
               s% d_gradT_dlnd00(k) = &
                  f*alfa*s% d_eos_dlnd(i_grad_ad, k) + one_m_f*s% d_gradr_dlnd00(k)
               s% d_gradT_dlnT00(k) = &
                  f*alfa*s% d_eos_dlnT(i_grad_ad, k) + one_m_f*s% d_gradr_dlnT00(k)
               if (k > 1) then               
                  s% d_gradT_dlndm1(k) = &
                     f*beta*s% d_eos_dlnd(i_grad_ad, k-1) + one_m_f*s% d_gradr_dlndm1(k)
                  s% d_gradT_dlnTm1(k) = &
                     f*beta*s% d_eos_dlnT(i_grad_ad, k-1) + one_m_f*s% d_gradr_dlndm1(k)               
               end if         
            end if
            
            
         end if

         s% gradT_sub_grada(k) = s% gradT(k) - s% grada_at_face(k) ! gradT_excess
                          
         call adjust_gradT_excess(s, k)        

      end subroutine adjust_gradT_fraction
      
         
      subroutine adjust_gradT_excess(s, k)
         use eos_def
         type (star_info), pointer :: s 
         integer, intent(in) :: k        
         
         real(dp) :: alfa, beta, log_tau, gradT_excess_alpha, &
            d_grada_at_face_dlnd00, d_grada_at_face_dlnT00, &
            d_grada_at_face_dlndm1, d_grada_at_face_dlnTm1
         
         include 'formats'
         
         !s% gradT_excess_alpha is calculated at start of step and held constant during iterations
         ! gradT_excess_alpha = 0 means no efficiency boost; = 1 means full efficiency boost
         
         gradT_excess_alpha = s% gradT_excess_alpha
         
         if (gradT_excess_alpha <= 0 .or. &
             s% gradT_sub_grada(k) <= s% gradT_excess_f1) return
             
         if (s% lnT(k)/ln10 > s% gradT_excess_max_logT) return
         
         log_tau = log10_cr(s% tau(k))
         if (log_tau < s% gradT_excess_max_log_tau_full_off) return
             
         ! boost efficiency of energy transport

         ! alfa is the fraction coming from k; (1-alfa) from k-1.
         if (k == 1) then
            alfa = 1
         else
            alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
         end if
         beta = 1 - alfa
         
         ! grada_face = alfa*s% grada(k) + beta*s% grada(k-1)
         d_grada_at_face_dlnd00 = alfa*s% d_eos_dlnd(i_grad_ad, k)
         d_grada_at_face_dlnT00 = alfa*s% d_eos_dlnT(i_grad_ad, k)
         if (k > 1) then
            d_grada_at_face_dlndm1 = beta*s% d_eos_dlnd(i_grad_ad, k-1)
            d_grada_at_face_dlnTm1 = beta*s% d_eos_dlnT(i_grad_ad, k-1)
         else
            d_grada_at_face_dlndm1 = 0
            d_grada_at_face_dlnTm1 = 0
         end if            
                  
         if (log_tau < s% gradT_excess_min_log_tau_full_on) &
            gradT_excess_alpha = gradT_excess_alpha* &
               (log_tau - s% gradT_excess_max_log_tau_full_off)/ &
               (s% gradT_excess_min_log_tau_full_on - s% gradT_excess_max_log_tau_full_off)     
         
         alfa = s% gradT_excess_f2 ! for full boost, use this fraction of gradT
         if (gradT_excess_alpha < 1) then ! only partial boost, so increase alfa
            ! alfa goes to 1 as gradT_excess_alpha goes to 0
            ! alfa unchanged as gradT_excess_alpha goes to 1
            alfa = alfa + (1d0 - alfa)*(1d0 - gradT_excess_alpha)
         end if
         beta = 1d0 - alfa
         
         s% gradT(k) = alfa*s% gradT(k) + beta*s% grada_at_face(k)
         
         s% d_gradT_dlnR(k) = alfa*s% d_gradT_dlnR(k)
         s% d_gradT_dL(k) = alfa*s% d_gradT_dL(k)
         
         s% d_gradT_dlnd00(k) = &
            alfa*s% d_gradT_dlnd00(k) + beta*d_grada_at_face_dlnd00
         s% d_gradT_dlnT00(k) = &
            alfa*s% d_gradT_dlnT00(k) + beta*d_grada_at_face_dlnT00
         
         if (k > 1) then
            s% d_gradT_dlndm1(k) = &
               alfa*s% d_gradT_dlndm1(k) + beta*d_grada_at_face_dlndm1
            s% d_gradT_dlnTm1(k) = &
               alfa*s% d_gradT_dlnTm1(k) + beta*d_grada_at_face_dlnTm1              
         end if         
         
      end subroutine adjust_gradT_excess         
         

      subroutine do1_mlt_eval(  &
            s, k, cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada, &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
            mixing_type, mlt_basics, mlt_partials1, ierr)
         use mlt_lib, only: mlt_eval
         type (star_info), pointer :: s 
         integer, intent(in) :: k        
         real(dp), intent(in) :: cgrav, m, r, T, Rho, L, P
         real(dp), intent(in) :: chiRho, chiT, Cp, Cv, csound, X, opacity, grada, gradr_factor
         real(dp), intent(in) :: gradL_composition_term
         real(dp), intent(in) :: alpha_semiconvection, thermohaline_coeff
         real(dp), intent(in) :: mixing_length_alpha, remove_small_D_limit
         logical, intent(in) :: alt_scale_height
         character (len=*), intent(in) :: &
            semiconvection_option, MLT_option, thermohaline_option 
         integer, intent(in) :: dominant_iso_for_thermohaline
         real(dp), intent(in) :: Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau
         logical, intent(in) :: MLT_dbg
         integer, intent(out) :: mixing_type ! none, convective, semiconvective, or salt finger
         real(dp), intent(out) :: mlt_basics(num_mlt_results)
         real(dp), intent(out), pointer :: mlt_partials1(:) ! =(num_mlt_partials, num_mlt_results)
            ! e.g., mlt_partials(mlt_dlnT,mlt_gradT) has partial wrt lnT of gradT
         integer, intent(out) :: ierr
         
         
         include 'formats'
         
         if (s% use_other_mlt) then
            call s% other_mlt(  &
               s% id, k, cgrav, m, r, T, rho, L, P, &
               chiRho, chiT, Cp, Cv, csound, X, opacity, grada, &
               gradr_factor, gradL_composition_term, &
               alpha_semiconvection, semiconvection_option, &
               thermohaline_coeff, thermohaline_option, &
               dominant_iso_for_thermohaline, &
               mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
               MLT_option, Henyey_y_param, Henyey_nu_param, &
               prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
               mixing_type, mlt_basics, mlt_partials1, ierr)
         else
            call mlt_eval(  &
               cgrav, m, r, T, rho, L, P, &
               chiRho, chiT, Cp, Cv, csound, X, opacity, grada, &
               gradr_factor, gradL_composition_term, &
               alpha_semiconvection, semiconvection_option, &
               thermohaline_coeff, thermohaline_option, &
               dominant_iso_for_thermohaline, &
               mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
               MLT_option, Henyey_y_param, Henyey_nu_param, &
               prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
               mixing_type, mlt_basics, mlt_partials1, ierr)
         end if
      end subroutine do1_mlt_eval
      

      subroutine set_grads(s, ierr)
         use chem_def, only: chem_isos
         use star_utils, only: smooth
         type (star_info), pointer :: s         
         integer, intent(out) :: ierr
         
         integer :: k, nz, j, cid, max_cid
         real(dp) :: val, max_val, A, Z
         real(dp), pointer, dimension(:) :: dlnP, dlnd, dlnT
            
         include 'formats'
         
         ierr = 0
         nz = s% nz
         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k = 2, nz
            dlnP(k) = s% lnP(k-1) - s% lnP(k)
            dlnd(k) = s% lnd(k-1) - s% lnd(k)
            dlnT(k) = s% lnT(k-1) - s% lnT(k)
         end do
         dlnP(1) = dlnP(2)
         dlnd(1) = dlnd(2)
         dlnT(1) = dlnT(2)
         
         call smooth(dlnP,nz)
         call smooth(dlnd,nz)
         call smooth(dlnT,nz)
         
         s% grad_density(1) = 0
         s% grad_temperature(1) = 0
         do k = 2, nz
            if (dlnP(k) >= 0) then
               s% grad_density(k) = 0
               s% grad_temperature(k) = 0
            else
               s% grad_density(k) = dlnd(k)/dlnP(k)
               s% grad_temperature(k) = dlnT(k)/dlnP(k)
            end if
         end do
         
         call smooth(s% grad_density,nz)
         call smooth(s% grad_temperature,nz)
         
         do k=1,nz
            s% gradL_composition_term(k) = s% unsmoothed_brunt_B(k)
         end do
         call smooth_gradL_composition_term
         
         call dealloc
                     
         do k=3,nz-2
            max_cid = 0
            max_val = -1d99
            do j=1,s% species
               cid = s% chem_id(j)
               A = dble(chem_isos% Z_plus_N(cid))
               Z = dble(chem_isos% Z(cid))
               val = (s% xa(j,k-2) + s% xa(j,k-1) - s% xa(j,k) - s% xa(j,k+1))*(1d0 + Z)/A
               if (val > max_val) then
                  max_val = val
                  max_cid = cid
               end if
            end do
            s% dominant_iso_for_thermohaline(k) = max_cid
         end do
         s% dominant_iso_for_thermohaline(1:2) = &
            s% dominant_iso_for_thermohaline(3)
         s% dominant_iso_for_thermohaline(nz-1:nz) = &
            s% dominant_iso_for_thermohaline(nz-2)
         
                  
         contains
         
         
         subroutine smooth_gradL_composition_term
            use star_utils, only: weighed_smoothing
            logical, parameter :: preserve_sign = .false.
            real(dp), pointer, dimension(:) :: work
            ierr = 0
            work => dlnd
            if (s% num_cells_for_smooth_gradL_composition_term <= 0) return
            call weighed_smoothing( &
               s% gradL_composition_term, s% nz, &
               s% num_cells_for_smooth_gradL_composition_term, preserve_sign, work)
         end subroutine smooth_gradL_composition_term
         
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            call get_work_array(s, dlnP, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnd, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
            call get_work_array(s, dlnT, nz, nz_alloc_extra, 'mlt', ierr)
            if (ierr /= 0) return            
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            call return_work_array(s, dlnP, 'mlt')            
            call return_work_array(s, dlnd, 'mlt')            
            call return_work_array(s, dlnT, 'mlt')            
         end subroutine dealloc
         
         
      end subroutine set_grads
      

      subroutine switch_to_no_mixing(s,k)
         type (star_info), pointer :: s
         integer, intent(in) :: k
         !write(*,*) 'switch_to_no_mixing', k
         s% mlt_mixing_type(k) = no_mixing
         s% mlt_mixing_length(k) = 0
         s% mlt_D(k) = 0
         s% mlt_vc(k) = 0
      end subroutine switch_to_no_mixing
      
      
      subroutine switch_to_radiative(s,k)
         type (star_info), pointer :: s
         integer, intent(in) :: k
         !write(*,*) 'switch_to_radiative', k
         s% gradT(k) = s% gradr(k)
         s% d_gradT_dlnd00(k) = s% d_gradr_dlnd00(k)
         s% d_gradT_dlnT00(k) = s% d_gradr_dlnT00(k)
         s% d_gradT_dlndm1(k) = s% d_gradr_dlndm1(k)
         s% d_gradT_dlnTm1(k) = s% d_gradr_dlnTm1(k)
         s% d_gradT_dlnR(k) = s% d_gradr_dlnR(k)
         s% d_gradT_dL(k) = s% d_gradr_dL(k)
      end subroutine switch_to_radiative
      
      
      subroutine switch_to_adiabatic(s,k)
         use eos_def, only: i_grad_ad
         type (star_info), pointer :: s
         integer, intent(in) :: k
         !write(*,*) 'switch_to_adiabatic', k
         s% gradT(k) = s% grada(k)
         s% d_gradT_dlnd00(k) = s% d_eos_dlnd(i_grad_ad, k)
         s% d_gradT_dlnT00(k) = s% d_eos_dlnT(i_grad_ad, k)
         s% d_gradT_dlndm1(k) = 0
         s% d_gradT_dlnTm1(k) = 0
         s% d_gradT_dlnR(k) = 0
         s% d_gradT_dL(k) = 0
      end subroutine switch_to_adiabatic
      
      
      subroutine set_gradT_excess_alpha(s, ierr)
         use alloc
         use star_utils, only: get_Lrad_div_Ledd, after_C_burn
         use chem_def, only: ih1, ihe4
         
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         real(dp) :: beta, lambda, phi, tmp, alpha, alpha2
         real(dp) :: &
            beta_limit, & 
            lambda1, &
            beta1, &
            lambda2, &
            beta2, &
            dlambda, &
            dbeta
         
         integer :: k, k_beta, k_lambda, nz, h1, he4
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,*) 'enter set_gradT_excess_alpha'
         
         ierr = 0
         if (.not. s% okay_to_reduce_gradT_excess) then
            s% gradT_excess_alpha = 0
            if (dbg) write(*,1) 'okay_to_reduce_gradT_excess'
            return
         end if
         nz = s% nz
         
         h1 = s% net_iso(ih1) 
         if (h1 /= 0) then
            if (s% xa(h1,nz) > s% gradT_excess_max_center_h1) then
               s% gradT_excess_alpha = 0
               if (dbg) write(*,1) 'gradT_excess_max_center_h1'
               return
            end if
         end if

         he4 = s% net_iso(ihe4) 
         if (he4 /= 0) then
            if (s% xa(he4,nz) < s% gradT_excess_min_center_he4) then
               s% gradT_excess_alpha = 0
               if (dbg) write(*,1) 'gradT_excess_min_center_he4'
               return
            end if
         end if

         beta = 1d0 ! beta = min over k of Pgas(k)/P(k)
         k_beta = 0
         do k=1,nz
            tmp = s% Pgas(k)/s% P(k)
            if (tmp < beta) then
               k_beta = k
               beta = tmp
            end if
         end do
         
         beta = beta*(1d0 + s% xa(1,nz))

         s% gradT_excess_min_beta = beta
         if (dbg) write(*,2) 'gradT_excess_min_beta', k_beta, beta
         
         lambda = 0d0 ! lambda = max over k of Lrad(k)/Ledd(k)
         do k=2,k_beta
            tmp = get_Lrad_div_Ledd(s,k)
            if (tmp > lambda) then
               k_lambda = k
               lambda = tmp
            end if
         end do         
         lambda = min(1d0,lambda)
         s% gradT_excess_max_lambda = lambda
         if (dbg) write(*,2) 'gradT_excess_max_lambda', k_lambda, lambda
         
         lambda1 = s% gradT_excess_lambda1
         beta1 = s% gradT_excess_beta1
         lambda2 = s% gradT_excess_lambda2
         beta2 = s% gradT_excess_beta2
         dlambda = s% gradT_excess_dlambda
         dbeta = s% gradT_excess_dbeta
         
         if (dbg) then
            write(*,1) 'lambda1', lambda1
            write(*,1) 'lambda2', lambda2
            write(*,1) 'lambda', lambda
            write(*,*)
            write(*,1) 'beta1', beta1
            write(*,1) 'beta2', beta2
            write(*,1) 'beta', beta
            write(*,*)
         end if
         
         ! alpha is fraction of full boost to apply
         ! depends on location in (beta,lambda) plane
         
         if (lambda1 < 0) then
            alpha = 1
         else if (lambda >= lambda1) then
            if (beta <= beta1) then
               alpha = 1
            else if (beta < beta1 + dbeta) then
               alpha = (beta1 + dbeta - beta)/dbeta
            else ! beta >= beta1 + dbeta
               alpha = 0
            end if
         else if (lambda >= lambda2) then
            beta_limit = beta2 + &
               (lambda - lambda2)*(beta1 - beta2)/(lambda1 - lambda2)
            if (beta <= beta_limit) then
               alpha = 1
            else if (beta < beta_limit + dbeta) then
               alpha = (beta_limit + dbeta - beta)/dbeta
            else
               alpha = 0
            end if
         else if (lambda > lambda2 - dlambda) then
            if (beta <= beta2) then
               alpha = 1
            else if (beta < beta2 + dbeta) then
               alpha = (lambda - (lambda2 - dlambda))/dlambda
            else ! beta >= beta2 + dbeta
               alpha = 0
            end if
         else ! lambda <= lambda2 - dlambda
            alpha = 0
         end if
         
         if (s% generations > 1 .and. lambda1 >= 0) then ! time smoothing
            s% gradT_excess_alpha = &
               (1d0 - s% gradT_excess_age_fraction)*alpha + &
               s% gradT_excess_age_fraction*s% gradT_excess_alpha_old
            if (s% gradT_excess_max_change > 0d0) then
               if (s% gradT_excess_alpha > s% gradT_excess_alpha_old) then
                  s% gradT_excess_alpha = min(s% gradT_excess_alpha, s% gradT_excess_alpha_old + &
                     s% gradT_excess_max_change)
               else
                  s% gradT_excess_alpha = max(s% gradT_excess_alpha, s% gradT_excess_alpha_old - &
                     s% gradT_excess_max_change)
               end if
            end if
         else
            s% gradT_excess_alpha = alpha
         end if
                  
         if (s% gradT_excess_alpha < 1d-4) s% gradT_excess_alpha = 0d0
         if (s% gradT_excess_alpha > 0.9999d0) s% gradT_excess_alpha = 1d0
         
         if (dbg) then
            write(*,1) 'gradT excess new', alpha
            write(*,1) 's% gradT_excess_alpha_old', s% gradT_excess_alpha_old
            write(*,1) 's% gradT_excess_alpha', s% gradT_excess_alpha
            write(*,*)
         end if
      
      end subroutine set_gradT_excess_alpha
      
      
      end module mlt_info
