! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module mix_smoothing

      use const_def
      use num_lib
      use utils_lib
      use star_private_def
      use mlt_def
      use star_utils, only: find_cell_for_mass

      implicit none


      contains


      subroutine set_newly_non_conv(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         integer :: num, num_old, j, nz
         logical, parameter :: dbg = .false.
         include 'formats'
         ierr = 0
         nz = s% nz
         s% newly_nonconvective(1:nz) = .false.
         num = s% n_conv_regions
         num_old = s% n_conv_regions_old
         if (num == 0 .or. num_old == 0) return
         if (dbg) then
            write(*,*)
            do j=1,s% n_conv_regions_old
               write(*,2) 'old conv region', j, s% cz_bot_mass_old(j)/Msun, s% cz_top_mass_old(j)/Msun
            end do
            write(*,*)
            do j=1,s% n_conv_regions
               write(*,2) 'conv region', j, s% cz_bot_mass(j)/Msun, s% cz_top_mass(j)/Msun
            end do
            write(*,*)
         end if
         if (dbg) write(*,*) 'set_newly_non_conv: call do_all_regions'
         call do_all_regions( &
            s, set_top_moved_down, set_bottom_moved_up, set_for_departed_region, &
            num_old, s% cz_top_mass_old, s% cz_bot_mass_old, & 
            num, s% cz_top_mass, s% cz_bot_mass, ierr)
      end subroutine set_newly_non_conv


      subroutine set_top_moved_down(s, nz, species, top_old, top_new)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, species
         real(dp), intent(in) :: top_old, top_new 
         integer :: ktop_old, ktop_new
         ktop_old = find_cell_for_mass(s,top_old)
         ktop_new = find_cell_for_mass(s,top_new)
         s% newly_nonconvective(ktop_old:ktop_new) = .true.
      end subroutine set_top_moved_down


      subroutine set_bottom_moved_up(s, nz, species, bot_old, bot_new)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, species
         real(dp), intent(in) :: bot_old, bot_new 
         integer :: kbot_old, kbot_new
         kbot_old = find_cell_for_mass(s,bot_old)
         kbot_new = find_cell_for_mass(s,bot_new)
         s% newly_nonconvective(kbot_new:kbot_old) = .true.
      end subroutine set_bottom_moved_up
      
      
      subroutine set_for_departed_region(s, nz, species, top, bot)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, species
         real(dp), intent(in) :: top, bot
         integer :: ktop, kbot
         ktop = find_cell_for_mass(s,top)
         kbot = find_cell_for_mass(s,bot)
         s% newly_nonconvective(ktop:kbot) = .true.
      end subroutine set_for_departed_region


      ! called after burn+mix to smooth abundances for newly non-convective
      subroutine smooth_newly_non_conv(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         integer :: num, num_old
         logical, parameter :: dbg = .false.
         include 'formats'
         ierr = 0
         if (dbg) write(*,1) 'smooth_newly_non_conv: initial Y(1200)', s% xa(4,1200)
         num = s% n_conv_regions
         num_old = s% n_conv_regions_old
         if (num == 0 .or. num_old == 0) then
            if (dbg) write(*,3) 'num == 0 .or. num_old == 0', num, num_old
            return
         end if
         if (dbg) write(*,*) 'call do_all_regions'
         call do_all_regions( &
            s, smooth_top_moved_down, smooth_bottom_moved_up, smooth_for_departed_region, &
            num_old, s% cz_top_mass_old, s% cz_bot_mass_old, & 
            num, s% cz_top_mass, s% cz_bot_mass, ierr)
         if (dbg) write(*,1) 'smooth_newly_non_conv: final Y(1200)', s% xa(4,1200)
      end subroutine smooth_newly_non_conv


      subroutine smooth_top_moved_down(s, nz, species, top_old, top_new)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, species
         real(dp), intent(in) :: top_old, top_new 
         ! mass coords of newly nonconvective region above downward moving top boundary
         
         integer :: ktop_old, k_above, ktop_new, k, j
         real(dp) :: dlnP, alfa, xa_above(species), xa_below(species), &
            lnP_top, lnP_bot, max_delta, dR, scale_height
         logical, parameter :: dbg = .false.
         
         include 'formats'

         ktop_new = find_cell_for_mass(s,top_new)
         ktop_old = find_cell_for_mass(s,top_old)
         k_above = min(ktop_old+1,nz)
         
         do k = k_above+1, ktop_new-1
            if (s% mixing_type(k) /= no_mixing) then
               if (dbg) write(*,2) 'smooth_top_moved_down mixing', k
               return
            end if
         end do
         dR = s% r(k_above) - s% r(ktop_new)
         scale_height = minval(s% scale_height(k_above:ktop_new))
         if (dR > scale_height*s% max_dR_div_Hp_for_smooth) then
            return
         end if
         
         do j=1,species
            xa_above(j) = s% xa(j,k_above)
         end do
         lnP_top = s% lnP(k_above)
         
         if (dbg) write(*,2) 'ktop_old', ktop_old, top_old/Msun
         if (dbg) write(*,2) 'k_above', k_above, s% m(k_above)/Msun

         do j=1,species
            xa_below(j) = s% xa(j,ktop_new)
         end do
         lnP_bot = s% lnP(ktop_new)
         
         if (dbg) write(*,2) 'ktop_new', ktop_new, top_new/Msun
         
         max_delta = maxval(abs(xa_below - xa_above))
         if (dbg) write(*,1) 'max_delta', max_delta, s% max_delta_limit_for_smooth
         if (max_delta > s% max_delta_limit_for_smooth) then
            if (dbg) write(*,3) 'smooth_top_moved_down: max_delta too large', &
               ktop_old, ktop_new, max_delta, lnP_top/ln10, lnP_bot/ln10
            return
         end if
         
         if (dbg) write(*,3) 'smooth_top_moved_down: k_above+1, ktop_new-1', &
            k_above+1, ktop_new-1, s% m(k_above+1)/Msun, s% m(ktop_new-1)/Msun, &
            s% xa(4,k_above+1), s% xa(4,ktop_new-1)
         dlnP = lnP_bot - lnP_top
         do k = k_above+1, ktop_new-1
            if (dlnP > 1d-50) then
               alfa = min(1d0, max(0d0, (s% lnP(k) - lnP_top)/dlnP)) ! alfa = 0 at top, =1 at bot
            else
               alfa = 0.5d0
            end if
            if (is_bad_num(alfa)) then
               write(*,2) 'alfa', k, alfa
               write(*,2) 'lnP_top', k, lnP_top
               write(*,2) 's% lnP(k)', k, s% lnP(k)
               write(*,2) 'lnP_bot', k, lnP_bot
               stop 'smooth_top_moved_down'
            end if
            do j=1,species
               s% xa(j,k) = alfa*xa_below(j) + (1-alfa)*xa_above(j)
               if (is_bad_num(s% xa(j,k))) then
                  write(*,3) 's% xa(j,k)', j, k, s% xa(j,k), xa_below(j), xa_above(j), alfa
                  stop 'smooth_top_moved_down'
               end if
            end do
         end do            
         if (dbg) write(*,1) 'smooth_top_moved_down: final Y(1200)', s% xa(4,1200)
         
      end subroutine smooth_top_moved_down


      subroutine smooth_bottom_moved_up(s, nz, species, bot_old, bot_new)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, species
         real(dp), intent(in) :: bot_old, bot_new 
         ! mass coords of newly nonconvective region above upward moving bottom boundary
         
         integer :: kbot_old, kbot_new, k, k_below, j
         real(dp) :: dlnP, alfa, xa_above(species), xa_below(species), &
            lnP_top, lnP_bot, max_delta, dR, scale_height
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         kbot_new = find_cell_for_mass(s,bot_new)
         kbot_old = find_cell_for_mass(s,bot_old)
         k_below = max(kbot_old-1,1)

         do k = kbot_new+1, k_below-1
            if (s% mixing_type(k) /= no_mixing) then
               if (dbg) write(*,2) 'smooth_bottom_moved_up mixing', k
               return
            end if
         end do
         dR = s% r(kbot_new) - s% r(k_below)
         scale_height = minval(s% scale_height(kbot_new:k_below))
         if (dR > scale_height*s% max_dR_div_Hp_for_smooth) then
            return
         end if

         do j=1,species
            xa_below(j) = s% xa(j,k_below)
         end do
         lnP_bot = s% lnP(k_below)

         do j=1,species
            xa_above(j) = s% xa(j,kbot_new)
         end do
         lnP_top = s% lnP(kbot_new+1)
         
         max_delta = maxval(abs(xa_below - xa_above))
         if (max_delta > s% max_delta_limit_for_smooth) then
            return
         end if
         
         if (dbg) write(*,3) 'smooth_bottom_moved_up: kbot_new+1, k_below-1', &
            kbot_new+1, k_below-1, s% m(kbot_new+1)/Msun, s% m(k_below-1)/Msun, &
            s% xa(4,kbot_new+1), s% xa(4,k_below-1)

         dlnP = lnP_bot - lnP_top
         do k = kbot_new+1, k_below-1
            if (dlnP > 1d-50) then
               alfa = max(0d0, min(1d0, (s% lnP(k) - lnP_top)/dlnP)) 
                  ! alfa = 0 at top, = 1 at bot
            else
               alfa = 0.5d0
            end if
            if (is_bad_num(alfa)) then
               write(*,2) 'alfa', k, alfa
               write(*,2) 'lnP_bot', k_below, lnP_bot
               write(*,2) 's% lnP(k)', k, s% lnP(k)
               write(*,2) 'lnP_top', kbot_new+1, lnP_top
               write(*,2) 'dlnP', k, dlnP
               write(*,2) 's% lnP(k) - lnP_top', k, s% lnP(k) - lnP_top
               stop 'smooth_bottom_moved_up'
            end if
            do j=1,species
               s% xa(j,k) = alfa*xa_below(j) + (1-alfa)*xa_above(j)
               if (is_bad_num(s% xa(j,k))) then
                  write(*,3) 's% xa(j,k)', j, k, s% xa(j,k), xa_below(j), xa_above(j), alfa
                  stop 'smooth_bottom_moved_up'
               end if
            end do
         end do

         if (dbg) write(*,1) 'smooth_bottom_moved_up: final Y(1200)', s% xa(4,1200)
         
      end subroutine smooth_bottom_moved_up
      
      
      subroutine smooth_for_departed_region(s, nz, species, top, bot)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, species
         real(dp), intent(in) :: top, bot
      
         integer :: ktop, k_above, kbot, k_below, k, j
         real(dp) :: mid, dm, alfa, xa_above(species), xa_below(species), m_top, m_bot, max_delta
         
         include 'formats'
         
         ktop = find_cell_for_mass(s,top)
         k_above = min(nz,ktop+1)
         do j=1,species
            xa_above(j) = s% xa(j,k_above)
         end do
         m_top = s% m(k_above) - 0.5d0*s% dm(k_above)
         
         kbot = find_cell_for_mass(s,bot)
         k_below = max(kbot-1,1)
         do j=1,species
            xa_below(j) = s% xa(j,k_below)
         end do
         m_bot = s% m(k_below) - 0.5d0*s% dm(k_below)
         
         max_delta = maxval(abs(xa_below - xa_above))
         if (max_delta > s% max_delta_limit_for_smooth) then
            !write(*,3) 'smooth_for_departed_region: max_delta', &
            !   ktop, kbot, max_delta, m_top/Msun, m_bot/Msun
            return
         end if

         dm = m_top - m_bot
         do k = k_above+1, k_below-1
            if (dm > 0) then
               mid = 0.5d0*(s% m(k) + s% m(k+1))
               alfa = (m_top - mid)/dm
            else
               alfa = 0.5d0
            end if
            if (is_bad_num(alfa) .or. alfa > 1 .or. alfa < 0) then
               write(*,2) 'alfa', k, alfa
               stop 'smooth_for_departed_region'
            end if
            do j=1,species
               s% xa(j,k) = alfa*xa_below(j) + (1-alfa)*xa_above(j)
               if (is_bad_num(s% xa(j,k))) then
                  write(*,3) 's% xa(j,k)', j, k, s% xa(j,k), xa_below(j), xa_above(j), alfa
                  stop 'smooth_for_departed_region'
               end if
            end do
         end do

      end subroutine smooth_for_departed_region


      subroutine do_all_regions( &
            s, top_moved_down, bottom_moved_up, departed_region, &
            num_old, cz_top_mass_old, cz_bot_mass_old, &
            num, cz_top_mass, cz_bot_mass, ierr)
         type (star_info), pointer :: s
         interface
            subroutine top_moved_down(s, nz, species, top, bot)
               use star_def, only: star_info
               use const_def, only: dp
               type (star_info), pointer :: s
               integer, intent(in) :: nz, species
               real(dp), intent(in) :: top, bot
            end subroutine top_moved_down
            subroutine bottom_moved_up(s, nz, species, top, bot)
               use star_def, only: star_info
               use const_def, only: dp
               type (star_info), pointer :: s
               integer, intent(in) :: nz, species
               real(dp), intent(in) :: top, bot
            end subroutine bottom_moved_up
            subroutine departed_region(s, nz, species, top, bot)
               use star_def, only: star_info
               use const_def, only: dp
               type (star_info), pointer :: s
               integer, intent(in) :: nz, species
               real(dp), intent(in) :: top, bot
            end subroutine departed_region
         end interface
         integer, intent(in) :: num, num_old
         real(dp), dimension(:) :: cz_top_mass_old, cz_bot_mass_old, &
            cz_top_mass, cz_bot_mass
         integer, intent(out) :: ierr
            
         real(dp), dimension(max_num_mixing_regions) :: region_dm, region_dm_old
         integer :: nz, species, i, j, k, j_top, j_bot
         real(dp) :: top, bot, top_old, bot_old
         logical, parameter :: dbg = .false.

         include 'formats'
         
         ierr = 0
         nz = s% nz
         species = s% species
         
         if (dbg) write(*,*)
         if (dbg) write(*,*) 'do_all_regions'
      
         do i=1,num
            region_dm(i) = cz_top_mass(i) - cz_bot_mass(i)
            if (dbg) write(*,2) 'cz_top_mass cz_bot_mass', i, &
               cz_top_mass(i)/Msun, cz_bot_mass(i)/Msun
         end do
         
         do i=1,num_old
            region_dm_old(i) = cz_top_mass_old(i) - cz_bot_mass_old(i)
            if (dbg) write(*,2) 'cz_top_mass_old cz_bot_mass_old', i, &
               cz_top_mass_old(i)/Msun, cz_bot_mass_old(i)/Msun
         end do
      
         do i=1,num
            j = maxloc(region_dm(1:num),dim=1)
            region_dm(j) = -1 ! mark as done
            top = cz_top_mass(j)
            bot = cz_bot_mass(j)
            j_top = minloc(abs(top - cz_top_mass_old(1:num_old)),dim=1)
            top_old = cz_top_mass_old(j_top)
            j_bot = minloc(abs(bot - cz_bot_mass_old(1:num_old)),dim=1)
            bot_old = cz_bot_mass_old(j_bot)
            if (min(top,top_old) > max(bot,bot_old)) then ! overlap
               if (top < top_old) then
                  if (dbg) write(*,2) 'top_moved_down: top_old, top', i, &
                     top_old/Msun, top/Msun
                  call top_moved_down(s, nz, species, top_old, top) ! top moved down
               end if
               if (bot > bot_old) then
                  if (dbg) write(*,2) 'bottom_moved_up: bot_old, bot', i, &
                     bot_old/Msun, bot/Msun
                  call bottom_moved_up(s, nz, species, bot_old, bot) ! bottom moved up
               end if
            end if
            do j=1,num_old
               if (cz_top_mass_old(j) <= top_old .and. cz_bot_mass_old(j) >= bot_old) &
                  region_dm_old(j) = -1 ! mark as used
            end do
         end do
      
         do j=1,num_old
            if (region_dm_old(j) < 0) cycle ! was used
            ! wasn't used, so region is no longer convective
            call departed_region(s, nz, species, cz_top_mass_old(j), cz_bot_mass_old(j))
            if (dbg) write(*,1) 'departed_region top bot', &
               cz_top_mass_old(j)/Msun, cz_bot_mass_old(j)/Msun
         end do
         
      end subroutine do_all_regions





      end module mix_smoothing
      
      