! ***********************************************************************
!
!   Copyright (C) 2011  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module mix_info

      use const_def
      use num_lib
      use utils_lib
      use star_private_def
      use mlt_def
      use star_utils, only: zero_D_mix_partials


      implicit none


      contains


      subroutine set_mixing_info(s, ierr)
         ! set convection variables cdc and conv_vel starting from local MLT results.
         ! overshooting can also be added.
         use rates_def, only: i_rate
         use chem_def, only: ipp, icno, i3alf
         use overshoot, only: add_overshooting
         use turbulent_diffusion, only: &
            add_turbulent_diffusion, add_radiation_turbulence
         use mix_smoothing, only: set_newly_non_conv
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: nz, i, k, max_conv_bdy, max_mix_bdy, k_dbg
         real(dp) :: c, rho_face, f
         real(dp), pointer, dimension(:) :: eps_h, eps_he, eps_z, cdc_factor
         
         logical :: dbg

         include 'formats'
         
         ierr = 0
         dbg = .false.
         k_dbg = -1152

         nullify(eps_h, eps_he, eps_z, cdc_factor)

         if (dbg) write(*, *) 'set_mixing_info'

         ierr = 0
         max_conv_bdy = 10 ! will automatically be increased if necessary
         max_mix_bdy = 10 ! will automatically be increased if necessary
         nz = s% nz
         
         s% num_conv_boundaries = 0
         if (.not. associated(s% conv_bdy_loc)) allocate(s% conv_bdy_loc(max_conv_bdy))
         if (.not. associated(s% conv_bdy_q)) allocate(s% conv_bdy_q(max_conv_bdy))
         if (.not. associated(s% top_conv_bdy)) allocate(s% top_conv_bdy(max_conv_bdy))
         if (.not. associated(s% burn_h_conv_region)) allocate(s% burn_h_conv_region(max_conv_bdy))
         if (.not. associated(s% burn_he_conv_region)) allocate(s% burn_he_conv_region(max_conv_bdy))
         if (.not. associated(s% burn_z_conv_region)) allocate(s% burn_z_conv_region(max_conv_bdy))
         
         s% num_mix_boundaries = 0
         if (.not. associated(s% mix_bdy_loc)) allocate(s% mix_bdy_loc(max_mix_bdy))
         if (.not. associated(s% mix_bdy_q)) allocate(s% mix_bdy_q(max_mix_bdy))
         if (.not. associated(s% top_mix_bdy)) allocate(s% top_mix_bdy(max_mix_bdy))
         if (.not. associated(s% burn_h_mix_region)) allocate(s% burn_h_mix_region(max_mix_bdy))
         if (.not. associated(s% burn_he_mix_region)) allocate(s% burn_he_mix_region(max_mix_bdy))
         if (.not. associated(s% burn_z_mix_region)) allocate(s% burn_z_mix_region(max_mix_bdy))
         
         if (s% redo_mlt_for_OPEC) then
            if (dbg) write(*, *) 'redo_mlt_for_OPEC'
            call redo_mlt_for_OPEC(s, ierr)
            if (failed('redo_mlt_for_OPEC')) return
         end if
         
         if (dbg) write(*,3) 'call check_for_redo_MLT', &
            k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call check_for_redo_MLT(s, nz, ierr)
         if (failed('check_for_redo_MLT')) return
         
         do k=1,nz
            call copy_mlt_results(s,k)
         end do
         
         if (s% remove_mixing_glitches) then
            
            if (dbg) write(*, *) 'remove_mixing_glitches'
            
            if (dbg) write(*,3) 'call remove_tiny_mixing', &
               k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
            call remove_tiny_mixing(s, ierr)
            if (failed('remove_tiny_mixing')) return
            
            if (dbg) write(*,3) 'call remove_mixing_singletons', &
               k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
            call remove_mixing_singletons(s, ierr)
            if (failed('remove_mixing_singletons')) return
            
            if (dbg) write(*,3) 'call close_convection_gaps', &
               k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
            call close_convection_gaps(s, ierr)
            if (failed('close_convection_gaps')) return
            
            if (dbg) write(*,3) 'call close_thermohaline_gaps', &
               k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
            call close_thermohaline_gaps(s, ierr)
            if (failed('close_thermohaline_gaps')) return
            
            if (dbg) write(*,3) 'call remove_thermohaline_dropouts', &
               k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
            call remove_thermohaline_dropouts(s, ierr)
            if (failed('remove_thermohaline_dropouts')) return
            
            if (dbg) write(*,3) 'call close_semiconvection_gaps', &
               k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
            call close_semiconvection_gaps(s, ierr)
            if (failed('close_semiconvection_gaps')) return
            
            if (dbg) write(*,3) 'call remove_embedded_semiconvection', &
               k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
            call remove_embedded_semiconvection(s, ierr)
              if (failed('remove_embedded_semiconvection')) return
            
         end if

         if (dbg) write(*,3) 'call do_mix_envelope'
         call do_mix_envelope(s)

         call do_alloc(ierr)
         if (ierr /= 0) return
         
         do k=1,nz
            eps_h(k) = s% eps_nuc_categories(ipp,k) + &
                       s% eps_nuc_categories(icno,k)
            eps_he(k) = s% eps_nuc_categories(i3alf,k)
            eps_z(k) = s% eps_nuc(k) - (eps_h(k) + eps_he(k))
         end do
         
         if (dbg) write(*,3) 'call set_mlt_cz_boundary_info', &
            k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call set_mlt_cz_boundary_info(s, ierr)
         if (failed('set_mlt_cz_boundary_info')) return
         
         if (dbg) write(*,3) 'call locate_convection_boundaries', &
            k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call locate_convection_boundaries( &
            s, nz, eps_h, eps_he, eps_z, s% mstar, s% q, s% cdc, s% grada_at_face, s% gradr, ierr)
         if (failed('locate_convection_boundaries')) return
        
         if (dbg) write(*,3) 'call add_overshooting', &
            k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call add_overshooting(s, ierr)
         if (failed('add_overshooting')) return
        
         if (dbg) write(*,3) 'call add_turbulent_diffusion', &
            k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call add_turbulent_diffusion(s, ierr)
         if (failed('add_turbulent_diffusion')) return
        
         if (dbg) write(*,3) 'call add_radiation_turbulence', &
            k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call add_radiation_turbulence(s, ierr)
         if (failed('add_radiation_turbulence')) return
                 
         if (dbg) write(*,3) 'call s% other_after_set_mixing_info', &
            k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call s% other_after_set_mixing_info(s% id, ierr)
         if (failed('other_after_set_mixing_info')) return         

         if (dbg) write(*,3) 'call set_cz_bdy_mass', &
            k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call set_cz_bdy_mass(s, ierr)
         if (failed('set_cz_bdy_mass')) return
         
         cdc_factor(1) = 1
         do k = 2, nz
            rho_face = (s% dq(k-1)*s% rho(k) + s% dq(k)*s% rho(k-1))/(s% dq(k-1) + s% dq(k))
            f = 4*pi*s% r(k)*s% r(k)*rho_face
            cdc_factor(k) = f*f
         end do
         
         if (s% set_min_D_mix .and. s% ye(nz) >= s% min_center_Ye_for_min_D_mix) then
            do k=1,nz
               if (s% D_mix(k) >= s% min_D_mix) cycle
               s% D_mix(k) = s% min_D_mix
               call zero_D_mix_partials(s,k)
               s% mixing_type(k) = minimum_mixing
            end do
         end if

         if (s% use_other_D_mix) then
            if (dbg) write(*,*) 'call other_D_mix'
            call s% other_D_mix(s% id, ierr)
            if (failed('other_D_mix')) return
         end if
         
         if (dbg) write(*,3) 'call set_newly_non_conv', k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)
         call set_newly_non_conv(s, ierr)
         if (failed('set_newly_non_conv')) return        

         do k=1,nz
            s% D_mix_non_rotation(k) = s% D_mix(k)
         end do
         
         if (s% rotation_flag) then

            if (s% trace_k > 0 .and. s% trace_k <= s% nz) then
               do k=1,nz
                  write(*,3) 'before update_rotation_mixing_info D_mix', &
                     s% model_number, k, s% D_mix(k)
               end do
            end if
         
            call update_rotation_mixing_info(s,ierr)
            if (failed('update_rotation_mixing_info')) return
                     
            do k = 2, nz
               s% cdc(k) = s% D_mix(k)*cdc_factor(k)               
               if (s% D_mix(k) /= 0 .and. s% mixing_type(k) == no_mixing) then
                  s% mixing_type(k) = rotation_mixing
               end if
            end do
            s% cdc(1) = s% cdc(2)

            if (s% trace_k > 0 .and. s% trace_k <= s% nz) then
               do k=1,nz
                  write(*,3) 'after do rotation mixing D_mix', &
                     s% model_number, k, s% D_mix(k)
               end do
            end if

         end if
         
         if (dbg) write(*,*) 'call locate_mixing_boundaries'
         call locate_mixing_boundaries(s, eps_h, eps_he, eps_z, ierr)
         if (failed('locate_mixing_boundaries')) return
         
         do k=2,nz ! as last thing, update conv_vel from D_mix and mixing length.
            if (s% mlt_mixing_length(k) > 0) then
               s% conv_vel(k) = 3*s% D_mix(k)/s% mlt_mixing_length(k)
            else
               s% conv_vel(k) = 0
            end if
         end do
         
         ! set these just for plotting.  not used.
         s% mixing_type(1) = s% mixing_type(2)
         s% D_mix(1) = s% D_mix(2)
         s% d_D_mix_dlnd00(1) = s% d_D_mix_dlnd00(2)
         s% d_D_mix_dlnT00(1) = s% d_D_mix_dlnT00(2)
         s% d_D_mix_dlndm1(1) = s% d_D_mix_dlndm1(2)
         s% d_D_mix_dlnTm1(1) = s% d_D_mix_dlnTm1(2)
         s% d_D_mix_dlnR(1) = s% d_D_mix_dlnR(2)
         s% d_D_mix_dL(1) = s% d_D_mix_dL(2)
         s% conv_vel(1) = s% conv_vel(2)

         call check

         if (dbg) write(*,3) 'done mixing', k_dbg, s% mixing_type(k_dbg), s% D_mix(k_dbg)

                  
         call dealloc
         
         s% have_mixing_info = .true.
         
         !if (dbg) stop 'mixing'

         contains
         
         logical function failed(str)
            character (len=*), intent(in) :: str
            if (ierr == 0) then
               failed = .false.
               return
            end if
            if (s% report_ierr .or. dbg) &
               write(*,*) 'set_mixing_info failed in call to ' // trim(str)
            failed = .true.
            call dealloc
         end function failed
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            call non_crit_get_work_array(s, eps_h, nz, nz_alloc_extra, 'mix_info', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, eps_he, nz, nz_alloc_extra, 'mix_info', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, eps_z, nz, nz_alloc_extra, 'mix_info', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, cdc_factor, nz, nz_alloc_extra, 'mix_info', ierr)
            if (ierr /= 0) return            
         end subroutine do_alloc
         
         subroutine dealloc
            use alloc
            use utils_lib
            call non_crit_return_work_array(s, eps_h, 'mix_info')            
            call non_crit_return_work_array(s, eps_he, 'mix_info')            
            call non_crit_return_work_array(s, eps_z, 'mix_info')            
            call non_crit_return_work_array(s, cdc_factor, 'mix_info')            
         end subroutine dealloc
         
         subroutine check
            integer :: k
            include 'formats'
            do k = 1, s% nz
               if (s% conv_vel(k) < 0d0 .or. &
                     (s% mixing_type(k) == 0 .and. s% conv_vel(k) > 0d0)) then
                  write(*,3) 'mixing_type conv_vel', k, s% mixing_type(k), s% conv_vel(k)
                  stop 'set mixing info'
               end if
               if (is_bad_num(s% D_mix(k))) then
                  write(*,3) 'D_mix mixing_type', k, s% mixing_type(k), s% D_mix(k)
                  if (s% rotation_flag) then
                     if (is_bad_num(s% D_mix_non_rotation(k))) &
                        write(*,2) 's% D_mix_non_rotation(k)', k, s% D_mix_non_rotation(k)
                     if (is_bad_num(s% D_visc(k))) write(*,2) 's% D_visc(k)', k, s% D_visc(k)
                     if (is_bad_num(s% D_DSI(k))) write(*,2) 's% D_DSI(k)', k, s% D_DSI(k)
                     if (is_bad_num(s% D_SH(k))) write(*,2) 's% D_SH(k)', k, s% D_SH(k)
                     if (is_bad_num(s% D_SSI(k))) write(*,2) 's% D_SSI(k)', k, s% D_SSI(k)
                     if (is_bad_num(s% D_ES(k))) write(*,2) 's% D_ES(k)', k, s% D_ES(k)
                     if (is_bad_num(s% D_GSF(k))) write(*,2) 's% D_GSF(k)', k, s% D_GSF(k)
                     if (is_bad_num(s% D_ST(k))) write(*,2) 's% D_ST(k)', k, s% D_ST(k)
                  end if
                  if (is_bad_num(s% D_mix(k))) stop 'mix info'
               end if
            end do
         end subroutine check
      
      end subroutine set_mixing_info

      
      subroutine check_for_redo_MLT(s, nz, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: nz
         integer, intent(out) :: ierr
         
         logical :: in_convective_region
         integer :: k, k_bot
         real(dp) :: bot_Hp, bot_r, top_Hp, top_r, dr
         logical :: dbg
         
         include 'formats'

         ierr = 0
         dbg = .false.

         bot_Hp = 0; bot_r = 0; top_Hp = 0; top_r = 0; dr = 0
         
         in_convective_region = (s% mixing_type(nz) == convective_mixing)
         k_bot = nz
         
         do k=nz-1, 2, -1
            if (in_convective_region) then
               if (s% mixing_type(k) /= convective_mixing) then
                  call end_of_convective_region
               end if
            else ! in non-convective region
               if (s% mixing_type(k) == convective_mixing) then ! start of a convective region
                  k_bot = k+1
                  in_convective_region = .true.
                  bot_r = s% r(k_bot)
                  bot_Hp = s% scale_height(k_bot)
               end if
            end if
         end do
         
         if (in_convective_region) then
            k = 1 ! end at top
            call end_of_convective_region
         end if
         
         
         contains         
         
         
         subroutine end_of_convective_region()
            integer :: kk, op_err, mix_type
            real(dp) :: Hp
            logical :: end_dbg
            
            9 format(a40, 3i7, 99(1pd26.16))
            include 'formats'
            
            in_convective_region = .false.
            
            end_dbg = .false.
            
            top_r = s% r(k)
            top_Hp = s% scale_height(k)
            dr = top_r - bot_r
            Hp = (bot_Hp + top_Hp)/2
            
            if (dr < s% mixing_length_alpha*min(top_Hp, bot_Hp) .and. &
                  s% redo_conv_for_dr_lt_mixing_length) then
!$OMP PARALLEL DO PRIVATE(kk,op_err)
               do kk = k, k_bot
                  op_err = 0
                  call redo1_mlt(s,kk,dr,op_err)
                  if (op_err /= 0) ierr = op_err
               end do
!$OMP END PARALLEL DO
            else if (s% limit_mixing_length_by_dist_to_bdy > 0) then
!$OMP PARALLEL DO PRIVATE(kk,op_err)
               do kk = k, k_bot
                  op_err = 0
                  call redo1_mlt(s, kk, s% limit_mixing_length_by_dist_to_bdy* &
                     min(top_r - s% r(kk), s% r(kk) - bot_r), op_err)
                  if (op_err /= 0) ierr = op_err
               end do
!$OMP END PARALLEL DO
            end if
            
            
         end subroutine end_of_convective_region
         
         
         subroutine redo1_mlt(s,k,dr,ierr)
            use mlt_info, only: do1_mlt
            type (star_info), pointer :: s         
            integer, intent(in) :: k
            real(dp), intent(in) :: dr
            integer, intent(out) :: ierr
            real(dp) :: Hp, reduced_alpha
            include 'formats'
            ierr = 0
            if (dr >= s% mlt_mixing_length(k)) return
            ! if convection zone is smaller than mixing length
            ! redo MLT with reduced alpha so mixing_length = dr
            Hp = s% scale_height(k)
            reduced_alpha = dr/Hp
            call do1_mlt(s, k, reduced_alpha, -1d0, &
               -1d0, -1d0, -1d0, -1d0, -1d0, -1d0, -1d0, -1d0, -1d0, &
               ierr)
            call copy_mlt_results(s,k)
         end subroutine redo1_mlt
         
                
      end subroutine check_for_redo_MLT
         
      
      
      subroutine copy_mlt_results(s,k)
         type (star_info), pointer :: s         
         integer, intent(in) :: k
         s% cdc(k) = s% mlt_cdc(k)
         s% conv_vel(k) = s% mlt_vc(k)
         s% mixing_type(k) = s% mlt_mixing_type(k)
         s% D_mix(k) = s% mlt_D(k)
         s% d_D_mix_dlnd00(k) = s% d_mlt_D_dlnd00(k)
         s% d_D_mix_dlnT00(k) = s% d_mlt_D_dlnT00(k)
         s% d_D_mix_dlndm1(k) = s% d_mlt_D_dlndm1(k)
         s% d_D_mix_dlnTm1(k) = s% d_mlt_D_dlnTm1(k)
         s% d_D_mix_dlnR(k) = s% d_mlt_D_dlnR(k)
         s% d_D_mix_dL(k) = s% d_mlt_D_dL(k)
      end subroutine copy_mlt_results
      
      
      subroutine set_mlt_cz_boundary_info(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, mt, mt1, mt2, nz
         real(dp) :: dg0, dg1
         
         include 'formats'

         ! NOTE: this routine must be called BEFORE overshooting is done.
         
         ! convective zone boundary is where gradL = gradr
         ! semi-convective zone boundary is where grada_at_face = gradr
         
         ierr = 0
         nz = s% nz
         s% cz_bdy_dq(1:nz) = 0d0
         
         do k = 2, nz
            mt1 = s% mixing_type(k-1)
            mt2 = s% mixing_type(k)
            if (mt1 == mt2) cycle
            if (mt2 == convective_mixing .or. mt1 == convective_mixing) then
               dg0 = s% gradL(k-1) - s% gradr(k-1)
               dg1 = s% gradL(k) - s% gradr(k)
            else if (mt2 == semiconvective_mixing .or. mt1 == semiconvective_mixing) then
               dg0 = s% grada_at_face(k-1) - s% gradr(k-1)
               dg1 = s% grada_at_face(k) - s% gradr(k)
            else
               cycle
            end if
            if (dg0*dg1 >= 0) cycle
            s% cz_bdy_dq(k-1) = find0(0d0,dg0,s% dq(k-1),dg1)
            if (s% cz_bdy_dq(k-1) < 0d0 .or. s% cz_bdy_dq(k-1) > s% dq(k-1)) then
               write(*,2) 'bad cz_bdy_dq', k-1, s% cz_bdy_dq(k-1), s% dq(k-1)
               stop 'set_mlt_cz_boundary_info'
               ierr = -1
               return
            end if
         end do

      end subroutine set_mlt_cz_boundary_info
      
      
      subroutine set_cz_bdy_mass(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr

         logical :: in_convective_region
         integer :: k, j, nz
         logical, parameter :: dbg = .false.

         include 'formats'
         ierr = 0
         nz = s% nz

         s% n_conv_regions = 0
         in_convective_region = (s% mixing_type(nz) == convective_mixing)
         if (in_convective_region) then
            s% n_conv_regions = 1
            s% cz_bot_mass(1) = s% M_center
         end if
         
         if (dbg) write(*,*) 'initial in_convective_region', in_convective_region
         
         do k=nz-1, 2, -1
            if (in_convective_region) then
               if (s% mixing_type(k) /= convective_mixing) then ! top of convective region
                  s% cz_top_mass(s% n_conv_regions) = &
                     s% M_center + (s% q(k) - s% cz_bdy_dq(k))*s% xmstar
                  in_convective_region = .false.
               end if
            else
               if (s% mixing_type(k) == convective_mixing) then ! bottom of convective region
                  if (s% n_conv_regions < max_num_mixing_regions) then
                     s% n_conv_regions = s% n_conv_regions + 1
                     s% cz_bot_mass(s% n_conv_regions) = &
                        s% M_center + (s% q(k) - s% cz_bdy_dq(k))*s% xmstar
                  end if
                  in_convective_region = .true.
               end if
            end if
         end do
         if (in_convective_region) then
            s% cz_top_mass(s% n_conv_regions) = s% mstar
         end if

         if (dbg) then
            write(*,*)
            write(*,2) 'set_mixing_info s% n_conv_regions', s% n_conv_regions
            do j = 1, s% n_conv_regions
               write(*,2) 'conv region', j, s% cz_bot_mass(j)/Msun, s% cz_top_mass(j)/Msun
            end do
            write(*,*)
         end if
         
      end subroutine set_cz_bdy_mass

      
      subroutine locate_convection_boundaries( &
            s, nz, eps_h, eps_he, eps_z, mstar, q, cdc, grada, gradr, ierr)
         !use star_utils, only: scale_height
         type (star_info), pointer :: s
         integer, intent(in) :: nz
         real(dp), dimension(:), intent(in) :: eps_h, eps_he, eps_z
         real(dp), intent(in) :: mstar
         real(dp), pointer, dimension(:) :: q, cdc, grada, gradr
         integer, intent(out) :: ierr
         
         logical :: in_convective_region
         integer :: k, k_bot, i, j, iounit, max_conv_bdy
         real(dp) :: dgrad00, dgradp1, turnover_time, &
            bot_Hp, bot_r, top_Hp, top_r, dr
         
         logical :: dbg
         logical, parameter :: write_debug = .false.
         
         include 'formats'

         ierr = 0
         dbg = .false.
         
         if (write_debug) then
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) then
               write(*, *) 'alloc_iounit failed'
               return
            end if
            open(iounit, file=trim('debug.data'), action='write', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'open debug.data failed'
               call free_iounit(iounit)
               return
            end if
            write(*,*) 'write debug.data'
            write(iounit,*) 'nz', nz
            write(iounit,1) 'mstar', mstar
            do k=1,nz
               write(iounit,2) 'q', k, q(k)
               write(iounit,2) 'cdc', k, cdc(k)
               write(iounit,2) 'eps_h', k, eps_h(k)
               write(iounit,2) 'eps_he', k, eps_he(k)
               write(iounit,2) 'eps_z', k, eps_z(k)
            end do
         end if

         max_conv_bdy = size(s% conv_bdy_q, dim=1)
         s% conv_bdy_q(:) = 0
         s% conv_bdy_loc(:) = 0
         s% top_conv_bdy(:) = .false.
         s% burn_h_conv_region(:) = .false.
         s% burn_he_conv_region(:) = .false.
         s% burn_z_conv_region(:) = .false.
         bot_Hp = 0; bot_r = 0; top_Hp = 0; top_r = 0; dr = 0
         
         s% num_conv_boundaries = 0
         in_convective_region = (s% mixing_type(nz) == convective_mixing)
         k_bot = nz
         turnover_time = 0
         
         do k=nz-1, 2, -1
            if (in_convective_region) then
               if (s% mixing_type(k) /= convective_mixing) then
                  call end_of_convective_region
               else
                  turnover_time = turnover_time + (s% rmid(k-1) - s% rmid(k))/s% conv_vel(k)
               end if
            else ! in non-convective region
               if (s% mixing_type(k) == convective_mixing) then ! start of a convective region
                  if (s% num_conv_boundaries == max_conv_bdy) then
                     call realloc(ierr)
                     if (ierr /= 0) then
                        if (write_debug) call free_iounit(iounit)
                        return
                     end if
                  end if
                  s% num_conv_boundaries = s% num_conv_boundaries+1
                  i = s% num_conv_boundaries
                  k_bot = k+1
                  if (k == 1) then
                     s% conv_bdy_q(i) = 1
                  else ! bottom of region is between k+1 and k
                     s% conv_bdy_q(i) = s% q(k) - s% cz_bdy_dq(k)
                  end if
                  s% top_conv_bdy(i) = .false.
                  s% conv_bdy_loc(i) = k_bot 
                     ! bottom of region is between k_bot and k_bot-1
                  in_convective_region = .true.
                  bot_r = s% r(k_bot)
                  !bot_Hp = scale_height(s,k_bot,.true.)
                  !write(*,2) 'bot_Hp', k_bot, bot_Hp, s% scale_height(k_bot)
                  bot_Hp = s% scale_height(k_bot)
                  !stop 'mix_info'
                  turnover_time = 0
               end if
            end if
         end do

         if (in_convective_region) then
            k = 1 ! end at top
            call end_of_convective_region
         end if

         if (write_debug) then
            write(iounit,*) 's% num_conv_boundaries', s% num_conv_boundaries
            do j=1,s% num_conv_boundaries
               write(iounit,2) 's% conv_bdy_q', j, s% conv_bdy_q(j)
               write(iounit,*) 's% top_conv_bdy', j, s% top_conv_bdy(j)
               write(iounit,*) 's% burn_h_conv_region', j, s% burn_h_conv_region(j)
               write(iounit,*) 's% burn_he_conv_region', j, s% burn_he_conv_region(j)
               write(iounit,*) 's% burn_z_conv_region', j, s% burn_z_conv_region(j)
               write(iounit,*) 's% conv_bdy_loc', j, s% conv_bdy_loc(j)
            end do
            close(iounit)
            call free_iounit(iounit)
         end if
         
         if (dbg) then
            write(*,*) 's% num_conv_boundaries', s% num_conv_boundaries
            do j=1,s% num_conv_boundaries
               write(*,*) 's% conv_bdy_q', j, s% conv_bdy_q(j)
               write(*,*) 's% top_conv_bdy', j, s% top_conv_bdy(j)
               write(*,*) 's% burn_h_conv_region', j, s% burn_h_conv_region(j)
               write(*,*) 's% burn_he_conv_region', j, s% burn_he_conv_region(j)
               write(*,*) 's% burn_z_conv_region', j, s% burn_z_conv_region(j)
               write(*,*) 's% conv_bdy_loc', j, s% conv_bdy_loc(j)
               write(*,*) 'mixing type', s% mixing_type(s% conv_bdy_loc(j)-3:s% conv_bdy_loc(j)+3)
            end do
            write(*,*)
            write(*,3) 'mixing_type', 1152, s% mixing_type(1152)
            stop 'locate_convection_boundaries'
         end if

         
         contains         
         
         
         subroutine end_of_convective_region()
            integer :: max_eps_loc, kk, op_err, mix_type
            real(dp) :: max_eps, eps, Hp
            logical :: end_dbg
            
            9 format(a40, 3i7, 99(1pd26.16))
            include 'formats'
            
            in_convective_region = .false.
            
            end_dbg = .false.
            
            top_r = s% r(k)
            top_Hp = s% scale_height(k)
            dr = top_r - bot_r
            Hp = (bot_Hp + top_Hp)/2
            
            if (dr/Hp < s% prune_bad_cz_min_Hp_height .and. s% prune_bad_cz_min_Hp_height > 0) then
               max_eps = maxval(eps_h(k:k_bot) + eps_he(k:k_bot) + eps_z(k:k_bot))
               if (end_dbg) write(*,3) 'max_eps', k, k_bot, max_eps, &
                  exp10_cr(s% prune_bad_cz_min_log_eps_nuc)
               if (max_eps < exp10_cr(s% prune_bad_cz_min_log_eps_nuc) &
                     .and. all(s% mixing_type(k+1:k_bot-1) /= thermohaline_mixing)) then
                  do kk = k, k_bot ! this includes the radiative points at boundaries
                     call set_use_gradr(s,kk)
                     s% cdc(kk) = 0
                     s% D_mix(kk) = 0
                     call zero_D_mix_partials(s,kk)
                     s% conv_vel(kk) = 0
                     s% mixing_type(kk) = no_mixing
                  end do
                  if (s% num_conv_boundaries > 0) &
                     s% num_conv_boundaries = s% num_conv_boundaries-1
                  return
               end if
            end if
            
            if (s% num_conv_boundaries == max_conv_bdy) then
               call realloc(ierr)
               if (ierr /= 0) return
            end if            
            
            s% num_conv_boundaries = s% num_conv_boundaries+1
            i = s% num_conv_boundaries

            ! check for burning in region
            max_eps = -1d99
            max_eps_loc = 0
            do kk=k,min(nz,k_bot+1)
               eps = eps_h(kk) + eps_he(kk) + eps_z(kk)
               if (eps > max_eps) then
                  max_eps = eps
                  max_eps_loc = kk
               end if
            end do
            if (max_eps > 1d0) then
               if (eps_h(max_eps_loc) > 0.5d0*max_eps) then
                  s% burn_h_conv_region(i) = .true.
                  if (i > 1) s% burn_h_conv_region(i-1) = .true.
               else if (eps_he(max_eps_loc) > 0.5d0*max_eps) then
                  s% burn_he_conv_region(i) = .true.
                  if (i > 1) s% burn_he_conv_region(i-1) = .true.
               else
                  s% burn_z_conv_region(i) = .true.
                  if (i > 1) s% burn_z_conv_region(i-1) = .true.
               end if
            end if
            
            if (k == 1) then
               s% conv_bdy_q(i) = 1
            else 
               ! top of region is between k+1 and k
               s% conv_bdy_q(i) = s% q(k) - s% cz_bdy_dq(k)
            end if
            s% top_conv_bdy(i) = .true.
            s% conv_bdy_loc(i) = k
            
         end subroutine end_of_convective_region
         
      
         subroutine realloc(ierr)
            use utils_lib
            integer, intent(out) :: ierr
            
            integer :: sz
            
            sz = size(s% conv_bdy_q, dim=1)
            
            ierr = 0
            max_conv_bdy = 2*(10+max_conv_bdy)
            
            call realloc_double(s% conv_bdy_q,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_integer(s% conv_bdy_loc,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% top_conv_bdy,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_h_conv_region,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_he_conv_region,max_conv_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_z_conv_region,max_conv_bdy,ierr)
            if (ierr /= 0) return

            s% conv_bdy_q(sz+1:max_conv_bdy) = 0
            s% conv_bdy_loc(sz+1:max_conv_bdy) = 0
            s% top_conv_bdy(sz+1:max_conv_bdy) = .false.
            s% burn_h_conv_region(sz+1:max_conv_bdy) = .false.
            s% burn_he_conv_region(sz+1:max_conv_bdy) = .false.
            s% burn_z_conv_region(sz+1:max_conv_bdy) = .false.
         
         end subroutine realloc
                
      end subroutine locate_convection_boundaries
      

      subroutine set_use_gradr(s,k)
         use mlt_info
         type (star_info), pointer :: s
         integer, intent(in) :: k
         call switch_to_no_mixing(s,k)
         call switch_to_radiative(s,k)
         !s% use_gradr_for_gradT(k) = .true.
      end subroutine set_use_gradr      
      
     
      subroutine locate_mixing_boundaries(s, eps_h, eps_he, eps_z, ierr)
         type (star_info), pointer :: s
         real(dp), dimension(:), intent(in) :: eps_h, eps_he, eps_z
         integer, intent(out) :: ierr
         
         logical :: in_mixing_region
         integer :: k, k_bot, i, j, iounit, max_mix_bdy, nz
         
         logical, parameter :: dbg = .false.
         
         include 'formats'

         ierr = 0
         nz = s% nz
         
         max_mix_bdy = size(s% mix_bdy_q, dim=1)
         s% mix_bdy_q(:) = 0
         s% mix_bdy_loc(:) = 0
         s% top_mix_bdy(:) = .false.
         s% burn_h_mix_region(:) = .false.
         s% burn_he_mix_region(:) = .false.
         s% burn_z_mix_region(:) = .false.
         
         s% num_mix_boundaries = 0
         s% num_mix_regions = 0
         in_mixing_region = (s% mixing_type(nz) /= no_mixing)
         k_bot = nz
         do k=nz-1, 2, -1
            if (in_mixing_region) then
               if (s% mixing_type(k) == no_mixing) call end_of_mixing_region
            else ! in non-mixing region
               if (s% mixing_type(k) /= no_mixing) then ! start of a mixing region
                  if (s% num_mix_boundaries == max_mix_bdy) then
                     call realloc(ierr)
                     if (ierr /= 0) return
                  end if
                  s% num_mix_boundaries = s% num_mix_boundaries+1
                  i = s% num_mix_boundaries
                  k_bot = k+1
                  if (k == 1) then
                     s% mix_bdy_q(i) = 1
                  else ! bottom of region is between k+1 and k
                     s% mix_bdy_q(i) = s% q(k) - s% cz_bdy_dq(k)
                  end if
                  s% top_mix_bdy(i) = .false.
                  s% mix_bdy_loc(i) = k_bot
                  in_mixing_region = .true.
               end if
            end if
         end do
         
         if (in_mixing_region) then
            k = 1 ! end at top
            call end_of_mixing_region
         end if

         
         contains         
         
         
         subroutine end_of_mixing_region()
            integer :: max_eps_loc, kk
            real(dp) :: max_eps, eps
            
            9 format(a40, 3i7, 99(1pd26.16))
            include 'formats'
            
            in_mixing_region = .false.
            
            if (s% num_mix_boundaries == max_mix_bdy) then
               call realloc(ierr)
               if (ierr /= 0) return
            end if            
            
            s% num_mix_regions = s% num_mix_regions+1
            s% num_mix_boundaries = s% num_mix_boundaries+1
            i = s% num_mix_boundaries

            ! check for burning in region
            max_eps = -1d99
            max_eps_loc = 0
            do kk=k,min(nz,k_bot+1)
               eps = eps_h(kk) + eps_he(kk) + eps_z(kk)
               if (eps > max_eps) then
                  max_eps = eps
                  max_eps_loc = kk
               end if
            end do
            if (max_eps > 1d0) then
               if (eps_h(max_eps_loc) > 0.5d0*max_eps) then
                  s% burn_h_mix_region(i) = .true.
                  if (i > 1) s% burn_h_mix_region(i-1) = .true.
               else if (eps_he(max_eps_loc) > 0.5d0*max_eps) then
                  s% burn_he_mix_region(i) = .true.
                  if (i > 1) s% burn_he_mix_region(i-1) = .true.
               else
                  s% burn_z_mix_region(i) = .true.
                  if (i > 1) s% burn_z_mix_region(i-1) = .true.
               end if
            end if
            
            if (k == 1) then
               s% mix_bdy_q(i) = 1
            else 
               ! top of region is between k+1 and k
               s% mix_bdy_q(i) = s% q(k) - s% cz_bdy_dq(k)
            end if
            s% top_mix_bdy(i) = .true.
            s% mix_bdy_loc(i) = k
            
         end subroutine end_of_mixing_region
         
      
         subroutine realloc(ierr)
            use utils_lib
            integer, intent(out) :: ierr
            
            integer :: sz
            
            sz = size(s% mix_bdy_q, dim=1)
            
            ierr = 0
            max_mix_bdy = 2*(10+max_mix_bdy)
            
            call realloc_double(s% mix_bdy_q,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_integer(s% mix_bdy_loc,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% top_mix_bdy,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_h_mix_region,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_he_mix_region,max_mix_bdy,ierr)
            if (ierr /= 0) return
            
            call realloc_logical(s% burn_z_mix_region,max_mix_bdy,ierr)
            if (ierr /= 0) return

            s% mix_bdy_q(sz+1:max_mix_bdy) = 0
            s% mix_bdy_loc(sz+1:max_mix_bdy) = 0
            s% top_mix_bdy(sz+1:max_mix_bdy) = .false.
            s% burn_h_mix_region(sz+1:max_mix_bdy) = .false.
            s% burn_he_mix_region(sz+1:max_mix_bdy) = .false.
            s% burn_z_mix_region(sz+1:max_mix_bdy) = .false.
         
         end subroutine realloc
         
                
      end subroutine locate_mixing_boundaries
      
      
      subroutine redo_mlt_for_OPEC(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz, kk, k_xa
         logical, parameter :: dbg = .false.
         real(dp) :: tiny
         
         include 'formats'
         
         if (dbg) write(*,*) 'redo_mlt_for_OPEC'
         
         ierr = 0
         nz = s% nz
         k = 2
         do 
            if (s% mlt_mixing_type(k-1) /= convective_mixing .and. &
                s% mlt_mixing_type(k) == convective_mixing) then
               ! at top of convection region.  extend toward surface
               k_xa = min(nz,k+1)
               do kk = k-1, 1, -1
                  call extend1(kk, k_xa, ierr)
                  if (ierr /= 0) return
                  if (s% mlt_mixing_type(kk) == no_mixing) then
                     if (dbg) write(*,2) 'OPEC stop extend convection upward', kk
                     exit
                  end if
                  if (dbg) write(*,2) 'OPEC extend convection upward', kk
               end do
            else if (s% mlt_mixing_type(k-1) == convective_mixing .and. &
                     s% mlt_mixing_type(k) /= convective_mixing) then
               ! at bottom of convection region.  extend toward center
               k_xa = max(1, k-2)
               do kk = k, nz
                  call extend1(kk, k_xa, ierr)
                  if (ierr /= 0) return
                  if (s% mlt_mixing_type(kk) == no_mixing) then
                     k = kk
                     if (dbg) write(*,2) 'OPEC stop extend convection downward', kk
                     exit
                  end if
                  if (dbg) write(*,2) 'OPEC extend convection downward', kk
               end do
            end if
            if (k == nz) exit
            k = k+1
         end do
         
         contains
         
         subroutine extend1(k, k_xa, ierr)
            use opacities, only: get1_kap
            use micro, only: eos_get
            use mlt_info, only: do1_mlt
            use eos_def
            
            integer, intent(in) :: k, k_xa
            integer, intent(out) :: ierr
            
            real(dp) :: xh, Y, Z, kap_frac_Type2, dlnkap_dlnd, dlnkap_dlnT, &
               opacity1, opacity2, opacity, alfa, beta, Pgas, Prad, &
               gamma1, Cv, chiRho, chiT, Cp, grada, P, gradL_composition_term
            real(dp), dimension(num_eos_basic_results) :: &
               res1, res2, d_eos_dlnd, d_eos_dlnT, d_eos_dabar, d_eos_dzbar
            logical :: dbg
            integer :: species
            real(dp), pointer :: xa(:)
            
            include 'formats'

            ierr = 0
            if (k == 1) return
            
            dbg = k >= 1190

            if (dbg) then
               write(*,2) 'at start of step', s% model_number
               write(*,2) 'old Z', k, max(0d0,1d0-(s% X(k) + s% Y(k)))
               write(*,2) 'old Y', k, s% Y(k)
               write(*,2) 'old opacity', k, s% opacity(k)
               write(*,2) 'old gradr', k, s% gradr(k)
               write(*,2) 'old grada', k, s% grada(k)
               write(*,3) 'old mlt_mixing_type', k, s% mlt_mixing_type(k)
            end if
            
            species = s% species
            
            ! call eos for k using composition from k_xa
            xh = s% X(k_xa)
            Y = s% Y(k_xa)
            Z = max(0d0,1d0-(xh + Y))
            xa(1:species) => s% xa(1:species,k_xa)
            call eos_get( &
               s, 0, Z, xh, s% abar(k_xa), s% zbar(k_xa), xa, &
               s% rho(k), s% lnd(k)/ln10, s% T(k), s% lnT(k)/ln10, &
               res1, d_eos_dlnd, d_eos_dlnT, d_eos_dabar, d_eos_dzbar, ierr)
            if (ierr /= 0) return
            call eos_get( &
               s, 0, Z, xh, s% abar(k_xa), s% zbar(k_xa), xa, &
               s% rho(k-1), s% lnd(k-1)/ln10, s% T(k-1), s% lnT(k-1)/ln10, &
               res2, d_eos_dlnd, d_eos_dlnT, d_eos_dabar, d_eos_dzbar, ierr)
            if (ierr /= 0) return
            
            ! call kap for k using composition from k_xa 
            opacity1 = get1_kap( &
               s, 0, s% zbar(k_xa), xa, s% q(k), 1d0, &
               s% lnd(k)/ln10, s% lnT(k)/ln10, 0d0, &
               s% lnfree_e(k), s% d_eos_dlnd(i_lnfree_e,k), &
               s% d_eos_dlnT(i_lnfree_e,k), &
               kap_frac_Type2, dlnkap_dlnd, dlnkap_dlnT, ierr)
            if (ierr /= 0) return
            opacity2 = get1_kap( &
               s, 0, s% zbar(k_xa), xa, s% q(k-1), 1d0, &
               s% lnd(k-1)/ln10, s% lnT(k-1)/ln10, 0d0, &
               s% lnfree_e(k-1), s% d_eos_dlnd(i_lnfree_e,k-1), &
               s% d_eos_dlnT(i_lnfree_e,k-1), &
               kap_frac_Type2, dlnkap_dlnd, dlnkap_dlnT, ierr)
            if (ierr /= 0) return
            
            alfa = s% dq(k-1)/(s% dq(k-1) + s% dq(k))
            beta = 1 - alfa

            gradL_composition_term = 0 ! kill off the stabilizing composition gradient
            gamma1 = alfa*res1(i_gamma1) + beta*res2(i_gamma1)
            Cv = alfa*res1(i_Cv) + beta*res2(i_Cv)
            chiRho = alfa*res1(i_chiRho) + beta*res2(i_chiRho)
            chiT = alfa*res1(i_chiT) + beta*res2(i_chiT)
            Cp = alfa*res1(i_Cp) + beta*res2(i_Cp)
            grada = alfa*res1(i_grad_ad) + beta*res2(i_grad_ad)
            Pgas = alfa*exp_cr(res1(i_lnPgas)) + beta*exp_cr(res2(i_lnPgas))
            Prad = alfa*s% Prad(k) + beta*s% Prad(k-1)
            P = Pgas + Prad
            opacity = alfa*opacity1 + beta*opacity2

            ! call MLT for k with the new eos and kap results
            
            call do1_mlt(s, k, s% mixing_length_alpha, gradL_composition_term, &
               opacity, gamma1, Cv, chiRho, chiT, Cp, grada, P, xh, &
               ierr)
            if (ierr /= 0) return
            
            if (dbg) then
               write(*,2) 'gamma1', k, gamma1
               write(*,2) 'Cv', k, Cv
               write(*,2) 'chiRho', k, chiRho
               write(*,2) 'chiT', k, chiT
               write(*,2) 'Cp', k, Cp
               write(*,2) 'grada', k, grada
               write(*,2) 'Pgas', k, Pgas
               write(*,2) 'Prad', k, Prad
               write(*,2) 'P', k, P
               write(*,2) 'opacity', k, opacity
               write(*,2) 'L', k, s% L(k)
               write(*,2) 'gradr', k, s% gradr(k)
               write(*,2) 'gradr/grada', k, s% gradr(k)/grada
               write(*,3) 'new mlt_mixing_type', k, s% mlt_mixing_type(k)
               write(*,*)
            end if
         
         end subroutine extend1
      
      end subroutine redo_mlt_for_OPEC
      
      
      subroutine remove_tiny_mixing(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz
         logical, parameter :: dbg = .false.
         real(dp) :: tiny
         
         include 'formats'
         
         if (dbg) write(*,*) 'remove_tiny_mixing'
         
         ierr = 0
         nz = s% nz
         
         tiny = s% clip_D_limit
         do k=1,nz
            if (s% D_mix(k) < tiny) then
               s% cdc(k) = 0
               s% D_mix(k) = 0
               call zero_D_mix_partials(s,k)
               s% conv_vel(k) = 0
               s% mixing_type(k) = no_mixing
            end if
         end do
      
      end subroutine remove_tiny_mixing
      
      
      ! remove single point mixing or non-mixing regions
      ! NOTE: does not remove thermohaline singletons
      subroutine remove_mixing_singletons(s, ierr)
         !use star_utils, only: scale_height
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz
         logical, parameter :: dbg = .false.
         real(dp) :: lambda
         
         include 'formats'
         
         if (dbg) write(*,*) 'remove_mixing_singletons'
         
         ierr = 0
         nz = s% nz
         
         do k=2,nz-1
            if (s% cdc(k) == 0) then
               if (s% cdc(k-1) /= 0 .and. s% cdc(k+1) /= 0) then
                  s% cdc(k) = (s% cdc(k-1) + s% cdc(k+1))/2
                  s% D_mix(k) = s% cdc(k)/pow2(4*pi*s% r(k)*s% r(k)*s% rho(k))
                  call zero_D_mix_partials(s,k)
                  lambda = s% mixing_length_alpha*(s% scale_height(k-1) + s% scale_height(k+1))/2
                  s% conv_vel(k) = 3*s% D_mix(k)/lambda
                  s% mixing_type(k) = max(s% mixing_type(k-1), s% mixing_type(k+1))
               if (dbg) write(*,3) 'remove radiative singleton', k, nz
               end if
            else if (s% okay_to_remove_mixing_singleton) then 
               if (s% cdc(k-1) == 0 .and. s% cdc(k+1) == 0) then
                  call set_use_gradr(s,k)
                  s% cdc(k) = 0
                  s% D_mix(k) = 0
                  call zero_D_mix_partials(s,k)
                  s% conv_vel(k) = 0
                  s% mixing_type(k) = no_mixing
                  if (dbg) write(*,3) 'remove mixing singleton', k, nz
               end if
            end if
         end do
         
         if (s% cdc(1) == 0) then
            if (s% cdc(2) /= 0) then
               s% cdc(1) = s% cdc(2)
               s% D_mix(1) = s% D_mix(2)
               call zero_D_mix_partials(s,1)
               s% conv_vel(1) = s% conv_vel(2)
               s% mixing_type(1) = s% mixing_type(2)
               if (dbg) write(*,3) 'remove radiative singleton', 1, nz
            end if
         else
            if (s% cdc(2) == 0) then
               call set_use_gradr(s,1)
               s% cdc(1) = 0
               s% D_mix(1) = 0
               call zero_D_mix_partials(s,1)
               s% conv_vel(1) = 0
               s% mixing_type(1) = no_mixing
               if (dbg) write(*,2) 'remove mixing singleton', 1
            end if
         end if
         
         if (s% cdc(nz) == 0) then
            if (s% cdc(nz-1) /= 0) then
               s% cdc(nz) = s% cdc(nz-1)
               s% D_mix(nz) = s% D_mix(nz-1)
               call zero_D_mix_partials(s,nz)
               s% conv_vel(nz) = s% conv_vel(nz-1)
               s% mixing_type(nz) = s% mixing_type(nz-1)
               if (dbg) write(*,2) 'remove radiative singleton: s% cdc(nz-1)', nz, s% cdc(nz-1)
            end if
         else
            if (s% cdc(nz-1) == 0) then
               call set_use_gradr(s,nz)
               s% cdc(nz) = 0
               s% D_mix(nz) = 0
               call zero_D_mix_partials(s,nz)
               s% conv_vel(nz) = 0
               s% mixing_type(nz) = no_mixing
               if (dbg) write(*,2) 'remove mixing singleton: s% cdc(nz)', nz, s% cdc(nz)
            end if
         end if
      
      end subroutine remove_mixing_singletons
      
      
      subroutine close_convection_gaps(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call close_gaps(s, convective_mixing, s% min_convective_gap, ierr)
      end subroutine close_convection_gaps
      
      
      subroutine close_thermohaline_gaps(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call close_gaps(s, thermohaline_mixing, s% min_thermohaline_gap, ierr)
      end subroutine close_thermohaline_gaps
      
      
      subroutine close_semiconvection_gaps(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         call close_gaps(s, semiconvective_mixing, s% min_semiconvection_gap, ierr)
      end subroutine close_semiconvection_gaps
      
      
      subroutine close_gaps(s, mix_type, min_gap, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: mix_type
         real(dp), intent(in) :: min_gap
         integer, intent(out) :: ierr
         
         integer :: k, kk, nz
         logical :: in_region, dbg
         real(dp) :: rtop, rbot, Hp
         integer :: ktop, kbot ! k's for gap         
         include 'formats'  
         
         dbg = .false.
         !dbg = (mix_type == convective_mixing)
         if (dbg) write(*,*) 'close_gaps convective_mixing'
         if (dbg) write(*,3) 'mixing_type', 1152, s% mixing_type(1152)
         ierr = 0         
         if (min_gap < 0) return         
         nz = s% nz        
         in_region = (s% mixing_type(nz) == mix_type)
         rbot = 0
         kbot = nz
         do k=nz-1, 2, -1
            if (in_region) then
               if (s% mixing_type(k) /= mix_type) then ! end of region
                  kbot = k+1
                  rbot = s% r(kbot)
                  in_region = .false.
                  if (dbg) write(*,2) 'end of region', kbot, rbot
               end if
            else
               if (s% mixing_type(k) == mix_type) then ! start of region
                  ktop = k
                  rtop = s% r(ktop)
                  Hp = s% P(ktop)/(s% rho(ktop)*s% grav(ktop))
                  if (dbg) write(*,2) 'start of region', ktop, rtop
                  if (dbg) write(*,1) 'rtop - rbot < Hp*min_gap', (rtop - rbot) - Hp*min_gap, &
                     rtop - rbot, Hp*min_gap, Hp, min_gap, (rtop-rbot)/Hp
                  if (rtop - rbot < Hp*min_gap) then
                     if (kbot < nz) then
                        s% cdc(ktop+1:kbot-1) = (s% cdc(ktop) + s% cdc(kbot))/2
                        s% D_mix(ktop+1:kbot-1) = &
                           (s% D_mix(ktop) + s% D_mix(kbot))/2
                        do kk=ktop+1,kbot-1
                           call zero_D_mix_partials(s,kk)
                        end do
                        s% conv_vel(ktop+1:kbot-1) = (s% conv_vel(ktop) + s% conv_vel(kbot))/2
                        s% mixing_type(ktop+1:kbot-1) = mix_type
                        if (dbg) write(*,3) 'close mixing gap', &
                              ktop+1, kbot-1, (rtop - rbot)/Hp, rtop - rbot, Hp
                     else
                        s% cdc(ktop+1:kbot) = s% cdc(ktop)
                        s% D_mix(ktop+1:kbot) = s% D_mix(ktop)
                        do kk=ktop+1,kbot
                           call zero_D_mix_partials(s,kk)
                        end do
                        s% conv_vel(ktop+1:kbot) = s% conv_vel(ktop)
                        s% mixing_type(ktop+1:kbot) = mix_type
                        if (dbg) write(*,3) 'close mixing gap', &
                           ktop+1, kbot, (rtop - rbot)/Hp, rtop - rbot, Hp
                     end if
                  end if
                  in_region = .true.
               end if
            end if
         end do
         if (dbg) write(*,3) 'mixing_type', 1152, s% mixing_type(1152)
         if (dbg) write(*,*) 'done close_gaps'
         !if (dbg) stop 'done close_gaps'
         
      end subroutine close_gaps
      
      
      ! if find radiative region embedded in thermohaline,
      ! and max(gradL - grada) in region is < 1d-3
      ! and region height is < min_thermohaline_dropout
      ! then convert the region to thermohaline
      subroutine remove_thermohaline_dropouts(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz, j
         logical :: in_region
         real(dp) :: rtop, rbot, Hp, q_upper, q_lower, alfa, beta
         integer :: ktop, kbot ! k's for gap
         logical :: all_small 
         logical, parameter :: dbg = .false.        
         include 'formats'         
         ierr = 0         
         nz = s% nz        
         rbot = s% r(nz)
         kbot = nz-1
         in_region = (s% mixing_type(kbot) == no_mixing)
         all_small = .false.
         do k=nz-2, 2, -1
            if (in_region) then
               if (s% mixing_type(k) == no_mixing) then ! check if okay
                  if (s% gradL(k) - s% grada_at_face(k) > s% max_dropout_gradL_sub_grada) &
                     all_small = .false.
               else ! end of radiative region
                  ktop = k+1
                  rtop = s% r(ktop)
                  Hp = s% P(ktop)/(s% rho(ktop)*s% grav(ktop))
                  q_upper = s% q(ktop-1)
                  q_lower = s% q(kbot+1)
                  !write(*,3) 'ktop kbot', ktop, kbot
                  if (.false. .and. ktop > 1100 .and. kbot < 1400) &
                     write(*,3) '(rtop - rbot)/Hp', ktop, kbot, (rtop - rbot)/Hp
                  if (rtop - rbot < Hp*s% min_thermohaline_dropout .and. &
                      s% mixing_type(ktop-1) == thermohaline_mixing .and. &
                      s% mixing_type(kbot+1) == thermohaline_mixing .and. &
                      q_upper - q_lower > 1d-20 .and. all_small) then
                     do j = ktop, kbot ! interpolate in q
                        alfa = (s% q(j) - q_lower)/(q_upper - q_lower)
                        beta = 1 - alfa
                        s% cdc(j) = alfa*s% cdc(ktop-1) + beta*s% cdc(kbot+1)
                        s% D_mix(j) = alfa*s% D_mix(ktop-1) + beta*s% D_mix(kbot+1)
                        call zero_D_mix_partials(s,j)
                        s% conv_vel(j) = alfa*s% conv_vel(ktop-1) + beta*s% conv_vel(kbot+1)
                        s% mixing_type(j) = thermohaline_mixing
                     end do
                  end if
                  in_region = .false.
               end if
            else
               if (s% mixing_type(k) == no_mixing) then ! start of region
                  kbot = k
                  rbot = s% r(kbot)
                  in_region = .true.
                  all_small = &
                     (s% gradL(k) - s% grada_at_face(k) <= s% max_dropout_gradL_sub_grada)
               end if
            end if
         end do
         
      end subroutine remove_thermohaline_dropouts
      
      
      subroutine remove_embedded_semiconvection(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz
         logical :: in_region
         integer :: kbot, ktop
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         ierr = 0
         if (.not. s% remove_embedded_semiconvection) return
         
         nz = s% nz
         
         in_region = check(nz)
         kbot = nz
         do k=nz-1, 2, -1
            if (in_region) then
               if (.not. check(k)) then ! end of region
                  ktop = k+1
                  in_region = .false.
                  call clean_region
               end if
            else ! not in region
               if (check(k)) then ! start of region
                  kbot = k
                  in_region = .true.
               end if
            end if
         end do
         
         if (in_region) then
            ktop = 1
            call clean_region
         end if
         
         
         contains
         
         
         subroutine clean_region
            integer :: kbot1, ktop1, kk
            include 'formats'
            if (dbg) write(*,3) 'clean_region semiconvective', kbot, ktop
            ! move top to below top convective region
            do while (s% mixing_type(ktop) == convective_mixing)
               ktop = ktop + 1
               if (ktop >= kbot) return
            end do
            if (dbg) write(*,2) 'new ktop 1', ktop
            ! move top to below top semiconvective region
            do while (s% mixing_type(ktop) == semiconvective_mixing)
               ktop = ktop + 1
               if (ktop >= kbot) return
            end do
            if (dbg) write(*,2) 'new ktop 2', ktop
            ! move bot to above bottom convective region
            do while (s% mixing_type(kbot) == convective_mixing)
               kbot = kbot - 1
               if (ktop >= kbot) return
            end do
            if (dbg) write(*,2) 'new kbot 1', kbot
            ! move bot to above bottom semiconvective region
            do while (s% mixing_type(kbot) == semiconvective_mixing)
               kbot = kbot - 1
               if (ktop >= kbot) return
            end do
            if (dbg) write(*,2) 'new kbot 2', kbot
            ! convert any semiconvective region between kbot and ktop
            kbot1 = kbot
            do while (kbot1 > ktop)
               ! move kbot1 to bottom of next semiconvective region
               do while (s% mixing_type(kbot1) == convective_mixing)
                  kbot1 = kbot1 - 1
                  if (kbot1 <= ktop) return
               end do
               ktop1 = kbot1
               ! move ktop1 to top of semiconvective region
               do while (s% mixing_type(ktop1) == semiconvective_mixing)
                  ktop1 = ktop1 - 1
                  if (ktop1 <= ktop) return
               end do
               s% D_mix(ktop1+1:kbot1) = s% D_mix(ktop1)
               do kk = ktop1+1,kbot1
                  call zero_D_mix_partials(s,kk)
               end do
               s% mixing_type(ktop1+1:kbot1) = convective_mixing
               s% conv_vel(ktop1+1:kbot1) = s% conv_vel(ktop1)
               s% mixing_type(ktop1+1:kbot1) = convective_mixing
               if (dbg) write(*,3) 'merge semiconvective island', kbot1, ktop1+1
               kbot1 = ktop1
            end do
         end subroutine clean_region
         
         
         logical function check(k)
            integer, intent(in) :: k
            check = &
               (s% mixing_type(k) == semiconvective_mixing) .or. &
               (s% mixing_type(k) == convective_mixing)
         end function check
         
      end subroutine remove_embedded_semiconvection

      
      subroutine do_mix_envelope(s)
         type (star_info), pointer :: s
         real(dp) :: T_mix_limit         
         integer :: j, k, i, nz
         
         include 'formats'
         
         T_mix_limit = s% T_mix_limit
         !write(*,1) 'T_mix_limit', T_mix_limit
         if (T_mix_limit <= 0) return
         nz = s% nz
         j = 0
         do k = 1, nz ! search inward until find T >= T_mix_limit
            if (s% T(k) >= T_mix_limit) then
               j = k; exit
            end if
         end do
         if (j==0) j=nz ! all T < T_mix_limit
         ! find base of innermost convection that has T < T_mix_limit
         i = 0
         do k = j, 1, -1
            if (s% mixing_type(k) == convective_mixing) then
               i = k; exit
            end if
         end do
         if (i == 0) then
            !write(*,*) 'no convection in region with T < T_mix_limit at point ', j
            return ! no convection in region with T < T_mix_limit
         end if
         ! extend convection region to surface
         j = maxloc(s% cdc(1:i), dim=1)
         s% conv_vel(1:i) = s% conv_vel(j)
         s% cdc(1:i) = s% cdc(j)
         do k = 1,i
            s% D_mix(k) = s% D_mix(j)
            call zero_D_mix_partials(s,k)
         end do
         s% mixing_type(1:i) = convective_mixing
         
         !write(*,*) 'extend convection to surface from point ', i
         !return

         
      end subroutine do_mix_envelope


      subroutine get_convection_sigmas(s, dt, ierr)
         use chem_def, only: chem_isos
         type (star_info), pointer :: s         
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: nz, k, j, species, ktop, kbot, bdy
         real(dp) :: sig_term_limit ! sig_term_limit is used to help convergence
                     
         real(dp) :: siglim, xmstar, dq00, dqm1, cdcterm, dmavg, rho_face, &
            cdc, max_sig, D, xm1, x00, xp1, dm, dX, X, cushion, limit, &
            Tc, full_off, full_on, X_lim, dX_lim, qbot, qtop, &
            f1, f, df_dlnd00, df_dlndm1, df_dlnR, df_d_rho_face, alfa, beta
         logical :: in_convective_region
         real(dp), dimension(:), pointer :: sig, &
            d_sig_dlnd00, d_sig_dlnT00, d_sig_dlndm1, &
            d_sig_dlnTm1, d_sig_dlnR, d_sig_dL
         real(dp), dimension(:), pointer :: D_mix, &
            d_D_mix_dlnd00, d_D_mix_dlnT00, d_D_mix_dlndm1, &
            d_D_mix_dlnTm1, d_D_mix_dlnR, d_D_mix_dL

         include 'formats'
         
         sig_term_limit = s% sig_term_limit

         ierr = 0
         nz = s% nz
         xmstar = s% xmstar
         sig => s% sig
         d_sig_dlnd00 => s% d_sig_dlnd00
         d_sig_dlnT00 => s% d_sig_dlnT00
         d_sig_dlndm1 => s% d_sig_dlndm1
         d_sig_dlnTm1 => s% d_sig_dlnTm1
         d_sig_dlnR => s% d_sig_dlnR
         d_sig_dL => s% d_sig_dL
         D_mix => s% D_mix
         d_D_mix_dlnd00 => s% d_D_mix_dlnd00
         d_D_mix_dlnT00 => s% d_D_mix_dlnT00
         d_D_mix_dlndm1 => s% d_D_mix_dlndm1
         d_D_mix_dlnTm1 => s% d_D_mix_dlnTm1
         d_D_mix_dlnR => s% d_D_mix_dlnR
         d_D_mix_dL => s% d_D_mix_dL
         
         sig(1) = 0d0
         do k = 2, nz
            D = D_mix(k)
            if (D == 0) then
               sig(k) = 0d0
               cycle
            end if
            rho_face = (s% dq(k-1)*s% rho(k) + s% dq(k)*s% rho(k-1))/(s% dq(k-1) + s% dq(k))
            f1 = 4*pi*s% r(k)*s% r(k)*rho_face
            f = f1*f1
            cdc = D*f
            cdcterm = s% mix_factor*cdc
            dq00 = s% dq(k)
            dqm1 = s% dq(k-1)
            dmavg = xmstar*(dq00+dqm1)/2
            sig(k) = cdcterm/dmavg
            if (is_bad_num(sig(k))) then
               sig(k) = 1d99
               d_sig_dlnd00(k) = 0
               d_sig_dlnT00(k) = 0
               d_sig_dlndm1(k) = 0
               d_sig_dlnTm1(k) = 0
               d_sig_dlnR(k) = 0
               d_sig_dL(k) = 0
            else ! partials of sig = D*f*s% mix_factor/dmavg
               d_sig_dL(k) = d_D_mix_dL(k)*f*s% mix_factor/dmavg
               d_sig_dlnT00(k) = d_D_mix_dlnT00(k)*f*s% mix_factor/dmavg
               d_sig_dlnTm1(k) = d_D_mix_dlnTm1(k)*f*s% mix_factor/dmavg               
               df_dlnR = 4*f
               d_sig_dlnR(k) = &
                  (d_D_mix_dlnR(k)*f + D*df_dlnR)*s% mix_factor/dmavg
               df_d_rho_face = 2*f/rho_face
               df_dlnd00 = s% dq(k-1)*s% rho(k)*df_d_rho_face/(s% dq(k-1) + s% dq(k))
               df_dlndm1 = s% dq(k)*s% rho(k-1)*df_d_rho_face/(s% dq(k-1) + s% dq(k))
               d_sig_dlnd00(k) = &
                  (d_D_mix_dlnd00(k)*f + D*df_dlnd00)*s% mix_factor/dmavg  
               d_sig_dlndm1(k) = &
                  (d_D_mix_dlndm1(k)*f + D*df_dlndm1)*s% mix_factor/dmavg                                    
            end if
         end do

         ! can get numerical problems unless limit sig
         max_sig = maxval(sig)
         if (max_sig < 1) return
         do k = 1, nz
            s% sig_raw(k) = sig(k)
            if (sig(k) == 0) cycle
            if (k > 1) then
               siglim = sig_term_limit*xmstar*min(s% dq(k),s% dq(k-1))/dt
            else
               siglim = sig_term_limit*xmstar*s% dq(k)/dt
            end if            
            if (sig(k) > siglim) sig(k) = siglim
         end do
         
         species = s% species
         ! limit sigma to avoid negative mass fractions
         cushion = 10d0
         Tc = s% T(s% nz)
         full_off = s% Tcenter_max_for_sig_min_factor_full_off
         if (Tc <= full_off) return
         full_on = s% Tcenter_min_for_sig_min_factor_full_on
         limit = s% sig_min_factor_for_high_Tcenter
         if (Tc < full_on) then
            alfa = (Tc - full_off)/(full_on - full_off)
            ! limit is full on for alfa = 1 and limit is 1 for alfa = 0
            limit = limit*alfa + (1d0 - alfa)
         end if
         if (limit >= 1d0 .or. s% num_mix_regions == 0) return
         ! boundaries are in order from center to surface
         ! no bottom boundary at loc=nz included if center is mixed
         ! however, do include top boundary at loc=1 if surface is mixed
         if (s% top_mix_bdy(1)) then
            kbot = s% nz
            qbot = 0d0
            bdy = 1
         else
            kbot = s% mix_bdy_loc(1)
            qbot = s% q(kbot)
            bdy = 2
         end if
         if (.not. s% top_mix_bdy(bdy)) then
            stop 'bad mix bdy info 1'
         end if
         ktop = s% mix_bdy_loc(bdy)
         qtop = s% q(ktop)
         call do1_region
         do while (bdy < s% num_mix_boundaries)
            bdy = bdy+1
            if (s% top_mix_bdy(bdy)) then
               stop 'bad mix bdy info 2'
            end if
            kbot = s% mix_bdy_loc(bdy)
            qbot = s% q(kbot)
            bdy = bdy+1
            if (.not. s% top_mix_bdy(bdy)) then
               stop 'bad mix bdy info 3'
            end if
            ktop = s% mix_bdy_loc(bdy)
            qtop = s% q(ktop)
            call do1_region
         end do
            
            
         contains
         
         
         subroutine do1_region
            real(dp) :: delta_m, max_lim, alfa, beta, delta_m_upper, delta_m_lower
            delta_m = s% m(ktop) - s% m(kbot)
            delta_m_upper = s% delta_m_upper_for_sig_min_factor
            delta_m_lower = s% delta_m_lower_for_sig_min_factor
            if (delta_m >= delta_m_upper) then
               max_lim = 1d0
            else if (delta_m <= delta_m_lower) then
               max_lim = limit
            else
               alfa = (delta_m - delta_m_lower)/(delta_m_upper - delta_m_lower)
               beta = 1d0 - alfa
               max_lim = alfa + beta*limit
            end if 
            do k=max(2,ktop),kbot
               call do1(k, max_lim, &
                  min(s% q(k) - qbot, qtop - s% q(k))*s% xmstar/Msun)
            end do
         end subroutine do1_region
         
         
         subroutine do1(k, max_lim, delta_m_to_bdy)
            integer, intent(in) :: k
            real(dp), intent(in) :: max_lim, delta_m_to_bdy
            real(dp) :: lim, max_delta_m_to_bdy
            include 'formats'
            siglim = sig(k)
            if (siglim == 0d0) return
            ! okay to increase limit up to max_lim
            max_delta_m_to_bdy = s% max_delta_m_to_bdy_for_sig_min_factor
            if (delta_m_to_bdy >= max_delta_m_to_bdy) return ! no change in sig
            lim = limit + (max_lim - limit)*delta_m_to_bdy/max_delta_m_to_bdy
            if (lim >= 1d0) return
            do j=1,species
               xm1 = s% xa(j,k-1)
               x00 = s% xa(j,k)
               if (xm1 > x00) then
                  X = xm1
                  dX = xm1 - x00
                  dm = s% dm(k-1)
               else
                  X = x00
                  dX = x00 - xm1
                  dm = s% dm(k)
                  dX = -dX
               end if
               if (X < 1d-5) cycle
               if (cushion*dt*dX*siglim > dm*X) then                     
                  siglim = dm*X/(dt*dX*cushion)
               end if
            end do
            if (siglim > lim*sig(k)) then
               sig(k) = siglim
               d_sig_dlnd00(k) = 0
               d_sig_dlnT00(k) = 0
               d_sig_dlndm1(k) = 0
               d_sig_dlnTm1(k) = 0
               d_sig_dlnR(k) = 0
               d_sig_dL(k) = 0
            else
               sig(k) = lim*sig(k)
               d_sig_dlnd00(k) = lim*d_sig_dlnd00(k)
               d_sig_dlnT00(k) = lim*d_sig_dlnT00(k)
               d_sig_dlndm1(k) = lim*d_sig_dlndm1(k)
               d_sig_dlnTm1(k) = lim*d_sig_dlnTm1(k)
               d_sig_dlnR(k) = lim*d_sig_dlnR(k)
               d_sig_dL(k) = lim*d_sig_dL(k)
            end if
         end subroutine do1
         
         
      end subroutine get_convection_sigmas
      
      
      subroutine update_rotation_mixing_info(s, ierr)
         use rotation_mix_info, only: set_rotation_mixing_info         
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         
         integer :: k, nz, k0
         logical :: set_min_am_nu_non_rot
         real(dp) :: &
            am_nu_visc_factor, &
            am_nu_DSI_factor, &
            am_nu_SH_factor, &
            am_nu_SSI_factor, &
            am_nu_ES_factor, &
            am_nu_GSF_factor, &
            am_nu_ST_factor, &
            f, lgT, full_off, full_on
         
         include 'formats'
         
         ierr = 0
         nz = s% nz
         
         call set_rotation_mixing_info(s, ierr)
         if (ierr /= 0) return
      
         ! include rotation part for mixing abundances
         full_on = s% D_mix_rotation_max_logT_full_on
         full_off = s% D_mix_rotation_min_logT_full_off
         do k = 2, nz
            lgT = s% lnT(k)/ln10
            if (lgT <= full_on) then
               f = 1d0
            else if (lgT >= full_off) then
               f = 0d0
            else ! lgT > full_on and < full_off
               f = (lgT - full_on) / (full_off - full_on)
            end if
            !!! Modification by C. Cadiou
            !!! see controls.default for default values of the params.
            s% D_mix(k) = &
               s% D_mix_non_rotation(k) + &
               f*s% am_D_mix_factor *(  &   
                  ! note: have dropped viscosity from mixing
                  s% D_DSI_factor * s% D_DSI(k)  + &
                  s% D_SH_factor  * s% D_SH(k)   + &
                  s% D_SSI_factor * s% D_SSI(k)  + &
                  s% D_ES_factor  * s% D_ES(k)   + &
                  s% D_GSF_factor * s% D_GSF(k)  + &
                  s% D_ST_factor  * s% D_ST(k))
            !!! End modification by C. Cadiou
         end do
         

         if (s% trace_k > 0 .and. s% trace_k <= s% nz) then
            do k=2,nz
               write(*,2) 's% D_visc(k)', k, s% D_visc(k)
               write(*,2) 's% D_DSI(k)', k, s% D_DSI(k)
               write(*,2) 's% D_SH(k)', k, s% D_SH(k)
               write(*,2) 's% D_SSI(k)', k, s% D_SSI(k)
               write(*,2) 's% D_ES(k)', k, s% D_ES(k)
               write(*,2) 's% D_GSF(k)', k, s% D_GSF(k)
               write(*,2) 's% D_ST(k)', k, s% D_ST(k)
            end do
         end if
         
         if (s% model_number == -1) then
            k = 3
            write(*,2) 's% D_visc(k)', k, s% D_visc(k)
            write(*,2) 's% D_DSI(k)', k, s% D_DSI(k)
            write(*,2) 's% D_SH(k)', k, s% D_SH(k)
            write(*,2) 's% D_SSI(k)', k, s% D_SSI(k)
            write(*,2) 's% D_ES(k)', k, s% D_ES(k)
            write(*,2) 's% D_GSF(k)', k, s% D_GSF(k)
            write(*,2) 's% D_ST(k)', k, s% D_ST(k)
            write(*,2) 's% D_mix_non_rotation(k)', k, s% D_mix_non_rotation(k)
            write(*,2) 's% D_mix(k)', k, s% D_mix(k)
         end if

         am_nu_DSI_factor = s% am_nu_DSI_factor
         am_nu_SH_factor = s% am_nu_SH_factor
         am_nu_SSI_factor = s% am_nu_SSI_factor
         am_nu_ES_factor = s% am_nu_ES_factor
         am_nu_GSF_factor = s% am_nu_GSF_factor
         am_nu_ST_factor = s% am_nu_ST_factor
         am_nu_visc_factor = s% am_nu_visc_factor
         ! If am_nu_..._factor < -1, use the D_..._factor
         if (am_nu_DSI_factor < 0) am_nu_DSI_factor = s% D_DSI_factor
         if (am_nu_SH_factor < 0) am_nu_SH_factor = s% D_SH_factor
         if (am_nu_SSI_factor < 0) am_nu_SSI_factor = s% D_SSI_factor
         if (am_nu_ES_factor < 0) am_nu_ES_factor = s% D_ES_factor
         if (am_nu_GSF_factor < 0) am_nu_GSF_factor = s% D_GSF_factor
         if (am_nu_ST_factor < 0) am_nu_ST_factor = s% D_ST_factor
         if (am_nu_visc_factor < 0) am_nu_visc_factor = s% D_visc_factor
         
         ! set set_min_am_nu_non_rot to s% set_min_am_nu_non_rot if
         ! ye > min_center_ye_for_min_am_nu_non_rot
         ! and s% set_min_am_nu_non_rot
         set_min_am_nu_non_rot = &
            s% set_min_am_nu_non_rot .and. &
            s% ye(nz) >= s% min_center_Ye_for_min_am_nu_non_rot .and. &
            (.not. s% set_uniform_am_nu_non_rot)
            
			do k=1,nz
            if (s% set_uniform_am_nu_non_rot) then
               s% am_nu_non_rot(k) = s% uniform_am_nu_non_rot
            else
               s% am_nu_non_rot(k) = s% am_nu_factor* &
                  s% am_nu_non_rotation_factor*s% D_mix_non_rotation(k)
            end if
            if (set_min_am_nu_non_rot) &
               s% am_nu_non_rot(k) = max(s% min_am_nu_non_rot, s% am_nu_non_rot(k))
            s% am_nu_rot(k) = s% am_nu_factor * ( &
               am_nu_visc_factor*s% D_visc(k) + &                  
               am_nu_DSI_factor*s% D_DSI(k) + &
               am_nu_SH_factor*s% D_SH(k) + &
               am_nu_SSI_factor*s% D_SSI(k) + &
               am_nu_ES_factor*s% D_ES(k) + &
               am_nu_GSF_factor*s% D_GSF(k) + &
               am_nu_ST_factor*s% nu_ST(k))                
				s% am_nu_omega(k) = &
               s% am_nu_omega_non_rot_factor*s% am_nu_non_rot(k) + &
               s% am_nu_omega_rot_factor*s% am_nu_rot(k)
				if (s% am_nu_omega(k) < 0) then
				   ierr = -1
				   return
				end if
				s% am_nu_j(k) = &
               s% am_nu_j_non_rot_factor*s% am_nu_non_rot(k) + &
               s% am_nu_j_rot_factor*s% am_nu_rot(k)
            !!! Modification by C. Cadiou
            if (s% am_nu_j(k) < 0) then
               ierr = -1
               return
            end if
           !!! End modification by C. Cadiou
			end do
      
         if (s% use_other_am_mixing) then
            call s% other_am_mixing(s% id, ierr)
            if (ierr /= 0) return
         end if
      
         
      end subroutine update_rotation_mixing_info
   
         
      
      end module mix_info
