! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module kuma

      use const_def
      use crlibm_lib

      implicit none


      contains


      subroutine eval_kuma_wind(xlogl,xlteff,stmass,xsurf,xmdfic,vinfkm)
         real(dp), intent(in) :: xlogl,xlteff,stmass,xsurf
         real(dp), intent(out) :: xmdfic,vinfkm
         ! input: xlogl= log L/Lsun; xlteff= log Teff, stmass=M/Msun,
         !        xsurf=surface hydrogen mass fraction               
         ! output: xmdfic=mass loss rate [Msun/yr], vinfkm=v_infinity
         ! ANALYTICAL APPROXIMATION FOR RADIATION DRIVEN WINDS 
         ! PROGRAM WRITTEN BY R.P. KUDRITZKI
         real(dp) :: teff, stlum, ttt, rst, yps, xihe, sig, ga, ganeu, &
            vesckm, vesc, vsound, vsquar, vsoukm, xk, alpha, beta, delta, &
            ucrit, xx, vc, vckms, xmdca, xdbaco, xdjage, xdlame
         
         include 'formats'
         
         teff=exp10_cr(xlteff)
         stlum=exp10_cr(xlogl)
         ttt=teff/5780.
         ttt=ttt*ttt*ttt*ttt
         rst=sqrt(stlum/ttt)
         call mdote(xlogl,rst,stmass,teff,xdjage,xdlame,xdbaco)
         yps=(0.98 - xsurf)/(4.*xsurf)
         xihe=2.
         sig=sigmae(yps,xihe)
         ga=gammma(stlum,stmass,sig)
         !ganeu=gamma2
         !ga=dmax1(ga,ganeu)
         vesckm=stmass*(1.-ga)/rst
         vesckm=617.*sqrt(vesckm)
         vesc=vesckm*1.d05
         vsound=(2.+(1.+xihe)*yps)/(1.+4.*yps)
         vsound=vsound*teff
         vsound=9.085d3*sqrt(vsound)
         vsquar=vsound/vesc
         vsquar=vsquar*vsquar
         vsoukm=vsound*1.d-5

         ! Pauldrach et al. 1993 (MPA 740) for Zeta Pup
         xk=0.085
         alpha=0.657
         beta=1.
         delta=0.095
         ucrit=uc(alpha,beta,delta,vsquar)

         ! Approx. wind integral
         xx=xintap(alpha,beta,delta,ucrit)
         vc=vaucr(vsound,vesc,alpha)
         xx=xx+(1.-alpha)*vc*vc/(alpha*vesc*vesc)
         vckms=vc*1.d-5
         vinfkm=vesckm*sqrt(xx)
         vinfkm=vinfkm*sqrt(alpha/(1.-alpha))

         ! === Approx. Mdot formulae ====
         xmdca=xmdcak(xk,alpha,stlum,stmass,yps,teff)
         xmdfic=xmdfc(xk,alpha,beta,delta,xx,ucrit,rst,vesc,yps,xmdca)
         
         !write(*,1) 'kuma mdot', xmdfic
         !stop

      end subroutine eval_kuma_wind


      real(dp) function cf(u,alpha,beta)
         real(dp), intent(in) :: u,alpha,beta
         ! cf IS THE FINITE CONE ANGLE CORRECTION FACTOR
         real(dp) :: xl, absxl, eps
         include 'formats'
         xl=(beta+1.)*u-1.
         xl=xl*u/beta
         absxl=abs(xl)
         eps=1.d-3
         if (absxl <= eps) then
            cf=1.
            return
         end if
         cf=1.-pow_cr(1.-xl,1.+alpha)
         cf=cf/xl
         cf=cf/(1.+alpha)
         !write(*,1) 'cf', cf
      end function cf


      real(dp) function gdel(u,beta,delta)
         real(dp), intent(in) :: u,beta,delta
         ! COMPUTES FUNCTION G(DELTA)
         real(dp) :: q
         include 'formats'
         q=qq(beta,delta)
         gdel=q*u*u+1.
         !write(*,1) 'gdel', gdel
      end function gdel


      real(dp) function uc(alpha,beta,delta,vsquar)
         real(dp), intent(in) :: alpha,beta,delta,vsquar
         ! ESTIMATE OF RECIPROCAL CRITICAL POINT
         real(dp) :: sigma, xphi, bbb, q, xc, hilf
         include 'formats'
         sigma=alpha*alpha*(1.-alpha)/vsquar
         xphi=phicr(vsquar)
         xphi=xphi*xphi
         sigma=sigma/xphi
         bbb=betcr(alpha,delta)
         q=qq(bbb,delta)
         xc=-2.*q/(sigma*3.)
         hilf=1.46/sigma
         hilf=hilf*(alpha+1.)*pow_cr(alpha,0.6d0)
         hilf=pow_cr(hilf,1d0/3d0)
         xc=hilf+xc
         if (xc < 1.03) xc=1.03
         uc=1./xc
         !write(*,1) 'uc', uc
      end function uc


      real(dp) function phicr(vsquar)
         real(dp), intent(in) :: vsquar
         ! CALCULATES PAULDRACH PHI FUNCTION AT UCRIT
         real(dp) :: vsq, xl
         include 'formats'
         vsq=sqrt(vsquar)
         xl=log10_cr(vsq)
         xl=0.36+xl
         xl=0.3*xl
         xl=pow_cr(vsq,xl)
         phicr=3.0*xl
         !write(*,1) 'phicr', phicr
      end function phicr


      real(dp) function vaucr(vsound,vesc,alpha)
         real(dp), intent(in) :: vsound,vesc,alpha
         ! COMPUTES VELOCITY AT CRITICAL POINT ACC. TO PAULDRACH
         real(dp) :: vsq, phi, hilf, h
         include 'formats'
         vsq=vsound*vsound/(vesc*vesc)
         phi=phicr(vsq)
         hilf=sqrt(phi/(1.-alpha))
         h=1.-pow_cr(1.-alpha,2./alpha)
         hilf=hilf/h
         vaucr=vsound*hilf
         !write(*,1) 'vaucr', vaucr
      end function vaucr


      real(dp) function betcr(alpha,delta)
         real(dp), intent(in) :: alpha,delta
         ! CALCULATES BETA VALUE FOR UCRIT ACC. TO PAULDRACH
         ! LVERS=2 MODIFIED BETACRIT ACC. TO SECOND VERSION
         include 'formats'
         if (alpha <= 0.7) then
            if (delta < 0.03) then
               betcr=2.
            else if (delta < 0.055) then
               betcr=1.
            else if (delta < 0.085) then
               betcr=0.7
            else if (delta < 0.095) then
               betcr=0.5
            else
               betcr=0.25
            end if
         else if (delta >= 0.03) then
            if (delta >= 0.095) then
               betcr=0.7
            else if (delta <= 0.055) then
               betcr=2.
            else
               betcr=1.
            end if
         else
            betcr=2.
         end if
         !write(*,1) 'betcr', betcr
      end function betcr


      real(dp) function qq(beta,delta)
         real(dp), intent(in) :: beta,delta
         ! CALCULATES Q AS FUNCTION OF BETA AND DELTA
         ! Q=A(BETA)**DELTA-1.
         ! A(BETA) LINEAR BETWEEN A(2)=22.1, A(1)=7.5
         !   A(0.7)=4.0, A(0.5)=2.5, A(0.25)=1.18
         real(dp) :: a, b
         include 'formats'
         b = min(2.5,max(0.2,beta))
         if (b >= 1.0) then
            a=15.*(b-1.)+7.5
         else if (b >= 0.7) then
            a=11.66667*(b-0.7)+4.0
         else if (b >= 0.5) then
            a=7.5*(b-0.5)+2.5
         else
            a=5.28*(b-0.25)+1.18
         end if
         qq=pow_cr(a,delta-1)
         !write(*,1) 'qq', qq
      end function qq


      real(dp) function xintap(alpha,beta,delta,ucrit)
         real(dp), intent(in) :: alpha,beta,delta,ucrit
         ! APPROX> WIND INTEGRAL = EQ. (60) +
         real(dp) :: q, bet1, bet2, bet3, bet4, bet5, &
            uc1, uc2, uc3, uc4, hilf, x, g, z, a0, a1, b1, b2
         include 'formats'
         q=qq(beta,delta)
         call coeff(alpha,beta,a0,a1,b1,b2)
         bet1=1./(1.+beta)
         bet2=bet1*bet1
         bet3=bet2*bet1
         bet4=bet3*bet1
         bet5=bet4*bet1
         uc1=ucrit
         uc2=uc1*uc1
         uc3=uc2*uc1
         uc4=uc3*uc1
         hilf=1./(1.-alpha)
         x=a0*(uc1-bet1+hilf*q*(uc3-bet3)/3.)
         x=x-a1*((uc2-bet2)/2.+hilf*q*(uc4-bet4)/4.)
         x=x+bet1
         x=x+b1*(bet2/2.+hilf*q*bet4/4.)
         x=x-b2*(bet3/3.+hilf*q*bet5/5.)
         x=x+hilf*q*bet3/3.
         g=1./(a0-a1*uc1)
         g=g*pow_cr(1./(q*uc2+1.),hilf)
         z=1./g
         z=2./alpha*(1.-z)
         z=g*(1.+sqrt(z))
         xintap=z*x
         !write(*,1) 'xintap', xintap
      end function xintap


      subroutine coeff(alpha,beta,a0,a1,b1,b2)
         real(dp), intent(in) :: alpha,beta
         real(dp), intent(out) :: a0,a1,b1,b2
         ! COEFFICIENTS A0 A1 B0 B1 FOR APPROX. TERMINAL VELOCITY +
         real(dp) :: hilfex, alaaf
         include 'formats'
         hilfex=1.-alpha
         hilfex=1./hilfex
         alaaf=pow_cr(1./(1.+alpha),hilfex)
         a0=(1.+beta-alaaf)/beta
         a1=a0-alaaf
         b1=alpha*hilfex/(2.*beta)
         b2=b1*(beta+1.)
         !write(*,1) 'a0', a0
         !write(*,1) 'a1', a1
         !write(*,1) 'b1', b1
         !write(*,1) 'b2', b2
      end subroutine coeff


      real(dp) function xmdcak(xk,alpha,stlum,stmass,yps,teff)
         real(dp), intent(in) :: xk,alpha,stlum,stmass,yps,teff
         ! MDOT AFTER CAK
         real(dp) :: hilfex, hilf, hiover, xmue, xihe, sig, ga, hi
         include 'formats'
         hilfex=1./alpha
         hilf=1.2762d23*xk*stlum
         hilf=hilf*1.d-10
         hilf=pow_cr(hilf,hilfex)
         hiover=pow_cr(1.d10,hilfex)
         xmue=1.
         hilf=hilf*alpha/vtherm(teff,xmue)
         xihe=2.
         sig=sigmae(yps,xihe)
         ga=gammma(stlum,stmass,sig)
         hilfex=hilfex*(1.-alpha)
         hi=sig*(1.-alpha)/(1.6671d27*stmass*(1.-ga))
         hi=pow_cr(hi,hilfex)
         xmdcak=hilf*hi*hiover
         !write(*,1) 'xmdcak', xmdcak
      end function xmdcak


      real(dp) function sigmae(yps,xihe)
         real(dp), intent(in) :: yps,xihe
         ! SIGMA = THOMSON ABS. COEFF. DIVIDED BY DENSITY
         ! XIHE  = NUMBER OF e- PROVIDIED BY He NUCLEUS
         ! YPS   = N(He)/N(H)
         sigmae=(1.+xihe*yps)/(1.+4.*yps)
         sigmae=sigmae*0.3978
      end function sigmae


      real(dp) function gammma(stlum,stmass,sig)
         real(dp), intent(in) :: stlum,stmass,sig
         ! EDDINGTON LIMIT GAMMA, SIG COMPUTED BY SIGMAE
         gammma=7.655d-5*sig*stlum/stmass
      end function gammma


      real(dp) function vtherm(t,xmue)
         real(dp), intent(in) :: t,xmue
         ! THERMAL VELOCITY OF ATOM WITH ATOMIC WHEIGHT XMUE
         vtherm=1.2848d4*sqrt(t/xmue)
      end function vtherm


      real(dp) function xmdfc(xk,alpha,beta,delta,xint,ucrit,rst,vesc,yps,xmdcak)
         real(dp), intent(in) :: xk,alpha,beta,delta,xint,ucrit,rst,vesc,yps,xmdcak
         ! MDOT FOR FINITE CONE ANGLE [Msun/yr],  EQ. (61)
         ! XMDCAK = CAK MDOT
         ! XINT   = F.C. VELOCITY INTEGRAL
         real(dp) :: xihe, d, alaaf, hiexa, hiexb, hiexc, a, b, c
         include 'formats'
         xihe=2.
         d=3.92859d1*(1.+xihe*yps)/(1.+4.*yps)
         d=d/(rst*rst)
         d=d/vesc
         alaaf=(1.-alpha)/alpha
         d=d*1.d-11*sqrt(alaaf)
         hiexa=1./(alpha-delta)
         hiexb=hiexa*delta
         hiexc=hiexa*alpha
         a=d/sqrt(xint)
         a=pow_cr(a,hiexb)
         b=cf(ucrit,alpha,beta)
         b=b*(1.+qq(beta,delta)*ucrit*ucrit)
         b=pow_cr(b,hiexa)
         c=pow_cr(xmdcak,hiexc)
         ! NORMALISATION TO [Msun/yr] +
         c=c*1.58661d-26
         xmdfc=a*b*c
         !write(*,1) 'xmdfc', xmdfc
      end function xmdfc


      subroutine mdote(xlogl,rst,stmass,teff,xdjage,xdlame,xdbaco)
         real(dp), intent(in) :: xlogl,rst,stmass,teff
         real(dp), intent(out) :: xdjage,xdlame,xdbaco
         ! COMPUTE EMPIRICAL MASS LOSS RATES ACCORDING TO:              +
         ! 1) DE JAGER ET AL.,                          ---> xdjage     +
         ! 2) Lamers, 1981, Ap.J. 245, 593              ---> xdlame     +
         ! 3) Barlow & Cohen, 1977, Ap.J. 213, 737      ---> xdbaco     +
         real(dp) :: x, y
         real(dp), parameter :: &
            a1=6.3168, a2=0.1104, a3=-0.4311, a4=3.579, a5=-1.571, &
            a6=-0.0109, a7=-0.2175, a8=-0.8381, a9=-1.2487, a10=1.5822
         include 'formats'
         x=log10_cr(teff)-4.
         y=xlogl-5.
         xdjage=a1+a2*x+a3*x*x+a4*x*x*x + a5*y+a6*y*y+a7*y*y*y + a8*x*y+a9*x*x*y+a10*x*y*y
         xdjage=-xdjage
         xdlame=1.42*xlogl+0.61*log10_cr(rst)-0.99*log10_cr(stmass)-12.71
         !write(*,1) 'xlogl', xlogl
         !write(*,1) 'rst', rst
         !write(*,1) 'stmass', stmass
         xdbaco=1.10*xlogl+log10_cr(1.35d-12)
         !write(*,1) 'xdjage', xdjage
         !write(*,1) 'xdlame', xdlame
         !write(*,1) 'xdbaco', xdbaco
       end subroutine mdote


       end module kuma

