! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


! Routine eval_fp_ft for computing rotation corrections to the stellar structure equations.
! Following Endal & Sofia, 1976, ApJ 210:184.
! Based on code from Evert Glebbeek which in turn was based on code from Alex Heger.



      module hydro_rotation

      use const_def, only: pi, pi4, standard_cgrav, ln10
      
      
      ! NOTE: we are not using the user supplied cgrav vector here.
      ! that can (should) be changed (if anyone cares).
      
      
      use star_private_def

      implicit none

      ! Angular factors appearing in different integrals
      ! Precompute these for a quarter circle
      ! Read-only after initialization
      integer, parameter, private :: intmax = 101 ! divisions of 1/4 circle
      real(dp), private :: &
         dtheta, dtheta2, theta(intmax), cost(intmax),sint(intmax), &
         cost2(intmax),sint2(intmax), utheta(intmax)
      logical, save, private :: have_initialized = .false.

      logical, parameter :: dbg = .false.   
      
      integer, parameter :: dbg_cell = -1
      

      contains


      subroutine get_rotation_sigmas(s, nzlo, nzhi, dt, ierr)
         use utils_lib, only: is_bad_num
         use alloc, only: non_crit_get_work_array, non_crit_return_work_array
         
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         
         integer :: k, nz
         real(dp), pointer :: am_nu(:), am_sig(:)

         include 'formats'

         ierr = 0
         nz = s% nz
         
         call non_crit_get_work_array(s, am_nu, nz, nz_alloc_extra, 'get_rotation_sigmas', ierr)
         if (ierr /= 0) return            
         call non_crit_get_work_array(s, am_sig, nz, nz_alloc_extra, 'get_rotation_sigmas', ierr)
         if (ierr /= 0) return            

         call get1_am_sig(s, nzlo, nzhi, s% am_nu_j, s% am_sig_j, dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,1) 'failed in get_rotation_sigmas'
            call dealloc
            return
         end if
         
         do k=1,nz
            am_nu(k) = s% am_nu_j(k) + s% am_nu_omega(k)
         end do
         ! do it this way so apply limit to sum; sum is used as diffusion coeff for omega
         call get1_am_sig(s, nzlo, nzhi, am_nu, am_sig, dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,1) 'failed in get_rotation_sigmas'
            call dealloc
            return
         end if
         
         do k=1,nz
            s% am_sig_omega(k) = max(0d0, am_sig(k) - s% am_sig_j(k))
         end do
         
         call dealloc
         
         contains
         
         subroutine dealloc
            call non_crit_return_work_array(s, am_nu, 'diffusion')
            call non_crit_return_work_array(s, am_sig, 'diffusion')
         end subroutine dealloc

      end subroutine get_rotation_sigmas
      
         
      subroutine get1_am_sig(s, nzlo, nzhi, am_nu, am_sig, dt, ierr)         
         type (star_info), pointer :: s         
         integer, intent(in) :: nzlo, nzhi
         real(dp), intent(in) :: dt
         real(dp), dimension(:), pointer :: am_nu, am_sig
         integer, intent(out) :: ierr
         
         integer :: k, nz, nz2                
         real(dp) :: r, D, am_nu_E00, am_nu_Ep1, dmbar, s1, &
            sig_term_limit, xmstar, siglim
            
         include 'formats'
         
         ierr = 0
         xmstar = s% xmstar
         sig_term_limit = s% am_sig_term_limit
         nz = s% nz
         ! note: am_sig is cell centered, so combine adjacent am_nu face values.
         am_nu_E00 = 0; am_nu_Ep1 = 0
         nz2 = nzhi
         if (nzhi == nz) then
            k = nz
            D = am_nu_E00
            r = 0.5d0*s% r(k)
            s1 = 4*pi*r*r*s% rho(k)
            am_sig(k) = s1*s1*D/s% dm(k)
            nz2 = nz-1
         end if
         do k = nzlo, nz2
            am_nu_E00 = max(0d0, am_nu(k))
            am_nu_Ep1 = max(0d0, am_nu(k+1))
            ! Meynet, Maeder, & Mowlavi, A&A 416, 1023-1036, 2004, eqn 51 with f = 1/2.
            D = 2*(am_nu_E00*am_nu_Ep1)/max(1d-99, am_nu_E00 + am_nu_Ep1)
            r = 0.5d0*(s% r(k) + s% r(k+1)) ! consistent with f = 1/2
            s1 = 4*pi*r*r*s% rho(k)
            am_sig(k) = s1*s1*D/s% dm(k)
         end do
         
         ! can get numerical problems unless limit am_sig
         ! adjust am_sig to make sure am_sig*dt/dmbar is < allowed limit
         do k = nzlo, nzhi
            if (k < nz) then
               dmbar = xmstar*min(s% dq(k),s% dq(k+1))
               siglim = sig_term_limit*dmbar/dt
            else
               dmbar = xmstar*s% dq(k)
               siglim = sig_term_limit*dmbar/dt
            end if
            if (am_sig(k) > siglim) then
               if (.false.) then
                  write(*,*) 'get1_am_sig'
                  write(*,2) 'log10(sig_term_limit)', k, log10_cr(sig_term_limit)
                  write(*,2) 'log10(am_sig(k))', k, log10_cr(am_sig(k))
                  write(*,2) 'log10(siglim)', k, log10_cr(siglim)
                  write(*,2) 'dmbar', k, dmbar
                  write(*,2) 'dt', k, dt
                  write(*,2) 'am_sig(k)*dt/dmbar', k, am_sig(k)*dt/dmbar
                  write(*,2) 'siglim*dt/dmbar', k, siglim*dt/dmbar
                  write(*,2) 'sig_term_limit', k, sig_term_limit
                  write(*,*)
                  !stop 'get1_am_sig'
               end if
               am_sig(k) = siglim
            end if
         end do

      end subroutine get1_am_sig

      
      subroutine set_uniform_omega(id, omega, ierr)
         use star_utils, only: &
            set_surf_avg_rotation_info, use_xh_to_update_i_rot
         integer, intent(in) :: id
         real(dp), intent(in) :: omega
         integer, intent(out) :: ierr         
         type (star_info), pointer :: s
         integer :: k
         include 'formats'
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         call use_xh_to_update_i_rot(s)
         do k=1, s% nz
            s% omega(k) = omega
            s% j_rot(k) = s% i_rot(k)*s% omega(k)
         end do
         call set_surf_avg_rotation_info(s)
      end subroutine set_uniform_omega


      subroutine set_rotation_info(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         integer :: k
         include 'formats'         
         ierr = 0
         if (.not. s% rotation_flag) return
         call eval_fp_ft( &
               s, s% nz, s% m, s% r, s% rho, s% omega, s% ft_rot, s% fp_rot, &
               s% r_polar, s% r_equatorial, s% report_ierr, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in eval_fp_ft'
         end if      
      end subroutine set_rotation_info


      ! Input variables:
      !  N     Number of meshpoints used by the model (arrays are this size)
      !  XM    Mass coordinate [gram]
      !  R     Radius coordinate [cm]
      !  RHO   Density [gram/cm^3]
      !  AW    Angular velocity [rad/sec]
      ! Output variables:
      !  Correction factor FT at each meshpoint
      !  Correction factor FP at each meshpoint
      !  r_polar, r_equatorial at each meshpoint
      subroutine eval_fp_ft( &
            s, nz, xm, r, rho, aw, ft, fp, r_polar, r_equatorial, report_ierr, ierr)
         use num_lib
         type (star_info), pointer :: s
         integer, intent(in) :: nz
         real(dp), intent(in) :: aw(:), r(:), rho(:), xm(:) ! (nz)
         real(dp), intent(out) :: ft(:), fp(:), r_polar(:), r_equatorial(:) ! (nz)
         logical, intent(in) :: report_ierr
         integer, intent(out) :: ierr
         
         integer :: j, k, kmax, ift_in, ift_out, ifp_in, ifp_out, kanz, i_in, i_out
         real(dp) :: gur(nz),eta(nz),rnull(nz),geff(nz),spgp(nz),spgm(nz), &
            gp(intmax),gmh(intmax)
         real(dp) :: rae(intmax),rov,etanu,wr,xm_out,&
            a,ft_min,fp_min,tegra,rnorm,rom1,rho_out,rho_in,rnull_out,rnull_in, &
            r_in,r_out,lnr_in, lnr_out, dat, dot, dut, r1, r2, dfdx, rnuv1, rnuv2, rrs, rrs1

         integer, parameter :: lipar = 2, lrpar = 5, imax = 50
         real(dp), parameter :: epsx=1d-8, epsy=epsx
         real(dp) :: apot1, apot2
      
         integer :: liwork, lwork
         integer, pointer :: iwork(:)
         real(dp), pointer :: work(:)
         
         integer, parameter :: out_io = 0, max_steps = 1000, itol = 0, lout = 0
         real(dp) :: rtol(1), atol(1), max_step_size, h
         integer :: idid
         
         real(dp), target :: veta_ary(1)
         real(dp), pointer :: veta(:)
         
         real(dp), target :: rpar_ary(lrpar) 
         integer, target :: ipar_ary(lipar)
         real(dp), pointer :: rpar(:)
         integer, pointer :: ipar(:)
         
         logical :: dbg
         
         rpar => rpar_ary
         ipar => ipar_ary
      
         include 'formats'
         
         dbg = .false. ! (s% model_number >= 5)

         ierr = 0
         veta => veta_ary
         if (.not. have_initialized) then
            write(*,*) 'must call init_rotation prior to getting rotation info'
            ierr = -1; return
         end if
         
         kmax = 0

         ft_min=1.0d0
         fp_min=1.0d0
         ift_in=0
         ift_out=0
         ifp_in=0
         ifp_out=0

         call cash_karp_work_sizes(1,liwork,lwork)
         allocate(work(lwork),iwork(liwork))
      
         ! main loop
         etanu = -1.00d-20
         tegra=0.d0
         rnull_out=0d0
         rho_out = rho(nz)
         r_out = 0
         lnr_out = 1d-10

         rnorm=1.d0
         do i_in=nz+1,2,-1
            i_out = i_in-1

            ! for each shell, compute the distortion (in terms of r0) of the shell. The
            ! inner edge of the shell is at r(i_in), the outer edge at r(i_out). The density is
            ! defined in the interior of each shell and needs to be interpolated when
            ! it is needed at the edge.


            rho_in = rho_out
            rho_out = rho(i_out)
            ! Compute mean density at this point
            r_in = r_out
            lnr_in = lnr_out
            r_out = r(i_out)
            lnr_out = log_cr(r_out)
            rom1 = xm(i_out)*0.75d0/(pi*r_out*r_out*r_out)
            rov = rho_in/rom1
            veta(1) = etanu
            kanz = 1
            dat = 1.d-04
            dot = (lnr_out-lnr_in)/8.d0
            dut = (lnr_out-lnr_in)/300.d0
         
            ! E&S, A6 by integration of A7         
            rtol = 1d-6
            atol = 1d-6
            h = lnr_out - lnr_in
            max_step_size = 0d0
            rpar(1) = rov
            call cash_karp( &
               1, rad_fcn, lnr_in, veta, lnr_out, & 
               h, max_step_size, max_steps, & 
               rtol, atol, itol, & 
               null_solout, out_io, & 
               work, lwork, iwork, liwork, & 
               lrpar, rpar, lipar, ipar, & 
               lout, idid)
            if (idid < 0) then
               if (report_ierr .or. dbg) write(*,2) 'eval_fp_ft failed in integration', i_out
               ierr = -1
               if (dbg) stop 'eval_fp_ft'
               exit
            end if         
         
            eta(i_out)=veta(1)
            xm_out=xm(i_out)
            wr=aw(i_out)
            etanu=veta(1)
            r1=4.00d0*r_out
            r2=0.01d0*r_out
            do j = 1, 20 ! exit when have bracketed root
               ! needs etanu,wr,r_out,xm_out; sets a
               rpar(1) = etanu
               rpar(2) = wr
               rpar(3) = r_out
               rpar(4) = xm_out
               ipar(1) = i_out
               ipar(2) = 0
               ierr = 0
               apot1 = apot_fcn(r1, dfdx, lrpar, rpar, lipar, ipar, ierr)
               if (ierr /= 0) then
                  if (report_ierr .or. dbg) write(*,2) 'eval_fp_ft failed in calculation of apot1', i_out
                  if (dbg) stop 'eval_fp_ft'
                  exit
               end if
               apot2 = apot_fcn(r2, dfdx, lrpar, rpar, lipar, ipar, ierr)
               if (ierr /= 0) then
                  if (report_ierr .or. dbg) write(*,2) 'eval_fp_ft failed in calculation of apot2', i_out
                  if (dbg) stop 'eval_fp_ft'
                  exit
               end if
               if (apot1*apot2 <= 0) exit
               r1=4.00d0*r1
               r2=0.01d0*r2
            end do

            if (apot1*apot2 > 0) then
               ierr = -1
               if (report_ierr .or. dbg) &
                  write(*,2) 'eval_fp_ft failed in calculation of apot1, apot2', i_out, apot1, apot2


               !exit
               write(*,2) 'i_out', i_out
               write(*,1) 'r_out', r_out
               write(*,1) 'r1', r1
               write(*,1) 'r2', r2
               write(*,1) 'apot1', apot1
               write(*,1) 'apot2', apot2
               write(*,1) 'epsx', epsx
               write(*,1) 'epsy', epsy
               write(*,2) 'imax', imax
               stop 'hydro_rotation: eval_fp_ft'


               exit
            end if
            
            rnuv1 = safe_root_with_initial_guess( &
               apot_fcn, r_out, r1, r2, apot1, apot2, &
               imax, epsx, epsy, lrpar, rpar, lipar, ipar, ierr)
            if (ierr /= 0) then
               if (report_ierr .or. dbg) write(*,2) 'eval_fp_ft failed in calculation of apot', i_out
               if (.not. dbg) exit
               write(*,2) 'i_out', i_out
               write(*,1) 'r_out', r_out
               write(*,1) 'r1', r1
               write(*,1) 'r2', r2
               write(*,1) 'apot1', apot1
               write(*,1) 'apot2', apot2
               write(*,1) 'epsx', epsx
               write(*,1) 'epsy', epsy
               write(*,2) 'imax', imax
               stop 'hydro_rotation: eval_fp_ft'
            end if
            
            a = rpar(5)
            rnuv2=rnuv1
            rnull(i_out) =  rnuv2
            rnull_in = rnull_out
            rnull_out = rnuv2

            ! compute integral for the potential calculation. See Endal&Sofia for details
            
            tegra = psiint(0.5d0*(rho_out+rho_in),xm_out,wr,etanu,rnull_in,rnull_out) + tegra

            ! calculate g and 1/g on a quarter circle.
            gur(i_out)=0.0d0
            do k=1,intmax,1
               gp(k)=gpsi(rnull(i_out), tegra, k, a, wr, xm_out, rae)
               gmh(k)=1.d0 / gp(k)
               gur(i_out)=gur(i_out)+gp(k)
            end do
            gur(i_out)=gur(i_out)*dtheta/pi*2.d0
            r_polar(i_out) = max(rae(1),r2)
            r_equatorial(i_out) = min(rae(intmax),r1)               

            ! find spsi*<g> and spsi*<1/g>. spsi is the surface area of the equipotential
            rrs=rae(1)*rae(1)*sint(1)
            rrs1=rae(intmax)*rae(intmax)*sint(intmax)
            spgp(i_out)=(gp(1)*rrs+gp (intmax)*rrs1)*dtheta2
            spgm(i_out)=(gmh(1)*rrs+gmh(intmax)*rrs1)*dtheta2
            
            do k=2,intmax-1
               rrs=rae(k)*rae(k)*sint(k)
               spgp(i_out)=spgp(i_out)+gp(k)*rrs*dtheta
               spgm(i_out)=spgm(i_out)+gmh(k)*rrs*dtheta
            enddo
            spgp(i_out)=spgp(i_out)*pi4
            spgm(i_out)=spgm(i_out)*pi4

            !  Find fp and ft

            fp(i_out) =  pi4 * r_out*r_out*r_out*r_out     / (standard_cgrav*xm_out*spgm(i_out))
            ft(i_out) = pow2(pi4 * r_out*r_out) / (spgp(i_out)*spgm(i_out))

            if (ft(i_out) < s% ft_error_limit) then
               ierr = -1
               if (report_ierr .or. dbg) then
                  write(*,2) 'FT too small', i_out, ft(i_out), s% ft_error_limit
                  stop 'eval_fp_ft'
               end if
               exit
               if (ift_in == 0) ift_in=i_out
               ift_out=i_out
               ft_min=min(ft_min,ft(i_out))
            elseif (ift_in /= 0) then 
               ift_in=0
               ift_out=0
               ft_min=1.0d0
            endif 

            if (fp(i_out) < s% fp_error_limit) then
               ierr = -1
               if (report_ierr .or. dbg) then
                  write(*,2) 'FP too small', i_out, fp(i_out), s% fp_error_limit
                  stop 'eval_fp_ft'
               end if
               exit
               if (ifp_in == 0) ifp_in=i_out
               ifp_out=i_out
               fp_min=min(fp_min,fp(i_out))
            elseif (ifp_in /= 0) then 
               ifp_in=0
               ifp_out=0
               fp_min=1.0d0
            endif
 
            ft(i_out)=max(s% ft_min, min(ft(i_out),1.d0))
            fp(i_out)=max(s% fp_min, min(fp(i_out),1.d0))
            
         end do
      
         deallocate(work, iwork)

         ft(1)=ft(2)
         fp(1)=fp(2)
         ft(nz)=ft(nz-1)
         fp(nz)=fp(nz-1)

      end subroutine eval_fp_ft


      subroutine rad_fcn(n, x, h, y, dydx, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: n, lrpar, lipar
         real(dp), intent(in) :: x, h
         real(dp), intent(inout) :: y(:) ! (n) 
         real(dp), intent(out) :: dydx(:) ! (n)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr ! nonzero means retry with smaller timestep.
         real(dp) :: rov
         ierr = 0
         rov = rpar(1)
         dydx(1)=(6.d0-6.d0*rov*(y(1)+1.d0)-y(1)*(y(1)-1.d0))
      end subroutine rad_fcn


      real(dp) function apot_fcn(rnu, df, lrpar, rpar, lipar, ipar, ierr) result(f)
         ! returns with ierr = 0 if was able to evaluate f and df/dx at x
         ! if df/dx not available, it is okay to set it to 0
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(in) :: rnu
         real(dp), intent(out) :: df
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr
         real(dp) :: wrnu2, a2, a3, dadrn, c1, rfk, drfk
         real(dp) :: etanu,wr,r,xm,a,r_psi
         real(dp), parameter :: c0=1.0d0/3.0d0
         real(dp), parameter :: c2=2.0d0/35.0d0
         real(dp), parameter :: c3=3.0d0*c2
         include 'formats'
         ierr = 0
         c1 = 0.6d0*standard_cgrav         
         etanu = rpar(1)
         wr = rpar(2)
         r = rpar(3)
         xm = rpar(4)
         wrnu2=wr*wr*rnu*rnu/(c1*xm*(2.d0+etanu)) ! (1/3)*da/dr0
         a=rnu*wrnu2 ! E&S, A10  NOTE: typo in paper where it gives r0^2 instead of r0^3.
         a2=a*a
         a3=a2*a
         drfk= pow_cr(max(1.0d-20,1.d0+0.6d0*a2-c2*a3),c0) ! dr_psi/dr0|A, E&S A13
         r_psi=rnu*drfk ! r_psi(r0)
         rfk=r_psi - r ! r_psi(r0) - true_r_psi
         dadrn=wrnu2*a ! (1/3)*a*da/dr0
         drfk=drfk+rnu/(drfk*drfk)*(1.2d0*dadrn-c3*a*dadrn) ! dr_psi/dr0
         ! divide by r to normalize
         f = rfk / r
         df = drfk / r
         if (rfk > 1.d30) f=1.d30
         if (df > 1.d30) df=1.d30
         rpar(5) = a
         
         return
         if (ipar(1) == 425) then
            ipar(2) = ipar(2) + 1
            write(*,2) 'ipar(2)', ipar(2)
            write(*,1) 'rnu', rnu
            write(*,1) 'r_psi', r_psi
            write(*,1) 'r', r
            write(*,1) 'rfk', rfk
            write(*,1) 'drfk', drfk
            write(*,1) 'etanu', etanu
            write(*,1) 'wr', wr
            write(*,1) 'xm', xm
            write(*,1) 'wrnu2', wrnu2
            write(*,1) 'a', a
            write(*,1) 'f', f
            write(*,1) 'df', df
            write(*,*)
         end if
      end function apot_fcn


      real(dp) function psiint(rho, xm, aw, eta, r0_in, r0_out)
         real(dp), intent(in) :: rho, xm, aw, eta, r0_in, r0_out
         psiint=rho*(pow8(r0_out)-pow8(r0_in))/xm
         psiint=psiint*(5.d0+eta)/(2.d0+eta)*0.125d0*aw*aw
      end function psiint


      real(dp) function gpsi(rnu, tegra, kr, a, wr, xm_out, rae)
         ! rnu = r0; tegra = integral in A9; kr is the theta index
         real(dp), intent(in) :: rnu, tegra, a, wr, xm_out
         real(dp), intent(out) :: rae(:)
         integer, intent(in) :: kr
         real(dp) :: r, ri, ri2, wr2, dpdr, dpdthe      
         r= rnu * (1.d0 - a*utheta(kr))
         ri=1.0d0/r
         ri2=ri*ri
         wr2=wr*wr
         rae(kr)=r
         dpdr=-standard_cgrav*xm_out*ri2 +pi4*ri2*ri2*utheta(kr)*tegra +wr2*r*sint2(kr)
            ! dpsi/dr from A9
         dpdthe= (pi4*ri2*ri*tegra + (wr2*r*r))*cost(kr)*sint(kr)
            ! dpsi/dtheta from A9
         gpsi=dsqrt(dpdr*dpdr+(dpdthe*dpdthe*ri2)) !   E&S, A14
      end function gpsi


      subroutine init_rotation(ierr)
         integer, intent(out) :: ierr
         integer :: i
         ierr = 0
         if (have_initialized) return
         dtheta=pi / (2.d0 * (intmax-1))
         dtheta2=0.5d0*dtheta
         do i=1,intmax
            theta(i)=(i-1)*dtheta
            cost(i)=cos_cr(theta(i))
            cost2(i)=cost(i)*cost(i)
            sint(i)=sin_cr(theta(i))
            sint2(i)=sint(i)*sint(i)
            utheta(i)=0.5d0*(3.d0*cost2(i)-1.d0)
         end do
         have_initialized = .true.
      end subroutine init_rotation      
      

      end module hydro_rotation

