! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module hydro_newton_procs
      use hydro_mtx
      
      use star_private_def
      use utils_lib, only: alloc_iounit, free_iounit, &
         is_bad_num, has_bad_num
      use const_def
      
      use num_def
      
      implicit none


      contains


      subroutine set_xscale_info(s, nvar, nz, xscale, ierr)
         type (star_info), pointer :: s   
         integer, intent(in) :: nvar, nz
         real(dp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(out) :: ierr

         integer :: i, j, k, nvar_hydro, i_v
         real(dp), parameter :: xscale_min = 1
         real(dp) :: var_scale, lum_scale, vel_scale, omega_scale
         
         include 'formats'
         
         ierr = 0
         
         if (dbg) write(*, *) 'set_xscale'
         nvar_hydro = s% nvar_hydro
         
         do k=1,nz
            do i=1,nvar
               if (i <= nvar_hydro) then ! structure variable
                  xscale(i,k) = max(xscale_min, abs(s% xh_pre(i,k)))
               else ! abundance variable
                  xscale(i,k) = max(s% xa_scale, s% xa_pre(i-nvar_hydro,k))                  
               end if
            end do
         end do
                  
         contains

         subroutine dump_xscale
            integer :: k, j, k0, k1
            include 'formats'
            !write(*,1) 's% xa_scale', s% xa_scale
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'xscale ' // trim(s% nameofvar(j)), k, xscale(j,k)
               end do
               write(*,*)
            end do
            stop 'set_xscale'      
         end subroutine dump_xscale

      end subroutine set_xscale_info


      subroutine eval_equations( &
            iter, nvar, nz, dx, xscale, equ, lrpar, rpar, lipar, ipar, ierr)
         use hydro_eqns, only: eval_equ
         use star_utils, only: update_time, total_times
         integer, intent(in) :: iter, nvar, nz
         real(dp), pointer, dimension(:,:) :: dx, xscale, equ ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

			integer :: cnt, i, j, k
         type (star_info), pointer :: s   
         integer :: id
         real(dp) :: dt, theta_dt
         
         logical, parameter :: check_bad_nums = .false., &
            convert_ODE_to_DAE_form = .true., &
            skip_partials = .true.
         
         include 'formats'
         
         ierr = 0

         id = ipar(ipar_id)
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return

         if (dbg) write(*, *) 'eval_equations: s% doing_numerical_jacobian', &
            s% doing_numerical_jacobian
         
         dt = rpar(rpar_dt)
         
         if (ipar(ipar_first_call) /= 0) then
            ipar(ipar_first_call) = 0
            if (dbg) write(*, *) 'skip set_newton_vars on call before 1st iter'
         else
            if (dbg) write(*, *) 'call set_newton_vars'
            call set_newton_vars(s, iter, dx, xscale, dt, ierr)
            if (ierr /= 0) then
               if (s% report_ierr) &
                  write(*,2) 'eval_equations: set_newton_vars returned ierr', ierr
               return
            end if
         end if
         
         if (ierr == 0) then
            equ = 0
            if (dbg) write(*, *) 'call eval_equ'
            call eval_equ(s, nvar, skip_partials, convert_ODE_to_DAE_form, xscale, ierr)         
            if (ierr /= 0) then
               if (s% report_ierr) write(*, *) 'eval_equations: eval_equ returned ierr', ierr
               return
            end if
         end if

         if (ierr /= 0 .or. .not. dbg .or. .not. check_bad_nums) return
         
         cnt = 0
         do i=1,nz
            do j=1, nvar
               if (is_bad_num(equ(j, i))) then
                  write(*,3) 'eval_equations: equ has a nan: ', j, i, equ(j, i)
                  cnt = cnt + 1
               end if
               if (abs(equ(j, i)) > 1d100) then
                  write(*,3) 'eval_equations: equ residual too big: ', j, i, equ(j, i)
                  cnt = cnt + 1
               end if
            end do
         end do
         if (cnt > 0) then
            ierr = -1
            return
         end if
         
         write(*,*) 'return from eval_equations'
         
         
         contains


         subroutine dump_eval_equ
            integer :: k, j, k0, k1
            include 'formats'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'dx ' // trim(s% nameofvar(j)), k, dx(j, k)
               end do
               write(*,*)
            end do
            stop 'dump_eval_equ'      
         end subroutine dump_eval_equ

         
      end subroutine eval_equations


      subroutine sizequ( &
            iter, nvar, nz, equ, equ_norm, equ_max, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: iter, nvar, nz
         real(dp), pointer :: equ(:,:) ! (nvar, nz)
         real(dp), intent(out) :: equ_norm, equ_max
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         integer :: j, k, num_terms, n, i_chem1, nvar_hydro, nvar_chem, &
            j_max, k_max, max_loc
         real(dp) :: sumequ, absq, max_energy_resid, avg_energy_resid
         type (star_info), pointer :: s 
         
         include 'formats'  

         if (dbg) write(*, *) 'enter sizequ'
         ierr = 0
         call get_star_ptr(ipar(ipar_id), s, ierr)
         if (ierr /= 0) return
         
         if (s% just_use_energy_resid) then
            max_energy_resid = maxval(abs(s% E_residual(1:nz)))
            avg_energy_resid = sum(s% E_residual(1:nz))/nz
            equ_max = max_energy_resid
            equ_norm = avg_energy_resid
            return
         end if
         
         nvar_hydro = s% nvar_hydro
         nvar_chem = s% nvar_chem
         n = nz
         num_terms = 0
         sumequ = 0
         equ_max = 0
         j_max = 0
         k_max = 0
         if (s% do_struct_hydro .or. s% do_struct_thermo) then
            if (s% do_burn .or. s% do_mix) then
               num_terms = num_terms + nvar*nz
               do k = 1, nz
                  do j = 1, nvar
                     absq = abs(equ(j,k))
                     sumequ = sumequ + absq
                     if (absq > equ_max) then
                        equ_max = absq
                        j_max = j
                        k_max = k
                     end if
                  end do
               end do
            else
               num_terms = num_terms + n*nvar_hydro
               do k = 1, nz
                  do j = 1, nvar_hydro
                     absq = abs(equ(j,k))
                     !write(*,3) 'equ(j,k)', j, k, equ(j,k)
                     sumequ = sumequ + absq
                     if (is_bad_num(sumequ)) then
                        if (dbg) then
                           write(*,3) trim(s% nameofequ(j)) // ' sumequ', j, k, sumequ
                           stop 'sizeq'
                        end if
                        ierr = -1
                        if (s% report_ierr) &
                           write(*,3) 'bad equ(j,k) ' // trim(s% nameofequ(j)), &
                              j, k, equ(j,k)
                        return
                     end if
                     if (absq > equ_max) then
                        equ_max = absq
                        j_max = j
                        k_max = k
                     end if
                  end do
               end do
            end if
         end if
         if (s% do_burn .or. s% do_mix) then
            i_chem1 = s% i_chem1
            num_terms = num_terms + nvar_chem*nz
            do k = 1, nz
               do j = i_chem1, nvar
                  absq = abs(equ(j,k))
                  sumequ = sumequ + absq
                  if (absq > equ_max) then
                     equ_max = absq
                     j_max = j
                     k_max = k
                  end if
               end do
            end do
         end if

         equ_norm = sumequ/num_terms
         if (dbg) write(*,2) trim(s% nameofequ(j_max)) // ' equ_max', k_max, equ_max, equ_norm

         if (dbg) write(*, *) 'exit sizequ'

         if (.false. .and. equ_max > 1d4) then
            write(*,2) 'equ_norm', num_terms, equ_norm
            write(*,2) trim(s% nameofequ(j_max)) // ' equ_max', k_max, equ_max, equ_norm
            stop 'sizequ'
            call dump_equ
         end if
         
         
         contains


         subroutine dump_equ
            integer :: k, j, k0, k1
            include 'formats'
            do k=1,s% nz
               do j=1,nvar
                  write(*,4) 'equ ' // trim(s% nameofequ(j)), &
                     k, iter, s% model_number, equ(j, k)
               end do
               write(*,*)
            end do
         end subroutine dump_equ


      end subroutine sizequ


      subroutine sizeB( &
            iter, nvar, nz, B, xscale, max_correction, correction_norm, &
            lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: iter, nvar, nz
         real(dp), pointer, dimension(:,:) :: B, xscale ! (nvar, nz)
         real(dp), intent(out) :: correction_norm ! a measure of the average correction
         real(dp), intent(out) :: max_correction ! magnitude of the max correction
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         integer :: k, i, num_terms, j, max_zone, max_var, n, nvar_hydro, &
            skip1, skip2, skip3, jmax, num_xa_terms
         real(dp) :: abs_corr, sum_corr, sum_xa_corr, x_limit, max_abs_corr_for_k
         type (star_info), pointer :: s
         
         logical, parameter :: dbg = .false.
         logical, parameter :: check_for_bad_nums = .true.
         logical, parameter :: save_max_abs_corr_for_k = .true.
         
         include 'formats'
         
         if (dbg) write(*, *) 'enter sizeB'

         ierr = 0
         call get_star_ptr(ipar(ipar_id), s, ierr)
         if (ierr /= 0) then
            write(*,*) 'bad id for sizeB'
            return
         end if

         n = nz
         nvar_hydro = s% nvar_hydro
         
         if (s% include_L_in_error_est) then
            skip1 = 0
         else
            skip1 = s% i_lum
         end if
         
         if (s% include_v_in_error_est) then
            skip2 = 0
         else
            skip2 = s% i_v
         end if
         
         skip3 = 0

         max_zone = 0
         max_var = 0
         num_terms = 0
         num_xa_terms = 0
         sum_corr = 0
         sum_xa_corr = 0
         max_correction = 0
         x_limit = s% correction_xa_limit
         do k = 1, nz
            max_abs_corr_for_k = 0
            
            if (s% do_struct_hydro .or. s% do_struct_thermo) then
               if (s% do_burn .or. s% do_mix) then
                  jmax = nvar
               else
                  jmax = nvar_hydro
               end if
               do j = 1, jmax
                  if (j == skip1 .or. &
                      j == skip2 .or. &
                      j == skip3) cycle
                  if (check_for_bad_nums) then
                     if (is_bad_num(B(j,k))) then
                        ierr = -1
                        if (.not. dbg) return
                        write(*,2) 'bad num for correction ' // s% nameofvar(j), k, B(j,k)
                        stop 'sizeB'
                        cycle
                     end if
                  end if
                  if (j > nvar_hydro) then
                     if (s% xa_pre(j-nvar_hydro,k) < x_limit) cycle
                  end if
                  
                  
                  if (j == s% i_lnT .and. &
                        abs(s% gam_start(k) - s% sizeB_gam0) < s% size_del_gam) cycle 
                  ! when going from liquid to solid, have trouble converging
                  ! e.g., can jump back and forth from gam = 165ish to 175ish
                  ! for each newton iteration -- and that can cause lnT to jump around
                  ! leading to failure to converge.
                  
                  
                  abs_corr = abs(B(j,k))
                  if (is_bad_num(abs_corr)) then
                     ierr = -1
                     if (.not. dbg) return
                     write(*,3) 'B(j,k)', j, k, B(j,k)
                     stop 'sizeB'
                  end if                  
                  if (abs_corr > max_abs_corr_for_k) max_abs_corr_for_k = abs_corr
                  if (abs_corr > max_correction) then
                     max_correction = abs_corr
                     max_zone = k
                     max_var = j
                  end if
                  if (j > nvar_hydro) then
                     num_xa_terms = num_xa_terms + 1
                     sum_xa_corr = sum_xa_corr + abs_corr
                  else
                     num_terms = num_terms + 1
                     sum_corr = sum_corr + abs_corr
                  end if
               end do
               if (num_xa_terms > 0) then
                  num_terms = num_terms + 1
                  sum_corr = sum_corr + sum_xa_corr/num_xa_terms
               end if               
            else if (s% do_burn .or. s% do_mix) then
               do j = s% i_chem1, nvar
                  i = j - s% nvar_hydro
                  if (check_for_bad_nums) then
                     if (is_bad_num(B(j,k))) then
                        ierr = -1
                        if (.not. dbg) return
                        write(*,3) 'chem B(j,k)', j, k, B(j,k)
                        stop 'sizeB'
                        cycle
                     end if
                  end if
                  ! recall that correction dx = B*xscale, so B is a relative correction
                  if (s% xa_pre(i,k) >= x_limit) then
                     abs_corr = abs(B(j,k))
                     if (abs_corr > max_abs_corr_for_k) max_abs_corr_for_k = abs_corr
                     if (abs_corr > max_correction) then
                        max_correction = abs_corr
                        max_zone = k
                        max_var = j
                     end if
                     sum_corr = sum_corr + abs_corr
                     num_terms = num_terms + 1
                  end if
               end do
            end if
         end do
         
         if (is_bad_num(sum_corr)) then
            ierr = -1
            if (.not. dbg) return
            write(*,*) 'sum_corr', sum_corr
            stop 'sizeB'
         end if
         
         if (s% split_mixing_choice < 0 .and. &
                  (.not. s% split_mix_do_burn)) then
            do k=1,nz
               if (is_bad_num(s% max_burn_correction(k))) then
                  ierr = -1
                  return
                  write(*,3) 'bad max_burn_correction: zone iter corr', &
                     k, iter, s% max_burn_correction(k)
                  stop 'sizeB'
               end if
               if (s% max_burn_correction(k) > max_correction) then
                  max_correction = s% max_burn_correction(k)
                  max_zone = k
                  max_var = -1
               end if
               sum_corr = sum_corr + s% avg_burn_correction(k)
               if (is_bad_num(s% avg_burn_correction(k))) then
                  ierr = -1
                  return
                  write(*,2) 's% avg_burn_correction(k)', k, s% avg_burn_correction(k)
                  stop 'sizeB'
               end if
               num_terms = num_terms + 1
            end do
            if (max_var == -1) then
               ierr = -1
               return
               write(*,3) 'max correction is from abundances: zone iter corr', &
                  max_zone, iter, max_correction
               max_var = 0
               stop 'sizeB'
            end if
         end if
         
         correction_norm = sum_corr/num_terms  !sqrt(sum_corr/num_terms)
         if (dbg) then
            write(*,2) 'sizeB: iter, correction_norm, max_correction', &
               iter, correction_norm, max_correction
            if (max_correction > 1d50 .or. is_bad_num(correction_norm)) then
               call show_stuff
               stop 'sizeB'
            end if
         end if

         if (.false. .and. s% hydro_call_number == 2 .and. iter == 1) then
            write(*,2) 'correction_norm', num_terms, correction_norm
            call dump_B
         end if
         
         if (s% hydro_show_correction_info) call show_stuff
         
         abs_corr = max_correction
         
         s% abs_max_corr2 = s% abs_max_corr1; s% abs_max_corr1 = abs_corr
         s% max_var2 = s% max_var1; s% max_var1 = max_var
         s% max_zone2 = s% max_zone1; s% max_zone1 = max_zone
         
         if (ierr /= 0) stop 'ierr in sizeB'
         
         if (iter < 3) return
         ! check for flailing
         if ( &
             abs_corr > s% tol_max_correction .and. &
             abs_corr > s% abs_max_corr1 .and. s% abs_max_corr1 > s% abs_max_corr2 .and. &
             max_zone == s% max_zone1 .and. s% max_zone1 == s% max_zone2 .and. &
             max_var == s% max_var1 .and. s% max_var1 == s% max_var2) then
            if (s% hydro_show_correction_info) then
               write(*,*) 'give up because diverging'
            end if
            max_correction = 1d99
         end if
         
         
         return
         
         
         if (s% hydro_call_number < 2) return
         
         call show_stuff
         stop 'sizeB'
         
         
         contains
         
         
         subroutine show_stuff
            include 'formats'
            if (iter == 1) then
               write(*,*)
               write(*,'(a15,2a7,2a8,99a20)') &
                  'corrections', 'model', 'iter', 'var', 'zone', &
                  'corr norm', 'max corr', 'xscale', 'max corr*xscale', &
                  'mass loc', 'log dt/yr', 'gam'
            end if
            write(*,'(15x,2i7,a8,i8,4e20.10,99f20.10)') &
               s% model_number, iter, trim(s% nameofvar(max_var)), max_zone, &
               correction_norm, &
               B(max_var,max_zone), &
               xscale(max_var,max_zone), &
               xscale(max_var,max_zone)*B(max_var,max_zone), &
               s% m(max_zone)/Msun, log10_cr(s% dt/secyer), s% gam(max_zone)
         end subroutine show_stuff


         subroutine dump_B
            integer :: k, j, k0, k1
            include 'formats'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'B ' // trim(s% nameofequ(j)), k, B(j, k)
               end do
               write(*,*)
            end do
            stop 'dump_equ'      
         end subroutine dump_B


      end subroutine sizeB


      ! the proposed change to dx is B*xscale*correction_factor
      ! edit correction_factor and/or B as necessary so that the new dx will be valid.
      ! set ierr nonzero if things are beyond repair.
      subroutine Bdomain( &
            iter, nvar, nz, B, dx, xscale, correction_factor, lrpar, rpar, lipar, ipar, ierr)
         use const_def, only: dp
         use chem_def, only: chem_isos
         use star_utils, only: current_min_xa_hard_limit
         integer, intent(in) :: iter, nvar, nz
         real(dp), pointer, dimension(:,:) :: dx, xscale, B ! (nvar, nz)
         real(dp), intent(inout) :: correction_factor
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr
         integer :: i, j, k, nvar_hydro, species, bad_j, bad_k
         real(dp) :: alpha, min_alpha, new_xa, old_xa, dxa, eps, min_xa_hard_limit
         type (star_info), pointer :: s   
         integer :: id
         include 'formats'
         ierr = 0         
         id = ipar(ipar_id)
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         if (.not. s% do_newton_damping_for_neg_xa) return
         if (.not. (s% do_mix .or. s% do_burn)) return
         
         nvar_hydro = s% nvar_hydro
         species = s% species
         min_alpha = 1d0
         min_xa_hard_limit = current_min_xa_hard_limit(s)
         eps = -0.5d0*min_xa_hard_limit ! allow xa to be this much below 0d0
         bad_j = 0
         bad_k = 0
         do k=1,nz
            do j=1,species
               i = j + nvar_hydro
               old_xa = s% xa_pre(j,k) + dx(i,k)
               if (old_xa <= 1d-90) cycle
               dxa = B(i,k)*xscale(i,k)*correction_factor
               new_xa = old_xa + dxa
               if (new_xa >= 0d0) cycle
               alpha = -(old_xa + eps)/dxa
               ! so dxa*alpha = -old_xa - eps,
               ! and therefore old_xa + alpha*dxa = -eps = 0.5*min_xa_hard_limit
               if (alpha < min_alpha) then
                  min_alpha = alpha
                  bad_j = j
                  bad_k = k
               end if 
            end do
         end do
         min_alpha = max(min_alpha, s% corr_coeff_limit)
         correction_factor = min_alpha*correction_factor
         if (s% trace_newton_damping .and. min_alpha < 1d0) then
            write(*,4) 'newton damping to avoid negative mass fractions: ' // &
               trim(chem_isos% name(s% chem_id(bad_j))), bad_k, &
               s% model_number, iter, min_alpha
         end if
         
         
      end subroutine Bdomain


      subroutine inspectB(iter, nvar, nz, dx, B, xscale, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: iter, nvar, nz
         real(dp), pointer, dimension(:,:) :: dx, B, xscale ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         type (star_info), pointer :: s   
         integer :: id
         integer, parameter :: inspectB_iter_stop = -1
         
         id = ipar(ipar_id)
         
         if (dbg) write(*, *) 'inspectB', iter
         ierr = 0

         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         !if (s% hydro_call_number == 3 .and. iter == 1) call dumpB
         
         if (.not. s% hydro_inspectB_flag) return
         
         if (s% hydro_call_number /= s% hydro_dump_call_number) return
         
         call write_solve_logs(s, iter, dx, B, xscale, ierr)

         if (iter == inspectB_iter_stop) then
            stop 'debug: inspectB'
         end if
         
         
         contains


         subroutine dumpB
            integer :: k, j, k0, k1
            include 'formats'
            do k=1,s% nz
               do j=1,nvar
                  write(*,2) 'B ' // trim(s% nameofvar(j)), k, B(j, k)
                  write(*,2) 'xscale ' // trim(s% nameofvar(j)), k, xscale(j, k)
                  write(*,2) 'dx ' // trim(s% nameofvar(j)), k, dx(j, k)
               end do
               write(*,*)
            end do
            stop 'dumpB'      
         end subroutine dumpB
         
      end subroutine inspectB
         
         
      subroutine write_solve_logs(s, iter, dx, B, xscale, ierr)
         use star_utils, only:std_write_internals_to_file
         use utils_lib, only:alloc_iounit, free_iounit, append_data
         type (star_info), pointer :: s   
         integer, intent(in) :: iter
         real(dp), pointer, dimension(:,:) :: dx, B, xscale ! (nvar, nz)
         integer, intent(out) :: ierr

         integer :: i, j, iounit, i_delta, nvar, nz
         character (len=strlen) :: logfilename, log_format_str, data_dir, filename
         logical :: initialize_log_file
         real(dp), pointer :: tmp(:), tmp2(:)

         tmp => null(); tmp2 => null()

         nz = s% nz
         if (s% do_burn .or. s% do_mix) then
            nvar = s% nvar
         else
            nvar = s% nvar_hydro
         end if
         
         !call std_write_internals_to_file(id, iter)

         data_dir = 'plot_data/solve_logs'

         if (iter == 1) then
            iounit = alloc_iounit(ierr); if (ierr /= 0) return
            write(filename, '(2a)') trim(data_dir), '/names.data'
            open(unit = iounit, file = trim(filename), iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'failed to open ', trim(filename)
               ierr = 0
               call free_iounit(iounit)
               return
            end if
            initialize_log_file = .true.
            write(*,*) 'allocate debug_previous_data'
            allocate(debug_previous_data(nz, nvar), stat=ierr)  
               ! we don't deallocate this, but it is only for debugging
               ! so isn't a serious memory leak
            if (ierr /= 0) then
               call free_iounit(iounit)
               return
            end if
         else
            initialize_log_file = .false.
         end if

         log_format_str = '(99999e20.12)'
         
         i_delta = 0
         
         allocate(tmp(nz), tmp2(nz), stat=ierr)
         if (ierr /= 0) then
            call free_iounit(iounit)
            return
         end if
         
         do j = 1, nvar
            tmp(1:nz) = B(j, 1:nz)
            call write_one(nz, tmp, 'corr_' // s% nameofvar(j))
         end do

         do j = 1, nvar
            tmp(1:nz) = B(j, 1:nz)
            call write_delta(nz, tmp, 'corr_' // s% nameofvar(j))
         end do

         do j=1, nvar
            tmp = dx(j,1:nz)
            call write_one(nz, tmp, 'd' // s% nameofvar(j))
         end do

         do j=1, nvar
            i = j - s% nvar_hydro
            if (i > 0) then
               tmp = s% xa_pre(i,1:nz) + dx(j,1:nz)
            else
               tmp = s% xh_pre(j,1:nz) + dx(j,1:nz)
            end if
            call write_one(nz, tmp, s% nameofvar(j))
         end do
                  
         !call write_one(nz, s% mlt_D, 'mlt_D')
         !call write_one(nz, s% gradT, 'gradT')
         !call write_one(nz, s% eps_grav, 'eps_grav')
         !call write_one(nz, s% eps_grav_dm_term_const_q, 'eps_grav_dm_term_const_q')
         !call write_one(nz, s% eps_grav_dt_term_const_q, 'eps_grav_dt_term_const_q')
         !call write_one(nz, s% dlnd_dt_const_q, 'dlnd_dt_const_q')
         !call write_one(nz, s% dlnT_dt_const_q, 'dlnT_dt_const_q')

         deallocate(tmp, tmp2) ! for write_solve_logs

         if (iter == 1) then
            close(iounit)
            call free_iounit(iounit)
         else
            initialize_log_file = .false.
         end if
         
         iounit = alloc_iounit(ierr); if (ierr /= 0) return
         write(filename, '(2a)') trim(data_dir), '/size.data'
         open(unit = iounit, file = trim(filename), iostat=ierr)
         if (ierr /= 0) then
            write(*, *) 'failed to open ', trim(filename)
            ierr = 0
            return
         end if
         write(iounit, *) nz, iter
         close(iounit)
         call free_iounit(iounit)

         contains

         subroutine write_delta(nz, vals, name)
            integer, intent(in) :: nz
            real(dp), intent(in) :: vals(:) ! (nz)
            character (len=*), intent(in) :: name
            real(dp) :: tmp(nz)
            i_delta = i_delta + 1
            if (iter == 1) then
               tmp = 0
            else if (i_delta <= nvar) then
               tmp = vals - debug_previous_data(:,i_delta)
            end if
            call write_one(nz, tmp, 'delta_' // name)          
            debug_previous_data(:,i_delta) = vals
         end subroutine write_delta

         subroutine write_one(nz, vals, varname)
            integer, intent(in) :: nz
            real(dp), intent(in) :: vals(:) ! (nz)
            character (len=*), intent(in) :: varname
            write(logfilename, '(4a)') trim(data_dir), '/', trim(varname), '.log'
            call append_data(nz, vals(1:nz), logfilename, log_format_str, initialize_log_file, ierr)
            if (ierr /= 0) then
               write(*, *) 'failed in append_data for ', trim(logfilename)
               ierr = 0
               return
            end if
            if (iter == 1) write(iounit, *) trim(varname)
         end subroutine write_one

      end subroutine write_solve_logs
      
      
      ! about to declare victory... but may want to do another iteration
      ! 1 means force another iteration
      ! 0 means don't need to force another
      ! -1 means failure. newton returns with non-convergence.
      integer function force_another_iteration(iter, itermin, lrpar, rpar, lipar, ipar)
         use hydro_mtx, only: ipar_id
         integer, intent(in) :: iter ! have finished this many iterations and have converged
         integer, intent(in) :: itermin ! this is the requested minimum.  iter may be < itermin.
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)

         type (star_info), pointer :: s   
         integer :: id, ierr, k, res
         
         include 'formats'
         
         if (iter < itermin) then
            force_another_iteration = 1
            return
         end if
         force_another_iteration = 0

         id = ipar(ipar_id)
         ierr = 0

         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then ! DISASTER
            force_another_iteration = -1
            return
         end if
                  
         !if (s% L(1) <= 0) then 
            ! probably won't help, but give it a try anyway
         !   force_another_iteration = 1
         !   return
         !end if
         
         if (s% k_below_just_added > 1 .and. &
               (.not. s% zero_eps_grav_in_just_added_material) .and. &
               s% num_surf_revisions < s% max_num_surf_revisions .and. &
               abs(s% lnS(1) - s% surf_lnS) > &
                  s% max_abs_rel_change_surf_lnS*max(s% lnS(1),s% surf_lnS)) then
!            write(*,'(a)') &
!               'force another newton iteration because of change in surface entropy'
            if (s% trace_force_another_iteration) &
               write(*,2) 'force extra iter to fix predicted final surface lnS', iter, &
                  abs(s% lnS(1) - s% surf_lnS)/max(s% lnS(1),s% surf_lnS), &
                  s% max_abs_rel_change_surf_lnS
            s% surf_lnT = s% lnT(1)    
            s% surf_E = s% energy(1)    
            s% surf_lnR = s% lnR(1)
            if (s% i_lnd /= 0) s% surf_lnd = s% lnd(1)
            if (s% i_lnPgas /= 0) s% surf_lnPgas = s% lnPgas(1)    
            if (s% i_v /= 0) s% surf_v = s% v(1)
            s% surf_lnS = s% lnS(1)
            s% num_surf_revisions = s% num_surf_revisions + 1
            force_another_iteration = 1
            s% used_extra_iter_in_newton_for_accretion = .true.
            return
         end if
         if (s% k_below_just_added > 1 .and. s% num_surf_revisions > 0) then
!            write(*,'(a)') 'okay: now have okay estimate of final surface entropy'
!            write(*,2) 'abs rel diff of predicted vs actual final lnS', iter, &
!               abs(s% lnS(1) - s% surf_lnS)/max(s% lnS(1),s% surf_lnS), &
!               s% max_abs_rel_change_surf_lnS
!            write(*,*)
         end if
      end function force_another_iteration
      

      end module hydro_newton_procs

