! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module hydro_mtx
      
      use star_private_def
      use utils_lib, only: is_bad_num, has_bad_num
      use const_def
      
      use num_def
      
      implicit none
      
      logical, parameter :: dbg = .false. 
      
      integer, parameter :: ipar_id = 1
      integer, parameter :: ipar_first_call = 2
      integer, parameter :: hydro_lipar = ipar_first_call
      
      integer, parameter :: rpar_dt = 1
      integer, parameter :: hydro_lrpar = 1

      
      ! for inspectB debugging
      real(dp), pointer :: debug_previous_data(:,:)


      contains
      

      subroutine set_newton_vars(s, iter, dx, xscale, dt, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: iter
         real(dp), dimension(:,:) :: dx, xscale
         real(dp), intent(in) :: dt
         integer, intent(out) :: ierr
         call set_vars_for_solver(s, 1, s% nz, iter, dx, xscale, ierr)
      end subroutine set_newton_vars


      subroutine set_vars_for_solver(s, nzlo, nzhi, iter, dx, xscale, ierr)
         use utils_lib, only: has_bad_num, is_bad_num
         use const_def, only: secyer, Msun, Lsun, Rsun
         use hydro_vars, only: set_hydro_vars, set_rmid_and_Amid
         use mix_info, only: get_convection_sigmas
         use chem_def
         use star_utils
         type (star_info), pointer :: s
         integer, intent(in) :: nzlo, nzhi, iter
         real(dp), dimension(:,:) :: dx, xscale
         integer, intent(out) :: ierr

         logical, parameter :: &
            skip_basic_vars = .true., &
            skip_micro_vars = .false., &
            skip_kap = .false., &
            skip_neu = .false., &
            skip_net = .false., &
            skip_grads = .true., &
            skip_rotation = .true., &
            skip_m_grav_and_grav = .true., &
            skip_brunt = .true., &
            skip_other_cgrav = .true.
         logical :: do_chem, do_struct, try_again, &
            skip_mixing_info, skip_mlt, do_edit_lnR
         integer :: i, j, k, kk, klo, khi, &
            i_lnd, i_lnPgas, i_lnT, i_E, i_lnR, i_lum, i_v, &
            fe56, nvar, nvar_chem, species, i_chem1, nz, nvar_hydro
         real(dp), dimension(:, :), pointer :: xh_pre, xa_pre
         integer :: op_err, kbad, &
            cnt, max_fixes, loc(2), k_lo, k_hi, k_const_mass
         real(dp) :: twoGmrc2, r2, xavg, &
            dq_sum, xa_err_norm, d_dxdt_dx, min_xa_hard_limit, sum_xa_hard_limit
         
         include 'formats'

         if (dbg) write(*, *) 'set_vars_for_solver'

         ierr = 0
                  
         do_chem = (s% do_burn .or. s% do_mix)
         do_struct = (s% do_struct_hydro .or. s% do_struct_thermo)
         
         skip_mixing_info = .not. s% redo_mix_info_at_each_iteration
         
         skip_mlt = (s% qmax_zero_non_radiative_luminosity >= 1d0) .and. (.not. s% do_mix)
         
         if (dbg .and. .not. skip_mixing_info) write(*,2) 'redo mix info iter', iter
         
         min_xa_hard_limit = current_min_xa_hard_limit(s)
         sum_xa_hard_limit = current_sum_xa_hard_limit(s)
         
         xh_pre => s% xh_pre
         xa_pre => s% xa_pre
         
         i_lnd = s% i_lnd
         i_lnPgas = s% i_lnPgas
         i_lnT = s% i_lnT
         i_E = s% i_E
         i_lnR = s% i_lnR
         i_lum = s% i_lum
         i_v = s% i_v
         
         nz = s% nz
         nvar_chem = s% nvar_chem
         species = s% species
         nvar_hydro = s% nvar_hydro
         i_chem1 = s% i_chem1

         if (s% trace_k > 0 .and. s% trace_k <= nz) then 
            k = s% trace_k
            if (i_lnd /= 0) write(*,4) 'set_vars_for_solver: lnd', &
               k, s% newton_iter, s% ebdf_stage, &
               xh_pre(i_lnd,k), dx(i_lnd,k)
         end if

         fe56 = s% net_iso(ife56)
         if (fe56 /= 0) fe56 = i_chem1+fe56-1
         
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = nvar_hydro
         end if

         d_dxdt_dx = s% dVARdot_dVAR
      
         if (do_chem) then
            do k=1,nz
               do j=1,species
                  s% xa_sub_xa_pre(j,k) = dx(j+nvar_hydro,k)
                  s% xa(j,k) = xa_pre(j,k) + dx(j+nvar_hydro,k)
               end do
            end do
            max_fixes = 0 !5
            do cnt=1,max_fixes
               loc = minloc(s% xa(1:species,1:nz))
               j = loc(1)
               k = loc(2)
               if (s% xa(j,k) >= 1d-3*min_xa_hard_limit) exit ! too good to fix
               if (s% xa(j,k) < min_xa_hard_limit) then
                  if (s% report_ierr) then
                     khi = nz
                     do kk=k+1,nz
                        if (s% xa(j,kk) < min_xa_hard_limit) cycle
                        khi = kk-1; exit
                     end do
                     klo = 1
                     do kk=k-1,1,-1
                        if (s% xa(j,kk) < min_xa_hard_limit) cycle
                        klo = kk+1; exit
                     end do
                     do k=klo,khi
                        write(*,2) &
                           'negative ' // trim(chem_isos% name(s% chem_id(j))), &
                           k, s% xa(j,k), xa_pre(j,k), dx(nvar_hydro+j,k), s% m(k)/Msun
                     end do
                  end if
                  ierr = -1
                  return
                  exit ! too bad to fix
               end if
               if (k == 1) then  
                  k_lo = 1; k_hi = 2
               else if (k == nz) then
                  k_lo = nz-1; k_hi = nz
               else if (s% sig(k) > s% sig(k+1)) then
                  k_lo = k-1; k_hi = k
               else if (s% sig(k+1) > 0) then
                  k_lo = k; k_hi = k+1
               else
                  exit
               end if
               try_again = .true.
               do while (try_again .and. sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi)) < 0)
                  try_again = .false.
                  if (k_lo > 1) then
                     if (s% sig(k_lo) > 0) then
                        k_lo = k_lo-1
                        try_again = .true.
                     end if
                  end if
                  if (k_hi < nz) then
                     if (s% sig(k_hi+1) > 0) then
                        k_hi = k_hi+1
                        try_again = .true.
                     end if
                  end if
               end do
               !write(*,3) 'no extend', k_lo, k_hi
               if (.not. try_again) exit
               dq_sum = sum(s% dq(k_lo:k_hi))
               if (s% report_ierr) then
                  write(*,5) 'fix xa(j,k_lo:k_hi)', j, k_lo, k, k_hi, s% xa(j,k), &
                     sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi))/dq_sum, dq_sum
                  !stop
                end if
               do j=1,species
                  xavg = sum(s% xa(j,k_lo:k_hi)*s% dq(k_lo:k_hi))/dq_sum
                  do kk=k_lo,k_hi
                     s% xa(j,kk) = xavg
                  end do
               end do
            end do
         end if

         kbad = 0
!$OMP PARALLEL DO PRIVATE(k,op_err)
         do k=1,nz
            if (ierr /= 0) cycle
            op_err = 0
            call set1(k,.false.,op_err)
            if (op_err /= 0) then
               kbad = k; ierr = op_err
            end if
         end do
!$OMP END PARALLEL DO
         if (ierr /= 0) then
            if (s% report_ierr) then
               do k=1,nz ! report the errors sequentially
                  call set1(k,.true.,op_err)
               end do
               write(*,4) 'set_vars_for_solver failed: model, k, nz', &
                  s% model_number, kbad, nz
               !stop 'set_vars_for_solver'
            end if            
            return
         end if
         
         if (s% L_flag) then
            do k=1,nz
               if (is_bad_num(s% L(k))) then
                  if (s% report_ierr) write(*,2) 'set_vars_for_solver L', k, s% L(k), &
                     xh_pre(i_lum,k) + dx(i_lum,k), &
                     xh_pre(i_lum,k), dx(i_lum,k)
                  ierr = -1
                  return               
                  stop
               end if
            end do
         end if
         
         if (ierr /= 0) then
            if (s% report_ierr) then
            
               do k=1,nz
                  if (abs(1d0 - sum(s% xa(:,k))) > 1d-3) then
                     write(*,2) 'set_vars_for_solver: bad xa sum', k, &
                        sum(s% xa(:,k)), sum(xa_pre(:,k)), sum(dx(i_chem1:nvar,k))
                     write(*,'(51x,a)') 'xa, xa_pre+dx, xa_pre, dx'
                     do j=1,species
                        write(*,2) trim(chem_isos% name(s% chem_id(j))), k, &
                           s% xa(j,k), xa_pre(j,k) + dx(i_chem1-1+j,k), &
                           xa_pre(j,k), dx(i_chem1-1+j,k)
                     end do
                     
                     exit
                     
                  end if
               end do
               write(*,*)
               
            end if
            return
         end if
         
         if (do_struct) then
            do_edit_lnR = .not. (s% doing_numerical_jacobian .or. s% ebdf_order > 0)
            if (do_edit_lnR) then
               if (s% lnPgas_flag) then
                  call edit_lnR_for_lnPgas(s, xh_pre, dx)
               else
                  call edit_lnR(s, xh_pre, dx)
               end if
            end if
!$OMP PARALLEL DO PRIVATE(k,twoGmrc2)
            do k=1,nz
               if (do_edit_lnR) s% r(k) = exp_cr(s% lnR(k))
               call set_rv_info(s,k)
               ! note: m_grav is held constant during newton iterations
               s% grav(k) = s% cgrav(k)*s% m_grav(k)/(s% r(k)*s% r(k))
               if (s% use_gr_factors) then ! GR gravity factor = 1/sqrt(1-2Gm/(rc^2))
                  twoGmrc2 = 2*s% cgrav(k)*s% m_grav(k)/(s% r(k)*clight*clight)
                  s% grav(k) = s% grav(k)/sqrt(1d0 - twoGmrc2)
               end if
            end do    
!$OMP END PARALLEL DO
         end if

         call set_rmid_and_Amid(s, 1, nz, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'set_rmid_and_Amid returned ierr', ierr
            return
         end if

         if (dbg) write(*, *) 'call set_hydro_vars'
         call set_hydro_vars( &
            s, 1, nz, skip_basic_vars, skip_micro_vars, &
            skip_m_grav_and_grav, skip_net, skip_neu, skip_kap, &
            skip_grads, skip_rotation, skip_brunt, skip_other_cgrav, &
            skip_mixing_info, skip_mlt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,2) 'set_hydro_vars returned ierr', ierr
            return
         end if

         if (s% rotation_flag) then 
            ! change i_rot to match new r, omega to match new i_rot
            call set_i_rot(s)
            call set_omega(s, 'hydro_mtx')            
         end if
         
         if (s% redo_mix_info_at_each_iteration) then
            call get_convection_sigmas(s, s% dt, ierr)
            if (ierr /= 0) then
               if (s% report_ierr) &
                  write(*,2) 'get_convection_sigmas returned ierr', ierr
               return
            end if
         end if

         
         contains
         
         
         subroutine set1(k,report,ierr)
            use chem_def, only: chem_isos
            integer, intent(in) :: k
            logical, intent(in) :: report
            integer, intent(out) :: ierr
            
            real(dp) :: x(nvar)
            ! setting x = xh_pre + dx is necessary because of numerical issues.
            ! we want to ensure that we've calculated the variables using exactly the
            ! same values for x as will be returned as the final result.
            real(dp) :: twoGmrc2, r2, sum_xa
            integer :: j, i
            real(dp) :: del_t, starting_value, alfa, beta, v, theta
            
            logical, parameter :: check_for_bad_nums = .false.
            
            include 'formats'
            ierr = 0
            v = 0
            
            if (do_struct) then
               
               do j=1,nvar_hydro
                  x(j) = xh_pre(j,k) + dx(j,k)
               end do
               
               if (i_lnT /= 0) then
               
                  s% lnT(k) = x(i_lnT)
                  s% T(k) = exp_cr(s% lnT(k))
                  if (check_for_bad_nums .and. is_bad_num(s% T(k))) then
                     if (report) write(*,2) 'bad num T', k, s% T(k)
                     ierr = -1
                  end if
                  if (abs(s% lnT(k) - s% lnT_start(k)) > &
                        ln10*s% hydro_mtx_max_allowed_abs_dlogT .and. &
                       s% min_logT_for_hydro_mtx_max_allowed < &
                        ln10*min(s% lnT(k),s% lnT_start(k))) then
                     if (s% report_ierr) &
                        write(*,4) 'hydro_mtx: change too large, dlogT, logT, logT_start', &
                           s% model_number, k, iter, &
                           (s% lnT(k) - s% lnT_start(k))/ln10, &
                           s% lnT(k)/ln10, s% lnT_start(k)/ln10
                     ierr = -1
                     return
                  end if
                  if (s% lnT(k) > ln10*s% hydro_mtx_max_allowed_logT .and. &
                       s% min_logT_for_hydro_mtx_max_allowed < &
                        ln10*min(s% lnT(k),s% lnT_start(k))) then
                     if (s% report_ierr) &
                        write(*,4) 'hydro_mtx: logT too large', &
                           s% model_number, k, iter, &
                           s% lnT(k)/ln10, s% lnT_start(k)/ln10
                     ierr = -1
                     return
                  end if
                  
               else if (i_E /= 0) then
               
                  s% energy(k) = x(i_E)
                  s% lnE(k) = log_cr(s% energy(k))
                  if (check_for_bad_nums .and. is_bad_num(s% lnE(k))) then
                     if (report) write(*,2) 'bad num lnE', k, s% lnE(k)
                     ierr = -1
                  end if

               else
                  write(*,*) 'must have either i_lnT or i_E nonzero'
                  stop 1
               end if
               
               if (i_lum /= 0) then
                  s% L(k) = x(i_lum)
                  if (check_for_bad_nums .and. is_bad_num(s% L(k))) then
                     if (report) write(*,2) 'bad num L', k, s% L(k)
                     ierr = -1
                  end if
               end if

               if (s% do_struct_hydro) then
         
                  if (i_v /= 0) then
                     s% v(k) = x(i_v)
                     if (check_for_bad_nums .and. is_bad_num(s% v(k))) then
                        if (report) write(*,2) 'bad num v', k, s% v(k)
                        ierr = -1
                     end if
                  end if
                  
                  s% lnR(k) = x(i_lnR)
                  if (check_for_bad_nums .and. is_bad_num(s% lnR(k))) then
                     if (report) write(*,2) 'bad num lnR', k, s% lnR(k)
                     ierr = -1
                  end if
                  s% r(k) = exp_cr(s% lnR(k))
                  
                  if (i_lnd /= 0) then
                     s% lnd(k) = x(i_lnd)
                     s% rho(k) = exp_cr(s% lnd(k))
                     if (check_for_bad_nums .and. is_bad_num(s% rho(k))) then
                        if (report) write(*,2) 'bad num rho', k, s% rho(k)
                        ierr = -1
                     end if
                     if (abs(s% lnd(k) - s% lnd_start(k)) > &
                           ln10*s% hydro_mtx_max_allowed_abs_dlogRho .and. &
                          s% min_logT_for_hydro_mtx_max_allowed < &
                           ln10*min(s% lnT(k),s% lnT_start(k))) then
                        if (s% report_ierr) &
                           write(*,4) &
                              'hydro_mtx: dlogRho, logRho, logRho_start', &
                              s% model_number, k, iter, &
                              (s% lnd(k) - s% lnd_start(k))/ln10, &
                              s% lnd(k)/ln10, s% lnd_start(k)/ln10
                        ierr = -1
                        return
                     end if            
                     if (s% lnd(k) > ln10*s% hydro_mtx_max_allowed_logRho .and. &
                          s% min_logT_for_hydro_mtx_max_allowed < &
                           ln10*min(s% lnT(k),s% lnT_start(k))) then
                        if (s% report_ierr) &
                           write(*,4) 'hydro_mtx: logRho too large', &
                              s% model_number, k, iter, &
                              s% lnd(k)/ln10, s% lnd_start(k)/ln10
                        ierr = -1
                        return
                     end if            
                  end if
            
                  if (i_lnPgas /= 0) then
                     s% lnPgas(k) = x(i_lnPgas)
                     s% Pgas(k) = exp_cr(s% lnPgas(k))
                     if (check_for_bad_nums .and. is_bad_num(s% Pgas(k))) then
                        if (report) write(*,2) 'bad num Pgas', k, s% Pgas(k)
                        ierr = -1
                     end if
                     if (abs(s% lnPgas(k) - s% lnPgas_start(k)) > &
                           ln10*s% hydro_mtx_max_allowed_abs_dlogPgas .and. &
                         s% min_logT_for_hydro_mtx_max_allowed < &
                           ln10*min(s% lnT(k),s% lnT_start(k))) then
                        if (s% report_ierr) &
                           write(*,4) &
                              'hydro_mtx: dlogPgas, logPgas, logPgas_start', &
                              s% model_number, k, iter, &
                              (s% lnPgas(k) - s% lnPgas_start(k))/ln10, &
                              s% lnPgas(k)/ln10, s% lnPgas_start(k)/ln10
                        ierr = -1
                        return
                     end if            
                     if (s% lnPgas(k) > ln10*s% hydro_mtx_max_allowed_logPgas .and. &
                         s% min_logT_for_hydro_mtx_max_allowed < &
                           ln10*min(s% lnT(k),s% lnT_start(k))) then
                        if (s% report_ierr) &
                           write(*,4) 'hydro_mtx: logPgas too large', &
                              s% model_number, k, iter, &
                              s% lnPgas(k)/ln10, s% lnPgas_start(k)/ln10
                        ierr = -1
                        return
                     end if            
                  end if
                  
               end if
               
               if (k == s% trace_k) then
                  if (i_lnd /= 0) &
                     write(*,5) 'hydro_mtx: lnd', &
                        k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% lnd(k)
                  if (i_lnPgas /= 0) &
                     write(*,5) 'hydro_mtx: lnPgas', &
                        k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% lnPgas(k)
                  write(*,5) 'hydro_mtx: lnT', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% lnT(k)
                  if (i_lum /= 0) &
                     write(*,5) 'hydro_mtx: L', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, &
                        s% L(k), xh_pre(i_lum,k), dx(i_lum,k)
                  write(*,5) 'hydro_mtx: lnR', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% lnR(k)
                  if (i_v /= 0) &
                     write(*,5) 'hydro_mtx: v', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% v(k)
               end if
               
               ! set time derivatives at constant q
               if (s% dt == 0) then
                  s% dlnT_dt_const_q(k) = 0                  
                  s% dE_dt_const_q(k) = 0                  
                  if (s% do_struct_hydro) then
                     s% dlnd_dt_const_q(k) = 0
                     s% dlnPgas_dt_const_q(k) = 0
                  end if
               else if (k < s% k_below_const_q .and. .not. s% eps_grav_dt_use_start_values) then 
                  ! use dx to get better accuracy
                  if (i_lnT /= 0) s% dlnT_dt_const_q(k) = dx(i_lnT,k)*d_dxdt_dx                               
                  if (i_E /= 0) s% dE_dt_const_q(k) = dx(i_E,k)*d_dxdt_dx                               
                  if (s% do_struct_hydro) then
                     if (i_lnd /= 0) s% dlnd_dt_const_q(k) = dx(i_lnd,k)*d_dxdt_dx                  
                     if (i_lnPgas /= 0) &
                        s% dlnPgas_dt_const_q(k) = dx(i_lnPgas,k)*d_dxdt_dx  
                  end if                
               else
                  if (i_lnT /= 0) s% dlnT_dt_const_q(k) = &
                     (x(i_lnT) - s% lnT_for_d_dt_const_q(k))*d_dxdt_dx                  
                  if (i_E /= 0) s% dE_dt_const_q(k) = &
                     (x(i_E) - s% E_for_d_dt_const_q(k))*d_dxdt_dx                  
                  if (s% do_struct_hydro) then
                     if (i_lnd /= 0) &
                        s% dlnd_dt_const_q(k) = &
                           (x(i_lnd) - s% lnd_for_d_dt_const_q(k))*d_dxdt_dx                  
                     if (i_lnPgas /= 0) &
                        s% dlnPgas_dt_const_q(k) = &
                           (x(i_lnPgas) - s% lnPgas_for_d_dt_const_q(k))*d_dxdt_dx                  
                  end if                
               end if

               ! set time derivatives at constant mass
               if (s% dt == 0) then
               
                  s% dlnT_dt(k) = 0
                  s% dE_dt(k) = 0
                  if (s% do_struct_hydro) then
                     s% dlnd_dt(k) = 0
                     s% dlnPgas_dt(k) = 0
                     s% dlnR_dt(k) = 0
                     if (i_v /= 0) s% dv_dt(k) = 0
                  end if
                  
               else if (k >= s% k_const_mass .and. .not. s% eps_grav_dt_use_start_values) then 
                  ! use dx to get better accuracy
               
                  if (i_lnT /= 0) s% dlnT_dt(k) = dx(i_lnT,k)*d_dxdt_dx
                  if (i_E /= 0) s% dE_dt(k) = dx(i_E,k)*d_dxdt_dx
                  
                  if (s% do_struct_hydro) then
                     if (i_lnd /= 0) s% dlnd_dt(k) = dx(i_lnd,k)*d_dxdt_dx                  
                     if (i_lnPgas /= 0) &
                        s% dlnPgas_dt(k) = dx(i_lnPgas,k)*d_dxdt_dx                                                     
                     if (i_v /= 0) s% dv_dt(k) = dx(i_v,k)*d_dxdt_dx
                     s% dlnR_dt(k) = dx(i_lnR,k)*d_dxdt_dx                     
                  end if
                  
                  if (k == s% trace_k) then
                     write(*,5) 'dx(i_E,k)', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, dx(i_E,k)
                     write(*,5) 'd_dxdt_dx', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, d_dxdt_dx
                     write(*,5) 's% dE_dt(k)', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% dE_dt(k)
                     write(*,2) 'd_dxdt_dx', k, d_dxdt_dx
                     write(*,*)
                  end if
                  
               else if (k >= s% k_below_just_added) then

                  if (i_lnT /= 0) &
                     s% dlnT_dt(k) = (x(i_lnT) - s% lnT_for_d_dt_const_m(k))*d_dxdt_dx                  
                  if (i_E /= 0) &
                     s% dE_dt(k) = (x(i_E) - s% E_for_d_dt_const_m(k))*d_dxdt_dx                  
                  if (s% do_struct_hydro) then
                     s% dlnR_dt(k) = (x(i_lnR) - s% lnR_for_d_dt_const_m(k))*d_dxdt_dx                  
                     if (i_lnd /= 0) &
                        s% dlnd_dt(k) = (x(i_lnd) - s% lnd_for_d_dt_const_m(k))*d_dxdt_dx                  
                     if (i_lnPgas /= 0) &
                        s% dlnPgas_dt(k) = &
                           (x(i_lnPgas) - s% lnPgas_for_d_dt_const_m(k))*d_dxdt_dx                  
                     if (i_v /= 0) s% dv_dt(k) = &
                        (x(i_v) - s% v_for_d_dt_const_m(k))*d_dxdt_dx 
                  end if
                  
                  if (k == s% trace_k) then
                     write(*,5) 's% dlnT_dt(k)', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% dlnT_dt(k)
                     write(*,5) 'x(i_lnT)', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, x(i_lnT)
                     write(*,5) 's% lnT_for_d_dt_const_m(k)', k, &
                        s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% lnT_for_d_dt_const_m(k)
                     write(*,5) 'xh_pre(i_lnT,1)', 1, s% newton_iter, s% ebdf_stage, &
                        s% model_number, xh_pre(i_lnT,1)
                     write(*,2) 'd_dxdt_dx', k, d_dxdt_dx
                     write(*,*)
                  end if
                  
               ! k < s% k_below_just_added
               else if (k > 1 .and. s% del_t_for_just_added(k) > 0d0) then 
               
                  ! del_t is time since this cell entered at the surface
                  del_t = s% del_t_for_just_added(k)
                  ! interpolate starting values
                  alfa = del_t/s% dt
                  beta = 1d0 - alfa
                  
                  if (i_lnT /= 0) then
                     starting_value = alfa*xh_pre(i_lnT,1) + beta*s% surf_lnT
                     s% dlnT_dt(k) = (x(i_lnT) - starting_value)/del_t 
                  end if
                    
                  if (i_E /= 0) then
                     starting_value = alfa*xh_pre(i_E,1) + beta*s% surf_E
                     s% dE_dt(k) = (x(i_E) - starting_value)/del_t 
                  end if
                         
                  if (s% do_struct_hydro) then
                     starting_value = alfa*xh_pre(i_lnR,1) + beta*s% surf_lnR
                     s% dlnR_dt(k) = (x(i_lnR) - starting_value)/del_t
                     if (i_lnd /= 0) then
                        starting_value = alfa*xh_pre(i_lnd,1) + beta*s% surf_lnd
                        s% dlnd_dt(k) = (x(i_lnd) - starting_value)/del_t    
                     end if    
                     if (i_lnPgas /= 0) then
                        starting_value = alfa*xh_pre(i_lnPgas,1) + beta*s% surf_lnPgas
                        s% dlnPgas_dt(k) = (x(i_lnPgas) - starting_value)/del_t   
                     end if          
                     if (i_v /= 0) then
                        starting_value = alfa*xh_pre(i_v,1) + beta*s% surf_v
                        s% dv_dt(k) = (x(i_v) - starting_value)/del_t
                     end if    
                  end if    
                  
                  if (k == s% trace_k) then
                     write(*,5) 's% dlnT_dt(k)', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% dlnT_dt(k)
                     write(*,5) 'x(i_lnT)', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, x(i_lnT)
                     write(*,5) 'starting_value', k, s% newton_iter, s% ebdf_stage, &
                        s% model_number, starting_value
                     write(*,5) 'xh_pre(i_lnT,1)', 1, s% newton_iter, s% ebdf_stage, &
                        s% model_number, xh_pre(i_lnT,1)
                     write(*,5) 's% surf_lnT', 1, s% newton_iter, s% ebdf_stage, &
                        s% model_number, s% surf_lnT
                     write(*,2) 'del_t', k, del_t
                     write(*,2) 'alfa', k, alfa
                     write(*,2) 'beta', k, beta
                     write(*,*)
                     
                  end if
                  
               else ! k == 1 and k < s% k_below_just_added, so new surface cell
               
                  s% dlnT_dt(k) = 0  
                  s% dE_dt(k) = 0  
                  if (s% do_struct_hydro) then
                     s% dlnR_dt(k) = 0
                     s% dlnd_dt(k) = 0    
                     s% dlnPgas_dt(k) = 0    
                     s% dv_dt(k) = 0
                  end if
                  
               end if
               
            end if
            
            if (do_chem) &
               call check1_chem( &
                  s, k, min_xa_hard_limit, sum_xa_hard_limit, report, ierr)
         
         end subroutine set1
         
         
      end subroutine set_vars_for_solver

               
      subroutine check1_chem( &
            s, k, min_xa_hard_limit, sum_xa_hard_limit, report, ierr)
         use chem_def, only: chem_isos
         type (star_info), pointer :: s
         integer, intent(in) :: k
         real(dp), intent(in) :: min_xa_hard_limit, sum_xa_hard_limit
         logical, intent(in) :: report
         integer, intent(out) :: ierr

         integer :: j, species, jmax
         real(dp) :: sum_xa, xsum
         logical :: okay

         include 'formats'
         
         ierr = 0
         species = s% species
         okay = .true.
         if (min_xa_hard_limit > -1d50) then
            do j=1,species
               if (s% xa(j,k) < min_xa_hard_limit) then
                  if (report) then
                     write(*,3) &
                        'bad negative xa, hard limit, sig, logT ' // &
                        trim(chem_isos% name(s% chem_id(j))), j, k, &
                        s% xa(j,k), min_xa_hard_limit, s% sig(k), s% lnT(k)/ln10
                     if (.false.) then
                        write(*,3) &
                           'bad negative xa xa_pre xa-xa_pre ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% xa(j,k), s% xa_pre(j,k), &
                           s% xa(j,k) - s% xa_pre(j,k)
                        write(*,3) &
                           'bad negative logT T logRho rho ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% lnT(k)/ln10, s% T(k), s% lnd(k)/ln10, s% rho(k)
                        write(*,3) &
                           'bad negative m/Msun q r/Rsun ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% m(k)/Msun, s% q(k), s% r(k)/Rsun
                        write(*,3) &
                           'bad negative sig ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% sig(max(1,k-1)), s% sig(k), &
                           s% sig(min(s% nz, k+1)), s% sig(min(s% nz, k+2))
                        write(*,3) &
                           'bad negative: xa neighbors ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% xa(j,max(1,k-2):min(s% nz, k+2))
                        write(*,3) &
                           'bad negative: xa_pre neighbors ' // &
                           trim(chem_isos% name(s% chem_id(j))), j, k, &
                           s% xa_pre(j,max(1,k-2):min(s% nz, k+2))
                        write(*,*)
                     end if
                     okay = .false.
                  end if
                  ierr = -1
                  s% why_Tlim = Tlim_neg_X
                  if (.not. report) return
               else if (.false. .and. report .and. s% xa(j,k) < -1d-6) then
                  write(*,2) &
                     'negative ' // &
                     trim(chem_isos% name(s% chem_id(j))), &
                     k, s% xa(j,k), s% xa_pre(j,k), &
                     s% xa(j,k) - s% xa_pre(j,k), &
                     s% m(k)/Msun
               end if
            end do
         else if (.false. .and. s% report_ierr) then
            do j=1,species
               if (s% xa(j,k) < -1d-3) then
                  write(*,2) &
                     'negative ' // &
                     trim(chem_isos% name(s% chem_id(j))), &
                     k, s% xa(j,k), s% xa_pre(j,k), &
                     s% xa(j,k) - s% xa_pre(j,k), &
                     s% m(k)/Msun
               end if
            end do
         end if
         if (.false. .and. report .and. .not. okay) then
            do j=1,species
               write(*,2) &
                  trim(chem_isos% name(s% chem_id(j))), &
                  k, s% xa(j,k), s% xa_pre(j,k), &
                  s% xa(j,k) - s% xa_pre(j,k)
            end do
            write(*,*)
            !stop 'check1_chem'
         end if

         do j=1,species
            s% xa(j,k) = max(0d0, min(1d0, s% xa(j,k)))
         end do

         sum_xa = sum(s% xa(1:species,k))
         if (is_bad_num(sum_xa)) then
            if (report) then
               write(*,'(a60,i8,99f20.10)') 'bad num sum X', k, s% m(k)/Msun, sum_xa
            end if
            ierr = -1
            s% why_Tlim = Tlim_bad_Xsum
            if (.not. report) return
         end if
         if (abs(sum_xa - 1d0) > sum_xa_hard_limit) then
            if (report) then
               write(*,2) &
                  'bad sumX', k, &
                  sum_xa, sum_xa_hard_limit, &
                  sum(s% xa_pre(1:species,k)), &
                  sum_xa - sum(s% xa_pre(1:species,k))
               if (.false.) then
                  write(*,2) &
                     'bad sumX sig', k, &
                     s% sig(max(1,k-1)), s% sig(k), &
                     s% sig(min(s% nz, k+1)), s% sig(min(s% nz, k+2))
                  write(*,2) &
                     'bad sumX logT T logRho rho', k, &
                     s% lnT(k)/ln10, s% T(k), s% lnd(k)/ln10, s% rho(k)
                  write(*,2) &
                     'bad sumX m/Msun q r/Rsun', k, &
                     s% m(k)/Msun, s% q(k), s% r(k)/Rsun
                  write(*,2) &
                     'bad sumX sum_xa sum(dxdt_nuc)', k, &
                     sum_xa, sum(s% dxdt_nuc(:,k))
                  write(*,*)
               end if
            end if
           ierr = -1
            s% why_Tlim = Tlim_bad_Xsum
            okay = .false.
            if (.not. report) return
         end if
         if (.false. .and. report .and. .not. okay) then
            do j=1,species
               write(*,2) &
                  trim(chem_isos% name(s% chem_id(j))), &
                  k, s% xa(j,k), s% xa_pre(j,k), &
                  s% xa(j,k) - s% xa_pre(j,k)
            end do
            write(*,*)
            write(*,2) 'sum_xa', k, sum_xa
            write(*,*)
            !stop 'check1_chem bad sum X'
         end if

         if (abs(sum_xa - 1d0) > 1d-12) then
            !jmax = maxloc(s% xa(1:species,k),dim=1)
            !xsum = sum(s% xa(1:species,k)) - s% xa(jmax,k)
            !if (1d0 > xsum) then
            !   s% xa(jmax,k) = 1d0 - xsum
            !else
               do j=1,species
                  s% xa(j,k) = s% xa(j,k)/sum_xa
               end do
            !end if
         end if
         
         if (s% xa_clip_limit > 0) then
            do j=1,species
               if (s% xa(j,k) < s% xa_clip_limit) s% xa(j,k) = 0d0
            end do
         end if
         
      end subroutine check1_chem
         
         
      subroutine dump_struct(s)
         type (star_info), pointer :: s
         integer :: k, j, i
         
         include 'formats.dek'
         
         do k=1,s% nz
            write(*,2) 'dq', k, s% dq(k)
            write(*,2) 'm', k, s% m(k)
            write(*,2) 'T', k, s% T(k)
            write(*,2) 'rho', k, s% rho(k)
            write(*,2) 'Pgas', k, s% Pgas(k)
            write(*,2) 'L', k, s% L(k)
            write(*,2) 'r', k, s% r(k)
            write(*,2) 'grada', k, s% grada(k)
            write(*,2) 'opacity', k, s% opacity(k)
            write(*,2) 'd_opacity_dlnd', k, s% d_opacity_dlnd(k)
            write(*,2) 'd_opacity_dlnT', k, s% d_opacity_dlnT(k)
            write(*,2) 'eps_nuc', k, s% eps_nuc(k)
            write(*,2) 'd_epsnuc_dlnd', k, s% d_epsnuc_dlnd(k)
            write(*,2) 'd_epsnuc_dlnT', k, s% d_epsnuc_dlnT(k)
            write(*,2) 'non_nuc_neu', k, s% non_nuc_neu(k)
            write(*,2) 'd_nonnucneu_dlnd', k, s% d_nonnucneu_dlnd(k)
            write(*,2) 'd_nonnucneu_dlnT', k, s% d_nonnucneu_dlnT(k)
            write(*,2) 'eps_grav', k, s% eps_grav(k)
            write(*,2) 'd_eps_grav_dlndm1', k, s% d_eps_grav_dlndm1(k)
            write(*,2) 'd_eps_grav_dlnd00', k, s% d_eps_grav_dlnd00(k)
            write(*,2) 'd_eps_grav_dlndp1', k, s% d_eps_grav_dlndp1(k)
            write(*,2) 'd_eps_grav_dlnTm1', k, s% d_eps_grav_dlnTm1(k)
            write(*,2) 'd_eps_grav_dlnT00', k, s% d_eps_grav_dlnT00(k)
            write(*,2) 'd_eps_grav_dlnTp1', k, s% d_eps_grav_dlnTp1(k)
            write(*,2) 'gradT', k, s% gradT(k)
            write(*,2) 'd_gradT_dlnd00', k, s% d_gradT_dlnd00(k)
            write(*,2) 'd_gradT_dlndm1', k, s% d_gradT_dlndm1(k)
            !write(*,2) '', k, s% (k)
            do j=1,s% species
               write(*,3) 'xa(j,k)', j, k, s% xa(j,k)
            end do
         end do
         
      
      end subroutine dump_struct
      
         
      subroutine edit_lnR(s, xh_pre, dx) 
         ! uses mass and density to set radius
         type (star_info), pointer :: s
         real(dp), dimension(:, :) :: xh_pre
         real(dp), dimension(:, :) :: dx
         real(dp) :: vol00, volp1, cell_vol
         integer :: k, nz
         include 'formats'
         if (s% ebdf_stage > 0) return ! edit_lnR incompatible with higher order solvers
         vol00 = (4*pi/3)*s% R_center*s% R_center*s% R_center
         nz = s% nz         
         do k=nz, 1, -1
            volp1 = vol00
            cell_vol = s% dm(k)/s% rho(k)
            vol00 = volp1 + cell_vol
            s% lnR(k) = log_cr(vol00/(4*pi/3))/3
            dx(s% i_lnR,k) = s% lnR(k) - xh_pre(s% i_lnR,k)
            if (k >= s% k_below_just_added) &
               s% dlnR_dt(k) = &
                  (s% lnR(k) - s% lnR_for_d_dt_const_m(k))*s% dVARDOT_dVAR
         end do
         call edit_dlnR_dt_above_k_below_just_added(s, xh_pre)
      end subroutine edit_lnR

      
      subroutine edit_lnR_for_lnPgas(s, xh_pre, dx) 
         type (star_info), pointer :: s
         ! cannot use density since it hasn't been calculated yet
         real(dp), dimension(:, :) :: xh_pre
         real(dp), dimension(:, :) :: dx
         integer :: k, kk, k_outer, k_inner, cnt, nz, k_const_mass
         real(dp) :: dq_total, dq_sum, &
            r3_outer, r3_inner, r3, alfa, beta
         logical, parameter :: dbg = .false.
         include 'formats'
         if (s% ebdf_stage > 0) return ! edit_lnR incompatible with higher order solvers
         cnt = 0
         nz = s% nz
         k_const_mass = s% k_const_mass
         do k=1, nz-1
            if (s% lnR(k) > s% lnR(k+1)) cycle
            ! find k_outer s.t. lnR(k+1) < lnR(k_outer)
            if (dbg) write(*,*)
            if (dbg) write(*,2) 'bad s% lnR(k)', k, s% lnR(k)
            if (dbg) write(*,2) 'bad s% lnR(k+1)', k+1, s% lnR(k+1)
            cnt = cnt+1
            k_outer = 1
            do kk = k,1,-1
               if (s% lnR(k+1) < s% lnR(kk)) then
                  k_outer = kk; exit
               end if
            end do
            ! find k_inner s.t. lnR(k_inner) < lnR(k)
            k_inner = nz
            do kk = k+1, nz-1
               if (s% lnR(kk) < s% lnR(k)) then
                  k_inner = kk; exit
               end if
            end do
            dq_total = sum(s% dq(k_outer:k_inner-1))
            dq_sum = 0
            r3_outer = exp_cr(3*s% lnR(k_outer))
            r3_inner = exp_cr(3*s% lnR(k_inner))
            if (dbg) write(*,2) 's% lnR(k_outer)', k_outer, s% lnR(k_outer)
            do kk=k_outer+1, k_inner-1
               dq_sum = dq_sum + s% dq(kk-1)
               alfa = dq_sum/dq_total
               beta = 1 - alfa
               r3 = alfa*r3_inner + beta*r3_outer
               s% lnR(kk) = log_cr(r3)/3
               if (dbg) write(*,2) 'new s% lnR(kk)', kk, s% lnR(kk)
               dx(s% i_lnR,kk) = s% lnR(kk) - xh_pre(s% i_lnR,kk)
               if (kk >= s% k_below_just_added) &
                  s% dlnR_dt(kk) = &
                     (s% lnR(kk) - s% lnR_for_d_dt_const_m(kk))*s% dVARDOT_dVAR
            end do
            if (dbg) write(*,2) 's% lnR(k_inner)', k_inner, s% lnR(k_inner)
         end do
         call edit_dlnR_dt_above_k_below_just_added(s, xh_pre)
      end subroutine edit_lnR_for_lnPgas     
      
      
      subroutine edit_dlnR_dt_above_k_below_just_added(s, xh_pre)
         type (star_info), pointer :: s
         real(dp), dimension(:, :) :: xh_pre
         integer :: k, k_below_just_added
         real(dp) :: lnR_pre
         k_below_just_added = s% k_below_just_added
         if (k_below_just_added == 1) return
         lnR_pre = xh_pre(s% i_lnR,1)
         s% dlnR_dt(1) = 0
         do k = 2, k_below_just_added - 1
            if (s% del_t_for_just_added(k) <= 0d0) then
               s% dlnR_dt(k) = 0d0
            else
               s% dlnR_dt(k) = (s% lnR(k) - lnR_pre)/s% del_t_for_just_added(k)
            end if
         end do
      end subroutine edit_dlnR_dt_above_k_below_just_added
            

      subroutine enter_setmatrix( &
            iter, nvar_in, nz, neqns, dx, &
            xscale, xder, need_solver_to_eval_jacobian, &
            ldA, A1, A1_qp, idiag, lrpar, rpar, lipar, ipar, ierr)
         use mtx_def, only: lapack
         integer, intent(in) :: iter, nvar_in, nz, neqns ! (neqns = nvar*nz)
         real(dp), pointer, dimension(:,:) :: dx, xscale, xder ! (nvar, nz)
         logical, intent(out) :: need_solver_to_eval_jacobian
         integer, intent(in) :: ldA ! leading dimension of A
         real(dp), pointer, dimension(:) :: A1
         real(qp), pointer, dimension(:) :: A1_qp
         integer, intent(inout) :: idiag 
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         integer, intent(out) :: ierr

         type (star_info), pointer :: s                  
         real(dp), pointer, dimension(:,:) :: A ! (ldA, neqns)
         real(qp), pointer, dimension(:,:) :: A_qp ! (ldA, neqns)
         integer :: i, j, k, cnt, i_lnR, nnz, nzlo, nzhi
         real(dp), parameter :: epsder_struct = 1d-4
         real(dp), parameter :: epsder_chem = 1d-4
         real(dp) :: dt, lnR00, lnRm1, lnRp1, dlnR_prev, ddx, ddx_limit
         integer :: id, nvar, nvar_hydro
         logical :: dbg_enter_setmatrix, do_chem
         real(dp), pointer :: blk3(:, :, :, :)
         
         include 'formats'
         
         id = ipar(ipar_id)
         dbg_enter_setmatrix = dbg

         ierr = 0   
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            ierr = -1
            return
         end if

         if (dbg_enter_setmatrix) write(*, '(/,/,/,/,/,/,a)') 'enter_setmatrix'
         
         if (s% model_number == 1) then
            s% num_newton_iterations = s% num_newton_iterations + 1
            if (s% num_newton_iterations > 60 .and. &
                  mod(s% num_newton_iterations,10) == 0) &
               write(*,*) 'first model is slow to converge: num tries', &
                  s% num_newton_iterations
         end if
         
         dt = rpar(rpar_dt)
         
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         
         if (nvar /= nvar_in) then
            write(*,3) 'enter_setmatrix: bad value for nvar_in', nvar, nvar_in
            ierr = -1
            return
         end if

         s% idiag = idiag
         s% jacobian(1:ldA,1:neqns) => A1(1:ldA*neqns)
         A(1:ldA,1:neqns) => A1(1:ldA*neqns)
                  
         if (s% hydro_matrix_type == block_tridiag_dble_matrix_type) then
         
!$OMP PARALLEL DO PRIVATE(i,j)
            do i=1,neqns
               do j=1,lda
                  A(j,i) = 0
               end do
            end do
!$OMP END PARALLEL DO
            i = nvar*nvar*nz
            if (size(A1,dim=1) < 3*i) then
               write(*,*) 'enter_setmatrix: size(A1,dim=1) < 3*i', size(A1,dim=1), 3*i
               ierr = -1
               return
            end if
            s% ublk(1:nvar,1:nvar,1:nz) => A1(1:i)
            s% dblk(1:nvar,1:nvar,1:nz) => A1(i+1:2*i)
            s% lblk(1:nvar,1:nvar,1:nz) => A1(2*i+1:3*i)
            
         else if (s% hydro_matrix_type == block_tridiag_quad_matrix_type) then
         
            A_qp(1:ldA,1:neqns) => A1_qp(1:ldA*neqns)
!$OMP PARALLEL DO PRIVATE(i,j)
            do i=1,neqns
               do j=1,lda
                  A_qp(j,i) = 0
               end do
            end do
!$OMP END PARALLEL DO
            i = nvar*nvar*nz
            if (size(A1_qp,dim=1) < 3*i) then
               write(*,*) 'enter_setmatrix: size(A1_qp,dim=1) < 3*i', &
                  size(A1_qp,dim=1), 3*i
               ierr = -1
               return
            end if
            s% ublk_qp(1:nvar,1:nvar,1:nz) => A1_qp(1:i)
            s% dblk_qp(1:nvar,1:nvar,1:nz) => A1_qp(i+1:2*i)
            s% lblk_qp(1:nvar,1:nvar,1:nz) => A1_qp(2*i+1:3*i)
            
         end if
         
         if (s% hydro_call_number == s% hydro_dump_call_number .and. &
               iter == s% hydro_dump_iter_number) s% numerical_jacobian = .true.

         if (s% numerical_jacobian) then
            if (dbg_enter_setmatrix) &
               write(*,*) 's% numerical_jacobian', s% numerical_jacobian
            
            need_solver_to_eval_jacobian = .true.
            s% doing_numerical_jacobian = .true.
            nvar_hydro = s% nvar_hydro
            do j=1, nvar_hydro
               do k=1,nz
                  xder(j,k) = epsder_struct* &
                     max(1d0, abs(s% xh_pre(j,k)+dx(j,k)), &
                                 abs(s% xh_pre(j,k)))
               end do
            end do
            if (nvar > nvar_hydro) then
               do j=nvar_hydro+1, nvar
                  do k=1,nz
                     xder(j,k) = epsder_chem* &
                        max(1d0, abs(s% xa_pre(j-nvar_hydro,k)+dx(j,k)), &
                                          abs(s% xa_pre(j-nvar_hydro,k)))
                  end do
               end do
            end if
            i_lnR = s% i_lnR
            do k=1,nz
               ddx = xder(i_lnR,k)
               lnR00 = s% xh_pre(i_lnR,k) + dx(i_lnR,k)
               if (ddx > 0) then ! compare lnR00 to lnRm1
                  if (k == 1) cycle
                  lnRm1 = s% xh_pre(i_lnR,k-1) + dx(i_lnR,k-1)
                  ddx_limit = 0.5d0*(lnRm1 - lnR00)
                  if (ddx > ddx_limit) then
                     xder(i_lnR,k) = ddx_limit
                  end if
               else ! compare lnR00 to lnRp1
                  if (k == nz) exit
                  lnRp1 = s% xh_pre(i_lnR,k+1) + dx(i_lnR,k+1)
                  ddx_limit = 0.5d0*(lnRp1 - lnR00) ! ddx and ddx_limit both < 0
                  if (ddx < ddx_limit) then
                     xder(i_lnR,k) = ddx_limit
                  end if
               end if
            end do            
            if (dbg_enter_setmatrix) write(*,*) 'done enter_setmatrix'
            return
         end if
         
         if (dbg_enter_setmatrix) &
            write(*, *) 'call eval_partials with doing_numerical_jacobian = .false.'
         call eval_partials(s, xscale, ierr)
         if (ierr /= 0) return

         if (dbg_enter_setmatrix) write(*, *) 'finished enter_setmatrix'
         need_solver_to_eval_jacobian = .false.
         
            
      end subroutine enter_setmatrix
      
      
      subroutine eval_partials(s, xscale, ierr)
         use hydro_eqns, only: eval_equ
         type (star_info), pointer :: s
         real(dp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(out) :: ierr
         
         logical :: do_chem
         integer :: nvar
         logical, parameter :: skip_partials = .false., &
            convert_ODE_to_DAE_form = .true.
         ierr = 0
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         call eval_equ(s, nvar, skip_partials, convert_ODE_to_DAE_form, xscale, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*, *) 'eval_partials: eval_equ returned ierr', ierr
            return
         end if
         
      end subroutine eval_partials
      

      subroutine exit_setmatrix( &
            iter, nvar_in, nz, neqns, dx, &
            xscale, lrpar, rpar, lipar, ipar, &
            num_xtra, xtra_names, &
            xtra_lblk_analytic, xtra_dblk_analytic, xtra_ublk_analytic, &
            xtra_lblk_numeric, xtra_dblk_numeric, xtra_ublk_numeric, &
            ierr)
         use const_def, only: Rsun
         use chem_def
         integer, intent(in) :: iter, nvar_in, nz, neqns, num_xtra
         real(dp), pointer :: dx(:,:) ! (nvar, nz)
         real(dp), pointer, dimension(:,:) :: xscale ! (nvar, nz)
         integer, intent(in) :: lrpar, lipar
         real(dp), intent(inout) :: rpar(:) ! (lrpar)
         integer, intent(inout) :: ipar(:) ! (lipar)
         real(dp), intent(inout), dimension(:,:,:) :: & ! (num_xtra,nvar,nz)
            xtra_lblk_analytic, xtra_dblk_analytic, xtra_ublk_analytic, &
            xtra_lblk_numeric, xtra_dblk_numeric, xtra_ublk_numeric
         character (len=*), intent(in) :: xtra_names(num_xtra)

         integer, intent(out) :: ierr

         type (star_info), pointer :: s
         real(dp), pointer, dimension(:,:,:) :: ublk, dblk, lblk
         real(dp), pointer, dimension(:) :: ublk1, dblk1, lblk1
         integer :: nvar, id, nvar_hydro, j, i, k, equ_k, var_k, nzlo, nzhi, &
            cnt_ublk, cnt_dblk, cnt_lblk, i_lnd, i_lnT, i_lnR, i_v, i_E
         logical :: save_numjac_plot_data, test_analytical_jacobian, dbg_exit, do_chem 
         real(dp) :: dt, dlnT, dlnP
         character (len=32), dimension(nvar_in) :: nmvar, nmequ
         integer, pointer :: chem_id(:)
         
         include 'formats'
         
         dbg_exit = .true. ! dbg         
         id = ipar(ipar_id)
         
         ierr = 0
         
         call get_star_ptr(id, s, ierr)
         if (ierr /= 0) then
            ierr = -1
            return
         end if
         
         dt = rpar(rpar_dt)
         
         do_chem = (s% do_burn .or. s% do_mix)
         if (do_chem) then ! include abundances
            nvar = s% nvar
         else ! only structure
            nvar = s% nvar_hydro
         end if
         
         if (nvar /= nvar_in .or. nz /= s% nz .or. neqns /= nvar*nz) then
            write(*,*) 'nvar', nvar
            write(*,*) 'nvar_in', nvar_in
            write(*,*) 'nz', nz
            write(*,*) 's% nz', s% nz
            ierr = -1
            return
         end if
         
         if (dbg_exit) then
            write(*, *) 'enter exit_setmatrix'
            write(*, 2) 'nvar', nvar
            write(*, 2) 'species', s% species
            write(*, 2) 'nvar_hydro', s% nvar_hydro
            write(*, 2) 'nz', nz
            write(*, 2) 'neqns', neqns
            write(*, *) 
         end if

         s% doing_numerical_jacobian = .false.

         if (.not. s% numerical_jacobian) return
         
         if ((s% hydro_call_number /= s% hydro_dump_call_number) &
            .and. (s% hydro_dump_call_number > 0)) return
            
         if (s% hydro_dump_iter_number > 0 &
            .and. iter /= s% hydro_dump_iter_number) return
                  
         if (s% hydro_matrix_type /= block_tridiag_dble_matrix_type) then
            ierr = -1
            write(*,*) 'must have hydro_matrix_type == block_tridiag_dble_matrix_type'
            return
         end if
         
         save_numjac_plot_data = s% hydro_save_numjac_plot_data        
         
         test_analytical_jacobian = save_numjac_plot_data &
                  .or. (s% hydro_dump_call_number > 0)
         
         nvar_hydro = s% nvar_hydro
         
         i = nvar*nvar*nz
         allocate(ublk1(i), dblk1(i), lblk1(i), stat=ierr)
         if (ierr /= 0) then
            write(*,*) 'allocate failed in exit_setmatrix'
            return
         end if
         
         ublk(1:nvar,1:nvar,1:nz) => ublk1(1:i)
         dblk(1:nvar,1:nvar,1:nz) => dblk1(1:i)
         lblk(1:nvar,1:nvar,1:nz) => lblk1(1:i)

         cnt_ublk = 0
         cnt_dblk = 0
         cnt_lblk = 0
         do k=1,nz
            do j=1,nvar
               do i=1,nvar
                  if (s% ublk(i,j,k) /= 0) cnt_ublk = cnt_ublk + 1
                  ublk(i,j,k) = s% ublk(i,j,k)
                  if (s% dblk(i,j,k) /= 0) cnt_dblk = cnt_dblk + 1
                  dblk(i,j,k) = s% dblk(i,j,k)
                  if (s% lblk(i,j,k) /= 0) cnt_lblk = cnt_lblk + 1
                  lblk(i,j,k) = s% lblk(i,j,k)                     
               end do
            end do
         end do
         
         !write(*,*)
         !write(*,2) 'cnt_ublk', cnt_ublk               
         !write(*,2) 'cnt_dblk', cnt_dblk               
         !write(*,2) 'cnt_lblk', cnt_lblk               
         write(*,*)
         write(*,*) 'exit_setmatrix: evaluate analytic partials' 
         
         ! evaluate analytic partials
         
         write(*,*) 'exit_setmatrix: call set_newton_vars'
         call set_newton_vars(s, iter, dx, xscale, dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) &
               write(*, *) 'exit_setmatrix: set_newton_vars returned ierr', ierr
            return
         end if

         write(*,*) 'exit_setmatrix: call eval_partials'
         call eval_partials(s, xscale, ierr)
         if (ierr /= 0) return
         write(*,*) 'exit_setmatrix: done eval_partials'

         if (save_numjac_plot_data) then
         
            if (xtra_names(1) /= 'opacity') &
               stop 'dvdt_visc bad xtra_names for exit_setmatrix'
            if (xtra_names(2) /= 'lnP') &
               stop 'dvdt_visc bad xtra_names for exit_setmatrix'
            if (xtra_names(3) /= 'lnT') &
               stop 'dvdt_visc bad xtra_names for exit_setmatrix'

            if (num_xtra == 6) then
               if (xtra_names(4) /= 'L_total') &
                  stop 'L_total bad xtra_names for exit_setmatrix'
               if (xtra_names(5) /= 'dlnT_div_dlnP') &
                  stop 'dlnT_div_dlnP bad xtra_names for exit_setmatrix'
               if (xtra_names(6) /= 'one_div_dlnP') &
                  stop 'dvdt_visc bad xtra_names for exit_setmatrix'
            else if (num_xtra == 7) then
               if (xtra_names(4) /= 'Qvisc') &
                  stop 'Qvisc bad xtra_names for exit_setmatrix'
               if (xtra_names(5) /= 'eps_visc') &
                  stop 'eps_visc bad xtra_names for exit_setmatrix'
               if (xtra_names(6) /= 'dvdt_visc') &
                  stop 'dvdt_visc bad xtra_names for exit_setmatrix'
               if (xtra_names(7) /= 'eta_visc') &
                  stop 'dvdt_visc bad xtra_names for exit_setmatrix'
            end if
            
            i_lnd = s% i_lnd
            i_lnT = s% i_lnT
            i_E = s% i_E
            i_lnR = s% i_lnR
            i_v = s% i_v

            xtra_lblk_analytic(:,:,:) = 0
            xtra_dblk_analytic(:,:,:) = 0
            xtra_ublk_analytic(:,:,:) = 0
            
            do k=1,nz
               ! dblk(i,j,k) = df(i,k)/dx(j,k)
               if (s% E_flag) then
                  xtra_dblk_analytic(1,i_lnd,k) = &
                     s% d_opacity_dlnd(k) + &
                     s% d_opacity_dlnT(k)*s% dlnT_dlnd_c_E(k)
                  xtra_dblk_analytic(1,i_E,k) = &
                     s% d_opacity_dlnT(k)*s% dlnT_dlnE_c_Rho(k)*s% energy(k)
               else
                  xtra_dblk_analytic(1,i_lnd,k) = s% d_opacity_dlnd(k)
                  xtra_dblk_analytic(1,i_lnT,k) = s% d_opacity_dlnT(k)
               end if
               if (k > 1) then
                  ! lblk(i,j,k) = df(i,k)/dx(j,k-1)
               end if
               if (k < nz) then
                  ! ublk(i,j,k) = df(i,k)/dx(j,k+1)
               end if
            end do
            
            do k=1,nz
               ! dblk(i,j,k) = df(i,k)/dx(j,k)
               if (s% E_flag) then
                  xtra_dblk_analytic(2,i_lnd,k) = s% dlnP_dlnd_c_E(k)
                  !s% chiRho(k) + s% chiT(k)*(-s% rho(k)*s% dE_dRho(k)/(s% Cv(k)*s% T(k)))
                  xtra_dblk_analytic(2,i_E,k) = s% dlnP_dlnE_c_Rho(k)*s% energy(k)
                  !s% chiT(k)*(s% energy(k)/(s% Cv(k)*s% T(k)))
               else
                  xtra_dblk_analytic(2,i_lnd,k) = s% chiRho(k)
                  xtra_dblk_analytic(2,i_lnT,k) = s% chiT(k)
               end if
            end do
            
            do k=1,nz
               ! dblk(i,j,k) = df(i,k)/dx(j,k)
               if (s% E_flag) then
                  xtra_dblk_analytic(3,i_lnd,k) = s% dlnT_dlnd_c_E(k)
                  xtra_dblk_analytic(3,i_E,k) = s% dlnT_dlnE_c_Rho(k)*s% energy(k)
               else
                  xtra_dblk_analytic(3,i_lnd,k) = 0d0
                  xtra_dblk_analytic(3,i_lnT,k) = 1d0
               end if
            end do
            
            if (num_xtra == 6) then
            
               do k=1,nz
                  ! dblk(i,j,k) = df(i,k)/dx(j,k)
                  !xtra_dblk_analytic(4,i_lnd,k) = s% d_L_total_dlnd00(k)
                  !xtra_dblk_analytic(4,i_lnT,k) = s% d_L_total_dlnT00(k)
                  !xtra_dblk_analytic(4,i_lnR,k) = s% d_L_total_dlnR(k)
                  if (k > 1) then
                     ! lblk(i,j,k) = df(i,k)/dx(j,k-1)
                     !xtra_lblk_analytic(4,i_lnd,k) = s% d_L_total_dlndm1(k)
                     !xtra_lblk_analytic(4,i_lnT,k) = s% d_L_total_dlnTm1(k)
                  end if
                  !if (k < nz) then
                     ! ublk(i,j,k) = df(i,k)/dx(j,k+1)
                  !end if
               end do
            
               do k=2,nz
                  ! dblk(i,j,k) = df(i,k)/dx(j,k)
                  ! gradT = dlnT/dlnP
                  dlnT = s% lnT(k-1) - s% lnT(k)
                  dlnP = s% lnP(k-1) - s% lnP(k)
                  xtra_dblk_analytic(5,i_lnd,k) = dlnT*s% chiRho(k)/(dlnP*dlnP)
                  xtra_dblk_analytic(5,i_lnT,k) = dlnT*s% chiT(k)/(dlnP*dlnP) - 1d0/dlnP
                  if (k > 1) then
                     ! lblk(i,j,k) = df(i,k)/dx(j,k-1)
                     xtra_lblk_analytic(5,i_lnd,k) = -dlnT*s% chiRho(k-1)/(dlnP*dlnP)
                     xtra_lblk_analytic(5,i_lnT,k) = &
                        1d0/dlnP - dlnT*s% chiT(k-1)/(dlnP*dlnP)
                  end if
                  if (k < nz) then
                     ! ublk(i,j,k) = df(i,k)/dx(j,k+1)
                     !xtra_ublk_analytic(5,i_lnR,k) = s% d_eps_visc_dlnRp1(k)
                     !xtra_ublk_analytic(5,i_v,k) = s% d_eps_visc_dvelp1(k)
                  end if
               end do
            
               do k=2,nz ! no values for dvdt_visc(1)
                  ! dblk(i,j,k) = df(i,k)/dx(j,k)
                  dlnP = s% lnP(k-1) - s% lnP(k)
                  xtra_dblk_analytic(6,i_lnd,k) = s% chiRho(k)/(dlnP*dlnP)
                  xtra_dblk_analytic(6,i_lnT,k) = s% chiT(k)/(dlnP*dlnP)
                  if (k > 1) then
                     ! lblk(i,j,k) = df(i,k)/dx(j,k-1)
                     xtra_lblk_analytic(6,i_lnd,k) = -s% chiRho(k-1)/(dlnP*dlnP)
                     xtra_lblk_analytic(6,i_lnT,k) = -s% chiT(k-1)/(dlnP*dlnP)
                  end if
                  if (k < nz) then
                     ! ublk(i,j,k) = df(i,k)/dx(j,k+1)
                     !xtra_ublk_analytic(6,i_lnR,k) = s% d_dvdt_visc_dlnRp1(k)
                     !xtra_ublk_analytic(6,i_v,k) = s% d_dvdt_visc_dvp1(k)
                  end if
               end do
            
            else if (num_xtra == 7) then
            
               do k=1,nz
                  ! dblk(i,j,k) = df(i,k)/dx(j,k)
                  xtra_dblk_analytic(4,i_lnd,k) = s% dQvisc_dlnd(k)
                  xtra_dblk_analytic(4,i_lnR,k) = s% dQvisc_dlnR00(k)
                  xtra_dblk_analytic(4,i_v,k) = s% dQvisc_dvel00(k)
                  if (k > 1) then
                     ! lblk(i,j,k) = df(i,k)/dx(j,k-1)
                     !xtra_lblk_analytic(4,,k) = 
                  end if
                  if (k < nz) then
                     ! ublk(i,j,k) = df(i,k)/dx(j,k+1)
                     xtra_ublk_analytic(4,i_lnR,k) = s% dQvisc_dlnRp1(k)
                     xtra_ublk_analytic(4,i_v,k) = s% dQvisc_dvelp1(k)
                  end if
               end do
            
               do k=1,nz
                  ! dblk(i,j,k) = df(i,k)/dx(j,k)
                  xtra_dblk_analytic(5,i_lnd,k) = s% d_eps_visc_dlnd(k)
                  xtra_dblk_analytic(5,i_lnR,k) = s% d_eps_visc_dlnR00(k)
                  xtra_dblk_analytic(5,i_v,k) = s% d_eps_visc_dvel00(k)
                  if (k > 1) then
                     ! lblk(i,j,k) = df(i,k)/dx(j,k-1)
                     !xtra_lblk_analytic(5,,k) = 
                  end if
                  if (k < nz) then
                     ! ublk(i,j,k) = df(i,k)/dx(j,k+1)
                     xtra_ublk_analytic(5,i_lnR,k) = s% d_eps_visc_dlnRp1(k)
                     xtra_ublk_analytic(5,i_v,k) = s% d_eps_visc_dvelp1(k)
                  end if
               end do
            
               do k=2,nz ! no values for dvdt_visc(1)
                  ! dblk(i,j,k) = df(i,k)/dx(j,k)
                  xtra_dblk_analytic(6,i_lnd,k) = s% d_dvdt_visc_dlnd00(k)
                  xtra_dblk_analytic(6,i_lnR,k) = s% d_dvdt_visc_dlnR00(k)
                  xtra_dblk_analytic(6,i_v,k) = s% d_dvdt_visc_dv00(k)
                  if (k > 1) then
                     ! lblk(i,j,k) = df(i,k)/dx(j,k-1)
                     xtra_lblk_analytic(6,i_lnd,k) = s% d_dvdt_visc_dlndm1(k)
                     xtra_lblk_analytic(6,i_lnR,k) = s% d_dvdt_visc_dlnRm1(k)
                     xtra_lblk_analytic(6,i_v,k) = s% d_dvdt_visc_dvm1(k)
                  end if
                  if (k < nz) then
                     ! ublk(i,j,k) = df(i,k)/dx(j,k+1)
                     xtra_ublk_analytic(6,i_lnR,k) = s% d_dvdt_visc_dlnRp1(k)
                     xtra_ublk_analytic(6,i_v,k) = s% d_dvdt_visc_dvp1(k)
                  end if
               end do
            
               do k=1,nz
                  ! dblk(i,j,k) = df(i,k)/dx(j,k)
                  xtra_dblk_analytic(7,i_lnd,k) = s% d_eta_visc_dlnd(k)
                  xtra_dblk_analytic(7,i_lnR,k) = s% d_eta_visc_dlnR00(k)
                  xtra_dblk_analytic(7,i_v,k) = s% d_eta_visc_dvel00(k)
                  if (k > 1) then
                     ! lblk(i,j,k) = df(i,k)/dx(j,k-1)
                     !xtra_lblk_analytic(7,,k) = 
                  end if
                  if (k < nz) then
                     ! ublk(i,j,k) = df(i,k)/dx(j,k+1)
                     xtra_ublk_analytic(7,i_lnR,k) = s% d_eta_visc_dlnRp1(k)
                     xtra_ublk_analytic(7,i_v,k) = s% d_eta_visc_dvelp1(k)
                  end if
               end do
            
            end if
         
            nmvar(1:nvar_hydro) = s% nameofvar(1:nvar_hydro)
            nmequ(1:nvar_hydro) = s% nameofequ(1:nvar_hydro)
            if (nvar > nvar_hydro) then
               chem_id => s% chem_id
               do i=nvar_hydro+1,nvar
                  write(nmvar(i),'(a)') 'var_' // &
                     trim(chem_isos% name(chem_id(i-nvar_hydro)))
                  write(nmequ(i),'(a)') 'equ_' // &
                     trim(chem_isos% name(chem_id(i-nvar_hydro)))
               end do
            end if

            nzlo = s% hydro_jacobian_nzlo
            nzhi = s% hydro_jacobian_nzhi
            if (nzhi < nzlo) nzhi = nz

            write(*, *) 'save analytic jacobian plot data'
            call write_jacobian_info( &
                  nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
                  s% lblk, s% ublk, s% dblk, &
                  xscale, nmvar, nmequ, num_xtra, xtra_names, &
                  xtra_lblk_analytic, xtra_dblk_analytic, xtra_ublk_analytic, &
                  'plot_data/jacobian_data')
            !write(*, *) 'done write_jacobian_info'
         
            do k=1,nz
               do j=1,nvar
                  do i=1,nvar
                     s% ublk(i,j,k) = s% ublk(i,j,k) - ublk(i,j,k)
                     s% dblk(i,j,k) = s% dblk(i,j,k) - dblk(i,j,k)
                     s% lblk(i,j,k) = s% lblk(i,j,k) - lblk(i,j,k)                   
                  end do
               end do
               do j=1,nvar
                  do i=1,num_xtra
                     xtra_ublk_analytic(i,j,k) = &
                        xtra_ublk_analytic(i,j,k) - xtra_ublk_numeric(i,j,k)
                     xtra_dblk_analytic(i,j,k) = &
                        xtra_dblk_analytic(i,j,k) - xtra_dblk_numeric(i,j,k)
                     xtra_lblk_analytic(i,j,k) = &
                        xtra_lblk_analytic(i,j,k) - xtra_lblk_numeric(i,j,k)                   
                  end do
               end do
            end do

            write(*, *) 'save analytic minus numerical plot data'
            call write_jacobian_info( &
                  nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
                  s% lblk, s% ublk, s% dblk, &
                  xscale, nmvar, nmequ, num_xtra, xtra_names, &
                  xtra_lblk_analytic, xtra_dblk_analytic, xtra_ublk_analytic, &
                  'plot_data/jacobian_diff_data')
               
            write(*, *) 'save numerical jacobian plot data'
            call write_jacobian_info( &
                  nvar, nz, nzlo, nzhi, 1, nvar, 1, nvar, &
                  lblk, ublk, dblk, &
                  xscale, nmvar, nmequ, num_xtra, xtra_names, &
                  xtra_lblk_numeric, xtra_dblk_numeric, xtra_ublk_numeric, &
                  'plot_data/numerical_jacobian')
                  
         end if
                  
         write(*,*) 'done writing jacobians'
         
         stop 'exit_setmatrix'
         
         
      end subroutine exit_setmatrix


      subroutine write_jacobian_info( &
           nvar, nz, z_low, z_hi, var_lo, var_hi, eqn_lo, equ_hi, &
           lblk, ublk, dblk, xscale, var_names, equ_names, &
           num_xtra, xtra_names, xtra_lblk, xtra_dblk, xtra_ublk, &
           data_dir)
         use utils_lib,only:alloc_iounit,free_iounit
         integer, intent(in) :: nvar ! number of variables per zone
         integer, intent(in) :: nz ! number of zones.  n = nvar*nz.
         integer, intent(in) :: z_low ! first zone to include in output
         integer, intent(in) :: z_hi ! last zone to include in output

         integer, intent(in) :: var_lo
         integer, intent(in) :: var_hi
         integer, intent(in) :: eqn_lo
         integer, intent(in) :: equ_hi

         integer, intent(in) :: num_xtra

         real(dp), intent(in), dimension(:,:,:) :: &
            lblk, ublk, dblk ! (nvar,nvar,nz)
         real(dp), intent(in), dimension(:,:,:) :: &
            xtra_lblk, xtra_dblk, xtra_ublk ! (num_xtra,nvar,nz)
         real(dp), intent(in) :: xscale(:,:) ! (nvar,nz)
         character (len=*), intent(in) :: &
            var_names(nvar), equ_names(nvar), xtra_names(num_xtra), data_dir

         integer :: ionames, iodata, ierr, i, j, k
         real(dp), dimension(nz) :: Am1, A00, Ap1
         character (len=100) :: fig_name, fname

         ionames = alloc_iounit(ierr); if (ierr /= 0) return
         iodata = alloc_iounit(ierr); if (ierr /= 0) return
         ierr = 0
         fname = trim(data_dir) // '/names.data'
         open(unit=ionames,file=trim(fname),action='write',status='replace',iostat=ierr)
         if (ierr /= 0) then
            write(*,*) 'failed to open ', trim(fname)
            return
         end if
         
         do i = eqn_lo, equ_hi ! the equation number
            do j = var_lo, var_hi ! the variable number
            
               write(fig_name,'(a)') 'd_' // trim(equ_names(i)) // &
                  '_d' // trim(var_names(j))
               fname = trim(data_dir) // '/' // trim(fig_name) // '.data'
               open(unit=iodata,file=trim(fname),&
                  action='write',status='replace',iostat=ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed to open ', trim(fname)
                  return
               end if
               write(ionames,*) trim(fig_name)

               do k = z_low, z_hi
                  if (k == 1) then
                     Am1(k) = 0
                  else
                     Am1(k) = lblk(i,j,k)/xscale(j,k-1)
                  end if
                  A00(k) = dblk(i,j,k)/xscale(j,k)
                  if (k == nz) then
                     Ap1(k) = 0
                  else
                     Ap1(k) = ublk(i,j,k)/xscale(j,k+1)
                  end if
               end do
               write(iodata,'(9999e20.8)') Am1(z_low:z_hi)
               write(iodata,'(9999e20.8)') A00(z_low:z_hi)
               write(iodata,'(9999e20.8)') Ap1(z_low:z_hi)
               close(iodata)
               
            end do
         end do         
         
         do i = 1, num_xtra
            do j = var_lo, var_hi ! the variable number
            
               write(fig_name,'(a)') 'd_' // trim(xtra_names(i)) // &
                  '_d' // trim(var_names(j))
               fname = trim(data_dir) // '/' // trim(fig_name) // '.data'
               open(unit=iodata,file=trim(fname),&
                  action='write',status='replace',iostat=ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed to open ', trim(fname)
                  return
               end if
               write(ionames,*) trim(fig_name)
               
               do k = z_low, z_hi
                  if (k == 1) then
                     Am1(k) = 0
                  else
                     Am1(k) = xtra_lblk(i,j,k)
                  end if
                  A00(k) = xtra_dblk(i,j,k)
                  if (k == nz) then
                     Ap1(k) = 0
                  else
                     Ap1(k) = xtra_ublk(i,j,k)
                  end if
               end do
               write(iodata,'(9999e20.8)') Am1(z_low:z_hi)
               write(iodata,'(9999e20.8)') A00(z_low:z_hi)
               write(iodata,'(9999e20.8)') Ap1(z_low:z_hi)
               close(iodata)
               
            end do
         end do         
         
         close(ionames)

         fname = trim(data_dir) // '/jacobian_rows.data'
         open(unit=iodata,file=trim(fname),action='write',&
                  status='replace',iostat=ierr)
         write(iodata,'(i3)') -1, 0, 1
         close(iodata)

         fname = trim(data_dir) // '/jacobian_cols.data'
         open(unit=iodata,file=trim(fname),action='write',&
                  status='replace',iostat=ierr)
         do k=z_low, z_hi
            write(iodata,'(i3)') k
         end do
         close(iodata)

         call free_iounit(iodata)
         call free_iounit(ionames)

      end subroutine write_jacobian_info


      end module hydro_mtx

