! ***********************************************************************
!
!   Copyright (C) 2012  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module hydro_chem_eqns

      use star_private_def
      use const_def
      use utils_lib, only: is_bad_num

      implicit none
      

      contains


      subroutine do_chem_eqns( &
            s, xscale, nvar, equchem1, species, skip_partials, &
            convert_ODE_to_DAE_form, equ, ierr)
         type (star_info), pointer :: s
         real(dp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: nvar, equchem1, species
         logical, intent(in) :: skip_partials, convert_ODE_to_DAE_form
         real(dp), intent(out) :: equ(:,:)
         integer, intent(out) :: ierr         
         integer :: k, op_err
         include 'formats'         
         ierr = 0
!$OMP PARALLEL DO PRIVATE(k,op_err)
         do k = 1, s% nz
            if (ierr /= 0) cycle
            call do1_chem_eqns( &
               s, xscale, k, nvar, equchem1, species, skip_partials, &
               convert_ODE_to_DAE_form, equ, op_err)
            if (op_err /= 0) ierr = op_err      
         end do
!$OMP END PARALLEL DO
      end subroutine do_chem_eqns


      subroutine do1_chem_eqns( &
            s, xscale, k, nvar, equchem1, species, skip_partials, &
            convert_ODE_to_DAE_form, equ, ierr)
            
         use chem_def
         use net_lib, only: show_net_reactions, show_net_params
         use rates_def, only: reaction_Name, i_rate
         use star_utils, only: em1, e00, ep1
         use solve_mix, only: revise_avg_mix_dxdt
         
         type (star_info), pointer :: s
         real(dp), pointer :: xscale(:,:) ! (nvar, nz)
         integer, intent(in) :: k, nvar, equchem1, species
         logical, intent(in) :: skip_partials, convert_ODE_to_DAE_form
         real(dp), intent(out) :: equ(:,:)
         integer, intent(out) :: ierr

         integer, pointer :: reaction_id(:) ! maps net reaction number to reaction id
         integer :: nz, i_lnd, i_lnPgas, i_lnT, i_E, i_lnR, j, i, jj, ii
         real(dp), pointer, dimension(:) :: sig
         real(dp) :: &
            dxdt_expected_dxa, dxdt_expected, dxdt_actual, dVARdot_dVAR, &
            dxdt_expected_dlnd, dxdt_expected_dlnT, &
            dq, dm, dequ, dxdt_nuc, dxdt_mix, &
            sum_dxdt_nuc, dx_expected_dlnd, dx_expected_dlnT, &
            d_dxdt_mix_dx00, d_dxdt_mix_dxm1, d_dxdt_mix_dxp1, &
            sum_dx_burning, sum_dx_mixing, &
            x00, xp1, xm1, dx00, dxp1, &
            sig00, sigp1, flux00, dflux00_dxm1, dflux00_dx00, &
            fluxp1, dfluxp1_dx00, dfluxp1_dxp1, eqn_scale, d_dxdt_dx, &
            dequ_dlnd, dequ_dlnT, dequ_dlnPgas_const_T, dequ_dlnT_const_Pgas, &
            dlnT_dlnd_const_E, dlnT_dlnE_const_Rho, &
            dequ_dlnd_const_E, dequ_dE_const_Rho

         include 'formats'
         
         ierr = 0
         
         dVARdot_dVAR = s% dVARdot_dVAR
         
         nz = s% nz
         i_lnd = s% i_lnd
         i_lnPgas = s% i_lnPgas
         i_lnT = s% i_lnT
         i_E = s% i_E
         i_lnR = s% i_lnR

         sig => s% sig
         
         dq = s% dq(k)
         dm = s% dm(k)
                  
         sum_dxdt_nuc = 0
         d_dxdt_mix_dxp1 = 0
         d_dxdt_mix_dxm1 = 0
         d_dxdt_mix_dx00 = 0
         dxp1 = 0
         dx00 = 0

         sig00 = sig(k)
   
         if (k < s% nz) then
            sigp1 = sig(k+1)
         else
            sigp1 = 0
         end if

         if (s% do_mix) then
      
            if (k > 1) then
               dflux00_dxm1 = -sig00
               dflux00_dx00 = sig00
            else
               dflux00_dxm1 = 0
               dflux00_dx00 = 0
            end if
      
            if (k < s% nz) then
               dfluxp1_dx00 = -sigp1
               dfluxp1_dxp1 = sigp1
            else
               dfluxp1_dx00 = 0
               dfluxp1_dxp1 = 0
            end if
         
            d_dxdt_mix_dx00 = (dfluxp1_dx00 - dflux00_dx00)/dm
            d_dxdt_mix_dxm1 = -dflux00_dxm1/dm
            d_dxdt_mix_dxp1 = dfluxp1_dxp1/dm
            
         end if
         
         sum_dx_burning = 0
         sum_dx_mixing = 0

         do j=1,species ! composition equation for species j in cell k
      
            i = equchem1+j-1
            
            x00 = s% xa(j,k)
            dx00 = s% xa_sub_xa_pre(j,k)
            dxdt_actual = dx00*dVARdot_dVAR
         
            if (s% do_burn) then            
               dxdt_nuc = s% dxdt_nuc(j,k)               
            else            
               dxdt_nuc = 0              
            end if
            
            if (s% do_mix) then            
               if (k > 1) then
                  xm1 = s% xa(j,k-1)
                  dx00 = xm1 - x00
                  flux00 = -sig00*dx00
               else
                  flux00 = 0
               end if         
               if (k < s% nz) then
                  xp1 = s% xa(j,k+1)
                  dxp1 = x00 - xp1
                  fluxp1 = -sigp1*dxp1
               else
                  fluxp1 = 0
               end if
               dxdt_mix = (fluxp1 - flux00)/dm               
            else           
               dxdt_mix = 0              
            end if

            dxdt_expected = dxdt_mix + dxdt_nuc           
         
            if (convert_ODE_to_DAE_form .and. associated(xscale)) then
               eqn_scale = xscale(i,k)*dVARdot_dVAR
               equ(i,k) = (dxdt_expected - dxdt_actual)/eqn_scale
            else
               equ(i,k) = dxdt_expected
               eqn_scale = 1d0
            end if         
            
            if (s% using_ode_form) s% ode(i,k) = dxdt_expected
      
            if (skip_partials) cycle
         
            if (convert_ODE_to_DAE_form) & ! partial of -dxdt_actual/eqn_scale
               call e00(s, xscale, i, i, k, nvar, -dVARdot_dVAR/eqn_scale)
         
            ! all the rest are jacobian terms for dxdt_expected/eqn_scale
            
            if (s% do_burn) then
         
               do jj=1,species
                  ii = equchem1+jj-1
                  dxdt_expected_dxa = s% d_dxdt_dx(j,jj,k)
                  dequ = dxdt_expected_dxa/eqn_scale
                  call e00(s, xscale, i, ii, k, nvar, dequ)
               end do
      
               dxdt_expected_dlnd = s% dxdt_drho(j,k)*s% rho(k)
               dequ_dlnd = dxdt_expected_dlnd/eqn_scale
               dxdt_expected_dlnT = s% dxdt_dT(j,k)*s% T(k) 
               dequ_dlnT = dxdt_expected_dlnT/eqn_scale
            
               if (s% do_struct_hydro) then ! partial wrt lnd or lnPgas
                  if (s% lnPgas_flag) then
                     dequ_dlnPgas_const_T = dequ_dlnd*s% dlnRho_dlnPgas_const_T(k)
                     call e00(s, xscale, i, i_lnPgas, k, nvar, dequ_dlnPgas_const_T)
                  else if (s% E_flag) then
                     dlnT_dlnd_const_E = s% dlnT_dlnd_c_E(k)
                     dequ_dlnd_const_E = dequ_dlnd + dequ_dlnT*dlnT_dlnd_const_E
                     call e00(s, xscale, i, i_lnd, k, nvar, dequ_dlnd_const_E)
                  else
                     call e00(s, xscale, i, i_lnd, k, nvar, dequ_dlnd)
                  end if
               end if

               if (s% do_struct_thermo) then ! partial wrt lnT or E
                  if (s% lnPgas_flag) then
                     dequ_dlnT_const_Pgas = &
                        dequ_dlnT + dequ_dlnd*s% dlnRho_dlnT_const_Pgas(k)
                     call e00(s, xscale, i, i_lnT, k, nvar, dequ_dlnT_const_Pgas)
                  else if (s% E_flag) then
                     dequ_dE_const_Rho = dequ_dlnT*s% dlnT_dlnE_c_Rho(k)/s% energy(k)
                     call e00(s, xscale, i, i_E, k, nvar, dequ_dE_const_Rho)
                  else
                     call e00(s, xscale, i, i_lnT, k, nvar, dequ_dlnT)
                  end if
               end if

            end if
         
            if (s% do_mix) then 
                       
               dxdt_expected_dxa = d_dxdt_mix_dx00
               dequ = dxdt_expected_dxa/eqn_scale
               call e00(s, xscale, i, i, k, nvar, dequ)               
               if (k > 1) then
                  dxdt_expected_dxa = d_dxdt_mix_dxm1
                  dequ = dxdt_expected_dxa/eqn_scale
                  call em1(s, xscale, i, i, k, nvar, dequ)
               end if               
               if (k < nz) then
                  dxdt_expected_dxa = d_dxdt_mix_dxp1
                  dequ = dxdt_expected_dxa/eqn_scale
                  call ep1(s, xscale, i, i, k, nvar, dequ)
               end if    
               
            end if
         
         end do
         

      end subroutine do1_chem_eqns


      end module hydro_chem_eqns

