! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


      module element_diffusion

      use star_private_def
      use const_def
      use utils_lib, only:is_bad_num, has_bad_num

      implicit none
      
      logical, parameter :: dbg = .false.
      
      
      contains
      

      logical function do_element_diffusion(s, dt_in) 
         ! return false if cannot satisfy accuracy requirements
         use mlt_def, only: no_mixing
         use chem_def, only: chem_isos, ihe4, ih1
         use chem_lib, only: chem_get_iso_id
         use diffusion, only: &
            do_solve_diffusion, set_diffusion_classes, diffusion_min_nc
         type (star_info), pointer :: s
         real(dp), intent(in) :: dt_in
         
         integer :: ierr, i, j, k, kk, nc, m, nzlo, nzhi, nz, species, iounit, &
            steps_used, total_num_iters, total_num_retries, cid, he4
         
         integer, dimension(:), pointer :: &
            class, class_chem_id, mixing_type, mixing_type_arg
         real(dp) :: s1, s2, dqsum, dist, r, Hp, dt, &
            gradT_mid, gradRho_mid, alfa, gradRho_face, chiRho_face, chiT_face
         real(dp), dimension(:), pointer :: &
            gamma, free_e, &
            dlnP_dm_face, dlnT_dm_face, dlnRho_dm_face, &
            dlnP_dm, dlnT_dm, dlnP_dm_mid, dlnT_dm_mid, dlnRho_dm_mid, dm_hat
         real(dp), dimension(:,:), pointer :: &
            X_init, X_final, typical_charge, &
            D_self, v_advection, &
            vlnP, vlnT, v_rad, g_rad, GT, xa_save
         real(dp), dimension(:,:,:), pointer :: CD
         character (len=8), pointer :: class_name(:)
         real(dp), pointer, dimension(:) :: CD1, &
            X_init1, X_final1, typical_charge1, &
            D_self1, v_advection1, &
            vlnP1, vlnT1, v_rad1, g_rad1, GT1, xa_save1
         
         real(dp), parameter :: min_D_mix = 0d0

         logical :: dumping, okay

         include 'formats'
         
         ierr = 0
         dt = dt_in
         nz = s% nz
         
         s% num_diffusion_solver_steps = 0
         
         if ((.not. s% do_element_diffusion) .or. dt < s% diffusion_dt_limit) then
            do_element_diffusion = .true.
            if (s% do_element_diffusion) then
               s% edv(:,1:nz) = 0
            end if
            if (s% report_ierr) then
               if (.not. s% do_element_diffusion) &
                  write(*,*) '.not. s% do_element_diffusion'
               if (dt < s% diffusion_dt_limit) &
                  write(*,1) 'dt < s% diffusion_dt_limit', dt, s% diffusion_dt_limit
            end if
            return
         end if

         if (s% use_other_diffusion) then
            call s% other_diffusion(s% id, dt_in, ierr)
            do_element_diffusion = (ierr == 0)
            return
         end if

         do_element_diffusion = .false.
         nz = s% nz
         nzlo = 1
         nzhi = nz
         species = s% species
         nc = s% diffusion_num_classes
         m = nc+1
         
         call do_alloc(ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'do_element_diffusion failed in allocate'
            return
         end if
         
         call set_extras(ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'do_element_diffusion failed in set_extras'
            return
         end if
         
         
         !write(*,1) 's% diffusion_min_T_at_surface', s% diffusion_min_T_at_surface
         !reset nzlo if necessary; nzlo=1 above
         nzlo = 1
         if (s% diffusion_min_T_at_surface > 0) then
            k = nzlo
            do while (s% T(k) < s% diffusion_min_T_at_surface)
               k = k+1
               if (k > nz) exit
               nzlo = k
            end do
         end if
         
         !write(*,2) 'diffusion_min_T_at_surface', nzlo
         
         if (s% diffusion_min_dq_ratio_at_surface > 0) then
            dqsum = sum(s% dq(1:nzlo))
            k = nzlo
            do while (dqsum < s% diffusion_min_dq_ratio_at_surface*s% dq(k+1))
               k = k+1
               if (k >= nz) exit
               dqsum = dqsum + s% dq(k)
               nzlo = k
            end do
         end if
         
         !write(*,2) 'before diffusion_min_dq_ratio_at_surface', nzlo, s% diffusion_min_dq_at_surface
         
         if (s% diffusion_min_dq_at_surface > 0) then
            dqsum = sum(s% dq(1:nzlo))
            k = nzlo
            do while (dqsum < s% diffusion_min_dq_at_surface)
               k = k+1
               if (k > nz) exit
               ! don't go across composition transition
               if (maxloc(s% xa(:,k),dim=1) /= maxloc(s% xa(:,k-1),dim=1)) exit
               dqsum = dqsum + s% dq(k)
               nzlo = k
            end do
            !write(*,2) 'after diffusion_min_dq_ratio_at_surface', nzlo, dqsum
         end if
         
         !write(*,3) 'nzlo mixing_type', nzlo, s% mixing_type(nzlo)
         
         if (s% D_mix(nzlo) > min_D_mix) then
            kk = nzlo
            do k = nzlo+1, nz-1
               if (maxloc(s% xa(:,k),dim=1) /= maxloc(s% xa(:,k-1),dim=1) .or. &
                   (s% D_mix(k) > min_D_mix .and. s% D_mix(k+1) <= min_D_mix)) then
                  nzlo=k; exit
               endif
            end do
            nzlo = (nzlo+kk)/2 ! back up into the convection zone
         end if
         
         !write(*,3) 'nzlo mixing_type', nzlo, s% mixing_type(nzlo)
         !write(*,3) 'nzlo-1 mixing_type', nzlo-1, s% mixing_type(nzlo-1)
         !write(*,*)
         !write(*,*)

         !reset nzhi if necessary; nzhi=nz above
         if (s% mixing_type(nzhi) > min_D_mix) then
            do k = nz, 2, -1
               if (s% mixing_type(k) > min_D_mix .and. &
                     s% mixing_type(k-1) <= min_D_mix) then
                  nzhi=k; exit
               end if
            end do
            nzhi = (3*nzhi+nz)/4 ! back up some into the convection zone
         end if
         
         if (.false. .and. nzhi < nzlo) then
            write(*,*) 'check for non convective zone below nzlo'
            okay = .true.
            do k=nzlo+1,nz
               if (s% mixing_type(k) > min_D_mix) cycle
               okay = .false.
               write(*,3) 's% mixing_type(k)', k, s% mixing_type(k)
            end do
            if (.not. okay) stop 'diff'
            do_element_diffusion = .true.
            return
         end if
         
         do j=1,nc
            cid = chem_get_iso_id(s% diffusion_class_representative(j))
            if (cid <= 0) then
               write(*,'(a,3x,i3)') 'bad entry for diffusion_class_representative: ' // &
                  trim(s% diffusion_class_representative(j)), j
               return
            end if
            class_chem_id(j) = cid
         end do

         call set_diffusion_classes( &
            nc, species, s% chem_id, class_chem_id, s% diffusion_class_A_max, &
            class, class_name)
         
         s% diffusion_call_number = s% diffusion_call_number + 1
         dumping = (s% diffusion_call_number == s% diffusion_dump_call_number)
         
         if (.not. s% diffusion_calculates_ionization) then
            do j=1,nc
               typical_charge(j,1:nz) = s% diffusion_class_typical_charge(j)
            end do
         end if
         
         do k=1,nz
            do j=1,species
               xa_save(j,k) = s% xa(j,k)
            end do
         end do
         
         mixing_type(1:nz) = no_mixing
         mixing_type_arg => mixing_type
         
         do k=1,nz-1
            s1 = dlnP_dm(k)
            s2 = dlnP_dm(k+1)
            if (s1*s2 <= 0) then
               dlnP_dm_mid(k) = 0
            else
               dlnP_dm_mid(k) = 2*s1*s2/(s1+s2)
            end if
            gradT_mid = 0.5d0*(s% gradT(k) + s% gradT(k+1))
            dlnT_dm_mid(k) = gradT_mid*dlnP_dm_mid(k)
            gradRho_mid = (1 - s% chiT(k)*gradT_mid)/s% chiRho(k)
            dlnRho_dm_mid(k) = gradRho_mid*dlnP_dm_mid(k)
         end do
         dlnP_dm_mid(nz) = dlnP_dm(nz)
         dlnT_dm_mid(nz) = dlnT_dm(nz)
         gradRho_mid = (1 - s% chiT(nz)*s% gradT(nz))/s% chiRho(nz)
         dlnRho_dm_mid(nz) = gradRho_mid*dlnP_dm_mid(nz)
            
         do k=2,nz
            dlnP_dm_face(k) = dlnP_dm(k)
            dlnT_dm_face(k) = dlnT_dm(k)
            alfa = s% dm(k-1)/(s% dm(k) + s% dm(k-1))
            chiT_face = alfa*s% chiT(k) + (1-alfa)*s% chiT(k-1)
            chiRho_face = alfa*s% chiRho(k) + (1-alfa)*s% chiRho(k-1)
            gradRho_face = (1 - chiT_face*s% gradT(k))/chiRho_face
            dlnRho_dm_face(k) = gradRho_face*dlnP_dm(k)
         end do
         dlnP_dm_face(1) = 0
         dlnT_dm_face(1) = 0
         dlnRho_dm_face(1) = 0
         
         if (dumping) call dump_diffusion_info
         
         ! args are at cell center points.
         !if (s% show_diffusion_info) write(*,*) 'call solve_diffusion'
         !write(*,4) 'call do_solve_diffusion nzlo nzhi nz', nzlo, nzhi, nz, &
         !   sum(s% xa(1,1:nzlo))
         call do_solve_diffusion( &
            s, nz, species, nc, m, class, class_chem_id, s% net_iso, s% chem_id, &
            s% abar, s% ye, free_e, s% dm_bar, s% dm, &
            s% T, s% lnT, s% rho, s% lnd, s% rmid, &
            dlnP_dm_mid, dlnT_dm_mid, dlnRho_dm_mid, &
            s% L, s% r, dlnP_dm_face, dlnT_dm_face, dlnRho_dm_face, &
            s% diffusion_use_iben_macdonald, dt, s% diffusion_dt_div_timescale, &
            s% diffusion_steps_hard_limit, s% diffusion_iters_hard_limit, &
            s% diffusion_max_iters_per_substep, &
            s% diffusion_calculates_ionization, typical_charge, &
            s% diffusion_nsmooth_typical_charge, &
            s% diffusion_min_T_for_radaccel, &
            s% diffusion_max_T_for_radaccel, &
            s% diffusion_screening_for_radaccel, &
            s% op_mono_data_path, s% op_mono_data_cache_filename, &
            s% diffusion_v_max, s% R_center, &
            gamma, s% diffusion_gamma_full_on, s% diffusion_gamma_full_off, &
            s% diffusion_T_full_on, s% diffusion_T_full_off, &
            s% diffusion_class_factor, s% xa, &
            steps_used, total_num_iters, total_num_retries, nzlo, nzhi, X_init, X_final, &
            D_self, v_advection, vlnP, vlnT, v_rad, g_rad, &
            s% E_field, s% g_field_element_diffusion, ierr )
         do_element_diffusion = (ierr == 0)
         s% num_diffusion_solver_steps = steps_used
         s% num_diffusion_solver_iters = total_num_iters
         !write(*,1) 'done solve_diffusion', s% xa(1,332)
         
         if (dbg .or. s% show_diffusion_info) then
            if (do_element_diffusion) then
               write(*,'(a,f6.3,3x,a,1pe10.3,3x,99(a,i5,3x))') &
                  'log_dt', log10_cr(s% dt/secyer), 'age', s% star_age, 'model', s% model_number, &
                  'iters', total_num_iters, 'steps', steps_used, 'retries', total_num_retries, &
                  'nzlo', nzlo, 'nzhi', nzhi, 'n', nzhi-nzlo+1, 'nz', nz, &
                  'diffusion_call_number', s% diffusion_call_number
            else
               write(*,'(a,2x,f10.3,3x,99(a,i5,3x))') &
                  'diffusion FAILED: log_dt', log10_cr(s% dt/secyer), 'model', s% model_number, &
                  'iters', total_num_iters, 'steps', steps_used, 'retries', total_num_retries, &
                  'nzlo', nzlo, 'nzhi', nzhi, 'n', nzhi-nzlo+1, 'nz', nz, &
                  'diffusion_call_number', s% diffusion_call_number
            end if
         end if
         
         if (.not. do_element_diffusion) then
            do k=1,nz
               do j=1,species
                  s% xa(j,k) = xa_save(j,k)
               end do
            end do
         end if

         if (.not. do_element_diffusion .and. s% report_ierr) then
            write(*, *) 
            write(*, *) 'solve_diffusion returned false'
            write(*, *) 's% model_number', s% model_number
            write(*, *) 's% diffusion_call_number', s% diffusion_call_number
            write(*, *) 
         end if
            
         if (dumping) stop 'debug: dump_diffusion_info'
         
         do k=nzlo+1,nzhi 
            do j=1,species
               i = class(j)
               s% diffusion_D_self(j,k) = D_self(i,k)
               s% edv(j,k) = v_advection(i,k)
               s% v_rad(j,k) = v_rad(i,k)
               s% g_rad(j,k) = g_rad(i,k)
               s% typical_charge(j,k) = typical_charge(i,k)
               s% diffusion_dX(j,k) = s% xa(j,k) - xa_save(j,k)
            end do
         end do
         
         do k=1,nzlo
            do j=1,species
               s% diffusion_D_self(j,k) = s% diffusion_D_self(j,nzlo+1)
               s% edv(j,k) = s% edv(j,nzlo+1)
               s% v_rad(j,k) = s% v_rad(j,nzlo+1)
               s% g_rad(j,k) = s% g_rad(j,nzlo+1)
               s% typical_charge(j,k) = s% typical_charge(j,nzlo+1)
               s% diffusion_dX(j,k) = s% xa(j,k) - xa_save(j,k)
            end do
            s% E_field(k) = s% E_field(nzlo+1)
            s% g_field_element_diffusion(k) = s% g_field_element_diffusion(nzlo+1)
         end do
         
         do k=nzhi+1,nz
            do j=1,species
               s% diffusion_D_self(j,k) = s% diffusion_D_self(j,nzhi)
               s% edv(j,k) = s% edv(j,nzhi)
               s% v_rad(j,k) = s% v_rad(j,nzhi)
               s% g_rad(j,k) = s% g_rad(j,nzhi)
               s% typical_charge(j,k) = s% typical_charge(j,nzhi)
               s% diffusion_dX(j,k) = s% xa(j,k) - xa_save(j,k)
            end do
            s% E_field(k) = s% E_field(nzhi)
            s% g_field_element_diffusion(k) = s% g_field_element_diffusion(nzhi)
         end do
         

         call dealloc


         contains
         
         
         subroutine do_alloc(ierr)
            use alloc
            use utils_lib
            integer, intent(out) :: ierr
            
            call get_integer_work_array(s, mixing_type, nz, nz_alloc_extra, ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, gamma, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, dlnP_dm, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, dlnT_dm, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, dlnP_dm_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, dlnT_dm_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, dlnRho_dm_mid, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            
            call non_crit_get_work_array(s, free_e, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return            

            call non_crit_get_work_array(s, dlnP_dm_face, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, dlnT_dm_face, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            call non_crit_get_work_array(s, dlnRho_dm_face, nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return

            call non_crit_get_work_array(s, X_init1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            X_init(1:nc,1:nz) => X_init1(1:nc*nz)
            call non_crit_get_work_array(s, X_final1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            X_final(1:nc,1:nz) => X_final1(1:nc*nz)
            call non_crit_get_work_array(s, typical_charge1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            typical_charge(1:nc,1:nz) => typical_charge1(1:nc*nz)
            
            call non_crit_get_work_array(s, D_self1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            D_self(1:nc,1:nz) => D_self1(1:nc*nz)
            
            call non_crit_get_work_array(s, v_advection1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            v_advection(1:nc,1:nz) => v_advection1(1:nc*nz)
            
            call non_crit_get_work_array(s, vlnP1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            vlnP(1:nc,1:nz) => vlnP1(1:nc*nz)
            
            call non_crit_get_work_array(s, vlnT1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            vlnT(1:nc,1:nz) => vlnT1(1:nc*nz)
            
            call non_crit_get_work_array(s, v_rad1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            v_rad(1:nc,1:nz) => v_rad1(1:nc*nz)
            
            call non_crit_get_work_array(s, g_rad1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            g_rad(1:nc,1:nz) => g_rad1(1:nc*nz)
            
            call non_crit_get_work_array(s, GT1, nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            GT(1:nc,1:nz) => GT1(1:nc*nz)
            call non_crit_get_work_array(s, xa_save1, species*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            xa_save(1:species,1:nz) => xa_save1(1:species*nz)

            call non_crit_get_work_array(s, CD1, nc*nc*nz, nz_alloc_extra, 'diffusion', ierr)
            if (ierr /= 0) return
            CD(1:nc,1:nc,1:nz) => CD1(1:nc*nc*nz)
            
            allocate(class(species), class_chem_id(nc), class_name(nc), stat=ierr)

         end subroutine do_alloc
            
            
         subroutine dealloc
            use alloc
            use utils_lib
            
            call return_integer_work_array(s, mixing_type)
            call non_crit_return_work_array(s, gamma, 'diffusion')
            call non_crit_return_work_array(s, dlnP_dm, 'diffusion')
            call non_crit_return_work_array(s, dlnT_dm, 'diffusion')
            call non_crit_return_work_array(s, dlnP_dm_mid, 'diffusion')
            call non_crit_return_work_array(s, dlnT_dm_mid, 'diffusion')
            call non_crit_return_work_array(s, dlnRho_dm_mid, 'diffusion')
            call non_crit_return_work_array(s, free_e, 'diffusion')
            call non_crit_return_work_array(s, dlnP_dm_face, 'diffusion')
            call non_crit_return_work_array(s, dlnT_dm_face, 'diffusion')
            call non_crit_return_work_array(s, dlnRho_dm_face, 'diffusion')
            call non_crit_return_work_array(s, X_init1, 'diffusion')
            call non_crit_return_work_array(s, X_final1, 'diffusion')
            call non_crit_return_work_array(s, typical_charge1, 'diffusion')
            call non_crit_return_work_array(s, D_self1, 'diffusion')
            call non_crit_return_work_array(s, v_advection1, 'diffusion')
            call non_crit_return_work_array(s, vlnP1, 'diffusion')
            call non_crit_return_work_array(s, vlnT1, 'diffusion')
            call non_crit_return_work_array(s, v_rad1, 'diffusion')
            call non_crit_return_work_array(s, g_rad1, 'diffusion')
            call non_crit_return_work_array(s, GT1, 'diffusion')
            call non_crit_return_work_array(s, xa_save1, 'diffusion')

            call non_crit_return_work_array(s, CD1, 'diffusion')

            deallocate(class, class_chem_id,  class_name)

         end subroutine dealloc
         
         
         subroutine check_xa_sums(ierr)
            integer, intent(out) :: ierr
            integer :: k
            include 'formats'
            do k=1, nz
               if (abs(sum(s% xa(1:species, k)) - 1d0) > 1d-3) then
                  write(*,*) 'k', k
                  write(*,1) 'sum', sum(s% xa(1:species, k))
                  write(*,1) 'sum-1d0', sum(s% xa(1:species, k))-1d0
                  write(*,1) 'abs(sum-1d0)', abs(sum(s% xa(1:species, k))-1d0)
                  do j=1,species
                     write(*,2) 's% xa(j,k)', j, s% xa(j, k)
                  end do
                  ierr = -1
               end if
            end do
         end subroutine check_xa_sums
         
         
         subroutine dump_diffusion_info
            use utils_lib
            use chem_def, only: chem_isos
            integer :: i, k, ierr
            real(dp) :: alfa, rho_face, chiT_face, chiRho_face, &
               dm_dr, gradRho, dlnRho_dm
            
            ierr = 0
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) return
            write(*, *)
            write(*, *) 'write diffusion.data'
            write(*, *)
            open(iounit, file='diffusion.data', action='write', status='replace', iostat=ierr)
            if (ierr /= 0) then
               write(*, *) 'failed to open diffusion dump file'
               call free_iounit(iounit)
               return
            end if
            
            ! args
            write(iounit, '(99i20)') nz, nzlo, nzhi, species, nc, &
               s% diffusion_steps_hard_limit, &
               s% diffusion_nsmooth_typical_charge
            
            do i=1,species
               write(iounit,*) trim(chem_isos% name(s% chem_id(i)))
            end do
            
            write(iounit, '(99i20)') class(1:species)
            
            write(iounit, '(99i20)') chem_isos% Z(s% chem_id(1:species))

            write(iounit, '(99i20)') chem_isos% Z_plus_N(s% chem_id(1:species))

            write(iounit, '(99e22.10)') chem_isos% W(s% chem_id(1:species))
            
            do i=1,nc
               write(iounit, '(a)') trim(chem_isos% name(class_chem_id(i)))
            end do
            
            do i=1,nc
               write(iounit, '(a)') trim(class_name(i))
            end do
            
            if (s% diffusion_calculates_ionization) then
               write(iounit,*) 1
            else
               write(iounit,*) 0
            end if
            
            if (s% diffusion_use_iben_macdonald) then
               write(iounit,*) 1
            else
               write(iounit,*) 0
            end if
            
            if (s% diffusion_screening_for_radaccel) then
               write(iounit,*) 1
            else
               write(iounit,*) 0
            end if
            
            write(iounit, '(99(1pd26.16))') &
               s% mstar, s% dt, &
               s% diffusion_v_max, s% diffusion_gamma_full_on, s% diffusion_gamma_full_off, &
               s% diffusion_T_full_on, s% diffusion_T_full_off, &
               s% diffusion_max_T_for_radaccel, s% diffusion_dt_div_timescale, &
               s% R_center
               
            do k=1, nz
               write(iounit, '(99(1pd26.16))') &
                  gamma(k), s% abar(k), s% ye(k), free_e(k), s% dm_bar(k), s% dm(k), &
                  s% T(k), s% lnT(k), s% Rho(k), s% lnd(k), s% rmid(k), &
                  dlnP_dm_mid(k), dlnT_dm_mid(k), dlnRho_dm_mid(k), &
                  s% L(k), s% r(k), dlnP_dm_face(k), dlnT_dm_face(k), dlnRho_dm_face(k)
               write(iounit, '(99(1pd26.16))') s% xa(1:species, k)
            end do
            
            close(iounit)
            call free_iounit(iounit)
            
         end subroutine dump_diffusion_info
         
         
         subroutine set_extras(ierr)
            integer, intent(out) :: ierr
            integer :: k, op_err
            op_err = 0
            ierr = 0
!$OMP PARALLEL DO PRIVATE(k, op_err)
            do k=1,nz
               call set1_extras(k, op_err)
               if (op_err /= 0) ierr = op_err
            end do
!$OMP END PARALLEL DO
         end subroutine set_extras
         
         
         subroutine set1_extras(k,ierr)
            use eos_lib, only: Plasma_Coupling_Parameter
            use hydro_eqns, only: eval_dlnPdm_qhse
            integer, intent(in) :: k
            integer, intent(out) :: ierr
            real(dp) :: &
               m, d_dlnPdm_dlnR, d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
               d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, d_dlnPdm_dL, &
               P, dP_dlnd00, dP_dlndm1, dP_dlnT00, dP_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
               dP_dlnPgas00_const_T, dP_dlnPgasm1_const_T, &
               dP_dlnT00_const_Pgas, dP_dlnTm1_const_Pgas
            ierr = 0
            free_e(k) = exp_cr(s% lnfree_e(k))
            gamma(k) = Plasma_Coupling_Parameter(s% T(k), s% rho(k), s% abar(k), s% zbar(k))
            if (k==1) then
               dlnP_dm(k) = 0; dlnT_dm(k) = 0; return
            end if
            ! the following are needed by eval_dlnPdm_qhse but haven't been set yet
            s% mass_correction_start(k) = s% mass_correction(k)
            s% P_div_rho_start(k) = s% P(k)/s% rho(k)
            call eval_dlnPdm_qhse(s, k, m, &
               dlnP_dm(k), d_dlnPdm_dlnR, d_dlnPdm_dL, &
               d_dlnPdm_dlnd00, d_dlnPdm_dlnT00, &
               d_dlnPdm_dlndm1, d_dlnPdm_dlnTm1, &
               d_dlnPdm_dlnPgas00_const_T, d_dlnPdm_dlnT00_const_Pgas, &
               d_dlnPdm_dlnPgasm1_const_T, d_dlnPdm_dlnTm1_const_Pgas, &
               P, &
               dP_dlnd00, dP_dlndm1, dP_dlnT00, dP_dlnTm1, &
               dP_dlnPgas00_const_T, dP_dlnPgasm1_const_T, &
               dP_dlnT00_const_Pgas, dP_dlnTm1_const_Pgas, &
               ierr)
            if (ierr /= 0) return
            dlnT_dm(k) = s% gradT(k)*dlnP_dm(k)
         end subroutine set1_extras


      end function do_element_diffusion
            

      end module element_diffusion







         
         



