! ***********************************************************************
!
!   Copyright (C) 2014  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module dirk

      use star_private_def
      use const_def
      use utils_lib, only: is_bad_num

      implicit none


      contains
      
      
      integer function do_sdirk32( &
            s, itermin, nvar, do_chem, &
            skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         use hydro_eqns, only: eval_equ
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp), parameter :: &
            g = 0.43586652d0, &
            a21 = 0.28206674d0, &
            a31 = 1.2084966d0, &
            a32 = -0.64436317d0, &
            d1 = 0.77263013d0, &
            d2 = 1d0 - d1, &             
            g2 = g*g, &            
            d21 = a21/g, &
            d20 = 1d0 - d21, &
            d31 = -a21*a32/g2 + a31/g, &
            d32 = a32/g, &
            d30 = 1d0 - (d31 + d32), &
            e1 = -a21*d2/g2 + d1/g, &
            e2 = d2/g, &
            e0 = 1d0 - (e1 + e2)

         integer :: ierr, j, k, nz, species, nvar_hydro
         real(dp) :: dt_stage, tol_correction_norm, tol_max_correction
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,*) 'do_sdirk32'
         
         do_sdirk32 = terminate
         ierr = 0
         max_order = 3
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         dt_stage = dt*g ! same for all stages
         s% dVARdot_dVAR = 1/dt_stage

         call set_tol_correction(s, s% T(nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
                  
         call do_initial_setup( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, ierr)
         if (ierr /= 0) return
         ! at this point, s% xh0 holds starting s% xh
         ! and s% xa0 has starting s% xa.
         ! s% xh_pre and s% xa_pre have been set.

         ! 1st stage
         do_sdirk32 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_sdirk32 /= keep_going) return 
         s% num_newton_iters_stage1 = s% num_newton_iterations
         ! at this point (s% xh, s% xa) hold 1st stage results
         
         ! 2nd stage         
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh1, s% xa1, s% xh1_1, s% xa1_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh1(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = d21*s% xh1(j,k) + d20*s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa1(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = d21*s% xa1(j,k) + d20*s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         do_sdirk32 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_sdirk32 /= keep_going) return   
         s% num_newton_iters_stage2 = s% num_newton_iterations
         ! at this point (s% xh, s% xa) hold 2nd stage results
         
         ! 3rd stage         
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh2, s% xa2, s% xh2_1, s% xa2_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh2(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  d32*s% xh2(j,k) + d31*s% xh1(j,k) + d30*s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa2(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  d32*s% xa2(j,k) + d31*s% xa1(j,k) + d30*s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         do_sdirk32 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_sdirk32 /= keep_going) return   
         s% num_newton_iters_stage3 = s% num_newton_iterations
         ! at this point (s% xh, s% xa) hold 3rd stage results
         
         call ebdf_set_vars(s, .false., dt, ierr)
         if (ierr /= 0) return
                  
         ! for local truncation error estimates
         s% xh_compare => s% xh0
         do k=1,nz
            do j=1,nvar_hydro
               s% xh_compare(j,k) = &
                  e2*s% xh2(j,k) + e1*s% xh1(j,k) + e0*s% xh0(j,k)
            end do
         end do
         
         if (do_chem) then
            s% xa_compare => s% xa0
            do k=1,nz
               do j=1,species
                  s% xa_compare(j,k) = &
                     e2*s% xa2(j,k) + e1*s% xa1(j,k) + e0*s% xa0(j,k)
               end do
            end do
         end if

         do_sdirk32 = keep_going

      end function do_sdirk32



      ! 5(4) 5 stage SDIRK
      ! singly diagonally implicit Runge-Kutta.
      ! order 5 with embedded 4th order solution.
      ! L stable, stiffly accurate.
      ! G. Wanner and E. Hairer, 
      ! Solving Ordinary Differential Equations II, 2nd ed.
      ! Springer, New York, 1993.
      
      ! S.K. Burger and W. Yang,
      ! J. Chem. Phys. 125, 244108, 2006.
      ! Automatic integration of the reaction path
      ! using diagonally implicit Runge-Kutta methods
      
      integer function do_sdirk54( &
            s, itermin, nvar, do_chem, &
            skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         use hydro_eqns, only: eval_equ
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         real(dp), parameter :: &
            g = 4d0/15d0, &
            a21 = 0.5d0, &
            a31 = 0.35415395d0, &
            a32 = -0.054153953d0, &
            a41 = 0.085154941d0, &
            a42 = -0.064843323d0, &
            a43 = 0.079153253d0, &
            a51 = 2.1001157d0, &
            a52 = -0.76778003d0, &
            a53 = 2.3998164d0, &
            a54 = -2.9988187d0, &
            d1 = 2.8852642d0, &
            d2 = -0.14587935d0, &
            d3 = 2.3900087d0, &
            d4 = -4.1293935d0, &     
            g2 = g*g, &
            g3 = g*g2, &
            g4 = g*g3, &
            d21 = a21/g, &
            d20 = 1d0 - d21, &
            d31 = -a21*a32/g2 + a31/g, &
            d32 = a32/g, &
            d30 = 1d0 - (d31 + d32), &
            d41 = a21*a32*a43/g3 - (a21*a42 + a31*a43)/g2 + a41/g, &
            d42 = -a32*a43/g2 + a42/g, &
            d43 = a43/g, &
            d40 = 1d0 - (d41 + d42 + d43), &
            d51 = -a21*a32*a43*a54/g4 + &
                  (a21*a32*a53 + a21*a42*a54 + a31*a43*a54)/g3 - &
                  (a21*a52 + a31*a53 + a41*a54)/g2 + a51/g, &
            d52 = a32*a43*a54/g3 - (a32*a53 + a42*a54)/g2 + a52/g, &
            d53 = -a43*a54/g2 + a53/g, &
            d54 = a54/g, &
            d50 = 1d0 - (d51 + d52 + d53 + d54), &
            e1 = -a21*a32*a43*d4/g4 + &
                  (a21*a32*d3 + a21*a42*d4 + a31*a43*d4)/g3 - &
                  (a21*d2 + a31*d3 + a41*d4)/g2 + d1/g, &
            e2 = a32*a43*d4/g3 - (a32*d3 + a42*d4)/g2 + d2/g, &
            e3 = -a43*d4/g2 + d3/g, &
            e4 = d4/g, &
            e0 = 1d0 - (e1 + e2 + e3 + e4)

         integer :: ierr, j, k, nz, species, nvar_hydro
         real(dp) :: dt_stage, tol_correction_norm, tol_max_correction
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,*) 'do_sdirk54'
         
         do_sdirk54 = terminate
         ierr = 0
         max_order = 5
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         dt_stage = dt*g ! same for all stages
         s% dVARdot_dVAR = 1/dt_stage

         call set_tol_correction(s, s% T(nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
                  
         call do_initial_setup( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, ierr)
         if (ierr /= 0) return
         ! at this point, s% xh0 holds starting s% xh
         ! and s% xa0 has starting s% xa.
         ! s% xh_pre and s% xa_pre have been set.

         ! 1st stage
         do_sdirk54 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_sdirk54 /= keep_going) return 
         s% num_newton_iters_stage1 = s% num_newton_iterations
         ! at this point (s% xh, s% xa) hold 1st stage results
         
         ! 2nd stage         
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh1, s% xa1, s% xh1_1, s% xa1_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh1(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = d21*s% xh1(j,k) + d20*s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa1(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = d21*s% xa1(j,k) + d20*s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         do_sdirk54 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_sdirk54 /= keep_going) return   
         s% num_newton_iters_stage2 = s% num_newton_iterations
         ! at this point (s% xh, s% xa) hold 2nd stage results
         
         ! 3rd stage         
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh2, s% xa2, s% xh2_1, s% xa2_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh2(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  d32*s% xh2(j,k) + d31*s% xh1(j,k) + d30*s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa2(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  d32*s% xa2(j,k) + d31*s% xa1(j,k) + d30*s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         do_sdirk54 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_sdirk54 /= keep_going) return   
         s% num_newton_iters_stage3 = s% num_newton_iterations
         ! at this point (s% xh, s% xa) hold 3rd stage results
         
         ! 4th stage         
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh3, s% xa3, s% xh3_1, s% xa3_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh3(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  d43*s% xh3(j,k) + d42*s% xh2(j,k) + &
                  d41*s% xh1(j,k) + d40*s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa3(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  d43*s% xa3(j,k) + d42*s% xa2(j,k) + &
                  d41*s% xa1(j,k) + d40*s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         do_sdirk54 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_sdirk54 /= keep_going) return   
         s% num_newton_iters_stage4 = s% num_newton_iterations
         ! at this point (s% xh, s% xa) hold 4th stage results
         
         ! 5th stage         
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh4, s% xa4, s% xh4_1, s% xa4_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh4(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  d54*s% xh4(j,k) + d53*s% xh3(j,k) + d52*s% xh2(j,k) + &
                  d51*s% xh1(j,k) + d50*s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa4(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  d54*s% xa4(j,k) + d53*s% xa3(j,k) + d52*s% xa2(j,k) + &
                  d51*s% xa1(j,k) + d50*s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         do_sdirk54 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_sdirk54 /= keep_going) return   
         s% num_newton_iters_stage5 = s% num_newton_iterations
         ! at this point (s% xh, s% xa) hold 5th stage results

         call ebdf_set_vars(s, .false., dt, ierr)
         if (ierr /= 0) return
                  
         ! for local truncation error estimates
         s% xh_compare => s% xh0
         do k=1,nz
            do j=1,nvar_hydro
               s% xh_compare(j,k) = &
                  e4*s% xh4(j,k) + e3*s% xh3(j,k) + e2*s% xh2(j,k) + &
                  e1*s% xh1(j,k) + e0*s% xh0(j,k)
            end do
         end do
         
         if (do_chem) then
            s% xa_compare => s% xa0
            do k=1,nz
               do j=1,species
                  s% xa_compare(j,k) = &
                     e4*s% xa4(j,k) + e3*s% xa3(j,k) + e2*s% xa2(j,k) + &
                     e1*s% xa1(j,k) + e0*s% xa0(j,k)
               end do
            end do
         end if

         do_sdirk54 = keep_going

      end function do_sdirk54

      
      integer function do_symplectic1( &
            s, itermin, nvar, top_level, do_chem, &
            skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: top_level, do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         integer :: ierr, j, k, nz, species, nvar_hydro
         real(dp) :: theta, dt_stage, tol_correction_norm, tol_max_correction
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,*) 'do_symplectic1'         
         do_symplectic1 = terminate
         ierr = 0
         max_order = 2
         
         theta = 0.5d0 ! note: even using 0.51 causes big increase in conservation errors

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro
         
         ! at this point, s% xh0 holds starting s% xh
         ! and s% xa0 has starting s% xa.
         ! s% xh_pre and s% xa_pre have been set.
         
         dt_stage = dt*theta
         s% dVARdot_dVAR = 1/dt_stage
            
         if (dbg) write(*,*) 'do_symplectic1 1st stage do_hydro_converge'
         do_symplectic1 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic1 /= keep_going) return 
         s% num_newton_iters_stage1 = s% num_newton_iterations

         if (dbg) write(*,*) 'do_symplectic1 store final results'
         ! store final results
         do k=1,nz
            do j=1,nvar_hydro
               s% xh(j,k) = s% xh_pre(j,k) + (s% xh(j,k) - s% xh_pre(j,k))/theta
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa(j,k) = s% xa_pre(j,k) + (s% xa(j,k) - s% xa_pre(j,k))/theta
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic1 final call ebdf_set_vars'
         call ebdf_set_vars(s, .false., dt, ierr)
         if (ierr /= 0) return

         call store_results_by_stage(s,1)
         call store_bdf_average_results(s,1)
         do_symplectic1 = keep_going

      end function do_symplectic1
      

      ! 2nd order 2 stage symplectic DIRK
      
      ! Yn1 = yn + h/4*f(Yn1)
      ! Yn2 = yn + h/2*f(Yn1) + h/4*f(Yn2)
      ! y_(n+1) = yn + h/2*f(Yn1) + h/2*f(Yn2)

      ! f(Yn1) = (Yn1 - yn)/(h/4)
      ! f(Yn2) = (Yn2 - (2*Yn1 - yn))/(h/4)
      ! y_(n+1) = yn + 2*(Yn2 - Yn1)

      ! stage times c_i*dt for Yn_i
      ! c1 = 0.25
      ! c2 = 0.75
      
      integer function do_symplectic2( &
            s, itermin, nvar, top_level, do_chem, &
            skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: top_level, do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         integer :: ierr, j, k, nz, species, nvar_hydro
         real(dp) :: dt_stage, tol_correction_norm, tol_max_correction
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,*) 'do_symplectic2'         
         do_symplectic2 = terminate
         ierr = 0
         max_order = 2

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro

         if (top_level .and. s% use_truncation_ratio_limit) then 
            ! at this point (xh,xa) have starting quess and (xh_pre,xa_pre) have starting model
            if (dbg) write(*,*) 'do_symplectic2 do_stage_bdf1'
            do_symplectic2 = do_symplectic1( &
               s, itermin, nvar, .false., do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            if (do_symplectic2 /= keep_going) return            
            if (dbg) write(*,*) 'do_symplectic2 save_for_comparison'            
            call save_for_comparison(s, nz, nvar_hydro, species, do_chem, ierr)
            if (ierr /= 0) return
         end if
         
         call do_initial_setup( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh0, s% xa0, s% xh0_1, s% xa0_1, ierr)
         if (ierr /= 0) return
         
         ! at this point, s% xh0 holds starting s% xh
         ! and s% xa0 has starting s% xa.
         ! s% xh_pre and s% xa_pre have been set.
         
         ! 1st stage
         dt_stage = dt/4
         s% dVARdot_dVAR = 1/dt_stage
            
         if (dbg) write(*,*) 'do_symplectic2 1st stage do_hydro_converge'
         do_symplectic2 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic2 /= keep_going) return 
         s% num_newton_iters_stage1 = s% num_newton_iterations
         call store_results_by_stage(s,1)
         
         ! 2nd stage

         ! at this point (s% xh, s% xa) hold 1st stage results
            
         if (dbg) write(*,*) 'do_symplectic2 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh1, s% xa1, s% xh1_1, s% xa1_1, ierr)
         if (ierr /= 0) return         
         
         do k=1,nz
            do j=1,nvar_hydro
               s% xh1(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = 2d0*s% xh1(j,k) - s% xh0(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa1(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = 2d0*s% xa1(j,k) - s% xa0(j,k)
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic2 2nd stage do_hydro_converge'         
         do_symplectic2 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic2 /= keep_going) return   
         s% num_newton_iters_stage2 = s% num_newton_iterations
         call store_results_by_stage(s,2)
         call store_bdf_average_results(s,2)
         
         ! at this point (s% xh, s% xa) hold 2nd stage results

         if (dbg) write(*,*) 'do_symplectic2 store final results'
         ! store final results
         do k=1,nz
            do j=1,nvar_hydro
               s% xh(j,k) = s% xh0(j,k) + 2d0*(s% xh(j,k) - s% xh1(j,k))
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa(j,k) = s% xa0(j,k) + 2d0*(s% xa(j,k) - s% xa1(j,k))
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic2 final call ebdf_set_vars'
         call ebdf_set_vars(s, .false., dt, ierr)
         if (ierr /= 0) return

         do_symplectic2 = keep_going

      end function do_symplectic2
      





      ! 2nd order 3 stage symplectic DIRK
      
      ! Yn1 = yn + h/6*f(Yn1)
      ! Yn2 = yn + h/3*f(Yn1) + h/6*f(Yn2)
      ! Yn3 = yn + h/3*f(Yn1) + h/3*f(Yn2) + h/6*f(Yn3)     
      ! y_(n+1) = yn + h/3*f(Yn1) + h/3*f(Yn2) + h/3*f(Yn3)

      ! f(Yn1) = (Yn1 - yn)/(h/6)      
      ! f(Yn2) = (Yn2 - (2*Yn1 - yn))/(h/6)      
      ! f(Yn3) = (Yn3 - (2*(Yn2 - Yn1) + yn))/(h/6)
      ! y_(n+1) = 2*(Yn3 - Yn2 + Yn1) - yn

      ! stage times c_i*dt for Yn_i
      ! c1 = 1/6
      ! c2 = 3/6
      ! c3 = 5/6
      
      integer function do_symplectic_3stage( &
            s, itermin, nvar, top_level, do_chem, &
            skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: top_level, do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         integer :: ierr, j, k, nz, species, nvar_hydro
         real(dp) :: dt_stage, tol_correction_norm, tol_max_correction
         
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,*) 'do_symplectic_3stage'         
         do_symplectic_3stage = terminate
         ierr = 0
         max_order = 2

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro

         if (top_level .and. s% use_truncation_ratio_limit) then 

            
            if (dbg) write(*,*) 'do_symplectic_3stage call do_symplectic1'
            do_symplectic_3stage = do_symplectic2( &
               s, itermin, nvar, .false., do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            
            if (do_symplectic_3stage /= keep_going) return            
            if (dbg) write(*,*) 'do_symplectic_3stage save_for_comparison'            
            call save_for_comparison(s, nz, nvar_hydro, species, do_chem, ierr)
            if (ierr /= 0) return

         else
            call do_initial_setup( &
               s, nz, nvar_hydro, species, do_chem, &
               s% xh0, s% xa0, s% xh0_1, s% xa0_1, ierr)
            if (ierr /= 0) return
         end if
         
         ! at this point, s% xh0 holds starting s% xh
         ! and s% xa0 has starting s% xa.
         ! s% xh_pre and s% xa_pre have been set.
         
         ! 1st stage
         dt_stage = dt/6
         s% dVARdot_dVAR = 1/dt_stage
            
         if (dbg) write(*,*) 'do_symplectic_3stage 1st stage do_hydro_converge'
         do_symplectic_3stage = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic_3stage /= keep_going) return 
         s% num_newton_iters_stage1 = s% num_newton_iterations
         call store_results_by_stage(s,1)
         
         ! 2nd stage
         ! at this point (s% xh, s% xa) hold 1st stage results
            
         if (dbg) write(*,*) 'do_symplectic_3stage 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh1, s% xa1, s% xh1_1, s% xa1_1, ierr)
         if (ierr /= 0) return         
         
         do k=1,nz
            do j=1,nvar_hydro
               s% xh1(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = 2d0*s% xh1(j,k) - s% xh0(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa1(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = 2d0*s% xa1(j,k) - s% xa0(j,k)
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic_3stage 2nd stage do_hydro_converge'         
         do_symplectic_3stage = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic_3stage /= keep_going) return   
         s% num_newton_iters_stage2 = s% num_newton_iterations
         call store_results_by_stage(s,2)
         
         ! 3rd stage
         ! at this point (s% xh, s% xa) hold 2nd stage results
            
         if (dbg) write(*,*) 'do_symplectic_3stage 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh2, s% xa2, s% xh2_1, s% xa2_1, ierr)
         if (ierr /= 0) return         
         
         do k=1,nz
            do j=1,nvar_hydro
               s% xh2(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = s% xh0(j,k) + 2d0*(s% xh(j,k) - s% xh1(j,k))
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa2(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = s% xa0(j,k) + 2d0*(s% xa(j,k) - s% xa1(j,k))
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic_3stage 2nd stage do_hydro_converge'         
         do_symplectic_3stage = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic_3stage /= keep_going) return   
         s% num_newton_iters_stage3 = s% num_newton_iterations
         call store_results_by_stage(s,3)
         call store_bdf_average_results(s,3)
         
         ! at this point (s% xh, s% xa) hold 3rd stage results

         if (dbg) write(*,*) 'do_symplectic_3stage store final results'
         ! store final results
         do k=1,nz
            do j=1,nvar_hydro
               s% xh(j,k) = &
                  2d0*(s% xh(j,k) - s% xh2(j,k) + s% xh1(j,k)) - s% xh0(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa(j,k) = &
                  2d0*(s% xa(j,k) - s% xa2(j,k) + s% xa1(j,k)) - s% xa0(j,k)
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic_3stage final call ebdf_set_vars'
         call ebdf_set_vars(s, .false., dt, ierr)
         if (ierr /= 0) return

         do_symplectic_3stage = keep_going

      end function do_symplectic_3stage


      ! 3rd order 3 stage symplectic DIRK

      ! G.J. Cooper, 
      ! Stability of Runge-Kutta Methods for Trajectory Problems,
      ! IMA Journal of Numerical Analysis, 7:1-13, 1897.

      ! Yn1 = yn + h*a/2*f(Yn1)
      ! Yn2 = yn + h*a*f(Yn1) + h*a/2*f(Yn2)
      ! Yn3 = yn + h*a*f(Yn1) + h*a*f(Yn2) + h*(1/2 - a)*f(Yn3)
      ! y_(n+1) = yn + h*a*f(Yn1) + h*a*f(Yn2) + h*(1 - 2*a)*f(Yn3)

      ! f(Yn1) = (Yn1 - yn)/(h*a/2)
      ! f(Yn2) = (Yn2 - (2*Yn1 - yn))/(h*a/2)
      ! f(Yn3) = (Yn3 - (2*(Yn2 - Yn1) + yn))/(h*(1/2 - a))
      ! y_(n+1) = 2*(Yn3 - Yn2 + Yn1) - yn

      ! a = real root of 6*x^3 - 12*x^2 + 6*x - 1 = 1.3512071920

      ! stage times c_i*dt for Yn_i
      ! c1 = 0.675604
      ! c2 = 2.02681
      ! c3 = 1.85121
      ! Note that this uses future times well beyond end of timestep.
      ! Then combines future and past results to get the final result.
      
      integer function do_symplectic3( &
            s, itermin, nvar, top_level, do_chem, &
            skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: top_level, do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         integer :: ierr, j, k, nz, species, nvar_hydro
         real(dp) :: dt_stage, tol_correction_norm, tol_max_correction
         real(dp), parameter :: a = 1.3512071920d0
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,*) 'do_symplectic3'
         
         do_symplectic3 = terminate
         ierr = 0
         max_order = 3

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro

         if (top_level .and. s% use_truncation_ratio_limit) then 
            ! at this point (xh,xa) have starting quess and (xh_pre,xa_pre) have starting model
            if (.true.) then
      
            s% use_energy_conservation_form = .true.
            if (dbg) write(*,*) 'do_symplectic3 do_stage_bdf1'
            do_symplectic3 = do_stage_bdf1( &
               s, itermin, nvar, nz, nvar_hydro, species, &
               do_chem, skip_global_corr_coeff_limit, &
               dt, tol_correction_norm, tol_max_correction, &
               s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
            s% use_energy_conservation_form = .false.
            
            else
            
            if (dbg) write(*,*) 'do_symplectic3 call do_symplectic2'
            do_symplectic3 = do_symplectic2( &
               s, itermin, nvar, .false., do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            
            end if
            
            if (do_symplectic3 /= keep_going) return            
            if (dbg) write(*,*) 'do_symplectic3 save_for_comparison'            
            call save_for_comparison(s, nz, nvar_hydro, species, do_chem, ierr)
            if (ierr /= 0) return
         else
            call do_initial_setup( &
               s, nz, nvar_hydro, species, do_chem, &
               s% xh0, s% xa0, s% xh0_1, s% xa0_1, ierr)
            if (ierr /= 0) return
         end if
         
         ! at this point, s% xh0 holds starting s% xh
         ! and s% xa0 has starting s% xa.
         ! s% xh_pre and s% xa_pre have been set.
         
         ! 1st stage
         dt_stage = dt*a/2
         s% dVARdot_dVAR = 1/dt_stage
            
         if (dbg) write(*,*) 'do_symplectic3 1st stage do_hydro_converge'
         do_symplectic3 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic3 /= keep_going) return 
         s% num_newton_iters_stage1 = s% num_newton_iterations
         
         ! 2nd stage
         
         ! at this point (s% xh, s% xa) hold 1st stage results
         if (dbg) write(*,*) 'do_symplectic3 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh1, s% xa1, s% xh1_1, s% xa1_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh1(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = 2d0*s% xh1(j,k) - s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa1(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = 2d0*s% xa1(j,k) - s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         if (dbg) write(*,*) 'do_symplectic3 2nd stage do_hydro_converge'
         do_symplectic3 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic3 /= keep_going) return   
         s% num_newton_iters_stage2 = s% num_newton_iterations
         
         ! at this point (s% xh, s% xa) hold 2nd stage results
         
         ! 3rd stage
         
         ! at this point (s% xh, s% xa) hold 1st stage results
         if (dbg) write(*,*) 'do_symplectic3 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh2, s% xa2, s% xh2_1, s% xa2_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh2(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = 2d0*(s% xh2(j,k) - s% xh1(j,k)) + s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa2(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = 2d0*(s% xa2(j,k) - s% xa1(j,k)) + s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do

         dt_stage = dt*(0.5d0 - a)
         s% dVARdot_dVAR = 1/dt_stage
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         if (dbg) write(*,*) 'do_symplectic3 3rd stage do_hydro_converge'
         do_symplectic3 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic3 /= keep_going) return   
         s% num_newton_iters_stage3 = s% num_newton_iterations
         
         ! at this point (s% xh, s% xa) hold 3rd stage results

         if (dbg) write(*,*) 'do_symplectic3 store final results'
         ! store final results
         do k=1,nz
            do j=1,nvar_hydro
               s% xh(j,k) = &
                  2d0*(s% xh(j,k) - s% xh2(j,k) + s% xh1(j,k)) - s% xh0(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa(j,k) = &
                  2d0*(s% xa(j,k) - s% xa2(j,k) + s% xa1(j,k)) - s% xa0(j,k)
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic3 final call ebdf_set_vars'
         call ebdf_set_vars(s, .false., dt, ierr)
         if (ierr /= 0) return

         do_symplectic3 = keep_going

      end function do_symplectic3





      ! 4th order 5 stage symplectic DIRK

      ! J.M. Franco & I. Gomez, 
      ! Fourth-order symmetric DIRK methods for periodic stiff problems,
      ! Numerical Algorithms, 32:317-336, 2003.

      ! Yn1 = yn + h*b1/2*f(Yn1)
      ! Yn2 = yn + h*b1*f(Yn1) + h*b2/2*f(Yn2)
      ! Yn3 = yn + h*b1*f(Yn1) + h*b2*f(Yn2) + h*b3/2*f(Yn3)
      ! Yn4 = yn + h*b1*f(Yn1) + h*b2*f(Yn2) + h*b3*f(Yn3) + h*b2/2*f(Yn4)
      ! Yn5 = yn + h*b1*f(Yn1) + h*b2*f(Yn2) + h*b3*f(Yn3) + h*b2*f(Yn4) + h*b1/2*f(Yn5)
      ! y_(n+1) = yn + h*b1*f(Yn1) + h*b2*f(Yn2) + h*b3*f(Yn3) + h*b2*f(Yn4) + h*b1*f(Yn5)

      ! f(Yn1) = (Yn1 - yn)/(h*b1/2)
      ! f(Yn2) = (Yn2 - (2*Yn1 - yn))/(h*b2/2)
      ! f(Yn3) = (Yn3 - (2*(Yn2 - Yn1) + yn))/(h*b3/2)
      ! f(Yn4) = (Yn4 - (2*(Yn3 - Yn2 + Yn1) - yn))/(h*b2/2)
      ! f(Yn5) = (Yn5 - (2*(Yn4 - Yn3 + Yn2 - Yn1) + yn))/(h*b1/2)
      ! y_(n+1) = 2*(Yn5 - Yn4 + Yn3 - Yn2 + Yn1) - yn

      ! b1 = 1.45222305916765
      ! b2 = −2.150611289942164
      ! b3 = 2.396776461549028

      ! stage times c_i*dt for Yn_i
      ! c1 = 0.726112
      ! c2 = 0.376917
      ! c3 = 0.5
      ! c4 = 0.623083
      ! c5 = 0.273888

      integer function do_symplectic4( &
            s, itermin, nvar, top_level, do_chem, &
            skip_global_corr_coeff_limit, dt, max_order)
         ! return keep_going, retry, backup, or terminate
         use solve_hydro, only: &
            set_tol_correction, set_surf_info, do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar
         logical, intent(in) :: top_level, do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: dt ! for entire step
         integer, intent(out) :: max_order
         
         integer :: ierr, j, k, nz, species, nvar_hydro
         real(dp) :: dt_stage, tol_correction_norm, tol_max_correction
         real(dp), parameter :: &
            b1 = 1.45222305916765d0, &
            b2 = -2.150611289942164d0, &
            b3 = 2.396776461549028d0
         logical, parameter :: dbg = .false.
         
         include 'formats'
         
         if (dbg) write(*,*) 'do_symplectic4'
         
         do_symplectic4 = terminate
         ierr = 0
         max_order = 4

         call set_tol_correction(s, s% T(s% nz), &
            tol_correction_norm, tol_max_correction)
         call set_surf_info(s)
         
         nz = s% nz
         species = s% species
         nvar_hydro = s% nvar_hydro

         if (top_level .and. s% use_truncation_ratio_limit) then 
            ! at this point (xh,xa) have starting quess and (xh_pre,xa_pre) have starting model
            if (.true.) then
      
            s% use_energy_conservation_form = .true.
            if (dbg) write(*,*) 'do_symplectic4 do_stage_bdf1'
            do_symplectic4 = do_stage_bdf1( &
               s, itermin, nvar, nz, nvar_hydro, species, &
               do_chem, skip_global_corr_coeff_limit, &
               dt, tol_correction_norm, tol_max_correction, &
               s% xh0, s% xa0, s% xh0_1, s% xa0_1, s% num_newton_iters_stage1)
            s% use_energy_conservation_form = .false.
            
            else
            
            if (dbg) write(*,*) 'do_symplectic4 call do_symplectic3'
            do_symplectic4 = do_symplectic3( &
               s, itermin, nvar, .false., do_chem, &
               skip_global_corr_coeff_limit, dt, max_order)
            
            end if
            
            if (do_symplectic4 /= keep_going) return            
            if (dbg) write(*,*) 'do_symplectic4 save_for_comparison'            
            call save_for_comparison(s, nz, nvar_hydro, species, do_chem, ierr)
            if (ierr /= 0) return
         else
            call do_initial_setup( &
               s, nz, nvar_hydro, species, do_chem, &
               s% xh0, s% xa0, s% xh0_1, s% xa0_1, ierr)
            if (ierr /= 0) return
         end if
         
         ! at this point, s% xh0 holds starting s% xh
         ! and s% xa0 has starting s% xa.
         ! s% xh_pre and s% xa_pre have been set.
         
         ! 1st stage
         dt_stage = dt*b1/2
         s% dVARdot_dVAR = 1/dt_stage
            
         if (dbg) write(*,*) 'do_symplectic4 1st stage do_hydro_converge'
         do_symplectic4 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic4 /= keep_going) return 
         s% num_newton_iters_stage1 = s% num_newton_iterations
         
         ! 2nd stage
         dt_stage = dt*b2/2
         s% dVARdot_dVAR = 1/dt_stage
         
         ! at this point (s% xh, s% xa) hold 1st stage results
         if (dbg) write(*,*) 'do_symplectic4 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh1, s% xa1, s% xh1_1, s% xa1_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh1(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = 2d0*s% xh1(j,k) - s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa1(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = 2d0*s% xa1(j,k) - s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         if (dbg) write(*,*) 'do_symplectic4 2nd stage do_hydro_converge'
         do_symplectic4 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic4 /= keep_going) return   
         s% num_newton_iters_stage2 = s% num_newton_iterations
         
         ! at this point (s% xh, s% xa) hold 2nd stage results
         
         ! 3rd stage
         dt_stage = dt*b3/2
         s% dVARdot_dVAR = 1/dt_stage
         
         ! at this point (s% xh, s% xa) hold 2nd stage results
         if (dbg) write(*,*) 'do_symplectic4 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh2, s% xa2, s% xh2_1, s% xa2_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh2(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = 2d0*(s% xh2(j,k) - s% xh1(j,k)) + s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa2(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = 2d0*(s% xa2(j,k) - s% xa1(j,k)) + s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         if (dbg) write(*,*) 'do_symplectic4 3rd stage do_hydro_converge'
         do_symplectic4 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic4 /= keep_going) return   
         s% num_newton_iters_stage3 = s% num_newton_iterations

         ! 4th stage
         dt_stage = dt*b2/2
         s% dVARdot_dVAR = 1/dt_stage
         
         ! at this point (s% xh, s% xa) hold 3rd stage results
         if (dbg) write(*,*) 'do_symplectic4 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh3, s% xa3, s% xh3_1, s% xa3_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh3(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  2d0*(s% xh3(j,k) - s% xh2(j,k) + s% xh1(j,k)) - s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa3(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  2d0*(s% xa3(j,k) - s% xa2(j,k) + s% xa1(j,k)) - s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
         
         call ebdf_set_vars(s, .false., dt_stage, ierr)
         if (ierr /= 0) return

         if (dbg) write(*,*) 'do_symplectic4 4th stage do_hydro_converge'
         do_symplectic4 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic4 /= keep_going) return   
         s% num_newton_iters_stage4 = s% num_newton_iterations

         ! 5th stage
         dt_stage = dt*b1/2
         s% dVARdot_dVAR = 1/dt_stage
         
         ! at this point (s% xh, s% xa) hold 4th stage results
         if (dbg) write(*,*) 'do_symplectic4 1st stage after do_hydro_converge'
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            s% xh4, s% xa4, s% xh4_1, s% xa4_1, ierr)
         if (ierr /= 0) return         

         do k=1,nz
            do j=1,nvar_hydro
               s% xh4(j,k) = s% xh(j,k)
               s% xh_pre(j,k) = &
                  2d0*(s% xh4(j,k) - s% xh3(j,k) + &
                        s% xh2(j,k) - s% xh1(j,k)) + s% xh0(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa4(j,k) = s% xa(j,k)
               s% xa_pre(j,k) = &
                  2d0*(s% xa4(j,k) - s% xa3(j,k) + &
                        s% xa2(j,k) - s% xa1(j,k)) + s% xa0(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic4 5th stage do_hydro_converge'
         do_symplectic4 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_symplectic4 /= keep_going) return   
         s% num_newton_iters_stage5 = s% num_newton_iterations
         
         ! at this point (s% xh, s% xa) hold 5th stage results
         if (dbg) write(*,*) 'do_symplectic4 store final results'
         do k=1,nz
            do j=1,nvar_hydro
               s% xh(j,k) = &
                  2d0*(s% xh(j,k) - s% xh4(j,k) + s% xh3(j,k) - &
                           s% xh2(j,k) + s% xh1(j,k)) - s% xh0(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa(j,k) = &
                  2d0*(s% xa(j,k) - s% xa4(j,k) + s% xa3(j,k) - &
                           s% xa2(j,k) + s% xa1(j,k)) - s% xa0(j,k)
            end do
         end do

         if (dbg) write(*,*) 'do_symplectic4 final call ebdf_set_vars'
         call ebdf_set_vars(s, .false., dt, ierr)
         if (ierr /= 0) return

         do_symplectic4 = keep_going

      end function do_symplectic4


      subroutine save_for_comparison(s, nz, nvar_hydro, species, do_chem, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nvar_hydro, species
         logical, intent(in) :: do_chem
         integer, intent(out) :: ierr
         integer :: k, j
         !real(dp), pointer :: p1(:), p2(:,:)
         
         
! NOTE: change to swap pointers xh and xh_compare so xh_compare has final results 
! then copy xh_pre to xh so xh has starting model
 
         
         ierr = 0
         call do_ebdf_alloc(s, nz, nvar_hydro, species, do_chem, &
               s% xh_compare, s% xa_compare, s% xh_compare_1, s% xa_compare_1, ierr)
         if (ierr /= 0) return
         do k=1,nz ! save the results and reset the starting guess
            do j=1,nvar_hydro
               s% xh_compare(j,k) = s% xh(j,k)
               s% xh(j,k) = s% xh_pre(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               s% xa_compare(j,k) = s% xa(j,k)
               s% xa(j,k) = s% xa_pre(j,k)
            end do
         end do
      end subroutine save_for_comparison


      subroutine do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            xh, xa, xh_1, xa_1, ierr)
         use utils_lib, only: realloc_if_needed_1
         type (star_info), pointer :: s
         real(dp), pointer, dimension(:,:) :: xh, xa
         real(dp), pointer, dimension(:) :: xh_1, xa_1
         integer, intent(in) :: nz, nvar_hydro, species
         logical, intent(in) :: do_chem
         integer, intent(out) :: ierr
         ierr = 0
         call realloc_if_needed_1(xh_1, nvar_hydro*nz, nvar_hydro*10, ierr)
         if (ierr /= 0) return
         xh(1:nvar_hydro,1:nz) => xh_1(1:nvar_hydro*nz)
         if (.not. do_chem) return
         call realloc_if_needed_1(xa_1, species*nz, species*10, ierr)
         if (ierr /= 0) return
         xa(1:species,1:nz) => xa_1(1:species*nz)
      end subroutine do_ebdf_alloc
      
      
      subroutine ebdf_redo_mix_coeffs(s,ierr)
         use hydro_vars, only: set_hydro_vars
         use mix_info, only: get_convection_sigmas
         type (star_info), pointer :: s
         integer, intent(out) :: ierr    
         logical, parameter :: &
            skip_basic_vars = .true., skip_micro_vars = .true., skip_m_grav_and_grav = .true., &
            skip_net = .true., skip_neu = .true., skip_kap = .true., skip_brunt = .true., &
            skip_grads = .true., skip_rotation = .true., skip_other_cgrav = .true., &
            skip_mixing_info = .false., skip_mlt = .false.
         ierr = 0
         call set_hydro_vars( &
            s, 1, s% nz, skip_basic_vars, skip_micro_vars, skip_m_grav_and_grav, &
            skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, &
            skip_brunt, skip_other_cgrav, skip_mixing_info, skip_mlt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'set_hydro_vars failed'
            return
         end if
         call get_convection_sigmas(s, s% dt, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'get_convection_sigmas failed'
            return
         end if
      end subroutine ebdf_redo_mix_coeffs
      
      
      subroutine ebdf_set_vars(s, do_mixing_coeffs, dt_stage, ierr)
         use hydro_vars, only: set_some_vars
         use mix_info, only: get_convection_sigmas
         type (star_info), pointer :: s
         logical, intent(in) :: do_mixing_coeffs
         real(dp), intent(in) :: dt_stage
         integer, intent(out) :: ierr    
         logical, parameter :: &
            skip_time_derivatives = .false., &
            skip_m_grav_and_grav = .false., &
            skip_net = .false., &
            skip_neu = .false., &
            skip_kap = .false., &
            skip_grads = .false., &
            skip_rotation = .false., &
            skip_brunt = .false.
         logical :: skip_mixing
         ierr = 0
         skip_mixing = .not. do_mixing_coeffs
         call set_some_vars( &
            s, skip_time_derivatives, skip_m_grav_and_grav, &
            skip_net, skip_neu, skip_kap, skip_grads, skip_rotation, &
            skip_brunt, skip_mixing, dt_stage, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'set_some_vars failed'
            return
         end if
         if (skip_mixing) return
         call get_convection_sigmas(s, dt_stage, ierr)
         if (ierr /= 0) then
            if (s% report_ierr) write(*,*) 'get_convection_sigmas failed'
            return
         end if
      end subroutine ebdf_set_vars
      

      integer function do_stage_bdf1( &
            s, itermin, nvar, nz, nvar_hydro, species, &
            do_chem, skip_global_corr_coeff_limit, &
            h, tol_correction_norm, tol_max_correction, &
            xh0, xa0, xh0_1, xa0_1, num_iters)
         use solve_hydro, only: do_hydro_converge
         type (star_info), pointer :: s
         integer, intent(in) :: itermin, nvar, nz, nvar_hydro, species
         logical, intent(in) :: do_chem, skip_global_corr_coeff_limit
         real(dp), intent(in) :: h, tol_correction_norm, tol_max_correction
         real(dp), pointer, dimension(:,:) :: xh0, xa0
         real(dp), pointer, dimension(:) :: xh0_1, xa0_1
         integer, intent(out) :: num_iters    
         integer :: k, j, ierr
         real(dp) :: dt_stage      
         include 'formats'  
         ierr = 0
         do_stage_bdf1 = terminate
         call do_initial_setup( &
            s, nz, nvar_hydro, species, do_chem, &
            xh0, xa0, xh0_1, xa0_1, ierr)
         if (ierr /= 0) return
         dt_stage = h
         s% dVARdot_dVAR = 1/dt_stage         
         do_stage_bdf1 = &
            do_hydro_converge( &
               s, itermin, nvar, skip_global_corr_coeff_limit, &
               tol_correction_norm, tol_max_correction, dt_stage)
         if (do_stage_bdf1 /= keep_going) return   
         num_iters = s% num_newton_iterations
      end function do_stage_bdf1
      
      
      subroutine do_initial_setup( &
            s, nz, nvar_hydro, species, do_chem, &
            xh0, xa0, xh0_1, xa0_1, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: nz, nvar_hydro, species
         logical, intent(in) :: do_chem
         real(dp), pointer, dimension(:,:) :: xh0, xa0
         real(dp), pointer, dimension(:) :: xh0_1, xa0_1
         integer, intent(out) :: ierr
         integer :: j, k    
         call do_ebdf_alloc( &
            s, nz, nvar_hydro, species, do_chem, &
            xh0, xa0, xh0_1, xa0_1, ierr)
         if (ierr /= 0) return
         do k=1,nz
            do j=1,nvar_hydro
               xh0(j,k) = s% xh(j,k)
            end do
            if (.not. do_chem) cycle
            do j=1,species
               xa0(j,k) = s% xa(j,k)
            end do
         end do
      end subroutine do_initial_setup
      
      
      subroutine store_results_by_stage(s, stage_number)
         use star_utils, only: set_acoustic_L
         type (star_info), pointer :: s
         integer, intent(in) :: stage_number

         integer :: nz
         nz = s% nz
      
         s% eps_nuc_dot_product_dm_by_stage(stage_number) = &
            dot_product(s% dm(1:nz), s% eps_nuc(1:nz))
            
         s% non_nuc_neu_dot_product_dm_by_stage(stage_number) = &
            dot_product(s% dm(1:nz), s% non_nuc_neu(1:nz))
         
         s% irradiation_heat_dot_product_dm_by_stage(stage_number) = &
            dot_product(s% dm(1:nz), s% irradiation_heat(1:nz))
         
         s% extra_heat_dot_product_dm_by_stage(stage_number) = &
            dot_product(s% dm(1:nz), s% extra_heat(1:nz))
         
         s% eps_grav_dot_product_dm_by_stage(stage_number) = &
            dot_product(s% dm(1:nz), s% eps_grav(1:nz))
         
         if (s% use_artificial_viscosity) &
            s% eps_visc_dot_product_dm_by_stage(stage_number) = &
               dot_product(s% dm(1:nz), s% eps_visc(1:nz))
         
         call set_acoustic_L(s)
         s% acoustic_L_by_stage(stage_number) = s% acoustic_L
         s% surface_L_by_stage(stage_number) = s% L(1)
      
      end subroutine store_results_by_stage
      
      
      subroutine store_ebdf_average_results(s,order)
         type (star_info), pointer :: s
         integer, intent(in) :: order

         s% eps_nuc_dot_product_dm_average = avg1(s% eps_nuc_dot_product_dm_by_stage)            
         s% non_nuc_neu_dot_product_dm_average = avg1(s% non_nuc_neu_dot_product_dm_by_stage)         
         s% irradiation_heat_dot_product_dm_average = &
            avg1(s% irradiation_heat_dot_product_dm_by_stage)         
         s% extra_heat_dot_product_dm_average = avg1(s% extra_heat_dot_product_dm_by_stage)         
         s% eps_grav_dot_product_dm_average = avg1(s% eps_grav_dot_product_dm_by_stage)        
         if (s% use_artificial_viscosity) then
            s% eps_visc_dot_product_dm_average = avg1(s% eps_visc_dot_product_dm_by_stage)  
         else
            s% eps_visc_dot_product_dm_average = 0
         end if      
         s% acoustic_L_average = avg1(s% acoustic_L_by_stage)
         s% surface_L_average = avg1(s% surface_L_by_stage)
         
         contains
         
         real(dp) function avg1(p)
            real(dp) :: p(:)
            select case(order)
            case (1)
               avg1 = p(1)
            case (2)
               avg1 = p(3)
            case default
               avg1 = (sum(p(1:order-2)) + p(order+1))/(order-1)
            end select
         end function avg1
      
      end subroutine store_ebdf_average_results
      
      
      subroutine store_bdf_average_results(s,order)
         type (star_info), pointer :: s
         integer, intent(in) :: order
         
         s% eps_nuc_dot_product_dm_average = &
            avg1(s% eps_nuc_dot_product_dm_by_stage)            
         s% non_nuc_neu_dot_product_dm_average = &
            avg1(s% non_nuc_neu_dot_product_dm_by_stage)         
         s% irradiation_heat_dot_product_dm_average = &
            avg1(s% irradiation_heat_dot_product_dm_by_stage)         
         s% extra_heat_dot_product_dm_average = &
            avg1(s% extra_heat_dot_product_dm_by_stage)         
         s% eps_grav_dot_product_dm_average = &
            avg1(s% eps_grav_dot_product_dm_by_stage)        
         if (s% use_artificial_viscosity) then
            s% eps_visc_dot_product_dm_average = &
            avg1(s% eps_visc_dot_product_dm_by_stage)  
         else
            s% eps_visc_dot_product_dm_average = 0
         end if      
         s% acoustic_L_average = avg1(s% acoustic_L_by_stage)
         s% surface_L_average = avg1(s% surface_L_by_stage)
         
         contains
         
         real(dp) function avg1(p)
            real(dp) :: p(:)
            avg1 = sum(p(1:order))/order
         end function avg1
      
      end subroutine store_bdf_average_results


      end module dirk


