! ***********************************************************************
!
!   Copyright (C) 2013  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 

      module gyre_support
      
      use astero_data
      use star_lib
      use star_def
      use const_def
      use utils_lib
      use gyre_lib
      
      implicit none
      
      logical, parameter :: gyre_is_enabled = .true.

      
      contains
      
      
      subroutine init_gyre(ierr)

         use const_def

         integer, intent(out) :: ierr

         ierr = 0

         call gyre_init()

         ! Set constants

         G_GRAVITY = standard_cgrav
         C_LIGHT = clight
         A_RADIATION = crad

         M_SUN = msol
         R_SUN = rsol
         L_SUN = lsol
         
         write(*,*) 'done init_gyre'

      end subroutine init_gyre
      
      
      subroutine do_gyre_get_modes(s, el, gyre_file, non_ad, store_model, ierr)
         type (star_info), pointer :: s
         integer, intent(in) :: el
         character (len=*), intent(in) :: gyre_file
         logical, intent(in) :: non_ad
         logical, intent(in) :: store_model
         integer, intent(out) :: ierr
         integer  :: ipar(1)
         real(dp) :: rpar(1)
         
         integer :: i, time0, time1, clock_rate
         real(dp) :: time
         
         include 'formats'
         
         ierr = 0
         
         if (store_model) then
            call store_model_for_gyre( &
               s, keep_surface_point, add_atmosphere, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in store_model_for_gyre'
               return
            end if
         end if
         
         if (trace_time_in_oscillation_code) then
            call system_clock(time0, clock_rate)
         end if
         
         call gyre_get_modes(el, gyre_file, non_ad, gyre_call_back, ipar, rpar)  
!         write(*,*) 'gyre_file ' // trim(gyre_file)
!         do i=1,100
!            write(*,2) 'extra call on gyre_get_modes', i
!            call gyre_get_modes(gyre_file, null_gyre_call_back, ipar, rpar)  
!         end do
         
         if (trace_time_in_oscillation_code) then
            call system_clock(time1, clock_rate)
            time = dble(time1-time0)/clock_rate
            total_time_in_oscillation_code = total_time_in_oscillation_code + time
            write(*,1) 'time_in_oscillation_code and total', time, total_time_in_oscillation_code
         end if

         !write(*,2) 'done gyre_get_modes ' // trim(gyre_file), s% model_number
         !write(*,*) 
             
      end subroutine do_gyre_get_modes
         

      subroutine null_gyre_call_back(md, ipar, rpar, ierr)         
         type(mode_t), intent(in) :: md
         integer, intent(inout) :: ipar(:)
         real(dp), intent(inout) :: rpar(:)
         integer, intent(out) :: ierr
         ierr = 0
      end subroutine null_gyre_call_back
      
      
      subroutine store_model_for_gyre( &
            s, keep_surface_point, add_atmosphere, ierr)
            
         type (star_info), pointer :: s
         logical, intent(in) :: keep_surface_point, add_atmosphere
         integer, intent(out) :: ierr
         
         real(dp) :: M_star, R_star, L_star
         ! arrays will be allocated by star_get_gyre_info
         ! must deallocate when done with them
         real(dp), dimension(:), pointer :: &
            r, w, L, p, rho, T, &
            N2, Gamma_1, nabla_ad, delta, nabla,  &
            kappa, kappa_rho, kappa_T, &
            epsilon, epsilon_rho, epsilon_T, omega
            
         integer :: k, nn
         character (len=2000) :: format_string, num_string, filename
         
         logical, parameter :: dbg = .false.
            
         include 'formats'
         
         if (write_fgong_for_each_model) then
            write(format_string,'( "(i",i2.2,".",i2.2,")" )') &
               model_num_digits, model_num_digits
            write(num_string,format_string) s% model_number
            filename = trim(fgong_prefix) // trim(num_string) // trim(fgong_postfix)
            call star_write_fgong(s% id, &
               add_center_point, keep_surface_point, &
               add_atmosphere, filename, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in star_write_fgong'
               stop 1
            end if
         end if

         if (dbg) write(*,2) 'call star_get_gyre_info', s% model_number
         call star_get_gyre_info( &
            s% id, keep_surface_point, add_atmosphere, &
            M_star, R_star, L_star, r, w, L, p, rho, T, &
            N2, Gamma_1, nabla_ad, delta, nabla,  &
            kappa, kappa_rho, kappa_T, &
            epsilon, epsilon_rho, epsilon_T, omega, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_get_pulsation_info'
            stop 1
         end if
         if (dbg) write(*,2) 'done star_get_gyre_info', s% model_number
         
         if (dbg) write(*,2) 'call gyre_set_model', s% model_number
         call gyre_set_model( &
            M_star, R_star, L_star, r, w, p, rho, T, &
            N2, Gamma_1, nabla_ad, delta, nabla,  &
            kappa, kappa_rho, kappa_T, &
            epsilon, epsilon_rho, epsilon_T, omega, 'MONO')
         if (dbg) write(*,2) 'done gyre_set_model', s% model_number
         
         if (write_gyre_for_each_model) then
            write(format_string,'( "(i",i2.2,".",i2.2,")" )') &
               model_num_digits, model_num_digits
            write(num_string,format_string) s% model_number
            filename = trim(gyre_prefix) // trim(num_string) // trim(gyre_postfix)
            call star_write_gyre_data_to_file(filename,  &
               M_star, R_star, L_star, &
               r, w, L, p, rho, T, &
               N2, Gamma_1, nabla_ad, delta, nabla,  &
               kappa, kappa_rho, kappa_T, &
               epsilon, epsilon_rho, epsilon_T, omega, &
               max_num_gyre_points, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in star_write_gyre'
               stop 1
            end if
         end if
            
         deallocate( &
            r, w, L, p, rho, T, &
            N2, Gamma_1, nabla_ad, delta, nabla,  &
            kappa, kappa_rho, kappa_T, &
            epsilon, epsilon_rho, epsilon_T, omega)
      
      end subroutine store_model_for_gyre
      

      subroutine gyre_call_back(md, ipar, rpar, ierr)
         use astero_data, only: store_new_oscillation_results
         
         type(mode_t), intent(in) :: md
         integer, intent(inout) :: ipar(:)
         real(dp), intent(inout) :: rpar(:)
         integer, intent(out) :: ierr

         integer :: new_el, new_order, new_em
         real(dp) :: new_inertia, new_cyclic_freq
         
         include 'formats'

         ierr = 0

         new_el = md% mp% l
         new_order = md% n_pg
         new_inertia = md% E_norm()
         new_cyclic_freq = md% freq('UHZ')
         new_em = 0
         
         call store_new_oscillation_results( &
            new_el, new_order, new_em, new_inertia, new_cyclic_freq, ierr)
         
         if (.false.) then
            write(*,'(3a8,99a20)') 'el', 'order', 'em', 'freq (microHz)', 'inertia'
            write(*,'(3i8,f20.10,e20.10,i20)') &
               new_el, new_order, new_em, new_cyclic_freq, new_inertia
            write(*,*)
         end if
            
         call save_gyre_mode_info( &
            new_el, new_order, new_em, new_inertia, new_cyclic_freq, &
            md, ipar, rpar, ierr)

      end subroutine gyre_call_back

      
      subroutine save_gyre_mode_info( &
            new_el, new_order, new_em, new_inertia, new_cyclic_freq, &
            md, ipar, rpar, ierr)
         integer, intent(in) :: new_el, new_order, new_em
         real(dp), intent(in) :: new_inertia, new_cyclic_freq
         type(mode_t), intent(in) :: md
         integer, intent(inout) :: ipar(:)
         real(dp), intent(inout) :: rpar(:)
         integer, intent(out) :: ierr

         integer :: iounit, i, j, skip, nn
         real(dp) :: y_r, y_h
         include 'formats'
         
         !if (use_other_gyre_mode_info) then
         !   call astero_other_procs% other_gyre_mode_info(md, ipar, rpar, ierr)
         !end if
         
         if (star_model_number /= save_mode_model_number) return
         if (new_el /= el_to_save .or. new_order /= order_to_save) return
         
         if (len_trim(save_mode_filename) <= 0) save_mode_filename = 'save_mode.data'
         write(*,*) 'save eigenfunction info to file ' // trim(save_mode_filename)
         write(*,'(3a8,99a20)') 'el', 'order', 'em', 'freq (microHz)', 'inertia'
         write(*,'(3i8,f20.10,e20.10,i20)') &
            new_el, new_order, new_em, new_cyclic_freq, new_inertia
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         open(unit=iounit, file=trim(save_mode_filename), action='write', iostat=ierr)
         if (ierr /= 0) return
         write(iounit,'(3a8,99a20)') 'el', 'order', 'em', 'freq (microHz)', 'inertia'!, 'nn'
         write(iounit,'(3i8,f20.10,e20.10,i20)') &
            new_el, new_order, new_em, new_cyclic_freq, new_inertia

         ! write eigenfunction

         close(iounit)
         call free_iounit(iounit)         
      end subroutine save_gyre_mode_info



      end module gyre_support
