! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   this file is part of mesa.
!
!   mesa is free software; you can redistribute it and/or modify
!   it under the terms of the gnu general library public license as published
!   by the free software foundation; either version 2 of the license, or
!   (at your option) any later version.
!
!   mesa is distributed in the hope that it will be useful, 
!   but without any warranty; without even the implied warranty of
!   merchantability or fitness for a particular purpose.  see the
!   gnu library general public license for more details.
!
!   you should have received a copy of the gnu library general public license
!   along with this software; if not, write to the free software
!   foundation, inc., 59 temple place, suite 330, boston, ma 02111-1307 usa
!
! ***********************************************************************
 
      module extras_support

      use star_lib
      use star_def
      use const_def
      use crlibm_lib
      use utils_lib
      use astero_support
      use astero_data

      implicit none


      contains
      
      
      subroutine get_all_el_info(s, ierr)
         type (star_info), pointer :: s
         integer, intent(out) :: ierr
         logical :: store_model
         store_model = .true.
         ierr = 0
         if (nl0 > 0) then
            call get_one_el_info(s, 0, &
               nu_lower_factor*l0_obs(1), &
               nu_upper_factor*l0_obs(nl0), &
               iscan_factor_l0*nl0, 1, nl0, store_model, &
               oscillation_code, ierr)
            if (ierr /= 0) return
            store_model = .false.
         end if         
         if (nl1 > 0) then
            call get_one_el_info(s, 1, &
               nu_lower_factor*l1_obs(1), &
               nu_upper_factor*l1_obs(nl1), &
               iscan_factor_l1*nl1, 1, nl1, store_model, &
               oscillation_code, ierr)
            if (ierr /= 0) return
            store_model = .false.
         end if        
         if (nl2 > 0) then
            call get_one_el_info(s, 2, &
               nu_lower_factor*l2_obs(1), &
               nu_upper_factor*l2_obs(nl2), &
               iscan_factor_l2*nl2, 1, nl2, store_model, &
               oscillation_code, ierr)
            if (ierr /= 0) return
            store_model = .false.
         end if         
         if (nl3 > 0) then
            call get_one_el_info(s, 3, &
               nu_lower_factor*l3_obs(1), &
               nu_upper_factor*l3_obs(nl3), &
               iscan_factor_l3*nl3, 1, nl3, store_model, &
               oscillation_code, ierr)
            if (ierr /= 0) return
            store_model = .false.
         end if
      end subroutine get_all_el_info
      

      integer function do_astero_extras_check_model(s, id, id_extra)
         use mlt_def, only: convective_mixing

         type (star_info), pointer :: s
         integer, intent(in) :: id, id_extra
         
         integer :: max_el_for_chi2, ierr, i, j, n
         logical :: store_model, checking_age
         real(dp) :: age_limit, model_limit, err, target_l0, X, Y, Z, &
            frac, surface_X, surface_Z, chi2_freq_and_ratios_fraction, &
            remaining_years, prev_max_years, min_max
         
         include 'formats'
         
         do_astero_extras_check_model = keep_going
         astero_max_dt_next = 1d99
         chi2 = -1
         chi2_seismo = -1
         chi2_spectro = -1
         FeH = -1
         delta_nu_model = -1
         nu_max_model = -1
         a_div_r = -1
         correction_r = -1
         checking_age = &
            eval_chi2_at_target_age_only .or. include_age_in_chi2_spectro
         
         if (checking_age) then
            if (num_smaller_steps_before_age_target <= 0 .or. &
                dt_for_smaller_steps_before_age_target <= 0) then
               write(*,*) 'ERROR: must set num_smaller_steps_before_age_target'
               write(*,*) 'and dt_for_smaller_steps_before_age_target'
               stop 1
            end if
            if (age_target > s% star_age) then
               remaining_years = age_target - s% star_age
               if (s% astero_using_revised_max_yr_dt) &
                  s% max_years_for_timestep = s% astero_revised_max_yr_dt
               n = floor(remaining_years/s% max_years_for_timestep + 1d-6)
               j = num_smaller_steps_before_age_target
               if (remaining_years <= s% max_years_for_timestep) then
                  write(*,'(a40,i6,f20.10)') '(age_target - star_age)/age_sigma', &
                     s% model_number, (age_target - s% star_age)/age_sigma
                  s% max_years_for_timestep = remaining_years
                  s% astero_using_revised_max_yr_dt = .true.
                  s% astero_revised_max_yr_dt = s% max_years_for_timestep
                  astero_max_dt_next = s% max_years_for_timestep*secyer
               else if (n <= j) then
                  write(*,'(a40,i6,f20.10)') '(age_target - star_age)/age_sigma', &
                     s% model_number, (age_target - s% star_age)/age_sigma
                  prev_max_years = s% max_years_for_timestep
                  i = floor(remaining_years/dt_for_smaller_steps_before_age_target + 1d-6)
                  if ((i+1d-9)*dt_for_smaller_steps_before_age_target < remaining_years) then
                     s% max_years_for_timestep = remaining_years/(i+1)
                  else
                     s% max_years_for_timestep = remaining_years/i
                  end if
                  min_max = prev_max_years*s% reduction_factor_for_max_timestep
                  if (s% max_years_for_timestep < min_max) &
                     s% max_years_for_timestep = min_max
                  if (.not. s% astero_using_revised_max_yr_dt) then
                     s% astero_using_revised_max_yr_dt = .true.
                     write(*,2) 'begin reducing max timestep prior to age target', &
                        s% model_number, remaining_years
                  else if (s% astero_revised_max_yr_dt > s% max_years_for_timestep) then
                     write(*,2) 'reducing max timestep prior to age target', &
                        s% model_number, remaining_years
                  else if (s% max_years_for_timestep <= dt_for_smaller_steps_before_age_target) then
                     i = floor(remaining_years/s% max_years_for_timestep + 1d-6)
                     write(*,3) 'remaining steps and years until age target', &
                        s% model_number, i, remaining_years
                  else 
                     write(*,2) 'remaining_years until age target', &
                        s% model_number, remaining_years
                  end if
                  s% astero_revised_max_yr_dt = s% max_years_for_timestep
                  if (s% dt_next/secyer > s% max_years_for_timestep) &
                     astero_max_dt_next = s% max_years_for_timestep*secyer
               end if
            else if (include_age_in_chi2_spectro) then
               write(*,'(a40,i6,f20.10)') '(age_target - star_age)/age_sigma', &
                  s% model_number, (age_target - s% star_age)/age_sigma
               if (abs(s% max_years_for_timestep - dt_for_smaller_steps_before_age_target) > &
                     dt_for_smaller_steps_before_age_target*1d-2) then
                  write(*,1) 'dt_for_smaller_steps_before_age_target', &
                     dt_for_smaller_steps_before_age_target
                  write(*,1) 'max_years_for_timestep', &
                     s% max_years_for_timestep
                  stop 'bad max_years_for_timestep'
               end if
            end if
         else
            if (s% star_age < min_age_for_chi2) return
            s% max_years_for_timestep = max_yrs_dt_when_cold
         end if

         if (include_age_in_chi2_spectro .and. s% star_age < min_age_for_chi2) return         
         if (eval_chi2_at_target_age_only .and. s% star_age < age_target) return
         
         delta_nu_model = s% delta_nu
         nu_max_model = s% nu_max
         
         chi2_seismo_delta_nu_fraction = &
            min(1d0, max(0d0, chi2_seismo_delta_nu_fraction))
         chi2_seismo_nu_max_fraction = &
            min(1d0, max(0d0, chi2_seismo_nu_max_fraction))
         chi2_seismo_r_010_fraction = &
            min(1d0, max(0d0, chi2_seismo_r_010_fraction))
         chi2_seismo_r_02_fraction = &
            min(1d0, max(0d0, chi2_seismo_r_02_fraction))
         chi2_seismo_freq_fraction = min(1d0, max(0d0, 1d0 - &
            (chi2_seismo_r_010_fraction + &
               chi2_seismo_r_02_fraction + &
               chi2_seismo_delta_nu_fraction + &
               chi2_seismo_nu_max_fraction)))
         
         if (s% L_nuc_burn_total < s% L_phot*Lnuc_div_L_limit .or. &
               s% star_age < min_age_limit) then
            return
         end if
         
         if (.not. checking_age) then
         
            age_limit = avg_age_top_samples + avg_age_sigma_limit*avg_age_sigma
            if (s% star_age > age_limit) then
               write(*,1) 'star age > limit from top samples', s% star_age, age_limit
               write(*,1) 'avg_age_top_samples', avg_age_top_samples
               write(*,1) 'avg_age_sigma_limit', avg_age_sigma_limit
               write(*,1) 'avg_age_sigma', avg_age_sigma
               do_astero_extras_check_model = terminate
               return
            end if
         
            model_limit = &
               avg_model_number_top_samples + &
                  avg_model_number_sigma_limit*avg_model_number_sigma
            if (dble(s% model_number) > model_limit) then
               write(*,2) 'model number > limit from top samples', &
                  s% model_number, model_limit
               write(*,1) 'avg_model_number_top_samples', avg_model_number_top_samples
               write(*,1) 'avg_model_number_sigma_limit', avg_model_number_sigma_limit
               write(*,1) 'avg_model_number_sigma', avg_model_number_sigma
               do_astero_extras_check_model = terminate
               return
            end if

         end if
         
         surface_X = max(s% surface_h1, 1d-10)
         surface_He = s% surface_he3 + s% surface_he4
         surface_Z = max(1d-99, min(1d0, 1 - (surface_X + surface_He)))
         surface_Z_div_X = surface_Z/surface_X
         FeH = log10_cr((surface_Z_div_X)/Z_div_X_solar)
         logg = log10_cr(s% grav(1))
         logR = log10_cr(s% photosphere_r)
         if (.not. include_Rcz_in_chi2_spectro) then
            Rcz = 0
         else
            do i = 1, s% nz-1 ! locate bottom of solar convective zone
               if (s% mixing_type(i+1) /= convective_mixing &
                     .and. s% mixing_type(i) == convective_mixing) then
                  if (s% r(i+1) > 0.25*Rsun .and. s% r(i) < 0.9*Rsun) then
                     Rcz = s% r(i)/Rsun
                     exit
                  end if
               end if
            end do
         end if
         if (include_csound_rms_in_chi2_spectro .or. report_csound_rms) then
            csound_rms = calc_current_rms(s, s% nz)
         else
            csound_rms = 0
         end if
         
         call check_limits
         if (do_astero_extras_check_model /= keep_going) return

         chi2_spectro = get_chi2_spectro(s)         
         if (is_bad_num(chi2_spectro)) then
            write(*,1) 'bad chi2_spectro', chi2_spectro
            write(*,1) 'FeH', FeH
            write(*,1) 'surface_Z', surface_Z
            write(*,1) 'surface_X', surface_X
            write(*,1) 'Z_div_X_solar', Z_div_X_solar
            chi2_spectro = 1d99
            do_astero_extras_check_model = terminate
            return
            !stop
         end if
         
         have_radial = .false.
         have_nonradial = .false.
         model_ratios_n = 0
         
         l0_freq(1:nl0) = 0
         l1_freq(1:nl1) = 0
         l2_freq(1:nl2) = 0
         l3_freq(1:nl3) = 0
         
         if (delta_nu_sigma > 0) then
            chi2_delta_nu = pow2((delta_nu - delta_nu_model)/delta_nu_sigma)
            if (trace_chi2_seismo_delta_nu_info) &
               write(*,1) 'chi2_delta_nu', chi2_delta_nu
         else
            chi2_delta_nu = 0
         end if
         
         chi2_nu_max = 0
         if (chi2_seismo_nu_max_fraction > 0) then
            if (nu_max <= 0) then
               write(*,2) 'must supply nu_max'
               do_astero_extras_check_model = terminate
               return
            end if         
            if (nu_max_sigma <= 0) then
               write(*,2) 'must supply nu_max_sigma'
               do_astero_extras_check_model = terminate
               return
            end if
            chi2_nu_max = pow2((nu_max - nu_max_model)/nu_max_sigma)
            if (trace_chi2_seismo_nu_max_info) &
               write(*,1) 'chi2_nu_max', chi2_nu_max 
         end if
         
         chi2_freq_and_ratios_fraction = &
            chi2_seismo_freq_fraction + &
            chi2_seismo_r_010_fraction + &
            chi2_seismo_r_02_fraction
            
         if (chi2_seismo_fraction <= 0d0) then
            ! no need to get frequencies
            chi2_seismo = &
               chi2_seismo_delta_nu_fraction*chi2_delta_nu + &
               chi2_seismo_nu_max_fraction*chi2_nu_max
            frac = chi2_seismo_fraction
            chi2 = frac*chi2_seismo + (1-frac)*chi2_spectro         
            write(*,'(a50,i6,99f16.2)') 'chi^2 combined, chi^2 seismo, chi^2 spectro', &
               s% model_number, chi2, chi2_seismo, chi2_spectro
            if (best_chi2 < 0 .or. chi2 < best_chi2) call save_best_info(s)
            if (chi2 < chi2_radial_limit .and. .not. checking_age) &
               s% max_years_for_timestep = max_yrs_dt_when_warm
            call final_checks
            return
         end if

         if (chi2_spectro > chi2_spectroscopic_limit) then
            write(*,'(a50,i6,99f16.2)') 'chi2_spectro > limit', &
                  s% model_number, chi2_spectro, chi2_spectroscopic_limit
            call check_too_many_bad
            return
         end if
         
         if (chi2_delta_nu > chi2_delta_nu_limit) then
            write(*,'(a50,i6,99f16.2)') 'chi2_delta_nu > limit', &
                  s% model_number, chi2_delta_nu, chi2_delta_nu_limit, &
                  delta_nu_model, delta_nu
            call check_too_many_bad
            return
         end if

         ! chi2_spectro <= limit and chi2_delta_nu <= limit        

         if (.not. checking_age) then
            s% max_years_for_timestep = max_yrs_dt_when_warm
            if (s% dt > max_yrs_dt_when_warm*secyer) then
               s% dt = max_yrs_dt_when_warm*secyer
               s% timestep_hold = s% model_number + 10
               write(*,'(a50,i6,1p,99e16.4)') &
                  'redo with smaller timestep for "warm" limit', &
                     s% model_number, max_yrs_dt_when_warm
               do_astero_extras_check_model = redo
               return
            end if            
         end if
         
         if (.not. checking_age) then
            s% max_years_for_timestep = max_yrs_dt_when_hot 
            if (s% dt > max_yrs_dt_when_hot*secyer) then
               s% dt = max_yrs_dt_when_hot*secyer
               s% timestep_hold = s% model_number + 10
               write(*,'(a50,i6,1p,99e16.4)') &
                  'redo with smaller timestep for "hot" limit', &
                  s% model_number, max_yrs_dt_when_hot
               do_astero_extras_check_model = redo
               return
            end if
         end if
         
         store_model = .true.
         if (nl0 > 0 .and. chi2_freq_and_ratios_fraction > 0d0) then
            if (.not. get_radial(oscillation_code)) then
               write(*,'(a65,i6)') 'failed to find all required l=0 modes', s% model_number
               if (trace_chi2_seismo_frequencies_info) then
                  write(*,2) 'results for l=0'
                  i = 0
                  do j = 1, num_results
                     if (el(j) /= 0) cycle
                     i = i+1
                     write(*,2) 'freq', i, cyclic_freq(j)
                  end do
                  write(*,*)
               end if
               call check_too_many_bad
               return 
            end if
            store_model = .false.
            have_radial = .true.
         end if
                  
         !write(*,2) 'nl1', nl1
         if (nl1 > 0 .and. chi2_freq_and_ratios_fraction > 0d0) then
            call get_one_el_info(s, 1, &
               nu_lower_factor*l1_obs(1), &
               nu_upper_factor*l1_obs(nl1), &
               iscan_factor_l1*nl1, 1, nl1, store_model, &
               oscillation_code, ierr)
            if (ierr /= 0) then
               if (trace_chi2_seismo_frequencies_info) write(*,*)
               write(*,'(a65,i6)') 'failed to find all required l=1 modes', s% model_number
               if (trace_chi2_seismo_frequencies_info) then
                  write(*,2) 'results for l=1'
                  i = 0
                  do j = 1, num_results
                     if (el(j) /= 1) cycle
                     i = i+1
                     write(*,2) 'freq', i, cyclic_freq(j)
                  end do
                  write(*,*)
               end if
               call check_too_many_bad
               return
            end if
            store_model = .false.
         end if
         
         !write(*,2) 'nl2', nl2
         if (nl2 > 0 .and. chi2_freq_and_ratios_fraction > 0d0) then
            call get_one_el_info(s, 2, &
               nu_lower_factor*l2_obs(1), &
               nu_upper_factor*l2_obs(nl2), &
               iscan_factor_l2*nl2, 1, nl2, store_model, &
               oscillation_code, ierr)
            if (ierr /= 0) then
               if (trace_chi2_seismo_frequencies_info) write(*,*)
               write(*,'(a65,i6)') 'failed to find all required l=2 modes', s% model_number
               if (trace_chi2_seismo_frequencies_info) then
                  write(*,2) 'results for l=2'
                  i = 0
                  do j = 1, num_results
                     if (el(j) /= 2) cycle
                     i = i+1
                     write(*,2) 'freq', i, cyclic_freq(j)
                  end do
                  write(*,*)
               end if
               call check_too_many_bad
               return
            end if
            store_model = .false.
         end if
         
         !write(*,2) 'nl3', nl3
         if (nl3 > 0 .and. chi2_freq_and_ratios_fraction > 0d0) then
            call get_one_el_info(s, 3, &
               nu_lower_factor*l3_obs(1), &
               nu_upper_factor*l3_obs(nl3), &
               iscan_factor_l3*nl3, 1, nl3, store_model, &
               oscillation_code, ierr)
            if (ierr /= 0) then
               if (trace_chi2_seismo_frequencies_info) write(*,*)
               write(*,'(a65,i6)') 'failed to find all required l=3 modes', s% model_number
               if (trace_chi2_seismo_frequencies_info) then
                  write(*,2) 'results for l=3'
                  i = 0
                  do j = 1, num_results
                     if (el(j) /= 3) cycle
                     i = i+1
                     write(*,2) 'freq', i, cyclic_freq(j)
                  end do
                  write(*,*)
               end if
               call check_too_many_bad
               return
            end if
            store_model = .false.
         end if
         
         have_nonradial = .true.
         
         if (chi2_freq_and_ratios_fraction > 0d0) then
            call get_freq_corr(s, .false., ierr)
            if (ierr /= 0) then
               write(*,'(a65,i6)') 'failed in get_freq_corr', s% model_number
               return
            end if
            if (nl3 > 0) then
               max_el_for_chi2 = 3
            else if (nl2 > 0) then
               max_el_for_chi2 = 2
            else if (nl1 > 0) then
               max_el_for_chi2 = 1
            else
               max_el_for_chi2 = 0
            end if
         else
            max_el_for_chi2 = -1
         end if
         
         !write(*,2) 'max_el_for_chi2', max_el_for_chi2
         if (chi2_seismo_r_010_fraction > 0 .and. max_el_for_chi2 >= 1) then

            call get_frequency_ratios( &
               .false., nl0, l0_freq_corr, nl1, l1_freq_corr, &
               model_ratios_n, model_ratios_l0_first, model_ratios_l1_first, &
               model_ratios_r01, model_ratios_r10)
            
            if (model_ratios_n /= ratios_n .or. &
                  model_ratios_l0_first /= ratios_l0_first .or. &
                     model_ratios_l1_first /= ratios_l1_first) then
               ierr = -1
               write(*,'(a,i6)') 'cannot calculate necessary chi^2 ratios for this model', s% model_number
               if (model_ratios_n /= ratios_n) &
                  write(*,*) '              model_ratios_n /= ratios_n', model_ratios_n, ratios_n
               if (model_ratios_l0_first /= ratios_l0_first) &
                  write(*,*) 'model_ratios_l0_first /= ratios_l0_first', &
                     model_ratios_l0_first, ratios_l0_first
               if (model_ratios_l1_first /= ratios_l1_first) &
                  write(*,*) 'model_ratios_l1_first /= ratios_l1_first', &
                     model_ratios_l1_first, ratios_l1_first
               call check_too_many_bad
               return               
            end if
            
         end if
               
         if (chi2_seismo_r_02_fraction > 0 .and. max_el_for_chi2 >= 2) then
            call get_r02_frequency_ratios( &
               .false., nl0, l0_freq_corr, nl1, l1_freq_corr, nl2, l2_freq_corr, model_ratios_r02)
         end if
         
         chi2 = get_chi2(s, max_el_for_chi2, .true., ierr)
         if (ierr /= 0) then
            write(*,'(a40,i6)') 'failed to calculate chi^2', s% model_number
            call check_too_many_bad
            return
         end if
         write(*,'(a50,i6,99f16.2)') 'chi^2 total, chi^2 radial', &
            s% model_number, chi2, chi2_radial
         
         if (use_other_after_get_chi2) then
            ierr = 0
            call astero_other_procs% other_after_get_chi2(s% id, ierr)
            if (ierr /= 0) then
               do_astero_extras_check_model = terminate
               return
            end if
         end if
          
         if (checking_age) then
            ! leave max_years_for_timestep as is
         else if (chi2 <= chi2_limit_for_smallest_timesteps) then
            s% max_years_for_timestep = max_yrs_dt_chi2_smallest_limit 
            if (s% dt > max_yrs_dt_chi2_smallest_limit*secyer) then
               s% dt = max_yrs_dt_chi2_smallest_limit*secyer
               s% timestep_hold = s% model_number + 10
               write(*,'(a50,i6,2f16.2,1p,99e16.4)') 'redo timestep for "smallest" chi^2 limit', &
                  s% model_number, chi2, chi2_limit_for_smallest_timesteps, &
                  max_yrs_dt_chi2_smallest_limit
               do_astero_extras_check_model = redo
               return
            end if         
         else if (chi2 <= chi2_limit_for_smaller_timesteps) then
            s% max_years_for_timestep = max_yrs_dt_chi2_smaller_limit 
            if (s% dt > max_yrs_dt_chi2_smaller_limit*secyer) then
               s% dt = max_yrs_dt_chi2_smaller_limit*secyer
               s% timestep_hold = s% model_number + 10
               write(*,'(a50,i6,2f16.2,1p,99e16.4)') 'redo timestep for "smaller" chi^2 limit', &
                  s% model_number, chi2, chi2_limit_for_smaller_timesteps, &
                  max_yrs_dt_chi2_smaller_limit
               do_astero_extras_check_model = redo
               return
            end if         
         else if (chi2 <= chi2_limit_for_small_timesteps) then
            s% max_years_for_timestep = max_yrs_dt_chi2_small_limit 
            if (s% dt > max_yrs_dt_chi2_small_limit*secyer) then
               s% dt = max_yrs_dt_chi2_small_limit*secyer
               s% timestep_hold = s% model_number + 10
               write(*,'(a50,i6,2f16.2,1p,99e16.4)') 'redo timestep for "small" chi^2 limit', &
                  s% model_number, chi2, chi2_limit_for_small_timesteps, &
                  max_yrs_dt_chi2_small_limit
               do_astero_extras_check_model = redo
               return
            end if         
         end if
         
         if (best_chi2 <= 0 .or. chi2 < best_chi2) then
            call save_best_info(s)
         end if
            
         call final_checks

         
         contains
         
         
         subroutine check_too_many_bad
            if (best_chi2 > 0) then
               num_chi2_too_big = num_chi2_too_big + 1
               if (num_chi2_too_big > limit_num_chi2_too_big) then
                  write(*,*) 'have reached too many bad chi2 limit'
                  do_astero_extras_check_model = terminate
               end if
               return
            end if
            num_chi2_too_big = 0
         end subroutine check_too_many_bad
         
         
         subroutine final_checks
            if (include_age_in_chi2_spectro .and. s% star_age >= max_age_for_chi2) then
               write(*,*) 'have reached max_age_for_chi2'
               do_astero_extras_check_model = terminate
            end if
            if (eval_chi2_at_target_age_only .and. s% star_age >= age_target) then
               write(*,*) 'have reached age_target'
               do_astero_extras_check_model = terminate
            end if
            if (best_chi2 > 0) then
               if (best_chi2 <= chi2_search_limit1 .and. &
                  chi2 >= chi2_search_limit2) then
                  write(*,*) 'have reached chi2_search_limit2'
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (chi2 >= chi2_relative_increase_limit*best_chi2) then
                  num_chi2_too_big = num_chi2_too_big + 1
                  if (num_chi2_too_big > limit_num_chi2_too_big) then
                     write(*,*) 'have reached too many bad chi2 limit'
                     do_astero_extras_check_model = terminate
                  end if
                  return
               end if
               num_chi2_too_big = 0
            end if
         end subroutine final_checks
         

         logical function get_radial(code)
            character (len=*), intent(in) :: code
            integer :: ierr
            include 'formats'
            ierr = 0
            get_radial = .false.
            l0_freq(:) = 0.
            call get_one_el_info(s, 0, &
               nu_lower_factor*l0_obs(1), &
               nu_upper_factor*l0_obs(nl0), &
               iscan_factor_l0*nl0, 1, nl0, store_model, &
               code, ierr)
            if (ierr /= 0) then
               !write(*,'(a65,i6)') 'failed in oscillation code', s% model_number
               !stop
               return
            end if
            if (.not. have_all_l0_freqs()) then
               write(*,'(a65,i6)') 'failed to find all required l=0 frequencies', &
                  s% model_number
               l0_freq_corr(:) = l0_freq(:) ! for plotting
               return
            end if
            call get_freq_corr(s, .true., ierr)
            if (ierr /= 0) then
               write(*,'(a65,i6)') 'failed in get_freq_corr', s% model_number
               return
            end if
            chi2_radial = get_chi2(s, 0, .false., ierr)
            if (ierr /= 0) then
               write(*,'(a65,i6)') 'failed to get chi2_radial', s% model_number
               return
            end if
            !write(*,1) trim(code) // ' chi2_radial', chi2_radial
            if (chi2_radial > chi2_radial_limit .and. nl1 + nl2 + nl3 > 0) then
               write(*,'(a50,i6,99f16.2)') &
                  'chi2_radial > chi2_radial_limit', &
                  s% model_number, chi2_radial, chi2_radial_limit
               return
            end if
            get_radial = .true.
         end function get_radial
         
         
         logical function have_all_l0_freqs()
            integer :: i, cnt
            real(dp) :: prev
            cnt = 0
            have_all_l0_freqs = .true.
            if (nl0 <= 0) return
            prev = l0_freq(1)
            do i=2,nl0
               if (l0_obs(i) < 0) cycle
               if (l0_freq(i) == prev) then
                  have_all_l0_freqs = .false.
                  if (cnt == 0) write(*,'(i30,4x,a)',advance='no') &
                     s% model_number, 'missing l=0 freq number:'
                  cnt = cnt+1
                  write(*,'(i3)',advance='no') i
               end if
            end do
            if (cnt > 0) write(*,*)
         end function have_all_l0_freqs
         
         
         subroutine check_limits
            real(dp) :: logg_limit, logL_limit, Teff_limit, delta_nu_limit, &
               logR_limit, surface_Z_div_X_limit, surface_He_limit, Rcz_limit, &
               csound_rms_limit, my_var1_limit, my_var2_limit, my_var3_limit
            integer :: nz
            include 'formats'
            nz = s% nz

            if (sigmas_coeff_for_Teff_limit /= 0 .and. Teff_sigma > 0) then
               Teff_limit = Teff_target + Teff_sigma*sigmas_coeff_for_Teff_limit
               if ((sigmas_coeff_for_Teff_limit > 0 .and. s% Teff > Teff_limit) .or. &
                   (sigmas_coeff_for_Teff_limit < 0 .and. s% Teff < Teff_limit)) then
                  write(*,*) 'have reached Teff limit'
                  write(*,1) 'Teff', s% Teff
                  write(*,1) 'Teff_limit', Teff_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if    
               if (trace_limits) then
                  write(*,1) 'Teff', s% Teff
                  write(*,1) 'Teff_limit', Teff_limit
               end if
            end if
            
            if (sigmas_coeff_for_logg_limit /= 0 .and. logg_sigma > 0) then    
               logg_limit = logg_target + logg_sigma*sigmas_coeff_for_logg_limit
               if ((sigmas_coeff_for_logg_limit > 0 .and. logg > logg_limit) .or. &
                   (sigmas_coeff_for_logg_limit < 0 .and. logg < logg_limit)) then
                  write(*,*) 'have reached logg limit'
                  write(*,1) 'logg', logg
                  write(*,1) 'logg_limit', logg_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'logg', logg
                  write(*,1) 'logg_limit', logg_limit
               end if
            end if
            
            if (sigmas_coeff_for_logL_limit /= 0 .and. logL_sigma > 0) then
               logL_limit = logL_target + logL_sigma*sigmas_coeff_for_logL_limit
               if ((sigmas_coeff_for_logL_limit > 0 .and. s% log_surface_luminosity > logL_limit) .or. &
                   (sigmas_coeff_for_logL_limit < 0 .and. s% log_surface_luminosity < logL_limit)) then
                  write(*,*) 'have reached logL limit'
                  write(*,1) 'logL', s% log_surface_luminosity
                  write(*,1) 'logL_limit', logL_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'logL', s% log_surface_luminosity
                  write(*,1) 'logL_limit', logL_limit
               end if
            end if
            
            if (sigmas_coeff_for_delta_nu_limit /= 0 .and. delta_nu_sigma > 0 .and. delta_nu > 0) then
               delta_nu_limit = &
                  delta_nu + delta_nu_sigma*sigmas_coeff_for_delta_nu_limit
               if ((sigmas_coeff_for_delta_nu_limit > 0 .and. delta_nu_model > delta_nu_limit) .or. &
                   (sigmas_coeff_for_delta_nu_limit < 0 .and. delta_nu_model < delta_nu_limit)) then
                  write(*,*) 'have reached delta_nu limit'
                  write(*,1) 'delta_nu_model', delta_nu_model
                  write(*,1) 'delta_nu_limit', delta_nu_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'delta_nu_model', delta_nu_model
                  write(*,1) 'delta_nu_limit', delta_nu_limit
               end if
            end if
            
            if (sigmas_coeff_for_logR_limit /= 0 .and. logR_sigma > 0) then
               logR_limit = logR_target + logR_sigma*sigmas_coeff_for_logR_limit
               if ((sigmas_coeff_for_logR_limit > 0 .and. logR > logR_limit) .or. &
                   (sigmas_coeff_for_logR_limit < 0 .and. logR < logR_limit)) then
                  write(*,*) 'have reached logR limit'
                  write(*,1) 'logR', logR
                  write(*,1) 'logR_limit', logR_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'logR', logR
                  write(*,1) 'logR_limit', logR_limit
               end if
            end if
            
            if (sigmas_coeff_for_surface_Z_div_X_limit /= 0 .and. surface_Z_div_X_sigma > 0) then
               surface_Z_div_X_limit = surface_Z_div_X_target + &
                  surface_Z_div_X_sigma*sigmas_coeff_for_surface_Z_div_X_limit
               if ((sigmas_coeff_for_surface_Z_div_X_limit > 0 .and. &
                     surface_Z_div_X > surface_Z_div_X_limit) .or. &
                     (sigmas_coeff_for_surface_Z_div_X_limit < 0 .and. &
                      surface_Z_div_X < surface_Z_div_X_limit)) then
                  write(*,*) 'have reached surface_Z_div_X limit'
                  write(*,1) 'surface_Z_div_X', surface_Z_div_X
                  write(*,1) 'surface_Z_div_X_limit', surface_Z_div_X_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'surface_Z_div_X', surface_Z_div_X
                  write(*,1) 'surface_Z_div_X_limit', surface_Z_div_X_limit
               end if
            end if
            
            if (sigmas_coeff_for_surface_He_limit /= 0 .and. surface_He_sigma > 0) then
               surface_He_limit = surface_He_target + &
                     surface_He_sigma*sigmas_coeff_for_surface_He_limit
               if ((sigmas_coeff_for_surface_He_limit > 0 .and. &
                     surface_He > surface_He_limit) .or. &
                   (sigmas_coeff_for_surface_He_limit < 0 .and. &
                     surface_He < surface_He_limit)) then
                  write(*,*) 'have reached surface_He limit'
                  write(*,1) 'surface_He', surface_He
                  write(*,1) 'surface_He_limit', surface_He_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'surface_He', surface_He
                  write(*,1) 'surface_He_limit', surface_He_limit
               end if
            end if
            
            if (sigmas_coeff_for_Rcz_limit /= 0 .and. Rcz_sigma > 0) then
               Rcz_limit = Rcz_target + Rcz_sigma*sigmas_coeff_for_Rcz_limit
               if ((sigmas_coeff_for_Rcz_limit > 0 .and. Rcz > Rcz_limit) .or. &
                   (sigmas_coeff_for_Rcz_limit < 0 .and. Rcz < Rcz_limit)) then
                  write(*,*) 'have reached Rcz limit'
                  write(*,1) 'Rcz', Rcz
                  write(*,1) 'Rcz_limit', Rcz_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'Rcz', Rcz
                  write(*,1) 'Rcz_limit', Rcz_limit
               end if
            end if
            
            if (sigmas_coeff_for_csound_rms_limit /= 0 .and. csound_rms_sigma > 0) then
               csound_rms_limit = csound_rms_target + &
                     csound_rms_sigma*sigmas_coeff_for_csound_rms_limit
               if ((sigmas_coeff_for_csound_rms_limit > 0 .and. &
                     csound_rms > csound_rms_limit) .or. &
                   (sigmas_coeff_for_csound_rms_limit < 0 .and. &
                     csound_rms < csound_rms_limit)) then
                  write(*,*) 'have reached csound_rms limit'
                  write(*,1) 'csound_rms', csound_rms
                  write(*,1) 'csound_rms_limit', csound_rms_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'csound_rms', csound_rms
                  write(*,1) 'csound_rms_limit', csound_rms_limit
               end if
            end if
            
            if (sigmas_coeff_for_my_var1_limit /= 0 .and. my_var1_sigma > 0) then
               my_var1_limit = &
                  my_var1_target + my_var1_sigma*sigmas_coeff_for_my_var1_limit
               if ((sigmas_coeff_for_my_var1_limit > 0 .and. &
                        my_var1 > my_var1_limit) .or. &
                   (sigmas_coeff_for_my_var1_limit < 0 .and. &
                        my_var1 < my_var1_limit)) then
                  write(*,*) 'have reached my_var1 limit'
                  write(*,1) 'my_var1', my_var1
                  write(*,1) 'my_var1_limit', my_var1_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'my_var1', my_var1
                  write(*,1) 'my_var1_limit', my_var1_limit
               end if
            end if
            
            if (sigmas_coeff_for_my_var2_limit /= 0 .and. my_var2_sigma > 0) then
               my_var2_limit = &
                  my_var2_target + my_var2_sigma*sigmas_coeff_for_my_var2_limit
               if ((sigmas_coeff_for_my_var2_limit > 0 .and. &
                        my_var2 > my_var2_limit) .or. &
                   (sigmas_coeff_for_my_var2_limit < 0 .and. &
                        my_var2 < my_var2_limit)) then
                  write(*,*) 'have reached my_var2 limit'
                  write(*,1) 'my_var2', my_var2
                  write(*,1) 'my_var2_limit', my_var2_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'my_var2', my_var2
                  write(*,1) 'my_var2_limit', my_var2_limit
               end if
            end if
            
            if (sigmas_coeff_for_my_var3_limit /= 0 .and. my_var3_sigma > 0) then
               my_var3_limit = &
                  my_var3_target + my_var3_sigma*sigmas_coeff_for_my_var3_limit
               if ((sigmas_coeff_for_my_var3_limit > 0 .and. &
                        my_var3 > my_var3_limit) .or. &
                   (sigmas_coeff_for_my_var3_limit < 0 .and. &
                        my_var3 < my_var3_limit)) then
                  write(*,*) 'have reached my_var3 limit'
                  write(*,1) 'my_var3', my_var3
                  write(*,1) 'my_var3_limit', my_var3_limit
                  write(*,*)
                  do_astero_extras_check_model = terminate
                  return
               end if
               if (trace_limits) then
                  write(*,1) 'my_var3', my_var3
                  write(*,1) 'my_var3_limit', my_var3_limit
               end if
            end if
            
         end subroutine check_limits


         subroutine setup_solar_data_for_calc_rms(ierr)
            use const_def, only: mesa_data_dir
            integer, intent(out) :: ierr
         
            integer, parameter :: lines_to_skip = 11
            integer :: iounit, i, k
            real(dp) :: jnk
         
            character (len=256) :: fname
         
            have_sound_speed_data = .true.
            ierr = 0
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) then
               return
            end if
            fname = trim(mesa_data_dir) // '/star_data/solar_csound.data'
            open(iounit, file=trim(fname), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed to open ' // trim(fname)
               call free_iounit(iounit)
               return
            end if                  
         
            do i=1,lines_to_skip
               read(iounit,fmt=*,iostat=ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed when skipping line', i
                  close(iounit)
                  call free_iounit(iounit)
                  return
               end if
            end do
         
            do i=1,npts
               read(iounit,fmt=*,iostat=ierr) &
                  data_r(i), jnk, data_csound(i), jnk, jnk, jnk, data_width(i)
               if (ierr /= 0) then
                  write(*,*) 'failed when reading data point', i
                  close(iounit)
                  call free_iounit(iounit)
                  return
               end if
            end do

            close(iounit)
            call free_iounit(iounit)
      
      
         end subroutine setup_solar_data_for_calc_rms


         real(dp) function calc_current_rms(s, nz) ! dR weighted
            use interp_1d_lib
            use interp_1d_def
            type (star_info), pointer :: s
            integer, intent(in) :: nz
         
            logical, parameter :: dbg = .false.
            real(dp), target :: calc_rms_f1_ary(4*nz)
            real(dp), pointer :: calc_rms_f1(:), calc_rms_f(:,:)
            real(dp) :: sumy2, sumdr, dr, y2, cs
            real(dp), parameter :: min_R = 0.094, max_R = 0.94
            real(dp), target :: pm_work_ary(nz*pm_work_size)
            real(dp), pointer :: pm_work(:)
            integer :: k, i, ierr
         
            include 'formats'
         
            calc_current_rms = -1  
            pm_work => pm_work_ary
            calc_rms_f1 => calc_rms_f1_ary
            calc_rms_f(1:4,1:nz) => calc_rms_f1(1:4*nz)

            ierr = 0
            if (.not. have_sound_speed_data) then
               call setup_solar_data_for_calc_rms(ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in setup_solar_data_for_calc_rms'
                  return
               end if
            end if
         
            do k=1,nz
               if (k == 1) then
                  calc_rms_f(1,k) = s% csound(k)
               else
                  calc_rms_f(1,k) = &
                     (s% csound(k)*s% dq(k-1) + s% csound(k-1)*s% dq(k))/(s% dq(k-1) + s% dq(k))
               end if
            end do

            call interp_pm( &
               s% r, nz, calc_rms_f1, pm_work_size, pm_work, 'calc_current_rms', ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in interp_pm'
               return
            end if

            sumy2 = 0
            sumdr = 0
            do i=1,npts
               if (data_r(i) < min_R .or. data_r(i) > max_R) cycle
               call interp_value(s% r, nz, calc_rms_f1, Rsun*data_r(i), cs, ierr)
               if (ierr /= 0) then
                  write(*,*) 'failed in interp_value', i
                  return
               end if
               if (i == 1) then
                  dr = data_r(2) - data_r(1)
               else
                  dr = data_r(i) - data_r(i-1)
               end if
               if (dr < 0) dr = -dr
               ! change to weigh by point rather than by dr
               dr = 1
            
               sumdr = sumdr + dr
               y2 = dr*pow2((cs - data_csound(i))/data_csound(i))
               sumy2 = sumy2 + y2
               if (dbg) write(*,2) 'rms cs, data_cs, reldiff, y2, dr', i, cs, data_csound(i), &
                  (cs - data_csound(i))/data_csound(i), y2, dr
            end do
         
            calc_current_rms = sqrt(sumy2/sumdr)
            if (dbg) write(*,1) 'calc_current_rms', calc_current_rms

         end function calc_current_rms
         

      end function do_astero_extras_check_model
      
      
      real(dp) function get_chi2_spectro(s)
         type (star_info), pointer :: s
         integer :: cnt
         real(dp) :: logL, sum
         include 'formats'
         cnt = 0
         sum = 0
         if (include_logL_in_chi2_spectro) then
            cnt = cnt + 1
            logL = s% log_surface_luminosity
            sum = sum + pow2((logL - logL_target)/logL_sigma)
         end if
         if (include_logg_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2((logg - logg_target)/logg_sigma)
         end if
         if (include_Teff_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2((s% Teff - Teff_target)/Teff_sigma)
         end if
         if (include_FeH_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2((FeH - FeH_target)/FeH_sigma)
         end if
         if (include_logR_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2((logR - logR_target)/logR_sigma)
         end if
         if (include_age_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2((s% star_age - age_target)/age_sigma)
         end if
         if (include_surface_Z_div_X_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + &
               pow2( &
               (surface_Z_div_X - surface_Z_div_X_target)/surface_Z_div_X_sigma)
         end if
         if (include_surface_He_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2( &
               (surface_He - surface_He_target)/surface_He_sigma)
         end if
         if (include_Rcz_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2((Rcz - Rcz_target)/Rcz_sigma)
         end if
         if (include_csound_rms_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2( &
               (csound_rms - csound_rms_target)/csound_rms_sigma)
         end if
         if (include_my_var1_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2( &
               (my_var1 - my_var1_target)/my_var1_sigma)
         end if
         if (include_my_var2_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2( &
               (my_var2 - my_var2_target)/my_var2_sigma)
         end if
         if (include_my_var3_in_chi2_spectro) then
            cnt = cnt + 1
            sum = sum + pow2( &
               (my_var3 - my_var3_target)/my_var3_sigma)
         end if
         get_chi2_spectro = sum/cnt
      end function get_chi2_spectro
      
      
      subroutine store_best_info(s)
         type (star_info), pointer :: s
         integer :: i
      
         best_chi2 = chi2
         best_chi2_seismo = chi2_seismo
         best_chi2_spectro = chi2_spectro
         
         best_age = s% star_age
         best_model_number = s% model_number
         best_radius = s% photosphere_r
         best_logL = s% log_surface_luminosity
         best_Teff = s% Teff
         best_logg = logg
         best_FeH = FeH
         
         best_logR = logR
         best_surface_Z_div_X = surface_Z_div_X
         best_surface_He = surface_He
         best_Rcz = Rcz
         best_csound_rms = csound_rms
         best_my_var1 = my_var1
         best_my_var2 = my_var2
         best_my_var3 = my_var3
         
         best_delta_nu = delta_nu_model
         best_nu_max = nu_max_model
         best_a_div_r = a_div_r
         best_correction_r = correction_r
         
         do i=1, nl0
            best_l0_order(i) = l0_order(i)
            best_l0_freq(i) = l0_freq(i)
            best_l0_freq_corr(i) = l0_freq_corr(i)
            best_l0_inertia(i) = l0_inertia(i)
         end do

         do i=1, nl1
            best_l1_order(i) = l1_order(i)
            best_l1_freq(i) = l1_freq(i)
            best_l1_freq_corr(i) = l1_freq_corr(i)
            best_l1_inertia(i) = l1_inertia(i)
         end do
      
         do i=1, nl2
            best_l2_order(i) = l2_order(i)
            best_l2_freq(i) = l2_freq(i)
            best_l2_freq_corr(i) = l2_freq_corr(i)
            best_l2_inertia(i) = l2_inertia(i)
         end do
      
         do i=1, nl3
            best_l3_order(i) = l3_order(i)
            best_l3_freq(i) = l3_freq(i)
            best_l3_freq_corr(i) = l3_freq_corr(i)
            best_l3_inertia(i) = l3_inertia(i)
         end do
         
         best_ratios_r01(:) = 0d0
         best_ratios_r10(:) = 0d0
         best_ratios_r02(:) = 0d0

         do i=1,ratios_n
            best_ratios_r01(i) = model_ratios_r01(i)
            best_ratios_r10(i) = model_ratios_r10(i)
         end do
         
         do i=1,nl0
            best_ratios_r02(i) = model_ratios_r02(i)
         end do
      
      end subroutine store_best_info


      subroutine set_current_from_best(s)
         type (star_info), pointer :: s
         integer :: i
      
         chi2 = best_chi2
         chi2_seismo = best_chi2_seismo
         chi2_spectro = best_chi2_spectro
         
         delta_nu_model = best_delta_nu
         nu_max_model = best_nu_max
         a_div_r = best_a_div_r
         correction_r = best_correction_r
         
         do i=1, nl0
            l0_order(i) = best_l0_order(i)
            l0_freq(i) = best_l0_freq(i)
            l0_freq_corr(i) = best_l0_freq_corr(i)
            l0_inertia(i) = best_l0_inertia(i)
         end do

         do i=1, nl1
            l1_order(i) = best_l1_order(i)
            l1_freq(i) = best_l1_freq(i)
            l1_freq_corr(i) = best_l1_freq_corr(i)
            l1_inertia(i) = best_l1_inertia(i)
         end do
      
         do i=1, nl2
            l2_order(i) = best_l2_order(i)
            l2_freq(i) = best_l2_freq(i)
            l2_freq_corr(i) = best_l2_freq_corr(i)
            l2_inertia(i) = best_l2_inertia(i)
         end do
      
         do i=1, nl3
            l3_order(i) = best_l3_order(i)
            l3_freq(i) = best_l3_freq(i)
            l3_freq_corr(i) = best_l3_freq_corr(i)
            l3_inertia(i) = best_l3_inertia(i)
         end do
         
         do i=1,ratios_n
            model_ratios_r01(i) = best_ratios_r01(i)
            model_ratios_r10(i) = best_ratios_r10(i)
         end do
         
         do i=1,nl0
            model_ratios_r02(i) = best_ratios_r02(i)
         end do
      
      end subroutine set_current_from_best
      
      
      subroutine save_best_info(s)     
         use pgstar_astero_plots, only: write_plot_to_file
         use run_star_extras, only: &
            how_many_extra_profile_columns, data_for_extra_profile_columns
         type (star_info), pointer :: s
         integer :: ierr
         logical :: write_controls_info_with_profile
         
         include 'formats'
         
         if (save_model_for_best_model) then
            ierr = 0
            call star_write_model(s% id, best_model_save_model_filename, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in star_write_model'
               stop 1
            end if
            write(*, '(a,i7)') 'save ' // trim(best_model_save_model_filename), s% model_number
         end if
         
         if (write_fgong_for_best_model) then
            ierr = 0
            call star_write_fgong(s% id, &
               add_center_point, keep_surface_point, &
               add_atmosphere, best_model_fgong_filename, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in star_write_fgong'
               stop 1
            end if
         end if
         
         if (write_gyre_for_best_model) then
            ierr = 0
            call star_write_gyre(s% id, &
               keep_surface_point, add_atmosphere, &
               best_model_gyre_filename, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in star_write_gyre'
               stop 1
            end if
         end if
         
         if (write_profile_for_best_model) then
            ierr = 0
            write_controls_info_with_profile = s% write_controls_info_with_profile
            s% write_controls_info_with_profile = .false.
            call star_write_profile_info(s% id, best_model_profile_filename, 0, &
               how_many_extra_profile_columns, data_for_extra_profile_columns, ierr)
            s% write_controls_info_with_profile = write_controls_info_with_profile
            if (ierr /= 0) then
               write(*,*) 'failed in star_write_profile_info'
               stop 1
            end if
            call save_profile(s% id, 0, &
               how_many_extra_profile_columns, &
               data_for_extra_profile_columns, 3, ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in save_profile'
               stop 1
            end if
         end if         
         
         if (len_trim(echelle_best_model_file_prefix) > 0) then
            ! note: sample_number hasn't been incremented yet so must add 1
            call write_plot_to_file( &
               s, p_echelle, echelle_best_model_file_prefix, sample_number+1, ierr)        
         end if
         
         if (len_trim(ratios_best_model_file_prefix) > 0) then
            ! note: sample_number hasn't been incremented yet so must add 1
            call write_plot_to_file( &
               s, p_ratios, ratios_best_model_file_prefix, sample_number+1, ierr)   
         end if
         
         call store_best_info(s)
            
      end subroutine save_best_info
      
      
      subroutine write_best(num)
         integer, intent(in) :: num
         integer :: ierr, iounit
         character (len=256) :: format_string, num_string, filename
         integer, parameter :: max_len_out = 2000
         character (len=max_len_out) :: script         
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         write(format_string,'( "(i",i2.2,".",i2.2,")" )') num_digits, num_digits
         write(num_string,format_string) num
         filename = trim(sample_results_prefix) // trim(num_string) // trim(sample_results_postfix)
         open(unit=iounit, file=trim(filename), action='write', status='replace', iostat=ierr)
         if (ierr == 0) then
            call show_best(iounit)
            close(iounit)
            write(*,*) 'save best model results to ' // trim(filename)
         else
            write(*,*) 'failed to open ' // trim(filename)
         end if
         call free_iounit(iounit)
         if (len(shell_script_for_each_sample) > 0) then
            call create_shell_script_for_sample(num_digits, num_string, script, ierr)
            write(*,*) 'shell script for sample: ' // trim(script)
            if (ierr == 0 .and. len_trim(script) > 0) then
               call system(trim(script))
            end if
         end if
      end subroutine write_best
      
      
      subroutine create_shell_script_for_sample(num_digits, num_string, script, ierr)
         integer, intent(in) :: num_digits
         character (len=*), intent(in) :: num_string
         character (len=*), intent(out) :: script
         integer, intent(out) :: ierr
         integer :: i, j, k, len_in, max_len_out   
         include 'formats'  
         ierr = 0    
         len_in = len_trim(shell_script_for_each_sample)
         if (len_in == 0) then
            script = ''
            return
         end if
         max_len_out = len(script)
         write(*,2) 'len_in', len_in
         write(*,2) 'max_len_out', max_len_out
         write(*,*) 'shell_script_num_string_char: ' // trim(shell_script_num_string_char)
         write(*,*) 'script in: ' // trim(shell_script_for_each_sample)
         write(*,*)
         i = 0
         do j=1,len_in
            if (i+num_digits > max_len_out) then
               write(*,*) 'shell script is too large'
               ierr = -1
               return
            end if
            if (shell_script_for_each_sample(j:j) == &
                  shell_script_num_string_char(1:1)) then
               do k=1,num_digits
                  i = i+1
                  script(i:i) = num_string(k:k)
               end do
            else
               i = i+1
               script(i:i) = shell_script_for_each_sample(j:j)
            end if
         end do
         do j=i+1,max_len_out
            script(j:j) = ' '
         end do      
      end subroutine create_shell_script_for_sample         


      integer function astero_extras_check_model(id, id_extra)
         use run_star_extras, only: extras_check_model
            
         integer, intent(in) :: id, id_extra
         integer :: other_check, ierr
         type (star_info), pointer :: s
         
         include 'formats'
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return

         if (s% job% astero_just_call_my_extras_check_model) then
            astero_extras_check_model = extras_check_model(id, id_extra)
            best_chi2 = 0
         else
            other_check = extras_check_model(id, id_extra)
            astero_extras_check_model = &
                  do_astero_extras_check_model(s, id, id_extra)
            if (other_check > astero_extras_check_model) &
               astero_extras_check_model = other_check
         end if
         
         star_model_number = s% model_number
         if (star_model_number /= save_mode_model_number) return
         call get_all_el_info(s,ierr)
               
      end function astero_extras_check_model

      
      integer function astero_extras_finish_step(id, id_extra)
         use run_star_extras, only: extras_finish_step
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         if (s% job% pgstar_flag) then
            ierr = 0
            call read_astero_pgstar_controls(inlist_astero_fname, ierr)
            if (ierr /= 0) then
               astero_extras_finish_step = terminate
               return
            end if
         end if
         astero_extras_finish_step = extras_finish_step(id, id_extra)
         call store_extra_info(s)
         
         s% dt_next = min(s% dt_next, astero_max_dt_next)
         
      end function astero_extras_finish_step
      

      subroutine astero_extras_controls(id, ierr)
         use run_star_extras, only: extras_controls
         use pgstar_astero_plots, only: astero_pgstar_plots_info
         integer, intent(in) :: id
         integer, intent(out) :: ierr
         real(dp) :: X, Y, Z, FeH, f_ov, a, b, c
         type (star_info), pointer :: s
         include 'formats'
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         write(*,*) 'enter astero_extras_controls'
         
         call extras_controls(id, ierr)
         if (ierr /= 0) return
         
         if (s% job% astero_just_call_my_extras_check_model) return
         
         s% other_pgstar_plots_info => astero_pgstar_plots_info
         s% use_other_pgstar_plots = .true.
         
         
         ! overwrite various inlist controls

         if (vary_alpha) then
            s% mixing_length_alpha = next_alpha_to_try
         else
            s% mixing_length_alpha = first_alpha
         end if

         if (vary_f_ov) then
            f_ov = next_f_ov_to_try
         else
            f_ov = first_f_ov
         end if
      
         if (vary_FeH) then
            FeH = next_FeH_to_try
         else
            FeH = first_FeH
         end if

         initial_FeH = FeH
         initial_Z_div_X = Z_div_X_solar*exp10_cr(FeH)

         if (Y_depends_on_Z) then
            a = initial_Z_div_X
            b = dYdZ
            c = 1d0 + a*(1d0 + b)
            X = (1d0 - Y0)/c
            Y = (Y0 + a*(b + Y0))/c
            Z = 1d0 - (X + Y)
            !write(*,1) 'init X', X
            !write(*,1) 'init Y', Y
            !write(*,1) 'init Z', Z
            !stop
         else 
            if (vary_Y) then
               Y = next_Y_to_try
            else
               Y = first_Y
            end if
            X = (1d0 - Y)/(1d0 + initial_Z_div_X)
            Z = X*initial_Z_div_X
         end if

         if (vary_mass) then
            s% job% new_mass = next_mass_to_try
         else
            s% job% new_mass = first_mass
         end if
         
         s% job% relax_initial_mass = .true.
         s% initial_mass = s% job% new_mass
         
         initial_Y = Y
         !s% initial_Z = Z << don't do this. it interferes with use of zams file.
         
         s% job% initial_h1 = X
         s% job% initial_h2 = 0
         s% job% initial_he3 = Y_frac_he3*Y
         s% job% initial_he4 = Y - s% job% initial_he3
         s% job% set_uniform_initial_composition = .true. 
         
         current_Y = Y
         current_FeH = FeH
         current_mass = s% job% new_mass
         current_alpha = s% mixing_length_alpha
         current_f_ov = f_ov
         
         current_h1 = X
         current_he3 = s% job% initial_he3
         current_he4 = s% job% initial_he4
         current_Z = Z
        
         s% overshoot_f_above_nonburn_core = f_ov
         s% overshoot_f_above_nonburn_shell = f_ov
         s% overshoot_f_below_nonburn_shell = f_ov
         
         s% overshoot_f_above_burn_h_core = f_ov
         s% overshoot_f_above_burn_h_shell = f_ov
         s% overshoot_f_below_burn_h_shell = f_ov
         
         s% overshoot_f_above_burn_he_core = f_ov
         s% overshoot_f_above_burn_he_shell = f_ov
         s% overshoot_f_below_burn_he_shell = f_ov
         
         s% overshoot_f_above_burn_z_core = f_ov
         s% overshoot_f_above_burn_z_shell = f_ov
         s% overshoot_f_below_burn_z_shell = f_ov
         
         s% overshoot_f0_above_nonburn_core = f0_ov_div_f_ov*f_ov
         s% overshoot_f0_above_nonburn_shell = f0_ov_div_f_ov*f_ov
         s% overshoot_f0_below_nonburn_shell = f0_ov_div_f_ov*f_ov
         
         s% overshoot_f0_above_burn_h_core = f0_ov_div_f_ov*f_ov
         s% overshoot_f0_above_burn_h_shell = f0_ov_div_f_ov*f_ov
         s% overshoot_f0_below_burn_h_shell = f0_ov_div_f_ov*f_ov
         
         s% overshoot_f0_above_burn_he_core = f0_ov_div_f_ov*f_ov
         s% overshoot_f0_above_burn_he_shell = f0_ov_div_f_ov*f_ov
         s% overshoot_f0_below_burn_he_shell = f0_ov_div_f_ov*f_ov
         
         s% overshoot_f0_above_burn_z_core = f0_ov_div_f_ov*f_ov
         s% overshoot_f0_above_burn_z_shell = f0_ov_div_f_ov*f_ov
         s% overshoot_f0_below_burn_z_shell = f0_ov_div_f_ov*f_ov
         
      end subroutine astero_extras_controls
      
      
      integer function astero_extras_startup(id, restart, ierr)
         use run_star_extras, only: extras_startup
         integer, intent(in) :: id
         logical, intent(in) :: restart
         integer, intent(out) :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         astero_extras_startup = extras_startup(id, restart, ierr)
         if (.not. restart) then
            call alloc_extra_info(s)
         else ! it is a restart
            call unpack_extra_info(s)
         end if
      end function astero_extras_startup


      integer function astero_how_many_extra_history_columns(id, id_extra)
         use run_star_extras, only: how_many_extra_history_columns
         integer, intent(in) :: id, id_extra
         integer :: ierr
         type (star_info), pointer :: s
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         astero_how_many_extra_history_columns = how_many_extra_history_columns(id, id_extra)
         if (.not. s% job% astero_just_call_my_extras_check_model) &
            astero_how_many_extra_history_columns = &
               astero_how_many_extra_history_columns + num_extra_history_columns
      end function astero_how_many_extra_history_columns
      
      
      subroutine astero_data_for_extra_history_columns(id, id_extra, n, names, vals, ierr)
         use run_star_extras, only: how_many_extra_history_columns, data_for_extra_history_columns
         integer, intent(in) :: id, id_extra, n
         character (len=maxlen_history_column_name) :: names(n)
         real(dp) :: vals(n)
         integer, intent(out) :: ierr
         integer :: i, num_extra
         type (star_info), pointer :: s
         
         include 'formats'
         
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         
         call data_for_extra_history_columns(id, id_extra, n, names, vals, ierr)
         if (ierr /= 0) return
         if (s% job% astero_just_call_my_extras_check_model) return
         
         num_extra = how_many_extra_history_columns(id, id_extra)
         
         i = num_extra+1
         names(i) = 'chi2'         
         i = i+1
         names(i) = 'delta_nu'         
         i = i+1
         names(i) = 'delta_nu_model'         
         i = i+1
         names(i) = 'correction_r'         
         i = i+1
         names(i) = 'a_div_r'
         
         if (i /= num_extra_history_columns) then
            write(*,2) 'i', i
            write(*,2) 'num_extra_history_columns', num_extra_history_columns
            stop 'bad num_extra_history_columns'
         end if
         
         i = num_extra+1
         vals(i) = chi2
         i = i+1
         vals(i) = delta_nu
         i = i+1
         vals(i) = delta_nu_model
         i = i+1
         vals(i) = correction_r
         i = i+1
         vals(i) = a_div_r
         
         
      end subroutine astero_data_for_extra_history_columns

      
      integer function astero_how_many_extra_profile_columns(id, id_extra)
         use run_star_extras, only: how_many_extra_profile_columns
         integer, intent(in) :: id, id_extra
         astero_how_many_extra_profile_columns = &
            how_many_extra_profile_columns(id, id_extra)
      end function astero_how_many_extra_profile_columns
      
      
      subroutine astero_data_for_extra_profile_columns( &
            id, id_extra, n, nz, names, vals, ierr)
         use run_star_extras, only: data_for_extra_profile_columns
         integer, intent(in) :: id, id_extra, n, nz
         character (len=maxlen_profile_column_name) :: names(n)
         real(dp) :: vals(nz,n)
         integer, intent(out) :: ierr
         integer :: k
         ierr = 0
         call data_for_extra_profile_columns(id, id_extra, n, nz, names, vals, ierr)
      end subroutine astero_data_for_extra_profile_columns

      
      subroutine astero_extras_after_evolve(id, id_extra, ierr)
         use run_star_extras, only: extras_after_evolve
         integer, intent(in) :: id, id_extra
         integer, intent(out) :: ierr
         integer :: iounit, ckm
         type (star_info), pointer :: s
         include 'formats'
         ierr = 0
         call star_ptr(id, s, ierr)
         if (ierr /= 0) return
         if ((eval_chi2_at_target_age_only .and. s% star_age >= age_target) .or. &
             (include_age_in_chi2_spectro .and. s% star_age >= max_age_for_chi2) .or. &
             save_info_for_last_model) then
            !write(*,*) 'call do_astero_extras_check_model before terminate'
            ckm = do_astero_extras_check_model(s, id, id_extra)
            !write(*,*) 'done do_astero_extras_check_model before terminate'
         end if         
         call extras_after_evolve(id, id_extra, ierr)
         if (save_info_for_last_model) then
            write(*,1) 'chi2', chi2
            call get_all_el_info(s,ierr)
            if (ierr /= 0) return
            call store_best_info(s)
            iounit = alloc_iounit(ierr)
            if (ierr /= 0) return
            open(unit=iounit, file=trim(last_model_save_info_filename), &
               action='write', status='replace', iostat=ierr)
            if (ierr /= 0) then
               write(*,'(a)') 'failed to open last_model_save_info_filename ' // &
                  trim(last_model_save_info_filename)
               return
            end if
            write(*,*) 'write ' // trim(last_model_save_info_filename)
            write(*,*) 'call show_best'
            call show_best(iounit)
            write(*,*) 'done show_best'
            call free_iounit(iounit)
         end if
         
         if (s% job% astero_just_call_my_extras_check_model) return

      end subroutine astero_extras_after_evolve
      
      
      ! routines for saving and restoring extra data so can do restarts
      
      subroutine alloc_extra_info(s)
         integer, parameter :: extra_info_alloc = 1
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_alloc)
      end subroutine alloc_extra_info
      
      
      subroutine unpack_extra_info(s)
         integer, parameter :: extra_info_get = 2
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_get)
      end subroutine unpack_extra_info
      
      
      subroutine store_extra_info(s)
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         call move_extra_info(s,extra_info_put)
      end subroutine store_extra_info
      
      
      subroutine move_extra_info(s,op)
         integer, parameter :: extra_info_alloc = 1
         integer, parameter :: extra_info_get = 2
         integer, parameter :: extra_info_put = 3
         type (star_info), pointer :: s
         integer, intent(in) :: op
         
         integer :: i, j, num_ints, num_dbls, ierr
         
         i = 0
         ! call move_int or move_flg 
         num_ints = i
         
         i = 0
         ! call move_dbl
         num_dbls = i
         
         if (op /= extra_info_alloc) return
         if (num_ints == 0 .and. num_dbls == 0) return
         
         ierr = 0
         call star_alloc_extras(s% id, num_ints, num_dbls, ierr)
         if (ierr /= 0) then
            write(*,*) 'failed in star_alloc_extras'
            write(*,*) 'alloc_extras num_ints', num_ints
            write(*,*) 'alloc_extras num_dbls', num_dbls
            stop 1
         end if
         
         contains
         
         subroutine move_dbl(dbl)
            real(dp) :: dbl
            i = i+1
            select case (op)
            case (extra_info_get)
               dbl = s% extra_work(i)
            case (extra_info_put)
               s% extra_work(i) = dbl
            end select
         end subroutine move_dbl
         
         subroutine move_int(int)
            integer :: int
            i = i+1
            select case (op)
            case (extra_info_get)
               int = s% extra_iwork(i)
            case (extra_info_put)
               s% extra_iwork(i) = int
            end select
         end subroutine move_int
         
         subroutine move_flg(flg)
            logical :: flg
            i = i+1
            select case (op)
            case (extra_info_get)
               flg = (s% extra_iwork(i) /= 0)
            case (extra_info_put)
               if (flg) then
                  s% extra_iwork(i) = 1
               else
                  s% extra_iwork(i) = 0
               end if
            end select
         end subroutine move_flg
      
      end subroutine move_extra_info



      end module extras_support
