module test_ecapture

  use const_def
  use eos_def
  use eos_lib
  use rates_def
  use rates_lib
  use num_lib
  use utils_lib
  use crlibm_lib

  implicit none

  real(dp) :: X, Z, Y, abar, zbar, z2bar, ye
  integer, parameter :: species = 7
  integer, parameter :: h1=1, he4=2, c12=3, n14=4, o16=5, ne20=6, mg24=7
  integer, pointer, dimension(:) :: net_iso, chem_id
  real(dp), dimension(species) :: xa, ya, za, aa
  integer :: handle, i, ierr

  real(dp) :: log10T, T, log10Rho, Rho

contains


  subroutine do_test_ecapture

    call Init_Composition
    call Setup_eos(handle)

    log10Rho = 9.5
    Rho = exp10_cr(log10Rho)
    log10T = 8.5
    T = exp10_cr(log10T)

    ! first test the phase space integral functions

    write(*,*)
    write(*,*) 'do_test_derivs'

    call do_test_derivs(-10.0_dp,1d-3,0.0_dp)
    call do_test_derivs(-10.0_dp,0.0_dp,1d-3)

    call do_test_derivs(10.0_dp,1d-3,0.0_dp)
    call do_test_derivs(10.0_dp,0.0_dp,1d-3)

    write(*,*) 'done'
    write(*,*)

    ! check that the coulomb corrections are behaving

    write(*,*)
    write(*,*) 'do_test_coulomb'

    call do_test_coulomb

    write(*,*) 'done'
    write(*,*)


    ! check that the special weak reactions are working

    write(*,*)
    write(*,*) 'do_test_special_weak'

    call do_test_special_weak(.false.)
    call do_test_special_weak(.true.)

    write(*,*) 'done'
    write(*,*)

    ! deallocate the eos tables
    call free_eos_handle(handle)
    call eos_shutdown

  end subroutine do_test_ecapture


  subroutine do_test_derivs(q,dx,dy)

    real(dp) :: beta   ! mec2 / kT
    real(dp) :: zeta   ! Q_n / kT
    real(dp) :: eta    ! chemical potential / kT
    real(dp) :: deta_dlnT, deta_dlnRho ! and derivs

    real(dp) :: I, J    ! phase space integral
    real(dp) :: dI_dlnT, dI_dlnRho ! and derivatives
    real(dp) :: dJ_dlnT, dJ_dlnRho ! and derivatives

    real(dp) :: Im, Jm    ! phase space integral
    real(dp) :: dIm_dlnT, dIm_dlnRho ! and derivatives
    real(dp) :: dJm_dlnT, dJm_dlnRho ! and derivatives

    real(dp) :: Ip, Jp    ! phase space integral
    real(dp) :: dIp_dlnT, dIp_dlnRho ! and derivatives
    real(dp) :: dJp_dlnT, dJp_dlnRho ! and derivatives

    integer :: k, neta
    real(dp) :: etamin, etamax, deltaeta
    real(dp) :: dx, dy
    real(dp) :: q,mu,D9,T8
    character(len=80) :: fmt

    do k = 1, 3

       select case (k)
       case (1)
          D9 = 1
          T8 = 1
       case (2)
          D9 = 1 + dx
          T8 = 1 + dy
       case(3)
          D9 = 1 - dx
          T8 = 1 - dy
       end select

       mu= 10.1 * pow_cr(D9,1d0/3d0)
       beta = 59.3 / T8

       zeta = q * beta
       eta = mu * beta
       deta_dlnRho = eta / 3.0
       deta_dlnT = -eta

       select case (k)
       case(1)

          if (q < 0) then
             call psi_Iec_and_Jec(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)
          else
             call psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  I, dI_dlnT, dI_dlnRho, J, dJ_dlnT, dJ_dlnRho)
          end if

       case(2)

          if (q < 0) then
             call psi_Iec_and_Jec(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  Ip, dIp_dlnT, dIp_dlnRho, Jp, dJp_dlnT, dJp_dlnRho)
          else
             call psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  Ip, dIp_dlnT, dIp_dlnRho, Jp, dJp_dlnT, dJp_dlnRho)
          end if

       case(3)

          if (q < 0) then
             call psi_Iec_and_Jec(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  Im, dIm_dlnT, dIm_dlnRho, Jm, dJm_dlnT, dJm_dlnRho)
          else
             call psi_Iee_and_Jee(beta, zeta, eta, deta_dlnT, deta_dlnRho, &
                  Im, dIm_dlnT, dIm_dlnRho, Jm, dJm_dlnT, dJm_dlnRho)
          end if

       end select
    end do

    include 'formats'
    if (dx.ne.0) then
       write(*,'(6X, A12, 3F12.6)') "dI_lnRho", dI_dlnRho, (Ip - Im) / (2 * dx), dI_dlnRho/(Ip - Im) * dx * 2
       write(*,'(6X, A12, 3F12.6)') "dJ_lnRho", dJ_dlnRho, (Jp - Jm) / (2 * dx), dJ_dlnRho/(Jp - Jm) * dx * 2
    end if
    if (dy.ne.0) then
       write(*,'(6X, A12, 3F12.6)') "dI_dlnT", dI_dlnT, (Ip - Im) / (2 * dy), dI_dlnT/(Ip - Im) * dy * 2
       write(*,'(6X, A12, 3F12.6)') "dJ_dlnT", dJ_dlnT, (Jp - Jm) / (2 * dy), dJ_dlnT/(Jp - Jm) * dy * 2
    end if

  end subroutine do_test_derivs



  subroutine do_test_coulomb

    use eval_coulomb
    use rates_def, only: Coulomb_Info, which_mui_coulomb, which_vs_coulomb

    real(dp) :: mu1, mu2, vs
    real(dp) :: z1, z2
    type(Coulomb_Info), pointer :: cc

    include 'formats'

    which_mui_coulomb = PCR2009
    which_vs_coulomb = Itoh2002

    z1 = 12
    z2 = 11

    allocate(cc)

    call coulomb_set_context(cc, T, Rho, log10T, log10Rho, &
         zbar, abar, z2bar, 0d0, species, ya, za)

    mu1 = do_mui_coulomb(cc, z1)
    mu2 = do_mui_coulomb(cc, z2)

    vs = do_vs_coulomb(cc, z1)

    deallocate(cc)

    write(*,'(6X, A4, 3F26.16)') 'mu', mu1, mu2, abs(mu1-mu2) * kev * T
    write(*,'(6X, A4, F26.16)') 'vs', vs

  end subroutine do_test_coulomb


  subroutine do_test_special_weak(use_special)

    use const_lib
    use utils_lib
    use eos_def
    use eos_lib
    use chem_def
    use chem_lib
    use rates_def, only: Coulomb_Info
    use eval_coulomb

    logical, intent(in) :: use_special

    real(dp) :: Rho, T, Pgas, log10Rho, log10T
    real(dp) :: dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, d_dlnRho_const_T, d_dlnT_const_Rho
    real(dp), dimension(num_eos_basic_results) :: res, d_dlnd, d_dlnT, d_dabar, d_dzbar
    integer :: ierr

    integer :: i, ir, nr
    integer, pointer :: ids(:)
    type(Coulomb_Info), pointer :: cc

    real(dp), dimension(:), pointer :: &
         ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
         lambda, dlambda_dlnT, dlambda_dlnRho, &
         Q, dQ_dlnT, dQ_dlnRho, &
         Qneu, dQneu_dlnT, dQneu_dlnRho
    real(dp) :: lntwo, logT, T9, dT9, dlnT, YeRho, &
         ye, logRho, dlogRho, eta, d_eta_dlnT, d_eta_dlnRho
    character(len=iso_name_length), dimension(2) :: weak_lhs, weak_rhs

    allocate(cc)

    nr = 2

    allocate( &
         ids(nr), ldecay(nr), d_ldecay_dT9(nr), d_ldecay_dlYeRho(nr), &
         lcapture(nr), d_lcapture_dT9(nr), d_lcapture_dlYeRho(nr), &
         lneutrino(nr), d_lneutrino_dT9(nr), d_lneutrino_dlYeRho(nr), &
         lambda(nr), dlambda_dlnT(nr), dlambda_dlnRho(nr), &
         Q(nr), dQ_dlnT(nr), dQ_dlnRho(nr), &
         Qneu(nr), dQneu_dlnT(nr), dQneu_dlnRho(nr), &
         stat=ierr)


    ! pick reactions
    weak_lhs(1) = 'mg24'
    weak_rhs(1) = 'na24'

    weak_lhs(2) = 'na24'
    weak_rhs(2) = 'mg24'

    do i = 1, nr
       ids(i) = get_weak_rate_id(weak_lhs(i), weak_rhs(i))
    enddo

    logT = 8.3d0
    logRho = 9.8d0
    Ye = 0.5d0

    T = exp10_cr(logT)
    T9 = T*1d-9
    rho = exp10_cr(logRho)
    YeRho = Ye*rho

    ! get a set of results for given temperature and density
    call eosDT_get( &
         handle, Z, X, abar, zbar,   &
         species, chem_id, net_iso, xa, &
         Rho, logRho, T, logT,   &
         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, ierr)

    call coulomb_set_context(cc, T, Rho, log10T, log10Rho, &
         zbar, abar, z2bar, 0d0, species, ya, za)

    eta = res(i_eta)
    d_eta_dlnT = d_dlnT(i_eta)
    d_eta_dlnRho = d_dlnd(i_eta)


    if (use_special) then

       do_ecapture = .true.
       ecapture_states_file = 'test_special.states'
       ecapture_transitions_file = 'test_special.transitions'
       which_mui_coulomb = PCR2009
       which_vs_coulomb = Itoh2002

       call eval_ecapture_reaction_info( &
            nr, ids, cc, T9, YeRho, zbar, &
            eta, d_eta_dlnT, d_eta_dlnRho, &
            ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
            lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
            lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
            lambda, dlambda_dlnT, dlambda_dlnRho, &
            Q, dQ_dlnT, dQ_dlnRho, &
            Qneu, dQneu_dlnT, dQneu_dlnRho, &
            ierr)

       write(*,*) "special weak rates"
    else
       do_ecapture = .false.
       call eval_weak_reaction_info( &
            nr, ids, cc, T9, YeRho, -1d0, &
            eta, d_eta_dlnT, d_eta_dlnRho, &
            ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
            lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
            lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
            lambda, dlambda_dlnT, dlambda_dlnRho, &
            Q, dQ_dlnT, dQ_dlnRho, &
            Qneu, dQneu_dlnT, dQneu_dlnRho, &
            ierr)

       write(*,*) "weaklib weak rates"

    end if

    do i = 1, nr
       write(*,'(6X, 2A6, ES26.16)') weak_lhs(i), weak_rhs(i), lambda(i)
    enddo
    write(*,*)

    deallocate( &
         ids, ldecay, d_ldecay_dT9, d_ldecay_dlYeRho, &
         lcapture, d_lcapture_dT9, d_lcapture_dlYeRho, &
         lneutrino, d_lneutrino_dT9, d_lneutrino_dlYeRho, &
         lambda, dlambda_dlnT, dlambda_dlnRho, &
         Q, dQ_dlnT, dQ_dlnRho, &
         Qneu, dQneu_dlnT, dQneu_dlnRho)


  end subroutine do_test_special_weak

  subroutine Init_Composition
    use chem_def
    use chem_lib

    real(dp) :: frac, dabar_dx(species), dzbar_dx(species),   &
         sumx, xh, xhe, mass_correction, dmc_dx(species)

    allocate(net_iso(num_chem_isos), chem_id(species), stat=ierr)
    if (ierr /= 0) stop 'allocate failed'
    X = 0.70
    Z = 0.02

    net_iso(:) = 0

    chem_id(h1) = ih1; net_iso(ih1) = h1
    chem_id(he4) = ihe4; net_iso(ihe4) = he4
    chem_id(c12) = ic12; net_iso(ic12) = c12
    chem_id(n14) = in14; net_iso(in14) = n14
    chem_id(o16) = io16; net_iso(io16) = o16
    chem_id(ne20) = ine20; net_iso(ine20) = ne20
    chem_id(mg24) = img24; net_iso(img24) = mg24

    xa(h1) = 0
    xa(he4) = 0
    xa(c12) = 0
    xa(n14) = 0
    xa(o16) = 0.0
    xa(ne20) = 0.5
    xa(mg24) = 0.5

    do i = 1, species
       za(i) = chem_isos% Z(chem_id(i))
       aa(i) = chem_isos% W(chem_id(i))
       ya(i) = xa(i) / aa(i)
    enddo

    call composition_info( &
         species, chem_id, xa, xh, xhe, abar, zbar, z2bar, ye,   &
         mass_correction, sumx, dabar_dx, dzbar_dx, dmc_dx)

  end subroutine Init_Composition


  subroutine Setup_eos(handle)
    ! allocate and load the eos tables
    use eos_def
    use eos_lib
    integer, intent(out) :: handle

    character (len=256) :: eos_file_prefix
    integer :: ierr
    logical, parameter :: use_cache = .true.

    eos_file_prefix = 'mesa'

    call eos_init(eos_file_prefix, '', '', '', use_cache, ierr)
    if (ierr /= 0) then
       write(*,*) 'eos_init failed in Setup_eos'
       stop 1
    end if

    handle = alloc_eos_handle(ierr)
    if (ierr /= 0) then
       write(*,*) 'failed trying to allocate eos handle'
       stop 1
    end if

  end subroutine Setup_eos


end module test_ecapture
