! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************
 
      module reaclib_support
      use rates_def
      use reaclib_support_mic
      use crlibm_lib
      
      implicit none

      contains
      

      subroutine set_up_network_information(rates)
      	type(reaction_data), intent(inout) :: rates
      	integer :: current_chapter, i

      	! set up bookmarks
      	current_chapter = 0
      	rates% nchapters_included = 0
      	rates% chapters_present = 0
      	rates% bookmarks = 0
      	do i = 1, rates% nreactions
      		new_chapter : if (rates% chapter(i) /= current_chapter) then
      			! close out the chapter we just left.
      			if (current_chapter /= 0) rates% bookmarks(2,current_chapter) = i-1
      			! set up information on the new chapter
      			current_chapter = rates% chapter(i)
      			rates% nchapters_included = rates% nchapters_included + 1
      			rates% chapters_present(rates% nchapters_included) = current_chapter
      			rates% bookmarks(1,current_chapter) = i
      		end if new_chapter
      	end do
      	! mark the end of the last chapter
      	rates% bookmarks(2,current_chapter) = rates% nreactions
      end subroutine set_up_network_information
      

      subroutine assign_weights(rates)
      	type(reaction_data), intent(inout) :: rates
      	integer :: i, i1, i2, i3, i4
      	
      	include 'formats.dek'

      	! check for allocation
      	if (.not.associated(rates% weight)) then
      		return
      	end if
      	
      	do i = 1, rates% nreactions
      	   i1 = -1; i2 = -2; i3 = -3; i4 = -4
      	   select case (rates% chapter(i))
      	      case (r_one_one)
      	      case (r_one_two)
      	      case (r_one_three)
      	      case (r_two_one)
      	         i1 = rates% pspecies(1,i)
      	         i2 = rates% pspecies(2,i)
      	      case (r_two_two)
      	         i1 = rates% pspecies(1,i)
      	         i2 = rates% pspecies(2,i)
      	      case (r_two_three)
      	         i1 = rates% pspecies(1,i)
      	         i2 = rates% pspecies(2,i)
      	      case (r_two_four)
      	         i1 = rates% pspecies(1,i)
      	         i2 = rates% pspecies(2,i)
      	      case (r_three_one)
      	         i1 = rates% pspecies(1,i)
      	         i2 = rates% pspecies(2,i)
      	         i3 = rates% pspecies(3,i)
      	      case (r_three_two)
      	         i1 = rates% pspecies(1,i)
      	         i2 = rates% pspecies(2,i)
      	         i3 = rates% pspecies(3,i)
      	      case (r_four_two)
      	         i1 = rates% pspecies(1,i)
      	         i2 = rates% pspecies(2,i)
      	         i3 = rates% pspecies(3,i)
      	         i4 = rates% pspecies(4,i)
      	      case (r_one_four)
      	   end select
            call set_weight(rates% weight(i))
         end do
      	
      	do i = 1, rates% nreactions
      	   i1 = -1; i2 = -2; i3 = -3; i4 = -4
      	   select case (rates% chapter(i))
      	      case (r_one_one)
      	      case (r_one_two)
      	         i1 = rates% pspecies(2,i)
      	         i2 = rates% pspecies(3,i)
      	      case (r_one_three)
      	         i1 = rates% pspecies(2,i)
      	         i2 = rates% pspecies(3,i)
      	         i3 = rates% pspecies(4,i)
      	      case (r_two_one)
      	      case (r_two_two)
      	         i1 = rates% pspecies(3,i)
      	         i2 = rates% pspecies(4,i)
      	      case (r_two_three)
      	         i1 = rates% pspecies(3,i)
      	         i2 = rates% pspecies(4,i)
      	         i3 = rates% pspecies(5,i)
      	      case (r_two_four)
      	         i1 = rates% pspecies(3,i)
      	         i2 = rates% pspecies(4,i)
      	         i3 = rates% pspecies(5,i)
      	         i4 = rates% pspecies(6,i)
      	      case (r_three_one)
      	      case (r_three_two)
      	         i1 = rates% pspecies(4,i)
      	         i2 = rates% pspecies(5,i)
      	      case (r_four_two)
      	         i1 = rates% pspecies(5,i)
      	         i2 = rates% pspecies(6,i)
      	      case (r_one_four)
      	         i1 = rates% pspecies(2,i)
      	         i2 = rates% pspecies(3,i)
      	         i3 = rates% pspecies(4,i)
      	         i4 = rates% pspecies(5,i)
      	   end select

            call set_weight(rates% weight_reverse(i))

         end do
         
         
         contains
         
         
         subroutine set_weight(w)
            ! nuclei are sorted, so if identical, then are adjacent in list
            double precision, intent(out) :: w
            if (i1 == i2 .and. i2 == i3 .and. i3 == i4) then
               w = 1d0/24d0
            else if (i2 == i3 .and. (i1 == i2 .or. i3 == i4)) then
               w = 1d0/6d0
            else if (i1 == i2) then
               if (i3 == i4) then
                  w = 1d0/4d0
               else
                  w = 1d0/2d0
               end if
            else if (i2 == i3 .or. i3 == i4) then
               w = 1d0/2d0
            else
               w = 1d0
            end if
         end subroutine set_weight
         

      end subroutine assign_weights
      

      subroutine compute_rev_ratio(rates,winvn)
      	use const_def, only : pi, kB=>boltzm, NA=>avo, hbar, &
      		c=>clight, conv=>mev_to_ergs
      	type(reaction_data), intent(inout) :: rates
      	type(nuclide_data), intent(in) :: winvn
      	double precision, parameter :: mp = 1.00727646688d0, mn = 1.00866491578d0
      	double precision,  dimension(max_species_per_reaction) :: g	! statistical weights of nuclides
      	double precision, dimension(max_species_per_reaction) :: mass	! mass no's of nuclides
      	integer, dimension(max_species_per_reaction) :: ps
      	integer :: Ni,No,Nt,i
      	double precision :: fac, massfac, sum1, sum2, tmp
      	
      	include 'formats.dek'
      	
      	fac = pow_cr(1d9*kB/(2d0*pi*hbar*hbar*NA),1.5d0)/NA
      	massfac = conv*NA/c**2

      	rates% weak_mask = 1.0
      	loop_over_rates: do i = 1,rates% nreactions
      		! weak rates don't have inverses, so set to innocuous values and mask out
      		if (rates% reaction_flag(i) == 'w' .or. rates% reaction_flag(i) == 'e') then
      			rates% weak_mask(i) = 0.0
      			rates% inverse_coefficients(:,i) = (/-huge(1.0),0.0/)
      			rates% inverse_exp(i) = 0.0
      			rates% inverse_part(:,i) = 1.0
      			cycle
      		end if
      		Ni = Nin(rates% chapter(i))
      		No = Nout(rates% chapter(i))
      		Nt = Ni+No
      		ps(1:Nt) = rates% pspecies(1:Nt,i)
      		g(1:Nt) = 2.0*winvn% spin(ps(1:Nt))+1.0
      		mass(1:Nt) = winvn% Z(ps(1:Nt))*mp + winvn% N(ps(1:Nt))*mn - winvn% binding_energy(ps(1:Nt))*massfac

      		! log(prefactor of reverse_ratio)
      		tmp = product(mass(1:Ni))/product(mass(Ni+1:Nt))
      		rates% inverse_coefficients(1,i) = log_cr(pow_cr(fac,dble(Ni-No)) &
      		 	&	*tmp*sqrt(tmp)*(product(g(1:Ni))/product(g(Ni+1:Nt))) &
      			& * rates% weight(i))

      		! -Q/(kB*10**9)
      		sum1 = sum(winvn% binding_energy(ps(1:Ni)))
      		sum2 = sum(winvn% binding_energy(ps(Ni+1:Nt)))
      		rates% inverse_coefficients(2,i) = (sum1-sum2)*conv/kB/1d9
      		rates% inverse_exp(i) = Ni-No

      		rates% inverse_part(:,i) = product(winvn% pfcn(1:npart,ps(1:Ni)),dim=2)/ &
      		 	& product(winvn% pfcn(1:npart,ps(Ni+1:Nt)),dim=2)
      	end	do loop_over_rates
      end subroutine compute_rev_ratio


      end module reaclib_support
