! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************






      module ratelib_mic
      use const_def, only: dp
      use crlibm_lib
      
      implicit none
      
      real(dp), parameter :: lowT9_cutoff = 1d-3 ! all non-pp rates except decays go to 0 below this

      contains


      

#ifdef offload
      !dir$ attributes offload: mic :: mazurek_init
#endif         
      subroutine mazurek_init(ierr)
         use rates_def_mic, only: tv,rv,rfdm,rfd0,rfd1,rfd2,tfdm,tfd0,tfd1,tfd2
         integer, intent(out) :: ierr
         integer :: k,j
         rv(:) = (/ 6D0, 7D0, 8D0, 9D0, 10D0, 11D0 /)
         tv(:) = (/ 2D0, 4D0, 6D0, 8D0, 10D0, 12D0, 14D0 /)
         ierr = 0
         do k=2,4 
            rfdm(k)=1./((rv(k-1)-rv(k))*(rv(k-1)-rv(k+1))*(rv(k-1)-rv(k+2))) 
            rfd0(k)=1./((rv(k)-rv(k-1))*(rv(k)-rv(k+1))*(rv(k)-rv(k+2))) 
            rfd1(k)=1./((rv(k+1)-rv(k-1))*(rv(k+1)-rv(k))*(rv(k+1)-rv(k+2))) 
            rfd2(k)=1./((rv(k+2)-rv(k-1))*(rv(k+2)-rv(k))*(rv(k+2)-rv(k+1))) 
         enddo
         do j=2,5 
            tfdm(j)=1./((tv(j-1)-tv(j))*(tv(j-1)-tv(j+1))*(tv(j-1)-tv(j+2))) 
            tfd0(j)=1./((tv(j)-tv(j-1))*(tv(j)-tv(j+1))*(tv(j)-tv(j+2))) 
            tfd1(j)=1./((tv(j+1)-tv(j-1))*(tv(j+1)-tv(j))*(tv(j+1)-tv(j+2))) 
            tfd2(j)=1./((tv(j+2)-tv(j-1))*(tv(j+2)-tv(j))*(tv(j+2)-tv(j+1))) 
         enddo
      end subroutine mazurek_init

#ifdef offload
      !dir$ attributes offload: mic :: mazurek
#endif         
      subroutine mazurek(btemp,bden,y56,ye,rn56ec,sn56ec)       
      use rates_def_mic, only: tv,rv,rfdm,rfd0,rfd1,rfd2,tfdm,tfd0,tfd1,tfd2
      real(dp), intent(in) :: btemp,bden,y56,ye
      real(dp), intent(out) :: rn56ec,sn56ec

!  this routine evaluates mazurek's 1973 fits for the ni56 electron 
!  capture rate rn56ec and neutrino loss rate sn56ec 

!  input: 
!  y56 = nickel56 molar abundance
!  ye  = electron to baryon number, zbar/abar

!  output:
!  rn56ec = ni56 electron capture rate
!  sn56ec = ni56 neutrino loss rate

!  declare 
      integer          ifirst,jp,kp,jr,jd,ii,ik,ij,j,k 
      real(dp) rnt(2),rne(2,7),datn(2,6,7),  &
                       t9,r,rfm,rf0,rf1,rf2,dfacm,dfac0,dfac1,dfac2,  &
                       tfm,tf0,tf1,tf2,tfacm,tfac0,tfac1,tfac2

!  initialize 
      data (((datn(ii,ik,ij),ik=1,6),ij=1,7),ii=1,1) /  &
          -3.98, -2.84, -1.41,  0.20,  1.89,  3.63,  &
          -3.45, -2.62, -1.32,  0.22,  1.89,  3.63,  &
          -2.68, -2.30, -1.19,  0.27,  1.91,  3.62,  &
          -2.04, -1.87, -1.01,  0.34,  1.94,  3.62,  &
          -1.50, -1.41, -0.80,  0.45,  1.99,  3.60,  &
          -1.00, -0.95, -0.54,  0.60,  2.06,  3.58,  &
          -0.52, -0.49, -0.21,  0.79,  2.15,  3.55 / 
      data (((datn(ii,ik,ij),ik=1,6),ij=1,7),ii=2,2) /  &
          -3.68, -2.45, -0.80,  1.12,  3.13,  5.19,  &
          -2.91, -2.05, -0.64,  1.16,  3.14,  5.18,  &
          -1.95, -1.57, -0.40,  1.24,  3.16,  5.18,  &
          -1.16, -0.99, -0.11,  1.37,  3.20,  5.18,  &
          -0.48, -0.40,  0.22,  1.54,  3.28,  5.16,  &
           0.14,  0.19,  0.61,  1.78,  3.38,  5.14,  &
           0.75,  0.78,  1.06,  2.07,  3.51,  5.11 / 

!  calculate ni56 electron capture and neutrino loss rates 
      rn56ec = 0.0 
      sn56ec = 0.0 

      if (btemp*1d-9 < lowT9_cutoff) return
          
      if ( (btemp .lt. 2.0e9) .or. (bden*ye .lt. 1.0e6)) return 
      t9    = max(btemp,1.4d10) * 1.0d-9 
      r     = max(6.0d0,min(11.0d0,log10_cr(bden*ye))) 
      jp    = min(max(2,int(0.5d0*t9)),5) 
      kp    = min(max(2,int(r)-5),4) 
      rfm   = r - rv(kp-1) 
      rf0   = r - rv(kp) 
      rf1   = r - rv(kp+1) 
      rf2   = r - rv(kp+2) 
      dfacm = rf0*rf1*rf2*rfdm(kp) 
      dfac0 = rfm*rf1*rf2*rfd0(kp) 
      dfac1 = rfm*rf0*rf2*rfd1(kp) 
      dfac2 = rfm*rf0*rf1*rfd2(kp) 
      tfm   = t9 - tv(jp-1) 
      tf0   = t9 - tv(jp) 
      tf1   = t9 - tv(jp+1) 
      tf2   = t9 - tv(jp+2) 
      tfacm = tf0*tf1*tf2*tfdm(jp) 
      tfac0 = tfm*tf1*tf2*tfd0(jp) 
      tfac1 = tfm*tf0*tf2*tfd1(jp) 
      tfac2 = tfm*tf0*tf1*tfd2(jp) 

!  evaluate the spline fits
      do jr = 1,2 
       do jd = jp-1,jp+2 
        rne(jr,jd) =   dfacm*datn(jr,kp-1,jd) + dfac0*datn(jr,kp,jd)  &
                     + dfac1*datn(jr,kp+1,jd) + dfac2*datn(jr,kp+2,jd) 
       enddo
       rnt(jr) =  tfacm*rne(jr,jp-1) + tfac0*rne(jr,jp)  &
                + tfac1*rne(jr,jp+1) + tfac2*rne(jr,jp+2) 
      enddo

!  set the output
      rn56ec = exp10_cr(rnt(1))
      sn56ec = 6.022548d+23 * 8.18683d-7 * y56 * exp10_cr(rnt(2))
      return 
      end subroutine mazurek

      
#ifdef offload
      !dir$ attributes offload: mic :: n14_electron_capture_rate
#endif         
      subroutine n14_electron_capture_rate(T,Rho,UE,rate)
         real(dp), intent(in) :: T ! temperature
         real(dp), intent(in) :: Rho ! density
         real(dp), intent(in) :: UE ! electron molecular weight
         real(dp), intent(out) :: rate ! (s^-1)
         
         real(dp) :: Q, AMC2, AMULTIP, AL92, T8, X, XFER, EF, Y, AA, GUESS, ELCAP
         
         ! from Lars
         
      
!      Inputs are T in K, rho in gr/cm^3, and UE=electron mean mol. weight
!
!     Gives a reasonable estimate (i.e. within factor of 50% or so) of the 
!     electron capture rate for electrons on 14N in a plasma assumed to be quite 
!     degenerate. 
!
!         x=KT/Q, y=E_FERMI/Q 
!   
!      ELCAP is the rate in 1/seconds 
!
!
!     Let's start by putting in the Q value, electron rest mass and 
!     temperature in units of keV.
!
!
      Q=667.479
      AMC2=510.999
      AMULTIP=0.693/1.104e9
      AL92=LOG_CR(9d0/2d0)
      T8=T/1E8
      X=8.617*T8/Q
!
!     For this value of the density, find the electron fermi momentum 
!     assuming that the KT corrections to the electron EOS are not
!     important. 
!
      XFER=pow_cr(RHO/(0.9739E6*UE),1d0/3d0) 
!
!      The parameter we need that is used in the fitting formula is
!      the electron Fermi energy 
!
      EF=AMC2*SQRT(1.0D0+XFER*XFER)
      Y=EF/Q
      IF(Y.LT.(1.0D0+AL92*X)) THEN 
          AA=(Y-1.0D0)/X
          GUESS=2.0*X*X*X*exp_cr(AA)
      ELSE
          GUESS=pow3(Y-1.0D0+(3.0-AL92)*X)/3.0D0 
      ENDIF
!
!     Now multiply by the prefactors .. .
!
      ELCAP=GUESS*AMULTIP


         rate = ELCAP
         
      end subroutine n14_electron_capture_rate


#ifdef offload
      !dir$ attributes offload: mic :: ecapnuc
#endif         
      subroutine ecapnuc(etakep,temp,rpen,rnep,spen,snep)
      real(dp), intent(in) :: etakep,temp
      real(dp), intent(out) :: rpen,rnep,spen,snep

!  given the electron degeneracy parameter etakep (chemical potential
!  without the electron's rest mass divided by kt) and the temperature temp,
!  this routine calculates rates for 
!  electron capture on protons rpen (captures/sec/proton),
!  positron capture on neutrons rnep (captures/sec/neutron), 
!  and their associated neutrino energy loss rates 
!  spen (ergs/sec/proton) and snep (ergs/sec/neutron)

!  declare
      integer          iflag
      real(dp) t9,t5,qn,etaef,etael,zetan,eta,etael2, &
                       etael3,etael4,f1l,f2l,f3l,f4l,f5l,f1g, &
                       f2g,f3g,f4g,f5g,exmeta,eta2,eta3,eta4, &
                       fac0,fac1,fac2,fac3,rie1,rie2,facv0,facv1, &
                       facv2,facv3,facv4,rjv1,rjv2,spenc,snepc, &
                       pi2,exeta,zetan2,f0,etael5, &
                       qn1,ft,twoln,cmk5,cmk6,bk,pi,qn2,c2me, &
                       xmp,xmn,qndeca,tmean
      parameter        (qn1    = -2.0716446d-06, &
                        ft     = 1083.9269d0, &
                        twoln  = 0.6931472d0, &
                        cmk5   = 1.3635675d-49, &
                        cmk6   = 2.2993864d-59, &
                        bk     = 1.38062e-16, &
                        pi     = 3.1415927d0, &
                        pi2    = pi * pi, &
                        qn2    = 2.0716446d-06, &
                        c2me   = 8.1872665d-07, &
                        xmp    = 1.6726485d-24, &
                        xmn    = 1.6749543d-24, &
                        qndeca = 1.2533036d-06, &
                        tmean  = 886.7d0)
      


!  tmean and qndeca are the mean lifetime and decay energy of the neutron
!  xmp,xnp are masses of the p and n in grams.
!  c2me is the constant used to convert the neutrino energy
!  loss rate from mec2/s (as in the paper) to ergs/particle/sec.

!  initialize
      rpen  = 0.0d0
      rnep  = 0.0d0
      spen  = 0.0d0
      snep  = 0.0d0
      t9    = temp * 1.0d-9

      if (t9 < lowT9_cutoff) return

      iflag = 0
      qn    = qn1
          

!  chemical potential including the electron rest mass
      etaef = etakep + c2me/bk/temp


!  iflag=1 is for electrons,  iflag=2 is for positrons
502   iflag = iflag + 1
      if (iflag.eq.1) etael = qn2/bk/temp
      if (iflag.eq.2) etael = c2me/bk/temp
      if (iflag.eq.2) etaef = -etaef

      t5    = temp*temp*temp*temp*temp
      zetan = qn/bk/temp
      eta   = etaef - etael

!  protect from overflowing with large eta values
      if (eta .le. 6.8e+02) then
       exeta = exp_cr(eta)
      else 
       exeta = 0.0d0
      end if
      etael2 = etael*etael
      etael3 = etael2*etael
      etael4 = etael3*etael
      etael5 = etael4*etael
      zetan2 = zetan*zetan
      if (eta .le. 6.8e+02) then
       f0 = log1p_cr(exeta)
      else
       f0 = eta
      end if

!  if eta le. 0., the following fermi integrals apply
      f1l = exeta
      f2l = 2.0d0   * f1l
      f3l = 6.0d0   * f1l
      f4l = 24.0d0  * f1l
      f5l = 120.0d0 * f1l

!  if eta gt. 0., the following fermi integrals apply:
      f1g = 0.0d0
      f2g = 0.0d0
      f3g = 0.0d0
      f4g = 0.0d0
      f5g = 0.0d0
      if (eta .gt. 0.0) then
       exmeta = exp_cr(-eta)
       eta2   = eta*eta
       eta3   = eta2*eta
       eta4   = eta3*eta
       f1g = 0.5d0*eta2 + 2.0d0 - exmeta
       f2g = eta3/3.0d0 + 4.0d0*eta + 2.0d0*exmeta
       f3g = 0.25d0*eta4 + 0.5d0*pi2*eta2 + 12.0d0 - 6.0d0*exmeta
       f4g = 0.2d0*eta4*eta + 2.0d0*pi2/3.0d0*eta3 + 48.0d0*eta &
             + 24.0d0*exmeta
       f5g = eta4*eta2/6.0d0 + 5.0d0/6.0d0*pi2*eta4  &
             + 7.0d0/6.0d0*pi2*eta2  + 240.0d0 -120.d0*exmeta
       end if

!  factors which are multiplied by the fermi integrals
      fac3 = 2.0d0*zetan + 4.0d0*etael
      fac2 = 6.0d0*etael2 + 6.0d0*etael*zetan + zetan2
      fac1 = 4.0d0*etael3 + 6.0d0*etael2*zetan + 2.0d0*etael*zetan2
      fac0 = etael4 + 2.0d0*zetan*etael3 + etael2*zetan2

!  electron capture rates onto protons with no blocking
      rie1 = f4l + fac3*f3l + fac2*f2l + fac1*f1l + fac0*f0
      rie2 = f4g + fac3*f3g + fac2*f2g + fac1*f1g + fac0*f0

!  neutrino emission rate for electron capture:
      facv4 = 5.0d0*etael + 3.0d0*zetan
      facv3 = 10.0d0*etael2 + 12.0d0*etael*zetan + 3.0d0*zetan2
      facv2 = 10.0d0*etael3 + 18.0d0*etael2*zetan &
              + 9.0d0*etael*zetan2 + zetan2*zetan
      facv1 = 5.0d0*etael4 + 12.0d0*etael3*zetan  &
              + 9.0d0*etael2*zetan2 + 2.0d0*etael*zetan2*zetan
      facv0 = etael5 + 3.0d0*etael4*zetan &
              + 3.0d0*etael3*zetan2 + etael2*zetan2*zetan
      rjv1  = f5l + facv4*f4l + facv3*f3l &
              + facv2*f2l + facv1*f1l + facv0*f0
      rjv2  = f5g + facv4*f4g + facv3*f3g &
              + facv2*f2g + facv1*f1g + facv0*f0

!  for electrons capture onto protons
      if (iflag.eq.2) go to 503
      if (eta.gt.0.) go to 505
      rpen  = twoln*cmk5*t5*rie1/ft
      spen  = twoln*cmk6*t5*temp*rjv1/ft
      spenc = twoln*cmk6*t5*temp*rjv1/ft*c2me
      go to 504
505   rpen = twoln*cmk5*t5*rie2/ft
      spen = twoln*cmk6*t5*temp*rjv2/ft
      spenc = twoln*cmk6*t5*temp*rjv2/ft*c2me
504   continue
      qn = qn2
      go to 502

!  for positrons capture onto neutrons
503   if (eta.gt.0.) go to 507
      rnep  = twoln*cmk5*t5*rie1/ft
      snep  = twoln*cmk6*t5*temp*rjv1/ft
      snepc = twoln*cmk6*t5*temp*rjv1/ft*c2me
!      if (rho.lt.1.0e+06) snep=snep+qndeca*xn(9)/xmn/tmean
      go to 506
507   rnep  = twoln*cmk5*t5*rie2/ft
      snep  = twoln*cmk6*t5*temp*rjv2/ft
      snepc = twoln*cmk6*t5*temp*rjv2/ft*c2me
!      if (rho.lt.1.0e+06) snep=snep+qndeca*xn(9)/xmn/tmean
506   continue
      return
      end subroutine ecapnuc
      end module ratelib_mic
      
      


