! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module load_weak
      
      use rates_def
      use const_def, only: dp
      
      implicit none
      
      integer, parameter :: cache_version = 6
      
      private :: private_load_weak_tables


      contains
      
      
      subroutine load_weak_data(ierr)
         integer, intent(out) :: ierr         
         ierr = 0         
         call private_load_weak_tables(ierr)
         if (ierr /= 0) return
         call load_weak_info_list(ierr)
      end subroutine load_weak_data
      
      
      subroutine load_weak_info_list(ierr)
         use utils_lib
         use crlibm_lib, only: str_to_vector
         integer, intent(out) :: ierr
         
         integer :: iounit, i, nvec
         character (len=256) :: filename, string
	      character(len=iso_name_length) :: lhs, rhs
	      character(len=2*iso_name_length+1) :: key
	      real(dp), target :: vec_ary(2)
	      real(dp), pointer :: vec(:)
	      integer, parameter :: max_num_weak_info = 1000

         logical, parameter :: dbg = .false.

         include 'formats.dek'
         
         ierr = 0
         vec => vec_ary
         
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return

         filename = trim(weak_data_dir) // '/weak_info.list'
         ierr = 0
         open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
         if (ierr /= 0) then
            call free_iounit(iounit)
            write(*,*) 'failed to open ' // trim(filename)
            call free_iounit(iounit)
            return
         end if
         
         if (dbg) then
            write(*,*)
            write(*,*) 'weak info filename <' // trim(filename) // '>'
            write(*,*)
         end if

         do ! skip to line starting with 'from '
            read(iounit,'(a)',iostat=ierr) string
            if (failed('read weak info comments')) return
            if (len_trim(string) > 4) then
               if (string(1:5) == 'from ') exit
            end if
         end do

         nullify(weak_info_list_dict)
         allocate(weak_info_list_halflife(max_num_weak_info))
         allocate(weak_info_list_Qneu(max_num_weak_info))
         num_weak_info_list_reactions = 0
         do i = 1, max_num_weak_info ! keep reading until end of file         
            read(iounit,fmt='(a5,a5,a)',iostat=ierr) lhs, rhs, string
            if (ierr == 0) then
               call str_to_vector(string, vec, nvec, ierr)
               if (nvec < 2) ierr = -1
            end if
            if (ierr /= 0) then
               ierr = 0; exit
            end if
            weak_info_list_halflife(i) = vec(1)
            weak_info_list_Qneu(i) = vec(2)    
            call create_weak_dict_key(lhs, rhs, key)
            !write(*,'(a)') 'weak info list key ' // trim(key)
            call integer_dict_define(weak_info_list_dict, key, i, ierr)
            if (failed('integer_dict_define')) return
            num_weak_info_list_reactions = i
         end do
         
         close(iounit)
         
         call free_iounit(iounit)
         
         if (num_weak_info_list_reactions == 0) then
            ierr = -1
            write(*,*) 'failed trying to read weak_info.list -- no reactions?'
            return
         end if
         
         if (num_weak_info_list_reactions == max_num_weak_info) then
            ierr = -1
            write(*,*) 'failed trying to read weak_info.list -- too many reactions?'
            return
         end if
         
         call integer_dict_create_hash(weak_info_list_dict, ierr)
         if (ierr /= 0) return
         
         call realloc_double(weak_info_list_halflife, num_weak_info_list_reactions, ierr)
         if (ierr /= 0) return
         
         call realloc_double(weak_info_list_Qneu, num_weak_info_list_reactions, ierr)
         if (ierr /= 0) return
         
         
         contains
         
         logical function failed(str)
            character (len=*) :: str
            failed = (ierr /= 0)
            if (failed) then
               call free_iounit(iounit)
               write(*,*) 'failed: ' // trim(str)
            end if
         end function failed
         
         
      end subroutine load_weak_info_list
      
      
      subroutine private_load_weak_tables(ierr)
         use utils_lib
         use chem_lib, only: chem_get_iso_id
         use chem_def, only: iso_name_length
         integer, intent(out) :: ierr
         
         integer :: iounit, i, ios, id
         character (len=256) :: filename, cache_filename, string
	      character(len=iso_name_length) :: lhs1, rhs1, lhs2, rhs2, weak_lhs, weak_rhs
	      character(len=2*iso_name_length+1) :: key

         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0
         iounit = alloc_iounit(ierr)
         if (ierr /= 0) return
         
         cache_filename = trim(rates_cache_dir) // '/weakreactions.bin'
         ios = 0
         
         open(unit=iounit,file=trim(cache_filename),action='read', &
            status='old',iostat=ios,form='unformatted')
         if (ios == 0) then ! opened it okay
            call read_weak_cache(iounit,ios)
            close(iounit)
         end if         
         
         if (ios /= 0) then ! need to read data file
         
            filename = trim(weak_data_dir) // '/weakreactions.tables'
            ierr = 0
            open(unit=iounit, file=trim(filename), action='read', status='old', iostat=ierr)
            if (ierr /= 0) then
               call free_iounit(iounit)
               write(*,*) 'failed to open ' // trim(filename)
               call free_iounit(iounit)
               return
            end if
         
            if (dbg) then
               write(*,*)
               write(*,*) 'weaklib filename <' // trim(filename) // '>'
               write(*,*)
            end if

            do ! skip to after line starting with '='
               read(iounit,'(a)',iostat=ierr) string
               if (failed('read header')) return
               if (len_trim(string) > 0) then
                  if (string(1:1) == '=') exit
               end if
            end do

            if (.not. skip_line()) return
         
            read(iounit,*,iostat=ierr) num_weak_reactions
            if (failed('read num_weak_reactions')) return
         
            if (dbg) write(*,2) 'num_weak_reactions', num_weak_reactions
         
            call alloc
            if (failed('allocate')) return
         
            do i = 1, num_weak_reactions
               if (.not. skip_line()) return
               if (mod(i,2)==1) then ! first of pair
                  if (.not. skip_line()) return
                  if (.not. skip_line()) return
                  read(iounit,fmt='(2a5)',iostat=ierr) lhs1, rhs1
                  if (failed('read lhs1, rhs1')) return
                  if (lhs1 == 'al-6') lhs1 = 'al26-1'
                  if (rhs1 == 'al-6') rhs1 = 'al26-1'
                  if (lhs1 == 'al*6') lhs1 = 'al26-2'
                  if (rhs1 == 'al*6') rhs1 = 'al26-2'
                  read(iounit,fmt='(2a5)',iostat=ierr) lhs2, rhs2
                  if (failed('read lhs2, rhs2')) return
                  if (lhs2 == 'al-6') lhs2 = 'al26-1'
                  if (rhs2 == 'al-6') rhs2 = 'al26-1'
                  if (lhs2 == 'al*6') lhs2 = 'al26-2'
                  if (rhs2 == 'al*6') rhs2 = 'al26-2'
                  if (.not. skip_line()) return
                  if (.not. skip_line()) return
                  weak_lhs = lhs1
                  weak_rhs = rhs1
               else
                  weak_lhs = lhs2
                  weak_rhs = rhs2
               end if
               call adjust_name(weak_lhs)
               call adjust_name(weak_rhs)
               id = chem_get_iso_id(weak_lhs)
               if (id <= 0) then
                  write(*,*) 'weaklib FATAL ERROR: unknown nuclide ' // weak_lhs
                  stop 1
               end if
               weak_lhs_nuclide_id(i) = id
               id = chem_get_iso_id(weak_rhs)
               if (id <= 0) then
                  write(*,*) 'weaklib FATAL ERROR: unknown nuclide ' // weak_rhs
                  stop 1
               end if
               weak_reaclib_id(i) = 0
               weak_rhs_nuclide_id(i) = id
               weak_lhs_nuclide_name(i) = weak_lhs
               weak_rhs_nuclide_name(i) = weak_rhs
               if (.not. skip_line()) return
               call read_table(i,i_ldecay)
               if (failed('read ldecay')) return
               if (.not. skip_line()) return
               call read_table(i,i_lcapture)
               if (failed('read lcapture')) return
               if (.not. skip_line()) return
               call read_table(i,i_lneutrino)
               if (failed('read lneutrino')) return
            end do
         
            close(iounit)
         
            open(unit=iounit, file=trim(cache_filename), iostat=ios, &
                  action='write', form='unformatted')
            if (ios == 0) then
               call write_weak_cache(iounit)
               close(iounit)
            end if
         
         end if
         
         call free_iounit(iounit)
         
         nullify(weak_reactions_dict)
         do i = 1, num_weak_reactions
            call create_weak_dict_key(weak_lhs_nuclide_name(i), weak_rhs_nuclide_name(i), key)
            call integer_dict_define(weak_reactions_dict, key, i, ierr)
            if (failed('integer_dict_define')) return
         end do
         
         call integer_dict_create_hash(weak_reactions_dict, ierr)
         if (failed('integer_dict_create_hash')) return
         
         if (weak_bicubic) then
            do i = 1, num_weak_reactions
               if (ierr == 0) call create_bicubic_interpolant(i,i_ldecay,ierr)
               if (ierr == 0) call create_bicubic_interpolant(i,i_lcapture,ierr)
               if (ierr == 0) call create_bicubic_interpolant(i,i_lneutrino,ierr)
            end do
            if (failed('create_bicubic_interpolant')) return
         else
            do i = 1, num_weak_reactions
               if (ierr == 0) call create_pm_T9_interpolant(i,i_ldecay,ierr)
               if (ierr == 0) call create_pm_T9_interpolant(i,i_lcapture,ierr)
               if (ierr == 0) call create_pm_T9_interpolant(i,i_lneutrino,ierr)
            end do
            if (failed('create_pm_T9_interpolant')) return
         end if
         
         if (dbg) write(*,*) 'finished load_weak_tables'
         
         
         contains


         subroutine read_weak_cache(iounit,ios)
            integer, intent(in) :: iounit
            integer, intent(out) :: ios
            integer :: n
            
            include 'formats'
         
            read(iounit,iostat=ios) num_weak_reactions
            if (ios /= 0) return
         
            if (dbg) write(*,2) 'num_weak_reactions', num_weak_reactions
         
            call alloc
            if (failed('allocate')) return
            
            n = num_weak_reactions
            
            read(iounit,iostat=ios) &
               weak_lhs_nuclide_id(1:n), &
               weak_rhs_nuclide_id(1:n), &
               weak_reaclib_id(1:n), &
               weak_lhs_nuclide_name(1:n), &
               weak_rhs_nuclide_name(1:n), &
               weak_reactions_data1(1:4*weak_num_T9*weak_num_lYeRho*3*n)
               
         end subroutine read_weak_cache


         subroutine write_weak_cache(iounit)
            integer, intent(in) :: iounit
            integer :: n
            
            include 'formats'
         
            write(iounit) num_weak_reactions
            
            n = num_weak_reactions
            
            write(iounit) &
               weak_lhs_nuclide_id(1:n), &
               weak_rhs_nuclide_id(1:n), &
               weak_reaclib_id(1:n), &
               weak_lhs_nuclide_name(1:n), &
               weak_rhs_nuclide_name(1:n), &
               weak_reactions_data1(1:4*weak_num_T9*weak_num_lYeRho*3*n)
               
         end subroutine write_weak_cache
                        
         
         subroutine alloc
         
            allocate( &
               weak_reaclib_id(num_weak_reactions), &
               weak_lhs_nuclide_name(num_weak_reactions), &
               weak_rhs_nuclide_name(num_weak_reactions), &
               weak_lhs_nuclide_id(num_weak_reactions), &
               weak_rhs_nuclide_id(num_weak_reactions), &
               weak_reactions_data1(4*weak_num_T9*weak_num_lYeRho*3*num_weak_reactions), &
               stat=ierr)
               
            weak_reactions_data(1:4, 1:weak_num_T9, 1:weak_num_lYeRho, 1:3, 1:num_weak_reactions) => &
               weak_reactions_data1(1:4*weak_num_T9*weak_num_lYeRho*3*num_weak_reactions)
               
         end subroutine alloc
         
         
         subroutine adjust_name(nm)
            character(len=iso_name_length) :: nm
            nm = adjustl(nm)
            if (nm == 'p') then
               nm = 'h1'
            else if (nm == 'n') then
               nm = 'neut'
            end if
         end subroutine adjust_name
         
         
         subroutine create_pm_T9_interpolant(i,ii,ierr)
            ! piecewise monotonic interpolation in T9 for each lYeRho in table
            use interp_1d_def
            use interp_1d_lib
            integer, intent(in) :: i, ii
            integer, intent(out) :: ierr
            integer :: j, m, n

            integer :: nx       ! length of x vector (>= 2)
            real(dp), pointer :: x(:) ! (nx)    ! junction points, strictly monotonic
            real(dp), pointer :: f1(:), f(:,:) ! (4,nx)  ! data & interpolation coefficients
            integer, parameter :: nwork = pm_work_size
            real(dp), pointer :: work(:) ! =(nx,nwork)
            
            ierr = 0
            
            nx = weak_num_T9
            allocate(x(nx), f1(4*nx), work(nx*nwork), stat=ierr)
            if (ierr /= 0) return
            
            f(1:4,1:nx) => f1(1:4*nx)
            
            x = dble(weak_reaction_T9s)
            
            do j=1,weak_num_lYeRho
               do m=1,nx
                  f(1,m) = dble(weak_reactions_data(1,m,j,ii,i))
               end do
               call interp_pm(x, nx, f1, nwork, work, 'create_pm_T9_interpolant', ierr)
               if (ierr /= 0) return
               do n=1,nx
                  do m=1,4
                     weak_reactions_data(m,n,j,ii,i) = real(f(m,n))
                  end do
               end do
            end do

            deallocate(x, f1, work)

         end subroutine create_pm_T9_interpolant
         
         
         subroutine create_bicubic_interpolant(i,ii,ierr)
            use interp_2d_lib_sg
            integer, intent(in) :: i, ii
            integer, intent(out) :: ierr
            integer :: ibcxmin                   ! bc flag for x=xmin
            real :: bcxmin(weak_num_lYeRho)    ! bc data vs. y at x=xmin
            integer :: ibcxmax                   ! bc flag for x=xmax
            real :: bcxmax(weak_num_lYeRho)     ! bc data vs. y at x=xmax
            integer :: ibcymin                   ! bc flag for y=ymin
            real :: bcymin(weak_num_T9)   ! bc data vs. x at y=ymin
            integer :: ibcymax                   ! bc flag for y=ymax
            real :: bcymax(weak_num_T9)   ! bc data vs. x at y=ymax         
            integer :: il1, il2, j, k, m
            
            real, target :: f1_ary(4*weak_num_T9*weak_num_lYeRho)
            real, pointer :: f1(:), f3(:,:,:)
            
            ! just use "not a knot" bc's at edges of tables
            ibcxmin = 0; bcxmin(:) = 0
            ibcxmax = 0; bcxmax(:) = 0
            ibcymin = 0; bcymin(:) = 0
            ibcymax = 0; bcymax(:) = 0
            
            f1 => f1_ary
            f3(1:4,1:weak_num_T9,1:weak_num_lYeRho) => f1(1:4*weak_num_T9*weak_num_lYeRho)
            do k = 1,weak_num_T9
               do j = 1,4
                  do m = 1,weak_num_lYeRho
                     f3(j,k,m) = weak_reactions_data(j,k,m,ii,i)
                  end do
               end do
            end do
            call interp_mkbicub_sg( &
               weak_reaction_T9s, weak_num_T9, &
               weak_reaction_lYeRhos, weak_num_lYeRho, &
               f1, weak_num_T9, &
               ibcxmin, bcxmin, ibcxmax, bcxmax, &
               ibcymin, bcymin, ibcymax, bcymax, &
               il1, il2, ierr)
            do k = 1,weak_num_T9
               do j = 1,4
                  do m = 1,weak_num_lYeRho
                     weak_reactions_data(j,k,m,ii,i) = f3(j,k,m)
                  end do
               end do
            end do
               
         end subroutine create_bicubic_interpolant
         
         
         subroutine read_table(i,ii)
            use crlibm_lib, only: str_to_vector
            integer, intent(in) :: i, ii
            integer :: k, j, skip, nvec
            !real :: buffer(weak_num_T9)
            character (len=256) :: buf
            real(dp), target :: vec_ary(50)
            real(dp), pointer :: vec(:)
            logical, parameter :: dbg = .false.
            vec => vec_ary
            skip = -1
            do j = 1, weak_num_lYeRho
               !read(iounit,fmt=*,iostat=ierr) skip, buffer
               read(iounit,fmt='(a)',iostat=ierr) buf
               if (ierr == 0) then
                  call str_to_vector(buf, vec, nvec, ierr)
                  skip = int(vec(1))
                  if (nvec < weak_num_T9+1) ierr = -1
               end if
               if (ierr /= 0 .or. j /= skip) then
                  if (dbg) then
                     write(*,*) 'error in reading table', j, skip
                     write(*,*) 'these are the NEXT lines after the error'
                     do k=1,20
                        read(iounit,fmt='(a)') string
                        write(*,'(a)') trim(string)
                     end do
                     write(*,*)
                     stop 'read_table'
                  end if
                  return
               end if
               do k=1,weak_num_T9
                  weak_reactions_data(1,k,j,ii,i) = vec(k+1)
               end do
               !if (dbg) write(*,'(a,2i6,99f9.3)') 'read_table', j, skip, buffer
            end do
         end subroutine read_table
         
         
         logical function failed(str)
            character (len=*) :: str
            failed = (ierr /= 0)
            if (failed) then
               call free_iounit(iounit)
               write(*,*) 'failed: ' // trim(str)
            end if
         end function failed
         
         
         logical function skip_line()
            logical, parameter :: dbg = .false.
            if (dbg) then
               read(iounit,fmt='(a)') string
               write(*,'(a)') 'skip line ' // trim(string)
            else
               read(iounit,'(a)',iostat=ierr)
            end if
            skip_line = .not. (failed('skip line'))
         end function skip_line


      end subroutine private_load_weak_tables
      
#ifdef offload
      subroutine copy_weak_info_to_coprocessor(ierr) ! on host
         use utils_lib, only: integer_dict_size, get_dict_entries
         use utils_def, only: maxlen_key_string
         integer, intent(out) :: ierr
      
         integer :: num_info_list_reactions
         real(dp), pointer :: info_list_halflife(:), info_list_Qneu(:)
         character (len=maxlen_key_string), pointer :: info_list_dict_keys(:)
         integer, pointer :: info_list_dict_values(:)
         real(dp) :: full_off, full_on
         integer :: blend_hi_Z
         real(dp) :: full_off_hi_Z, full_on_hi_Z
         integer :: num_reactions
         real, pointer :: reactions_data1(:)
         integer, pointer :: lhs_nuclide_id(:), rhs_nuclide_id(:), reaclib_id(:)
         character(len=iso_name_length), pointer :: &
            lhs_nuclide_name(:), rhs_nuclide_name(:)
         character (len=maxlen_key_string), pointer :: reactions_dict_keys(:)
         integer, pointer :: reactions_dict_values(:)
         logical :: bicubic 
         
         integer :: sz

         ierr = 0

         num_info_list_reactions = num_weak_info_list_reactions
         info_list_halflife => weak_info_list_halflife
         info_list_Qneu => weak_info_list_Qneu
         
         ! copy weak_info_list_dict
         sz = integer_dict_size(weak_info_list_dict)
         allocate(info_list_dict_keys(sz), info_list_dict_values(sz))
         call get_dict_entries( &
            weak_info_list_dict, info_list_dict_keys, info_list_dict_values)
         
         full_off = T9_weaklib_full_off
         full_on = T9_weaklib_full_on
         blend_hi_Z = weaklib_blend_hi_Z
         full_off_hi_Z = T9_weaklib_full_off_hi_Z
         full_on_hi_Z = T9_weaklib_full_on_hi_Z
         num_reactions = num_weak_reactions
         reactions_data1 => weak_reactions_data1
         lhs_nuclide_id => weak_lhs_nuclide_id
         rhs_nuclide_id => weak_rhs_nuclide_id
         reaclib_id => weak_reaclib_id
         lhs_nuclide_name => weak_lhs_nuclide_name
         rhs_nuclide_name => weak_rhs_nuclide_name
         bicubic = weak_bicubic
         
         ! copy weak_reactions_dict
         sz = integer_dict_size(weak_reactions_dict)
         allocate(reactions_dict_keys(sz), reactions_dict_values(sz))
         call get_dict_entries( &
            weak_reactions_dict, reactions_dict_keys, reactions_dict_values)

         !dir$ offload target(mic) out(ierr) in( &
            num_info_list_reactions, info_list_halflife, info_list_Qneu, &
            info_list_dict_keys, info_list_dict_values, &
            full_off, full_on, blend_hi_Z, full_off_hi_Z, full_on_hi_Z, &
            num_reactions, reactions_data1, &
            lhs_nuclide_id, rhs_nuclide_id, reaclib_id, &
            lhs_nuclide_name, rhs_nuclide_name, &
            reactions_dict_keys, reactions_dict_values, bicubic)
         call do_copy_weak_info_to_coprocessor( &
            num_info_list_reactions, info_list_halflife, info_list_Qneu, &
            info_list_dict_keys, info_list_dict_values, &
            full_off, full_on, blend_hi_Z, full_off_hi_Z, full_on_hi_Z, &
            num_reactions, reactions_data1, &
            lhs_nuclide_id, rhs_nuclide_id, reaclib_id, &
            lhs_nuclide_name, rhs_nuclide_name, &
            reactions_dict_keys, reactions_dict_values, bicubic, &
            ierr)
         
         deallocate(info_list_dict_keys, info_list_dict_values)
         deallocate(reactions_dict_keys, reactions_dict_values)
         
      end subroutine copy_weak_info_to_coprocessor     
#endif         

      end module load_weak

