c ***********************************************************************
!
!   Copyright (C) 2008  Bill Paxton
!
!   This file is part of MEBDFI_95.
!
!   MEBDFI_95 is free software; you can redistribute it and/or modify
!   it under the terms of the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License, or
!   (at your option) any later version.
!
!   MEBDFI_95 is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!   GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
c ***********************************************************************

! test derived from the "Medical Akzo Nobel problem"
! it is a system of 2 PDE's forming a stiff ODE to exercise the solver 

! for more information about this test, and lots of other good stuff too,
! visit the website with the Bari Test Set for IVP solvers.
! http://pitagora.dm.uniba.it/~testset/problems/medakzo.htm

		module test_mebdfi_medakzo

		use mebdfi_solver_def
		use num_lib, only: MEBDFI_95_ibvp, null_monitor, basic_monitor
		use mtx_lib, only: bcyclic_dble_decsolblk
      use crlibm_lib
		
		integer, parameter :: num_points = 200		
		integer, parameter :: num_components = 2	
		integer, parameter :: i_u = 1, i_v = 2 ! names for the components
		
		! names for our integer parameters
		integer, parameter :: i_ml = 1
		integer, parameter :: i_mu = 2
		integer, parameter :: i_first_pt = 3
		integer, parameter :: i_num_pts = 5
		
		integer, parameter :: num_test_ipars = i_first_pt + i_num_pts	- 1
		integer, parameter :: num_test_rpars = 0
		
		
		contains
		
		
      subroutine do_mebdfi_medakzo_test(verbose)
			
			implicit none
			
			logical, intent(in) :: verbose
	
			integer, parameter :: m = num_components, n = num_points

     		double precision :: t0, tend, h0, hmax, scd, inf_nrms(4), y_min, y_max
			integer :: max_order, max_steps, max_n,
     >			stencil_pts_lower, stencil_pts_higher, ierr, i, lout, verbosity, stage2_min
			logical :: numerical_jacobian, jdebug, equilibrate
			
			integer, target :: ipar_target(num_test_ipars), idata_target(num_idata)
			integer, pointer :: ipar(:), idata(:)
			double precision, target :: Z1_target(m*n), Zprime1_target(m*n), 
     >			rpar_target(num_test_rpars), rdata_target(num_rdata),
     >			abs_tol_target(m), rel_tol_target(m)
			double precision, pointer :: Z1(:), Zprime1(:), Z(:,:), Zprime(:,:), 
     >			rpar(:), rdata(:), abs_tol(:), rel_tol(:)

         integer :: caller_id, nvar, nz
         double precision, dimension(:), pointer :: 
     >      lblk, dblk, ublk ! =(nvar,nvar,nz)
			
			include 'formats'

         caller_id = 0
         nvar = num_components
         nz = num_points
         allocate(lblk(nvar*nvar*nz), dblk(nvar*nvar*nz), ublk(nvar*nvar*nz))
			
			ipar => ipar_target
			rpar => rpar_target
			Z1 => Z1_target
			Zprime1 => Zprime1_target
			Z(1:m,1:n) => Z1(1:m*n)
			Zprime(1:m,1:n) => Zprime1(1:m*n)
			idata => idata_target
			rdata => rdata_target
			abs_tol => abs_tol_target
			rel_tol => rel_tol_target
			
			scd = 0; scd = scd ! for compiler
			
			write(*,*) 'test mebdfi medakzo'
			
			t0 = 0
			tend = 20
			abs_tol = 1d-7
			rel_tol = abs_tol
			h0 = rel_tol(1)*1d-5
			hmax = 0
			
			y_min = -1d199
			y_max = 1d199
			
			inf_nrms = 1
			
			max_order = 7
			max_steps = 10000
			stencil_pts_lower = 1
			stencil_pts_higher = 1
			stage2_min = 2
			numerical_jacobian = .false.
			jdebug = .false.
			equilibrate = .false.
		
			ipar(i_ml) = 3
			ipar(i_mu) = 3
			
			ierr = 0			
			ipar(i_first_pt : i_first_pt+i_num_pts-1) = (/ 1, 40, 67, 86, 100 /)

			call initial_conditions(m,n,Z,Zprime)
			
			lout = 6
			verbosity = 0
			
			nvar = 0
			
			call MEBDFI_95_ibvp(
     >		m, n, Z1, Zprime1, 
     >		t0, tend, h0, hmax, abs_tol, rel_tol, 
     >      y_min, y_max, inf_nrms,
     >      max_order, max_steps,
     >		num_test_rpars, rpar, num_test_ipars, ipar,
     >		num_rdata, rdata, num_idata, idata,
     >		stencil_pts_lower, stencil_pts_higher,
     >		eval_resid, call_back, eval_pderv, null_monitor,
     >      caller_id, nvar, nz, lblk, dblk, ublk, bcyclic_dble_decsolblk, 
     >      numerical_jacobian, equilibrate, stage2_min, jdebug, 
     >		lout, verbosity, ierr)	
     
     		if (ierr /= 0) stop 1
			
			if (verbose) call report(m, n, Z, num_idata, idata, abs_tol, rel_tol, scd)
			! don't include report in test_output since compiler differences
			! can yield integer differences in number of steps taken.
			
			write(*,*)
			if (verbose) then
			   max_n = n
			else
			   max_n = 120
			end if
			do i = 1, max_n, (n-1)/11
				write(*,2) 'Z', i, Z(1,i)
			end do
			write(*,*)

         deallocate(lblk, dblk, ublk)

      end subroutine do_mebdfi_medakzo_test


      subroutine initial_conditions(m, n, Z, Zprime)
	      integer, intent(in) :: m, n
	      double precision, intent(out) :: Z(m,n), Zprime(m,n)
			double precision, parameter :: v0 = 1d0
			Z(i_u,:) = 0
			Z(i_v,:) = v0
			Zprime = 0
      end subroutine initial_conditions


		subroutine eval_pderv(
     >			time,h,con,y,jac1,neqns,lblk1,dblk1,ublk1,cid,nvar,nz,
     >         yprime,ldj,num_rpar,rpar,num_ipar,ipar,ierr)			
         use test_medakzo, only: medakzo_jeval_for_blk_dble
	      integer, intent(in) :: neqns, ldj, cid, nvar, nz, num_ipar, num_rpar
     		double precision, intent(in) :: time,h,con
	     	real(dp), pointer :: y(:), yprime(:) ! (neqns)
	     	real(dp), pointer :: jac1(:) ! (ldj*neqns)
         real(dp), dimension(:), pointer, intent(inout) :: lblk1,dblk1,ublk1 ! =(nvar,nvar,nz)
			integer, pointer :: ipar(:) ! (num_ipar)
			real(dp), pointer :: rpar(:) ! (num_rpar)
			integer, intent(out) :: ierr
			real(dp), pointer :: jac(:,:)			
			integer, parameter :: m = num_components, n = num_points			
         real(dp),dimension(:,:,:),pointer :: lblk,dblk,ublk ! =(nvar,nvar,nz)

         integer, parameter :: ld_dfdy = 5 ! for medakzo
         real(dp), target :: dfdy1(ld_dfdy*neqns)
         real(dp), pointer :: dfdy(:,:)

         include 'formats'
			ierr = 0
			if (neqns /= m*n) stop 'bad args for eval_pderv'
			if (nvar == 0) then 
			   jac(1:ldj,1:neqns) => jac1(1:ldj*neqns)
			   call pdervode(time,y,jac,neqns,yprime,ldj,con,ipar,rpar,ierr,ipar(i_ml),ipar(i_mu))
			   return
			end if
			! bcyclic
         dfdy(1:ld_dfdy,1:neqns) => dfdy1(1:ld_dfdy*neqns)
         call medakzo_jeval_for_blk_dble(ld_dfdy,neqns,time,y,yprime,dfdy,ierr,rpar,ipar)
		   if (ierr /= 0) then
		      return
		   end if
         lblk(1:nvar,1:nvar,1:nz) => lblk1(1:nvar*nvar*nz)
         dblk(1:nvar,1:nvar,1:nz) => dblk1(1:nvar*nvar*nz)
         ublk(1:nvar,1:nvar,1:nz) => ublk1(1:nvar*nvar*nz)
         ! convert from banded (jac) to block tridiagonal (lblk,dblk,ublk)
         call convert_to_block
         ! convert ODE's to PDE's by subtracting 1/con along diagonal
         do k=1,nz
            do j=1,nvar
               dblk(j,j,k) = dblk(j,j,k) - 1d0/con
            end do
         end do
			
			contains
			
			subroutine convert_to_block
			   integer :: i, k
            ! convert from banded to block tridiagonal
            ! lblk(:,:,1) is not used; ublk(:,:,nz) is not used.
            k = 1         
            dblk(1,1,k) = dfdy(3,1) ! partial of f(1,k) wrt var(1,k)    dfdy(3,i)
            dblk(1,2,k) = dfdy(2,2) ! partial of f(1,k) wrt var(2,k)    dfdy(2,i+1)
            dblk(2,1,k) = dfdy(4,1) ! partial of f(2,k) wrt var(1,k)    dfdy(4,i)
            dblk(2,2,k) = dfdy(3,2) ! partial of f(2,k) wrt var(2,k)    dfdy(3,i+1)
            ublk(1,1,k) = dfdy(1,3) ! partial of f(1,k) wrt var(1,k+1)  dfdy(1,i+2)         
            !dfdy(1,i+2) partial of f(1,k) wrt var(1,k+1)
            !dfdy(2,i+1) partial of f(1,k) wrt var(2,k)
            !dfdy(3,i)   partial of f(1,k) wrt var(1,k)
            !dfdy(3,i+1) partial of f(2,k) wrt var(2,k)
            !dfdy(4,i)   partial of f(2,k) wrt var(1,k)
            !dfdy(5,i-2) partial of f(1,k) wrt var(1,k-1)         
            do k=2,nz-1
               i = 2*k-1            
               ! set lblk
               lblk(1,1,k) = dfdy(5,i-2) ! partial of f(1,k) wrt var(1,k-1)
               lblk(1,2,k) = 0 ! partial of f(1,k) wrt var(2,k-1)
               lblk(2,1,k) = 0 ! partial of f(2,k) wrt var(1,k-1)
               lblk(2,2,k) = 0 ! partial of f(2,k) wrt var(2,k-1)
               ! set dblk
               dblk(1,1,k) = dfdy(3,i)   ! partial of f(1,k) wrt var(1,k)  dfdy(3,i)
               dblk(1,2,k) = dfdy(2,i+1) ! partial of f(1,k) wrt var(2,k)  dfdy(2,i+1)
               dblk(2,1,k) = dfdy(4,i)   ! partial of f(2,k) wrt var(1,k)  dfdy(4,i)
               dblk(2,2,k) = dfdy(3,i+1) ! partial of f(2,k) wrt var(2,k)  dfdy(3,i+1)
               ! set ublk
               ublk(1,1,k) = dfdy(1,i+2) ! partial of f(1,k) wrt var(1,k+1)   dfdy(1,i+2)
               ublk(2,1,k) = 0 ! partial of f(2,k) wrt var(1,k+1)
               ublk(1,2,k) = 0 ! partial of f(1,k) wrt var(2,k+1)
               ublk(2,2,k) = 0 ! partial of f(2,k) wrt var(2,k+1)
            end do            
            k = nz
            i = 2*k-1
            dblk(1,1,k) = dfdy(3,i)   ! partial of f(1,k) wrt var(1,k)
            dblk(1,2,k) = dfdy(2,i+1) ! partial of f(1,k) wrt var(2,k)
            dblk(2,1,k) = dfdy(4,i)   ! partial of f(2,k) wrt var(1,k)
            dblk(2,2,k) = dfdy(3,i+1) ! partial of f(2,k) wrt var(2,k)
			end subroutine convert_to_block
			
		end subroutine eval_pderv

c-----------------------------------------------------------------------
  
 
      subroutine pdervode(t,y,pd,n,dy,meband,con,ipar,rpar,ierr,mljac,mujac) 
      integer meband,n,ierr 
		integer, pointer :: ipar(:) ! (num_ipar)
		real(dp), pointer :: rpar(:) ! (num_rpar)
      double precision t,y(n),pd(meband,n)
      double precision dy(n),con 
      integer mljac,mujac,j

      ierr = 0 
      call jeval(meband,n,t,y,dy,pd(1,1),ierr,rpar,ipar) 
      
     
      if (ierr.ne.0) then 
         print *, 'MEBDFID: ERROR: ', 
     +            'MEBDFI can not handle JEVAL IERR' 
         stop 
      endif 
c compute pd = df/dy - 1d0/con*I 
      if (mljac.lt.n) then 
         do 10 j=1,n 
            pd( mujac+1,j) = 
     +       pd( mujac+1,j) - 1d0/con 
   10    continue 
      else 
         do 20 j=1,n 
            pd(j,j) = 
     +       pd(j,j) - 1d0/con 
   20    continue 
      endif 

      return 
      end subroutine
 
 
      subroutine jeval(ldim,neqn,t,y,yprime,dfdy,ierr,rpar,ipar)
      integer ldim,neqn,ierr
      double precision t,y(neqn),yprime(neqn),dfdy(ldim,neqn)
		   integer, pointer :: ipar(:) ! (num_ipar)
		   real(dp), pointer :: rpar(:) ! (num_rpar)

      integer N,i,j
      double precision zeta,dzeta,dzeta2,alpha,beta,k,c,dum,bz
      parameter(k=100d0,c=4d0)
		integer ml,mu,mbnd1,mbnd2,mbnd3,mbnd4
		parameter(ml=3,mu=3,mbnd1=ml,mbnd2=mu,mbnd3=ml+mu+1,mbnd4=mbnd3+ml)
c
      do 20 j=1,neqn
         do 10 i=1,mbnd3
            dfdy(i,j) = 0d0
   10    continue
   20 continue

      N      = neqn/2
      dzeta  = 1d0/dble(N)
      dzeta2 = dzeta*dzeta
      dum    = (dzeta-1d0)*(dzeta-1d0)/c
      alpha  = 2d0*(dzeta-1d0)*dum/c
      beta   = dum*dum

      dfdy(mu+1,1) = -beta*2d0/dzeta2-k*y(2)
      dfdy(mu-1,3) = beta/dzeta2+alpha/(2d0*dzeta)
      dfdy(mu,2) = -k*y(1)
      dfdy(mu+2,1) = -k*y(2)
      dfdy(mu+1,2) = -k*y(1)
c
      do 30 j=2,N-1
         i          = 2*j-1
         zeta       = j*dzeta
         dum        = (zeta-1d0)*(zeta-1d0)/c
         alpha      = 2d0*(zeta-1d0)*dum/c
         beta       = dum*dum
         bz         = beta/dzeta2
         dfdy(mu+3,i-2) = bz-alpha/(2d0*dzeta)
         dfdy(mu+1,i)   = -2d0*bz-k*y(i+1)
         dfdy(mu-1,i+2) = bz+alpha/(2d0*dzeta)
         dfdy(mu,i+1) = -k*y(i)
         i          = 2*j
         dfdy(mu+2,i-1) = -k*y(i)
         dfdy(mu+1,i)   = -k*y(i-1)
   30 continue
c
      dfdy(mu+1,2*N-1) = -k*y(2*N)
      dfdy(mu,2*N)   = -k*y(2*N-1)
      dfdy(mu+2,2*N-1) = -k*y(2*N)
      dfdy(mu+1,2*N)   = -k*y(2*N-1)
		
		return
		
		open(unit=50,file='jeval.txt')
		write(50,'(e24.14)') dfdy(1:5,1:neqn)
		close(50)
		write(*,*) 'ldim', ldim
		write(*,*) 'neqn', neqn
		write(*,*) 't', t
		write(*,*) 
		stop

      return
      end subroutine


		subroutine eval_resid(
     >		   neqns,caller_id,nvar,nz,time,h,con,
     >		   y,delta,yprime,num_rpar,rpar,num_ipar,ipar,stage,iteration,ierr)
         use test_medakzo, only: medakzo_fcn_blk_dble
     		integer, intent(in) :: neqns, caller_id, nvar, nz, 
     >     		num_ipar, num_rpar, stage, iteration
     		real(dp), intent(in) :: time, h, con
     		real(dp), pointer :: y(:), yprime(:) ! (neqns)
     		real(dp), pointer :: delta(:) ! (neqns)
		   integer, pointer :: ipar(:) ! (num_ipar)
		   real(dp), pointer :: rpar(:) ! (num_rpar)
			integer, intent(out) :: ierr
			
			integer, parameter :: m = num_components, n = num_points
			
			double precision :: jnk; jnk = h; jnk = jnk
			include 'formats'
			
			ierr = iteration; ierr = stage ! for compiler

			if (neqns /= m*n) then
			   write(*,2) 'neqns', neqns
			   write(*,2) 'nvar', nvar
			   write(*,2) 'nz', nz
			   write(*,2) 'caller_id', caller_id
			   write(*,2) 'num_components', num_components
			   write(*,2) 'num_points', num_points
			   write(*,2) 'm*n', m*n
			   stop 'bad args for eval_resid'
			end if
			call do_eval_resid(y,yprime,delta)
			ierr = 0
			
			contains
			
			subroutine do_eval_resid(Z,Zprime,resid)
				double precision, intent(in) :: Z(m,n), Zprime(m,n)
				double precision, intent(out) :: resid(m,n)
				
				double precision, parameter :: k = 1d2, c = 4
	      	double precision :: phi, zeta, dzeta, dzeta2, alpha, beta, dum, dZdt(m,n)
				integer :: i
				
				dZdt(i_v,1:n) = -k*Z(i_u,1:n)*Z(i_v,1:n)				

				if (time <= 5) then; phi = 2; else; phi = 0; end if
		      dzeta = 1/dble(n)
		      dzeta2 = dzeta**2
		      dum = pow2(dzeta - 1)/c
		      alpha = 2*(dzeta - 1)*dum/c
		      beta = pow2(dum)

				dZdt(i_u,1) = dZdt(i_v,1) +
     >			(phi - 2*Z(i_u,1) + Z(i_u,2))*beta/dzeta2 
     >			+ alpha*(Z(i_u,2) - phi)/(2*dzeta)

		      do i = 2, n-1
		         zeta  = i*dzeta
		         dum   = pow2(zeta-1)/c
		         alpha = 2*(zeta-1)*dum/c
		         beta  = pow2(dum)
		         dZdt(i_u,i) = dZdt(i_v,i) +
     >					(Z(i_u,i-1) - 2*Z(i_u,i) + Z(i_u,i+1))*beta/dzeta2 +
     >						alpha*(Z(i_u,i+1) - Z(i_u,i-1))/(2*dzeta)
		   	end do

				dZdt(i_u,n) = dZdt(i_v,n)

				resid = dZdt - Zprime

			end subroutine do_eval_resid
		
		end subroutine eval_resid
		

		subroutine call_back(
     >			neqns,time,h,con,y,yprime,tout,tend,hmax,lout,verbosity,
     >			num_rdata,num_idata,rdata,idata,maxder,maxstp,equil,jdebug,
     >			itol,num_tol,rtol,atol,y_min,y_max,inf_nrms,
     >			num_rpar,rpar,num_ipar,ipar,idid,kgo,ierr)
	      integer, intent(in) :: neqns,num_rdata,num_idata,num_tol,num_ipar,num_rpar,kgo
	      real(dp), intent(in) :: time,h,con,y_min,y_max			
     		real(dp), pointer :: y(:), yprime(:) ! (neqns)
			integer, intent(inout) :: idid,ierr,itol,lout,verbosity
			integer, intent(inout) :: maxder,maxstp,equil,jdebug		
		   integer, pointer :: idata(:) ! (num_idata)
		   real(dp), pointer :: rdata(:) ! (num_rdata)
		   integer, pointer :: ipar(:) ! (num_ipar)
		   real(dp), pointer :: rpar(:) ! (num_rpar)
		   real(dp), pointer :: atol(:), rtol(:) ! (num_tol)
			real(dp), intent(inout) :: inf_nrms(4)
			real(dp), intent(inout) :: tout,tend,hmax
			
			integer :: i
			double precision :: jnk
			jnk = y(1); jnk = yprime(1); jnk = jnk ! to keep the compiler quiet

			ierr = 0
			
		end subroutine call_back
		
		
		subroutine report(m,n,Z,num_id,idata,abs_tol,rel_tol,scd)
			integer, intent(in) :: m, n, num_id, idata(num_id)
			double precision, intent(in) :: Z(m,n), abs_tol(m), rel_tol(m)
			double precision, intent(out) :: scd
			
			double precision :: hi_res_solution(m,n), mescd
			integer :: nsteps
			
			call medakzo_solut(m*n, hi_res_solution)
			call getscd(mescd, scd, m*n, hi_res_solution, Z)

			nsteps = idata(i_nstep)+idata(i_nfail)

 1			format(a)
			write(*,1) '	MEDAKZO -- Medical Akzo Nobel Problem'
			write(*,1)
			write(*,1) '	"atol" is the requested absolute error tolerance.'
			write(*,1) '	"rtol" is the requested relative error tolerance.'
			write(*,1) '	"tries" is the total number of attempts to take a step.'
			write(*,1) '	"steps" is the sum of steps taken to reach the ending time for the problem.'
			write(*,1) '	"fail" is the number of tries that had to be redone with a smaller timestep.'
			write(*,1) '	"jacob" is the total number of jacobians created.'
			write(*,1) '	"solve" is the total number matrix solves performed.'
			write(*,1) '	"avg k" is the average order used in the solution.'
			write(*,*)
      	write(*,'(2a12,3x,7a8)')
     >			'atol', 'rtol', 'tries', 'steps', 'fail', 'jacob',
     >			'solve', 'avg k'
			
			write(*,'(3x,2e12.3,5i8,2f8.3)') 
     +      abs_tol(1), rel_tol(1), nsteps, idata(i_nstep), idata(i_nfail), idata(i_nje), 
     >		idata(i_nbsol), dble(idata(i_sum_nq))/nsteps

			write(*,*) 
			write(*,'(f14.6,a)') scd, ' significant correct digits in result'
			write(*,*) 

		end subroutine report
		

      subroutine getscd(mescd,scd,neqn,yref,y)
	      integer, intent(in) :: neqn
	      double precision, intent(in) :: yref(neqn), y(neqn)
	      double precision, intent(out) :: mescd, scd

	      integer :: i
	      double precision :: aerr, aerrmx, aerrl2, rerr, rerrmx, rerrl2
	      double precision ::  mixed_rerr, mixed_rerrmx
	      integer :: numa, numr

	      numa   = 0
	      numr   = 0
	      aerrl2 = 0d0
	      rerrl2 = 0d0 
	      aerrmx = 0d0
	      rerrmx = 0d0
			mixed_rerrmx = 0d0
      
	      do i=1,neqn

	         aerr = abs(yref(i)-y(i))
	         aerrmx = max(aerrmx,aerr)
	         aerrl2 = aerrl2 + aerr*aerr
	         numa   = numa + 1

	         if (yref(i).ne.0d0) then       
	            rerr = abs((yref(i)-y(i))/(yref(i)))
	            rerrmx = max(rerrmx,rerr)
	            rerrl2 = rerrl2 + rerr*rerr
	            numr   = numr + 1
	         endif

	         mixed_rerr = abs((yref(i)-y(i)) / (1+abs(yref(i))))
	         mixed_rerrmx = max(mixed_rerrmx,mixed_rerr)

	      end do

	      mescd = -log10_cr(mixed_rerrmx)
	      scd = -log10_cr(aerrmx)

      end subroutine getscd


      subroutine medakzo_solut(neqn,y)
      	integer, intent(in) :: neqn
      	double precision, intent(out) :: y(neqn)
			
			if (neqn /= 400) stop 'only have hi res solution for neqn=400'
			
	      y(  1) =  0.5113983840919909d-005
	      y(  2) =  0.1925112884312553d-143
	      y(  3) =  0.1027858770570419d-004
	      y(  4) =  0.1890518289312031d-142
	      y(  5) =  0.1549349862635799d-004
	      y(  6) =  0.1774199325357386d-142
	      y(  7) =  0.2075835344757462d-004
	      y(  8) =  0.5897341137981092d-143
	      y(  9) =  0.2607273610116854d-004
	      y( 10) =  0.1093527900908030d-143
	      y( 11) =  0.3143617475695002d-004
	      y( 12) =  0.1188834841626416d-144
	      y( 13) =  0.3684813884509626d-004
	      y( 14) =  0.9968323236025642d-147
	      y( 15) =  0.4230803594492533d-004
	      y( 16) = -0.2801994001528093d-146
	      y( 17) =  0.4781520853483223d-004
	      y( 18) = -0.7337417669341249d-147
	      y( 19) =  0.5336893059800053d-004
	      y( 20) = -0.1209033101530330d-147
	      y( 21) =  0.5896840407836044d-004
	      y( 22) = -0.1430357497530360d-148
	      y( 23) =  0.6461275518112516d-004
	      y( 24) = -0.1063952641824646d-149
	      y( 25) =  0.7030103051210320d-004
	      y( 26) =  0.7939969136126717d-152
	      y( 27) =  0.7603219304985662d-004
	      y( 28) =  0.1568246940545520d-150
	      y( 29) =  0.8180511794465543d-004
	      y( 30) =  0.4074950357924872d-150
	      y( 31) =  0.8761858813806752d-004
	      y( 32) =  0.5592746648679992d-150
	      y( 33) =  0.9347128979692480d-004
	      y( 34) = -0.5510388943414421d-151
	      y( 35) =  0.9936180755532036d-004
	      y( 36) = -0.2724738349250769d-149
	      y( 37) =  0.1052886195582220d-003
	      y( 38) = -0.9327772452398718d-149
	      y( 39) =  0.1112500923002360d-003
	      y( 40) = -0.2182885200987554d-148
	      y( 41) =  0.1172444752530255d-003
	      y( 42) = -0.4041450806475518d-148
	      y( 43) =  0.1232698952748828d-003
	      y( 44) = -0.5608157478395261d-148
	      y( 45) =  0.1293243507959787d-003
	      y( 46) = -0.2639662630908699d-148
	      y( 47) =  0.1354057057728661d-003
	      y( 48) =  0.1801866277537073d-147
	      y( 49) =  0.1415116834059119d-003
	      y( 50) =  0.8464449882759417d-147
	      y( 51) =  0.1476398596134615d-003
	      y( 52) =  0.2245234937355967d-146
	      y( 53) =  0.1537876562567258d-003
	      y( 54) =  0.3359213489153582d-146
	      y( 55) =  0.1599523341096154d-003
	      y( 56) = -0.3085721171916412d-146
	      y( 57) =  0.1661309855680449d-003
	      y( 58) = -0.4465322607423735d-145
	      y( 59) =  0.1723205270935920d-003
	      y( 60) = -0.1970925996866384d-144
	      y( 61) =  0.1785176913868402d-003
	      y( 62) = -0.6070953121563027d-144
	      y( 63) =  0.1847190192862588d-003
	      y( 64) = -0.1412011918930335d-143
	      y( 65) =  0.1909208513890961d-003
	      y( 66) = -0.2378861987352203d-143
	      y( 67) =  0.1971193193914910d-003
	      y( 68) = -0.2380432473186974d-143
	      y( 69) =  0.2033103371458565d-003
	      y( 70) = -0.6522557638254663d-145
	      y( 71) =  0.2094895914345677d-003
	      y( 72) =  0.1784305601809064d-143
	      y( 73) =  0.2156525324601176d-003
	      y( 74) = -0.1007474781780816d-142
	      y( 75) =  0.2217943640531935d-003
	      y( 76) = -0.5281511349479423d-142
	      y( 77) =  0.2279100336016016d-003
	      y( 78) = -0.1117525482975987d-141
	      y( 79) =  0.2339942217046434d-003
	      y( 80) = -0.1127916494884468d-141
	      y( 81) =  0.2400413315594459d-003
	      y( 82) = -0.1633306916231411d-142
	      y( 83) =  0.2460454780878912d-003
	      y( 84) =  0.2708874035585891d-143
	      y( 85) =  0.2520004768152150d-003
	      y( 86) = -0.2501941069702609d-142
	      y( 87) =  0.2578998325140575d-003
	      y( 88) = -0.2642308070750020d-141
	      y( 89) =  0.2637367276308081d-003
	      y( 90) = -0.3684887530751217d-139
	      y( 91) =  0.2695040105145025d-003
	      y( 92) = -0.3647274179805887d-138
	      y( 93) =  0.2751941834723564d-003
	      y( 94) = -0.1255641406397419d-137
	      y( 95) =  0.2807993906802854d-003
	      y( 96) = -0.1694257216823904d-138
	      y( 97) =  0.2863114059815211d-003
	      y( 98) = -0.1785516142939602d-136
	      y( 99) =  0.2917216206117258d-003
	      y(100) = -0.3935939757647002d-135
	      y(101) =  0.2970210308948898d-003
	      y(102) = -0.2514765666933440d-134
	      y(103) =  0.3022002259608294d-003
	      y(104) = -0.7200873856605984d-134
	      y(105) =  0.3072493755423352d-003
	      y(106) = -0.7539683247227422d-134
	      y(107) =  0.3121582179180383d-003
	      y(108) =  0.3738577086039426d-135
	      y(109) =  0.3169160480759169d-003
	      y(110) = -0.2493582962172335d-131
	      y(111) =  0.3215117061821543d-003
	      y(112) =  0.3039632438293726d-130
	      y(113) =  0.3259335664508512d-003
	      y(114) =  0.5321044068586611d-128
	      y(115) =  0.3301695265219917d-003
	      y(116) = -0.1918129324351378d-126
	      y(117) =  0.3342069974681551d-003
	      y(118) = -0.1336929159252586d-124
	      y(119) =  0.3380328945648600d-003
	      y(120) =  0.9521748754010357d-123
	      y(121) =  0.3416336289752354d-003
	      y(122) =  0.1001197393324181d-120
	      y(123) =  0.3449951005170561d-003
	      y(124) =  0.2703860993866771d-119
	      y(125) =  0.3481026916991771d-003
	      y(126) =  0.4365133580297076d-119
	      y(127) =  0.3509412632351946d-003
	      y(128) =  0.4898111237855383d-115
	      y(129) =  0.3534951512648823d-003
	      y(130) =  0.1621439381962246d-112
	      y(131) =  0.3557481665387581d-003
	      y(132) =  0.3003220203772183d-110
	      y(133) =  0.3576835958481664d-003
	      y(134) =  0.5931668289615909d-108
	      y(135) =  0.3592842060126915d-003
	      y(136) =  0.2235590472383775d-105
	      y(137) =  0.3605322507686931d-003
	      y(138) =  0.1025457293602057d-102
	      y(139) =  0.3614094809374544d-003
	      y(140) =  0.3496613568296336d-100
	      y(141) =  0.3618971582890092d-003
	      y(142) =  0.4767073568395508d-098
	      y(143) =  0.3619760735583436d-003
	      y(144) = -0.2410784286794997d-095
	      y(145) =  0.3616265691144918d-003
	      y(146) = -0.9188398110576038d-093
	      y(147) =  0.3608285668302233d-003
	      y(148) =  0.1146623087995081d-089
	      y(149) =  0.3595616017506735d-003
	      y(150) =  0.1649638439865233d-086
	      y(151) =  0.3578048622135169d-003
	      y(152) =  0.1215140240350217d-083
	      y(153) =  0.3555372371311931d-003
	      y(154) =  0.7134490346394154d-081
	      y(155) =  0.3527373712073181d-003
	      y(156) =  0.4502515392738464d-078
	      y(157) =  0.3493837289247301d-003
	      y(158) =  0.7138395988310312d-075
	      y(159) =  0.3454546682115489d-003
	      y(160) =  0.9941693919247076d-071
	      y(161) =  0.3409285247640208d-003
	      y(162) =  0.2012859826753015d-066
	      y(163) =  0.3357837080804970d-003
	      y(164) =  0.3598261520662423d-062
	      y(165) =  0.3299988103392750d-003
	      y(166) =  0.5466580008990664d-058
	      y(167) =  0.3235527293336597d-003
	      y(168) =  0.6945384844951550d-054
	      y(169) =  0.3164248067597393d-003
	      y(170) =  0.7275415527806026d-050
	      y(171) =  0.3085949832350532d-003
	      y(172) =  0.6193143746524996d-046
	      y(173) =  0.3000439715082906d-003
	      y(174) =  0.4219255556214135d-042
	      y(175) =  0.2907534493998412d-003
	      y(176) =  0.2263678154715720d-038
	      y(177) =  0.2807062740884081d-003
	      y(178) =  0.9401607967545219d-035
	      y(179) =  0.2698867194275612d-003
	      y(180) =  0.2968231730793053d-031
	      y(181) =  0.2582807380350103d-003
	      y(182) =  0.6987463944434805d-028
	      y(183) =  0.2458762499428408d-003
	      y(184) =  0.1201641789884051d-024
	      y(185) =  0.2326634596245027d-003
	      y(186) =  0.1477169946829840d-021
	      y(187) =  0.2186352032185982d-003
	      y(188) =  0.1268462422099779d-018
	      y(189) =  0.2037873277440060d-003
	      y(190) =  0.7425015664001834d-016
	      y(191) =  0.1881191040379240d-003
	      y(192) =  0.2886826929895103d-013
	      y(193) =  0.1716336750388461d-003
	      y(194) =  0.7252477041900172d-011
	      y(195) =  0.1543385408702044d-003
	      y(196) =  0.1143390654212691d-008
	      y(197) =  0.1362460820444338d-003
	      y(198) =  0.1096625145716966d-006
	      y(199) =  0.1173741304462833d-003
	      y(200) =  0.6190822732534586d-005
	      y(201) =  0.9774701310627047d-004
	      y(202) =  0.1986273404756002d-003
	      y(203) =  0.7740788649977313d-004
	      y(204) =  0.3489773624098464d-002
	      y(205) =  0.5657119003189305d-004
	      y(206) =  0.3234526094359604d-001
	      y(207) =  0.3643334879766658d-004
	      y(208) =  0.1548747348410801d+000
	      y(209) =  0.2003152841880950d-004
	      y(210) =  0.4026980529594953d+000
	      y(211) =  0.9608297851720770d-005
	      y(212) =  0.6649744834198490d+000
	      y(213) =  0.4215537698495267d-005
	      y(214) =  0.8409284546320647d+000
	      y(215) =  0.1753504402754791d-005
	      y(216) =  0.9314946676956936d+000
	      y(217) =  0.7048158429518009d-006
	      y(218) =  0.9720896201631835d+000
	      y(219) =  0.2760943506466737d-006
	      y(220) =  0.9890204872799944d+000
	      y(221) =  0.1057554501281432d-006
	      y(222) =  0.9957930123519514d+000
	      y(223) =  0.3965142250779033d-007
	      y(224) =  0.9984246531478463d+000
	      y(225) =  0.1455273204279008d-007
	      y(226) =  0.9994229325942358d+000
	      y(227) =  0.5226348147846279d-008
	      y(228) =  0.9997932125999319d+000
	      y(229) =  0.1835610545325733d-008
	      y(230) =  0.9999275409325039d+000
	      y(231) =  0.6301078589385454d-009
	      y(232) =  0.9999751869380269d+000
	      y(233) =  0.2112538351365564d-009
	      y(234) =  0.9999917015131560d+000
	      y(235) =  0.6912550453447044d-010
	      y(236) =  0.9999972914302640d+000
	      y(237) =  0.2205932132514696d-010
	      y(238) =  0.9999991378543379d+000
	      y(239) =  0.6860095639285670d-011
	      y(240) =  0.9999997325855174d+000
	      y(241) =  0.2077324462852526d-011
	      y(242) =  0.9999999192384585d+000
	      y(243) =  0.6120038908594393d-012
	      y(244) =  0.9999999762710279d+000
	      y(245) =  0.1752695518797070d-012
	      y(246) =  0.9999999932230490d+000
	      y(247) =  0.4875001992978682d-013
	      y(248) =  0.9999999981203191d+000
	      y(249) =  0.1315706848908981d-013
	      y(250) =  0.9999999994941428d+000
	      y(251) =  0.3442274192104633d-014
	      y(252) =  0.9999999998680372d+000
	      y(253) =  0.8721783456154470d-015
	      y(254) =  0.9999999999666630d+000
	      y(255) =  0.2137938962858872d-015
	      y(256) =  0.9999999999918528d+000
	      y(257) =  0.5064735930780995d-016
	      y(258) =  0.9999999999980759d+000
	      y(259) =  0.1158284928109727d-016
	      y(260) =  0.9999999999995613d+000
	      y(261) =  0.2554350586347124d-017
	      y(262) =  0.9999999999999036d+000
	      y(263) =  0.5425563935887811d-018
	      y(264) =  0.9999999999999796d+000
	      y(265) =  0.1108623976460997d-018
	      y(266) =  0.9999999999999958d+000
	      y(267) =  0.2176490922739810d-019
	      y(268) =  0.9999999999999992d+000
	      y(269) =  0.4100180074816888d-020
	      y(270) =  0.9999999999999998d+000
	      y(271) =  0.7401919443964595d-021
	      y(272) =  0.1000000000000000d+001
	      y(273) =  0.1278745657114596d-021
	      y(274) =  0.1000000000000000d+001
	      y(275) =  0.2111087049605767d-022
	      y(276) =  0.1000000000000000d+001
	      y(277) =  0.3325632734364699d-023
	      y(278) =  0.1000000000000000d+001
	      y(279) =  0.4991515592566292d-024
	      y(280) =  0.1000000000000000d+001
	      y(281) =  0.7126950428617158d-025
	      y(282) =  0.1000000000000000d+001
	      y(283) =  0.9664740804131475d-026
	      y(284) =  0.1000000000000000d+001
	      y(285) =  0.1242716896959521d-026
	      y(286) =  0.1000000000000000d+001
	      y(287) =  0.1512543532243458d-027
	      y(288) =  0.1000000000000000d+001
	      y(289) =  0.1739533019752215d-028
	      y(290) =  0.1000000000000000d+001
	      y(291) =  0.1886942537979667d-029
	      y(292) =  0.1000000000000000d+001
	      y(293) =  0.1926965705022792d-030
	      y(294) =  0.1000000000000000d+001
	      y(295) =  0.1849021812823421d-031
	      y(296) =  0.1000000000000000d+001
	      y(297) =  0.1663798767415642d-032
	      y(298) =  0.1000000000000000d+001
	      y(299) =  0.1401076830818626d-033
	      y(300) =  0.1000000000000000d+001
	      y(301) =  0.1101818149402153d-034
	      y(302) =  0.1000000000000000d+001
	      y(303) =  0.8074224739509168d-036
	      y(304) =  0.1000000000000000d+001
	      y(305) =  0.5501249196662931d-037
	      y(306) =  0.1000000000000000d+001
	      y(307) =  0.3476859813132770d-038
	      y(308) =  0.1000000000000000d+001
	      y(309) =  0.2033489290876775d-039
	      y(310) =  0.1000000000000000d+001
	      y(311) =  0.1097880013869247d-040
	      y(312) =  0.1000000000000000d+001
	      y(313) =  0.5457825200381417d-042
	      y(314) =  0.1000000000000000d+001
	      y(315) =  0.2491675366427318d-043
	      y(316) =  0.1000000000000000d+001
	      y(317) =  0.1041801880291617d-044
	      y(318) =  0.1000000000000000d+001
	      y(319) =  0.3978066491064419d-046
	      y(320) =  0.1000000000000000d+001
	      y(321) =  0.1383174699098532d-047
	      y(322) =  0.1000000000000000d+001
	      y(323) =  0.4365911791079500d-049
	      y(324) =  0.1000000000000000d+001
	      y(325) =  0.1247057764661705d-050
	      y(326) =  0.1000000000000000d+001
	      y(327) =  0.3212728839963712d-052
	      y(328) =  0.1000000000000000d+001
	      y(329) =  0.7439366703571565d-054
	      y(330) =  0.1000000000000000d+001
	      y(331) =  0.1542770387822259d-055
	      y(332) =  0.1000000000000000d+001
	      y(333) =  0.2854454245592573d-057
	      y(334) =  0.1000000000000000d+001
	      y(335) =  0.4693220411250150d-059
	      y(336) =  0.1000000000000000d+001
	      y(337) =  0.6828458274546624d-061
	      y(338) =  0.1000000000000000d+001
	      y(339) =  0.8752952529541412d-063
	      y(340) =  0.1000000000000000d+001
	      y(341) =  0.9838541433761416d-065
	      y(342) =  0.1000000000000000d+001
	      y(343) =  0.9649177728609193d-067
	      y(344) =  0.1000000000000000d+001
	      y(345) =  0.8213596936190817d-069
	      y(346) =  0.1000000000000000d+001
	      y(347) =  0.6033986647865674d-071
	      y(348) =  0.1000000000000000d+001
	      y(349) =  0.3802531117966294d-073
	      y(350) =  0.1000000000000000d+001
	      y(351) =  0.2042261117698575d-075
	      y(352) =  0.1000000000000000d+001
	      y(353) =  0.9282595096128614d-078
	      y(354) =  0.1000000000000000d+001
	      y(355) =  0.3543587864454877d-080
	      y(356) =  0.1000000000000000d+001
	      y(357) =  0.1126779423370979d-082
	      y(358) =  0.1000000000000000d+001
	      y(359) =  0.2957534367766753d-085
	      y(360) =  0.1000000000000000d+001
	      y(361) =  0.6344600529877694d-088
	      y(362) =  0.1000000000000000d+001
	      y(363) =  0.1100279075462365d-090
	      y(364) =  0.1000000000000000d+001
	      y(365) =  0.1523845293461783d-093
	      y(366) =  0.1000000000000000d+001
	      y(367) =  0.1662696161555950d-096
	      y(368) =  0.1000000000000000d+001
	      y(369) =  0.1407578290673998d-099
	      y(370) =  0.1000000000000000d+001
	      y(371) =  0.9086150803567186d-103
	      y(372) =  0.1000000000000000d+001
	      y(373) =  0.4384339596163745d-106
	      y(374) =  0.1000000000000000d+001
	      y(375) =  0.1545482064392824d-109
	      y(376) =  0.1000000000000000d+001
	      y(377) =  0.3874172613928345d-113
	      y(378) =  0.1000000000000000d+001
	      y(379) =  0.6689452219441953d-117
	      y(380) =  0.1000000000000000d+001
	      y(381) =  0.7655680935317283d-121
	      y(382) =  0.1000000000000000d+001
	      y(383) =  0.5538543899545850d-125
	      y(384) =  0.1000000000000000d+001
	      y(385) =  0.2386173886563501d-129
	      y(386) =  0.1000000000000000d+001
	      y(387) =  0.5664887497790931d-134
	      y(388) =  0.1000000000000000d+001
	      y(389) =  0.6671124967149171d-139
	      y(390) =  0.1000000000000000d+001
	      y(391) =  0.3351973480286951d-144
	      y(392) =  0.1000000000000000d+001
	      y(393) =  0.5684315818559200d-150
	      y(394) =  0.1000000000000000d+001
	      y(395) =  0.2142121793294590d-156
	      y(396) =  0.1000000000000000d+001
	      y(397) =  0.6727117900187205d-164
	      y(398) =  0.1000000000000000d+001
	      y(399) =  0.0000000000000000d+000
	      y(400) =  0.1000000000000000d+001
      end subroutine medakzo_solut


		end module test_mebdfi_medakzo

