! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module num_lib
      ! various numerical routines
      
      use const_def, only: dp
      use crlibm_lib
      use num_def
      
      ! NOTE: because of copyright restrictions, 
      !       mesa doesn't use any routines from Numerical Recipes.

      
      implicit none

#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif

      contains ! the procedure interface for the library
      ! client programs should only call these routines.

      
      ! safe root finding
      ! uses alternating bisection and inverse parabolic interpolation
      ! also have option to use derivative as accelerator (newton method)
      include "num_safe_root.dek"

      
      ! solvers for ODEs and DAEs.
      
      
      ! sometimes you just want a simple runge-kutta
         include "num_rk2.dek"
         include "num_rk4.dek"
         
      
      ! but there are lots of fancier options too.
      

      ! selections from the Hairer family of ODE/DAE integrators.
      ! from Ernst Hairer's website: http://www.unige.ch/~hairer/


      ! explicit ODE solver based on method of Cash and Karp (for non-stiff problems)

         ! explicit Runge-Kutta ODE integrator with variable step size
         include "num_cash_karp.dek"


      ! explicit ODE solvers based on methods of Dormand and Prince (for non-stiff problems)

         ! explicit Runge-Kutta ODE integrator of order 5
         ! with dense output of order 4
         include "num_dopri5.dek" !  "DOramand PRInce order 5"

         ! explicit Runge-Kutta ODE integrator of order 8
         ! with dense output of order 7
         include "num_dop853.dek" !  "DOramand Prince order 8(5, 3)"
         
         ! both integrators have automatic step size control and monitoring for stiffness.
      
         ! For a description see:
         ! Hairer, Norsett and Wanner (1993): 
         ! Solving Ordinary Differential Equations. Nonstiff Problems. 2nd edition. 
         ! Springer Series in Comput. Math., vol. 8.
         ! http://www.unige.ch/~hairer/books.html
      
      
      ! implicit solvers (for stiff problems)
      
         ! there are a bunch of implicit solvers to pick from (listed below), 
         ! but they all have pretty much the same arguments, 
         ! so I've provided a general routine, called "isolve", that let's you
         ! pass an extra argument to specify which one of the particular solvers
         ! that you want to use.
         include "num_isolve.dek"
         
         ! if possible, you should write your code to call isolve
         ! rather than calling one of the particular solvers.
         ! only call a specific solver if you need a feature it provides
         ! that isn't supported by isolve.
         
         ! you can find an example program using isolve in num/test/src/sample_ode_solver.f
         
         
      ! the implicit solver routines
      
         ! for detailed descriptions of these routines see: 
         ! Hairer and Wanner (1996): 
         ! Solving Ordinary Differential Equations. 
         ! Stiff and Differential-Algebraic Problems. 2nd edition. 
         ! Springer Series in Comput. Math., vol. 14.
         ! http://www.unige.ch/~hairer/books.html
      
         ! linearly implicit Runge-Kutta method (Rosenbrock)
         include "num_ros2.dek"    ! L-stable; 2 stages; order 2, 2 function evaluations.
            ! ros2 is suitable for use with approximate jacobians such as from numerical differences.
            ! ros2 is designed for use with Strang splitting and is reported to be able to cope
            ! with large time steps and artificial transients introduced at the beginning of split intervals.
            ! see Verwer et al, "A Second-Order Rosenbrock Method Applied to Photochemical Dispersion Problems", 
            ! SIAM J. Sci. Comput. (20), 1999, 1456-1480.
            
         include "num_rose2.dek"    ! L-stable; 3 stages; order 2, 3 function evaluations.   
            ! rose2 is unique among the implicit solvers in that the final function evaluation
            ! uses the solution vector for the step.

         include "num_rodas3.dek"  ! L-stable; 4 stages; order 3, 3 function evaluations.
         include "num_rodas4.dek"  ! L-stable; 6 stages; order 4, 6 function evaluations.     

         ! 3rd order; for parabolic equations.
         include "num_ros3p.dek"   ! A-stable; 3 stages; order 3, 2 function evaluations.
         include "num_ros3pl.dek"  ! L-stable; 4 stages; order 3, 3 function evaluations.
         ! 4th order; for parabolic equations.
         include "num_rodasp.dek"  ! L-stable; 6 stages; order 4, 6 function evaluations. 
         
         include "num_solvers_options.dek"


      ! which implicit solver should you use?
         
         ! somewhat surprisingly, in some cases the solvers
         ! that work well at high tolerances will fail with low
         ! tolerances and vice-versa.  so you need to match
         ! the solver to the problem.
      
         ! your best bet is to try them all on some typical cases.
         ! happily this isn't too hard to do since they all
         ! use the same function arguments and have (almost)
         ! identical calling sequences and options.
      
      
      ! flexible choice of linear algebra routines
      
         ! the solvers need to solve linear systems.
         ! this is typically done by first factoring the matrix A
         ! and then repeatedly using the factored form to solve
         ! A*x=b for various vectors b.
         
         ! rather than build-in a particular matrix solver, 
         ! the mesa versions of the solvers take as arguments
         ! routines to perform these tasks.  the mesa/mtx package
         ! includes several choices for implementations of the
         ! required routines.
         
      
      ! dense, banded, or sparse matrix
      
         ! All the packages allow the matrix to be in dense or banded form.
         ! the choice of sparse matrix package is not fixed by the solvers.
         ! the only constraint is the the sparse format must be either
         ! compressed row sparse or compressed column sparse.
         ! the mesa/mtx package comes with one option for a sparse
         ! package (based on SPARSKIT), and also has hooks for another (Super_LU).
         ! Since the sparse routines are passed as arguments to the solvers, 
         ! it is possible to experiment with different linear algebra
         ! packages without a great deal of effort.
         
         
      ! analytical or numerical jacobian
      
         ! to solve M*y' = f(y), the solvers need to have the jacobian matrix, df/dy.
         ! the jacobian can either be calculated analytically by a user supplied routine, 
         ! or the solver can form a numerical difference estimate by repeatedly
         ! evaluating f(y) with slightly different y's.  Such numerical jacobians
         ! are supported by all the solvers for both dense and banded matrix forms.
         ! For the sparse matrix case, only analytical jacobians are allowed.
         
         ! NOTE: for most implicit solvers, the accuracy of the jacobian influences
         ! the rate of convergence, but doesn't impact the accuracy of the solution.
         ! however, the rodas solvers are an exception to this rule.
         ! they are based on the rosenbrock method which replaces the newton iteration
         ! by formulas that directly use the jacobian in the formula for
         ! the result.  as a result, the rodas solvers depend on having
         ! accurate jacobians in order to produce accurate results.
      
      
      ! explicit or implicit ODE systems
      
         ! systems of the form y' = f(y) are called "explicit ODE systems".
         
         ! systems of the form M*y' = f(y), with M not equal to the identity matrix, 
         ! are called "implicit ODE systems".
         
         ! in addition to the usual explicit systems, 
         ! the solvers can also handle implicit ODE systems
         ! in which M is an arbitrary constant matrix, 
         ! even including the case of M singular.
         
         ! for M non-constant, see the discussion of "problems with special structure"
      
      
      ! problems with special structure
      
         ! 3 special cases can be handled easily
         
            ! case 1, second derivatives: y'' = f(t, y, y')
            ! case 2, nonconstant matrix: C(x, y)*y' = f(t, y)
            ! case 3, both of the above: C(x, y)*y'' = f(t, y, y')
            
         ! these all work by adding auxiliary variables to the problem and
         ! converting back to the standard form with a constant matrix M.
         
            ! case 1: y'' = f(t, y, y')
               ! after add auxiliary variables z, this becomes
               ! y' = z
               ! z' = f(t, y, z)
               
            ! case 2: C(x, y)*y' = f(t, y)
               ! after add auxiliary variables z, this becomes
               ! y' = z
               ! 0 = C(x, y)*z - f(t, y)
               
            ! case 3: C(x, y)*y'' = f(t, y, y')
               ! after add auxiliary variables z and u, this becomes
               ! y' = z
               ! z' = u
               ! 0 = C(x, y)*u - f(t, y, z)
         
         ! The last two cases take advantage of the ability to have M singular.
         
         ! If the matrix for df/dy is dense in these special cases, all the solvers
         ! can reduce the cost of the linear algebra operations by special treatment
         ! of the auxiliary variables.
         
      
      ! "projection" of solution to valid range of values.
      
         ! it is often the case that the n-dimensional solution
         ! is actually constrained to a subspace of full n dimensional
         ! space of numbers.  The proposed solutions at each step
         ! need to be projected back to the allowed subspace in order
         ! to maintain valid results.  The routines all provide for this
         ! option by calling a "solout" routine, supplied by the user, 
         ! after every accepted step.  The user's solout routine can modify
         ! the solution y before returning to continue the integration.
      
         
      ! "dense output"
      
         ! the routines provide estimates of the solution over entire step.
         ! useful for tabulating the solution at prescribed points
         ! or for smooth graphical presentation of the solution.
         ! also very useful for "event location" -- e.g., at what
         ! value of x do we get a solution y(x) s.t. some relation
         ! g(x, y(x))=0.  The dense output option is very helpful here.
         ! All of the solvers support dense output.
         ! BTW: there is typically a certain overhead associated with
         ! providing the option for dense output, so don't request it
         ! unless you'll really be using it.

         ! here is a special version of safe_root for use with dense output "solout" routines
         include "num_solout_root.dek"



      ! "null" implementations of routines used by the solvers
      ! are for use in cases in which you aren't actually using the routine, 
      ! but something must be provided for the required argument.


      subroutine null_fcn(n, x, h, y, f, lrpar, rpar, lipar, ipar, ierr)
         use const_def, only: dp
         integer, intent(in) :: n, lrpar, lipar
         real(dp), intent(in) :: x, h
         real(dp), intent(inout) :: y(:) ! (n) 
            ! okay to edit y if necessary (e.g., replace negative values by zeros)
         real(dp), intent(out) :: f(:) ! (n) ! dy/dx
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr ! nonzero means retry with smaller timestep.
         f=0; ierr=0
      end subroutine null_fcn


      subroutine null_fcn_blk_dble(n, caller_id, nvar, nz, x, h, y, f, lrpar, rpar, lipar, ipar, ierr)
         use const_def, only: dp
         integer, intent(in) :: n, caller_id, nvar, nz, lrpar, lipar
         real(dp), intent(in) :: x, h
         real(dp), intent(inout), pointer :: y(:) 
            ! (n) okay to edit y if necessary (e.g., replace negative values by zeros)
         real(dp), intent(out), pointer :: f(:) ! (n) dy/dx
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr ! nonzero means retry with smaller timestep.
         f=0; ierr=0
      end subroutine null_fcn_blk_dble
      

      subroutine null_jac(n, x, h, y, f, dfy, ldfy, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: n, ldfy, lrpar, lipar
         real(dp), intent(in) :: x, h
         real(dp), intent(inout) :: y(:) ! (n)
         real(dp), intent(out) :: f(:) ! (n) ! dy/dx
         real(dp), intent(out) :: dfy(:,:) ! (ldfy, n)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr ! nonzero means terminate integration
         f=0; dfy=0; ierr=0
      end subroutine null_jac


      subroutine null_jac_blk_dble(n, caller_id, nvar, nz, x, h, y, f, lblk, dblk, ublk, lrpar, rpar, lipar, ipar, ierr)
         use const_def, only: dp
         integer, intent(in) :: n, caller_id, nvar, nz, lrpar, lipar
         real(dp), intent(in) :: x, h
         real(dp), intent(inout), pointer :: y(:) ! (n)
         real(dp), intent(out), pointer :: f(:) ! (n) dy/dx
         real(dp), dimension(:), pointer, intent(inout) :: lblk, dblk, ublk ! =(nvar,nvar,nz)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr ! nonzero means terminate integration
         f=0; y=0; ierr=0
      end subroutine null_jac_blk_dble


      subroutine null_sjac(n, x, h, y, f, nzmax, ia, ja, values, lrpar, rpar, lipar, ipar, ierr)  
         ! sparse jacobian. format either compressed row or compressed column.
         integer, intent(in) :: n, nzmax, lrpar, lipar
         real(dp), intent(in) :: x, h
         real(dp), intent(inout) :: y(:) ! (n)
         real(dp), intent(out) :: f(:) ! (n) ! dy/dx
         integer, intent(out) :: ia(:) ! (n+1)
         integer, intent(out) :: ja(:) ! (nzmax)
         real(dp), intent(out) :: values(:) ! (nzmax)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr ! nonzero means terminate integration
         f=0; values = 0; ia = 0; ja = 0; ierr = 0
      end subroutine null_sjac


      subroutine null_mas(n, am, lmas, lrpar, rpar, lipar, ipar)
         integer, intent(in) :: n, lmas, lrpar, lipar
         real(dp), intent(out) :: am(:,:) ! (lmas, n)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         am = 0
      end subroutine null_mas


      subroutine null_solout(nr, xold, x, n, y, rwork, iwork, interp_y, lrpar, rpar, lipar, ipar, irtrn)
         integer, intent(in) :: nr, n, lrpar, lipar
         real(dp), intent(in) :: xold, x
         real(dp), intent(inout) :: y(:) ! (n)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         real(dp), intent(inout), target :: rwork(*)
         integer, intent(inout), target :: iwork(*)
         interface
            real(dp) function interp_y(i, s, rwork, iwork, ierr)
               use const_def, only: dp
               integer, intent(in) :: i ! result is interpolated approximation of y(i) at x=s.
               real(dp), intent(in) :: s ! interpolation x value (between xold and x).
               real(dp), intent(inout), target :: rwork(*)
               integer, intent(inout), target :: iwork(*)
               integer, intent(out) :: ierr
            end function interp_y
         end interface
         integer, intent(out) :: irtrn
         irtrn = 0
      end subroutine null_solout
      
      
      subroutine null_dfx(n, x, y, fx, lrpar, rpar, lipar, ipar, ierr)
         integer, intent(in) :: n, lrpar, lipar
         real(dp), intent(in) :: x, y(:) ! (n)
         real(dp), intent(out) :: fx(:) ! (n)
         integer, intent(inout), pointer :: ipar(:) ! (lipar)
         real(dp), intent(inout), pointer :: rpar(:) ! (lrpar)
         integer, intent(out) :: ierr
         ierr = 0
         fx = 0
      end subroutine null_dfx
      
      
      ! Newton-Raphson iterative solver for nonlinear systems
      ! square, banded, or sparse matrix      
      ! analytic or numerical difference jacobian      
      ! where possible, reuses jacobian to improve efficiency      
      ! uses line search method to improve "global" convergence
      include "num_newton.dek"            
      


      ! minimize scalar function of many variables without using derivatives.

      ! NEWUOA
      !     This subroutine seeks the least value of a function of many variables,
      !     by applying a trust region method that forms quadratic models by
      !     interpolation. There is usually some freedom in the interpolation
      !     conditions, which is taken up by minimizing the Frobenius norm of
      !     the change to the second derivative of the model, beginning with the
      !     zero matrix.
      ! by M.J.D. Powell (mjdp@cam.ac.uk)    
      ! M.J.D. Powell, "Developments of NEWUOA for unconstrained minimization without derivatives",
      ! Department of Applied Mathematics and Theoretical Physics, Cambridge, England, report NA05, 2007.  
      include "num_newuoa.dek"


      ! BOBYQA
      !     Similar to NEWUOA, but the values of the variables are constrained 
      !     by upper and lower bounds.
      ! by M.J.D. Powell (mjdp@cam.ac.uk)      
      ! M.J.D. Powell, "The BOBYQA algorithm for bound constrained optimization without derivatives",
      ! Department of Applied Mathematics and Theoretical Physics, Cambridge, England, report NA06, 2009.  
      include "num_bobyqa.dek"
      
      
      ! Nelder-Mead Simplex Method
      !     doesn't use interpolation, so robust with noise data.
      include "num_simplex.dek"
         ! Nelder, J. A. and Mead, R.
         ! "A Simplex Method for Function Minimization."
         ! Comput. J. 7, 308-313, 1965.

      
      
      ! global or local minimum of scalar function of 1 variable
      include "num_brent.dek"


      
      ! QuickSort. ACM Algorithm 402, van Emden, 1970
      ! mesa's implementation from Joseph M. Krahn
      ! http://fortranwiki.org/fortran/show/qsort_inline
      
      subroutine qsort(index,n,vals)
         use mod_qsort, only: sortp_dp
         integer :: index(:), n
         real(dp) :: vals(:)
         call sortp_dp(n,index,vals)
      end subroutine qsort
      
      subroutine qsort_strings(index,n,strings)
         use mod_qsort, only: sortp_string
         integer :: index(:), n
         character(len=*), intent(in) :: strings(:)
         call sortp_string(n,index,strings)
      end subroutine qsort_strings
      
      subroutine qsort_string_index(index,n,string_index,strings)
         use mod_qsort, only: sortp_string_index
         integer :: index(:), n
         integer, intent(in) :: string_index(:) ! (n)
         character(len=*), intent(in) :: strings(:) ! 1..maxval(string_index)
         call sortp_string_index(n,index,string_index,strings)
      end subroutine qsort_string_index
      
      
      ! random numbers
      real(dp) function get_dp_uniform_01(seed)
         ! returns a unit pseudorandom real(dp)
         use mod_random, only: r8_uniform_01
         integer ( kind = 4 ) seed
         get_dp_uniform_01 = r8_uniform_01(seed)
      end function get_dp_uniform_01


      function get_i4_uniform(a, b, seed)
         ! The pseudorandom integer will be scaled to be uniformly distributed
         ! between a and b.
         use mod_random, only: i4_uniform
         integer ( kind = 4 ) a, b, seed, get_i4_uniform
         get_i4_uniform = i4_uniform(a, b, seed)
      end function get_i4_uniform
      
      
      subroutine get_perm_uniform ( n, base, seed, p )
         ! selects a random permutation of n integers
         use mod_random, only: perm_uniform
         integer ( kind = 4 ) n
         integer ( kind = 4 ) base
         integer ( kind = 4 ) p(n)
         integer ( kind = 4 ) seed
         call perm_uniform ( n, base, seed, p )
      end subroutine get_perm_uniform
      
      
      subroutine get_seed_for_random(seed)
         ! returns a seed for the random number generator
         use mod_random, only: get_seed
         integer ( kind = 4 ) seed
         call get_seed(seed)
      end subroutine get_seed_for_random

      
      ! binary search
      include "num_binary_search.dek"

      
      real(dp) function linear_interp(x1, y1, x2, y2, x)
         real(dp), intent(in) :: x1, y1, x2, y2, x
         if (x2 == x1) then
            linear_interp = (y1+y2)/2
         else
            linear_interp = y1 + (y2-y1)*(x-x1)/(x2-x1)
         end if
      end function linear_interp


      real(dp) function find0(xx1, yy1, xx2, yy2) result(x)
         ! find x between xx1 and xx2 s.t. linear_interp(xx1, yy1, xx2, yy2, x) == 0
         real(dp), intent(in) :: xx1, yy1, xx2, yy2
         real(dp) :: a, b
         a = (xx1*yy2)-(xx2*yy1)
         b = yy2-yy1
         if ((abs(a) .ge. abs(b)*1d99) .or. 
     >        ((abs(a) .ge. abs(b)*1d30) .and. 
     >           ((yy1 .ge. 0d0 .and. yy2 .le. 0d0) .or. (yy1 .le. 0d0 .and. yy2 .ge. 0d0)))) then
            x = 0.5d0*(xx1+xx2)
         else
            x = a/b
         end if
         if (yy1*yy2 <= 0) then ! sanity check
            if (x > max(xx1,xx2)) x = max(xx1,xx2)
            if (x < min(xx1,xx2)) x = min(xx1,xx2)
         end if
      end function find0


      real(dp) function find0_quadratic(xx1, yy1, xx2, yy2, xx3, yy3, ierr) result(x)
         ! find x between xx1 and xx3 s.t. quad_interp(xx1, yy1, xx2, yy2, xx3, yy3, x) == 0
         ! xx2 between xx1 and xx3; yy1 and yy3 different sign; yy2 between yy1 and yy3.
         real(dp), intent(in) :: xx1, yy1, xx2, yy2, xx3, yy3
         integer, intent(out) :: ierr
         real(dp) :: a, b, s2, denom
         ierr = 0; x = 0
         s2 = (xx3**2*(-yy1 + yy2) + xx2**2*(yy1 - yy3) + xx1**2*(-yy2 + yy3))**2 -  
     >        4*(xx3*(-yy1 + yy2) + xx2*(yy1 - yy3) + xx1*(-yy2 + yy3)) 
     >           *(xx1*xx3*(-xx1 + xx3)*yy2 +  
     >              xx2**2*(xx3*yy1 - xx1*yy3) + xx2*(-xx3**2*yy1 + xx1**2*yy3))
         if (s2 < 0) then
            ierr = -1
            return
         end if
         b = sqrt(s2)
         a = xx3**2*(yy1 - yy2) + xx1**2*(yy2 - yy3) + xx2**2*(-yy1 + yy3)
         denom = 2*(xx3*(yy1 - yy2) + xx1*(yy2 - yy3) + xx2*(-yy1 + yy3))
         x = (a + b)/denom
         if (x > max(xx1,xx2,xx3)) x = (a - b)/denom
         if (x < min(xx1,xx2,xx3) .or. x > max(xx1,xx2,xx3)) ierr = -1
      end function find0_quadratic
      
      
      subroutine two_piece_linear_coeffs(x, x0, x1, x2, a0, a1, a2, ierr)
         ! interpolation value at x is a0*f(x0) + a1*f(x1) + a2*f(x2)
         real(dp), intent(in) :: x, x0, x1, x2
         real(dp), intent(out) :: a0, a1, a2
         integer, intent(out) :: ierr
         ierr = 0
         if (x0 < x1 .and. x1 < x2) then
            if (x <= x0) then
               a0 = 1; a1 = 0; a2 = 0
            else if (x >= x2) then
               a0 = 0; a1 = 0; a2 = 1
            else if (x <= x1) then
               a1 = min(1d0, max(0d0, (x - x0)/(x1 - x0)))
               a0 = 1 - a1; a2 = 0
            else if (x < x2) then
               a2 = min(1d0, max(0d0, (x - x1)/(x2 - x1))) ! a2 => 1 as x => x2
               a1 = 1 - a2; a0 = 0
            end if
         else if (x0 > x1 .and. x1 > x2) then
            if (x >= x0) then
               a0 = 1; a1 = 0; a2 = 0
            else if (x <= x2) then
               a0 = 0; a1 = 0; a2 = 1
            else if (x >= x1) then
               a1 = min(1d0, max(0d0, (x - x0)/(x1 - x0)))
               a0 = 1 - a1; a2 = 0
            else if (x > x2) then
               a2 = min(1d0, max(0d0, (x - x1)/(x2 - x1))) ! a2 => 1 as x => x2
               a1 = 1 - a2; a0 = 0
            end if
         else
            ierr = -1
         end if
      end subroutine two_piece_linear_coeffs







      
      
      
      !  MEBDFI_95 is a routine for solving stiff initial value problems
      !  with fully implicit systems of differential algebraic equations:
      !  
      !      g(t,y,y')=0, with vector y=(y(1),y(2),y(3),.....,y(n)).
      !  
      !  The code is based on MEBDFI, backward differentiation formulas (BDF)
      !  as modified (M) and extended (E) by Jeff Cash, and then modified some 
      !  more by him to handle fully implicit equations (I).  So the name 
      !  can be parsed as M-E-BDF-I, a modified extended backward difference 
      !  formula scheme for stiff fully implicit initial value problems.
      !  
      !      References
      !  
      !  J. R. Cash, Efficient numerical methods for the solution of stiff 
      !         initial-value problems and differential algebraic equations", 
      !         Proc. Roy. Soc. London, Ser. A, vol 459, (2003) pp 797-815.
      !  
      !  E. Hairer and G. Wanner, Solving Ordinary Differential
      !         Equations II, Stiff and Differential-Algebraic Problems,
      !         Springer 1996, Page 267.
      !
! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************



c ***********************************************************************
c
c    Original version of mebdfi written by T.J. Abdulla and J.R. Cash,
c    Department of Mathematics,
c    Imperial College,
c    London SW7 2AZ
c    England
c
c    t.abdulla@ic.ac.uk   or  j.cash@ic.ac.uk
c
c ***********************************************************************
c
c    MEBDFI_95_solver created by Bill Paxton
c    October, 2006
c    http://theory.kitp.ucsb.edu/~paxton/
c
c    Here's a fairly complete checklist of the changes I made.
c
c    1) fortran95
c    2) implicit none everywhere
c    3) numeric labels are now only for format's, not for control
c    4) vector notation replaces inner loops where possible
c    5) LAPACK matrix routines (so can use low level OpenMP parallelism)
c    6) no COMMON's, no SAVE's,  no globals of any kind -- so is thread safe.
c    7) a few "object-oriented" additions to the interface
c
c    And the most important of all:
c
c    8) no changes in the basic numerical algorithms -- 
c    	  carefully checked for consistent results with the fortran77 mebdfi.
c
c ***********************************************************************






* ***********************************************************************
*
*  Background
*  ==========
*  
*  MEBDFI_95 is a routine for solving stiff initial value problems
*  with fully implicit systems of differential algebraic equations:
*  
*      g(t,y,y')=0, with vector y=(y(1),y(2),y(3),.....,y(n)).
*  
*  The code is based on MEBDFI, backward differentiation formulas (BDF)
*  as modified (M) and extended (E) by Jeff Cash, and then modified some 
*  more by him to handle fully implicit equations (I).  So the name 
*  can be parsed as M-E-BDF-I, a modified extended backward difference 
*  formula scheme for stiff fully implicit initial value problems.
*  
*      References
*  
*  J. R. Cash, Efficient numerical methods for the solution of stiff 
*         initial-value problems and differential algebraic equations", 
*         Proc. Roy. Soc. London, Ser. A, vol 459, (2003) pp 797-815.
*  
*  E. Hairer and G. Wanner, Solving Ordinary Differential
*         Equations II, Stiff and Differential-Algebraic Problems,
*         Springer 1996, Page 267.
*
*  
*
*  Description
*  ===========
*
*  MEBDFI schemes combine the multistep approach of typical BDF solvers
*  with the multistage approach of solvers such as Runge-Kutta.
*
*  The time-step and order of accuracy are automatically adjusted.
*  As in standard order-k BDF schemes, up to k previous approximate 
*  solutions are used for time derivatives by backward differencing.
*  The previous solutions are also used in extrapolating to create
*  predicted solutions for use as input to a Newton iteration that
*  attempts to converge to a corrected solution that satisfies the
*  given error tolerances.
*
*  Three stages are used by MEBDFI for each timestep.  The first two 
*  stages use a standard (k-1)-order BDF algorithm, while the last stage 
*  uses the 2 previous stages' results to produce an order k solution.  
*
*  For a current time of t and a time-step of dt, the first and last 
*  stages are evaluated at t+dt while the middle stage is evaluated
*  at the "super-future" time of t+2*dt.  The placement of the 2nd 
*  stage in the super-future gives the higher order accuracy and 
*  the improved stability of the method (A-stable up to order 4 and 
*  A(alpha)-stable up to order 9).
*
*  The converged (k-1)-order solution from the 1st stage is compared
*  to the converged k-order solution from the 3rd stage to provide
*  a reliable time-step error estimate.
*  
*  A modified Newton method is employed where the iteration matrix
*  and its LU decomposition are updated only when deemed necessary.
*  The Newton iteration matrix used for the 1st stage can usually be
*  reused for the 3rd stage, and the 1st stage solution provides a
*  good starting prediction for the 3rd stage Newton iteration.
*
*  In addition, the previous time-step's 2nd stage can often be used 
*  as a good starting prediction for the next time-step's 1st stage.
*
*  Since an approximate solution is not accepted without already having
*  produced a converged solution for a future time, the rate of failure
*  in attempting a next time-step is usually low.  However, if such a
*  failure does occur, a shorter time-step can be tried.  The attempt
*  to do the integration is only abandoned when repeated trials have
*  either failed to converge or failed to produce tolerable errors.
*
*  A change of parameters can be made after either a successful or 
*  an unsuccessful return from MEBDFI.  If a normal return occured and 
*  a normal continuation is desired, simply update the requested stopping
*  time and call again.
*
*
*  =====================================================================
*
*  Arguments
*  =========
*
*  n       (input) INTEGER
*          The number of dependent variables.
*          NOTE: the equations must be defined so that the
*          index 1 variables precede the index 2 variables,
*          which in turn precede the index 3 variables.
*
*  t0      (input/output) DOUBLE PRECISION
*          On entry for the first call, the starting value for t.
*          On exit, the new value of t.
*
*  h0      (input/output) DOUBLE PRECISION
*          On entry for the first call, the starting step size in t.
*          On exit, the stepsize h used last, whether successfully or not.
*
*  y0      (input/output) DOUBLE PRECISION array, dimension (n)
*          On entry for the first call, the initial values of y.
*          On exit, the new values of y at the new t = t0.
*
*  yprime  (input/output) DOUBLE PRECISION array, dimension (n)
*          On entry for the first call, the initial values of y' = dy/dt.
*          On exit, the new values of y' at the new t = t0.
*
*  tout    (input) DOUBLE PRECISION
*          The value of t at which output is desired next.
*          Integration will normally go slightly beyond tout,
*          and the package will interpolate y and yprime to t = tout.
*
*  tend    (input) DOUBLE PRECISION
*          The end of the range of integration.
*
*  hmax    (input) DOUBLE PRECISION
*          The maximum allowed magnitude for step-size. (0 means any size is ok)
*
*  njflag  (input) INTEGER
*          The "numerical jacobian" flag. If non-zero, then
*			  the jacobian will be constructed by numeric differencing.
*			  If zero, then the pderv routine will be called
*			  to create the jacobian.
*
*  mtx_shape  (input) INTEGER
*          The matrix shape indicator.
*          		=  0:  square.
*          		=  1:  banded.
*          		=  2:  sparse (NOT IMPLEMENTED).
*
*  equil   (input) INTEGER
*          The "equilibration" flag. If non-zero, then
*			  the jacobian will equilibrated.  This is done
*          by the LAPACK routines DGESVX or DGBSVX, and
*          their documentation should be consulted for details.
*
*  idid    (input/output) INTEGER
*          On entry, indicates the type of call.
*          		=  1:  this is the first call for the problem.
*          		=  0:  this is not the first call for this problem,
*          		       and integration is to continue with the same parameters.
*          		= -1:  this is not the first call for the problem,
*          		       and the user has modified the parameters.
*          		=  2:  same as 0 except that tout has to be hit
*          		       exactly (no interpolation is done).
*          		       assumes tout >= the current t.
*          		=  3:  same as 0 except control returns to calling
*          		       program after one step. tout is ignored, until the
*          		       integration reaches tout or beyond. if it passes tout
*          		       the program interpolates the solution values and
*          		       returns the solution value at tout.
*          On exit, indicates the type of result.
*          		=  1:  this is the return from the first call for the problem.
*          		=  0:  integration was completed to tout or beyond.
*          		= -1:  the integration was halted after failing to pass the
*          		       error test even after reducing h by a factor of
*          		       1.e10 from its initial value.
*          		= -2:  after some initial success, the integration was
*          		       halted either by repeated error test failures or by
*          		       a test on rtol/atol.  too much accuracy has been requested.
*          		= -3:  the integration was halted after failing to achieve
*          		       corrector convergence even after reducing h by a
*          		       factor of 1.e10 from its initial value.
*          		= -4:  immediate halt because of illegal values of input
*          		       parameters.  see printed message.
*          		= -5:  idid was -1 on input, but the desired changes of
*          		       parameters were not implemented because tout
*          		       was not beyond t.  interpolation at t = tout was
*          		       performed as on a normal return.  to try again,
*          		       simply call again with idid = -1 and a new tout.
*          		= -6:  maximum allowable number of integration steps exceeded.
*          		       to continue the user should increase iwork(14).
*          		= -7:  stepsize is too small for machine accuracy.
*          		= -8:  call_back returned with ierr nonzero.
*          		=-11:  insufficient real workspace for the integration.
*          		=-12:  insufficient integer workspace for the integration.
*
*          Since the normal output value of idid is 0,
*          it need not be reset for normal continuation.
*          The first call to the driver is with idid=1 and for
*          a successful step the driver returns with idid=1. Thus
*          the call with idid = 1 is simply the first
*          initialising step for the code.  The user
*          then needs to continue with idid=0,-1,2 or 3 as above.
*          
*  lout    (input) INTEGER
*          The logical output channel for message passing (e.g., 6 for terminal)
*
*  verbosity (input) INTEGER
*          Controls amount of extra debugging information written to lout (0 means none).
*
*  lwork   (input) INTEGER
*          The size of the real(dp) work array.
*          lwork  >= (n_dwork_alpha + n_dwork_beta3*mbnd(3) + n_dwork_beta4*mbnd(4))*n + n_dwork_gamma
*
*  work    (workspace/output) DOUBLE PRECISION array, dimension (lwork)
*
*  liwork  (input) INTEGER
*          The size of the integer work array.
*          liwork  >= (n_iwork_alpha + n_iwork_beta3*mbnd(3) + n_iwork_beta4*mbnd(4))*n + n_iwork_gamma
*
*  iwork   (workspace/output) INTEGER array, dimension (liwork)
*
*  rdata   (output) DOUBLE PRECISION array, dimension (num_rdata) [num_rdata is defined in mebdfi_solver_def]
*          On exit, 
*			  rdata(r_hused) = the last stepsize successfully used by the integrator
*			  rdata(r_lte) = the last estimate of the local truncation error (stage3 compared to stage1)
*			  rdata(r_rcond) = the last estimate of the reciprocal condition number of the matrix after 
*			  		equilibration (only set if the 'equil' argument is nonzero). If it is less than the machine 
*          		precision (in particular, if it is 0), the matrix is singular to working precision.
*			  rdata(r_con) = the last time derivative conversion factor
*
*  idata   (output) INTEGER array, dimension (num_idata) [num_idata is defined in mebdfi_solver_def]
*          On exit, 
*			  idata(i_maxord) = the maximum order used so far in the integration
*			  idata(i_sum_nq) = the sum of nqused values so far
*			  idata(i_sum_nq2) = the sum of nqused^2 so far
*			  idata(i_nqused) = the last order successfully used
*			  idata(i_nstep) = the number of successful steps taken so far
*			  idata(i_nfail) = the number of failed steps
*			  idata(i_nre) = the number of residual evaluations so far
*			  idata(i_nje) = the number of jacobian evaluations so far
*			  idata(i_ndec) = the number of lu decompositions so far
*			  idata(i_nbsol) = the number of 'backsolves' so far
*			  idata(i_npset) = the number of times a new coefficient matrix
*			                      has been formed so far
*			  idata(i_ncoset) = the number of times the order of the method
*			                      used has been changed so far
*          idata(i_nsteps_prevJ) = the number of steps done with the previous jacobian
*          idata(i_nbsols_prevJ) = the number of backsolves done with the previous jacobian
*          idata(i_step_J) = the step number (nstep) when the current jacobian was created
*          idata(i_sol_J) = the solve number (nbsol) when the current jacobian was created
*
*  mbnd    (input) INTEGER array, dimension (4)
*          An integer array of dimension 4 for use when the newton iteration
*          matrix is banded.  If this matrix has ml diagonals
*          below the main diagonal and mu diagonals above the
*          main diagonal then on entry:
*          		mbnd(1) = ml
*          		mbnd(2) = mu
*          		mbnd(3) = mu + ml + 1
*          		mbnd(4) = 2*ml + mu + 1
*
*  nind1  (input) INTEGER
*			  The number of variables of index 1.
*
*  nind2  (input) INTEGER
*			  The number of variables of index 2.
*
*  nind3  (input) INTEGER
*			  The number of variables of index 3.
*
*  maxder  (input) INTEGER
*          The maximum order allowed for the solver is maxder + 1.
*          The value of maxder cannot exceed 7.  This is the
*          value recommended unless it is believed that there
*          are severe stability problems in which case maxder = 3
*          or 4 should be tried instead.
*
*  maxstp  (input) INTEGER
*			  The maximum allowed number of steps.
*
*  itol    (input) INTEGER
*          An indicator of the type of error control.  The
*          solver will control the vector e = (e(i)) of estimated
*          local errors in y according to an inequality of the form
*              rms-norm of (e(i)/ewt(i))  <=  1
*          where
*              rms-norm(v) = sqrt((sum v(i)**2)/n).
*          Here ewt = (ewt(i)) is a vector of weights which must
*          always be positive, and the values of rtol and atol
*          should be non-negative. If itol = 1 then single step error
*          estimates divided by ymax(i) will be kept less than 1
*          in root-mean-square norm.  The vector ymax of weights is
*          computed as follows: initially ymax(i) is set as
*          the maximum of 1 and abs(y(i)).  Thereafter ymax(i) is
*          the largest value of abs(y(i)) seen so far, or the
*          initial value ymax(i) if that is larger.
*          The following table gives the lengths (1 or n) of rtol
*          and atol, and the corresponding form of ewt(i)
*
*          itol   rtol   atol  ewt(i)
*           1      1      1    rtol(1)*ymax(i)													
*           2      1      1    rtol(1)*abs(y(i)) + atol(1)
*           3      1      n    rtol(1)*abs(y(i)) + atol(i)
*           4      n      1    rtol(i)*abs(y(i)) + atol(1)
*           5      n      n    rtol(i)*abs(y(i)) + atol(i)
*
*  num_tol (input) INTEGER
*          length of rtol and atol arrays (1 or n)
*
*  rtol    (input) DOUBLE PRECISION array, dimension (num_tol)
*          A relative error tolerance parameter. See description
*          of itol.
*
*  atol    (input) DOUBLE PRECISION array, dimension (num_tol)
*          An absolute error tolerance parameter. See description
*          of itol.

*           y_min -- smallest allowed value for element of solution y
*           y_max -- largest allowed value for element of solution y

*
*  inf_nrms (input) DOUBLE PRECISION array, dimension (4)
*          Bounds for infinity norms.  Don't consider stage i newton
*          iteration to have converged until max(abs(scaled correction))
*          is less than inf_nrms(i). Don't accept new solution unless
*          max(abs(scaled error estimate)) is less than inf_nrms(4).
*
*  num_rpar (input) INTEGER
*          length of rpar array
*
*  rpar    (input/output) DOUBLE PRECISION array, dimension (num_rpar)
*          Holds parameters for communication between the
*
*  num_ipar (input) INTEGER
*          length of ipar array
*
*  ipar    (input/output) INTEGER array, dimension (num_ipar)
*          Holds parameters for communication between the
*          calling program and the resid and pderv subroutines.
*
*  stage2_min  (input) INTEGER (>= 0)
*          Minimum number of Newton iterations for stage 2.
*          Typically use 1, but 2 is also a common choice.
*
*  jdebug  (input) INTEGER
*          An integer flag that can be used with pderv when 
*          debugging analytical jacobians.
*
*  pderv   SUBROUTINE
*          This routine is for creating analytic jacobians
*          rather than numerical approximations.
*          If you are going to have the system do a
*          numerical approximation, you can simply pass
*          the null_pderv routine which is provided for this case.
*
*          If miter = 1 or 3 pderv is called to
*          compute the Newton iteration matrix
*          of partial derivatives pd=dg/dy+(dg/dy')/con
*          where con is a scalar.  The iteration 
*          matrix is stored as an n by n array if the 
*          matrix is full.  If the iteration matrix is 
*          banded the array pd is of size mbnd(4)*n. 
*          If the iteration matrix is full, pd(i,j) is  
*          to be set to the partial derivative of the 
*          ith component of g(t,y,y') with
*          respect to y(j).  If the jacobian is banded
*          with mu diagonals above the main diagonal
*          the partial derivative of the ith component
*          of g(t,y,y') with respect to y(j) should be
*          put in pd(mu+1+i-j,j).
*
*			  If a numerical jacobian is being used, and jdebug
*          is equal to 1, then pderv is called with the
*          numerical jacobian after it is created.  This is
*	        typically so that the jacobian can be studied for debugging, 
*          but it is also possible to replace various entries by 
*          analytically calculated partial derivatives.  This can
*          be very handy if most of the equations are best done 
*          numerically but one or two require an analytic treatment.
*
*  resid   SUBROUTINE(neqns,t,h,y,delta,yprime,num_rpar,rpar,num_ipar,ipar,stage,iteration,ierr)
*          Computes the residual vector, delta, for the approximate solution y
*          and yprime at time t.
*			  stage is 0 or -1 when resid is called for doing a numerical jacobian.
*					stage == 0 for the "base" call with no modified variables.
*					stage == -1 when one or more variables have been modified.
*          otherwise stage is 1, 2, or 3 and iteration is the current Newton iteration number.
*
*  call_back   SUBROUTINE
*          This provides an alternative mechanism for monitoring the
*			  step-by-step progress of the solver, both to record selected
*			  intermediate results and to modify various parameters.
*          These things can also be done by setting idid = 3, but
*          some users may find using the call_back routine more convenient.
*			  If you are not going to use the call_back mechanism, simply
*          pass the dummy routine named null_call_back that is provided
*          for this situation.
*
*  monitor   SUBROUTINE
*          A routine for monitoring the results of the matrix solves.
*          Can be useful when debugging.  A dummy routine, null_monitor,
*          is provided.
*
*  ierr    (input/output) INTEGER
*          An integer flag which is always equal to zero
*          on input.  Subroutines resid and pderv should alter
*          ierr only if one of them encounters an illegal operation such
*          as the square root of a negative number or exponent
*          overflow. The caller can then alter parameters and call the
*          subroutine again with idid = -1.
*
*  =====================================================================
*

		! The routine MEBDFI_95_ibvp is an alternative interface to the general MEBDFI_95 solver.
		
		! Arguments (see above for more info)
		
		! m						number of components per point in solution
		! n						number of points in solution
		! U(m,n)					solution array
		! Uprime(m,n)			time derivative of solution (can be initialized to 0)
		! t0						starting time
		! tend					ending time
		! h0						initial step size
		! hmax					max allowed step size, or 0 to means anything goes
		! abs_tol(m)			absolute error tolerance parameters (eg, 1d-4)
		! rel_tol(m)			relative error tolerance parameters (eg, 1d-4)
		!							let err(i,k) be the estimated error for U(i,k)
		!							let ewt(i,k) = rel_tol(i)*abs(U(i,k)) + abs_tol(i)
		!							adjust timestep to keep RMS of err(i,k)/ewt(i,k) less than 1.0
		!							i.e., Sqrt[Sum(i=1,m;k=1,n)(err(i,k)/ewt(i,k))^2/(m*n)] < 1.0
		!							err(i,k) is highest order U(i,k) minus next lower order U(i,k)
		! max_order				maxiumum allowed order for solver (eg, 7)
		! max_steps				maximum number of allowed steps (eg, 10000)
		! num_rpar				number of your own real(dp) parameters (>= 0)
		! rpar(num_rpar) 		your parameter values
		! num_ipar				number of your own integer parameters (>= 0)
		! ipar(num_ipar) 		your parameter values
		! num_rd					the size of rdata (= num_rdata in mebdfi_solver_def)
		! rdata(num_rd)		the data values (see description above)
		! num_id					the size of idata (= num_idata in mebdfi_solver_def)
		! idata(num_id)		the data values (see description above)
		! ml_cells         	for banded jacobian, number of lower numbered cells  (0 for square matrix)
		! mu_cells				for banded jacobian, number of higher numbered cells  (0 for square matrix)
		! resid					your routine for calculating residuals
		! call_back				your routine for monitoring the step-by-step progress of the solution
		! numerical_jacobian	true means the system will do a numerical estimate for jacobian
		!							false means the system will call the following routine that you supply
		! equilibrate			true means the jacobian will equilibrated
		! pderv					if numerical_j, this is your routine for calculating jacobians
		!							otherwise, pass a dummy routine such as null_pderv (from solver_lib)
		! monitor				routine for monitoring the results of the matrix solves (for debugging)
		! lout    				the logical output channel for message passing (e.g., 6 for terminal)
		! verbosity 			controls amount of extra debugging information written to lout (0 means none).
		! ierr					0 result means AOK.

		subroutine MEBDFI_95_ibvp(
     >		m, n, U, Uprime, 
     >		t0, tend, h0, hmax, abs_tol, rel_tol, y_min, y_max, inf_nrms,
     >		max_order, max_steps,
     >		num_rpar, rpar, num_ipar, ipar,
     >		num_rd, rdata, num_id, idata,
     >		ml_cells, mu_cells,
     >		resid, call_back, pderv, monitor,
     >      caller_id, nvar, nz, lblk, dblk, ublk, decsolblk, 
     >		numerical_jacobian, equilibrate, stage2_min, jdebug, 
     >		lout, verbosity, ierr)

			use mebdfi_solver_def
         use const_def, only: dp
      	implicit none
			integer, intent(in) :: 
     >			m, n, num_ipar, num_rpar, num_rd, num_id, ml_cells, mu_cells
     		real(dp), intent(in) :: y_min, y_max
     		real(dp), intent(inout) :: 
     >			t0, tend, h0, hmax, inf_nrms(4)
			real(dp), pointer :: abs_tol(:), rel_tol(:) ! (m)
			real(dp), pointer :: U(:), Uprime(:) ! (m*n)
			integer, intent(inout) :: 
     >			lout, verbosity, max_order, max_steps, stage2_min
			integer, pointer :: ipar(:) ! (num_ipar)
			real(dp), pointer :: rpar(:) ! (num_rpar)
			integer, pointer :: idata(:) ! (num_id)
			real(dp), pointer :: rdata(:) ! (num_rd)
			logical, intent(in) :: numerical_jacobian, equilibrate, jdebug
			interface			
				include "mebdfi_call_back.dek"
				include "mebdfi_resid.dek"
				include "mebdfi_pderv.dek"
				include "mebdfi_monitor.dek"
			end interface
			integer, intent(out) :: ierr			

         ! bcyclic
         integer, intent(in) :: caller_id, nvar, nz
            ! nvar > 0 means use decsolblk
         real(dp), dimension(:), pointer, intent(inout) :: 
     >      lblk, dblk, ublk ! =(nvar,nvar,nz)
			interface
            include "mtx_decsolblk.dek"
			end interface

			! locals
			integer :: mbnd(4), ml, mu, liwork, lwork, idid, itol, num_tol, i, j
			integer :: equil, banded, njflag, nind1, nind2, nind3, jdbug, status
			real(dp) :: tout
			real(dp), target :: atol1_target(m*n), rtol1_target(m*n)
			real(dp), pointer :: atol1(:), rtol1(:)
			real(dp), pointer :: atol(:,:), rtol(:,:)
			
			atol1 => atol1_target; atol(1:m,1:n) => atol1(1:m*n)
			rtol1 => rtol1_target; rtol(1:m,1:n) => rtol1(1:m*n)

			if (ml_cells <= 0 .and. mu_cells <= 0) then
				banded = 0
				mbnd = m*n
			else
				banded = 1
			
				ml = m*(ml_cells+1) - 1
				mu = m*(mu_cells+1) - 1
			
				mbnd(1) = ml	
				mbnd(2) = mu
				mbnd(3) = ml + mu + 1
				mbnd(4) = 2*ml + mu + 1
			end if

			liwork = (n_iwork_alpha + n_iwork_beta3*mbnd(3)
     >					+ n_iwork_beta4*mbnd(4))*m*n + n_iwork_gamma
         lwork  = (n_dwork_alpha + n_dwork_beta3*mbnd(3)
     >					+ n_dwork_beta4*mbnd(4))*m*n + n_dwork_gamma
			
			ierr = 0
			
			num_tol = m*n
			do i=1,n
			   do j=1,m
   				atol(j,i) = abs_tol(j)
   				rtol(j,i) = rel_tol(j)
				end do
			end do
					
			nind1 = m*n
			nind2 = 0
			nind3 = 0
		
			if (equilibrate) then
				equil = 1
			else
				equil = 0
			end if
			if (jdebug) then
				jdbug = 1
			else
				jdbug = 0
			end if
			if (numerical_jacobian) then
				njflag = 1
			else
				njflag = 0
			end if
			idid = 1 ! indicating first call for the problem
			ierr = 0 ! error flag
			itol = 5 ! errors use vectors for both rtol and atol

			tout = tend
			
		 	call do_one_ibvp(lwork, liwork)

			return
			
			contains
			
#ifdef offload
         !dir$ attributes offload: mic :: do_one_ibvp
#endif
			subroutine do_one_ibvp(lwork, liwork)
	      	implicit none
			   integer, intent(in) :: lwork, liwork
   			real(dp), target :: work_target(lwork)
   			integer, target :: iwork_target(liwork)
   			real(dp), pointer :: work(:)
   			integer, pointer :: iwork(:)
   			
   			work => work_target
   			iwork => iwork_target
			
				do while (t0 < tend)

					call MEBDFI_95_solver(m*n,t0,h0,U,Uprime,
     >				tout,tend,hmax,njflag,banded,equil,idid,lout,verbosity,
     >				lwork,work,liwork,iwork,num_rd,rdata,num_id,idata,mbnd,nind1,nind2,nind3,
     >				max_order,max_steps,itol,num_tol,rtol1,atol1,
     >				y_min,y_max,inf_nrms,num_rpar,rpar,
     >				num_ipar,ipar,jdbug,stage2_min,pderv,resid,call_back,monitor,
     >            caller_id, nvar, nz, lblk, dblk, ublk, decsolblk, 
     >            ierr)
				
		         if (idid == 1) then
						idid = 0
					else if (ierr /= 0) then
						exit
					else if (idid /= 0 .and. idid /= 3) then
						ierr = -999
						exit
	         	endif
				
				end do 
		
			end subroutine do_one_ibvp
		
		end subroutine MEBDFI_95_ibvp
		
		
      subroutine MEBDFI_95_solver(
     >		n,t0,h0,y0,yprime,tout,tend,hmax,njflag,mtx_shape,equil,idid,lout,verbosity,
     >		lwork,work,liwork,iwork,num_rd,rdata,num_id,idata,mbnd,
     >		nind1,nind2,nind3,maxder,maxstp,itol,num_tol,rtol,atol,y_min,y_max,inf_nrms,
     >		num_rpar,rpar,num_ipar,ipar,jdebug,stage2_min,pderv,resid,call_back,monitor,
     >      caller_id,nvar,nz,lblk,dblk,ublk,decsolblk,ierr)
         use const_def, only: dp
			use mebdfi_solver_def
			use mebdfi_stiff
	      integer, intent(in) :: n,njflag,mtx_shape,lwork,liwork,
     >			nind1,nind2,nind3,num_tol,num_ipar,num_rpar,num_rd,num_id
			integer, intent(out) :: ierr
			integer, pointer :: idata(:) ! (num_id)
			integer, intent(inout) :: idid,mbnd(4),
     >			maxder,maxstp,lout,verbosity,itol,jdebug,equil,stage2_min
			
			real(dp), pointer :: y0(:), yprime(:) ! (n)
			real(dp), intent(inout) :: t0,h0,inf_nrms(4),tend,hmax
			real(dp), intent(out) :: tout
			real(dp), pointer :: rdata(:) ! (num_rd)
			real(dp), pointer :: rtol(:), atol(:) ! (num_tol)
     		real(dp), intent(in) :: y_min, y_max
			
			integer, pointer :: iwork(:) ! (liwork)
			real(dp), pointer :: work(:) ! (lwork)
			integer, pointer :: ipar(:) ! (num_ipar)
			real(dp), pointer :: rpar(:) ! (num_rpar)
			
			interface
				include "mebdfi_call_back.dek"
				include "mebdfi_resid.dek"
				include "mebdfi_pderv.dek"
				include "mebdfi_monitor.dek"
			end interface
         
         ! bcyclic
         integer, intent(in) :: caller_id, nvar, nz
            ! nvar > 0 means use decsolblk
         real(dp), dimension(:), pointer, intent(inout) :: 
     >      lblk, dblk, ublk ! =(nvar,nvar,nz)
			interface
            include "mtx_decsolblk.dek"
			end interface
			
			! locals

			real(dp) :: hstpsz(2,14), dlamch
	      integer :: equed_flag,i1,i2,i3,i4,i5,i6,i7,i8,i9,i10,i11,i12,i13,i,j,mf
         integer, pointer, dimension(:) :: ipw0, ipw1, ipw2, ipw3, ipw4, ipw5, ipw6,
     >         ipw7, ipw8, ipw9, ipw10, ipw11, ipw12
         real(dp), pointer, dimension(:) :: pw0, pw1, pw2, pw3, pw4, pw5, pw6,
     >         pw7, pw8, pw9, pw10, pw11, pw12
	      
			! for compiler
			i1=0; i2=0; i3=0; i4=0; i5=0; i6=0; i7=0; i8=0; i9=0; i10=0; i11=0; i12=0; i13=0
	
			if (num_rd < num_rdata) then
				idid = -12
				write (lout,9030) num_rdata
				return
			end if
			
			if (num_id < num_idata) then
				idid = -12
				write (lout,9040) num_idata
				return
			end if

			if (njflag == 0) then
				if (mtx_shape == 0) then
					mf = 21
				else
					mf = 23
				end if
			else
				if (mtx_shape == 0) then
					mf = 22
				else
					mf = 24
				end if
			end if
		
			if (idid /= 1) then
			
				i1 = iwork(n_iwork_basic+1)
				i2 = iwork(n_iwork_basic+2)
				i3 = iwork(n_iwork_basic+3)
				i4 = iwork(n_iwork_basic+4)
				i5 = iwork(n_iwork_basic+5)
				i6 = iwork(n_iwork_basic+6)
				i7 = iwork(n_iwork_basic+7)
				i8 = iwork(n_iwork_basic+8)
				i9 = iwork(n_iwork_basic+9)
				i10 = iwork(n_iwork_basic+10)
				i11 = iwork(n_iwork_basic+11)
				i12 = iwork(n_iwork_basic+12)
				equed_flag = iwork(n_iwork_basic+17)
				do i = 1, 2
					do j = 1, 14
						hstpsz(i,j) = work(n_dwork_basic + j + (i-1)*14)
					end do
				end do
				
	      else ! first call for this problem

				idata(1:num_idata) = 0
				rdata(1:num_rdata) = 0d0
				
				iwork(15:16) = 0
				iwork(18:liwork) = 0
				
				work(1:lwork) = 0d0
				hstpsz(1:2,1:14) = 0d0
	         if (n <= 0) then
	            write (lout,9020) n
	            idid = -4            
	         else
	            if(mf.lt.23) then	
						mbnd(3)=n
						mbnd(4)=n
					end if
	
					! y starts at n_dwork_ttl + 1
	            i1 = n_dwork_ttl + 1 + n*12 ! yhold
	            i2 = i1 + n*12 ! ynhold
	            i3 = i2 + n*2 ! ymax
	            i4 = i3 + n ! errors
	            i5 = i4 + n ! save1
	            i6 = i5 + n ! save2
	            i7 = i6 + n ! scale
	            i8 = i7 + n ! arh
	
	            i9 = i8   + n         ! pw starts here (factored matrix)
	            i10 = i9  + mbnd(4)*n ! pw1 starts here (unfactored version of matrix)
	            i11 = i10 + mbnd(3)*n ! pw_R starts here (row equilibration scales)
	            i12 = i11 + n         ! pw_C starts here (column equilibration scales)
	
	            i13 = i12 + n         ! end of work area
					
					equed_flag = 0

					dlamch = EPSILON(1d0); work(1) = dlamch  ! EPSILON is standard in f95

	            work(3) = sqrt(work(1))
	        		work(4) = 0d0
	            if (lwork.lt.i13) then
	               idid = -11
	               write (lout,9000) i13
	            endif          
	            if (liwork.lt.n+n_iwork_ttl) then
	               idid = -12
	               write (lout,9010) n+n_iwork_ttl
	            end if            
	         end if    
	         if (idid.lt.0) return
	
	      end if

	      ierr = 0
	               
         pw0(1:n*12) => work(n_dwork_ttl+1:n_dwork_ttl+n*12)
         pw1(1:n*12) => work(i1:i1-1+n*12)
         pw2(1:n*2) => work(i2:i2-1+n*2)
         pw3(1:n) => work(i3:i3-1+n)
         pw4(1:n) => work(i4:i4-1+n)
         pw5(1:n) => work(i5:i5-1+n)
         pw6(1:n) => work(i6:i6-1+n)
         pw7(1:n) => work(i7:i7-1+n)
         pw8(1:n) => work(i8:i8-1+n)
         pw9(1:mbnd(4)*n) => work(i9:i9-1+mbnd(4)*n)
         pw10(1:mbnd(3)*n) => work(i10:i10-1+mbnd(3)*n)
         pw11(1:n) => work(i11:i11-1+n)
         pw12(1:n) => work(i12:i12-1+n)
         
         ipw0(1:n) => iwork(n_iwork_ttl+1:n_iwork_ttl+n)
         
      	call ovdriv(
     >		n,t0,h0,y0,yprime,tout,tend,hmax,mf,idid,lout,pw0,
     >		pw1,pw2,pw3,pw4,pw5,pw6,pw7,pw8,
     >		pw9,pw10,pw11,pw12,equil,equed_flag,
     >		ipw0,mbnd,nind1,nind2,nind3,
     >		maxder,maxstp,itol,num_tol,rtol,atol,y_min,y_max,inf_nrms,
     >		rdata,idata,num_rpar,rpar,num_ipar,ipar,pderv,resid,call_back,monitor,
     >      caller_id,nvar,nz,lblk,dblk,ublk,decsolblk, 
     >		work(1),work(3),hstpsz,jdebug,stage2_min,ierr,verbosity,
     >		lwork,work,liwork,iwork,
     >		iwork(first_isaves_ovdriv:liwork),work(first_dsaves_ovdriv:lwork),
     >		iwork(first_isaves_stiff:liwork),work(first_dsaves_stiff:lwork))

			iwork(n_iwork_basic+1) = i1
			iwork(n_iwork_basic+2) = i2
			iwork(n_iwork_basic+3) = i3
			iwork(n_iwork_basic+4) = i4
			iwork(n_iwork_basic+5) = i5
			iwork(n_iwork_basic+6) = i6
			iwork(n_iwork_basic+7) = i7
			iwork(n_iwork_basic+8) = i8
			iwork(n_iwork_basic+9) = i9
			iwork(n_iwork_basic+10) = i10
			iwork(n_iwork_basic+11) = i11
			iwork(n_iwork_basic+12) = i12
			iwork(n_iwork_basic+17) = equed_flag
			do i = 1, 2
				do j = 1, 14
					work(n_dwork_basic + j + (i-1)*14) = hstpsz(i,j)
				end do
			end do

      return

 9000 format (/,/,' ***** error ***** integration halted in mebdfi',/,/,
     +       '   >>> real workspace is insufficient <<< ',/,
     +       '       workspace must be at least ',i8,' elements long')
 9010 format (/,/,' ***** error ***** integration halted in mebdfi',/,/,
     +       '   >>> integer workspace is insufficient <<< ',/,
     +       '       workspace must be at least ',i6,' elements long')
 9020 format (/,/,' ***** error ***** integration halted in mebdfi',/,/,
     +       '   >>> illegal value for number of equations <<< ',/,
     +       '                     with n = ',i6)
 9030 format (/,/,' ***** error ***** integration halted in mebdfi',/,/,
     +       '   >>> space for returned real data is insufficient <<< ',/,
     +       '       must be at least ',i8,' elements long')
 9040 format (/,/,' ***** error ***** integration halted in mebdfi',/,/,
     +       '   >>> space for returned integer data is insufficient <<< ',/,
     +       '       must be at least ',i6,' elements long')
		
			
      end subroutine MEBDFI_95_solver


		subroutine null_pderv(
     >			time,h,con,y,jac1,neqns,lblk,dblk,ublk,cid,nvar,nz,
     >         yprime,ldj,num_rpar,rpar,num_ipar,ipar,ierr)			
	      integer, intent(in) :: neqns, ldj, cid, nvar, nz, num_ipar, num_rpar
     		double precision, intent(in) :: time,h,con
	     	real(dp), pointer :: y(:), yprime(:) ! (neqns)
	     	real(dp), pointer :: jac1(:) ! (ldj*neqns)
         real(dp), dimension(:), pointer, intent(inout) :: lblk,dblk,ublk ! =(nvar,nvar,nz)
			integer, pointer :: ipar(:) ! (num_ipar)
			real(dp), pointer :: rpar(:) ! (num_rpar)
			integer, intent(out) :: ierr
			ierr = 0
		end subroutine null_pderv


		subroutine null_call_back(n,time,h,con,y,yprime,tout,tend,hmax,lout,verbosity,
     >			num_rdta,num_idta,rdata,idata,maxder,maxstp,equil,jdebug,
     >			itol,num_tol,rtol,atol,inf_nrms,num_rpar,rpar,num_ipar,ipar,idid,ierr)
         use const_def, only: dp
			use mebdfi_solver_def
	      integer, intent(in) :: n,num_rdta,num_idta,num_tol,num_ipar,num_rpar
	      real(dp), intent(in) :: time,h,con					
     		real(dp), pointer :: y(:), yprime(:) ! (n) 
			integer, intent(inout) :: idid,ierr,itol,lout,verbosity
			integer, intent(inout) :: maxder,maxstp,equil,jdebug
			real(dp), intent(inout) :: inf_nrms(4)
			real(dp), pointer :: rtol(:), atol(:) ! (num_tol)
			integer, pointer :: idata(:) ! (num_idta)		
			real(dp), pointer :: rdata(:) ! (num_rd)
			integer, pointer :: ipar(:) ! (num_ipar)
			real(dp), pointer :: rpar(:) ! (num_rpar)
			real(dp), intent(inout) :: tout,tend,hmax
			
			ierr = 0
			
		end subroutine null_call_back


		subroutine null_monitor(
     >				neqns,time,h,con,old_y,new_y,error,del_err,yprime,scale,d,crate,bnd,
     >            R,C,ipiv,rcond,recip_pivot_growth_factor,
     >				num_rpar,rpar,num_ipar,ipar,
     >				nqused,nstep,nbsol,stage,iteration,ierr)
         use const_def, only: dp
     		integer, intent(in) :: neqns, num_ipar, num_rpar, nqused, nstep, nbsol, stage, iteration
     		real(dp), intent(in) :: time, h, con, d, crate, bnd
     		real(dp), dimension(:), pointer :: old_y,new_y,error,del_err,yprime,scale ! (neqns)
	     	real(dp), intent(inout) :: rcond,recip_pivot_growth_factor
	     	real(dp), dimension(:), pointer :: R,C ! (neqns)
	     	integer, dimension(:), pointer :: ipiv ! (neqns)
			integer, pointer :: ipar(:) ! (num_ipar)
			real(dp), pointer :: rpar(:) ! (num_rpar)
			integer, intent(out) :: ierr
			
			ierr = 0
			
		end subroutine null_monitor


		subroutine basic_monitor(
     >				neqns,time,h,con,old_y,new_y,error,del_err,yprime,scale,d,crate,bnd,
     >            R,C,ipiv,rcond,recip_pivot_growth_factor,
     >				num_rpar,rpar,num_ipar,ipar,
     >				nqused,nstep,nbsol,stage,iteration,ierr)
         use const_def, only: dp
     		integer, intent(in) :: neqns, num_ipar, num_rpar, nqused, nstep, nbsol, stage, iteration
     		real(dp), intent(in) :: time, h, con, d, crate, bnd
     		real(dp), dimension(:), pointer :: old_y,new_y,error,del_err,yprime,scale ! (neqns)
	     	real(dp), intent(inout) :: rcond,recip_pivot_growth_factor
	     	real(dp), dimension(:), pointer :: R,C ! (neqns)
	     	integer, dimension(:), pointer :: ipiv ! (neqns)
			integer, pointer :: ipar(:) ! (num_ipar)
			real(dp), pointer :: rpar(:) ! (num_rpar)
			integer, intent(out) :: ierr
			
			ierr = 0
         
         if (stage == 1 .and. iteration == 1) write(*,'(/,5a9,4x,99(a10,4x))') 
     >         'nstep', 'stage', 'iter', 'nbsol', 'nqused', 'err est', 'bnd', 'd', 'crate', 'time', 'h'
         write(*,'(5i9,4x,99e14.6)') nstep, stage, iteration,  nbsol, nqused,
     >         sqrt(min(1d0,2d0*crate)*d/neqns), sqrt(bnd/neqns), d, crate, time, h
			
		end subroutine basic_monitor


		! the following routines are provided as an option for recording
		! information about the solver.

		subroutine std_open_mebdfi_history(io_unit,ierr)
			integer, intent(in) :: io_unit
			integer, intent(out) :: ierr
			
			call open_mebdfi_history('mebdfi_history_data',io_unit,ierr)
			
		end subroutine std_open_mebdfi_history


		subroutine open_mebdfi_history(data_dir,io_unit,ierr)
			use mebdfi_solver_def
			character(*), intent(in) :: data_dir
			integer, intent(in) :: io_unit
			integer, intent(out) :: ierr
			
			ierr = 0
			open(unit = io_unit, file = trim(data_dir) // '/names.data', iostat=ierr)
			if (ierr /= 0) return
         write(io_unit,'(a)')
     >      'time', 'log_dt', 'order', 'solves_for_recent_J', 'steps_for_recent_J',
     >      'overall_solves_per_J', 'overall_steps_per_J'
			close(io_unit)
			
			open(unit = io_unit, file = trim(data_dir) // '/test.data', iostat=ierr)
			
		end subroutine open_mebdfi_history		


		subroutine append_to_mebdfi_history(io_unit, time, h, num_id, idata)
         use const_def, only: dp
			use mebdfi_solver_def
			integer, intent(in) :: io_unit, num_id, idata(num_id)
			real(dp), intent(in) :: time, h

         write(unit=io_unit,fmt='(2e24.12,3i5,99e24.12)')
     >               time, log10(h), idata(i_nqused),
     >               idata(i_nbsols_prevJ), idata(i_nsteps_prevJ),
     >               dble(idata(i_nbsol))/idata(i_nje),
     >               dble(idata(i_nstep))/idata(i_nje)   
		
		end subroutine append_to_mebdfi_history
		
		
		subroutine report_mebdfi_stats(test_name,num_id,idata,tend,abs_tol,rel_tol,show_info)
         use const_def, only: dp
			use mebdfi_solver_def
			character(*), intent(in) :: test_name
			integer, intent(in) :: num_id, idata(num_id)
			real(dp), intent(in) :: tend, abs_tol, rel_tol
			logical, intent(in) :: show_info
			
			integer :: nsteps

			nsteps = idata(i_nstep)+idata(i_nfail)

			if (show_info) then
				write(*,'(a)') '	' // trim(test_name) // ' mebdfi statistics'
				write(*,'(a)')
				write(*,'(a)') '	"abs_tol" is the absolute error tolerance.'
				write(*,'(a)') '	"rel_tol" is the absolute error tolerance.'
				write(*,'(a)') '	"tries" is the total number of attempts to take a step.'
				write(*,'(a)') '	"steps" is the sum of steps taken to reach the ending time for the problem.'
				write(*,'(a)') '	"fail" is the number of tries that had to be redone with a smaller timestep.'
				write(*,'(a)') '	"jacob" is the total number of jacobians created.'
				write(*,'(a)') '	"solve" is the total number matrix solves performed.'
				write(*,'(a)') '	"avg k" is the average order used in the solution.'
				write(*,*)
			end if
			
      	write(*,'(a12,5x,a12,3x,a12,1x,7a8)')
     >			'tend', 'abs_tol', 'rel_tol', 'tries', 'steps', 'fail', 'jacob', 'solve', 'avg k'
			
			write(*,'(3(3x,e12.3),5i8,2f8.1)') 
     >      tend, abs_tol, rel_tol, nsteps, idata(i_nstep), idata(i_nfail), idata(i_nje), 
     >		idata(i_nbsol), dble(idata(i_sum_nq))/nsteps

		end subroutine report_mebdfi_stats
		





#ifdef offload
      !dir$ end options
#endif

      end module num_lib

