

      module mod_qsort
      use const_def, only: dp
      
      
      implicit none

#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif

      contains


  
           ! FILE: sort.f
         ! PURPOSE: demonstrate the use of "qsort_inline.inc" and
         ! "qsort_inline_index.inc". These can be used as specific
         ! sort procedures under a common SORT generic name.
         !---------------------------------------------------------------
         ! Sort a string array, with any string length.
         subroutine sortp_string(array_size,index,string)
           integer, intent(in) :: array_size
           integer, intent(out) :: index(:) ! (array_size)
           character(len=*), intent(in) :: string(:) ! (array_size)
           include "qsort_inline.inc"
         contains
#ifdef offload
            !dir$ attributes offload: mic :: init
#endif
         ! set up initial index:
           subroutine init()
             integer :: i
             do i=1,array_size
               index(i)=i
             end do
           end subroutine init

#ifdef offload
            !dir$ attributes offload: mic :: swap
#endif
         ! swap indices a,b
           subroutine swap(a,b)
             integer, intent(in) :: a,b
             integer :: hold
             hold=index(a)
             index(a)=index(b)
             index(b)=hold
           end subroutine swap

#ifdef offload
            !dir$ attributes offload: mic :: rshift
#endif
         ! circular shift-right by one:
           subroutine rshift(left,right)
             implicit none
             integer, intent(in) :: left, right
             integer :: hold, i
             hold=index(right)
             ! This sytnax is valid, but has poor optimization in GFortran:
             ! index(left+1:right)=index(left:right-1)
             do i=right,left+1,-1
               index(i)=index(i-1)
             end do
             index(left)=hold
           end subroutine rshift
  
#ifdef offload
            !dir$ attributes offload: mic :: less_than
#endif
           logical &
           function less_than(a,b)
             integer, intent(in) :: a,b
             if ( string(index(a)) == string(index(b))  ) then
               less_than = ( index(a) < index(b) )
             else
               less_than = ( string(index(a)) < string(index(b)) )
             end if
           end function less_than
           
         end subroutine sortp_string
         !---------------------------------------------------------------
         ! Sort an array of indices into a string array, with any string length.
         subroutine sortp_string_index(array_size,index,str_index,string)
           integer, intent(in) :: array_size
           integer, intent(out) :: index(:) ! (array_size)
           integer, intent(in) :: str_index(:) ! (array_size)
           character(len=*), intent(in) :: string(:) ! 1..maxval(str_index)
           include "qsort_inline.inc"
         contains
#ifdef offload
            !dir$ attributes offload: mic :: init
#endif
         ! set up initial index:
           subroutine init()
             integer :: i
             do i=1,array_size
               index(i)=i
             end do
           end subroutine init

#ifdef offload
            !dir$ attributes offload: mic :: swap
#endif
         ! swap indices a,b
           subroutine swap(a,b)
             integer, intent(in) :: a,b
             integer :: hold
             hold=index(a)
             index(a)=index(b)
             index(b)=hold
           end subroutine swap

#ifdef offload
            !dir$ attributes offload: mic :: rshift
#endif
         ! circular shift-right by one:
           subroutine rshift(left,right)
             implicit none
             integer, intent(in) :: left, right
             integer :: hold, i
             hold=index(right)
             ! This sytnax is valid, but has poor optimization in GFortran:
             ! index(left+1:right)=index(left:right-1)
             do i=right,left+1,-1
               index(i)=index(i-1)
             end do
             index(left)=hold
           end subroutine rshift
           
#ifdef offload
            !dir$ attributes offload: mic :: less_than
#endif
           logical &
           function less_than(a,b)
             integer, intent(in) :: a,b
             if ( string(str_index(index(a))) == string(str_index(index(b)))  ) then
               less_than = ( str_index(index(a)) < str_index(index(b)) )
             else
               less_than = ( string(str_index(index(a))) < string(str_index(index(b))) )
             end if
           end function less_than
           
         end subroutine sortp_string_index
         !---------------------------------------------------------------
         ! Sort a double-precision array by index
         subroutine sortp_dp(array_size,index,value)
           integer, intent(in) :: array_size
           integer, intent(inout) :: index(:) ! (array_size)
           double precision, intent(in) :: value(:) ! (array_size)
           include "qsort_inline.inc"
         contains
#ifdef offload
            !dir$ attributes offload: mic :: init
#endif
         ! set up initial index:
           subroutine init()
             integer :: i
             do i=1,array_size
               index(i)=i
             end do
           end subroutine init

#ifdef offload
            !dir$ attributes offload: mic :: swap
#endif
         ! swap indices a,b
           subroutine swap(a,b)
             integer, intent(in) :: a,b
             integer :: hold
             hold=index(a)
             index(a)=index(b)
             index(b)=hold
           end subroutine swap

#ifdef offload
            !dir$ attributes offload: mic :: rshift
#endif
         ! circular shift-right by one:
           subroutine rshift(left,right)
             implicit none
             integer, intent(in) :: left, right
             integer :: hold, i
             hold=index(right)
             ! This sytnax is valid, but has poor optimization in GFortran:
             ! index(left+1:right)=index(left:right-1)
             do i=right,left+1,-1
               index(i)=index(i-1)
             end do
             index(left)=hold
           end subroutine rshift
  
#ifdef offload
            !dir$ attributes offload: mic :: less_than
#endif
           logical &
           function less_than(a,b)
             integer, intent(in) :: a,b
             less_than = value(index(a)) < value(index(b))
           end function less_than
           
         end subroutine sortp_dp
         !---------------------------------------------------------------
         ! Sort a single-precision real array by index, with a fuzzy equality test
         subroutine sortp_1r4(array_size,index,value)
           integer, intent(in) :: array_size
           integer, intent(inout) :: index(:) ! (array_size)
           real(4), intent(in) :: value(:) ! (array_size)
           include "qsort_inline.inc"
         contains
#ifdef offload
            !dir$ attributes offload: mic :: init
#endif
         ! set up initial index:
           subroutine init()
             integer :: i
             do i=1,array_size
               index(i)=i
             end do
           end subroutine init

#ifdef offload
            !dir$ attributes offload: mic :: swap
#endif
         ! swap indices a,b
           subroutine swap(a,b)
             integer, intent(in) :: a,b
             integer :: hold
             hold=index(a)
             index(a)=index(b)
             index(b)=hold
           end subroutine swap

#ifdef offload
            !dir$ attributes offload: mic :: rshift
#endif
         ! circular shift-right by one:
           subroutine rshift(left,right)
             implicit none
             integer, intent(in) :: left, right
             integer :: hold, i
             hold=index(right)
             ! This sytnax is valid, but has poor optimization in GFortran:
             ! index(left+1:right)=index(left:right-1)
             do i=right,left+1,-1
               index(i)=index(i-1)
             end do
             index(left)=hold
           end subroutine rshift
  
#ifdef offload
            !dir$ attributes offload: mic :: less_than
#endif
           logical &
           function less_than(a,b)
             integer, intent(in) :: a,b
             real(4), parameter :: small=1.0e-6
             if ( abs(value(index(a))-value(index(b))) < small ) then
               less_than = index(a) < index(b)
             else
               less_than = value(index(a)) < value(index(b))
             end if
           end function less_than
           
         end subroutine sortp_1r4
         !---------------------------------------------------------------
         ! Sort an array of integers
         subroutine sort_1i(array_size,i1)
           integer, intent(in) :: array_size
           integer, intent(inout) :: i1(:) ! (array_size)
           include "qsort_inline.inc"
         contains
#ifdef offload
            !dir$ attributes offload: mic :: init
#endif
           subroutine init()
           end subroutine init
           
#ifdef offload
            !dir$ attributes offload: mic :: swap
#endif
           subroutine swap(a,b)
             integer, intent(in) :: a,b
             integer :: hold
             hold=i1(a); i1(a)=i1(b); i1(b)=hold
           end subroutine swap
           
#ifdef offload
            !dir$ attributes offload: mic :: rshift
#endif
         ! circular shift-right by one:
           subroutine rshift(left,right)
             integer, intent(in) :: left, right
             integer :: hold
             hold=i1(right); i1(left+1:right)=i1(left:right-1); i1(left)=hold
           end subroutine rshift
           
#ifdef offload
            !dir$ attributes offload: mic :: less_than
#endif
           logical &
           function less_than(a,b)
             integer, intent(in) :: a,b
             if ( i1(a) == i1(b) ) then
               less_than = a < b
             else
               less_than = i1(a) < i1(b)
             end if
           end function less_than
           
         end subroutine sort_1i
         !---------------------------------------------------------------



#ifdef offload
      !dir$ end options
#endif
      
      
      end module mod_qsort
      
