! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************



      module mebdfi_stiff
      use const_def, only: dp
      use crlibm_lib
      
		implicit none

		contains
		
      subroutine ovdriv(
     >		n,t0,h0,y0,yprime,tout,tend,caller_hmax,mf,idid,lout,y1,
     >		yhold1,ynhold1,ymax,errors,
     >		save1,save2,scale,arh,
     >		pw,pw1,pw_R,pw_C,equil,equed_flag,
     >		ipiv,mbnd,nind1,nind2,nind3,maxder,maxstp,
     >      itol,num_tol,rtol,atol,y_min,y_max,inf_nrms,
     >		rdata,idata,num_rpar,rpar,num_ipar,ipar,pderv,resid,call_back,monitor,
     >      caller_id,nvar,nz,lblk,dblk,ublk,decsolblk, 
     >		uround,epsjac,hstpsz,jdebug,stage2_min,ierr,verbosity,
     >		lwork,work,liwork,iwork,
     >		ovdriv_integer_saves,ovdriv_double_saves,
     >		stiff_integer_saves,stiff_double_saves)

			use mebdfi_solver_def
			use mebdfi_support, only : interp
		
			integer, intent(in) :: n,mf,mbnd(4),nind1,nind2,nind3,lwork,liwork,
     >			num_tol,num_rpar,num_ipar
			integer, intent(inout) :: idid,maxder,maxstp,lout,itol,jdebug,
     >			equil,equed_flag,stiff_integer_saves(n_isaves_stiff), 
     >			ovdriv_integer_saves(n_isaves_ovdriv),verbosity,stage2_min
			integer, intent(out) :: ierr
		   integer, pointer :: idata(:) ! (num_idata)
		   integer, pointer :: ipiv(:) ! (n)
		   integer, pointer :: iwork(:) ! (liwork)
		   integer, pointer :: ipar(:) ! (num_ipar)
		   real(dp), pointer :: rpar(:) ! (num_rpar)
	
			real(dp), intent(in) :: epsjac,uround,y_min,y_max
			real(dp), intent(inout) :: t0,h0,
     >		stiff_double_saves(n_dsaves_stiff), ovdriv_double_saves(n_dsaves_ovdriv),
     >		inf_nrms(4),tend,caller_hmax
			real(dp), intent(out) :: tout,hstpsz(2,14)
			real(dp), pointer, dimension(:) :: pw,y0,yprime,errors,save1,save2,
     >		scale,arh,pw1,pw_R,pw_C,ymax,rdata,y1,yhold1,ynhold1,work,atol,rtol
			interface
				include "mebdfi_call_back.dek"
				include "mebdfi_resid.dek"
				include "mebdfi_pderv.dek"
				include "mebdfi_monitor.dek"
			end interface

         ! bcyclic
         integer, intent(in) :: caller_id, nvar, nz
            ! nvar > 0 means use decsolblk
         real(dp), dimension(:), pointer, intent(inout) :: 
     >      lblk, dblk, ublk ! =(nvar,nvar,nz)
			interface
            include "mtx_decsolblk.dek"
			end interface
					
			! SAVED
			real(dp) :: t, h, hmin, hmax
			integer :: kflag, kgo, jstart, nhcut
			
			! kflag -- a completion flag set by stiff
			!     0  the step was successful
			!    -1  the requested error could not be achieved
			!          with abs(h) = hmin.
			!    -2  the requested error is smaller than can
			!          be handled for this problem.
			!    -3  corrector convergence could not be
			!          achieved for abs(h)=hmin.
			
			! jstart -- for communication with stiff
			!   on input to stiff it has the following values and meanings..
			!    = 0  perform the first step.
			!    > 0  take a new step continuing from the last
			!    < 0  take the next step with a new value of h or n.
			!   on exit from stiff, jstart is the order of the method last used.
			!		it is needed for interpolation.
			
			! nhcut = number of times h has been cut in size
			

			! NOT SAVED
	      integer :: nn
			real(dp) :: tmp
         real(dp) :: dlamch, sfmin

        	real(dp), pointer :: y(:,:) ! (n,12)
        	real(dp), pointer :: yhold(:,:) ! (n,12)
        	real(dp), pointer :: ynhold(:,:) ! (n,2)
        	
        	integer :: lrd, lid
         real(dp), pointer :: rpar_decsol(:) ! (lrd)
         integer, pointer :: ipar_decsol(:) ! (lid)
     	
        	logical :: need_free ! if true, need to free bcyclic storage
        	real(dp), pointer :: brhs(:)
        	
         need_free = .false.
         lrd = 0; lid = 0; nullify(rpar_decsol, ipar_decsol)
			
			y(1:n,1:12) => y1(1:n*12)
			yhold(1:n,1:12) => yhold1(1:n*12)
			ynhold(1:n,1:2) => ynhold1(1:n*2)
			
			if (verbosity > 0) then
			   write(*,*) 'enter mebdfi'
			end if
		
         sfmin = dlamch('S')  
			rdata(r_lte) = 0d0

			if (idid == 1) then ! initialize the saved variables
				kflag = 0
				jstart = 0
				nhcut = 0
				t = 0d0
				h = 0d0
				hmin = 0d0
				hmax = 0d0
			else
				call unpack
			end if

	      if (idid == 0 .or. idid == 3) then
				! continue integration
	         t0=t
	         hmax = dabs(tend-t0)/10d0
	         if ((t-tout)*h >= 0d0) then
					call overshot(kflag)
      			return
	         end if

	      else if (idid == 2) then
				! i.e. continuing integration but wish to hit tout
	         t0 = t
	         hmax = dabs(tend-t0)*10d0
	         if (((t+h)-tout)*h > 0d0) then
					! we have already overshot the output point or we will
					! do so on the next step
	            if (((t-tout)*h >= 0d0)  .or.  (dabs(t-tout) <= 
     +           		100d0*uround*hmax)) then
						call overshot(kflag)
      				return
	            else
						! will pass tout on next step with current stepsize
						! so reduce stepsize to hit tout 'exactly'
	               h = (tout-t)* (1d0-4d0*uround)
	               jstart = -1
	            end if           
	         end if
         
	      else if (idid == -1) then
				! not first call but parameters reset
	         h = h0
	         if(h.lt.epsjac/100d0) then
	            write(lout,9160)
	            idid = -7
	            call repack
      			return
	         endif
	         t0 = t
	         if ((t-tout)*h >= 0d0) then
	            write (lout,9080) t,tout,h
					call overshot(-5)
      			return
	         else
	            jstart = -1
	         end if

	      else
				! idid should be 1 and this is the first call for this problem
				! check the arguments that were passed for correctness
	         if (idid /= 1) then
					! value of idid not allowed
	            write (lout,9070) idid
	            idid = -4
	         end if

				call stiff_init_data(n_isaves_stiff,stiff_integer_saves,
     >				n_dsaves_stiff,stiff_double_saves)

	         nn=n
	         if(itol <= 3) nn = 1
            if (minval(rtol(1:nn)).lt.0d0) then
					! illegal value for relative error tolerence
               write (lout,9040)
               idid = -4
            end if

	         nn=n
	         if(itol == 1 .or. itol == 2 .or. itol == 4) nn=1
            if (minval(atol(1:nn)).lt.0d0) then
					! illegal absolute error tolerance
               write(lout,9045)
               idid=-4
            endif
	         if(itol == 1 .and. rtol(1) == 0) then
					! illegal error tolerance
	            write(lout,9040)
	            idid = -4
	         endif
	         if(itol /= 1) then
	            tmp = 0d0
					select case (itol)
	            case (2) 
	               tmp = dmax1(rtol(1),atol(1))
	            case (3) 
	               tmp = dmax1(rtol(1),maxval(atol(1:n)))
	            case (4) 
	               tmp = dmax1(maxval(rtol(1:n)),atol(1))
	            case (5) 
	               tmp = dmax1(maxval(rtol(1:n)),maxval(atol(1:n)))
	            end select
               if(tmp <= 0d0) then
                  write(lout,9040)
                  idid = -4
               endif
	         endif

	         if (n <= 0) then
					! illegal value for the number of equations
	            write (lout,9050)
	            idid = -4
	         end if

	         if ((t0-tout)*h0 >= 0d0) then
					! parameters for integration are illegal
	            write (lout,9060)
	            idid = -4
	         end if

	         if ((mf /= 21) .and. (mf /= 22) .and. (mf /= 23) .and. (mf /= 24)) then
					! illegal value for method flag
	            write (lout,9090) mf
	            idid = -4
	         end if

	         if(itol.lt.1 .or. itol > 5) then
					! illegal value for error control parameter
	            write (lout,9110)
	            idid=-4
	         endif

	         if(maxder.lt.1 .or. maxder > 7) then
					! illegal value for maximum order
	            write(lout,9120)
	            idid = -4
	         endif

	         if(nind1 + nind2 + nind3  /=  n) then
					! sum of variables of different index should be n.
	            write(lout,9140)
	            idid = -4
	         endif

	         if (idid /= 1) then
	            call repack
      			return
	         else
					! the initial parameters are o.k. so initialise everything
					! else necessary for the integration.
					! if values of ymax other than those set below are desired,
					! they should be set here. all ymax(i) must be positive. if
					! values for hmin or hmax, the bounds on dabs(h), other than
					! those below are desired, they should be set below.

	            if (itol == 1) then
	               ymax(1:n) = dmax1(dabs(y0(1:n)),1.0d0)
	            endif
	            y(1:n,1)=y0(1:n)
	            t = t0
	            h = h0
	            hmin = dabs(h0)*1d-10
	            hmax = dabs(t0-tend)*10d0
	            jstart = 0
	            nhcut = 0
	         end if
         
	      end if
	
			!  <<<<<<<<<<<<<<<<<
			!  <  take a step  >
			!  <<<<<<<<<<<<<<<<<

			do_step: do

      	if ((t+h) == t) then
	         write (lout,9000)
	      end if

			tmp = hmax
			if (caller_hmax > 0d0 .and. caller_hmax < hmax) tmp = caller_hmax
			
	      call stiff(h,tmp,hmin,jstart,kflag,mf,mbnd,sfmin,
     +    	nind1,nind2,nind3,t,tout,tend,y1,yprime,n,
     +    	ymax,errors,save1,save2,scale,
     +    	pw,pw1,pw_R,pw_C,equil,equed_flag,
     +    	yhold1,ynhold1,arh,ipiv,lout,maxder,itol,num_tol,rtol,atol,
     +    	y_min,y_max,inf_nrms,rdata(r_con),
     >		rdata(r_lte),num_rpar,rpar,num_ipar,ipar,
     +    	pderv,resid,call_back,monitor,
     >      caller_id,nvar,nz,lblk,dblk,ublk,decsolblk, 
     >      need_free,lrd,rpar_decsol,lid,ipar_decsol, 
     >		idata(i_nqused),idata(i_nstep),idata(i_nfail),
     >		idata(i_nre),idata(i_nje),idata(i_ndec),idata(i_nbsol),
     +    	idata(i_npset),idata(i_ncoset),
     +      idata(i_nsteps_prevJ),idata(i_nbsols_prevJ),
     >      idata(i_step_J),idata(i_sol_J),
     >      idata(i_maxord),idata(i_sum_nq),
     >		idata(i_sum_nq2),maxstp,uround,epsjac,rdata(r_rcond),
     >	  	rdata(r_hused),hstpsz,jdebug,stage2_min,ierr,verbosity,
     >		n_isaves_stiff,stiff_integer_saves,
     >		n_dsaves_stiff,stiff_double_saves)
			
	      kgo = 1 - kflag			
			! kgo  a completion flag with the following meanings..
			!    1  the step was successful
			!    2  the requested error could not be achieved
			!          with abs(h) = hmin.
			!    3  the requested error is smaller than can
			!          be handled for this problem.
			!    4  corrector convergence could not be
			!          achieved for abs(h)=hmin.
			
			if (ierr == 0) then
				call call_back(n,t,h,rdata(r_con),y1,yprime,tout,tend,caller_hmax,lout,verbosity,
     >                  num_rdata,num_idata,rdata,idata,maxder,maxstp,equil,jdebug,
     >					   itol,num_tol,rtol,atol,y_min,y_max,inf_nrms,
     >                  num_rpar,rpar,num_ipar,ipar,idid,kgo,ierr)
			end if
			
			if (ierr /= 0) then
            call repack
     			return
			endif
			
			select case (kgo)
			case (1)
				! ---------------------------------------------------------------------
				!  normal return from the integrator.
				!
				!  the weights ymax(i) are updated if itol=1.
				!  if different values are desired, they should be set here.
				!
				!  any other tests or calculations that are required after every
				!  step should be inserted here.
				!
				!  if idid = 3, y0 is set to the current y values on return.
				!  if idid = 2, h is controlled to hit tout (within roundoff
				!  error), and then the current y values are put in y0 on return.
				!  for any other value of idid, control returns to the integrator
				!  unless tout has been reached.  then interpolated values of y are
				!  computed and stored in y0 on return.
				!  if interpolation is not desired, the call to interp should be
				!  removed and control transferred to statement 500 instead of 520.
				! --------------------------------------------------------------------
		      if (idata(i_nstep) > maxstp) then
		         kflag=4 ! too much work
		         if (lout > 0) write(lout,9130)
		         idid = -6
		         exit do_step
		      end if

		      if(itol == 1) then
		         ymax(1:n) = dmax1(ymax(1:n),dabs(y(1:n,1)))
		      endif
		
		      if (idid == 1) exit do_step

		      if (dabs(t-tout) <= dabs(15d0*uround*tout)) then
					! effectively we have hit tout
		         idid = kflag
		         t0 = tout
		         y0(1:n) = y(1:n,1)
		         h0 = h
	            call repack
	     			return
		      end if
		
		      if (idid == 3) exit do_step

		      if (idid == 2) then
					! continuing integration but must hit tout exactly
		         if (((t+h)-tout)*h > 0d0) then
						! we have already overshot the output point or we will do
						! so on the next step
		            if (((t-tout)*h >= 0d0)  .or.  (dabs(t-tout) <= 
     +           		100d0*uround*hmax)) then
							call overshot(kflag)
	     					return
		            else
							! will pass tout on next step with current stepsize
							! so reduce stepsize to hit tout 'exactly'
		               h = (tout-t)* (1d0-4d0*uround)
		               jstart = -1
		            end if

		         end if

		      else if ((t-tout)*h >= 0d0) then
					call overshot(kflag)
	     			return
		      end if
      
		      cycle do_step

			case (2)
				! -------------------------------------------------------------------
				!  h and hmin are reduced by a factor of .1 up to 10 times
				!  before giving up.
				! --------------------------------------------------------------------
		      if (nhcut == 10) then ! have reduced h ten times
		         write (lout,9100)
		         exit do_step
		      end if
      
		      nhcut = nhcut + 1
		      hmin = 0.1d0*hmin
		      h = 0.1d0*h
		      jstart = -1
		      cycle do_step

			case (3)
				! error requirement smaller than can be handled for this problem
	         write (lout,9010) t,h
	         exit do_step

			case (4)
				! could not achieve convergence with hmin
	         write (lout,9030) t
	         exit do_step

			case default			
	         write (lout,9039) t
	         exit do_step
	      end select

      
      	end do do_step

      
      	if(dabs(t-tout) <= 1000d0*uround) then
				call overshot(kflag)
	         return
	      end if
	
         y0(1:n) = y(1:n,1)
         t0 = t
	      h0 = h
	      if(kflag /= 0) idid = kflag

			call repack
			
   		if (need_free) then
   		   nullify(brhs)
            call decsolblk(
     &         2,caller_id,nvar,nz,lblk,dblk,ublk,brhs,ipiv,
     &         lrd,rpar_decsol,lid,ipar_decsol,ierr)
		   end if

      	return

! -------------------------- end of subroutine ovdriv -----------------
 9000 format (' warning..  t + h = t on next step.')
 9010 format (/,/,' kflag = -2 from integrator at t = ',e16.8,'  h =',
     +       e16.8,/,
     +       '  the requested error is smaller than can be handled',/,/)
 9030 format (/,/,' kflag = -3 from integrator at t = ',e16.8,/,
     +       '  corrector convergence could not be achieved',/)
 9039 format (/,/,' unrecoverable error from integrator at t = ',e16.8,/,/)
 9040 format (/,/,' illegal input.. rtol  <=  0.',/,/)
 9045 format (/,/,' illegal input.. atol  <=  0.',/,/)
 9050 format (/,/,' illegal input.. n  <=  0',/,/)
 9060 format (/,/,' illegal input.. (t0-tout)*h  >=  0.',/,/)
 9070 format (/,/,' illegal input.. idid =',i5,/,/)
 9080 format (/,/,' idid = -1 on input with (t-tout)*h  >=  0.',/,
     +       ' t =',e16.8,'   tout =',e16.8,'   h =',e16.8,/,
     +       ' interpolation was done as on normal return.',/,
     +       ' desired parameter changes were not made.')
 9090 format (/,/,' illegal input.. method flag, mf, = ',i6,/,
     +       '         allowed values are 21 or 22',/)
 9100 format (/,/,' problem appears unsolvable with given input',/,
     +       '         hmin reduced by a factor of 1.0e10',/,/)
 9110 format (/,/,' illegal value for itol',/,/)
 9120 format (/,/,' illegal value for maxder',/,/)
 9130 format (/,/,' number of steps exceeds maximum',/,/)
 9140 format (/,/,'bad input for number of variables of index 1,2,3'
     +,/,/)
 9160 format (/,/,'stepsize is too small')

		contains
		
#ifdef offload
      !dir$ attributes offload: mic :: overshot
#endif
			subroutine overshot(flg)
				integer, intent(IN) :: flg
            call interp(n,jstart,h,t,y1,tout,y0)
            idid = flg
            t0 = tout
            h0 = h
            call repack			
			end subroutine overshot
		
#ifdef offload
      !dir$ attributes offload: mic :: unpack
#endif
			subroutine unpack
				kflag = ovdriv_integer_saves(1)
				jstart = ovdriv_integer_saves(2)
				nhcut = ovdriv_integer_saves(3)			
				t = ovdriv_double_saves(1)
				h = ovdriv_double_saves(2)
				hmin = ovdriv_double_saves(3)
				hmax = ovdriv_double_saves(4)			
			end subroutine unpack

#ifdef offload
      !dir$ attributes offload: mic :: repack
#endif
			subroutine repack
				ovdriv_integer_saves(1) = kflag
				ovdriv_integer_saves(2) = jstart	
				ovdriv_integer_saves(3) = nhcut
				ovdriv_double_saves(1) = t
				ovdriv_double_saves(2) = h
				ovdriv_double_saves(3) = hmin
				ovdriv_double_saves(4) = hmax	
			end subroutine repack


      end subroutine ovdriv

c----------------------------------------------------------------------------


      subroutine stiff(h,hmax,hmin,jstart,kflag,mf,mbnd,sfmin,
     >		nind1,nind2,nind3,t,tout,tend,y1,yprime,n,
     >		ymax,error,save1,save2,scale,
     +    	pw,pw1,pw_R,pw_C,equil,equed_flag,
     >		yhold1,ynhold1,arh,ipiv,lout,maxder,
     >      itol,num_tol,rtol,atol,y_min,y_max,inf_nrms,con,
     >		lte,num_rpar,rpar,num_ipar,ipar,
     >		pderv,resid,call_back,monitor,
     >      caller_id,nvar,nz,lblk,dblk,ublk,decsolblk, 
     >      need_free,lrd,rpar_decsol,lid,ipar_decsol, 
     >		nqused,nstep,nfail,nre,nje,ndec,nbsol,npset,ncoset,
     >      nsteps_prevJ,nbsols_prevJ,step_J,sol_J,
     >		maxord,sum_nq,sum_nq2,maxstp,uround,epsjac,rcond,
     >		hused,hstpsz,jdebug,stage2_min,ierr,verbosity,
     >		n_isaves_stiff,stiff_integer_saves,
     >		n_dsaves_stiff,stiff_double_saves)      

		use mebdfi_support
	
		integer, intent(in) :: n ! the number of equations.
		integer, intent(in) :: nind1 ! number of index 1 variables
		integer, intent(in) :: nind2 ! number of index 2 variables
		integer, intent(in) :: nind3 ! number of index 3 variables
		integer, intent(in) :: itol ! indicator of the type of error control
		integer, intent(in) :: num_tol ! length of atol and rtol
		integer, intent(in) :: mbnd(4) ! info about band size for banded matrix
		integer, intent(in) :: mf ! the method flag.  (21,22,23 or 24 at present)
		integer, intent(in) :: maxder 
			! the maximum order allowed for the solver is maxder+1
		integer, intent(in) :: lout ! logical output unit
		integer, intent(in) :: maxstp ! the maximum allowed number of steps
		integer, intent(in) :: jdebug ! if 1, pass numerical jacobians to pderv
		integer, intent(in) :: stage2_min ! min number of newton iterations for stage2
		integer, intent(in) :: num_ipar ! length of ipar array
		integer, intent(in) :: num_rpar ! length of rpar array
		integer, intent(in) :: verbosity ! 0 means no extra output
		
		integer, intent(inout) :: equil ! nonzero means equilibrate the matrix		
		integer, intent(inout) :: equed_flag ! info about state of the matrix		
		integer, intent(inout) :: jstart
			! on entry,
			! 		= 0  perform the first step
			! 		> 0  take a new step continuing from the last
			! 		< 0  take the next step with new parameters
			! on exit, the order of the method last used
		integer, intent(inout) :: nqused ! the last order successfully used
		integer, intent(inout) :: nstep ! the number of successful steps taken so far
		integer, intent(inout) :: nfail ! the number of failed attempts
		integer, intent(inout) :: nre ! the number of residual evaluations
		integer, intent(inout) :: nje ! the number of jacobian evaluations
		integer, intent(inout) :: ndec ! the number of lu matrix decompositions
		integer, intent(inout) :: nbsol ! the number of 'backsolves' 
		integer, intent(inout) :: npset ! number of times a new coefficient matrix has been formed
		integer, intent(inout) :: ncoset
			! the number of times the order of the method used has been changed
      integer, intent(inout) :: nsteps_prevJ ! number of steps done with the previous jacobian
      integer, intent(inout) :: nbsols_prevJ ! number of backsolves done with the previous jacobian
      integer, intent(inout) :: step_J ! step number (nstep) when the current jacobian was created
      integer, intent(inout) :: sol_J ! solve number (nbsol) when the current jacobian was created
		integer, intent(inout) :: maxord ! the maximum order used
		integer, intent(inout) :: sum_nq ! the sum of nqused values so far
		integer, intent(inout) :: sum_nq2 ! the sum of nqused^2 so far
		integer, intent(in) :: n_isaves_stiff, n_dsaves_stiff
		integer, intent(inout) :: stiff_integer_saves(n_isaves_stiff)
		
		integer, intent(out) :: ierr ! error flag (0 means AOK)
		integer, intent(out), pointer :: ipiv(:) ! (n) ! used for pivot information.
		integer, intent(out) :: kflag
			! a completion flag with the following meanings..
			!     0  the step was successful
			!    -1  the requested error could not be achieved
			!          with abs(h) = hmin.
			!    -2  the requested error is smaller than can
			!          be handled for this problem.
			!    -3  corrector convergence could not be
			!          achieved for abs(h)=hmin.
			!    -4  unrecoverable error in the algorithm.
			!  on a return with kflag negative, the values of t and
			!  the y array are as at the beginning of the last
			!  step attempted, and h is the last step size attempted.
	
     	real(dp), intent(in) :: sfmin, y_min, y_max
     	real(dp), intent(in) :: tend ! end of the range of integration
		real(dp), intent(in) :: uround ! the unit roundoff = dlamch('epsilon')
		real(dp), intent(in) :: epsjac ! sqrt(uround)
		real(dp), intent(in), pointer :: rtol(:) ! relative error tolerances
		real(dp), intent(in), pointer :: atol(:) ! absolute error tolerances
			! rtol,atol  the error bounds. see description in ovdriv.
		real(dp), intent(in) :: inf_nrms(4) ! infinity norms
		
		real(dp), intent(inout) :: lte ! estimate of local truncation error
		real(dp), intent(inout) :: rcond ! reciprocal of matrix condition number
	   integer, pointer :: ipar(:) ! (num_ipar)
	   real(dp), pointer :: rpar(:) ! (num_rpar)
		real(dp), intent(inout) :: stiff_double_saves(n_dsaves_stiff)
		
     	real(dp), intent(inout) :: h 
			! the stepsize to be attempted on the next step.
			! h is altered by the error control algorithm during
			! the problem.  h can be either positive or negative but
			! its sign must remain constant throughout the problem.
     	real(dp), intent(in) :: hmax ! max absolute value of the stepsize
     	real(dp), intent(in) :: hmin ! min absolute value of the stepsize
			! these may be changed at any time 
			! but will not take effect until the next h change.
     	real(dp), intent(out) :: hused
			! the last stepsize successfully used
     	real(dp), intent(out) :: hstpsz(2,14) ! history of step sizes
     	real(dp), intent(out), pointer :: arh(:) ! (n)

			! the following comments reflect the use of save1, save2, and error by itrat2			
     	real(dp), intent(out), pointer :: save1(:) ! (n) ! the modified value of y during iteration
     	real(dp), intent(out), pointer :: save2(:) ! (n) ! incremental change to save1 as iterate
     	real(dp), intent(out), pointer :: error(:) ! (n) ! save1 equals y + error; error = sum of save2's

     	real(dp), pointer :: pw(:)
			! a block of locations used for factored matrix of partial derivatives
     	real(dp), pointer :: pw_R(:), pw_C(:) ! (n) ! row and column equilibration info

     	real(dp), pointer :: pw1(:)
			! a block of locations used for unfactored matrix of partial derivatives

     	real(dp), intent(out), pointer :: yprime(:) ! (n)
			! On entry, the initial values of y' = dy/dt.
			! On exit, the new values of y' at the new t = t0
     	real(dp), intent(out), pointer :: y1(:) ! (n*12)
			! y contains the dependent variables and their backward differences.
			! y(i,1) contains the current value
			! y(i,j+1) contains the jth backward difference
     	real(dp), intent(out), pointer :: yhold1(:) ! (n*12)
			! backup storage for history array
     	real(dp), intent(out), pointer :: ymax(:) ! (n)
			! the estimated local errors in y are sometimes compared to ymax
     	real(dp), intent(out), pointer :: scale(:) ! (n) ! scalings for calculating norms
     	real(dp), intent(out) :: con
			! the most recently used qi.
     	real(dp), intent(out) :: t
			! the independent variable. t is updated on each step taken.
     	real(dp), intent(in) :: tout
			! value of t at which output is desired next
     	real(dp), intent(out), pointer :: ynhold1(:) ! (n*2)
			
		interface		
			include "mebdfi_call_back.dek"
			include "mebdfi_resid.dek"
			include "mebdfi_pderv.dek"
			include "mebdfi_monitor.dek"		
		end interface

      ! bcyclic
      integer, intent(in) :: caller_id, nvar, nz, lrd, lid
      logical, intent(inout) :: need_free
         ! nvar > 0 means use decsolblk
      real(dp), dimension(:), pointer, intent(inout) :: 
     >      lblk, dblk, ublk ! =(nvar,nvar,nz)
		interface
         include "mtx_decsolblk.dek"
		end interface
      real(dp), pointer, intent(inout) :: rpar_decsol(:) ! (lrd)
      integer, pointer, intent(inout) :: ipar_decsol(:) ! (lid)
		
		! SAVED
		integer :: miter ! remainder of mf div 10
		integer :: mfold ! old value of mf
		integer :: lmax ! max allowed order
		integer :: jsinup
				! set to -1 when convergence fails; incremented when it works; 
				! used as part of test of whether we're running into too much trouble
		integer :: jsnold
				! keeps track of number of steps taken with
				! present iteration matrix (be it formed by
				! a new j or not).
		integer :: isamp ! counts steps until need to force 2 newton iterations for stage1 
		integer :: jchang ! 1 if we have just changed step-size
				! if 1, then will not use the previous stage2 result for stage1 prediction
		integer :: kchang ! 1 if we have just changed order
				! if 1, then will not change time-step
		integer :: l ! number of terms in the history array
		integer :: nq ! the current order
		integer :: meqc1 ! running total of number of newton iterations for stage1
		integer :: meqc2 ! running total of number of newton iterations for stage2
		integer :: mq1tmp ! the value of meqc1 when the last jacobian was created
		integer :: mq2tmp ! the value of meqc2 when the last jacobian was created
		integer :: iemb ! =1 means failed because of embedded error estimate
		integer :: iweval
				! 0 means we've just done a check on the convergence
				! 1 means are using an old jacobian
				! initialized to miter when change mf and want partial updated
				! set to miter when abs(rc-1)>upbnd and want partial updated
				! set to miter when change parameters
		integer :: ier
		integer :: kfail ! counts failures in the error test
		integer :: idoub ! counts down steps until is okay to change timestep
		integer :: newpar 
				! =1 means that a jacobian has been evaluated for the current step

		! SAVED
      logical :: cfail ! convergence failed -- need a new jacobian
      logical :: jnewim
			! indicates if present iteration matrix was formed using a new j or old j.
      integer :: stage1_min ! min number newton iterations in stage1

		! SAVED
      real(dp) :: el(10) ! bdf coefficients
      real(dp) :: elst(10) ! mebdf coefficients
      real(dp) :: tq(5) ! truncation error coefficients
      real(dp) :: hold ! old stepsize
      real(dp) :: e_present ! error associated with present order
      real(dp) :: edn ! error associated with method of order one lower
      real(dp) :: eup ! error associated with higher order method
      real(dp) :: bnd ! error bound used to test for convergence of newton iterates
      real(dp) :: eddn ! error associated with method of order two lower
      real(dp) :: qi ! qi = h*el(1), for use in calculating yprime = (y - arh)/qi
			! set in first_step_setup
			! set in predict_stage1
			! used for calculating yprime in predict_stage1
			! passed to pset in try_stage1
			! stored in qqq in try_stage1 after calling pset
      real(dp) :: qqq ! value of qi when last called pset
			! used in try_stage3 for calculating yprime
			! used in check_converged_solution for calculating yprime
			! passed to itrat2 for calculating yprime during newton iterations
      real(dp) :: rc
			! rc is the ratio of the new to old values of the coefficient
			! h*el(1). when rc differs from 1 by more than 20 percent, iweval is
			! set to miter to force the partials to be updated
      real(dp) :: rh ! ratio of the new step size to the old step size
      real(dp) :: rmax ! maximum ratio by which h can be increased in a single step
      real(dp) :: told ! what t was at start of this timestep; used for backup
      real(dp) :: crate1 ! estimate of rate of convergence for stage1
      real(dp) :: crate2 ! estimate of rate of convergence for stage2
      real(dp) :: tcrat1 ! sum of recent crate1's used in deciding when to redo J
      real(dp) :: tcrat2 ! sum of recent crate2's -- not currently used
      real(dp) :: avnewj
			! avnewj stores value for average crate when iteration (using tcrat1)
			! matrix was formed by a new j.
      real(dp) :: avoldj 
			! avoldj stores value for average crate when iteration
			! matrix was formed by an old j.
      real(dp) :: upbnd ! upper bound on abs(rc-1.0) for forcing partials to be updated 
      real(dp) :: oldlo ! old low order coeff el(1)
      real(dp) :: lte_adj ! value of lte at last adjustment of timestep
      real(dp) :: rh_adj ! value of rh at last adjustment of timestep


		! NOT SAVED
      logical :: finish
			! if true, then reduce_time_step will pack_it_up
		real(dp) :: norm
			! set at end of try_stage1
			! used by adjust after a failure in try_stage1
	   integer :: ijus 
			! if (ijus == 0) in reduce_time_step, then call hchose
			! set to 0 if fail in try_stage1
			! set to 0 if fail in try_stage2
			! set to 1 if fail in try_stage3
			! set to 1 in check_converged_solution if fail embedded error check
			! set to 1 in try_stage1 if the matrix cannot be factored
	
		integer, parameter :: dfsc_start = 75 
		! let it run for this many steps before using digital filter timestep control

		real(dp), parameter :: one = 1d0, zero = 0d0, red = 0.5d0
		integer, parameter :: ibnd = 5
		
		integer, parameter ::
     >			have_new_params = 1,
     >			have_new_timestep = 2,
     >			need_adjustments = 3,
     >			need_prediction = 4,
     >			need_stage1 = 5,
     >			need_stage2 = 6,
     >			need_stage3 = 7,
     >			stage1_failed = 8,
     >			need_reduced_dt = 9,
     >			need_to_retry = 10,
     >			pack_it_up = 11

     	real(dp), pointer :: y(:,:) ! (n,12)
     	real(dp), pointer :: yhold(:,:) ! (n,12)
     	real(dp), pointer :: ynhold(:,:) ! (n,2)

		integer :: state
		state = maxstp ! to keep compiler quiet
		
		y(1:n,1:12) => y1(1:n*12)
		yhold(1:n,1:12) => yhold1(1:n*12)
		ynhold(1:n,1:2) => ynhold1(1:n*2)
      
		call unpack	
		state = start_new_step()
		
		step_loop: do
		
			select case (state)
				case (need_prediction);       state = predict_stage1()
				case (need_stage1);           state = try_stage1()
				case (stage1_failed);         state = after_a_failure()
				case (need_stage2);           state = try_stage2()
				case (need_stage3);           state = try_stage3()
				case (have_new_params);       state = setup_new_params()
				case (have_new_timestep);     state = setup_for_new_dt()
				case (need_adjustments);      state = make_adjustments()
				case (need_reduced_dt); 		state = reduce_time_step()
				case (need_to_retry);         state = prepare_to_retry()
				case (pack_it_up);            exit step_loop
				case default;         			state = unknown_state()
			end select	
		
		end do step_loop

      call repack
      con = qi
		return

		contains
			
#ifdef offload
      !dir$ attributes offload: mic :: unpack
#endif
		subroutine unpack
		
			el(1:10) = stiff_double_saves(1:10)
			elst(1:10) = stiff_double_saves(11:20)
			tq(1:5) = stiff_double_saves(21:25)
			hold = stiff_double_saves(26)

			edn = stiff_double_saves(28)
			eup = stiff_double_saves(29)
			bnd = stiff_double_saves(30)
			eddn = stiff_double_saves(31)
			e_present = stiff_double_saves(32)
			lte_adj = stiff_double_saves(33)
			rh_adj = stiff_double_saves(34)
			qi = stiff_double_saves(35)
			qqq = stiff_double_saves(36)
			rc = stiff_double_saves(37)

			rh = stiff_double_saves(40)
			rmax = stiff_double_saves(41)
			told = stiff_double_saves(42)
			crate1 = stiff_double_saves(43)
			crate2 = stiff_double_saves(44)
			tcrat1 = stiff_double_saves(45)
			tcrat2 = stiff_double_saves(46)
			avnewj = stiff_double_saves(47)
			avoldj = stiff_double_saves(48)
			upbnd = stiff_double_saves(49)
			oldlo = stiff_double_saves(52)

			mfold = stiff_integer_saves(1)
			lmax = stiff_integer_saves(2)
			jsinup = stiff_integer_saves(3)
			jsnold = stiff_integer_saves(4)
			kchang = stiff_integer_saves(5)
			isamp = stiff_integer_saves(6)
			jchang = stiff_integer_saves(7)
			l = stiff_integer_saves(8)
			nq = stiff_integer_saves(9)
			meqc1 = stiff_integer_saves(10)
			miter = stiff_integer_saves(11)

			iemb = stiff_integer_saves(13)
			iweval = stiff_integer_saves(14)
			ier = stiff_integer_saves(15)
			kfail = stiff_integer_saves(16)		
			idoub = stiff_integer_saves(17)
			cfail = (stiff_integer_saves(18)  ==  1)
			jnewim = (stiff_integer_saves(19)  ==  1)
			stage1_min = stiff_integer_saves(20)
			meqc2 = stiff_integer_saves(21)
			mq1tmp = stiff_integer_saves(22)
			mq2tmp = stiff_integer_saves(23)
			newpar = stiff_integer_saves(24)
		
		end subroutine unpack


#ifdef offload
      !dir$ attributes offload: mic :: repack
#endif
		subroutine repack
			stiff_integer_saves(1) = mfold
			stiff_integer_saves(2) = lmax
			stiff_integer_saves(3) = jsinup
			stiff_integer_saves(4) = jsnold
			stiff_integer_saves(5) = kchang
			stiff_integer_saves(6) = isamp
			stiff_integer_saves(7) = jchang
			stiff_integer_saves(8) = l
			stiff_integer_saves(9) = nq
			stiff_integer_saves(10) = meqc1
			stiff_integer_saves(11) = miter

			stiff_integer_saves(13) = iemb
			stiff_integer_saves(14) = iweval
			stiff_integer_saves(15) = ier
			stiff_integer_saves(16) = kfail
			stiff_integer_saves(17) = idoub
			if (cfail) then
				stiff_integer_saves(18) = 1
			else
				stiff_integer_saves(18) = 0
			end if
			if (jnewim) then
				stiff_integer_saves(19) = 1
			else
				stiff_integer_saves(19) = 0
			end if
			stiff_integer_saves(20) = stage1_min
			stiff_integer_saves(21) = meqc2
			stiff_integer_saves(22) = mq1tmp
			stiff_integer_saves(23) = mq2tmp
			stiff_integer_saves(24) = newpar

			stiff_double_saves(1:10) = el(1:10)
			stiff_double_saves(11:20) = elst(1:10) 
			stiff_double_saves(21:25) = tq(1:5)
			stiff_double_saves(26) = hold

			stiff_double_saves(28) = edn
			stiff_double_saves(29) = eup
			stiff_double_saves(30) = bnd
			stiff_double_saves(31) = eddn
			stiff_double_saves(32) = e_present
			stiff_double_saves(33) = lte_adj
			stiff_double_saves(34) = rh_adj
			stiff_double_saves(35) = qi
			stiff_double_saves(36) = qqq
			stiff_double_saves(37) = rc

			stiff_double_saves(40) = rh
			stiff_double_saves(41) = rmax
			stiff_double_saves(42) = told
			stiff_double_saves(43) = crate1
			stiff_double_saves(44) = crate2
			stiff_double_saves(45) = tcrat1
			stiff_double_saves(46) = tcrat2
			stiff_double_saves(47) = avnewj
			stiff_double_saves(48) = avoldj
			stiff_double_saves(49) = upbnd
			stiff_double_saves(52) = oldlo
		
		end subroutine repack

		
#ifdef offload
      !dir$ attributes offload: mic :: setup_new_params
#endif
		integer function setup_new_params()
		
			integer :: meth
			
			call cpyary(n*l,yhold,y)
			
	      if (mf /= mfold) then
	         meth = mf/10
	         miter = mf - 10*meth
	         mfold = mf
	         iweval = miter
	      end if
     
	      if (nstep <= 0) then
		      nje = 0
		      nre = 1
		      cfail = .true.
		      newpar = 0
		      mq1tmp = 0
		      mq2tmp = 0
		      meqc1 = 0
		      meqc2 = 0
		      tcrat1 = 0d0
		      tcrat2 = 0d0
		      crate1 = 1.0d-1
		      crate2 = 1.0d-1
		      nstep = 0
		      nbsol = 0
		      npset = 0
		      ncoset = 0
		      ndec = 0
				sum_nq = 0
				sum_nq2 = 0
	      end if
	
	      if (h /= hold) then
	         rh = h/hold
	         h = hold
				setup_new_params = need_adjustments
	      else
	         setup_new_params = need_prediction
	      end if
	
		end function setup_new_params
		
		
#ifdef offload
      !dir$ attributes offload: mic :: setup_for_new_dt
#endif
		integer function setup_for_new_dt()
		
			rh = dmax1(rh,hmin/dabs(h))
			setup_for_new_dt = need_adjustments
			
		end function setup_for_new_dt

	
#ifdef offload
      !dir$ attributes offload: mic :: make_adjustments
#endif
		integer function make_adjustments()
		
			rh = dmin1(rh,hmax/dabs(h),rmax)
	      call rscale(n,l,rh,y1)
	      rmax = 10d0
	      jchang = 1
	      h = h*rh
	      rc = rc*rh
	      if (jsnold > ibnd) then
	         cfail = .true.
	         newpar = 0
	         rc = zero
				! ********************************************************************
				! cfail=true and newpar=0 should force a new j to be evaluated
				! after 7 steps with an old j, if we have had a failure of any
				! kind on the first, second or third stage of the current step
				! ********************************************************************
	      end if
     
	      idoub = l + 1
	      call cpyary(n*l,y,yhold) ! save copy of y in yhold
			
			make_adjustments = need_prediction
	
		end function make_adjustments
		
		
#ifdef offload
      !dir$ attributes offload: mic :: first_step_setup
#endif
		subroutine first_step_setup
		
			real(dp) :: vtol
		
	      y(1:n,2) = h*yprime(1:n)
	      miter = mf - 20
	      upbnd=0.2d0
	      if (mf > 22) upbnd=0.1d0
	      nq = 1
	      nqused = nq
	      l = 2
	      idoub = 3
	      kfail = 0
	      rmax = 10000d0
	      ier=0
	      rc = zero
	      crate1 = 0.1d0
	      crate2 = 0.1d0
	      jsnold = 0 ! no steps taken yet with current matrix
	      jnewim = .true.
	      tcrat1 = zero
	      tcrat2 = zero
	      vtol=dmax1(rtol(1),atol(1))/10d0
	      hstpsz(1,1:12)=1d0
	      hstpsz(2,1:12)=vtol
	      hold = h
	      mfold = mf
	      nstep = 0
	      nre = 1
	      nje = 0
	      ndec = 0
	      npset = 0
	      ncoset = 0
         nsteps_prevJ = 0
         nbsols_prevJ = 0
         step_J = 0
         sol_J = 0
	      maxord = 1
	      nfail = 0
	      cfail = .true.
	      avnewj = zero
	      avoldj = zero
	      stage1_min = 1
	      isamp = 0
	      iemb=0
			! **************************************************
			! cfail=.true. ensures that we calculate a new
			! j on the first call.
			! **************************************************
	      meqc1 = 0
	      meqc2 = 0
	      mq1tmp = 0
	      mq2tmp = 0
	      nbsol = 0
	      hused = h
			! -----------------------------------------------------------------
			! if the caller has changed n , the constants e, edn, eup
			! and bnd must be reset.  e_present is a comparison for errors at the
			! current order nq.  eup is to test for increasing the order,
			! edn for decreasing the order.  bnd is used to test for convergence
			! of the corrector iterates.   if the caller has changed h, y must
			! be re-scaled.  if h is changed, idoub is set to l+1 to prevent
			! further changes in h for that many steps.
			! -----------------------------------------------------------------
	      call coset(nq,el,elst,tq,ncoset,maxord)
	      lmax = maxder + 1
	      rc = rc*el(1)/oldlo
	      oldlo = el(1)
	      iweval = miter
	      newpar = 0
			! *****************************************************
			! newpar is to indicate that
			! we wish a new j to be calculated for this step.
			! *****************************************************
	      call errors(n,tq,edn,e_present,eup,bnd,eddn)
	      arh(1:n) = el(2)*y(1:n,1)
	      call cpyary(n*l,y,yhold)
	      qi = h*el(1)
	
		end subroutine first_step_setup


#ifdef offload
      !dir$ attributes offload: mic :: predict_stage1
#endif
		integer function predict_stage1()
		
			integer :: j1, jp1
		
			! *********************************************
			! computes the predicted values of y
			! and the rhs, arh, for use in the newton iteration scheme.
			! rc is the ratio of the new to old values of the coefficient
			! h*el(1). when rc differs from 1 by more than 20 percent, iweval is
			! set to miter to force the partials to be updated.
			! *********************************************
			if (verbosity > 0 .and. lout > 0) write(lout,*) 'predict_stage1'
			if (dabs(rc-one) > upbnd) iweval = miter
	      hused = h
	      qi = h*el(1)
	      arh(1:n) = el(2)*y(1:n,1)
	      do j1 = 2,nq
	         jp1 =j1+1
	         arh(1:n) = arh(1:n) + el(jp1)*y(1:n,j1)
	      end do
	      if (jchang == 1) then
				! if we have changed stepsize then predict a value for y(t+h)
				! and evaluate the derivative there (stored in save2())
	         call prdict(t,h,y,l,n) ! prdict does t = t + h
	         yprime(1:n)=(y(1:n,1)-arh(1:n))/qi
	      else
				! use the values computed for the second bdf from the last step.
	         y(1:n,1)=y(1:n,lmax+3)                      
	         yprime(1:n) = (y(1:n,1)-arh(1:n))/qi
	         t = t + h
	      end if
	
			predict_stage1 = need_stage1
			
		end function predict_stage1

		
#ifdef offload
      !dir$ attributes offload: mic :: prepare_to_retry
#endif
		integer function prepare_to_retry()
			if (verbosity > 0 .and. lout > 0) write(lout,*) 'prepare_to_retry'
		
	      finish = .false.
	      t=told
	      rmax=2d0
			! restore y from yhold
			call cpyary(n*l,yhold,y)
	      if (dabs(h) <= hmin*1.00001d0) then
				! corrector convergence could not be achieved
	         if (nstep == 0) then
	            kflag=-1
	         else
	            kflag=-3
	         end if
				! to suppress error messages at start as h may
				! have been too large on the first step.
	         hold = h
	         finish = .true.
            if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >            '(dabs(h) <= hmin*1.00001d0)  h', h, 'hmin', hmin
	      end if
	      rh = red
			prepare_to_retry = need_reduced_dt
	
		end function prepare_to_retry
			

#ifdef offload
      !dir$ attributes offload: mic :: reduce_time_step
#endif
		integer function reduce_time_step()
		
			real(dp), target :: y0_target(n)
			real(dp), pointer :: y0(:)
      	logical :: ovride
      	
      	y0 => y0_target
			
			if (verbosity > 0 .and. lout > 0) write(lout,*) 'reduce_time_step'
			if (ierr /= 0) then
	         h= h/2         
	         if (h < epsjac/100d0) then
	            reduce_time_step = pack_it_up
               if (verbosity > 0 .and. lout > 0) write(lout,*) 'pack_it_up (h < epsjac/100d0)'
					return
	         endif
	         t = told ! restore t to its value at start of this timestep
	         if ((t-tout)*h >= 0d0) then ! have overshot tout
	            call interp(n,jstart,h,t,y1,tout,y0)				
               if (verbosity > 0 .and. lout > 0) 
     >               write(lout,*) 'pack_it_up (have overshot tout)'
	            reduce_time_step = pack_it_up
					return
	         endif
	         ierr = 0
	         jstart = -1
            if (verbosity > 0 .and. lout > 0) write(lout,*) 'have_new_params'
				reduce_time_step = have_new_params
				return
	      endif
	
         if (ijus == 0) then
            call hchose(rh,h,hstpsz,ovride)
            rh = dmax1(rh,1d-1) ! don't reduce timestep by too much
         end if
	
	      if (finish) then
            if (verbosity > 0 .and. lout > 0) write(lout,*) 'pack_it_up (finish)'
				reduce_time_step = pack_it_up
	      else
            if (verbosity > 0 .and. lout > 0) write(lout,*) 'have_new_timestep', h
				reduce_time_step = have_new_timestep
	      end if

		end function reduce_time_step
		
		
#ifdef offload
      !dir$ attributes offload: mic :: back_to_order1
#endif
		subroutine back_to_order1
		
      	logical :: ovride
			if (verbosity > 0 .and. lout > 0) write(lout,*) 'back_to_order1'

	      jchang = 1
	      rh = dmax1(hmin/dabs(h),0.1d0)
	      call hchose(rh,h,hstpsz,ovride)
	      h = h*rh
         y(1:n,1) = yhold(1:n,1)
         y(1:n,2) = yhold(1:n,2)
	      iweval = miter
	      cfail = .true.
			! since we have had problems proceed with this order
			! for 10 steps (if we can)
	      idoub = 10
	      if (nq == 1) return
	      nq = 1
	      l = 2
			! reset order, recalculate error bounds
	      call coset(nq,el,elst,tq,ncoset,maxord)
	      lmax = maxder + 1
	      rc = rc*el(1)/oldlo
	      oldlo = el(1)
	      call errors(n,tq,edn,e_present,eup,bnd,eddn)
		end subroutine back_to_order1
		
		
#ifdef offload
      !dir$ attributes offload: mic :: get_new_order_and_dt
#endif
		subroutine get_new_order_and_dt(newrh, newnq)
			real(dp), intent(OUT) :: newrh
			integer, intent(OUT) :: newnq

			real(dp) :: pr3, pr2, pr1, pr0, npr, fac, ddown, dddown, dup, d0
			! ********************************************************************
			! compute the factors pr1, pr2 and pr3, by which
			! h could be divided at order nq-1, order nq and order nq+1
			! respectively. the smallest of these is determined and the new
			! order chosen accordingly. if the order is to be increased we
			! must compute one more backward difference.
			! ********************************************************************
			if (verbosity > 0 .and. lout > 0) write(lout,*) 'get_new_order_and_dt'
			dup = 0d0
         fac = 1.5d0
         if (iemb == 1) fac = 1.8d0
			call set_scale(y1, n, itol, scale, ymax, 
     >				rtol, atol, nind1, nind2, nind3, hused)

         pr3 = 1d20
         if (l /= lmax) then
            dup = sum(((y(1:n,l+1)-y(1:n,lmax+4))/scale(1:n))**2)
            if (itol == 1) dup = dup/(rtol(1)**2)
            pr3 = pow_cr(dup/eup,0.5d0/dble(l+1))*(fac+0.2d0) + 1.8d-6
         end if
        
         d0 = sum((y(1:n,l+1)/scale(1:n))**2) ! weighted norm of term for order l+1 = nq
         if (itol == 1) d0 = d0/(rtol(1)**2)
         pr2 = pow_cr(d0/e_present,0.5d0/dble(l))*fac + 1.6d-6

         pr1 = 1d20
         if (nq > 1) then
            ddown = sum((y(1:n,l)/scale(1:n))**2) ! weighted norm of term for order l = nq-1
            dddown= sum((y(1:n,l-1)/scale(1:n))**2) ! weighted norm of term for order l-1 = nq-2
            if (itol == 1) ddown = ddown/(rtol(1)**2)
            pr1 = pow_cr(ddown/edn,0.5d0/dble(nq))*(fac+0.1d0) + 1.7d-6
            if (nq > 2) then
               pr0 = pow_cr(dddown/eddn,0.5d0/dble(nq-1))*(fac+0.1d0) + 1.7d-6
               if (pr0 > pr1) pr1 = pr0
               if (dddown < ddown) ddown = dddown
            endif
	         if (dup > d0 .and. d0 > ddown) then
	            pr2=1d30
	            pr3=1d30
				end if
         end if
			
			if (verbosity > 0 .and. lout > 0) then
			   write(lout,*) '   pr1', pr1
			   write(lout,*) '   pr2', pr2
			   write(lout,*) '   pr3', pr3
			end if
         if (pr2 <= pr3) then
            if (pr2 > pr1) then
               newnq = nq - 1
               npr = pr1            
            else
               newnq = nq
               npr = pr2
            end if          
         else if (pr3 < pr1) then ! increase order
            newnq = l
            npr = pr3          
         else
            newnq = nq - 1
            npr = pr1
         end if

         if (nind3 /= 0) then
            if (nq <= 2 .and. pr3 < 1d0) newnq=nq+1
         endif
			
			newrh = 1d0 / npr
		
		end subroutine get_new_order_and_dt
			
			
#ifdef offload
      !dir$ attributes offload: mic :: adjust
#endif
		subroutine adjust
			! returns true if okay to go ahead with timestep adjustment
		
			integer :: newq		
			real(dp) :: newrh
		
      	logical :: ovride
			real(dp), parameter :: dfsc_target = 1d0
			
			logical :: do_order, do_timestep
			
			do_order = (idoub == 0)
			do_timestep = (kchang==0) .and. (do_order)
			
			if (.not. do_order .and. .not. do_timestep) then
				kchang = 0 ! to indicate have taken a step at the current order
				return
			end if

			call get_new_order_and_dt(newrh, newq)			
			
			if (.not. do_order) then
				kchang = 0 ! to indicate have taken a step at the current order
			else
	         stage1_min = 1
	         isamp = isamp + 1
	         if (isamp == 4) then ! it is time to force 2 newton iterations for stage1
	            stage1_min = 2
	            isamp = 0
	         end if         
	         iemb=0
	         if (newrh > 1d0 .and. newrh < 1.1d0) then
	            idoub=10
	            nq=nqused
	            l=nq+1
					kchang = 0 ! to indicate have taken a step at the current order
	            return
	         endif
	         if ((jsinup <= 20) .and. (kflag == 0) .and. (newrh < 1.1d0)) then
					! we have run into problems
	            idoub = 10
	            nq = nqused
	            l = nq + 1
					kchang = 0 ! to indicate have taken a step at the current order
					return
	         end if
				! ********************************************************************
				! if there is a change in order, reset nq, l and the coefficients.
				! ********************************************************************
	         if (newq == nq) then ! changing order
					kchang = 0 ! to indicate have taken a step at the current order
				else
					kchang = 1 ! to indicate a change in order has just happened
	            if (newq > nq) then
						! add an extra term to the history array
	               y(1:n,l+1) = y(1:n,l) - yhold(1:n,l)
	            end if          
	            nq = newq
	            l = nq + 1
					! reset order,recalculate error bounds
	            call coset(nq,el,elst,tq,ncoset,maxord)
	            lmax = maxder + 1
	            rc = rc*el(1)/oldlo
	            oldlo = el(1)
	            call errors(n,tq,edn,e_present,eup,bnd,eddn)
	         end if

	         idoub = l + 1
	
			end if
			
			if (.not. do_timestep) return
			
			! adjust timestep
			
			rh = newrh
			
         rh = dmin1(rh,rmax)
         call hchose(rh,h,hstpsz,ovride)
         rh = dmax1(rh,hmin/dabs(h))
         rh = dmin1(rh,hmax/dabs(h),rmax)

         call rscale(n,l,rh,y1)
         rmax = 10d0
			jchang = 1   ! IF 1, THEN WILL NOT USE PREVIOUS STAGE2 AS PREDICTION FOR NEXT STAGE1			
         h = h*rh
         rc = rc*rh
         if (jsnold > ibnd) rc=zero
			lte_adj = lte
			rh_adj = rh

		end subroutine adjust
		
		
#ifdef offload
      !dir$ attributes offload: mic :: try_stage3
#endif
		integer function try_stage3()
			integer :: m3step, j1, info
			real(dp) :: delst, d0, norminf, recip_pivot_growth_factor

			if (verbosity > 0 .and. lout > 0) write(lout,*) 'try_stage3'
			
         y(1:n,lmax+3) = (save1(1:n)-arh(1:n))
         y(1:n,lmax+2) = y(1:n,lmax+3)*(one/qi)
         y(1:n,lmax+3) = save1(1:n)

			! we are now computing the third stage

	      t = told + h
	      delst = elst(1)-el(1)
	      arh(1:n) = h*(elst(nq+2)*y(1:n,lmax+2)+delst*ynhold(1:n,2))
	      do j1 = 1,nq
	         arh(1:n) = arh(1:n) + elst(j1+1)*yhold(1:n,j1)
	      end do
         save2(1:n) = ynhold(1:n,2)
         y(1:n,1) = ynhold(1:n,1)
	      m3step = 0

			m3step_loop: do
			
		      yprime(1:n)=(y(1:n,1)-arh(1:n))/qqq
		      call resid(n,caller_id,nvar,nz,t,h,qi,y1,save1,yprime,
     >		         num_rpar,rpar,num_ipar,ipar,3,m3step+1,ierr)
		      nre=nre+1
		      if (ierr  /=  0) then
		         ijus = 1
					finish = .false.
					try_stage3 = need_reduced_dt
					if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >					'stage3 failed because of nonzero resid ierr: ', ierr
					return
				end if
		
				info = 0
				call matrix_solver(
     >            caller_id,nvar,nz,lblk,dblk,ublk,decsolblk, 
     >            lrd,rpar_decsol,lid,ipar_decsol,
     >            mf, equil, equed_flag, mbnd, sfmin, 
     >		      y_min, y_max, pw, pw1, pw_R, pw_C, ipiv, n, save1, 
     >            rcond, recip_pivot_growth_factor,
     >            verbosity, lout, info)
		      if (info /= 0) then
		         ijus = 1
					finish = .false.
					try_stage3 = need_reduced_dt
					if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >					'stage3 failed because of nonzero matrix solver info: ', info
					return
				end if
		      nbsol = nbsol + 1
				
				call set_scale(y1, n, itol, scale, ymax, 
     >				rtol, atol, nind1, nind2, nind3, hused)
	
		      d0 = sum((save1(1:n)/scale(1:n))**2)
		      if (itol  ==  1) d0 = d0/(rtol(1)**2)
				
				call monitor(
     >				n,t,h,qi,y1,y1,save1,save1,yprime,scale,d0,crate1,bnd,
     >            pw_R,pw_C,ipiv,rcond,recip_pivot_growth_factor,
     >				num_rpar,rpar,num_ipar,ipar,
     >				nqused,nstep,nbsol,3,m3step+1,ierr)
		      if (ierr /= 0) then
		         ijus = 1
					finish = .false.
					try_stage3 = need_reduced_dt
					if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >					'stage3 failed because of nonzero monitor ierr: ', ierr
	            return
	         end if
		
		      y(1:n,1) = y(1:n,1) - save1(1:n)
		
		      if ((d0*dmin1(1d0,2d0*crate1)) <= bnd) then
		      
		         ! check infinity norm of last correction before accept as converged
      			norminf = maxval(abs(save1(1:n)/scale(1:n)))
	
      	      if (norminf < inf_nrms(3)) then
					   try_stage3 = check_converged_solution()
					   return
					end if
					
				end if
				
		      if (m3step == 4) then ! ****  step 3 fails
		         ijus = 1
		         nfail = nfail + 1
					try_stage3 = need_to_retry
					if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >					'try_stage3 newton iterations failed to converge'
					return
		      end if

		      m3step = m3step + 1

	      end do m3step_loop

		end function try_stage3
		
			
#ifdef offload
      !dir$ attributes offload: mic :: check_converged_solution
#endif
		integer function check_converged_solution()
		
			real(dp) :: trange, demb, norminf
			integer :: j2
			
			if (verbosity > 0 .and. lout > 0) write(lout,*) 'check_converged_solution'
			
	      demb = sum(((y(1:n,1)-ynhold(1:n,1))/scale(1:n))**2)
	      if (demb > 4d0*dble(n)) then
				! * failed because of embedded error estimate
	         iemb=1
	         ijus = 1
	         nfail = nfail + 1
				check_converged_solution = need_to_retry
				if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >				'try_stage3 failed because of embedded error estimate'
	         return
	      endif
	      
	      norminf = maxval(abs((y(1:n,1)-ynhold(1:n,1))/scale(1:n)))
	      if (norminf > inf_nrms(4)) then
				if (verbosity > 0 .and. lout > 0) write(lout,*)
     >				'check_converged_solution norminf too large', norminf, nstep
	         iemb=1
	         ijus = 1
	         nfail = nfail + 1
				check_converged_solution = need_to_retry
				return
	      end if
	
	      do j2 = 2,l+1
	         y(1:n,j2) = y(1:n,j2-1) - yhold(1:n,j2-1)
	      end do
	      yprime(1:n)=(y(1:n,1)-arh(1:n))/qqq
	
	      idoub = idoub - 1
	      trange=(tend-told-h)*h
	      if (trange < 0d0) then
	         idoub = idoub + 2
	      else
		      jchang = 0
		      call adjust ! have the option to adjust order or timestep size
	      end if
	
	      if (l /= lmax) then ! save highest order for error test on next step
	         y(1:n,lmax+4) = y(1:n,l+1)
	      end if
	
			! ----------------------------------------------------------------------
			! store the y array in the matrix yhold.  store in the y array the
			! information necessary to perform an interpolation to find the
			! solution at the specified output point if appropriate.
			! ----------------------------------------------------------------------
	      call cpyary(n*l,y,yhold)
	      nstep = nstep + 1
	      jsinup = jsinup + 1
	      jsnold = jsnold + 1
	      jstart = nqused
	      t = told + hused ! update t to new value
	      hold = h
	      kfail = 0
	      newpar = 0
	      cfail = .false.
			lte = dsqrt(demb/dble(n))
			check_converged_solution = pack_it_up

		end function check_converged_solution
		
		
#ifdef offload
      !dir$ attributes offload: mic :: after_a_failure
#endif
 		integer function after_a_failure()

			! uses norm as input

			real(dp) :: ffail, frfail, efail, prfail, plfail, pllfal, 
     >			twodwn, ddown
			
			kfail = kfail - 1
			! ********************************************************************
			! the error test failed. kfail keeps track of multiple failures.
			! restore t and the y array to their previous values and prepare to
			! try the step again. compute the optimal step size for this order
			! and one order lower.
			! ********************************************************************
			! ***  failed on step 1 because of accuracy
			! compute error in the solution
	      nfail = nfail + 1
			call set_scale(y1, n, itol, scale, ymax, 
     >				rtol, atol, nind1, nind2, nind3, hused)
	      ddown = sum(((y(1:n,l))/scale(1:n))**2)
	      twodwn = sum(((y(1:n,l-1))/scale(1:n))**2)
	      if (itol  ==  1) norm = norm/(rtol(1)**2)
	
	      t = told
	      hold = h
	      if (nq > 1) then; ffail = 0.5d0/dble(nq); else; ffail = 0; end if
	      if (nq > 2) then; frfail = 0.5d0/dble(nq-1); else; frfail = 0; end if
	      efail = 0.5d0/dble(l)
			
	      call cpyary(n*l,yhold,y) ! restore y from yhold
	
	      rmax = 2d0
	      if (dabs(h) <= hmin*1.00001d0) then
				! requested error not possible with given hmin
	         kflag = -1
	         hold = h
				after_a_failure = pack_it_up
				return
	      end if
      
	      if (kfail > -3) then
				! predicting a new h after insufficient accuracy
		      prfail = pow_cr(norm/(0.2d0*e_present),efail)*1.5d0 + 1.6d-6
		      plfail = pow_cr(ddown/(0.2d0*edn),ffail)*1.5d0+1.7d-6
		      if (nq > 2) then
					pllfal = pow_cr(twodwn/(0.2d0*eddn),frfail)*1.5d0+1.7d-6
				else
					pllfal=0
				end if
		      if (pllfal > plfail) plfail=pllfal
		      if (plfail < prfail .and. nq /= 1) then ! reduce order
		         nq=nq-1
		         rh=one/(plfail*dble(-kfail))
		         l=nq+1
		         call coset(nq,el,elst,tq,ncoset,maxord)
		         rc=rc*el(1)/oldlo
		         oldlo=el(1)
		         call errors(n,tq,edn,e_present,eup,bnd,eddn)
		      else
		         rh = one / (prfail*dble(-kfail))
		      endif
				after_a_failure = have_new_timestep
				return
			end if
			! ********************************************************************
			! control reaches this stage if 3 or more failures have occured.
			! it is assumed that the derivatives that have accumulated in the y
			! array have errors of the wrong order. hence the first derivative
			! is re-computed, and the order is set to 1. then h is reduced by a
			! factor of 10, and the step is retried. after a total of 7
			! failures an exit is taken with kflag=-2.
			! ********************************************************************
	      if (kfail == -7) then
				! error smaller than can be handled for problem
	         kflag = -2
	         hold = h
				after_a_failure = pack_it_up
				return
	      end if

			call back_to_order1
			after_a_failure = need_prediction
			
		end function after_a_failure 
			

#ifdef offload
      !dir$ attributes offload: mic :: try_stage2
#endif
		integer function try_stage2()
	      
			logical :: worked
			integer :: j1, jp1, m2

			if (verbosity > 0 .and. lout > 0) write(lout,*) 'try_stage2'
		
         ynhold(1:n,1) = y(1:n,1)
         ynhold(1:n,2) = save2(1:n)
      
	      kfail = 0
	      arh(1:n) = el(2)*y(1:n,1)
	      do j1 = 2,nq
	         jp1 = j1+1
	         arh(1:n) = arh(1:n) + el(jp1)*y(1:n,j1)
	      end do
	      call prdict(t,h,y,l,n)      
	      yprime(1:n)=(y(1:n,1)-arh(1:n))/qqq
         
	      call itrat2(
     >	      qqq,y1,yprime,n,t,h,qi,sfmin,y_min,y_max,bnd,arh,crate2,tcrat2,m2,
     +     		worked,ymax,error,save1,save2,scale,pw,pw1,pw_R,pw_C,mf,mbnd,
     +     		nind1,nind2,nind3,ipiv,stage2_min,itol,num_tol,rtol,atol,inf_nrms,
     >			num_rpar,rpar,num_ipar,ipar,
     +     		hused,nbsol,nre,nqused,nstep,equil,equed_flag,rcond,
     >			pderv,resid,call_back,monitor,2,verbosity,lout,
     >         caller_id,nvar,nz,lblk,dblk,ublk,decsolblk,
     >         lrd,rpar_decsol,lid,ipar_decsol,ierr) 
     
	      if (ierr /= 0) then
				ijus = 0
				finish = .false.
				try_stage2 = need_reduced_dt
				if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >				'try_stage2 failed with nonzero ierr in itrat2', ierr
				return
			end if
	      meqc2 = meqc2 + m2 + 1

	      if (.not.worked) then
	         nfail = nfail + 1
	         ijus = 0
	         try_stage2 = need_to_retry
				if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >				'try_stage2 newton iterations failed to converge'
				return
	      end if
		
			try_stage2 = need_stage3

		end function try_stage2
		
		
#ifdef offload
      !dir$ attributes offload: mic :: start_new_step
#endif
		integer function start_new_step()

			if (verbosity > 0 .and. lout > 0) write(lout,*) ''

			do
				told = t
		      kflag = 0
		      if (jstart > 0) then
					start_new_step = need_prediction
					return
				end if
		      if (jstart == 0) then
					call first_step_setup
					! *********************************************
					! on the first call, the order is set to 1.  
					! rmax is the maximum ratio by which h can be
					! increased in a single step.  rmax is set equal to 1.d4 initially
					! to compensate for the small initial h, but then is normally = 10.
					! if a failure occurs (in corrector convergence or error test),
					! rmax is set at 2. for the next increase.
					! *********************************************
			      call prdict(t,h,y,l,n)      
			      yprime(1:n)=(y(1:n,1)-arh(1:n))/h
					start_new_step = need_stage1
					return
				end if
				start_new_step = have_new_params
				return
			end do
		
		end function start_new_step


#ifdef offload
      !dir$ attributes offload: mic :: try_stage1
#endif
		integer function try_stage1()
			
			integer :: j, jm1, m1
			logical :: worked
	   	integer :: iiter
			! -----------------------------------------------------------------
			! if indicated, the matrix p = i/(h*el(2)) - j is re-evaluated before
			! starting the corrector iteration.  iweval is set = 0 to indicate
			! that this has been done. p is computed and processed in pset.
			! the processed matrix is stored in pw
			! -----------------------------------------------------------------
			
			if (verbosity > 0 .and. lout > 0) write(lout,*) 'try_stage1'
			
			if (iweval > 0) then ! check the matrix
		      iweval = 0
		      rc = one
		      iiter = meqc1 - mq1tmp ! the number of crate1's in the tcrat1 sum
		      if (jnewim) then ! current iteration matrix p was formed using a new j
		         if (jsnold >= 3) then ! we've taken enough steps with it to get average convergence rate
		            avnewj = tcrat1/dble(iiter) ! average rate with new j
		         else ! don't have enough info yet to form meaningful average
		            avnewj = one
		         end if
		      else ! current iteration matrix p was formed using an old j
		         if (jsnold >= 3) then ! we've taken enough steps with it to get average convergence rate
		            avoldj = tcrat1/dble(iiter)
		            if (avoldj < avnewj) then ! average convergence rate is actually improving
		               avnewj = avoldj
		            else if (((dabs(avoldj-avnewj)) > 0.3d0)  .or. 
     +          		((avoldj > 0.85d0) .and. (avoldj /= one))) then
		               cfail = .true.
		               crate1 = 0.1d0
		               crate2 = 0.1d0
		            end if            
		         else
		            cfail = .true.
		            crate1 = 0.1d0
		            crate2 = 0.1d0
						! *********************************************************
						! if we have reached here things must have gone wrong
						! *********************************************************
		         end if        
		      end if
     
		      if (cfail) then
		         newpar = 1 ! indicates that we've computed a new jacobian
		         jsinup = -1
		         jnewim = .true. ! indicates that we're using a new jacobian
		      end if

		      cfail = .false. ! reset the flag that indicates need new jacobian
		      jsnold = 0 ! reset count of steps taken with current matrix
		      tcrat1 = zero ! reset running sum of crate1's
		      tcrat2 = zero ! reset running sum of crate2's
		      mq1tmp = meqc1 ! reset running total of stage1 iterations
		      mq2tmp = meqc2 ! reset running total of stage2 iterations
				
		      call pset(
     +    			y1,yprime,n,h,t,uround,epsjac,qi,miter,mbnd,sfmin,
     >				nind1,nind2,nind3,ier,pderv,resid,call_back,
     >				ymax,save1,save2,scale,
     +    			pw,pw1,pw_R,pw_C,equil,equed_flag,
     >				error,ipiv,itol,num_tol,rtol,atol,npset,nje,nre,ndec,
     >				num_rpar,rpar,num_ipar,ipar,jdebug,verbosity,lout,
     >            caller_id,nvar,nz,lblk,dblk,ublk,decsolblk,
     >            need_free,lrd,rpar_decsol,lid,ipar_decsol,ierr) 

		      if (ierr /= 0) then
		         ijus = 1
					finish = .false.
					try_stage1 = need_reduced_dt
					if (verbosity > 0 .and. lout > 0) write(lout,*) 'try_stage1 failed in pset'
					return
				end if
		      qqq=qi

		      if (ier /= 0) then ! singularity in the iteration matrix
		         ijus = 1
		         nfail = nfail + 1
					try_stage1 = need_to_retry
					if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >					'try_stage1 failed with singularity in the iteration matrix'
		         return
				end if
				
	      end if
		
			! ********************************************************************
			! up to 4 corrector iterations are taken.  a convergence test is made
			! on the r.m.s. norm of each correction, using bnd, which depends
			! on atol and rtol.  the sum of the corrections is accumulated in the
			! vector  error(i).  the y array is not altered in the corrector
			! loop. the updated y vector is stored temporarily in save1.
			! ********************************************************************
	      call itrat2(
     >	      qqq,y1,yprime,n,t,h,qi,sfmin,y_min,y_max,bnd,arh,crate1,tcrat1,m1,
     >			worked,ymax,error,save1,save2,scale,pw,pw1,pw_R,pw_C,mf,mbnd,
     >			nind1,nind2,nind3,ipiv,stage1_min,itol,num_tol,rtol,atol,inf_nrms,
     >			num_rpar,rpar,num_ipar,ipar,
     >			hused,nbsol,nre,nqused,nstep,equil,equed_flag,rcond,
     >			pderv,resid,call_back,monitor,1,verbosity,lout,
     >         caller_id,nvar,nz,lblk,dblk,ublk,decsolblk,
     >         lrd,rpar_decsol,lid,ipar_decsol,ierr) 

         if (ierr /= 0) then
	         ijus = 0
				finish = .false.
				try_stage1 = need_reduced_dt
				if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >				'try_stage1 failed with nonzero ierr in itrat2', ierr
				return
			end if

	      meqc1 = meqc1 + m1 + 1 ! update the running total of stage1 newton iterations

			!   now test to see if it was successful or not

	      if (.not.worked) then
	         nfail = nfail + 1
				! ********************************************************************
				! the corrector iteration failed to converge in 4 tries. if
				! partials are not up to date, they are re-evaluated for the
				! next try. otherwise the y array is replaced by its values
				! before prediction and h is reduced if possible. if not a
				! non-convergence exit is taken
				! ********************************************************************
	         if (iweval == -1) then
					! have been using old partials, update them and try again
	            iweval = miter
	            cfail = .true.
	            yprime(1:n)=(y(1:n,1)-arh(1:n))*(one/qi)
					try_stage1 = need_stage1
					if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >				'try_stage1 failed with old partials; update them and try again'
					return
	         end if
         
	         ijus = 0
				try_stage1 = need_to_retry
				if (verbosity > 0 .and. lout > 0) 
     >				write(lout,*) 'try_stage1 newton iterations failed to converge'
				return
         
	      end if
      
	      iweval = -1
	      hused = h
	      nqused = nq
			sum_nq = sum_nq + nq
			sum_nq2 = sum_nq2 + nq*nq		
	      save2(1:n) = (save1(1:n)-arh(1:n))*(one/qi)
	      y(1:n,1) = save1(1:n)

			! update the differences
	      do j = 2,l
	         jm1 = j-1
	         y(1:n,j) = y(1:n,jm1) - yhold(1:n,jm1)
	   	end do

			! compute error in the solution
			call set_scale(y1, n, itol, scale, ymax, 
     >				rtol, atol, nind1, nind2, nind3, hused)
			
			! use the norm of the change in highest order terms as error estimate
	      norm = sum(((y(1:n,l)-yhold(1:n,l))/scale(1:n))**2)
	      if (itol  ==  1) norm = norm/(rtol(1)**2)
	
	      if (norm > e_present) then ! reject because estimated error too large
				try_stage1 = stage1_failed
				if (verbosity > 0 .and. lout > 0) write(lout,*) 
     >			'try_stage1 failed because estimated L2 error norm too large', 
     >              norm, e_present
				return
			end if
		
			try_stage1 = need_stage2
		
		end function try_stage1

		
#ifdef offload
      !dir$ attributes offload: mic :: unknown_state
#endif
		integer function unknown_state()
			kflag = -4			
			unknown_state = pack_it_up			
		end function unknown_state


      end subroutine stiff


		subroutine stiff_init_data(n_isaves_stiff,stiff_integer_saves,
     >				n_dsaves_stiff,stiff_double_saves)
			integer, intent(in) :: n_isaves_stiff, n_dsaves_stiff
			integer, intent(out) :: stiff_integer_saves(n_isaves_stiff)
	      real(dp), intent(out) :: stiff_double_saves(n_dsaves_stiff)

	      real(dp) :: el(10),elst(10),oldlo
			
			stiff_integer_saves = 0
			stiff_double_saves = 0d0

			el(2) = 1d0
			elst(2) = 1d0
			oldlo = 1d0

			stiff_double_saves(2) = el(2)
			stiff_double_saves(12) = elst(2)
			stiff_double_saves(52) = oldlo
		
		end subroutine stiff_init_data


      subroutine cpyary(nelem,source,target)
	      integer :: nelem
	      real(dp) ::  source(nelem),target(nelem)
	      target(1:nelem) = source(1:nelem)
      end subroutine cpyary


		end module mebdfi_stiff
		
		
