! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************


		module mtx_lib
      
      use const_def, only: dp, qp
      use mod_klu_dble, only: num_klu_ipar_decsol, num_klu_rpar_decsol
		
		implicit none

#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif
		
		contains
		
		! mesa includes sources for a subset of BLAS and dble.
		! you can use those, or, better yet, you can use a package optimized
		! for your machine such as GotoBLAS or Intel's MKL. 
		! see utils/makefile_header for details.

		! see mtx/blas_src for the subset of BLAS routines included in mtx_lib		
		! see mtx/dble_src for the subset of dble routines included in mtx_lib		
		
      ! subroutines for dense and banded matrix decompositions and solves
         
         include "mtx_dble_decsol.dek" ! dble versions
      
      ! subroutines for sparse matrix decomposition and solve

 !        include "mtx_umf_dble.dek" 
 !        include "mtx_umf_quad.dek" 
         ! interface to the UMFPACK sparse matrix solver (version 2.2d)
         ! if you use it, please reference this paper:
            ! Davis, T.A., and Duff, I.S.
            ! An Unsymmetric-Pattern Multifrontal Method for Sparse LU Factorization
            ! SIAM Journal on Matrix Analysis and Applications, vol 18, no. 1, pp. 140-158, Jan. 1997. 
         ! note: we are using an old version of UMFPACK.
         ! for newer versions, visit http://www.cise.ufl.edu/research/sparse/umfpack/

         include "mtx_klu_dble_decsol.dek" 
         include "mtx_klu_quad_decsol.dek" 
         ! interface to the KLU sparse matrix solver
         ! if you use it, please reference this paper:
            ! Davis, T.A., and Natarajan, E.P., 
            ! Algorithm 907: KLU, A Direct Sparse Solver for Circuit Simulation Problems,
            ! ACM Transactions on Mathematical Software, Vol. 37, No. 3, 2010.
      
      ! sometimes you just need a null version of a routine
      include "mtx_null_decsol.dek"
      
      ! sometimes you need to debug a jacobian by saving it to plotting data files
      include "mtx_debug_decsol.dek"

      ! sparse matrices come in many formats.  
      ! for example, compressed row sparse format is used by SPARSKIT,
      ! while compressed column sparse format is used by Super_LU.
      ! here are conversion routines for these two options.
      include "mtx_formats.dek"
      
      subroutine mtx_read_hbcode1(iounit, n, nnzero, values, rowind, colptr, ierr)
         use mtx_support, only: read_hbcode1
         integer, intent(in) :: iounit
         integer, intent(out) :: n, nnzero, ierr
         integer, pointer :: rowind(:) ! (nnzero) -- will be allocated
         integer, pointer :: colptr(:) ! (n+1) -- will be allocated
         real(dp), pointer :: values(:) ! (nnzero) -- will be allocated
         call read_hbcode1(iounit, n, n, nnzero, values, rowind, colptr, ierr)
      end subroutine mtx_read_hbcode1
      
      subroutine mtx_write_hbcode1(iounit, n, nnzero, values, rowind, colptr, ierr)
         use mtx_support, only: write_hbcode1
         integer, intent(in) :: iounit, n, nnzero
         integer :: rowind(:) ! (nnzero)
         integer :: colptr(:) ! (n+1)
         real(dp) :: values(:) ! (nnzero)
         integer, intent(out) :: ierr
         call write_hbcode1(iounit, n, n, nnzero, values, rowind, colptr, ierr)
      end subroutine mtx_write_hbcode1
      
      
      subroutine mtx_write_block_tridiagonal(iounit,nvar,nblk,lblk,dblk,ublk,ierr)
         use mtx_support, only: write_block_tridiagonal
         integer, intent(in) :: iounit, nvar, nblk
         real(dp), intent(in), dimension(:,:,:) :: lblk,dblk,ublk
         integer, intent(out) :: ierr
         call write_block_tridiagonal(iounit,nvar,nblk,lblk,dblk,ublk,ierr)
      end subroutine mtx_write_block_tridiagonal
         
      subroutine mtx_read_block_tridiagonal(iounit,nvar,nblk,lblk1,dblk1,ublk1,ierr)
         use mtx_support, only: read_block_tridiagonal
         integer, intent(in) :: iounit
         integer, intent(out) :: nvar, nblk
         real(dp), pointer, dimension(:) :: lblk1,dblk1,ublk1 ! =(nvar,nvar,nblk) will be allocated
         integer, intent(out) :: ierr
         call read_block_tridiagonal(iounit,nvar,nblk,lblk1,dblk1,ublk1,ierr)
      end subroutine mtx_read_block_tridiagonal
         
      subroutine mtx_read_quad_block_tridiagonal(iounit,nvar,nblk,lblk1,dblk1,ublk1,ierr)
         use mtx_support, only: read_quad_block_tridiagonal
         integer, intent(in) :: iounit
         integer, intent(out) :: nvar, nblk
         real(qp), pointer, dimension(:) :: lblk1,dblk1,ublk1 ! =(nvar,nvar,nblk) will be allocated
         integer, intent(out) :: ierr
         call read_quad_block_tridiagonal(iounit,nvar,nblk,lblk1,dblk1,ublk1,ierr)
      end subroutine mtx_read_quad_block_tridiagonal
   
      
      ! tridiagonal linear systems have a very efficient solver (using the "Thomas" algorithm)
      include "mtx_tridiag_decsol.dek"
      
      ! Thomas single-thread block tridiagonal
      include "mtx_block_thomas_dble_decsol.dek"
      include "mtx_block_thomas_quad_decsol.dek"
      ! Thomas single-thread block tridiagonal using KLU for individual blocks
      include "mtx_block_thomas_klu_decsol.dek"
      
      
      ! BCYCLIC multi-thread block tridiagonal
      include "mtx_bcyclic_dble_decsol.dek"  ! for dense blocks
      include "mtx_bcyclic_klu_decsol.dek"   ! for sparse blocks
         ! S.P.Hirshman, K.S.Perumalla, V.E.Lynch, & R.Sanchez,
         ! BCYCLIC: A parallel block tridiagonal matrix cyclic solver,
         ! J. Computational Physics, 229 (2010) 6392-6404.


      subroutine block_dble_mv(nvar, nz, lblk, dblk, ublk, b, prod)
         ! set prod = A*b with A = block tridiagonal given by lblk, dblk, ublk
         use mtx_support, only: do_block_dble_mv
         integer, intent(in) :: nvar, nz
         real(dp), pointer, dimension(:,:,:), intent(in) :: lblk, dblk, ublk ! (nvar,nvar,nz)
         real(dp), pointer, dimension(:,:), intent(in) :: b ! (nvar,nz)
         real(dp), pointer, dimension(:,:), intent(out) :: prod ! (nvar,nz)   
         call do_block_dble_mv(nvar, nz, lblk, dblk, ublk, b, prod)
      end subroutine block_dble_mv
      

      subroutine block_quad_mv(lblk, dblk, ublk, b, prod)
         ! set prod = A*b with A = block tridiagonal given by lblk, dblk, ublk
         use mtx_support, only: do_block_mv_quad
         real(qp), pointer, dimension(:,:,:), intent(in) :: lblk, dblk, ublk ! (nvar,nvar,nz)
         real(qp), pointer, dimension(:,:), intent(in) :: b ! (nvar,nz)
         real(qp), pointer, dimension(:,:), intent(out) :: prod ! (nvar,nz)   
         call do_block_mv_quad(lblk, dblk, ublk, b, prod)
      end subroutine block_quad_mv
      
      
      subroutine block_dble_LU_factored_mv(lblk, dblk, ublk, b, ipiv, prod)
         ! set prod = A*b with A = factored block tridiagonal given by lblk, dblk, ublk, ipiv
         use mtx_support, only: do_LU_factored_block_dble_mv
         real(dp), pointer, dimension(:,:,:), intent(in) :: lblk, dblk, ublk ! (nvar,nvar,nz)
         real(dp), pointer, dimension(:,:), intent(in) :: b ! (nvar,nz)
         integer, intent(in) :: ipiv(:,:) ! (nvar,nz)
         real(dp), pointer, dimension(:,:), intent(out) :: prod ! (nvar,nz) 
         call do_LU_factored_block_dble_mv(lblk, dblk, ublk, b, ipiv, prod)
      end subroutine block_dble_LU_factored_mv


      subroutine LU_factored_square_mv(a,b,ipiv,prod) 
         ! set prod = A*b with A factored in LU manner (see dgetrs).
         ! A^-1 = P*U^-1*L^-1*P^-1, so A = P*L*U*P^-1
         use mtx_support, only: do_LU_factored_square_mv
         real(dp), intent(in) :: a(:,:) ! (lda,m), lda >= m
         real(dp), intent(in) :: b(:) ! (m)
         integer, intent(in) :: ipiv(:) ! (m)
         real(dp), intent(out) :: prod(:) ! (m)
         call do_LU_factored_square_mv(size(ipiv,dim=1),a,b,ipiv,prod)
      end subroutine LU_factored_square_mv


      subroutine LU_factored_square_mm(A,B,ipiv,C) ! set C = A*B
         ! A factored in LU manner = P*L*U.
         use mtx_support, only: do_LU_factored_square_mm
         real(dp), intent(in) :: A(:,:) ! (lda,m), lda >= m
         real(dp), intent(in) :: B(:,:) ! (ldb,m), ldb >= m
         integer, intent(in) :: ipiv(:) ! (m)
         real(dp), intent(out) :: C(:,:) ! (ldc,m), ldc >= m
         call do_LU_factored_square_mm(size(ipiv,dim=1),A,B,ipiv,C)
      end subroutine LU_factored_square_mm


      subroutine multiply_xa(n, A1, x, b)
         !  calculates b = x*A
         use mtx_support, only: do_multiply_xa
         integer, intent(in) :: n
         real(dp), pointer, intent(in) :: A1(:) ! =(n, n)
         real(dp), pointer, intent(in) :: x(:) ! (n)
         real(dp), pointer, intent(out) :: b(:) ! (n)
         call do_multiply_xa(n, A1, x, b)
      end subroutine multiply_xa


      subroutine quad_multiply_xa(n, A1, x, b)
         !  calculates b = x*A
         use mtx_support, only: do_quad_multiply_xa
         integer, intent(in) :: n
         real(qp), pointer, intent(in) :: A1(:) ! =(n, n)
         real(qp), pointer, intent(in) :: x(:) ! (n)
         real(qp), pointer, intent(out) :: b(:) ! (n)
         call do_quad_multiply_xa(n, A1, x, b)
      end subroutine quad_multiply_xa


      subroutine multiply_xa_plus_c(n, A1, x, c, b)
         !  calculates b = x*A + c
         use mtx_support, only: do_multiply_xa_plus_c
         integer, intent(in) :: n
         real(dp), pointer, intent(in) :: A1(:) ! =(n,n)
         real(dp), pointer, intent(in) :: x(:), c(:) ! (n)
         real(dp), pointer, intent(out) :: b(:) ! (n)
         call do_multiply_xa_plus_c(n, A1, x, c, b)
      end subroutine multiply_xa_plus_c


      subroutine quad_multiply_xa_plus_c(n, A1, x, c, b)
         !  calculates b = x*A + c
         use mtx_support, only: do_quad_multiply_xa_plus_c
         integer, intent(in) :: n
         real(qp), pointer, intent(in) :: A1(:) ! =(n, n)
         real(qp), pointer, intent(in) :: x(:) ! (n)
         real(qp), pointer, intent(in) :: c(:) ! (n)
         real(qp), pointer, intent(out) :: b(:) ! (n)
         call do_quad_multiply_xa_plus_c(n, A1, x, c, b)
      end subroutine quad_multiply_xa_plus_c


      subroutine block_multiply_xa(nvar, nz, lblk1, dblk1, ublk1, x1, b1)
         !  calculates b = x*A
         use mtx_support, only: do_block_multiply_xa
         integer, intent(in) :: nvar, nz
         real(dp), dimension(:), intent(in), pointer :: lblk1, dblk1, ublk1 ! =(nvar,nvar,nz)
         real(dp), intent(in), pointer :: x1(:) ! =(nvar,nz)
         real(dp), intent(out), pointer :: b1(:) ! =(nvar,nz)
         call do_block_multiply_xa(nvar, nz, lblk1, dblk1, ublk1, x1, b1)
      end subroutine block_multiply_xa


      subroutine quad_block_multiply_xa(nvar, nz, lblk1, dblk1, ublk1, x1, b1)
         !  calculates b = x*A
         use mtx_support, only: do_quad_block_multiply_xa
         integer, intent(in) :: nvar, nz
         real(qp), dimension(:), intent(in), pointer :: lblk1, dblk1, ublk1 ! =(nvar,nvar,nz)
         real(qp), intent(in), pointer :: x1(:) ! =(nvar,nz)
         real(qp), intent(out), pointer :: b1(:) ! =(nvar,nz)
         call do_quad_block_multiply_xa(nvar, nz, lblk1, dblk1, ublk1, x1, b1)
      end subroutine quad_block_multiply_xa


      subroutine band_multiply_xa(n, kl, ku, ab1, ldab, x, b)
         !  calculates b = x*a = transpose(a)*x
         use mtx_support, only: do_band_multiply_xa         
         integer, intent(in) :: n
         !          the number of linear equations, i.e., the order of the
         !          matrix a.  n >= 0.
         integer, intent(in) :: kl
         !          the number of subdiagonals within the band of a.  kl >= 0.
         integer, intent(in) :: ku
         !          the number of superdiagonals within the band of a.  ku >= 0.
         integer, intent(in) :: ldab
         !          the leading dimension of the array ab.  ldab >= kl+ku+1.
         real(dp), intent(in), pointer :: ab1(:) ! =(ldab, n)
         !          the matrix a in band storage, in rows 1 to kl+ku+1;
         !          the j-th column of a is stored in the j-th column of the
         !          array ab as follows:
         !          ab(ku+1+i-j, j) = a(i, j) for max(1, j-ku)<=i<=min(n, j+kl)
         real(dp), intent(in), pointer :: x(:) ! (n)
         !          the input vector to be multiplied by the matrix.
         real(dp), intent(out), pointer :: b(:) ! (n)
         !          on exit, set to matrix product of x*a = b
         call do_band_multiply_xa(n, kl, ku, ab1, ldab, x, b)
      end subroutine band_multiply_xa
      
      
      include "mtx_lapack95.dek" 
      
      
      ! utilities for working with jacobians
      include "mtx_jac.dek"
		
		! the following call dble routines to estimate matrix condition numbers.
      include "mtx_rcond.dek"
      
      integer function decsol_option(which_decsol_option, ierr)
         use mtx_def
         character (len=*), intent(in) :: which_decsol_option
         integer, intent(out) :: ierr
         character (len=64) :: option
         ierr = 0
         option = which_decsol_option
         
         if (option == 'lapack') then
            decsol_option = lapack
            
         else if (option == 'bcyclic_dble') then
            decsol_option = bcyclic_dble
            
         else if (option == 'bcyclic_klu') then
            decsol_option = bcyclic_klu
            
         else if (option == 'block_thomas_dble') then
            decsol_option = block_thomas_dble
            
         else if (option == 'block_thomas_quad') then
            decsol_option = block_thomas_quad
            
         else if (option == 'block_thomas_klu') then
            decsol_option = block_thomas_klu
            
         else if (option == 'klu') then
            decsol_option = klu
            
         else
            ierr = -1
            decsol_option = -1
         end if 
      end function decsol_option
      
      
      subroutine decsol_option_str(which_decsol_option, decsol_option, ierr)
         use mtx_def
         integer, intent(in) :: which_decsol_option
         character (len=*), intent(out) :: decsol_option
         integer, intent(out) :: ierr
         ierr = 0
         
         if (which_decsol_option == lapack) then
            decsol_option = 'lapack'
            
         else if (which_decsol_option == bcyclic_dble) then
            decsol_option = 'bcyclic_dble'
            
         else if (which_decsol_option == bcyclic_klu) then
            decsol_option = 'bcyclic_klu'
            
         else if (which_decsol_option == block_thomas_dble) then
            decsol_option = 'block_thomas_dble'
            
         else if (which_decsol_option == block_thomas_quad) then
            decsol_option = 'block_thomas_quad'
            
         else if (which_decsol_option == block_thomas_klu) then
            decsol_option = 'block_thomas_klu'
            
         else if (which_decsol_option == klu) then
            decsol_option = 'klu'
            
         else
            ierr = -1
            decsol_option = ''
         end if 
         
      end subroutine decsol_option_str
      
      
      logical function is_sparse_decsol(which_decsol_option)
         use mtx_def
         integer, intent(in) :: which_decsol_option
         is_sparse_decsol = (which_decsol_option == klu)
      end function is_sparse_decsol
      
      
      logical function is_block_tridiagonal_decsol(which_decsol_option)
         use mtx_def
         integer, intent(in) :: which_decsol_option
         is_block_tridiagonal_decsol = &
            (which_decsol_option == bcyclic_dble) .or. &
            (which_decsol_option == bcyclic_klu) .or. &
            (which_decsol_option == block_thomas_dble) .or. & 
            (which_decsol_option == block_thomas_quad) .or. & 
            (which_decsol_option == block_thomas_klu)
      end function is_block_tridiagonal_decsol
      
      
      real(dp) function get_thomas_blck_sprs_clip_lim()
         use thomas_block_sparse, only: clip_limit
         get_thomas_blck_sprs_clip_lim = clip_limit
      end function get_thomas_blck_sprs_clip_lim
      
      
      subroutine set_thomas_blck_sprs_clip_lim(new_clip_limit)
         use thomas_block_sparse, only: clip_limit
         real(dp), intent(in) :: new_clip_limit
         clip_limit = new_clip_limit
      end subroutine set_thomas_blck_sprs_clip_lim

#ifdef offload
      !dir$ end options
#endif

		end module mtx_lib
