
#include <stdio.h>
#include "klu_version.h"
#include "klu.h"

#define HANDLE_SIZE  8
/* kind of integer to hold a pointer. */
typedef long long fptr;  /* 64-bit by default */

typedef struct {
   KLU_symbolic *Symbolic ;
   KLU_numeric *Numeric ;
   KLU_common Common ;
} factors_t;

#ifdef USE_FLOAT128
#undef c_fort
#define c_fort c_fortran_klu_quad_
#undef do_klu_test
#define do_klu_test do_klu_test_quad_
#undef basicfloat
#define basicfloat __float128
#else
#undef c_fort
#define c_fort c_fortran_klu_dble_
#undef do_klu_test
#define do_klu_test do_klu_test_dble_
#undef basicfloat
#define basicfloat double
#endif


static void do_klu_test();




void c_fort(int *iopt, int *n, int *nrhs, 
                 basicfloat Ax [ ], int Ai [ ], int Ap [ ],
                 basicfloat B [ ], int *ldb, basicfloat *result,
		 fptr *f_factors, /* a handle containing the address
				     pointing to the factored matrices */
		 int *info)
{
/* 
 * This routine can be called from Fortran.
 *
 * f_factors (input/output) fptr* 
 *      If iopt == 1, it is an output and contains the pointer pointing to
 *                    the structure of the factored matrices.
 *      Otherwise, it it an input.
 *
 */
    int i;
    factors_t *klu, *copy_klu;
    
    
	 klu = (factors_t*) *f_factors;
	      //printf("set klu = *f_factors\n");
    *info = 0;
	      //printf("set *info = 0\n");
    *result = 0.0;
	      //printf("set *result = 0.0\n");
	      //if ( *iopt == -1 ) printf("iopt %i\n", *iopt);

    if ( *iopt == -6 ) { /* turn off pivoting */

       klu->Common.tol = 0.0; /* force diagonal to be the pivot */

    } else if ( *iopt == -5 ) { /* copy klu */

	    copy_klu = (factors_t*) malloc(sizeof(factors_t));
       *copy_klu = *klu;
	    *f_factors = (fptr) copy_klu;

    } else if ( *iopt == -4 ) { /* free klu */
       
       free(klu);

    } else if ( *iopt == -3 ) { /* free Symbolic */

       KLU_free_symbolic (&klu->Symbolic, &klu->Common) ;
       *info = klu->Common.status;
       free(klu);

    } else if ( *iopt == -2 ) { /* free Numeric */

       KLU_free_numeric (&klu->Numeric, &klu->Common) ;
       *info = klu->Common.status;

    } else if ( *iopt == -1 ) { /* create Symbolic */

	      //printf("create Symbolic\n");
	    klu = (factors_t*) malloc(sizeof(factors_t));
	    *f_factors = (fptr) klu;
	   //printf("call KLU_defaults\n");
       KLU_defaults (&klu->Common) ;
       
       //klu->Common.tol = 0.0; /* force diagonal to be the pivot */
       //klu->Common.btf = 0; /* skip BTF since already have non-zero diagonals */
       //klu->Common.scale = 2; /* scale rows by max element */
       
	   //printf("call KLU_analyze \n");
       klu->Symbolic = KLU_analyze (*n, Ap, Ai, &klu->Common) ;
	   //printf("done KLU_analyze\n");
       *info = klu->Common.status;

    } else if ( *iopt == 0 ) { /* create Numeric */

       klu->Numeric = KLU_factor (Ap, Ai, Ax, klu->Symbolic, &klu->Common) ;
       *info = klu->Common.status;

    } else if ( *iopt == 1 ) { /* analyze and factor */

	    klu = (factors_t*) malloc(sizeof(factors_t));
	    *f_factors = (fptr) klu;
       KLU_defaults (&klu->Common) ;  
       
       klu->Common.tol = 0.0; /* force diagonal to be the pivot */
       klu->Common.btf = 0; /* skip BTF since already have non-zero diagonals */
       klu->Common.scale = 2; /* scale rows by max element */
       
       klu->Symbolic = KLU_analyze (*n, Ap, Ai, &klu->Common) ;
       if (klu->Common.status == 0) {
          klu->Numeric = KLU_factor (Ap, Ai, Ax, klu->Symbolic, &klu->Common) ;
       } else {
	       printf("c_fortran_klu failed in analyze\n");
       }
       *info = klu->Common.status;

    } else if ( *iopt == 2 ) { /* solve */
       
       KLU_solve (klu->Symbolic, klu->Numeric, *ldb, *nrhs, B, &klu->Common) ;
       *info = klu->Common.status;

    } else if ( *iopt == 3 ) { /* free storage */

       KLU_free_symbolic (&klu->Symbolic, &klu->Common) ;
       KLU_free_numeric (&klu->Numeric, &klu->Common) ;
       free(klu);

    } else if ( *iopt == 4 ) { /* refactor */

       KLU_refactor (Ap, Ai, Ax, klu->Symbolic, klu->Numeric, &klu->Common) ;
       *info = klu->Common.status;

    } else if ( *iopt == 5 ) { /* result = rcond */

       KLU_rcond (klu->Symbolic, klu->Numeric, &klu->Common) ;
       *result = klu->Common.rcond;
       *info = klu->Common.status;

    } else if ( *iopt == 6 ) { /* result = rgrowth */

       KLU_rgrowth (Ap, Ai, Ax, klu->Symbolic, klu->Numeric, &klu->Common) ;
       *result = klu->Common.rgrowth;
       *info = klu->Common.status;

    } else if ( *iopt == 7 ) { /* result = condest */
       KLU_condest (Ap, Ax, klu->Symbolic, klu->Numeric, &klu->Common) ;
       *result = klu->Common.condest;
       *info = klu->Common.status;

    } else {
	    printf("Invalid iopt=%d passed to c_fortran_klu\n",*iopt);
	    exit(-1);
    }
}



static void do_klu_test() {
   int    n = 5 ;
   int    Ap [ ] = {0, 2, 5, 9, 10, 12} ;
   int    Ai [ ] = { 0,  1,  0,   2,  4,  1,  2,  3,   4,  2,  1,  4} ;
   basicfloat Ax [ ] = {2., 3., 3., -1., 4., 4., -3., 1., 2., 2., 6., 1.} ;
   basicfloat B1 [ ] = {8., 45., -3., 3., 19.} ;
   basicfloat B2 [ ] = {8., 45., -3., 3., 19.} ;
   basicfloat err1, err2, err, result, rcond, rgrowth, condest;
   fptr klu_info;
   fptr *klu = &klu_info;
   int i, nrhs, ldb, info, iopt ;

	      printf("do_klu_test\n");
   
   nrhs = 1;
   ldb = n;
   
   iopt = 1; // factor
   c_fort(&iopt, &n, &nrhs, Ax, Ai, Ap, B1, &ldb, &result, klu, &info);
   if (info < 0) {
      printf("error in factoring\n");
      exit(1);
   }
   
   iopt = 5; // rcond
   c_fort(&iopt, &n, &nrhs, Ax, Ai, Ap, B1, &ldb, &rcond, klu, &info);
   if (info < 0) {
      printf("error in solve\n");
      exit(1);
   }
   
   iopt = 6; // rgrowth
   c_fort(&iopt, &n, &nrhs, Ax, Ai, Ap, B1, &ldb, &rgrowth, klu, &info);
   if (info < 0) {
      printf("error in solve\n");
      exit(1);
   }
   
   iopt = 7; // condest
   c_fort(&iopt, &n, &nrhs, Ax, Ai, Ap, B1, &ldb, &condest, klu, &info);
   if (info < 0) {
      printf("error in solve\n");
      exit(1);
   }
   
   iopt = 2; // solve
   c_fort(&iopt, &n, &nrhs, Ax, Ai, Ap, B1, &ldb, &result, klu, &info);
   if (info < 0) {
      printf("error in solve\n");
      exit(1);
   }
   
   err1 = 0.0;
   for (i = 0 ; i < n ; i++) {
      err = i + 1.0 - B1[i];
      err1 = err1 + err*err;
   }
   
   iopt = 4; // refactor
   c_fort(&iopt, &n, &nrhs, Ax, Ai, Ap, B2, &ldb, &result, klu, &info);
   if (info < 0) {
      printf("error in solve\n");
      exit(1);
   }
   
   iopt = 2; // solve
   c_fort(&iopt, &n, &nrhs, Ax, Ai, Ap, B2, &ldb, &result, klu, &info);
   if (info < 0) {
      printf("error in solve\n");
      exit(1);
   }

   iopt = 3; // dealloc
   c_fort(&iopt, &n, &nrhs, Ax, Ai, Ap, B2, &ldb, &result, klu, &info);
   if (info < 0) {
      printf("error in solve\n");
      exit(1);
   }
   
   err2 = 0.0;
   for (i = 0 ; i < n ; i++) {
      err = i + 1.0 - B2[i];
      err2 = err2 + err*err;
   }
   printf ("err1 %Lg       err2 %Lg       rcond %Lg       rgrowth %Lg       condest %Lg\n", 
      err1, err2, rcond, rgrowth, condest) ;
}



