      module test_mlt_support
      use mlt_lib
      use mlt_def
      use const_def
      use crlibm_lib

      implicit none

#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif
      
      logical, parameter :: do_standard_test = .false.

      ! inputs
      real(dp) :: cgrav
      real(dp) :: m       ! enclosed mass
      real(dp) :: r       ! radius
      real(dp) :: T       ! temperature
      real(dp) :: rho     ! density
      real(dp) :: L       ! luminosity
      real(dp) :: grad_star ! the actual dlnT/dlnP
      real(dp) :: P       ! pressure
      real(dp) :: chiRho  ! dlnP_dlnd at constant T
      real(dp) :: chiT    ! dlnP_dlnT at constant Rho
      real(dp) :: Cp      ! specific heat capacity at constant pressure
      real(dp) :: csound  ! sound speed
      real(dp) :: opacity
      real(dp) :: grada ! dlnT/dlnP at constant entropy (adiabatic gradient)
      real(dp) :: alpha_semiconvection
      real(dp) :: gradr_factor
      real(dp) :: Cv
      real(dp) :: X
      real(dp) :: mixing_length_alpha, thermohaline_coeff
      logical :: alt_scale_height, MLT_dbg
      real(dp) :: gradL_composition_term
      real(dp) :: Henyey_y_param, Henyey_nu_param, tau, &
         prev_conv_vel, max_conv_vel, g_theta, dt, remove_small_D_limit
      integer :: dominant_iso_for_thermohaline
      character (len=64) :: thermohaline_option, MLT_option, semiconvection_option
      
      
      ! results
      integer :: mixing_type
      real(dp) :: mlt_basics(num_mlt_results)
      real(dp) :: mlt_basics_alt(num_mlt_results)
      
      real(dp), target :: mlt_partials1_ary(num_mlt_partials*num_mlt_results)
      real(dp), pointer :: mlt_partials1(:), mlt_partials(:,:)
      real(dp), target :: mlt_partials1_alt_ary(num_mlt_partials*num_mlt_results)
      real(dp), pointer :: mlt_partials1_alt(:), mlt_partials_alt(:,:)
      
      real(dp) :: gradT, gradr, scale_height, convection_velocity, &
         d_gradT_dlnd, d_gradT_dlnT, d_gradT_dlnR, d_gradT_dL, d_gradT_dlnm, &
         log10_convection_velocity, log10_scale_height


      contains
      
      
      subroutine Do_MLT_Test
         use chem_lib
         use const_lib
         use atm_lib, only: eval_Paczynski_gradr
         integer :: ierr
         real(dp) :: lnR, dlnR, d_analytical, d_numerical, dL
         
         include 'formats'
         
         
         gradL_composition_term = 0
         dominant_iso_for_thermohaline = 6
         alt_scale_height = .true.
         remove_small_D_limit = 1d2
         thermohaline_coeff = 0
         Henyey_y_param = one_third
         Henyey_nu_param = 8
         tau = 1
         prev_conv_vel = -1
         max_conv_vel = 1d99
         g_theta = 0
         dt = -1
         
         thermohaline_option = 'Kippenhahn'
         semiconvection_option = 'Langer_85 mixing; gradT = gradr'
         MLT_option = 'Henyey'
         X = 0.7d0
         
         gradr_factor = 1
         alpha_semiconvection = 0
         
         ! test data from near top of central convective region of 4.0M ZAMS, Z=0.02
                             m =     6.0708798764978617D+32
                             r =     1.8104022869294819D+10
                             T =     1.6390574971606171D+07
                           rho =     2.1498086433161234D+01
                             L =     1.4437988700909283D+35
                             P =     6.3673785913580768D+16
                        chiRho =     1.0016146515292350D+00
                          chiT =     1.0039723441618589D+00
                            Cp =     4.5860132601655614D+08
                       opacity =     7.1145152433753656D-01
                         grada =     3.9495940002582286D-01
           mixing_length_alpha =     2.0000000000000000D+00
 
         ! expected results
                         gradT =     3.9495941506490573D-01
                         gradr =     5.8865301921953783D-01
           convection_velocity =     4.7462650334641930D+03
                  scale_height =     2.3964312124944618D+10
                  d_gradT_dlnd =     2.9566266663535595D-03
                  d_gradT_dlnT =    -9.8632412487315978D-03
                  d_gradT_dlnR =     1.1076053742688285D-18
                    d_gradT_dL =     2.1104128893990745D-43
                  d_gradT_dlnm =    -3.0470117470354128D-08
                  
         log10_convection_velocity = log10_cr(convection_velocity)
         log10_scale_height = log10_cr(scale_height)
         prev_conv_vel = -1
         max_conv_vel = 1d99
         g_theta = 0
         dt = -1
         MLT_dbg = .false.
         
         Cv = Cp ! need Cv
         csound = 0 ! need to get value of csound; it isn't used in current tests.
         
         call mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada,  &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
            mixing_type, mlt_basics, mlt_partials1, ierr)
                  
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            stop 1
         end if
      
         call show_mlt_input
         
         ! eval_Paczynski_gradr used to be part of mlt, hence the test here.
         gradr = eval_Paczynski_gradr(P,opacity,L,m,cgrav,crad*T**4/3,tau,T,r,rho)
         write(*,1) 'eval_Paczynski_gradr', gradr
         write(*,*)
         
         call check_gradT
         call check_convection_velocity
         call check_gradr
         write(*,*) 'MLT Results'
         call show_mlt_results
         write(*,*)
         
         
         return
         ! check partials
         
         dL = L*1d-7
         call mlt_eval( &
            cgrav, m, r, T, rho, L+dL, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada,  &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
            mixing_type, mlt_basics_alt, mlt_partials1_alt, ierr)
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            stop 1
         end if
         d_numerical = (mlt_basics_alt(mlt_D) - mlt_basics(mlt_D))/dL
         d_analytical = mlt_partials(mlt_dL, mlt_D)
         write(*,1) 'd_numerical', d_numerical
         write(*,1) 'd_analytical', d_analytical
         stop
         
         
         

      end subroutine Do_MLT_Test
      
         
         subroutine show_mlt_input
            include 'formats'
            write(*,*)
            write(*,*) 'MLT Input'
            write(*,*)
            write(*,1) 'm (Msun)', m / Msun
            write(*,1) 'r (Rsun)', r / Rsun
            write(*,1) 'logT', log10_cr(T)
            write(*,1) 'logRho', log10_cr(Rho)
            write(*,1) 'logP', log10_cr(P)
            write(*,1) 'log Cp', log10_cr(Cp)
            write(*,1) 'log kap', log10_cr(opacity)
            write(*,1) 'log L', log10_cr(L)
            write(*,1) 'grada', grada
            write(*,*)
         end subroutine show_mlt_input
         
         subroutine show_mlt_results
            include 'formats'
            write(*,*)               
            write(*,2) 'mixing_type', mixing_type
            write(*,1) 'gradT', mlt_basics(mlt_gradT)
            write(*,1) 'log velocity', log10_cr(mlt_basics(mlt_convection_velocity))
            write(*,1) 'gradr', mlt_basics(mlt_gradr)
            write(*,1) 'log scale height', log10_cr(mlt_basics(mlt_scale_height))
            write(*,1) 'D', mlt_basics(mlt_D)
         end subroutine show_mlt_results
         
         subroutine check_gradT
            include 'formats'
            if (abs(mlt_basics(mlt_gradT) - gradT) > 1d-2 * gradT) then
               write(*,*)
               write(*,*) 'WARNING'
               write(*,1) 'expected gradT', gradT
               write(*,1) ' but got gradT', mlt_basics(mlt_gradT)
               write(*,1) '    relative error',  &
                  abs(mlt_basics(mlt_gradT) - gradT) / gradT
            end if
         end subroutine check_gradT

         subroutine check_convection_velocity
            include 'formats'
            if (abs(log10_cr(mlt_basics(mlt_convection_velocity)) - log10_convection_velocity)  &
                  > 1d-2 * log10_convection_velocity) then
               write(*,*)
               write(*,*) 'WARNING'
               write(*,1) 'expected log10_convection_velocity',  &
                     log10_convection_velocity
               write(*,1) ' but got log10_convection_velocity',  &
                     log10_cr(mlt_basics(mlt_convection_velocity))
               write(*,1) '    relative error',  &
                     abs(log10_cr(mlt_basics(mlt_convection_velocity)) - log10_convection_velocity)  &
                     / log10_convection_velocity
            end if
         end subroutine check_convection_velocity

         subroutine check_gradr
            include 'formats'
            if (abs(mlt_basics(mlt_gradr) - gradr) > 1d-2 * gradr) then
               write(*,*)
               write(*,*) 'WARNING'
               write(*,1) 'expected gradr', gradr
               write(*,1) ' but got gradr', mlt_basics(mlt_gradr)
               write(*,1) '    relative error',  &
                  abs(mlt_basics(mlt_gradr) - gradr) / gradr
            end if
         end subroutine check_gradr
         
         
      
      
      
      subroutine do_semiconvection_test
         integer :: ierr
         real(dp) :: gradr, gradT, conv_vel, mixing_length, tau
         
         include 'formats.dek'
         
         write(*,*) 'do_semiconvection_test'
         
         
                                     m =    1.1176820612008070D+34
                                     r =    8.8645308490830048D+10
                                     T =    2.2250617568258464D+07
                                   rho =    2.6828952300338811D+00
                                     L =    7.6031961060516470D+37
                                     P =    8.6076623322211160D+15
                                chiRho =    9.2808438263133053D-01
                                  chiT =    1.2160620094805608D+00
                                    Cp =    5.5589182151158094D+08
                                    Cv =    3.2614122541475892D+08
                               opacity =    3.7300490123264862D-01
                                 grada =    3.3983015676206235D-01

                          gradr_factor =    1.0000000000000000D+00
                gradL_composition_term =    1.4062894973013990D-02
         dominant_iso_for_thermohaline =           6
                  alpha_semiconvection =    1.0000000000000001D-01
                   thermohaline_coeff =    0.0000000000000000D+00
                   mixing_length_alpha =    1.6000000000000001D+00
                         alt_scale_height = .true.
         remove_small_D_limit = 1d2

                        Henyey_y_param =    3.3333332999999998D-01
                       Henyey_nu_param =    8.0000000000000000D+00

         X = 0.7
         thermohaline_option = 'Kippenhahn'
         !semiconvection_option = 'Langer_85 mixing; gradT = gradr'
         semiconvection_option = 'Langer_85'
         MLT_option = 'Cox'
         tau = 1
         prev_conv_vel = -1
         max_conv_vel = 1d99
         g_theta = 0
         dt = -1
         MLT_dbg = .false.
         
         csound = 0 ! dummy value for now
 
         
         write(*,*) 'call mlt_eval'
         call mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada,  &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
            mixing_type, mlt_basics, mlt_partials1, ierr)
                  
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            stop 1
         end if
         
         gradr = mlt_basics(mlt_gradr)
         gradT = mlt_basics(mlt_gradT)
         conv_vel = mlt_basics(mlt_convection_velocity)
         mixing_length = mlt_basics(mlt_Lambda)
         write(*,*)
         write(*,1) 'gradr-grada', gradr-grada
         write(*,1) 'gradr', gradr
         write(*,1) 'grada', grada
         write(*,1) 'gradT', gradT
         write(*,*)
         write(*,1) 'conv_vel', conv_vel
         write(*,1) 'mixing_length', mixing_length
         write(*,*)


         write(*,*)
         write(*,2) 'mixing_type', mixing_type
         write(*,2) 'semiconvective_mixing', semiconvective_mixing
         write(*,*)
         write(*,1) trim(semiconvection_option)
         write(*,*)
         stop 'do_semiconvection_test'
         
      end subroutine do_semiconvection_test
         
      
      
      
      subroutine do1_test
         integer :: ierr, i, j
         real(dp) :: gradr, gradT, D_mix, conv_vel, mixing_length, tau
         real(dp) :: d_analytical, d_numerical, dP, lnT, dlnT
         
         include 'formats.dek'
         
         write(*,*) 'do1_test'
         
         
                                                cgrav =    6.6742800000000007D-08
                                                    m =    1.2963226545436946D+34
                                                    r =    9.5209648913830078D+10
                                                    T =    2.7612331159589186D+07
                                                  rho =    1.5529606267262894D+00
                                                    L =    4.2664390264588951D+38
                                                    P =    6.0826754777320240D+15
                                               chiRho =    7.5887984241494622D-01
                                                 chiT =    1.7234558856720641D+00
                                                   Cp =    1.1299336942200823D+09
                                                   Cv =    5.7384419242389536D+08
                                               csound =    7.6551630460922822D+07
                                                    X =    4.4434810132519137D-01
                                              opacity =    2.7607427771409393D-01
                                                grada =    2.8555878073959817D-01

                                         gradr_factor =    1.0003620918582783D+00
                               gradL_composition_term =    3.1457965208611330D+06
                        dominant_iso_for_thermohaline =          2

                                 alpha_semiconvection =    1.0000000000000000D-04
         semiconvection_option = 'Langer_85 mixing; gradT = gradr'

                                   thermohaline_coeff =    1.0000000000000000D+00
                     thermohaline_option = "Kippenhahn"
                        dominant_iso_for_thermohaline =           2

                                  mixing_length_alpha =    2.0000000000000000D+00
                         alt_scale_height = .true.

                                       Henyey_y_param =    3.3333332999999998D-01
                                      Henyey_nu_param =    8.0000000000000000D+00

                                     MLT_option = 'Cox'
         
         
         tau = 1
         prev_conv_vel = -1
         max_conv_vel = 1d99
         g_theta = 0
         dt = -1
         MLT_dbg = .true.
         
         csound = 0 ! dummy value for now
         
         write(*,*) 'call mlt_eval do1_test'
         call mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada,  &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
            mixing_type, mlt_basics, mlt_partials1, ierr)
                  
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            stop 1
         end if
         
         gradr = mlt_basics(mlt_gradr)
         gradT = mlt_basics(mlt_gradT)
         conv_vel = mlt_basics(mlt_convection_velocity)
         D_mix = mlt_basics(mlt_D)
         mixing_length = mlt_basics(mlt_Lambda)
         write(*,*)
         write(*,1) 'gradr-grada', gradr-grada
         write(*,1) 'gradr', gradr
         write(*,1) 'grada', grada
         write(*,1) 'gradT', gradT
         write(*,*)
         write(*,1) 'conv_vel', conv_vel
         write(*,1) 'D_mix', D_mix
         write(*,1) 'mixing_length', mixing_length
         write(*,*)


         write(*,*)
         write(*,2) 'mixing_type', mixing_type
         write(*,*)
         
         do i=1,num_mlt_results
            write(*,2) 'mlt_basics(i)', i, mlt_basics(i)
         end do

!         do i=1,num_mlt_results
!            do j=1,num_mlt_partials
!               write(*,3) 'mlt_partials(j,i)', j, i, mlt_partials(j,i)
!            end do
!         end do
         
         write(*,*)




         ! check partials
         
         dP = P*1d-7
         lnT = log_cr(T)
         dlnT = 1d-7*lnT
         call mlt_eval( &
            cgrav, m, r, exp_cr(lnT+dlnT), rho, L, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada,  &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
            mixing_type, mlt_basics_alt, mlt_partials1_alt, ierr)
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            stop 1
         end if
         
         d_numerical = (mlt_basics_alt(mlt_D) - mlt_basics(mlt_D))/dlnT
         d_analytical = mlt_partials(mlt_dlnT, mlt_D)
         write(*,1) 'dlnD/dlnT d_numerical', d_numerical/D_mix
         write(*,1) 'd_analytical', d_analytical/D_mix
         
         d_numerical = (mlt_basics_alt(mlt_Gamma) - mlt_basics(mlt_Gamma))/dlnT
         d_analytical = mlt_partials(mlt_dlnT, mlt_Gamma)
         write(*,1) 'dGamma/dlnT d_numerical', d_numerical
         write(*,1) 'd_analytical', d_analytical
         
         d_numerical = (mlt_basics_alt(mlt_gradT) - mlt_basics(mlt_gradT))/dlnT
         d_analytical = mlt_partials(mlt_dlnT, mlt_gradT)
         write(*,1) 'dgradT/dlnT d_numerical', d_numerical
         write(*,1) 'd_analytical', d_analytical
         
         d_numerical = (mlt_basics_alt(mlt_scale_height) - mlt_basics(mlt_scale_height))/dlnT
         d_analytical = mlt_partials(mlt_dlnT, mlt_scale_height)
         write(*,1) 'dln_scale_height/dlnT d_numerical', d_numerical/mlt_basics(mlt_scale_height)
         write(*,1) 'd_analytical', d_analytical/mlt_basics(mlt_scale_height)
         
         d_numerical = (mlt_basics_alt(mlt_convection_velocity) - mlt_basics(mlt_convection_velocity))/dlnT
         d_analytical = mlt_partials(mlt_dlnT, mlt_convection_velocity)
         write(*,1) 'dln_convection_velocity/dlnT d_numerical', d_numerical/mlt_basics(mlt_convection_velocity)
         write(*,1) 'd_analytical', d_analytical/mlt_basics(mlt_convection_velocity)
         stop
         
         




         stop 'do1_test'
         
      end subroutine do1_test
      
      
      subroutine do_thermohaline_test
         integer :: ierr
         real(dp) :: gradr, gradT, conv_vel, mixing_length, tau
         
         include 'formats.dek'
                  
                                                    m =    5.7032149651480377D+32
                                                    r =    1.0858006432604305D+10
                                                    T =    7.1015611551428121D+06
                                                  rho =    3.8762625920915750D-01
                                                    L =    3.6838894452152698D+35
                                                    P =    3.6800506868274644D+14
                                               chiRho =    9.8192428361020445D-01
                                                 chiT =    1.0544841817737272D+00
                                                   Cp =    3.7712160295735288D+08
                                                   Cv =    2.2580635453606457D+08
                                                    X =    6.7521599521551812D-01
                                              opacity =    5.8080847679803682D-01
                                                grada =    3.8050477761835633D-01
 
                                         gradr_factor =    1.0000000000000000D+00
                               gradL_composition_term =   -6.2142426904104969D-06
                        dominant_iso_for_thermohaline =           6
                                 alpha_semiconvection =    4.0000000000000001D-02
                                  thermohaline_coeff =    1.0000000000000000D+00
                                  mixing_length_alpha =    1.6000000000000001D+00
                         alt_scale_height = .true.
         remove_small_D_limit = 1d2

                                          MLT_option = 'Cox'
                                       Henyey_y_param =    3.3333332999999998D-01
                                      Henyey_nu_param =    8.0000000000000000D+00
                                   tau =    1
         prev_conv_vel = -1
         max_conv_vel = 1d99
         dt = -1
         MLT_dbg = .false.
         
         csound = 0 ! dummy value for now
         
         semiconvection_option = 'Langer_85 mixing; gradT = gradr'

         !thermohaline_option = 'Kippenhahn'
         !thermohaline_option = 'Traxler_Garaud_Stellmach_11'
         thermohaline_option = 'Brown_Garaud_Stellmach_13'
         
         write(*,*) 'call mlt_eval'
         call mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada,  &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, MLT_dbg, &
            mixing_type, mlt_basics, mlt_partials1, ierr)
                  
         if (ierr /= 0) then
            write(*,*) 'bad return from mlt_eval'
            stop 1
         end if
         
         gradr = mlt_basics(mlt_gradr)
         gradT = mlt_basics(mlt_gradT)
         conv_vel = mlt_basics(mlt_convection_velocity)
         mixing_length = mlt_basics(mlt_Lambda)
         write(*,*)
         write(*,1) trim(thermohaline_option)
         write(*,1) 'gradr-grada', gradr-grada
         write(*,1) 'gradr', gradr
         write(*,1) 'gradT', gradT
         write(*,1) 'conv_vel', conv_vel
         write(*,1) 'D', mlt_basics(mlt_D)
         write(*,1) 'mixing_length', mixing_length
         write(*,2) 'mixing_type', mixing_type
         write(*,*)


         write(*,*)
         if (mixing_type == thermohaline_mixing) then
            write(*,*) 'mixing_type == thermohaline_mixing'
         else
            write(*,*) 'mixing_type /= thermohaline_mixing'
         end if
         write(*,*)
         stop 'do_thermohaline_test'
         
      end subroutine do_thermohaline_test
      
#ifdef offload
      !dir$ end options
#endif

      end module


      program test_mlt
      use test_mlt_support
      use const_lib
      use chem_lib, only: chem_init
      
      character (len=32) :: my_mesa_dir

      my_mesa_dir = '../..'         
      call const_init(my_mesa_dir,ierr)     
   	if (ierr /= 0) then
   	   write(*,*) 'const_init failed'
   	   stop 1
   	end if        
         
      call crlibm_init

   	call chem_init('isotopes.data', ierr)
   	if (ierr /= 0) then
   	   write(*,*) 'chem_init failed'
   	   stop 1
   	end if
      
#ifdef offload
      !dir$ offload begin target(mic)
#endif
      
      cgrav = standard_cgrav
      mlt_partials1 => mlt_partials1_ary
      mlt_partials(1:num_mlt_partials,1:num_mlt_results) => &
         mlt_partials1(1:num_mlt_partials*num_mlt_results)
      mlt_partials1_alt => mlt_partials1_alt_ary
      mlt_partials_alt(1:num_mlt_partials,1:num_mlt_results) => &
         mlt_partials1_alt(1:num_mlt_partials*num_mlt_results)
      
      !call do1_test; stop
      !call do_semiconvection_test; stop
      
      !call do_thermohaline_test
      
      call Do_MLT_Test
      
#ifdef offload
      !dir$ end offload
#endif

      end program




