! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mlt_lib

      use mlt_def
      use const_def, only: dp
      
      implicit none

      contains 

#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif            

      ! temperature gradient calculation based on standard mixing length theory
      ! as, for example, described in Cox & Giuli, Chapter 14.  (see references below to C&G)
      
      subroutine mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada,  &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, dbg, &
            mixing_type, mlt_basics, mlt_partials1, ierr)

         use mlt,only:do_mlt_eval

         real(dp), intent(in) :: cgrav   ! gravitational constant
         real(dp), intent(in) :: m       ! enclosed mass
         real(dp), intent(in) :: r       ! radius
         real(dp), intent(in) :: T       ! temperature
         real(dp), intent(in) :: rho     ! density
         real(dp), intent(in) :: L       ! luminosity
         real(dp), intent(in) :: P       ! pressure
         real(dp), intent(in) :: chiRho  ! dlnP_dlnd at constant T and mu
         real(dp), intent(in) :: chiT    ! dlnP_dlnT at constant Rho and mu
         real(dp), intent(in) :: Cp      ! specific heat capacity at constant pressure
         real(dp), intent(in) :: Cv      ! specific heat capacity at constant volume
         real(dp), intent(in) :: csound  ! sound speed
         real(dp), intent(in) :: X ! hydrogen mass fraction
         real(dp), intent(in) :: opacity
         real(dp), intent(in) :: grada   ! dlnT/dlnP at constant entropy (adiabatic gradient)
         real(dp), intent(in) :: gradr_factor ! multiply standard gradr by this
            ! usually 1, but not when have rotation.
         
         real(dp), intent(in) :: gradL_composition_term
            ! gradL = grada + gradL_composition_term
            ! set gradL_composition_term = 0 for Schwarzschild criterion
         real(dp), intent(in) :: alpha_semiconvection
            ! determines efficiency of semiconvective mixing.
            ! values of 1d-1 to 1d-4 are used.
         character (len=*), intent(in) :: semiconvection_option 
            ! 'Langer_85 mixing; gradT = gradr'
            ! 'Langer_85' -- includes calculation of gradT following the paper
         real(dp), intent(in) :: thermohaline_coeff 
            ! 0 means no thermohaline mixing.
            ! > 0 determines efficiency of thermohaline mixing.
            ! values of up to 1000 are used.
         character (len=*), intent(in) :: thermohaline_option 
            ! determines which method to use for calculating thermohaline diffusion coef
            ! if == 'Traxler', use method of Traxler, Garaud, & Stellmach. ApJ Letters, 728:L29 (2011).
            ! if == 'Kippenhahn', use method of Kippenhahn, Ruschenplatt, & Thomas. 1980, A&A, 91, 175.
         integer, intent(in) :: dominant_iso_for_thermohaline
            ! chem id for dominant isotope in setting chemical gradient for thermohaline mixing.
            ! i.e. iso with largest local dX*(1+Z)/A
         real(dp), intent(in) :: mixing_length_alpha 
         logical, intent(in) :: alt_scale_height
            ! if false, then use the usual definition -- P/(g*rho)
            ! if true, use min of the usual and sound speed * hydro time scale, sqrt(P/G)/rho
         real(dp), intent(in) :: remove_small_D_limit
            ! if diffusion coeff D (cm^2/sec) is less than this limit
            ! then set D to zero and change the point to mixing_type == no_mixing.
         character (len=*), intent(in) :: MLT_option 
            ! options are:
            !    'none'       just return radiative values (gradT = gradr) with no mixing.
            !    'Cox'        MLT as developed in Cox & Giuli 1968, Chapter 14.
            !    'ML1'        Bohm-Vitense 1958
            !    'ML2'        Bohm and Cassinelli 1971
            !    'Mihalas'    Mihalas 1978, Kurucz 1979
            !    'Henyey'     Henyey, Vardya, and Bodenheimer 1965
            ! Values of the f1..f4 coefficients are taken from Table 1 of Ludwig et al. 1999, A&A, 346, 111
            ! with the following exception: their value of f3 for Henyey convection is f4/8 when it should be
            ! 8*f4, i.e., f3=32*pi**2/3 and f4=4*pi**2/3. f3 and f4 are related to the henyey y parameter, so
            ! for the 'Henyey' case they are set based on the value of Henyey_y_param.
         real(dp), intent(in) :: Henyey_y_param, Henyey_nu_param 
         real(dp), intent(in) :: prev_conv_vel, max_conv_vel, g_theta, dt
         real(dp), intent(in) :: tau
            ! if tau < 2/3, then, following B. Paczynski, 1969, Acta Astr., vol. 19, 1.  (eqn 14),
            ! the radiative temperature gradient, gradr, includes a factor to approximate the effect
            ! of the dilution of the radiation in the atmosphere.
            ! you should also use Henyey MLT in order to account for optically thin turbulent elements.
         logical, intent(in) :: dbg
         integer, intent(out) :: mixing_type ! none, convective, semiconvective, or thermohaline
            ! values defined in const_def
         real(dp), intent(out) :: mlt_basics(:) ! (num_mlt_results)
         real(dp), intent(out), pointer :: mlt_partials1(:) ! =(num_mlt_partials, num_mlt_results)
            ! e.g., mlt_partials(mlt_dlnT,mlt_gradT) has partial wrt lnT of gradT
         integer, intent(out) :: ierr

         call do_mlt_eval( &
            cgrav, m, r, T, rho, L, P, &
            chiRho, chiT, Cp, Cv, csound, X, opacity, grada,  &
            gradr_factor, gradL_composition_term, &
            alpha_semiconvection, semiconvection_option, &
            thermohaline_coeff, thermohaline_option, &
            dominant_iso_for_thermohaline, &
            mixing_length_alpha, alt_scale_height, remove_small_D_limit, &
            MLT_option, Henyey_y_param, Henyey_nu_param, &
            prev_conv_vel, max_conv_vel, g_theta, dt, tau, dbg, &
            mixing_type, mlt_basics, mlt_partials1, ierr)
         
      end subroutine mlt_eval
      
      
      subroutine mixing_type_str(which_mixing_type, mixing_type, ierr)
         integer, intent(in) :: which_mixing_type
         character (len=*), intent(out) :: mixing_type
         integer, intent(out) :: ierr
         ierr = 0
         
         if (which_mixing_type == no_mixing) then
            mixing_type = 'no_mixing'
            
         else if (which_mixing_type == convective_mixing) then
            mixing_type = 'convective'
            
         else if (which_mixing_type == softened_convective_mixing) then
            mixing_type = 'softened_convective'
            
         else if (which_mixing_type == overshoot_mixing) then
            mixing_type = 'overshoot'
            
         else if (which_mixing_type == semiconvective_mixing) then
            mixing_type = 'semiconvective'
            
         else if (which_mixing_type == thermohaline_mixing) then
            mixing_type = 'thermohaline'
            
         else if (which_mixing_type == rotation_mixing) then
            mixing_type = 'rotation'
            
         else if (which_mixing_type == minimum_mixing) then
            mixing_type = 'minimum'
            
         else if (which_mixing_type == anonymous_mixing) then
            mixing_type = 'anonymous'
            
         else
            ierr = -1
            mixing_type = ''
         end if 
         
      end subroutine mixing_type_str
      
#ifdef offload
      !dir$ end options
#endif
      
      end module mlt_lib

