      module test_kap_support
      
      use kap_lib
      use const_def, only: dp
      use crlibm_lib
      
      implicit none

      logical, parameter :: use_shared_data_dir = .true. ! if false, then test using local version data
      !logical, parameter :: use_shared_data_dir = .false.

      character (len=32) :: my_mesa_dir
      integer, parameter :: ionmax = 7
   
      integer :: nz
      real(dp), pointer, dimension(:) :: &
            x, y, z, c, n, o, ne, lgT, lgd, &
            old_lgKap, old_opacity, old_d_opacity_dlnd, old_d_opacity_dlnT
      real(dp), pointer, dimension(:,:) :: &
            new_lgKap, new_opacity, new_d_opacity_dlnd, new_d_opacity_dlnT
      integer, parameter :: num_new = 2
      
      character(len=256) :: kappa_file_prefix = 'gs98'
      character (len=256) :: kappa_CO_prefix = 'gn93_co'
      !character (len=256) :: kappa_lowT_prefix = 'lowT_Freedman11'
      character (len=256) :: kappa_lowT_prefix = 'lowT_fa05_gs98'
      
      
      logical :: cubic_interpolation_in_X = .true.
      logical :: cubic_interpolation_in_Z = .false.
      logical :: include_electron_conduction = .true.
      
      real(dp) :: kappa_blend_logT_upper_bdy = 4.1d0
      real(dp) :: kappa_blend_logT_lower_bdy = 3.93d0
      
      real(dp) :: kap_Type2_full_off_X = 0d0, kap_Type2_full_on_X = 0d0
      real(dp) :: kap_Type2_full_off_dZ = 0d0, kap_Type2_full_on_dZ = 0d0
      
      real(dp) :: kappa_type2_logT_lower_bdy = 3.80d0

      contains
      
      
      subroutine Do_One(quietly)
         logical, intent(in) :: quietly    
         integer :: handle     
         call setup(quietly, handle)  
         call Do_One_set_of_tests(quietly, handle)
         
         !return
         
         
#ifdef offload
         !dir$ offload target(mic) in(quietly, handle)
#endif
         call Do_One_set_of_tests(quietly, handle)
      end subroutine Do_One

#ifdef offload
      !dir$ attributes offload: mic :: Do_One_set_of_tests
#endif
      subroutine Do_One_set_of_tests(quietly, handle)
         logical, intent(in) :: quietly         
         integer, intent(in) :: handle     
         integer :: ierr
         ierr = 0         
         call test1(quietly, handle, 1, 'fixed metals', ierr)
         if (ierr /= 0) stop 1         
         call test1(quietly, handle, 2, 'C/O enhanced', ierr)
         if (ierr /= 0) stop 1      
      end subroutine Do_One_set_of_tests
      
      
      subroutine Do_Test1(quietly)
         logical, intent(in) :: quietly 
         integer :: ierr, handle
         ierr = 0         
         
         cubic_interpolation_in_Z = .false.
         !cubic_interpolation_in_Z = .true.
         
         !kap_Type2_full_off_X = 1
         !kap_Type2_full_on_X = 1
         
!         kap_Type2_full_off_dZ = 0.2; kap_Type2_full_on_dZ = 0.5
!         kap_Type2_full_off_dZ = 99; kap_Type2_full_on_dZ = 99
         
         kappa_file_prefix = 'a09'
         call setup(quietly, handle)
         
         call test1(quietly, handle, 0, '', ierr)
         stop

         call test1(quietly, handle, 0, '', ierr) ! testing
         call test1(quietly, handle, 1, 'fixed metals', ierr)
         call test1(quietly, handle, 2, 'C/O enhanced', ierr)
         call test1(quietly, handle, 3, 'op_mono', ierr)

         if (ierr /= 0) stop 1         
      end subroutine Do_Test1


      subroutine test1_op_mono(quietly, test_str, ierr)
         logical, intent(in) :: quietly         
         character (len=*), intent(in) :: test_str       
         integer, intent(out) :: ierr
         real(dp) :: &
            zbar, Z, xh, XC, XN, XO, XNe, frac, abar, &
            fC, fN, fO, fNe, Zbase, dXC, dXO, &
            frac_Type2, lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            logT, logRho, logR, kap, log10kap, dlnkap_dlnRho, dlnkap_dlnT
            
         logical :: CO_enhanced
         logical, parameter :: dbg = .false.
         integer :: chem_id(ionmax)
         
         include 'formats.dek'
         
         ierr = 0
         
         lnfree_e=0; d_lnfree_e_dlnRho=0; d_lnfree_e_dlnT=0
         Zbase = 0
         xc = 0d0
         xn = 0d0
         xo = 0d0
         xne = 0d0

            logT = 8d0
          logRho = 4.5d0
               z = 0.02d0
              xh = 0.65d0
                          
         
         call get_composition_info(Z, xh, abar, zbar, chem_id)
         
         frac_Type2 = 0d0
         call test_op_mono(ierr)

         log10kap = safe_log10_cr(kap)
         
         if (.not. quietly) then
            write(*,*) trim(test_str)
            write(*,*)
            call show_args
            call show_results
         end if
      
         contains
         
         subroutine test_op_mono(ierr)
            integer, intent(out) :: ierr
         
            real, pointer :: &
               umesh(:), ff(:,:,:,:), rs(:,:,:), ss(:,:,:,:)
            integer :: nel, nptot, ipe, nrad, i, iz(ionmax)
            real(dp) :: fap(ionmax), gp1(ionmax), xmass(ionmax)
            character (len=256) :: op_mono_data_path, op_mono_data_cache_filename
            logical, parameter :: screening = .true.
            
            include 'formats'
         
            ierr = 0
            
            op_mono_data_path = trim(my_mesa_dir) // '/data/kap_data/op_mono'
            op_mono_data_cache_filename = &
               trim(my_mesa_dir) // '/data/kap_data/op_mono_cache.bin'
            write(*,*) 'call load_op_mono_data'
            call load_op_mono_data( &
               op_mono_data_path, op_mono_data_cache_filename, ierr)
            if (ierr /= 0) return

            write(*,*) 'call get_op_mono_params'
            call get_op_mono_params(nptot, ipe, nrad)
            allocate( &
               umesh(nptot), ff(nptot,ipe,4,4), &
               rs(nptot,4,4), ss(nptot,nrad,4,4), stat=ierr)
            if (ierr /= 0) return
            
            write(*,*) 'call get_op_mono_args'
            call get_op_mono_args( &
   		      ionmax, xmass, 0d0, chem_id, &
   		      nel, iz, fap, ierr)
            if (ierr /= 0) then
               write(*,*) 'error in get_op_mono_args, ierr = ',ierr
               return
            end if

            do i=1,ionmax
               write(*,3) 'iz xmass fap', i, iz(i), xmass(i), fap(i)
            end do
         
            write(*,*) 'call op_mono_get_kap'
            call op_mono_get_kap( &
               ionmax, iz, fap, logT, logRho, screening, &
               log10kap, dlnkap_dlnRho, dlnkap_dlnT, gp1, &
               umesh, ff, rs, ss, ierr)
            deallocate(umesh, ff, rs, ss)
            if (ierr /= 0) then
               write(*,*) 'error in op_mono_get_kap, ierr = ',ierr
               return
            end if
            
            kap = exp10_cr(log10kap)
            
            write(*,*)
            write(*,1) 'kap', kap
            write(*,1) 'sum fap', sum(fap(:))
            write(*,*)

         end subroutine test_op_mono
         
      
         subroutine show_args
            1 format(a40,1pe26.16)
            write(*,*) 'CO_enhanced', CO_enhanced
            write(*,1) 'logT', logT
            write(*,1) 'logRho', logRho
            write(*,1) 'Z', Z
            write(*,1) 'Zbase', Zbase
            write(*,1) 'zbar', zbar
            write(*,1) 'xh', xh
            write(*,1) 'xc', xc
            write(*,1) 'xn', xn
            write(*,1) 'xo', xo
            write(*,1) 'xne', xne
            write(*,1) 'lnfree_e', lnfree_e
            write(*,*)
         end subroutine show_args
         
      
         subroutine show_results
            use utils_lib
            1 format(a40,1pe26.16)
            write(*,1) 'log10kap', log10kap
            write(*,1) 'dlnkap_dlnRho', dlnkap_dlnRho
            write(*,1) 'dlnkap_dlnT', dlnkap_dlnT
            write(*,*)
            write(*,1) 'kap', kap
            write(*,1) 'dkap_dlnd', dlnkap_dlnRho*kap
            write(*,1) 'dkap_dlnT', dlnkap_dlnT*kap
            write(*,*)
            write(*,1) 'frac_Type2', frac_Type2
            write(*,*)
            if (is_bad_num(log10kap)) then
               write(*,*) 'bad log10kap'
            end if
         end subroutine show_results
         
      end subroutine test1_op_mono
   
   
#ifdef offload
      !dir$ attributes offload: mic :: test1
#endif
      subroutine test1(quietly, handle, which, test_str, ierr)
         logical, intent(in) :: quietly         
         integer, intent(in) :: handle, which 
         character (len=*), intent(in) :: test_str       
         integer, intent(out) :: ierr
         real(dp) :: &
            zbar, Z, xh, XC, XN, XO, XNe, frac, abar, &
            fC, fN, fO, fNe, Zbase, dXC, dXO, &
            frac_Type2, lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
            logT, logRho, logR, kap, log10kap, dlnkap_dlnRho, dlnkap_dlnT
            
         logical :: CO_enhanced
         logical, parameter :: dbg = .false.
         integer :: chem_id(ionmax)
         include 'formats.dek'
         
         ierr = 0
         
         lnfree_e=0; d_lnfree_e_dlnRho=0; d_lnfree_e_dlnT=0
         Zbase = 0
         xc = 0d0
         xn = 0d0
         xo = 0d0
         xne = 0d0
         
         if (which == 1) then ! not enhanced in C/O

            logT =    6
            logRho =   -6
            Z =    0.018d0
            xh =    0.7d0
            CO_enhanced = .false.
             
         else if (which == 2) then ! C/O enhanced
         
            CO_enhanced = .true.
            logT =    6
            logRho =   -6
            Zbase = 0.018d0
            dXC = 0.021d0
            dXO = 0.019d0
            xh = 0.7d0
            fC = 0.173312d0
            fN = 0.053152d0
            fO = 0.482398d0
            fNe = 0.098668d0
            Z = Zbase + dXC + dXO
            xc = dXC + fC*Zbase
            xn = fN*Zbase
            xo = dXO + fO*Zbase
            xne = fNe*Zbase
             
         else if (which == 3) then ! op_mono

            stop 1
                          
         else ! DEBUGGING
 
            !CO_enhanced = .true.
            CO_enhanced = .false.

              xh        =    0d0
               Z        =    2d-2
           Zbase        =    2d-2
              XC        =    0
              XN        =    0
              XO        =    0
             XNe        =    0
          logRho        =    4.1511856827839102  
            logT        =    7.9665307474722908    
            
            lnfree_e = -0.69463592767715454     

            write(*,*) 'test'
             
         end if
         
         call get_composition_info(Z, xh, abar, zbar, chem_id)

         if (which == 0) &
            zbar        =    2.3584494590759277

         if (CO_enhanced) then
            call kap_get_Type2( &
                  handle, zbar, xh, Z, Zbase, XC, XN, XO, XNe, logRho, logT, &
                  lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
                  frac_Type2, kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         else
            frac_Type2 = 0d0
            call kap_get_Type1( &
                  handle, zbar, xh, Z, logRho, logT, &
                  lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
                  kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         end if
      

         log10kap = safe_log10_cr(kap)
         
         if (.not. quietly) then
            write(*,*) 'test number', which
            write(*,*) trim(test_str)
            write(*,*)
            call show_args
            call show_results
         end if
      
         contains
         
      
#ifdef offload
         !dir$ attributes offload: mic :: show_args
#endif
         subroutine show_args
            1 format(a40,1pe26.16)
            write(*,*) 'CO_enhanced', CO_enhanced
            write(*,1) 'logT', logT
            write(*,1) 'logRho', logRho
            write(*,1) 'Z', Z
            write(*,1) 'Zbase', Zbase
            write(*,1) 'zbar', zbar
            write(*,1) 'xh', xh
            write(*,1) 'xc', xc
            write(*,1) 'xn', xn
            write(*,1) 'xo', xo
            write(*,1) 'xne', xne
            write(*,1) 'lnfree_e', lnfree_e
            write(*,*)
         end subroutine show_args
         
      
#ifdef offload
         !dir$ attributes offload: mic :: show_results
#endif
         subroutine show_results
            use utils_lib
            1 format(a40,1pe26.16)
            write(*,1) 'log10kap', log10kap
            write(*,1) 'dlnkap_dlnRho', dlnkap_dlnRho
            write(*,1) 'dlnkap_dlnT', dlnkap_dlnT
            write(*,*)
            write(*,1) 'kap', kap
            write(*,1) 'dkap_dlnd', dlnkap_dlnRho*kap
            write(*,1) 'dkap_dlnT', dlnkap_dlnT*kap
            write(*,*)
            write(*,1) 'frac_Type2', frac_Type2
            write(*,*)
            if (is_bad_num(log10kap)) then
               write(*,*) 'bad log10kap'
            end if
         end subroutine show_results

      end subroutine test1
      

      subroutine setup(quietly, handle)
         use chem_lib
         use const_lib
         logical, intent(in) :: quietly
         integer, intent(out) :: handle
         
         character (len=256) :: kap_dir, opal_dir, cbeg_ferg
         integer :: ierr
         logical, parameter :: use_cache = .true.
         
         my_mesa_dir = '../..'         
         call const_init(my_mesa_dir,ierr)     
      	if (ierr /= 0) then
      	   write(*,*) 'const_init failed'
      	   stop 1
      	end if        
         
         call crlibm_init

      	call chem_init('isotopes.data', ierr)
      	if (ierr /= 0) then
      	   write(*,*) 'chem_init failed'
      	   stop 1
      	end if

         call kap_init( &
            kappa_file_prefix, kappa_CO_prefix, kappa_lowT_prefix, &
            kappa_blend_logT_upper_bdy, kappa_blend_logT_lower_bdy, &
            kappa_type2_logT_lower_bdy, use_cache, '', ierr) 
         if (ierr /= 0) stop 1
         
         handle = alloc_kap_handle(ierr) 
         if (ierr /= 0) stop 1
         
         call kap_set_choices( &
            handle, cubic_interpolation_in_X, cubic_interpolation_in_Z, &
            include_electron_conduction, &
            kap_Type2_full_off_X, kap_Type2_full_on_X, &
            kap_Type2_full_off_dZ, kap_Type2_full_on_dZ, &
            ierr)
         if (ierr /= 0) stop 1
      
      end subroutine setup


      subroutine finish(handle)
         integer, intent(in) :: handle
         call free_kap_handle(handle)
         call kap_shutdown
      end subroutine finish
      
      
#ifdef offload
      !dir$ attributes offload: mic :: get_composition_info
#endif
      subroutine get_composition_info(Z, X, abar, zbar, chem_id)
         use chem_def, only: ih1, ihe4, ic12, in14, io16, ine20, img24
         real(dp), intent(in) :: Z, X
         real(dp), intent(out) :: abar, zbar
         integer, intent(out) :: chem_id(ionmax)

         real(dp) :: Y
         real(dp) :: aion(ionmax),zion(ionmax),xmass(ionmax),ymass(ionmax),zbarxx,ytot1
         integer :: iz(ionmax)

         real(dp), parameter :: Zfrac_C = 0.173312d0
         real(dp), parameter :: Zfrac_N = 0.053177d0
         real(dp), parameter :: Zfrac_O = 0.482398d0
         real(dp), parameter :: Zfrac_Ne = 0.098675d0
         real(dp), parameter :: Zfrac_Mg = 1d0 - (Zfrac_C + Zfrac_N + Zfrac_O + Zfrac_Ne)
         
         integer :: i
         integer :: h1,he4,c12,n14,o16,ne20,mg24

         
         Y = 1 - (X+Z)
         if (Y < 0) then ! adjust XC and XO
            write(*,*) 'bad args to get_composition_info'
            stop 1
         end if
      
         h1        = 1
         iz(h1) = 1
         zion(h1)  = 1.0d0
         aion(h1)  = 1.0d0 
         xmass(h1) = X
         chem_id(h1) = ih1

         he4        = 2
         iz(he4) = 2
         zion(he4)  = 2.0d0
         aion(he4)  = 4.0d0 
         xmass(he4) = Y
         chem_id(he4) = ihe4

         c12        = 3
         iz(c12) = 6
         zion(c12)  = 6.0d0
         aion(c12)  = 12.0d0 
         xmass(c12) = Z * Zfrac_C
         chem_id(c12) = ic12

         n14        = 4
         iz(n14) = 7
         zion(n14)  = 7.0d0
         aion(n14)  = 14.0d0 
         xmass(n14) = Z * Zfrac_N
         chem_id(n14) = in14

         o16        = 5
         iz(o16) = 8
         zion(o16)  = 8.0d0
         aion(o16)  = 16.0d0 
         xmass(o16) = Z * Zfrac_O
         chem_id(o16) = io16

         ne20       = 6
         iz(ne20) = 10
         zion(ne20)  = 10.0d0
         aion(ne20)  = 20.0d0 
         xmass(ne20) = Z * Zfrac_Ne
         chem_id(ne20) = ine20

         mg24       = 7
         iz(mg24) = 12
         zion(mg24)  = 12.0d0
         aion(mg24)  = 24.0d0 
         xmass(mg24) = Z * Zfrac_Mg
         chem_id(mg24) = img24

         zbarxx  = 0.0d0
         ytot1   = 0.0d0
         do i=1,ionmax
            ymass(i) = xmass(i)/aion(i)
            ytot1    = ytot1 + ymass(i)
            zbarxx   = zbarxx + zion(i) * ymass(i)
         enddo
         abar   = 1.0d0/ytot1
         zbar   = zbarxx * abar

      end subroutine get_composition_info
      
      
      subroutine write_plot_data(handle)
         use chem_def
         integer, intent(in) :: handle
         integer, parameter :: io_unit0 = 40
         character (len=256) :: dir
         real(dp), pointer, dimension(:,:,:) :: output_values, co_output_values
         real(dp) :: kap_elect, Z, Zbase, xh, XC, XN, XO, XNe, abar, zbar, &
            logT_max, logT_min, logRho_max, logRho_min, dlogT, dlogRho, logT, logRho
         integer :: logT_points, logRho_points, num_out, io_params, io_rho, io_tmp, &
            io_first, io_last, i, j, k, ierr, io, chem_id(ionmax)
         logical, parameter :: compare_to_CO = .false.
         
         dir = 'plot_data'
         write(*,*) 'write data for opacity plots to ' // trim(dir)

         xh = 0.6905d0
         Z = 0.020113d0
         XC = 0.0
         XN = 0.0
         XO = 0.0
         XNe = 0.0
         Zbase = -1

         logT_points = 101
         logRho_points = 101
         
         logT_max = 9.2d0
         logT_min = 1.9d0
         logRho_max = 14d0
         logRho_min = -14d0
         
         logT_max = 5.7d0
         logT_min = 4.0d0
         logRho_max = -6.5d0
         logRho_min = -11d0
         
         logT_max = 4.5d0
         logT_min = 3.75d0
         logRho_max = 0d0
         logRho_min = -9.5d0
         
         if (.false.) then
         
         
         
                                                  z =    2.67d-2
                                                  zbase = 1d-5
                                                  xh =    2.8d-2
                                                  xc =    2d-3
                                                  xo =    3.56d-3

            logT_max = 5.67
            logT_min = 5.62
            logRho_max = -5.90
            logRho_min = -6.05

            logT_points = 201
            logRho_points = 201
            
         end if

         call get_composition_info(z, xh, abar, zbar, chem_id)
         
         kap_elect = 0.2d0*(1 + xh)
         
         io_params = io_unit0
         io_rho = io_unit0+1
         io_tmp = io_unit0+2
         io_first = io_unit0+3
         call Open_Plot_Outfiles(io_first, io_last, io_params, io_rho, io_tmp, dir, compare_to_CO)
         num_out = io_last - io_first + 1
         
         allocate(output_values(logRho_points,logT_points,num_out), &
            co_output_values(logRho_points,logT_points,num_out))

         write(io_params, '(99(f16.6,6x))') Z, xh, XC, XO
         write(io_params, '(99(i16,6x))') logRho_points, logT_points
         close(io_params)


         dlogT = (logT_max - logT_min)/(logT_points-1)
         dlogRho = (logRho_max - logRho_min)/(logRho_points-1)
         
         ierr = 0
         
         do j=1, logT_points
            logT = logT_min + dlogT*(j-1)
            do i=1,logRho_points
               logRho = logRho_min + dlogRho*(i-1)
               if (xc /= 0 .or. xo /= 0) then
                  call do1_CO_plot_data( &
                     handle, logRho, logT, zbar, xh, Z, Zbase, XC, XN, XO, XNe, &
                     output_values, i, j, compare_to_CO, ierr)
               else
                  call do1_plot_data( &
                     handle, logRho, logT, zbar, xh, Z, output_values, i, j, compare_to_CO, ierr)
                  if (compare_to_CO) then
                     call do1_CO_plot_data( &
                        handle, logRho, logT, zbar, xh, Z, Zbase, XC, XN, XO, XNe, &
                        co_output_values, i, j, compare_to_CO, ierr)
                  end if
               end if
               if (ierr /= 0) exit
            end do
         end do

 01   format(e30.22)
         ! write out the results
         do j=1,logT_points
            write(io_tmp,01) logT_min + dlogT*(j-1)
         end do
         close(io_tmp)

         do i=1,logRho_points
            write(io_rho,01) logRho_min + dlogRho*(i-1)
         enddo
         close(io_rho)
         
         if (compare_to_CO) then
            write(*,*) 1
            write(io_first,'(e14.6)') output_values(1:logRho_points,1:logT_points,1)
            write(*,*) 2
            write(io_first+1,'(e14.6)') co_output_values(1:logRho_points,1:logT_points,1)
            write(*,*) 3
            write(io_first+2,'(e14.6)') &
               output_values(1:logRho_points,1:logT_points,1) - &
               co_output_values(1:logRho_points,1:logT_points,1)
         else
            do k = 1, num_out
               write(*,*) k
               write(io_first+k-1,'(e14.6)') output_values(1:logRho_points,1:logT_points,k)
            end do
         end if
      
         do io=io_first,io_last
            close(io)
         end do
         
         deallocate(output_values, co_output_values)
         
      end subroutine write_plot_data
      

      subroutine Open_Plot_Outfiles( &
            io_first, io_last, io_params, io_rho, io_tmp, dir, compare_to_CO)
         integer, intent(IN) :: io_first, io_params, io_rho, io_tmp
         integer, intent(OUT) :: io_last
         character (len=256), intent(IN) :: dir
         logical, intent(in) :: compare_to_CO
         character (len=256) :: fname
         integer :: io
         
         fname = trim(dir) // '/params.data'
         open(unit=io_params,file=trim(fname))
         
         fname = trim(dir) // '/logRho.data'
         open(unit=io_rho,file=trim(fname))
         
         fname = trim(dir) // '/logT.data'
         open(unit=io_tmp,file=trim(fname))
         
      
         if (compare_to_CO) then
            io = io_first
            fname = trim(dir) // '/kap.data'
            open(unit=io,file=trim(fname))
            fname = trim(dir) // '/kapCO.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/kap_sub_kapCO.data'
            io = io+1; open(unit=io,file=trim(fname))
         else 
            io = io_first
            fname = trim(dir) // '/logK.data'
            open(unit=io,file=trim(fname))
            fname = trim(dir) // '/dlogK_dlogRho.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/dlogK_dlogT.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/logKec.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/dlogKec_dlogRho.data'
            io = io+1; open(unit=io,file=trim(fname))
            fname = trim(dir) // '/dlogKec_dlogT.data'
            io = io+1; open(unit=io,file=trim(fname))
         end if
            
         io_last = io
      
      end subroutine Open_Plot_Outfiles
         
      
      
      subroutine do1_plot_data( &
            handle, lgd, lgT, zbar, xh, Z, output_values, i, j, compare_to_CO, ierr)
         use utils_lib, only: is_bad_num
         real(dp), intent(in) :: lgd, lgT, zbar
         integer, intent(in) :: handle, i, j
         real(dp), intent(in) :: xh, Z
         real(dp), intent(out) :: output_values(:,:,:)
         logical, intent(in) :: compare_to_CO
         integer, intent(out) :: ierr
         
         real(dp) :: kap, dlnkap_dlnRho, dlnkap_dlnT, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
         ierr = 0
         lnfree_e=0; d_lnfree_e_dlnRho=0; d_lnfree_e_dlnT=0
         call kap_get_Type1( &
                  handle, zbar, xh, Z, lgd, lgT, &
                  lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
                  kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         if (ierr /= 0) then
            output_values(i,j,:) = -99
            ierr = 0
            return
         end if

         
         if (is_bad_num(kap)) then
            write(*,*) 'kap', kap
            stop 'do1_plot_data'
         end if

         if (compare_to_CO) then
            output_values(i,j,1) = kap
         else
            output_values(i,j,1) = safe_log10_cr(kap)
         end if
         output_values(i,j,2) = dlnkap_dlnRho
         output_values(i,j,3) = dlnkap_dlnT

         call kap_get_elect_cond_opacity( &
                  zbar, lgd, lgT, &
                  kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         if (ierr /= 0) then
            output_values(i,j,4) = -99
            output_values(i,j,5) = -99
            output_values(i,j,6) = -99
            ierr = 0
            return
         end if
         
         if (is_bad_num(kap)) then
            write(*,*) 'kap_get_elect_cond_opacity kap', kap
            stop 'do1_plot_data'
         end if

         output_values(i,j,4) = safe_log10_cr(kap)
         output_values(i,j,5) = dlnkap_dlnRho
         output_values(i,j,6) = dlnkap_dlnT

      end subroutine do1_plot_data
         
      
      subroutine do1_CO_plot_data( &
            handle, lgd, lgT, zbar, xh, Z, Zbase, XC, XN, XO, XNe, &
            output_values, i, j, compare_to_CO, ierr)
         use utils_lib, only: is_bad_num
         real(dp), intent(in) :: lgd, lgT, zbar
         integer, intent(in) :: handle, i, j
         real(dp), intent(in) :: xh, Z, Zbase, XC, XN, XO, XNe
         real(dp), intent(out) :: output_values(:,:,:)
         logical, intent(in) :: compare_to_CO
         integer, intent(out) :: ierr
         real(dp) :: frac_Type2, kap, dlnkap_dlnRho, dlnkap_dlnT, &
            lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT
         ierr = 0
         lnfree_e=0; d_lnfree_e_dlnRho=0; d_lnfree_e_dlnT=0
         call kap_get_Type2( &
                  handle, zbar, xh, Z, Zbase, XC, XN, XO, XNe, lgd, lgT, &
                  lnfree_e, d_lnfree_e_dlnRho, d_lnfree_e_dlnT, &
                  frac_Type2, kap, dlnkap_dlnRho, dlnkap_dlnT, ierr)
         if (ierr /= 0) then
            output_values(i,j,1) = -99
            output_values(i,j,2) = -99
            output_values(i,j,3) = -99
            ierr = 0
            return
         end if
         
         if (is_bad_num(kap)) then
            write(*,*) 'kap', kap, lgd, lgT, zbar, xh, Z, xc, xo
            stop 'do1_CO_plot_data'
         end if
         
         if (is_bad_num(dlnkap_dlnRho)) then
            write(*,*) 'dlnkap_dlnRho', dlnkap_dlnRho, lgd, lgT, zbar, xh, Z, xc, xo
            stop 'do1_CO_plot_data'
         end if
         
         if (is_bad_num(dlnkap_dlnT)) then
            write(*,*) 'dlnkap_dlnT', dlnkap_dlnT, lgd, lgT, zbar, xh, Z, xc, xo
            stop 'do1_CO_plot_data'
         end if

         if (compare_to_CO) then
            output_values(i,j,1) = kap
         else
            output_values(i,j,1) = safe_log10_cr(kap)
         end if
         output_values(i,j,2) = dlnkap_dlnRho
         output_values(i,j,3) = dlnkap_dlnT
      end subroutine do1_CO_plot_data
      

      end module test_kap_support

