! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module load_kap
      
      use kap_def
      use load_CO_kap, only: Setup_Kap_CO_Tables, min_version
      use const_def, only: dp
      use crlibm_lib

      implicit none

      
      logical, parameter :: dbg = .false.

      logical, parameter :: dbg_cache = .false.

      contains

#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif            
      
      subroutine Setup_Kap_Tables( &
            file_prefix, CO_prefix, lowT_prefix, &
            blend_logT_upper_bdy, blend_logT_lower_bdy, &
            type2_logT_lower_bdy, use_cache, load_on_demand, ierr)
         use const_def, only: mesa_data_dir
         use condint, only: init_potekhin
         character (*), intent(in) :: &
            file_prefix, CO_prefix, lowT_prefix
         real(dp), intent(in) :: blend_logT_upper_bdy, blend_logT_lower_bdy, &
            type2_logT_lower_bdy
         logical, intent(in) :: use_cache, load_on_demand
         integer, intent(out) :: ierr
      
         integer :: iz, ix, num_Xs
         real(dp) :: X, Z
         
         ierr = 0
         
         kap_dir = trim(mesa_data_dir) // '/kap_data'
         if (len_trim(file_prefix) > 0) kap_prefix = file_prefix
         if (len_trim(CO_prefix) > 0) kap_CO_prefix = CO_prefix
         if (len_trim(lowT_prefix) > 0) kap_lowT_prefix = lowT_prefix
         if (blend_logT_upper_bdy > 0) kap_blend_logT_upper_bdy = blend_logT_upper_bdy
         if (blend_logT_lower_bdy > 0) kap_blend_logT_lower_bdy = blend_logT_lower_bdy
         if (type2_logT_lower_bdy > 0) kap_type2_logT_lower_bdy = type2_logT_lower_bdy
         kap_use_cache = use_cache

         call Setup_Kap_CO_Tables(use_cache, load_on_demand, ierr)
         if (ierr /= 0) return
         
         call setup(.false., kap_z_tables)
         call setup(.true., kap_lowT_z_tables)
         
         call init_potekhin(ierr)
         
         contains
         
#ifdef offload
         !dir$ attributes offload: mic :: setup
#endif      
         subroutine setup(lowT_flag, z_tables)
            logical, intent(in) :: lowT_flag
            type (Kap_Z_Table), dimension(:), pointer :: z_tables
            integer :: iz
            if (lowT_flag .and. kap_using_lowT_Freedman) then
               if (.not. associated(z_tables)) then
                  allocate(z_tables(num_kap_Freedman_Zs), STAT=ierr)
                  if (ierr /= 0) return
                  do iz = 1, num_kap_Freedman_Zs
                     z_tables(iz)% lowT_flag = lowT_flag
                     z_tables(iz)% Z = kap_Freedman_Zs(iz)
                     allocate(z_tables(iz)% x_tables(1), STAT=ierr)
                     if (ierr /= 0) return
                  end do
               end if
               do iz = 1, num_kap_Freedman_Zs
                  Z = kap_Freedman_Zs(iz)
                  num_Xs = 1
                  z_tables(iz)% num_Xs = num_Xs
                  X = 0.00
                  ix = 1
                  call load_one(z_tables, iz, ix, X, Z, load_on_demand, ierr)
               end do
            else
               if (.not. associated(z_tables)) then
                  allocate(z_tables(num_kap_Zs), STAT=ierr)
                  if (ierr /= 0) return
                  do iz = 1, num_kap_Zs
                     z_tables(iz)% lowT_flag = lowT_flag
                     z_tables(iz)% Z = kap_Zs(iz)
                     allocate(z_tables(iz)% x_tables(num_kap_Xs), STAT=ierr)
                     if (ierr /= 0) return
                  end do
               end if
               do iz = 1, num_kap_Zs
                  Z = kap_Zs(iz)
                  num_Xs = num_kap_Xs_for_this_Z(iz)
                  z_tables(iz)% num_Xs = num_Xs
                  do ix = 1, num_Xs
                     if (ix < num_Xs) then
                        X = kap_Xs(ix)
                     else
                        X = 1 - Z
                     end if
                     call load_one(z_tables, iz, ix, X, Z, load_on_demand, ierr)
                     if (ierr /= 0) return
                  end do
               end do
            end if
        
         end subroutine setup

      end subroutine Setup_Kap_Tables
      
         
      subroutine load_one(z_tables, iz, ix, X, Z, read_later, ierr)
         use utils_lib
         type (Kap_Z_Table), dimension(:), pointer :: z_tables
         integer, intent(in) :: iz, ix
         real(dp), intent(in) :: X, Z
         logical, intent(in) :: read_later
         integer, intent(out) :: ierr

         character (len=256) :: fname, filename, cache_filename
         integer :: iounit1, iounit2
         
         ierr = 0
         iounit1 = alloc_iounit(ierr); if (ierr /= 0) return
         iounit2 = alloc_iounit(ierr); if (ierr /= 0) return
                  
         call Get_Filenames(z_tables, Z, X, kap_dir, fname, filename, cache_filename)
         
         call Prepare_Kap_X_Table( &
            z_tables, iz, z_tables(iz)% x_tables, ix, X, Z, read_later, &
            fname, filename, cache_filename, iounit1, iounit2, ierr)
            
         call free_iounit(iounit2)
         call free_iounit(iounit1)
         if (ierr /= 0) return

      end subroutine load_one
      
      
      subroutine Prepare_Kap_X_Table( &
                  z_tables, iz, x_tables, ix, X_in, Z_in, read_later, &
                  fname, filename, cache_filename, io_unit, cache_io_unit, ierr)
         type (Kap_Z_Table), dimension(:), pointer :: z_tables
         ! allocates the arrays and stores data
         type (Kap_X_Table), dimension(:), pointer :: x_tables
         integer, intent(in) :: iz, ix
         real(dp), intent(in) :: X_in, Z_in ! expected contents of the file
         logical, intent(in) :: read_later
         character (*), intent(in) :: fname, filename, cache_filename
         integer, intent(in) :: io_unit, cache_io_unit
         integer, intent(out) :: ierr
         
         real :: X, Z
         integer :: ios, form, version, n, num_logRs, num_logTs, status, i
         real :: xin, zz, logR_min, logR_max, logT_min, logT_max, xErr, zErr
         character (len=1000) :: message
         real(dp), target :: vec_ary(50)
         real(dp), pointer :: vec(:)
         integer :: nvec
         real, parameter :: tiny = 1d-6

         ierr = 0
         vec => vec_ary
         X = X_in
         Z = Z_in
         
         open(unit=io_unit,file=trim(filename),action='read',status='old',iostat=ios)
         if (ios /= 0) then
            !write(*,*) 'load kap tables ' // trim(filename)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*) 'NOTICE: missing kap data ' // trim(filename)
            write(*,*) 
            write(*,*) 'Please check the validity of the kap_prefix string for this file.'
            write(*,*) 
            write(*,*) 'If it is okay, you may need to install new kap data.'
            write(*,*) 'To do that, remove the directory mesa/data/kap_data,'
            write(*,*) 'and rerun the mesa ./install script.'
            write(*,*)
            stop 1
         end if

         version = -1
         read(io_unit,*,iostat=ierr) ! skip
         if (ierr == 0) then
            read(io_unit,*,iostat=ierr) ! skip
            if (ierr == 0) then
               read(io_unit,'(a)',iostat=ierr) message
               if (ierr == 0) call str_to_vector(message, vec, nvec, ierr)
               if (nvec < 10) ierr = -1
               if (ierr == 0) then
                  form = int(vec(1))
                  version = int(vec(2))
                  xin = vec(3)
                  zz = vec(4)
                  num_logRs = int(vec(5))
                  logR_min = vec(6)
                  logR_max = vec(7)
                  num_logTs = int(vec(8))
                  logT_min = vec(9)
                  logT_max = vec(10)
               end if
            end if
         end if
         
         if (ierr /= 0 .or. version < min_version) then
            write(*,*) 'load kap tables ' // trim(filename)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*) 'NOTICE: you need to install a new verion of the kap data.'
            write(*,*) 'Please remove the directory mesa/data/kap_data,'
            write(*,*) 'and rerun the mesa ./install script.'
            write(*,*)
            if (ierr /= 0) write(*,*) 'ierr', ierr
            if (version < min_version) &
               write(*,*) 'version < min_version', version, min_version
            write(*,*) 'form', form
            write(*,*) 'xin', xin
            write(*,*) 'zz', zz
            write(*,*) 'num_logRs', num_logRs
            write(*,*) 'logR_min', logR_min
            write(*,*) 'logR_max', logR_max
            write(*,*) 'num_logTs', num_logTs
            write(*,*) 'logT_min', logT_min
            write(*,*) 'logT_max', logT_max
            stop 1
         end if

         if (form /= kap_table_fixed_metal_form) then
            stop 'form /= kap_table_fixed_metal_form'
         end if
         
         call Setup_Kap_X_Table(ierr)
         if (ierr /= 0) return
         
         if (read_later) then
            close(io_unit)
            return
         end if
         
         if (kap_use_cache) then
            ios = 0
            if (dbg_cache) then
               open(unit=cache_io_unit,file=trim(cache_filename),action='read', &
                     status='old',iostat=ios)
            else
               open(unit=cache_io_unit,file=trim(cache_filename),action='read', &
                     status='old',iostat=ios,form='unformatted')
            end if
            if (ios == 0) then ! try reading the cached data
               !write(*,'(a)') 'loading ' // trim(cache_filename)
               call Read_Kap_X_Table(cache_io_unit, .true., ierr)
               close(cache_io_unit)
               if (ierr == 0) then
                  z_tables(iz)% x_tables(ix)% not_loaded_yet = .false.
                  close(io_unit)
                  return
               end if
               ierr = 0
            else
               !write(*,*) 'failed to open ' // trim(cache_filename)
            end if
         end if
         
         !write(*,*) 'loading ' // trim(filename)
         call Read_Kap_X_Table(io_unit, .false., ierr)
         close(io_unit)
         if (ierr /= 0) then
            write(*,*) 'failed in Read_Kap_X_Table ' // trim(filename)
            stop 1
            return
         end if

         z_tables(iz)% x_tables(ix)% not_loaded_yet = .false.
         
         if (.not. kap_use_cache) return
         
         if (dbg_cache) then
            open(unit=cache_io_unit,file=trim(cache_filename),iostat=ios,&
               action='write')
         else
            open(unit=cache_io_unit,file=trim(cache_filename),iostat=ios,&
               action='write',form='unformatted')
         end if
         
         if (ios == 0) then
            if (dbg) write(*,'(a)') 'write ' // trim(cache_filename)
            call Write_Kap_X_Table_Cache(z_tables(iz)% x_tables, ix, cache_io_unit,  version)
            close(cache_io_unit)
         end if


         contains
         
         
#ifdef offload
         !dir$ attributes offload: mic :: Setup_Kap_X_Table
#endif      
         subroutine Setup_Kap_X_Table(ierr)
            integer, intent(out) :: ierr
            
            integer :: i
         
            xErr = abs(xin - X); zErr = abs(zz - Z)
            if (xErr > tiny .or. zErr > tiny) then
               ierr = -1
               write(*,*) 'bug in file ' // trim(filename), xErr, zErr
               write(*,*) 'xErr > tiny', xErr > tiny
               write(*,*) 'zErr > tiny', zErr > tiny
               return
            end if
         
            x_tables(ix)% not_loaded_yet = .true.
            !x_tables(ix)% kap_data_dir = kap_dir
            !x_tables(ix)% fname = fname
            !x_tables(ix)% use_cache = kap_use_cache
            
            x_tables(ix)% X = X
            x_tables(ix)% Z = Z

            x_tables(ix)% logR_min = logR_min
            x_tables(ix)% logR_max = logR_max
            x_tables(ix)% num_logRs = num_logRs
            nullify(x_tables(ix)% logRs)

            x_tables(ix)% logT_min = logT_min
            x_tables(ix)% logT_max = logT_max
            x_tables(ix)% num_logTs = num_logTs
            nullify(x_tables(ix)% logTs)
         
            nullify(x_tables(ix)% kap1) ! allocate when read the data

         end subroutine Setup_Kap_X_Table


#ifdef offload
         !dir$ attributes offload: mic :: Read_Kap_X_Table
#endif      
         subroutine Read_Kap_X_Table(io_unit, reading_cache, ierr)
            integer, intent(in) :: io_unit ! use this for file access
            logical, intent(in) :: reading_cache
            integer, intent(out) :: ierr
            
            character (len=1000) :: message
            character (len=1) :: char
            integer :: i, j, c_version, c_num_logRs, c_num_logTs
            real :: c_xin, c_zz, c_logR_min, c_logR_max, c_logT_min, c_logT_max
            real :: kap_logKs(num_logRs), logT
            real, pointer :: kap(:,:,:), kap1(:)
            real(dp), target :: vec_ary(100)
            real(dp), pointer :: vec(:)
            integer :: nvec
            
            include 'formats'

            vec => vec_ary

            if (reading_cache) then
               
               ios = 0
               if (dbg_cache) then
                  write(*,*) 'io_unit', io_unit
                  read(io_unit, *, iostat=ios) c_version, c_num_logRs, c_num_logTs
               else
                  read(io_unit, iostat=ios) c_version, c_num_logRs, c_num_logTs
               end if
               if (ios /= 0 .or. c_version /= version) then
                  ierr = 1
                  if (ios /= 0) then
                     write(*,*) 'cache failed in read 1'
                  else if (c_version /= version) then
                     write(*,*) 'cache failed for c_version /= version'
                     write(*,*) 'c_version', c_version
                     write(*,*) 'version', version
                  else if (c_num_logRs /= num_logRs .or. c_num_logTs /= num_logTs) then
                  if (c_num_logRs /= num_logRs) write(*,*) 'cache failed for c_num_logRs /= num_logRs'
                  if (c_num_logTs /= num_logTs) write(*,*) 'cache failed for c_num_logTs /= num_logTs'
                  end if
                  return
               end if
            
               read(io_unit, iostat=ios) &
                  c_xin, c_zz, c_logR_min, c_logR_max, &
                  c_logT_min, c_logT_max
               if (ios /= 0) then
                  ierr = 1
                  if (ios /= 0) write(*,*) 'cache failed in read 2'
               end if
               
            end if

            xErr = abs(xin - X); zErr = abs(zz - Z)
            if (xErr > tiny .or. zErr > tiny) then
               if (reading_cache) then
                  if (xErr > tiny) write(*,*) 'cache failed for xErr > tiny'
                  if (zErr > tiny) write(*,*) 'cache failed for zErr > tiny'
                  ierr = 1; return
               end if
               ierr = -1
               return
            end if
            
            allocate(x_tables(ix)% logRs(num_logRs), x_tables(ix)% logTs(num_logTs), &
               x_tables(ix)% kap1(sz_per_Kap_point*num_logRs*num_logTs), STAT=status)
            if (status .ne. 0) then
               ierr = -1
               return
            end if
            
            kap1 => x_tables(ix)% kap1
            kap(1:sz_per_Kap_point,1:num_logRs,1:num_logTs) => &
               kap1(1:sz_per_Kap_point*num_logRs*num_logTs)

            if (.not. reading_cache) then

               read(io_unit,*,iostat=ierr) ! skip
               if (ierr /= 0) return
               read(io_unit,*,iostat=ierr) ! skip
               if (ierr /= 0) return
             
               read(io_unit,'(a)',iostat=ierr) message
               if (ierr == 0) call str_to_vector(message, vec, nvec, ierr)
               if (nvec < num_logRs) ierr = -1
               if (ierr /= 0) return
               do j=1,num_logRs
                  x_tables(ix)% logRs(j) = vec(j)
               end do
            
               read(io_unit,*,iostat=ierr) ! skip
               if (ierr /= 0) return
            
               do i = 1, num_logTs
                  read(io_unit,'(a)',iostat=ierr) message
                  if (ierr == 0) call str_to_vector(message, vec, nvec, ierr)
                  if (nvec < 1+num_logRs) ierr = -1
                  if (ierr /= 0) return
                  x_tables(ix)% logTs(i) = vec(1)
                  do j=1,num_logRs
                     kap_logKs(j) = vec(j+1)
                     kap(1,j,i) = kap_logKs(j)
                  end do
                  
                  if (.false.) then
                     write(*,2) 'logT', i, x_tables(ix)% logTs(i)
                     do j=1,num_logRs
                        write(*,3) 'kap', j, i, kap(1,j,i)
                     end do
                     write(*,'(a)') 'message ' // trim(message)
                  end if
                  
               end do
         
               call Make_Interpolation_Data( &
                  kap1, x_tables(ix)% logRs, num_logRs, &
                  x_tables(ix)% logTs, num_logTs, &
                  x_tables(ix)% ili_logRs, x_tables(ix)% ili_logTs, ierr)

            else ! reading_cache

               read(io_unit, iostat=ierr) &
                  x_tables(ix)% ili_logRs, x_tables(ix)% ili_logTs
               if (ierr /= 0) return
               
               read(io_unit, iostat=ierr) &
                  x_tables(ix)% logRs(1:num_logRs), &
                  x_tables(ix)% logTs(1:num_logTs)            
               if (ierr /= 0) return
            
               read(io_unit, iostat=ierr) kap1
               if (ierr /= 0) return
            
            end if

                     
         end subroutine Read_Kap_X_Table


      end subroutine Prepare_Kap_X_Table
      
                  
      subroutine Make_Interpolation_Data( &
            kap1, logRs, num_logRs, logTs, num_logTs, ili_logRs, ili_logTs, ierr)
         use interp_2d_lib_sg
         real, pointer :: kap1(:)
         integer, intent(in) :: num_logRs, num_logTs
         real, intent(in) :: logRs(:) ! (num_logRs)
         real, intent(in) :: logTs(:) ! (num_logTs)
         integer, intent(out) :: ili_logRs, ili_logTs, ierr
         
         character (len=256) :: message
         integer :: ibcxmin                   ! bc flag for x=xmin
         real :: bcxmin(num_logTs)               ! bc data vs. y at x=xmin
         integer :: ibcxmax                   ! bc flag for x=xmax
         real :: bcxmax(num_logTs)               ! bc data vs. y at x=xmax
         integer :: ibcymin                   ! bc flag for y=ymin
         real :: bcymin(num_logRs)               ! bc data vs. x at y=ymin
         integer :: ibcymax                   ! bc flag for y=ymax
         real :: bcymax(num_logRs)               ! bc data vs. x at y=ymax
         integer :: ier                       ! =0 on exit if there is no error.
         integer :: i, j
         real, pointer :: kap(:,:,:)
         
         kap(1:sz_per_kap_point,1:num_logRs,1:num_logTs) => &
            kap1(1:sz_per_kap_point*num_logRs*num_logTs)
      
         ! just use "not a knot" bc's at edges of tables
         ibcxmin = 0; bcxmin(1:num_logTs) = 0
         ibcxmax = 0; bcxmax(1:num_logTs) = 0
         ibcymin = 0; bcymin(1:num_logRs) = 0
         ibcymax = 0; bcymax(1:num_logRs) = 0

         call interp_mkbicub_sg( &
               logRs, num_logRs, logTs, num_logTs, kap1, num_logRs, &
               ibcxmin,bcxmin,ibcxmax,bcxmax, &
               ibcymin,bcymin,ibcymax,bcymax, &
               ili_logRs,ili_logTs,ier)

         if (ier /= 0) then
            write(*,*) 'interp_mkbicub_db error happened for Make_Interpolation_Data for table'
            ierr = -1
            return
         end if
         
         call Check_Interpolation_Data
         
         ierr = 0
         
         contains
         
#ifdef offload
         !dir$ attributes offload: mic :: Check_Interpolation_Data
#endif      
         subroutine Check_Interpolation_Data
            use utils_lib,only:is_bad_real
            integer :: i, iR, jtemp
            real :: val
            
            do i = 1, sz_per_kap_point
               do iR = 1, num_logRs
                  do jtemp = 1, num_logTs
                     val = kap(i,iR,jtemp)
                     if (is_bad_real(val)) then
                        if (.true.) then
                           write(*,*) 'bad value in xz', val, i, iR, jtemp
                           write(*,'(99(a15,3x,f15.8,3x))')  &
                                    'logR', logRs(iR), 'logT', logTs(jtemp)
                        end if
                        kap(i,iR,jtemp) = 0
                     end if
                  end do
               end do
            end do
         
         end subroutine Check_Interpolation_Data
         

      end subroutine Make_Interpolation_Data
      
      
      subroutine Write_Kap_X_Table_Cache(x_tables, ix, io_unit,  version)
         type (Kap_X_Table), dimension(:), pointer :: x_tables
         integer, intent(in) :: ix, io_unit, version
         
         
         if (dbg_cache) then
            write(*,*) 'write cache plain text', io_unit
            write(io_unit,*) version, x_tables(ix)% num_logTs, x_tables(ix)% num_logRs
            write(io_unit,'(999(1pe26.16))') &
              x_tables(ix)% X, &
              x_tables(ix)% Z, &
              x_tables(ix)% logR_min, &
              x_tables(ix)% logR_max, &
              x_tables(ix)% logT_min, &
              x_tables(ix)% logT_max
            !write(io_unit) &
            !   x_tables(ix)% logRs(:), &
            !   x_tables(ix)% logTs(:)                            
            !write(io_unit) x_tables(ix)% kap(:,:,:)   
         else
            write(io_unit) version, x_tables(ix)% num_logTs, x_tables(ix)% num_logRs
            write(io_unit) &
              x_tables(ix)% X, &
              x_tables(ix)% Z, &
              x_tables(ix)% logR_min, &
              x_tables(ix)% logR_max, &
              x_tables(ix)% logT_min, &
              x_tables(ix)% logT_max                           
            write(io_unit) x_tables(ix)% ili_logRs, x_tables(ix)% ili_logTs 
            write(io_unit) &
               x_tables(ix)% logRs(:), &
               x_tables(ix)% logTs(:)                             
            write(io_unit) x_tables(ix)% kap1(:) 
         end if
            
      end subroutine Write_Kap_X_Table_Cache


      subroutine Get_Filenames( &
            z_tables, Z, X, data_dir, fname, filename, cache_filename)
         type (Kap_Z_Table), dimension(:), pointer :: z_tables
         real(dp), intent(in) :: Z, X
         character (*), intent(in) :: data_dir
         character (*), intent(out) :: fname, filename, cache_filename
         character (len=256) :: cache_fname
         call create_fname(z_tables, Z, X, fname, cache_fname)
         filename = trim(data_dir) // '/' // fname
         cache_filename = trim(kap_cache_dir) // '/' // cache_fname
      end subroutine Get_Filenames

		
		! this must match the preprocessor naming scheme
		subroutine create_fname(z_tables, Z, X, fname, cache_fname)
         type (Kap_Z_Table), dimension(:), pointer :: z_tables
			real(dp), intent(in) :: Z, X
			character (len=*), intent(out) :: fname, cache_fname
			character (len=256) :: zstr, xstr, prefix	
			if (z_tables(1)% lowT_flag .and. kap_using_lowT_Freedman) then
			   call get_Freedman_output_Zstr(Z, zstr)
            fname = trim(kap_lowT_prefix)// '_z' // trim(zstr) // '.data'
            cache_fname = trim(kap_lowT_prefix)// '_z' // trim(zstr) // '.bin'
            return
			end if
			call get_output_Zstr(Z, zstr)			
			call get_output_Xstr(X, xstr)
			if (z_tables(1)% lowT_flag) then
			   prefix = kap_lowT_prefix
			else
			   prefix = kap_prefix
			end if
   		fname = &
   		   trim(prefix) // '_z' // trim(zstr) // '_x' // trim(xstr) // '.data'
         cache_fname = &
            trim(prefix) // '_z' // trim(zstr) // '_x' // trim(xstr) // '.bin'
		end subroutine create_fname
		
		
		! copy preprocessor for this
		subroutine get_output_Zstr(Z, zstr)
			real(dp), intent(in) :: Z
			character (len=*),intent(out) :: zstr
			integer :: iz
			iz = floor(Z*1d5 + 0.1d0)
			select case (iz)
			   case (0)
			      zstr = '0m0'
			   case (1)
			      zstr = '1m5'
			   case (3)
			      zstr = '3m5'
			   case (10)
			      zstr = '1m4'
			   case (30)
			      zstr = '3m4'
			   case (100)
			      zstr = '1m3'
			   case (200)
			      zstr = '2m3'
			   case (400)
			      zstr = '4m3'
			   case (1000)
			      zstr = '1m2'
			   case (2000)
			      zstr = '2m2'
			   case (3000)
			      zstr = '3m2'
			   case (4000)
			      zstr = '4m2'
			   case (5000)
			      zstr = '5m2'
			   case (6000)
			      zstr = '6m2'
			   case (8000)
			      zstr = '8m2'
			   case (10000)
			      zstr = '1m1'
			   case default
			      write(*,*) 'unexpected Z value Z value', Z
			      stop 'get_Zstr'
			end select
		end subroutine get_output_Zstr
		
		
      subroutine get_Freedman_output_Zstr(Z, zstr)
         real(dp), intent(in) :: Z
         character (len=*),intent(out) :: zstr
         integer :: iz
         iz = floor(Z*1d5 + 0.1d0)
         select case (iz)
         case (1000)
            zstr = '0.01'
         case (2000)
            zstr = '0.02'
         case (4000)
            zstr = '0.04'
         case (10000)
            zstr = '0.10'
         case (20000)
            zstr = '0.20'
         case (63000)
            zstr = '0.63'
         case (100000)
            zstr = '1.00'
         case default
            write(*,*) 'unexpected Z value for Freedman data', Z
            stop 'get_Zstr'
         end select
      end subroutine get_Freedman_output_Zstr
		
		
		! copy preprocessor for this
		subroutine get_output_Xstr(X, xstr)
			real(dp), intent(in) :: X
			character (len=*),intent(out) :: xstr

			integer :: ix
			ix = floor(X*1d5 + 0.1d0)
			
! 0,  .1, .2, .35, .5, .7, .8, .9, .95, 1-Z
			select case (ix)
			   case (0)
			      xstr = '00'
			   case (10000)
			      xstr = '10'
			   case (20000)
			      xstr = '20'
			   case (35000)
			      xstr = '35'
			   case (50000)
			      xstr = '50'
			   case (70000)
			      xstr = '70'
			   case (80000)
			      xstr = '80'
			   case (90000)
			      xstr = '90'
			   case (92000)
			      xstr = '92'
			   case (94000)
			      xstr = '94'
			   case (95000)
			      xstr = '95'
			   case (96000)
			      xstr = '96'
			   case (97000)
			      xstr = '97'
			   case (98000)
			      xstr = '98'
			   case (99000)
			      xstr = '99'
			   case (99600)
			      xstr = '99.6'
			   case (99800)
			      xstr = '99.8'
			   case (99900)
			      xstr = '99.9'
			   case (99970)
			      xstr = '99.97'
			   case (99990)
			      xstr = '99.99'
			   case (99997)
			      xstr = '99.997'
			   case (99999)
			      xstr = '99.999'
			   case (100000)
			      xstr = '100'
			   case default
			      write(*,*) 'unknown X value', X, ix
			      stop 'get_output_Xstr'
			end select
			
		end subroutine get_output_Xstr

      
      subroutine Free_Kap_Tables
         use load_CO_kap, only: Free_Kap_CO_Tables   
             
         call Free_Kap_CO_Tables
         if (associated(kap_z_tables)) &
            call free_tables(kap_z_tables, .false.)
         if (associated(kap_lowT_z_tables)) &
            call free_tables(kap_lowT_z_tables, .true.)
         
         contains
         
#ifdef offload
         !dir$ attributes offload: mic :: free_tables
#endif      
         subroutine free_tables(z_tables, lowT_flag)
            type (Kap_Z_Table), dimension(:), pointer :: z_tables
            logical, intent(in) :: lowT_flag
            integer :: iz, num_Zs
            if (lowT_flag .and. kap_using_lowT_Freedman) then
               num_Zs = num_kap_Freedman_Zs
            else
               num_Zs = num_kap_Zs
            end if
            do iz = 1, num_Zs
               if (associated(z_tables(iz)% x_tables)) then
                  call free_xtable(z_tables(iz)% x_tables, z_tables(iz)% num_Xs, iz)
               end if
            end do
            deallocate(z_tables)
            nullify(z_tables) 
         end subroutine free_tables
         
#ifdef offload
         !dir$ attributes offload: mic :: free_xtable
#endif      
         subroutine free_xtable(x_tables, num_Xs, iz)
            type (Kap_X_Table), dimension(:), pointer :: x_tables       
            integer, intent(in) :: num_Xs, iz         
            integer :: ix
            do ix = 1, num_Xs            
               if (associated(x_tables(ix)% logRs)) deallocate(x_tables(ix)% logRs)
               if (associated(x_tables(ix)% logTs)) deallocate(x_tables(ix)% logTs)            
               if (associated(x_tables(ix)% kap1)) deallocate(x_tables(ix)% kap1)
            end do            
            if (associated(x_tables)) deallocate(x_tables)            
         end subroutine free_xtable

      end subroutine Free_Kap_Tables

#ifdef offload
      !dir$ end options
#endif

      end module load_kap
      
