! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
! ***********************************************************************

      module mod_ionization

      use const_def, only: one_third, two_thirds, avo, dp, mesa_data_dir
      use crlibm_lib
      use ionization_def

      implicit none
      

      logical, parameter :: dbg = .false.
     

      contains


      subroutine do_init_ionization(ionization_cache_dir_in, use_cache, ierr)      
         use utils_lib, only : alloc_iounit, free_iounit
#ifdef offload
         use ion_tables_load, only: Load_ion_Table
#endif         
         character (len=*), intent(in) :: ionization_cache_dir_in
         logical, intent(in) :: use_cache
         integer, intent(out) :: ierr
         ierr = 0
         table_is_initialized = .false.
#ifdef offload
         call do_load(ierr) ! for special Fe56_in_He4
         if (ierr /= 0) return
         call Load_ion_Table(ierr) ! for other ionization
         if (ierr /= 0) return
         call copy_to_coprocessor(ionization_cache_dir_in, ierr) 
#endif         
      end subroutine do_init_ionization

#ifdef offload
      subroutine copy_to_coprocessor(ionization_cache_dir_in, ierr) ! runs on host
         use ionization_def
         character (len=*), intent(in) :: ionization_cache_dir_in
         integer, intent(out) :: ierr

         integer :: ion_version_in, ion_num_logQs_in, ion_num_logTs_in
         real(dp) :: ion_logQ_min_in, ion_logQ_max_in, ion_del_logQ_in
         real(dp) :: ion_logT_min_in, ion_logT_max_in, ion_del_logT_in
         real(dp), dimension(:), pointer :: ion_logQs_in, ion_logTs_in, ion_tbl1_in

         ! for Ionization_Info
         integer :: num_log_ne_in, num_logT_in
         real(dp), pointer :: log_ne_in(:), logT_in(:)
         real(dp), pointer :: f1_in(:)
         integer :: ilinx_in, iliny_in


         if (.not. fe_he_ptr% have_interpolation_info) then
            write(*,*) 'ionization copy_to_coprocessor failure'
            ierr = -1
            stop 1
         end if

         ierr = 0
         
         ion_version_in = ion_version
         ion_num_logQs_in = ion_num_logQs
         ion_num_logTs_in = ion_num_logTs
         ion_logQ_min_in = ion_logQ_min
         ion_logQ_max_in = ion_logQ_max
         ion_del_logQ_in = ion_del_logQ
         ion_logT_min_in = ion_logT_min
         ion_logT_max_in = ion_logT_max
         ion_del_logT_in = ion_del_logT
         ion_logQs_in => ion_logQs
         ion_logTs_in => ion_logTs
         ion_tbl1_in => ion_tbl1
         
         num_log_ne_in = fe_he_ptr% num_log_ne
         num_logT_in = fe_he_ptr% num_logT
         ilinx_in = fe_he_ptr% ilinx
         iliny_in = fe_he_ptr% iliny
         log_ne_in => fe_he_ptr% log_ne
         logT_in => fe_he_ptr% logT
         f1_in => fe_he_ptr% f1
                  
         !dir$ offload target(mic) out(ierr) in( &
            ion_version_in, &
            ion_num_logQs_in, &
            ion_num_logTs_in, &
            ion_logQ_min_in, &
            ion_logQ_max_in, &
            ion_del_logQ_in, &
            ion_logT_min_in, &
            ion_logT_max_in, &
            ion_del_logT_in, &
            ion_logQs_in, &
            ion_logTs_in, &
            ion_tbl1_in, &
            num_log_ne_in, &
            num_logT_in, &
            ilinx_in, &
            iliny_in, &
            log_ne_in, &
            logT_in, &
            f1_in)
         call do_copy_to_coprocessor( &
            ion_version_in, &
            ion_num_logQs_in, &
            ion_num_logTs_in, &
            ion_logQ_min_in, &
            ion_logQ_max_in, &
            ion_del_logQ_in, &
            ion_logT_min_in, &
            ion_logT_max_in, &
            ion_del_logT_in, &
            ion_logQs_in, &
            ion_logTs_in, &
            ion_tbl1_in, &
            num_log_ne_in, &
            num_logT_in, &
            ilinx_in, &
            iliny_in, &
            log_ne_in, &
            logT_in, &
            f1_in, &
            ierr)
         
      end subroutine copy_to_coprocessor
      
#ifdef offload
      !dir$ attributes offload: mic :: do_copy_to_coprocessor
#endif         
      subroutine do_copy_to_coprocessor( & ! runs on mic
            ion_version_in, &
            ion_num_logQs_in, &
            ion_num_logTs_in, &
            ion_logQ_min_in, &
            ion_logQ_max_in, &
            ion_del_logQ_in, &
            ion_logT_min_in, &
            ion_logT_max_in, &
            ion_del_logT_in, &
            ion_logQs_in, &
            ion_logTs_in, &
            ion_tbl1_in, &
            num_log_ne_in, &
            num_logT_in, &
            ilinx_in, &
            iliny_in, &
            log_ne_in, &
            logT_in, &
            f1_in, &
            ierr)
         use ionization_def
         integer, intent(in) :: &
            ion_version_in, ion_num_logQs_in, ion_num_logTs_in
         real(dp), intent(in) :: &
            ion_logQ_min_in, ion_logQ_max_in, ion_del_logQ_in
         real(dp), intent(in) :: &
            ion_logT_min_in, ion_logT_max_in, ion_del_logT_in
         real(dp), dimension(:), pointer, intent(in) :: &
            ion_logQs_in, ion_logTs_in, ion_tbl1_in
         integer :: num_log_ne_in, num_logT_in
         real(dp), pointer :: log_ne_in(:), logT_in(:)
         real(dp), pointer :: f1_in(:)
         integer :: ilinx_in, iliny_in

         integer, intent(out) :: ierr

         integer :: i, sz
         ierr = 0
         
         fe_he_ptr => fe_he_info
         fe_he_ptr% num_log_ne = num_log_ne_in
         fe_he_ptr% num_logT = num_logT_in
         fe_he_ptr% ilinx = ilinx_in
         fe_he_ptr% iliny = iliny_in

         sz = size(log_ne_in, dim=1)
         allocate(fe_he_ptr% log_ne(sz))
         do i=1,sz
            fe_he_ptr% log_ne(i) = log_ne_in(i)
         end do

         sz = size(logT_in, dim=1)
         allocate(fe_he_ptr% logT(sz))
         do i=1,sz
            fe_he_ptr% logT(i) = logT_in(i)
         end do

         sz = size(f1_in, dim=1)
         allocate(fe_he_ptr% f1(sz))
         do i=1,sz
            fe_he_ptr% f1(i) = f1_in(i)
         end do
         
         ion_version = ion_version_in
         ion_num_logQs = ion_num_logQs_in
         ion_num_logTs = ion_num_logTs_in
         ion_logQ_min = ion_logQ_min_in
         ion_logQ_max = ion_logQ_max_in
         ion_del_logQ = ion_del_logQ_in
         ion_logT_min = ion_logT_min_in
         ion_logT_max = ion_logT_max_in
         ion_del_logT = ion_del_logT_in
         
         sz = size(ion_logQs_in, dim=1)
         allocate(ion_logQs(sz))
         do i=1,sz
            ion_logQs(i) = ion_logQs_in(i)
         end do
         
         sz = size(ion_logTs_in, dim=1)
         allocate(ion_logTs(sz))
         do i=1,sz
            ion_logTs(i) = ion_logTs_in(i)
         end do
         
         sz = size(ion_tbl1_in, dim=1)
         allocate(ion_tbl1(sz))
         do i=1,sz
            ion_tbl1(i) = ion_tbl1_in(i)
         end do
         ion_tbl(1:sz_per_ion_point, 1:num_ion_vals,&
                 1:ion_num_logQs, 1:ion_num_logTs, &
                 1:num_ion_Xs, 1:num_ion_Zs) => &
            ion_tbl1(1:sz)

         call ion_def_init(ionization_cache_dir)

         ion_root_is_initialized = .true.
         ion_is_initialized = .true.
         
      end subroutine do_copy_to_coprocessor
#endif         

      subroutine do_load(ierr)
         use utils_lib, only : alloc_iounit, free_iounit
         use ionization_def
         integer, intent(out) :: ierr
      
         integer :: io_log_ne, io_logT, io_z
         integer, pointer :: ibound(:,:), tmp_version(:)
         integer, parameter :: num_log_ne_fe56_he4 = 105, num_logT_fe56_he4 = 30
      
         ierr = 0
         io_log_ne = alloc_iounit(ierr)
         if (ierr /= 0) return
         io_logT = alloc_iounit(ierr)
         if (ierr /= 0) return
         io_z = alloc_iounit(ierr)
         if (ierr /= 0) return

         fe_he_ptr => fe_he_info  
      
         call load_table_summary( &
            'log_ne_fe56_he4.data', 'logT_fe56_he4.data', 'z_fe56_he4.data', &
            num_log_ne_fe56_he4, num_logT_fe56_he4, fe_he_ptr, ierr)
         if (ierr /= 0) return         

         call free_iounit(io_log_ne)
         call free_iounit(io_logT)
         call free_iounit(io_z)
         
         call create_interpolants(fe_he_ptr,num_log_ne_fe56_he4,num_logT_fe56_he4,ierr)
         if (ierr /= 0) return         
         
         table_is_initialized = .true.
         
         contains
         
         subroutine openfile(filename, iounit, ierr)
            character(len=*) :: filename
            integer, intent(in) :: iounit
            integer, intent(out) :: ierr
            if (dbg) write(*,*) 'read ' // trim(filename)            
            ierr = 0
            open(iounit,file=trim(filename),action='read',status='old',iostat=ierr)
            if (ierr/= 0) then
               write(*,*) 'table_ionization_init: missing ionization data'
               write(*,*) filename
               write(*,*)
               write(*,*)
               write(*,*)
               write(*,*)
               write(*,*)
               write(*,*) 'FATAL ERROR: missing or bad ionization data.'
               write(*,*) 'Please update by removing the directory mesa/data/ionization_data,'
               write(*,*) 'and rerunning the mesa ./install script.'
               write(*,*)
               stop 1
            endif
         end subroutine openfile
      
      
         subroutine load_table_summary( &
               log_ne_fname, logT_fname, z_fname, num_log_ne, num_logT, p, ierr)
            character(len=*), intent(in) :: log_ne_fname, logT_fname, z_fname
            integer, intent(in) :: num_log_ne, num_logT
            type (Ionization_Info), pointer :: p
            integer, intent(out) :: ierr
      
            character(len=256) :: filename
            real(dp), pointer :: f(:,:,:)
            integer :: i, j
            
            ierr = 0
            p% have_interpolation_info = .false.
            p% num_log_ne = num_log_ne
            p% num_logT = num_logT
            allocate( &
               p% log_ne(num_log_ne), p% logT(num_logT), &
               p% f1(4*num_log_ne*num_logT), stat=ierr)
            if (ierr /= 0) then
               write(*,*) 'failed in allocate for ionization tables'
               stop 1
            end if
            f(1:4,1:num_log_ne,1:num_logT) => p% f1(1:4*num_log_ne*num_logT)
            
            filename = trim(mesa_data_dir) // '/ionization_data/' // trim(z_fname)            
            call openfile(filename, io_z, ierr)
            if (ierr /= 0) return
            do i=1,num_logT
               read(io_z,fmt=*,iostat=ierr) p% log_ne(1:num_log_ne)
               if (ierr /= 0) then
                  write(*,*) 'failed in reading ionization z ' // trim(filename)
                  stop 1
               end if
               !p% f(1,1:num_log_ne,i) = p% log_ne(1:num_log_ne)  << segfault on UBUNTU
               do j=1,num_log_ne
                  f(1,j,i) = p% log_ne(j) ! sets p% f1
               end do
            end do
            close(io_z)
            
            filename = trim(mesa_data_dir) // '/ionization_data/' // trim(log_ne_fname)            
            call openfile(filename, io_log_ne, ierr)
            if (ierr /= 0) return
            do i=1,num_log_ne
               read(io_log_ne,fmt=*,iostat=ierr) p% log_ne(i)
               if (ierr /= 0) then
                  write(*,*) 'failed in reading ionization log_ne ' // trim(filename)
                  stop 1
               end if
            end do
            close(io_log_ne)
            
            filename = trim(mesa_data_dir) // '/ionization_data/' // trim(logT_fname)            
            call openfile(filename, io_logT, ierr)
            if (ierr /= 0) return
            do i=1,num_logT
               read(io_logT,fmt=*,iostat=ierr) p% logT(i)
               if (ierr /= 0) then
                  write(*,*) 'failed in reading ionization logT ' // trim(filename)
                  stop 1
               end if
            end do
            close(io_logT)
         
         end subroutine load_table_summary


      end subroutine do_load

      
      subroutine create_interpolants(p,nx,ny,ierr)
         use interp_2d_lib_db
         type (Ionization_Info), pointer :: p
         integer, intent(in) :: nx, ny
         integer, intent(out) :: ierr
         integer :: ibcxmin, ibcxmax, ibcymin, ibcymax
         real(dp) :: bcxmin(ny), bcxmax(ny), bcymin(nx), bcymax(nx)
         ! use "not a knot" bc's
         ibcxmin = 0; bcxmin(:) = 0d0
         ibcxmax = 0; bcxmax(:) = 0d0
         ibcymin = 0; bcymin(:) = 0d0
         ibcymax = 0; bcymax(:) = 0d0
         call interp_mkbicub_db( &
            p% log_ne, p% num_log_ne, p% logT, p% num_logT, &
            p% f1, p% num_log_ne, &
            ibcxmin, bcxmin, ibcxmax, bcxmax, &
            ibcymin, bcymin, ibcymax, bcymax, &
            p% ilinx, p% iliny, ierr )
         if (ierr /= 0) then
            if (dbg) write(*,*) 'interp_mkbicub_db failed for ionization interpolant'
            return
         end if
         p% have_interpolation_info = .true.
      end subroutine create_interpolants

#ifdef offload
      !dir$ attributes offload: mic :: charge_of_Fe56_in_He4
#endif         
      real(dp) function charge_of_Fe56_in_He4(log_ne, logT, ierr)
         use interp_2d_lib_db
         real(dp), intent(in) :: log_ne ! ne=avo*rho*free_e
         real(dp), intent(in) :: logT
         integer, intent(out) :: ierr

         integer :: ict(6) ! code specifying output desired
         real(dp) :: fval(6) ! output data
         type (Ionization_Info), pointer :: p
         
         ierr = 0
         charge_of_Fe56_in_He4 = 0
         
#ifndef offload
         if (.not. table_is_initialized) then
!$omp critical (ionization_table)
            if (.not. table_is_initialized) call do_load(ierr)
!$omp end critical (ionization_table)
            if (ierr /= 0) return
         endif
#endif         
         
         ict = 0; ict(1) = 1 ! just the result; no partials
         p => fe_he_ptr
         call interp_evbicub_db( &
            log_ne, logT, p% log_ne, p% num_log_ne, p% logT, p% num_logT, &
            p% ilinx, p% iliny, p% f1, p% num_log_ne, ict, fval, ierr)
         
         charge_of_Fe56_in_He4 = fval(1)
         
      end function charge_of_Fe56_in_He4

#ifdef offload
      !dir$ attributes offload: mic :: chi_info
#endif         
      subroutine chi_info(a1, z1, T, log_T, rho, log_rho, chi, c0, c1, c2)
         real(dp), intent(in) :: a1, z1, T, log_T, rho, log_rho
         real(dp), intent(out) :: chi, c0, c1, c2
         chi = 1.987d-4*T*(-8.392d0 - log_rho + 1.5d0*log_T - log10_cr(z1/a1)) ! eqn 20
         ! coef's used in eqn 21
         c0 = 1.085d-4*rho*T/a1
         c1 = 1.617d4*sqrt(rho*(z1**2 + z1)/(T*a1))
         c2 = 29.38d0*z1/pow_cr(a1,one_third)
      end subroutine chi_info
      
#ifdef offload
      !dir$ attributes offload: mic :: chi_effective
#endif         
      real(dp) function chi_effective(chi, c0, c1, c2, z1, z2)
         real(dp), intent(in) :: chi, c0, c1, c2, z1, z2
         chi_effective = chi + c0/(z2*z2*z2) + &
            min(c1*z2, c2*(pow_cr(z2/z1,two_thirds) + 0.6d0))
      end function chi_effective
      
#ifdef offload
      !dir$ attributes offload: mic :: get_typical_charge
#endif         
      real(dp) function get_typical_charge(cid, a1, z1, T, log_T, rho, log_rho)
         use mod_tables
         use chem_def
         integer, intent(in) :: cid
         real(dp), intent(in) :: a1, z1, T, log_T, rho, log_rho      
         real(dp) :: chi, c0, c1, c2, z2, chi_eff
         integer :: i, izmax
         include 'formats'
         if (.not. ionization_tables_okay) then
            call set_ionization_potentials
            ionization_tables_okay = .true.
         end if
         izmax = int(chem_isos% Z(cid))
         get_typical_charge = dble(izmax)
         if (izmax > 30) return
         call chi_info(a1, z1, T, log_T, rho, log_rho, chi, c0, c1, c2)
         do i=1, izmax-1
            z2 = dble(i)
            chi_eff = chi_effective(chi, c0, c1, c2, z1, z2+1)
            if (chi_eff < ip(izmax,i+1)) then
               get_typical_charge = z2
               return
            end if
         end do
      end function get_typical_charge
      
      



      end module mod_ionization

