      module test_gyre_mod
      use const_def, only : dp
      use gyre_lib
      implicit none

      contains

      subroutine do_test

        real(dp)              :: M_star
        real(dp)              :: R_star
        real(dp)              :: L_star
        real(dp), allocatable :: r(:)
        real(dp), allocatable :: w(:)
        real(dp), allocatable :: p(:)
        real(dp), allocatable :: rho(:)
        real(dp), allocatable :: T(:)
        real(dp), allocatable :: N2(:)
        real(dp), allocatable :: Gamma_1(:)
        real(dp), allocatable :: nabla_ad(:)
        real(dp), allocatable :: delta(:)
        real(dp), allocatable :: nabla(:)
        real(dp), allocatable :: kappa(:)
        real(dp), allocatable :: kappa_rho(:)
        real(dp), allocatable :: kappa_T(:)
        real(dp), allocatable :: epsilon(:)
        real(dp), allocatable :: epsilon_rho(:)
        real(dp), allocatable :: epsilon_T(:)
        real(dp), allocatable :: Omega_rot(:)

        integer  :: ipar(1)
        real(dp) :: rpar(1)
        integer  :: retcode

        ! Initialize

        call gyre_init()

        ! Override constants

        G_GRAVITY = 6.67428d-8

        ! Read a model from file

        call gyre_read_model('model.dat', 'MONO')

        ! Find modes

        call gyre_get_modes(0, 'gyre.in', .FALSE., user_sub, ipar, rpar)
        call gyre_get_modes(1, 'gyre.in', .FALSE., user_sub, ipar, rpar)

        write(*,*) 'done file model'

        ! Load a model into memory

        call load_model('model.dat', M_star, R_star, L_star, r, w, p, rho, T, &
                         N2, Gamma_1, nabla_ad, delta, nabla, &
                         kappa, kappa_rho, kappa_T, &
                         epsilon, epsilon_rho, epsilon_T, &
                         Omega_rot)

        call gyre_set_model (M_star, R_star, L_star, r, w, p, rho, T, &
                             N2, Gamma_1, nabla_ad, delta, nabla,  &
                             kappa, kappa_rho, kappa_T, &
                             epsilon, epsilon_rho, epsilon_T, &
                             Omega_rot, 'MONO')

        ! Find modes

        call gyre_get_modes(0, 'gyre.in', .FALSE., user_sub, ipar, rpar)
        call gyre_get_modes(1, 'gyre.in', .FALSE., user_sub, ipar, rpar)

        write(*,*) 'done memory model'
         
      end subroutine do_test 

      subroutine user_sub (md, ipar, rpar, retcode)

        type(mode_t), intent(in) :: md
        integer, intent(inout)   :: ipar(:)
        real(dp), intent(inout)  :: rpar(:)
        integer, intent(out)     :: retcode

        integer :: n_p
        integer :: n_g
        integer :: n_pg

        ! Print out mode info

        write(*,*) md%mp%l, md%n_p, md%n_g, md%n_pg, REAL(md%freq('UHZ')), md%E_norm()

        ! Finish

        retcode = 0

        return

      end subroutine user_sub

      subroutine load_model (file, M_star, R_star, L_star, r, w, p, rho, T, &
                             N2, Gamma_1, nabla_ad, delta, nabla, &
                             kappa, kappa_rho, kappa_T, &
                             epsilon, epsilon_rho, epsilon_T, &
                             Omega_rot)

        character(LEN=*), intent(in)       :: file
        real(dp), intent(out)              :: M_star
        real(dp), intent(out)              :: R_star
        real(dp), intent(out)              :: L_star
        real(dp), allocatable, intent(out) :: r(:)
        real(dp), allocatable, intent(out) :: w(:)
        real(dp), allocatable, intent(out) :: p(:)
        real(dp), allocatable, intent(out) :: rho(:)
        real(dp), allocatable, intent(out) :: T(:)
        real(dp), allocatable, intent(out) :: N2(:)
        real(dp), allocatable, intent(out) :: Gamma_1(:)
        real(dp), allocatable, intent(out) :: nabla_ad(:)
        real(dp), allocatable, intent(out) :: delta(:)
        real(dp), allocatable, intent(out) :: nabla(:)
        real(dp), allocatable, intent(out) :: kappa(:)
        real(dp), allocatable, intent(out) :: kappa_rho(:)
        real(dp), allocatable, intent(out) :: kappa_T(:)
        real(dp), allocatable, intent(out) :: epsilon(:)
        real(dp), allocatable, intent(out) :: epsilon_rho(:)
        real(dp), allocatable, intent(out) :: epsilon_T(:)
        real(dp), allocatable, intent(out) :: Omega_rot(:)

        integer               :: unit
        integer               :: n
        integer               :: n_cols
        real(dp), allocatable :: var(:,:)

        ! Read a model from the MESA-format file

        open(NEWUNIT=unit, FILE=file, STATUS='OLD')

        ! Read the header

        read(unit, *) n, M_star, R_star, L_star, n_cols

        ! Determine the file variant, and read the data

        if(n_cols == 1) then

           ! Old variant (n_cols not specified)

           n_cols = 18

           backspace(unit)

           call read_mesa_data_old()

        else

           ! New variant (n_cols specified)

           call read_mesa_data_new()

        endif

        ! Finish

        return

      contains

        subroutine read_mesa_data_old ()

          integer :: k
          integer :: k_chk

          ! Read data from the old-variant file

          allocate(var(18,n))

          read_loop : do k = 1,n
             read(unit, *) k_chk, var(:,k)
             if(k /= k_chk) stop 'Index mismatch'
          end do read_loop

          close(unit)

          r = var(1,:)
          w = var(2,:)
          p = var(4,:)
          T = var(5,:)
          rho = var(6,:)
          nabla = var(7,:)
          N2 = var(8,:)
          Gamma_1 = var(12,:)*var(10,:)/var(9,:)
          delta = var(11,:)/var(12,:)
          kappa = var(13,:)
          kappa_T = var(14,:)
          kappa_rho = var(15,:)
          epsilon = var(16,:)
          epsilon_T = var(17,:)
          epsilon_rho = var(18,:)

          nabla_ad = p*delta/(rho*T*var(10,:))

          allocate(Omega_rot(n))
          Omega_rot = 0._WP

          ! Decide whether epsilon_T and epsilon_rho need rescaling

          k = MAXLOC(ABS(epsilon_T), DIM=1)

          if(ABS(epsilon_T(k)) < 1E-3*ABS(epsilon(k))) then
             epsilon_T = epsilon_T*epsilon
             epsilon_rho = epsilon_rho*epsilon
          endif

          ! Finish

          return

        end subroutine read_mesa_data_old

        subroutine read_mesa_data_new ()

          integer :: k
          integer :: k_chk

          if(n_cols < 18) stop 'Too few columns'

          ! Read data from the new-variant file

          allocate(var(n_cols-1,n))

          read_loop : do k = 1,n
             read(unit, *) k_chk, var(:,k)
             if(k /= k_chk) stop 'Index mismatch'
          end do read_loop

          close(unit)

          r = var(1,:)
          w = var(2,:)
          p = var(4,:)
          T = var(5,:)
          rho = var(6,:)
          nabla = var(7,:)
          N2 = var(8,:)
          Gamma_1 = var(9,:)
          nabla_ad = var(10,:)
          delta = var(11,:)
          kappa = var(12,:)
          kappa_T = var(13,:)
          kappa_rho = var(14,:)
          epsilon = var(15,:)
          epsilon_T = var(16,:)
          epsilon_rho = var(17,:)
          Omega_rot = var(18,:)

          ! Finish

          return

        end subroutine read_mesa_data_new

        subroutine add_center (x, y)
    
          real(dp), intent(in)                 :: x(:)
          real(dp), intent(inout), allocatable :: y(:)

          real(dp) :: y_0

          ! Add center (x=0) data to the array y(x), incrementing the
          ! dimension of y by 1. x is not altered.
      
          y_0 = (x(2)**2*y(1) - x(1)**2*y(2))/(x(2)**2 - x(1)**2)
      
          y = [y_0,y]

          ! Finish

          return

        end subroutine add_center
      end subroutine load_model

      end module test_gyre_mod

      program test_gyre
      use test_gyre_mod
      implicit none
      call do_test
      end program
