      module test_eos_support

      use eos_def
      use eos_lib
      use chem_lib
      use chem_def
      use const_def
      use eos_support
      use crlibm_lib
      
      implicit none

      real(dp), dimension(:), pointer :: rho_vector, T_vector
      real(dp), dimension(:,:), pointer :: results, results_saved
      
      contains
      
      
      subroutine get_logP20_contour
         real(dp) :: Z, X, logPgas, logT, logRho, logP, T, Prad, Pgas, P
         real(dp) :: logT_min, logT_max, dlogT
         integer :: iounit
         include 'formats.dek'
         call Setup_eos
         Z =  0.02
         X =  0.0
         logT_min = 6d0
         logT_max = 8.59d0
         dlogT = 0.01d0
         
         write(*,'(99(a20,4x))') 'logRho', 'logT', 'logP'
         
         logT = logT_min
         do while (logT < logT_max)
            T = exp10_cr(logT)
            logP = 20d0
            P = exp10_cr(logP)
            Prad = crad*T*T*T*T/3
            Pgas = P - Prad
            logPgas = log10_cr(Pgas)
            call test1_eosPT(Z, X, logPgas, logT, .true., .true., logRho, logP)
            write(*,'(99(f20.10,4x))') logRho, logT, logP
            logT = logT + dlogT
         end do
         
         write(*,*)
         
      end subroutine get_logP20_contour
      
      
      subroutine get_logP23_contour
         real(dp) :: Z, X, logPgas, logT, logRho, logP, T, Prad, Pgas, P
         real(dp) :: logT_min, logT_max, dlogT
         integer :: iounit
         include 'formats.dek'
         call Setup_eos
         Z =  0.02
         X =  0.0
         logT_min = 7d0
         logT_max = 8.9d0
         dlogT = 0.01d0
         
         write(*,'(99(a20,4x))') 'logRho', 'logT', 'logP'
         
         logT = logT_min
         do while (logT < logT_max)
            T = exp10_cr(logT)
            logP = 23d0
            P = exp10_cr(logP)
            Prad = crad*T*T*T*T/3
            Pgas = P - Prad
            logPgas = log10_cr(Pgas)
            call test1_eosPT(Z, X, logPgas, logT, .true., .true., logRho, logP)
            write(*,'(99(f20.10,4x))') logRho, logT, logP
            logT = logT + dlogT
         end do
         
      end subroutine get_logP23_contour
      
      
      subroutine test1_eosPT_for_ck(quietly)
         logical, intent(in) :: quietly
         real(dp) :: Z, X, logPgas, logT, logRho, logP
         ! pick args for test so that use both HELM and tables to get results.
         Z =  0.02d0
         X =  0.7d0
         logT = 0.814999961853D+01
         logPgas = 0.184108947214D+02
         call test1_eosPT(Z, X, logPgas, logT, .false., quietly, logRho, logP)
      end subroutine test1_eosPT_for_ck

      
      subroutine test_eosPT
         real(dp) :: Z, X, logPgas, logT, logRho, logP
         Z =  2d-01
         X =  0.00
         logT = 5.9338637087310975d+00
         logPgas = 2.2999999999999993d+01
         call test1_eosPT(Z, X, logPgas, logT, .true., .false., logRho, logP)
      end subroutine test_eosPT
      
      
      subroutine test1_eosPT(Z, X, logPgas, logT, do_compare, quietly, logRho, logP)
         logical, intent(in) :: quietly
         real(dp) :: Z, X, logPgas, logT
         real(dp), intent(out) :: logRho, logP
         logical, intent(in) :: do_compare
         real(dp) :: 
     >         P, Pgas, Prad, T, Rho, dlnRho_dlnPgas_c_T, dlnRho_dlnT_c_Pgas, 
     >         res(num_eos_basic_results), d_dlnd(num_eos_basic_results),
     >         d_dlnT(num_eos_basic_results),       
     >         d_dabar(num_eos_basic_results), d_dzbar(num_eos_basic_results),
     >         res2(num_eos_basic_results), d_dlnd2(num_eos_basic_results),
     >         d_dabar2(num_eos_basic_results), d_dzbar2(num_eos_basic_results),
     >         d_dlnT2(num_eos_basic_results)     
         integer:: ierr, i
         logical, parameter :: use_log10_for_other = .false.
         character (len=8) :: names(num_eos_basic_results)
         
         ! for offload, need local copies of globals
         integer :: handle1, species1
         real(dp) :: abar1, zbar1
         integer, pointer, dimension(:) :: chem_id1, net_iso1
         real(dp) :: xa1(species)
         
         
         include 'formats.dek'
 
         ierr = 0

         call Init_Composition(X, Z, 0d0, 0d0) ! sets abar and zbar

         handle1 = handle
         species1 = species
         chem_id1 => chem_id
         net_iso1 => net_iso
         abar1 = abar      
         zbar1 = zbar
         xa1 = xa
         
         if (.false.) then ! TESTING
            z =     1.9999871528153523D-02
            x =     7.0000210590897427D-01
            call Init_Composition(X, Z, 0d0, 0d0) ! sets abar and zbar
            abar =     1.2966413082679851D+00
            zbar =     1.1021433867453336D+00
            logPgas  = 4.8066181993619859D+00
            logT  = 3.7569035961895620D+00
         end if
         
         T = exp10_cr(logT)
         Pgas = exp10_cr(logPgas)

         if (.not. quietly) then
            write(*,*) 'test1_eosPT'
            write(*,1) 'logT', logT
            write(*,1) 'logPgas', logPgas
            write(*,1) 'T', T
            write(*,1) 'Pgas', Pgas
            write(*,1) 'Z', Z
            write(*,1) 'X', X
            write(*,1) 'abar', abar
            write(*,1) 'zbar', zbar
            write(*,*)
         end if

         call eosPT_get(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         Pgas, logPgas, T, logT, 
     >         Rho, logRho, dlnRho_dlnPgas_c_T, dlnRho_dlnT_c_Pgas, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in eosPT_get for test1_eosPT'
            stop 1
         end if
         
#ifdef offload
         !dir$ offload target(mic) 
     >      in(handle1, Z, X, abar1, zbar1, 
     >         species1, chem_id1, net_iso1, xa1,
     >         Pgas, logPgas, T, logT)
     >      out(Rho, logRho, dlnRho_dlnPgas_c_T, dlnRho_dlnT_c_Pgas, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, ierr) 
#endif
         call eosPT_get(
     >         handle1, Z, X, abar1, zbar1, 
     >         species1, chem_id1, net_iso1, xa1,
     >         Pgas, logPgas, T, logT, 
     >         Rho, logRho, dlnRho_dlnPgas_c_T, dlnRho_dlnT_c_Pgas, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in eosPT_get for test1_eosPT'
            stop 1
         end if

         Prad = crad*T*T*T*T/3
         P = Pgas + Prad
         logP = log10_cr(P)
         
         if (quietly) return
         
         write(*,*)
         write(*,1) 'rho', rho
         write(*,1) 'logRho', logRho
         write(*,*)
     
         call eos_result_names(names)
         
         if (.not. do_compare) then ! simple form of output
            write(*,1) 'dlnRho_dlnPgas_c_T', dlnRho_dlnPgas_c_T
            write(*,1) 'dlnRho_dlnT_c_Pgas', dlnRho_dlnT_c_Pgas
            write(*,1) 'P', P
            write(*,1) 'E', exp_cr(res(i_lnE))
            write(*,1) 'S', exp_cr(res(i_lnS))
            do i=4, num_eos_basic_results
               write(*,1) trim(names(i)), res(i)
            end do
            write(*,*)
            return
         end if

         call eosDT_get(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         rho, logRho, T, logT, 
     >         res2, d_dlnd2, d_dlnT2, d_dabar2, d_dzbar2, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in eosDT_get for test1_eosPT'
            stop 1
         end if
     
         write(*,*)
         
         write(*,1) 'dlnRho_dlnPgas_c_T', dlnRho_dlnPgas_c_T
         write(*,1) 'dlnRho_dlnT_c_Pgas', dlnRho_dlnT_c_Pgas
         do i=1, num_eos_basic_results
            write(*,1) trim(names(i)), res(i), res2(i), 
     >            (res(i)-res2(i)) / max(1d0, abs(res(i)), abs(res2(i)))
         end do
         write(*,*)
         
         do i=1, num_eos_basic_results
            write(*,1) 'd_dlnd ' // trim(names(i)), d_dlnd(i), d_dlnd2(i), 
     >            (d_dlnd(i)-d_dlnd2(i)) / max(1d0, abs(d_dlnd(i)), abs(d_dlnd2(i)))
         end do
         write(*,*)
         
         do i=1, num_eos_basic_results
            write(*,1) 'd_dlnT ' // trim(names(i)), d_dlnT(i), 
     >            d_dlnT2(i), (d_dlnT(i)-d_dlnT2(i)) / max(1d0, abs(d_dlnT(i)), abs(d_dlnT2(i)))
         end do
         write(*,*)
           
      end subroutine test1_eosPT
      
      
      subroutine test1_eosDE_for_ck(quietly)
         logical, intent(in) :: quietly
         real(dp) :: Z, X, logRho, logE, logT_guess, logT, logPgas
         logical :: do_compare
         
         !write(*,*) 'test in table'
         Z =  0.02d0
         X =  0.70d0
         logRho = -4.011
         logE = 14.011
         logT_guess = 4
         do_compare = .false.
         call test1_eosDE(
     >      Z, X, logRho, logE, logT_guess, do_compare, quietly, logT, logPgas)
     
         !write(*,*) 'test off table'
         Z =  0.02d0
         X =  0.70d0
         logRho = 6
         logE = 17.2
         logT_guess = 8
         call test1_eosDE(
     >      Z, X, logRho, logE, logT_guess, do_compare, quietly, logT, logPgas)
     
      end subroutine test1_eosDE_for_ck

      
      subroutine test_eosDE
         real(dp) :: Z, X, logRho, logE, logT_guess, logT, logPgas
         logical :: do_compare, quietly
         Z =  0.02d0
         X =  0.70d0
         
         ! off table
         logRho = 6
         logE = 17.2
         logT_guess = 8.0
         
         ! in table
         logRho = -4.011
         logE = 14.011
         logT_guess = 4
         
         ! test
         logRho = -3.2869927951535223D+00
         logE = 1.3390937632825320D+01
         logT_guess = 4.8604791525666036D+00

         do_compare = .false.
         quietly = .false.
         call test1_eosDE(
     >      Z, X, logRho, logE, logT_guess, do_compare, quietly, logT, logPgas)
      end subroutine test_eosDE
      
      
      subroutine test1_eosDE(
     >      Z, X, logRho, logE, logT_guess, do_compare, quietly, logT, logPgas)
         logical, intent(in) :: quietly
         real(dp) :: Z, X, logRho, logE, logT_guess
         real(dp), intent(out) :: logT, logPgas
         logical, intent(in) :: do_compare
         real(dp) :: 
     >         energy, rho, T, 
     >         dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, 
     >         dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, 
     >         res(num_eos_basic_results), d_dlnd(num_eos_basic_results),
     >         d_dlnT(num_eos_basic_results),     
     >         d_dabar(num_eos_basic_results), d_dzbar(num_eos_basic_results),
     >         res2(num_eos_basic_results), d_dlnd2(num_eos_basic_results),
     >         d_dabar2(num_eos_basic_results), d_dzbar2(num_eos_basic_results),
     >         d_dlnT2(num_eos_basic_results)    
         integer:: ierr, i
         logical, parameter :: use_log10_for_other = .false.
         character (len=8) :: names(num_eos_basic_results)
         
         ! for offload, need local copies of globals
         integer :: handle1, species1
         real(dp) :: abar1, zbar1
         integer, pointer, dimension(:) :: chem_id1, net_iso1
         real(dp) :: xa1(species)
         
         include 'formats.dek'
 
         ierr = 0

         call Init_Composition(X, Z, 0d0, 0d0) ! sets abar and zbar
         
         handle1 = handle
         species1 = species
         abar1 = abar      
         zbar1 = zbar
         chem_id1 => chem_id
         net_iso1 => net_iso
         xa1 = xa
         
         if (.false.) then ! TESTING
            z =     2D-02
            x =     7D-01
            call Init_Composition(X, Z, 0d0, 0d0) ! sets abar and zbar
            !abar =     1.2966438355619729D+00
            !zbar =     1.1021536997138426D+00
            !logRho  = -3.2869927951535223D+00
            !logE  = 1.3390937632825320D+01
         end if
         
         energy = exp10_cr(logE)
         rho = exp10_cr(logRho)

         if (.not. quietly) then
            write(*,*) 'test1_eosDE'
            write(*,1) 'logRho', logRho
            write(*,1) 'logE', logE
            write(*,1) 'logT_guess', logT_guess
            write(*,1) 'Z', Z
            write(*,1) 'X', X
            write(*,1) 'abar', abar
            write(*,1) 'zbar', zbar
            write(*,*)
         end if
         
         call eosDE_get(
     >         handle1, Z, X, abar1, zbar1, 
     >         species1, chem_id1, net_iso1, xa1,
     >         energy, logE, rho, logRho, logT_guess, 
     >         T, logT, res, d_dlnd, d_dlnT, d_dabar, d_dzbar, 
     >         dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, 
     >         dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, 
     >         ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in eosDE_get for test1_eosDE'
            stop 1
         end if
         
         if (quietly) return
         
         write(*,*)
         write(*,1) 'T', T
         write(*,1) 'logT', logT
         write(*,*)
     
         call eos_result_names(names)
         
         if (.not. do_compare) then ! simple form of output
            write(*,1) 'logPgas', res(i_lnPgas)/ln10
            write(*,1) 'logS', res(i_lnS)/ln10
            do i=4, num_eos_basic_results
               write(*,1) trim(names(i)), res(i)
            end do
            write(*,1) 'dlnT_dlnE_c_Rho', dlnT_dlnE_c_Rho
            write(*,1) 'dlnT_dlnd_c_E', dlnT_dlnd_c_E
            write(*,1) 'dlnPgas_dlnE_c_Rho', dlnPgas_dlnE_c_Rho
            write(*,1) 'dlnPgas_dlnd_c_E', dlnPgas_dlnd_c_E
            write(*,*)
            return
         end if

         call eosDT_get(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         rho, logRho, T, logT, 
     >         res2, d_dlnd2, d_dlnT2, d_dabar2, d_dzbar2, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in eosDT_get for test1_eosDE'
            stop 1
         end if
     
         write(*,*)
         
         do i=1, num_eos_basic_results
            write(*,1) trim(names(i)), res(i), res2(i), 
     >            (res(i)-res2(i)) / max(1d0, abs(res(i)), abs(res2(i)))
         end do
         write(*,*)
         
         do i=1, num_eos_basic_results
            write(*,1) 'd_dlnd ' // trim(names(i)), d_dlnd(i), d_dlnd2(i), 
     >            (d_dlnd(i)-d_dlnd2(i)) / max(1d0, abs(d_dlnd(i)), abs(d_dlnd2(i)))
         end do
         write(*,*)
         
         do i=1, num_eos_basic_results
            write(*,1) 'd_dlnT ' // trim(names(i)), d_dlnT(i), 
     >            d_dlnT2(i), (d_dlnT(i)-d_dlnT2(i)) / max(1d0, abs(d_dlnT(i)), abs(d_dlnT2(i)))
         end do
         write(*,*)
           
      end subroutine test1_eosDE
      
      
      
      
      
      subroutine test_eosDT
         real(dp) :: logT, logRho, T, Rho, X, Z, logPgas, Pgas, Prad, P, XC, XO,
     >         logRho1_OPAL_SCVH_limit, logRho2_OPAL_SCVH_limit, Z_all_HELM
         real(dp), dimension(num_eos_basic_results) :: 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar
         integer :: info
      
  1      format(a30,4x,1pe24.16)

                                                   z =     5.0835743564261371D-01
                                                   x =     9.9999999999999982-100

                                                 rho =     9.4747794945837904D+03
                                              logRho =     3.9765691114739008D+00
                                                   T =     8.0765312722926632D+07
                                                logT =     7.9072248789982353D+00
         
         XC = 0
         XO = 0
         
         call Init_Composition(X, Z, XC, XO)
         write(*,*) 'change abar and zbar for test'
         
         
         
                                                abar =     6.1653224270134785D+00
                                                zbar =     3.0826612135067393D+00
         
         if (.false.) then
            logRho1_OPAL_SCVH_limit = 3.1d0 ! don't use OPAL_SCVH for logRho > this
            logRho2_OPAL_SCVH_limit = 3.0d0 ! full OPAL_SCVH okay for logRho < this
            call eos_set_logRhos_OPAL_SCVH(handle, logRho1_OPAL_SCVH_limit, logRho2_OPAL_SCVH_limit, info)
            write(*,1) 'logRho1_OPAL_SCVH_limit', logRho1_OPAL_SCVH_limit
            write(*,1) 'logRho2_OPAL_SCVH_limit', logRho2_OPAL_SCVH_limit 
            if (info /= 0) stop 'call eos_set_logRhos_OPAL_SCVH'
         end if
         
         if (.false.) then
            Z_all_HELM = 1.0001
            write(*,1) 'Z_all_HELM', Z_all_HELM
            call eos_set_Z_all_HELM(handle, Z_all_HELM, info)
            if (info /= 0) stop 'call eos_set_Z_all_HELM'
         end if

         write(*,*) 'call eosDT_get'         
         info = 0
         call eosDT_get( 
     >         handle, Z, X, abar, zbar,  
     >         species, chem_id, net_iso, xa,
     >         Rho, logRho, T, logT, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, info)
         if (info /= 0) then
            write(*,*) 'failed in test1_eos'
            stop 1
         end if
         
         Pgas = exp_cr(res(i_lnPgas))
         Prad = Radiation_Pressure(T)
         P = Pgas + Prad
         
         write(*,1) 'T', T
         write(*,1) 'logT', logT
         write(*,1) 'Rho', Rho
         write(*,1) 'logRho', logRho
         write(*,1) 'abar', abar
         write(*,1) 'zbar', zbar
         write(*,1) 'X', X
         write(*,1) 'Z', Z
         write(*,*)
         write(*,1) 'grad_ad', res(i_grad_ad)
         write(*,1) 'd_dlnd(i_grad_ad)', d_dlnd(i_grad_ad)
         write(*,1) 'd_dlnT(i_grad_ad)', d_dlnT(i_grad_ad)
         write(*,1) 'chiT', res(i_chiT)
         write(*,1) 'd_dlnd(i_chiT)', d_dlnd(i_chiT)
         write(*,1) 'd_dlnT(i_chiT)', d_dlnT(i_chiT)
         write(*,1) 'Pgas', Pgas
         write(*,1) 'logPgas', res(i_lnPgas)/ln10
         !return
         
         Pgas = exp_cr(res(i_lnPgas))
         Prad = Radiation_Pressure(T)
         P = Pgas + Prad
         
         write(*,1) 'P', P
         write(*,1) 'logP', log10_cr(P)
         write(*,1) 'Pgas', Pgas
         write(*,1) 'logPgas', res(i_lnPgas)/ln10
         write(*,1) 'logE', res(i_lnE)/ln10
         write(*,1) 'logS', res(i_lnS)/ln10
         write(*,1) 'chiRho', res(i_chiRho)
         write(*,1) 'Cp', res(i_Cp)
         write(*,1) 'Cv', res(i_Cv)
         write(*,1) 'dE_dRho', res(i_dE_dRho)
         write(*,1) 'dS_dT', res(i_dS_dT)
         write(*,1) 'dS_dRho', res(i_dS_dRho)
         write(*,1) 'mu', res(i_mu)
         write(*,1) 'lnfree_e', res(i_lnfree_e)
         write(*,1) 'gamma1', res(i_gamma1)
         write(*,1) 'gamma3', res(i_gamma3)
         write(*,1) 'eta', res(i_eta)
         write(*,*)
         stop

      end subroutine test_eosDT
      
      
      subroutine test_eosDT_partials
         real(dp) :: logT, logRho, T, Rho, X, Z, logPgas, 
     >         Pgas, Prad, P, lnd, dlnd, Rho2, logRho2,
     >         dlnd_analytic, dlnd_numerical
         real(dp), dimension(num_eos_basic_results) :: 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, 
     >         res2, d_dlnd2, d_dlnT2, d_dabar2, d_dzbar2
         integer :: info, i
      
         include 'formats.dek'

         call Setup_eos
         X =    0.8d0
         Z =    0.02d0
         abar = 1.1817162154266392
         zbar = 1.0635445938839752

         logT = 6.5d0
         T = exp10_cr(logT)
         logRho = -1d0
         Rho = exp10_cr(logRho)
         
         info = 0
         call eosDT_get(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         Rho, logRho, T, logT, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, info)
         if (info /= 0) then
            write(*,*) 'failed in test1_eos'
            stop 1
         end if
         
         lnd = logRho*ln10
         dlnd = lnd*1d-4
         logRho2 = (lnd + dlnd)/ln10
         Rho2 = exp10_cr(logRho2)

         call eosDT_get(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         Rho2, logRho2, T, logT, 
     >         res2, d_dlnd2, d_dlnT2, d_dabar2, d_dzbar2, info)
         if (info /= 0) then
            write(*,*) 'failed in test1_eos'
            stop 1
         end if
         
         d_dlnd2(:) = (res2(:) - res(:)) / dlnd
         
         do i=1,num_eos_basic_results
            dlnd_analytic = d_dlnd(i)
            dlnd_numerical = d_dlnd2(i)
            write(*,2) 'dlnd_analytic, dlnd_numerical, diff, rel diff', i, 
     >         dlnd_analytic, dlnd_numerical, dlnd_analytic - dlnd_numerical,
     >         (dlnd_analytic - dlnd_numerical) / max(1d-99, abs(dlnd_analytic), abs(dlnd_numerical))
            write(*,*)
         end do
         
         write(*,1) 'logRho', logRho
         write(*,1) 'logRho2', logRho2
         write(*,1) 'logT', logT
         write(*,*)
         stop

      end subroutine test_eosDT_partials
      
      
      subroutine test_HELM_OPAL_transition_T
         real(dp), dimension(num_eos_basic_results) :: 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar
         real(dp) :: X, Zinit, dXO, dXC, Z, Y, T, rho, 
     >         Prad, Pgas, P, logT_all_HELM, logT_all_OPAL
         logical, parameter :: quietly = .true.
         integer :: ierr
         
         include 'formats.dek'

         call Setup_eos

         ! pure Helium
         X = 0.00d0
         Zinit = 0.00d0
         dXO = 0.00d0
         dXC = 0.00d0
         
         
         Z = Zinit + dXC + dXO
         Y = 1 - (X+Z)
      
         T = 1d7; rho = 1d-3

         write(*,1) 'T', T
         write(*,1) 'rho', rho
         write(*,*)
         
         logT_all_HELM = 6d0
         logT_all_OPAL = logT_all_HELM - 0.1d0
         call do1
         
         logT_all_HELM = 7d0
         logT_all_OPAL = logT_all_HELM - 0.1d0
         call do1
         
         logT_all_HELM = 8d0
         logT_all_OPAL = logT_all_HELM - 0.1d0
         call do1
      
         stop 'test_HELM_OPAL_transition_T'
         
         contains
         
         subroutine do1
            include 'formats.dek'
            call eos_set_HELM_OPAL_lgTs(handle, logT_all_HELM, logT_all_OPAL, ierr)
            if (ierr /= 0) stop 1
            write(*,1) 'logT_all_HELM', logT_all_HELM
            call Do_One_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res) ! opal
            Prad = crad*T*T*T*T/3
            Pgas = exp_cr(res(i_lnPgas))
            P = Pgas + Prad
            write(*,1) 'P', P
            write(*,*)
         end subroutine do1
         
      end subroutine test_HELM_OPAL_transition_T
      
      
      
      
      
      
      subroutine Do_One(quietly)
         logical, intent(in) :: quietly
         real(dp) :: T, rho, log10_rho, log10_T
         real(dp), dimension(num_eos_basic_results) :: res, d_dlnd, d_dlnT
         integer :: info, i
         real(dp) :: XC, XO, Ah, Zh, Yh, Ahe, Zhe, Yhe, Az, Zz, Yz, ddabar, ddzbar, 
     >         helm_ddXh, helm_ddXz, opal_ddXh, opal_ddXz, XC0, XO0,
     >         helm_P, helm_PX, helm_PZ, opal_P, opal_PX, opal_PZ, X1, X2, Z1, Z2,
     >         abar1, zbar1, dXC, dlnP_dabar, dlnP_dzbar, dlnP_dXC,
     >         dabar_dZ, dzbar_dZ, dlnP_dZ, P, logRhoguess
         
         if (.true.) then
            ! pure Helium
            X = 0.00d0
            Zinit = 0.00d0
            dXO = 0.00d0
            dXC = 0.00d0
            call doit('pure Helium')
            ! pure Hydrogen
            X = 1.00d0
            Zinit = 0.00d0
            dXO = 0.00d0
            dXC = 0.00d0
            call doit('pure Hydrogen')
            ! mixed Z with H&He 3:1 ratio
            Zinit = 0.03d0
            X = 0.75d0*(1 - Zinit)
            dXO = 0.00d0
            dXC = 0.00d0
            call doit('mixed Z with H&He 3:1 ratio')
            ! solar
            X = 0.70d0
            Zinit = 0.02d0
            dXO = 0.00d0
            dXC = 0.00d0
            call doit('solar')
         end if
         
         if (.true.) then ! do get_Rho and get_T
            X = 0.70d+00
            Zinit = 0.02d0
            dXO = 0.00d0
            dXC = 0.00d0
            T = exp10_cr(4.8d0)
            rho = 1d-7
            Z = Zinit + dXC + dXO
            Y = 1 - (X+Z)
            call Init_Composition(X, Zinit, dXC, dXO)
            res(i_lnS) = log_cr(2.9680645120000000d+09)
            call test_get_Rho_T
            if (.not. quietly) write(*,*)
         end if
         
         contains

         
         subroutine doit(str)
            character (len=*), intent(in) :: str
            
            if (.false.) then
               T = 2d8; rho = 100
               call Do_One_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res) ! scvh
               stop
            end if
            
            if (.not. quietly) write(*,*) trim(str)

            
            T = 1d2; rho = 1d-8
            call Do_One_Ideal_Gas_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res) 
            T = 1d6; rho = 1d-2
            call Do_One_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res) ! opal
            T = 1d4; rho = 1d-1
            call Do_One_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res) ! scvh
            T = 1d5; rho = 1d-1
            call Do_One_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res) ! opal-scvh overlap
            T = 2d8; rho = 1d2
            call Do_One_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res)  ! helm
            
            if (.not. quietly) write(*,*)
            
         end subroutine


         subroutine test_get_Rho_T ! using most recent values from subroutine Do_One_TRho
            real(dp) :: tol, othertol,
     >         result, result_log10, log10_T, log10_rho, lnS,
     >         clipped_log10rho, clipped_log10temp,
     >         logRho_guess, logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2,
     >         logT_guess, logT_bnd1, logT_bnd2
            integer :: i, which_other, max_iter, eos_calls, ierr
            real(dp), dimension(num_eos_basic_results) :: 
     >            d_dlnd, d_dlnT, d_dabar, d_dzbar
            type (EoS_General_Info), pointer :: rq
            logical, parameter :: use_log10_for_other = .false.
            
            if (.not. quietly) write(*,*)
            
            call eos_ptr(handle,rq,ierr)
            if (ierr /= 0) stop 1
            
            log10_rho = log10_cr(rho)
            log10_T = log10_cr(T)
            
            lnS = res(i_lnS)

            ierr = 0
            max_iter = 100
            tol = 1d-4
            othertol = 1d-12

 1          format(a30,1pe24.16)
            
            if (.not. quietly) then
               write(*,*)
               write(*,1) ' tolerance', tol
            end if
            if (.not. quietly) write(*,*)
            result = rho*0.5d0 ! initial guess
            result_log10 = log10_cr(result)
            res = 0
            logRho_guess = result_log10
            logRho_bnd1 = arg_not_provided
            logRho_bnd2 = arg_not_provided
            other_at_bnd1 = arg_not_provided
            other_at_bnd2 = arg_not_provided
            call eosDT_get_Rho(
     >            handle, Z, X, abar, zbar, 
     >            species, chem_id, net_iso, xa,
     >            log10_T, i_lnS, lnS,
     >            tol, othertol, max_iter, logRho_guess, 
     >            logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2,
     >            result_log10, res, d_dlnd, d_dlnT, 
     >            d_dabar, d_dzbar, eos_calls, ierr)
            result = exp10_cr(result_log10)
            if (ierr /= 0) then
               write(*,*) 'ierr in test_get_Rho_T'
               stop 2
            end if
            if (.not. quietly) then
               write(*,1) 'actual logRho', log10_rho
               write(*,1) ' guess logRho', logRho_guess
               write(*,1) ' found logRho', result_log10
               write(*,1) '  wanted logS', lnS/ln10
               write(*,1) '     got logS', res(i_lnS)/ln10
               write(*,*)
            end if
         
            result = T*2d0 ! initial guess
            result_log10 = log10_cr(result)
            res = 0
            logT_guess = result_log10
            logT_bnd1 = 3
            logT_bnd2 = 9
            call eosDT_get_T(
     >            handle, Z, X, abar, zbar, 
     >            species, chem_id, net_iso, xa,
     >            log10_rho, i_lnS, lnS,
     >            tol, othertol, max_iter, logT_guess, 
     >            logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, 
     >            result_log10, res, d_dlnd, d_dlnT,
     >            d_dabar, d_dzbar, eos_calls, ierr)
            result = exp10_cr(result_log10)
            if (ierr /= 0) then
               write(*,*) 'ierr in test_get_Rho_T'
               stop 1
            end if
            if (.not. quietly) then
               write(*,*)
               write(*,1) 'actual logT', log10_T
               write(*,1) ' guess logT', logT_guess
               write(*,1) ' found logT', result_log10
               write(*,1) '  wanted logS', lnS/ln10
               write(*,1) '     got logS', res(i_lnS)/ln10
               write(*,*)
            end if
         
         end subroutine test_get_Rho_T

      end subroutine Do_One
      
      
      
      subroutine test1_eosDT_get_T
         
         real(dp) :: 
     >         energy, abar, zbar, X, Z, logRho, logT_tol, other_tol, other,
     >         logT_guess, logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2,
     >         logT_result, new_energy,
     >         res(num_eos_basic_results), d_dlnd(num_eos_basic_results),
     >         d_dabar(num_eos_basic_results), d_dzbar(num_eos_basic_results),
     >         d_dlnT(num_eos_basic_results)      
         integer:: ierr, which_other, eos_calls, max_iter
         logical, parameter :: use_log10_for_other = .false.
         
 1       format(a40,1pe26.16)
 
         call Setup_eos
 
         ierr = 0

                                     Z =    2.0057910422533909D-02
                                     X =    6.8067344312014655D-01
                                  abar =    1.3213531256488331D+00
                                  zbar =    1.1105746510172319D+00
                                logRho =   -7.2923785725386976D+00
                                energy =    1.9116604229463992D+12
                              logT_tol =    9.9999999999999995D-07
                             other_tol =    2.3025850929940459D-07
                            logT_guess =    3.6800855464963247D+00
                             logT_bnd1 =   -8.9999999999999999D+99
                             logT_bnd2 =   -8.9999999999999999D+99
                         other_at_bnd1 =   -8.9999999999999999D+99
                         other_at_bnd2 =   -8.9999999999999999D+99


         which_other = i_lnE
         other = log_cr(energy)
         max_iter = 100


         write(*,1) 'logRho', logRho
         write(*,1) 'logT_guess', logT_guess
         write(*,1) 'logT_bnd1', logT_bnd1
         write(*,1) 'logT_bnd2', logT_bnd2
         write(*,1) 'energy', energy
         write(*,1) 'other', other
         write(*,1) 'Z', Z
         write(*,1) 'X', X
         write(*,1) 'abar', abar
         write(*,1) 'zbar', zbar
         write(*,1) 'logT_tol', logT_tol
         write(*,1) 'other_tol', other_tol
         write(*,*)

         call eosDT_get_T(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         logRho, which_other, other,
     >         logT_tol, other_tol, max_iter, logT_guess, 
     >         logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, 
     >         logT_result, res, d_dlnd, d_dlnT, d_dabar, d_dzbar,
     >         eos_calls, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in test1_eosDT_get_T'
            stop 1
         end if
         write(*,*)
         !write(*,1) 'actual logT', log10_T
         write(*,1) 'guess logT', logT_guess
         write(*,1) 'found logT', logT_result
         write(*,1) 'wanted logE', other/ln10
         write(*,1) 'got logE', res(i_lnE)/ln10
         write(*,*)
         write(*,*) 'eos_calls', eos_calls
         write(*,*)
         
      end subroutine test1_eosDT_get_T
      
      
      subroutine test1_eosDT_get_rho
         
         real(dp) :: 
     >         P, Prad, Pgas, T, rho_guess, abar, zbar, X, Z, logT, logRho_tol, other_tol, other,
     >         logRho_guess, logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, logRho_result,
     >         res(num_eos_basic_results), d_dlnd(num_eos_basic_results), Rho, logRho,
     >         d_dabar(num_eos_basic_results), d_dzbar(num_eos_basic_results), 
     >         d_dlnT(num_eos_basic_results),
     >         dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas
         integer:: ierr, which_other, eos_calls, max_iter
         logical, parameter :: use_log10_for_other = .false.
         
 1       format(a40,1pe26.16)
 
         call Setup_eos
 
         ierr = 0

         Z =     2.0062018516311897D-02
         X =     6.7967739184154219D-01
         abar =     1.3226610221256363D+00
         zbar =     1.1110240673800316D+00
         logT =     3.5999013770057040D+00
         other =     7.7701772367967239D+00
         logRho_tol =     9.9999999999999995D-07
         other_tol =     2.3025850929940460D-06
         logRho_guess =    -8.0208039694507622D+00

         logRho_bnd1 = arg_not_provided
         logRho_bnd2 = arg_not_provided
         other_at_bnd1 = arg_not_provided
         other_at_bnd2 = arg_not_provided

         which_other = i_lnPgas
         max_iter = 100
         
         write(*,1) 'logT', logT
         write(*,1) 'logRho_guess', logRho_guess
         write(*,1) 'Z', Z
         write(*,1) 'X', X
         write(*,1) 'abar', abar
         write(*,1) 'zbar', zbar
         write(*,1) 'logRho_tol', logRho_tol
         write(*,1) 'other_tol', other_tol
         write(*,*)

         call eosDT_get_Rho(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         logT, which_other, other,
     >         logRho_tol, other_tol, max_iter, logRho_guess, 
     >         logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, 
     >         logRho_result, res, d_dlnd, d_dlnT, d_dabar, d_dzbar,
     >         eos_calls, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in test1_eosDT_get_rho'
            stop 1
         end if
         write(*,*)
         write(*,1) 'guess logRho', logRho_guess
         write(*,1) 'found logRho', logRho_result
         write(*,1) 'wanted logPgas', other/ln10
         write(*,1) 'got logPgas', res(i_lnPgas)/ln10
         write(*,*)
         write(*,*) 'eos_calls', eos_calls
         write(*,*)
         
      end subroutine test1_eosDT_get_rho


      
      
      
      subroutine test1_eosPT_get_T
         
         real(dp) :: 
     >         energy, abar, zbar, X, Z, logPgas, logT_tol, other_tol, other,
     >         logT_guess, logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2,
     >         logT_result, new_energy,
     >         res(num_eos_basic_results), d_dlnd(num_eos_basic_results),
     >         d_dabar(num_eos_basic_results), d_dzbar(num_eos_basic_results),
     >         d_dlnT(num_eos_basic_results), 
     >         Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas         
         integer:: ierr, which_other, eos_calls, max_iter
         logical, parameter :: use_log10_for_other = .false.
         
 1       format(a40,1pe26.16)
 
         call Setup_eos
 
         ierr = 0
         
         write(*,*) 'test1_eosPT_get_T'

                                     Z =    0.02
                                     X =    0.70
                                  abar =    1.2966353559153956
                                  zbar =    1.1021400447995373
                                logPgas =   15
                                energy =    exp_cr(3.5034294596213336d+01)
                              logT_tol =    1d-6
                             other_tol =    ln10*1d-6
                            logT_guess =    7
                             logT_bnd1 =   arg_not_provided
                             logT_bnd2 =   arg_not_provided
                         other_at_bnd1 =   arg_not_provided
                         other_at_bnd2 =   arg_not_provided


         which_other = i_lnE
         other = log_cr(energy)
         max_iter = 100


         write(*,1) 'logPgas', logPgas
         write(*,1) 'logT_guess', logT_guess
         write(*,1) 'logT_bnd1', logT_bnd1
         write(*,1) 'logT_bnd2', logT_bnd2
         write(*,1) 'energy', energy
         write(*,1) 'other', other
         write(*,1) 'Z', Z
         write(*,1) 'X', X
         write(*,1) 'abar', abar
         write(*,1) 'zbar', zbar
         write(*,1) 'logT_tol', logT_tol
         write(*,1) 'other_tol', other_tol
         write(*,*)

         call eosPT_get_T(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         logPgas, which_other, other,
     >         logT_tol, other_tol, max_iter, logT_guess, 
     >         logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, 
     >         logT_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar,
     >         eos_calls, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in test1_eosPT_get_T'
            stop 1
         end if
         write(*,*)
         write(*,1) 'guess logT', logT_guess
         write(*,1) 'found logT', logT_result
         write(*,1) 'wanted logE', other/ln10
         write(*,1) 'got logE', res(i_lnE)/ln10
         write(*,*)
         write(*,*) 'eos_calls', eos_calls
         write(*,*)
         
      end subroutine test1_eosPT_get_T
      
      
      subroutine test1_eosPT_get_Pgas
         
         real(dp) :: 
     >         P, Prad, T, Pgas_guess, abar, zbar, X, Z, energy, logT, logPgas_tol, other_tol, other,
     >         logPgas_guess, logPgas_bnd1, logPgas_bnd2, other_at_bnd1, other_at_bnd2, logPgas_result,
     >         res(num_eos_basic_results), d_dlnd(num_eos_basic_results), Pgas, logPgas,
     >         d_dabar(num_eos_basic_results), d_dzbar(num_eos_basic_results),      
     >         d_dlnT(num_eos_basic_results), 
     >         Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,
     >         dlnPgas_dlnPgas_const_T, dlnPgas_dlnT_const_Pgas
         integer:: ierr, which_other, eos_calls, max_iter
         logical, parameter :: use_log10_for_other = .false.
         
 1       format(a40,1pe26.16)
 
         call Setup_eos
 
         ierr = 0

         call Init_Composition(X, Z, 0d0, 0d0) ! sets abar and zbar
         
         write(*,*) 'test1_eosPT_get_Pgas'


                                     Z =    0.02
                                     X =    0.70
                                  abar =    1.2966353559153956
                                  zbar =    1.1021400447995373
                                  logT =    6.9
                                energy = 1.6413485676831915d+15
                              logPgas_tol =    1d-6
                             other_tol =    ln10*1d-6
                            logPgas_guess = 14
                             logPgas_bnd1 =   arg_not_provided
                             logPgas_bnd2 =   arg_not_provided
                         other_at_bnd1 =   arg_not_provided
                         other_at_bnd2 =   arg_not_provided


         which_other = i_lnE
         other = log_cr(energy)
         max_iter = 100


         write(*,1) 'logT', logT
         write(*,1) 'logPgas_guess', logPgas_guess
         write(*,1) 'logPgas_bnd1', logPgas_bnd1
         write(*,1) 'logPgas_bnd2', logPgas_bnd2
         write(*,1) 'energy', energy
         write(*,1) 'other', other
         write(*,1) 'Z', Z
         write(*,1) 'X', X
         write(*,1) 'abar', abar
         write(*,1) 'zbar', zbar
         write(*,1) 'logPgas_tol', logPgas_tol
         write(*,1) 'other_tol', other_tol
         write(*,*)

         call eosPT_get_Pgas(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         logT, which_other, other,
     >         logPgas_tol, other_tol, max_iter, logPgas_guess, 
     >         logPgas_bnd1, logPgas_bnd2, other_at_bnd1, other_at_bnd2, 
     >         logPgas_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar,
     >         eos_calls, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in test1_eosPT_get_Pgas'
            stop 1
         end if
         write(*,*)
         write(*,1) 'guess logPgas', logPgas_guess
         write(*,1) 'found logPgas', logPgas_result
         write(*,1) 'wanted logE', other/ln10
         write(*,1) 'got logE', res(i_lnE)/ln10
         write(*,1) 'got log10Rho', log10Rho
         write(*,*)
         write(*,*) 'eos_calls', eos_calls
         write(*,*)
         
         
      end subroutine test1_eosPT_get_Pgas


      
      
      subroutine test1_eosPT_get_Pgas_for_Rho
         
         real(dp) :: 
     >         P, Prad, T, Pgas_guess, abar, zbar, X, Z, energy, logT, logPgas_tol, logRho_tol, logRho_want,
     >         logPgas_guess, logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2, logPgas_result,
     >         res(num_eos_basic_results), d_dlnd(num_eos_basic_results), Pgas, logPgas,
     >         d_dabar(num_eos_basic_results), d_dzbar(num_eos_basic_results), 
     >         d_dlnT(num_eos_basic_results), 
     >         Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,
     >         dlnPgas_dlnPgas_const_T, dlnPgas_dlnT_const_Pgas
         integer:: ierr, which_other, eos_calls, max_iter
         logical, parameter :: use_log10_for_other = .false.
         
 1       format(a40,1pe26.16)
 
         call Setup_eos
 
         ierr = 0

         call Init_Composition(X, Z, 0d0, 0d0) ! sets abar and zbar
         
         write(*,*) 'test1_eosPT_get_Pgas_for_Rho'


                                     Z =    1.4331866202912824D-02
                                     X =    7.4801683389855433D-01
                                  abar =    1.2372753933790150D+00
                                  zbar =    1.0813484340002424D+00
                                  logT =    7.6013756122152900D+00
                              logPgas_tol =    1d-6
                              logRho_tol = 1d-6
                            logPgas_guess = 1.7508981610296928D+01
                             logPgas_bnd1 =   arg_not_provided
                             logPgas_bnd2 =   arg_not_provided
                         logRho_at_bnd1 =   arg_not_provided
                         logRho_at_bnd2 =   arg_not_provided

         max_iter = 100
         logRho_want = 1.7645028058855445D+00

         write(*,1) 'logT', logT
         write(*,1) 'logPgas_guess', logPgas_guess
         write(*,1) 'logPgas_bnd1', logPgas_bnd1
         write(*,1) 'logPgas_bnd2', logPgas_bnd2
         write(*,1) 'logRho_want', logRho_want
         write(*,1) 'Z', Z
         write(*,1) 'X', X
         write(*,1) 'abar', abar
         write(*,1) 'zbar', zbar
         write(*,1) 'logPgas_tol', logPgas_tol
         write(*,1) 'logRho_tol', logRho_tol
         write(*,*)

         call eosPT_get_Pgas_for_Rho(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         logT, logRho_want,
     >         logPgas_tol, logRho_tol, max_iter, logPgas_guess, 
     >         logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2, 
     >         logPgas_result, Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar,
     >         eos_calls, ierr)
         if (ierr /= 0) then
            write(*,*) 'ierr in test1_eosPT_get_Pgas_for_Rho'
            stop 1
         end if
         write(*,*)
         write(*,1) 'guess logPgas', logPgas_guess
         write(*,1) 'found logPgas', logPgas_result
         write(*,1) 'wanted logRho', logRho_want
         write(*,1) 'got logRho', logRho
         write(*,*)
         write(*,*) 'eos_calls', eos_calls
         write(*,*)
         
         
      end subroutine test1_eosPT_get_Pgas_for_Rho



         
      
      subroutine Do_One_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res)
         logical, intent(in) :: quietly
         real(dp), intent(in) :: T, Rho, X, Zinit, dXC, dXO
         real(dp), intent(out) :: Y, Z
         real(dp), intent(out), dimension(num_eos_basic_results) :: res

         real(dp), dimension(num_eos_basic_results) :: 
     >         d_dlnd, d_dlnT, d_dabar, d_dzbar
         integer :: info, i
         real(dp) :: dlnT, dlnRho, lnRho_2, Prad, Pgas, P


         ! for offload, need local copies of globals
         integer :: handle1, species1
         real(dp) :: abar1, zbar1, arg_not_provided1
         integer, pointer, dimension(:) :: chem_id1, net_iso1
         real(dp) :: xa1(species)
         
         handle1 = handle
         species1 = species
         arg_not_provided1 = arg_not_provided
         abar1 = abar      
         zbar1 = zbar
         chem_id1 => chem_id
         net_iso1 => net_iso
         xa1 = xa


  101    format(a30,4x,1pe24.16)
  102    format(a30,3x,1pe24.16)
         
         
         Z = Zinit + dXC + dXO
         Y = 1 - (X+Z)
                        
         call Init_Composition(X, Zinit, dXC, dXO)
         
         if (.not. quietly) then
            write(*,*)
            write(*,*)
            write(*,102) 'X', X
            write(*,102) 'Y', Y
            write(*,102) 'Z', Z
            write(*,102) 'abar', abar
            write(*,102) 'zbar', zbar
            write(*,102) 'logRho', log10_cr(Rho)
            write(*,102) 'logT', log10_cr(T)
            write(*,102) 'T6', T * 1d-6
            write(*,*)
         end if
         
         if (.true.) call eosDT_get(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         Rho, arg_not_provided, T, arg_not_provided, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, info)
         if (.true. .and. info /= 0) then
            write(*,*) 'info', info, 'Rho', Rho, 'T', T
            write(*,*) 'failed in Do_One_TRho'
            stop 1
         end if

#ifdef offload
         !dir$ offload target(mic) 
     >      in(handle1, Z, X, abar1, zbar1, 
     >         species1, chem_id1, net_iso1, xa1,
     >         Rho, arg_not_provided1, T)
     >      out(res, d_dlnd, d_dlnT, d_dabar, d_dzbar, info) 
#endif
         call eosDT_get(
     >         handle1, Z, X, abar1, zbar1, 
     >         species1, chem_id1, net_iso1, xa1,
     >         Rho, arg_not_provided1, T, arg_not_provided1, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, info)
         if (info /= 0) then
            write(*,*) 'info', info, 'Rho', Rho, 'T', T
            write(*,*) 'failed in Do_One_TRho'
            stop 1
         end if
         
         if (.not. quietly) then
         
            write(*,*) 'eosDT_get'
            Prad = crad*T*T*T*T/3
            Pgas = exp_cr(res(i_lnPgas))
            P = Pgas + Prad
            write(*,101) 'P', P
            write(*,101) 'E', exp_cr(res(i_lnE))
            write(*,101) 'S', exp_cr(res(i_lnS))
            do i = 4, 9
               write(*,101) trim(eos_names(i)), res(i)
            end do
            write(*,101) trim(eos_names(i_gamma1)), res(i_gamma1)
            write(*,101) trim(eos_names(i_gamma3)), res(i_gamma3)
            write(*,101) trim(eos_names(i_eta)), res(i_eta)
            
            if (.false.) then ! debugging
               do i = 1, num_eos_basic_results
                  write(*,101) 'd_dlnd ' // trim(eos_names(i)), d_dlnd(i)
               end do
               write(*,*)
               do i = 1, num_eos_basic_results
                  write(*,101) 'd_dlnT ' // trim(eos_names(i)), d_dlnT(i)
               end do
               write(*,*)
            end if
            
         end if

      end subroutine Do_One_TRho

      
      subroutine Do_One_Ideal_Gas_TRho(quietly,T,Rho,X,Zinit,dXC,dXO,Y,Z,res)
         logical, intent(in) :: quietly
         real(dp), intent(in) :: T, Rho, X, Zinit, dXC, dXO
         real(dp), intent(out) :: Y, Z
         real(dp), intent(out), dimension(num_eos_basic_results) :: res

         real(dp), dimension(num_eos_basic_results) :: 
     >         d_dlnd, d_dlnT, d_dabar, d_dzbar
         integer :: info, i
         real(dp) :: dlnT, dlnRho, lnRho_2, Prad, Pgas, P

         ! for offload, need local copies of globals
         integer :: handle1, species1
         real(dp) :: abar1, zbar1, arg_not_provided1
         integer, pointer, dimension(:) :: chem_id1, net_iso1
         real(dp) :: xa1(species)
         
         handle1 = handle
         species1 = species
         arg_not_provided1 = arg_not_provided
         abar1 = abar      
         zbar1 = zbar
         chem_id1 => chem_id
         net_iso1 => net_iso
         xa1 = xa


  101    format(a30,4x,1pe24.16)
  102    format(a30,3x,1pe24.16)
         
         
         Z = Zinit + dXC + dXO
         Y = 1 - (X+Z)
                        
         call Init_Composition(X, Zinit, dXC, dXO)
         
         if (.not. quietly) then
            write(*,*)
            write(*,*)
            write(*,102) 'X', X
            write(*,102) 'Y', Y
            write(*,102) 'Z', Z
            write(*,102) 'abar', abar
            write(*,102) 'zbar', zbar
            write(*,102) 'logRho', log10_cr(Rho)
            write(*,102) 'logT', log10_cr(T)
            write(*,102) 'T6', T * 1d-6
            write(*,*)
         end if
         
         if (.true.) call eosDT_ideal_gas_get(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         Rho, arg_not_provided, T, arg_not_provided, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, info)
         if (.true. .and. info /= 0) then
            write(*,*) 'info', info, 'Rho', Rho, 'T', T
            write(*,*) 'failed in eosDT_ideal_gas_get'
            stop 1
         end if

#ifdef offload
         !dir$ offload target(mic) 
     >      in(handle1, Z, X, abar1, zbar1, 
     >         species1, chem_id1, net_iso1, xa1,
     >         Rho, arg_not_provided1, T)
     >      out(res, d_dlnd, d_dlnT, d_dabar, d_dzbar, info) 
#endif
         call eosDT_ideal_gas_get(
     >         handle, Z, X, abar, zbar, 
     >         species, chem_id, net_iso, xa,
     >         Rho, arg_not_provided, T, arg_not_provided, 
     >         res, d_dlnd, d_dlnT, d_dabar, d_dzbar, info)
         if (info /= 0) then
            write(*,*) 'info', info, 'Rho', Rho, 'T', T
            write(*,*) 'failed in eosDT_ideal_gas_get'
            stop 1
         end if
         
         if (.not. quietly) then
            
            write(*,*) 'eosDT_ideal_gas_get'
            Prad = crad*T*T*T*T/3
            Pgas = exp_cr(res(i_lnPgas))
            P = Pgas + Prad
            write(*,101) 'P', P
            write(*,101) 'E', exp_cr(res(i_lnE))
            write(*,101) 'S', exp_cr(res(i_lnS))
            do i = 4, 9
               write(*,101) trim(eos_names(i)), res(i)
            end do
            write(*,101) trim(eos_names(i_gamma1)), res(i_gamma1)
            write(*,101) trim(eos_names(i_gamma3)), res(i_gamma3)
            write(*,101) trim(eos_names(i_eta)), res(i_eta)
            
            if (.false.) then ! debugging
               do i = 1, num_eos_basic_results
                  write(*,101) 'd_dlnd ' // trim(eos_names(i)), d_dlnd(i)
               end do
               write(*,*)
               do i = 1, num_eos_basic_results
                  write(*,101) 'd_dlnT ' // trim(eos_names(i)), d_dlnT(i)
               end do
               write(*,*)
               stop
            end if
            
         end if

      end subroutine Do_One_Ideal_Gas_TRho
      
      
#ifdef offload
      !dir$ attributes offload : mic :: test_theta_e
#endif      
      subroutine test_theta_e
         integer :: i
         real(dp), parameter :: eta_min = -10
         real(dp), parameter :: eta_max = 75
         real(dp), parameter :: d_eta = 5d0
         real(dp) :: eta, theta_e, d_theta_e_deta
 3       format(f14.2,2x,2(1pe16.6,2x))
         write(*,*) 'test_theta_e'
         eta = eta_min
         write(*,'(a14,99a18)') 'eta', 'theta_e'
         do while (eta <= eta_max)
            theta_e = eos_theta_e(eta, d_theta_e_deta)
            write(*,3) eta, theta_e
            eta = eta + d_eta
         end do
         write(*,*)
      end subroutine test_theta_e
      
      
      subroutine test_dirac_integrals
         real(dp) :: dk, T, eta, theta, fdph, fdmh, fdeta, fdtheta, theta_e
 1       format(a40,1pe26.16)
         eta = 1.46722890948893d0    
         T = 11327678.5183021d0
         theta = (kerg*T)/(me*clight*clight)
         !zt   1.55623520289424d0      
         theta_e = 0.929542529701454d0
         call eos_fermi_dirac_integral(-0.5d0, eta, theta, fdmh, fdeta, fdtheta)
         call eos_fermi_dirac_integral(0.5d0, eta, theta, fdph, fdeta, fdtheta)
         write(*,*)
         write(*,*) 'test_dirac_integrals'
         write(*,1) 'calculated theta_e', fdmh/fdph
         write(*,*)
         stop
      end subroutine test_dirac_integrals


      end module test_eos_support  
