! ***********************************************************************
!
!   Copyright (C) 2010  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************




      module eos_lib

      use const_def, only: dp
      
      implicit none

      contains ! the procedure interface for the library
      ! client programs should only call these routines.
            
            
      subroutine eos_init( &
            eos_file_prefix, eosDT_cache_dir, eosPT_cache_dir, eosDE_cache_dir, &
            use_cache, info)      
         use eos_initialize, only : Init_eos
         character(*), intent(in) :: eos_file_prefix ! 'mesa' is the usual choice
         character(*), intent(in) :: eosDT_cache_dir ! '' means use default
         character(*), intent(in) :: eosPT_cache_dir ! '' means use default
         character(*), intent(in) :: eosDE_cache_dir ! '' means use default
         logical, intent(in) :: use_cache
         integer, intent(out) :: info ! 0 means AOK. 
         info = 0
         call Init_eos( &
            eos_file_prefix, eosDT_cache_dir, eosPT_cache_dir, eosDE_cache_dir, &
            use_cache, info)
         if (info /= 0) return      
      end subroutine eos_init
      
      
      subroutine eos_shutdown
         use eos_def
         use helm_alloc,only:free_helm_table
         integer :: ix, iz
         if (associated(eos_ht)) call free_helm_table(eos_ht)
         do ix = 1, num_eos_Xs
            do iz = 1, num_eosDT_Zs
               if (associated(eosDT_XZ_data(ix,iz)% tbl1)) then
                  deallocate(eosDT_XZ_data(ix,iz)% tbl1)
                  nullify(eosDT_XZ_data(ix,iz)% tbl1)
               end if
               if (associated(eosDT_XZ_data(ix,iz)% logQs)) then
                  deallocate(eosDT_XZ_data(ix,iz)% logQs)
                  nullify(eosDT_XZ_data(ix,iz)% logQs)
               end if
               if (associated(eosDT_XZ_data(ix,iz)% logTs)) then
                  deallocate(eosDT_XZ_data(ix,iz)% logTs)
                  nullify(eosDT_XZ_data(ix,iz)% logTs)
               end if
            end do
            do iz = 1, num_eosPT_Zs
               if (associated(eosPT_XZ_data(ix,iz)% tbl1)) then
                  deallocate(eosPT_XZ_data(ix,iz)% tbl1)
                  nullify(eosPT_XZ_data(ix,iz)% tbl1)
               end if
               if (associated(eosPT_XZ_data(ix,iz)% logWs)) then
                  deallocate(eosPT_XZ_data(ix,iz)% logWs)
                  nullify(eosPT_XZ_data(ix,iz)% logWs)
               end if
               if (associated(eosPT_XZ_data(ix,iz)% logTs)) then
                  deallocate(eosPT_XZ_data(ix,iz)% logTs)
                  nullify(eosPT_XZ_data(ix,iz)% logTs)
               end if
            end do
            do iz = 1, num_eosDE_Zs
               if (associated(eosDE_XZ_data(ix,iz)% tbl1)) then
                  deallocate(eosDE_XZ_data(ix,iz)% tbl1)
                  nullify(eosDE_XZ_data(ix,iz)% tbl1)
               end if
               if (associated(eosDE_XZ_data(ix,iz)% logVs)) then
                  deallocate(eosDE_XZ_data(ix,iz)% logVs)
                  nullify(eosDE_XZ_data(ix,iz)% logVs)
               end if
               if (associated(eosDE_XZ_data(ix,iz)% logEs)) then
                  deallocate(eosDE_XZ_data(ix,iz)% logEs)
                  nullify(eosDE_XZ_data(ix,iz)% logEs)
               end if
            end do
         end do
         eos_root_is_initialized = .false.
         eosDT_is_initialized = .false.
         eosPT_is_initialized = .false.
         eosDE_is_initialized = .false.
      end subroutine eos_shutdown
      
      
      subroutine set_eos_Z1_suffix(DT_Z1_suffix, PT_Z1_suffix) 
         ! suffix for Z=1.0 data file
         ! e.g., '_CO_1' or '_CO_0' or ''
         use eosDT_load_tables, only : eosDT_Z1_suffix
         use eosPT_load_tables, only : eosPT_Z1_suffix
         character(*), intent(IN) :: DT_Z1_suffix, PT_Z1_suffix
         eosDT_Z1_suffix = DT_Z1_suffix
         eosPT_Z1_suffix = PT_Z1_suffix
         !write(*,*) 'eosDT_Z1_suffix ' // trim(eosDT_Z1_suffix)
         !write(*,*) 'eosPT_Z1_suffix ' // trim(eosPT_Z1_suffix)
      end subroutine set_eos_Z1_suffix
      
      
      ! after eos_init has finished, you can allocate a "handle".
      
      integer function alloc_eos_handle(ierr)
         use eos_def,only:do_alloc_eos
         integer, intent(out) :: ierr ! 0 means AOK.   
         alloc_eos_handle = do_alloc_eos(ierr)
         if (ierr /= 0) return
      end function alloc_eos_handle      
      
      subroutine free_eos_handle(handle)
         ! frees the handle and all associated data
         use eos_def,only:do_free_eos_handle
         integer, intent(in) :: handle
         call do_free_eos_handle(handle)
      end subroutine free_eos_handle      
      
      
      subroutine eos_set_HELM_flags( &
            handle, include_radiation, always_skip_elec_pos, ierr)
         use eos_def, only: do_set_HELM_flags
         integer, intent(in) :: handle
         logical, intent(in) :: include_radiation, always_skip_elec_pos
         integer, intent(out) :: ierr ! 0 means AOK.
         call do_set_HELM_flags( &
            handle, include_radiation, always_skip_elec_pos, ierr)
         if (ierr /= 0) return
#ifdef offload
         !dir$ offload target(mic) out(ierr) &
            in(handle, include_radiation, always_skip_elec_pos)
         call do_set_HELM_flags( &
            handle, include_radiation, always_skip_elec_pos, ierr)
#endif
      end subroutine eos_set_HELM_flags

      
      subroutine eos_get_HELM_flags( &
            handle, include_radiation, always_skip_elec_pos, ierr)
         use eos_def
         integer, intent(in) :: handle
         logical, intent(out) :: include_radiation, always_skip_elec_pos
         integer, intent(out) :: ierr ! 0 means AOK.
         type (EoS_General_Info), pointer :: rq
         call eos_ptr(handle,rq,ierr)
         if (ierr /= 0) return
         include_radiation = rq% include_radiation
         always_skip_elec_pos = rq% always_skip_elec_pos
      end subroutine eos_get_HELM_flags

      
      subroutine eos_set_Z_all_HELM(handle, Z_all_HELM, ierr)
         use eos_def, only: do_set_Z_all_HELM
         integer, intent(in) :: handle
         real(dp), intent(in) :: Z_all_HELM ! HELM for Z >= this
         integer, intent(out) :: ierr ! 0 means AOK.
         call do_set_Z_all_HELM(handle, Z_all_HELM, ierr)
         if (ierr /= 0) return
#ifdef offload
         !dir$ offload target(mic) out(ierr) in(handle, Z_all_HELM)
         call do_set_Z_all_HELM(handle, Z_all_HELM, ierr)
#endif
      end subroutine eos_set_Z_all_HELM

      
      subroutine eos_get_Z_all_HELM(handle, Z_all_HELM, ierr)
         use eos_def
         integer, intent(in) :: handle
         real(dp), intent(out) :: Z_all_HELM ! HELM for Z >= this
         integer, intent(out) :: ierr ! 0 means AOK.
         type (EoS_General_Info), pointer :: rq
         call eos_ptr(handle,rq,ierr)
         if (ierr /= 0) return
         Z_all_HELM = rq% Z_all_HELM
      end subroutine eos_get_Z_all_HELM

      
      subroutine eos_set_logRhos_OPAL_SCVH( &
            handle, logRho1_OPAL_SCVH_limit, logRho2_OPAL_SCVH_limit, ierr)
         use eos_def, only: do_set_logRhos_OPAL_SCVH
         integer, intent(in) :: handle
         real(dp), intent(in) :: logRho1_OPAL_SCVH_limit 
            ! don't use OPAL_SCVH for logRho > this
         real(dp), intent(in) :: logRho2_OPAL_SCVH_limit 
            ! full OPAL_SCVH okay for logRho < this
         integer, intent(out) :: ierr ! 0 means AOK.
         call do_set_logRhos_OPAL_SCVH( &
            handle, logRho1_OPAL_SCVH_limit, logRho2_OPAL_SCVH_limit, ierr)
         if (ierr /= 0) return
#ifdef offload
         !dir$ offload target(mic) out(ierr) &
            in(handle, logRho1_OPAL_SCVH_limit, logRho2_OPAL_SCVH_limit)
         call do_set_logRhos_OPAL_SCVH( &
            handle, logRho1_OPAL_SCVH_limit, logRho2_OPAL_SCVH_limit, ierr)
#endif
      end subroutine eos_set_logRhos_OPAL_SCVH

      
      subroutine eos_get_logRhos_OPAL_SCVH( &
            handle, logRho1_OPAL_SCVH_limit, logRho2_OPAL_SCVH_limit, ierr)
         use eos_def
         integer, intent(in) :: handle
         real(dp), intent(out) :: logRho1_OPAL_SCVH_limit ! don't use OPAL_SCVH for logRho > this
         real(dp), intent(out) :: logRho2_OPAL_SCVH_limit ! full OPAL_SCVH okay for logRho < this
            ! don't use OPAL_SCVH for logRho > this
         integer, intent(out) :: ierr ! 0 means AOK.
         type (EoS_General_Info), pointer :: rq
         call eos_ptr(handle,rq,ierr)
         if (ierr /= 0) return
         logRho1_OPAL_SCVH_limit = rq% logRho1_OPAL_SCVH_limit
         logRho2_OPAL_SCVH_limit = rq% logRho2_OPAL_SCVH_limit
      end subroutine eos_get_logRhos_OPAL_SCVH

      
      ! transition temperature zone for OPAL to HELM at high T
      subroutine eos_set_HELM_OPAL_lgTs(handle, logT_all_HELM, logT_all_OPAL, ierr)
         use eos_def, only: do_set_HELM_OPAL_lgTs
         integer, intent(in) :: handle
         real(dp), intent(in) :: logT_all_HELM ! HELM for lgT >= this
         real(dp), intent(in) :: logT_all_OPAL ! OPAL/SCVH for lgT <= this
         integer, intent(out) :: ierr ! 0 means AOK.
         call do_set_HELM_OPAL_lgTs( &
            handle, logT_all_HELM, logT_all_OPAL, ierr)
         if (ierr /= 0) return
#ifdef offload
         !dir$ offload target(mic) out(ierr) &
            in(handle, logT_all_HELM, logT_all_OPAL)
         call do_set_HELM_OPAL_lgTs( &
            handle, logT_all_HELM, logT_all_OPAL, ierr)
#endif
      end subroutine eos_set_HELM_OPAL_lgTs
      
      
      subroutine eos_get_HELM_OPAL_lgTs(handle, logT_all_HELM, logT_all_OPAL, ierr)
         use eos_def
         integer, intent(in) :: handle
         real(dp), intent(out) :: logT_all_HELM ! HELM for lgT >= this
         real(dp), intent(out) :: logT_all_OPAL ! OPAL/SCVH for lgT <= this
         integer, intent(out) :: ierr ! 0 means AOK.
         type (EoS_General_Info), pointer :: rq
         call eos_ptr(handle,rq,ierr)
         if (ierr /= 0) return
         logT_all_HELM = rq% logT_all_HELM
         logT_all_OPAL = rq% logT_all_OPAL
      end subroutine eos_get_HELM_OPAL_lgTs

      
      ! transition temperature zone for SCVH to HELM at low T
      subroutine eos_set_HELM_SCVH_lgTs(handle, logT_low_all_HELM, logT_low_all_SCVH, ierr)
         use eos_def, only: do_set_HELM_SCVH_lgTs
         integer, intent(in) :: handle
         real(dp), intent(in) :: logT_low_all_HELM ! HELM for lgT <= this
         real(dp), intent(in) :: logT_low_all_SCVH ! SCVH for lgT >= this
         integer, intent(out) :: ierr ! 0 means AOK.
         call do_set_HELM_SCVH_lgTs( &
            handle, logT_low_all_HELM, logT_low_all_SCVH, ierr)
         if (ierr /= 0) return
#ifdef offload
         !dir$ offload target(mic) out(ierr) &
            in(handle, logT_low_all_HELM, logT_low_all_SCVH)
         call do_set_HELM_SCVH_lgTs( &
            handle, logT_low_all_HELM, logT_low_all_SCVH, ierr)
#endif
      end subroutine eos_set_HELM_SCVH_lgTs
      
      
      subroutine eos_get_HELM_SCVH_lgTs(handle, logT_low_all_HELM, logT_low_all_SCVH, ierr)
         use eos_def
         integer, intent(in) :: handle
         real(dp), intent(out) :: logT_low_all_HELM ! HELM for lgT <= this
         real(dp), intent(out) :: logT_low_all_SCVH ! SCVH for lgT >= this
         integer, intent(out) :: ierr ! 0 means AOK.
         type (EoS_General_Info), pointer :: rq
         call eos_ptr(handle,rq,ierr)
         if (ierr /= 0) return
         logT_low_all_HELM = rq% logT_low_all_HELM
         logT_low_all_SCVH = rq% logT_low_all_SCVH
      end subroutine eos_get_HELM_SCVH_lgTs
      
      
      ! the following routine sets the transition region from HELM to PC
      ! PC does some extra work for each species it considers
      ! the mass_fraction_limit_for_PC controls which species are included
      ! Gamma := Plasma Coupling Parameter
      subroutine eos_set_PC_parameters( &
            handle, mass_fraction_limit_for_PC,  &
            logRho1_PC_limit, logRho2_PC_limit, &
            log_Gamma_all_HELM, log_Gamma_all_PC, PC_min_Z, ierr)
         use eos_def, only: do_set_PC_parameters
         integer, intent(in) :: handle
         real(dp), intent(in) :: mass_fraction_limit_for_PC ! skip species if abundance < this
         real(dp), intent(in) :: logRho1_PC_limit ! okay for pure PC for logRho > this
         real(dp), intent(in) :: logRho2_PC_limit ! don't use PC for logRho < this (>= 2.8)
         real(dp), intent(in) :: log_Gamma_all_HELM ! HELM for log_Gamma <= this
         real(dp), intent(in) :: log_Gamma_all_PC ! PC for log_Gamma >= this
         real(dp), intent(in) :: PC_min_Z ! don't use PC for Z < this
         integer, intent(out) :: ierr ! 0 means AOK.
         call do_set_PC_parameters( &
            handle, mass_fraction_limit_for_PC,  &
            logRho1_PC_limit, logRho2_PC_limit, &
            log_Gamma_all_HELM, log_Gamma_all_PC, PC_min_Z, ierr)
         if (ierr /= 0) return
#ifdef offload
         !dir$ offload target(mic) out(ierr) in( &
            handle, mass_fraction_limit_for_PC,  &
            logRho1_PC_limit, logRho2_PC_limit, &
            log_Gamma_all_HELM, log_Gamma_all_PC, PC_min_Z)
         call do_set_PC_parameters( &
            handle, mass_fraction_limit_for_PC,  &
            logRho1_PC_limit, logRho2_PC_limit, &
            log_Gamma_all_HELM, log_Gamma_all_PC, PC_min_Z, ierr)
#endif
      end subroutine eos_set_PC_parameters
      
      
      subroutine eos_get_PC_parameters( &
            handle, mass_fraction_limit_for_PC,  &
            logRho1_PC_limit, logRho2_PC_limit, &
            log_Gamma_all_HELM, log_Gamma_all_PC, PC_min_Z, ierr)
         use eos_def
         integer, intent(in) :: handle
         real(dp), intent(out) :: mass_fraction_limit_for_PC ! skip species if abundance < this
         real(dp), intent(out) :: logRho1_PC_limit ! okay for pure PC for logRho > this
         real(dp), intent(out) :: logRho2_PC_limit ! don't use PC for logRho < this (>= 2.8)
         real(dp), intent(out) :: log_Gamma_all_HELM ! HELM for log_Gamma <= this
         real(dp), intent(out) :: log_Gamma_all_PC ! PC for log_Gamma >= this
         real(dp), intent(out) :: PC_min_Z ! don't use PC for Z < this
         integer, intent(out) :: ierr ! 0 means AOK.
         type (EoS_General_Info), pointer :: rq
         call eos_ptr(handle,rq,ierr)
         if (ierr /= 0) return
         mass_fraction_limit_for_PC = rq% mass_fraction_limit_for_PC
         logRho1_PC_limit = rq% logRho1_PC_limit
         logRho2_PC_limit = rq% logRho2_PC_limit
         log_Gamma_all_HELM = rq% log_Gamma_all_HELM
         log_Gamma_all_PC = rq% log_Gamma_all_PC
         PC_min_Z = rq% PC_min_Z
      end subroutine eos_get_PC_parameters

            
#ifdef offload
      !dir$ options /offload_attribute_target=mic
#endif      
      
      ! for typical use, you won't need the pointer corresponding to a handle,
      ! but just in case, this routine lets you get it.
      subroutine eos_ptr(handle,rq,ierr)
         use eos_def,only:EoS_General_Info,get_eos_ptr
         integer, intent(in) :: handle ! from alloc_eos_handle
         type (EoS_General_Info), pointer :: rq
         integer, intent(out):: ierr
         call get_eos_ptr(handle,rq,ierr)
      end subroutine eos_ptr
            
      
      subroutine eos_result_names(names)      
         use eos_def, only : get_result_names, num_eos_basic_results
         character (len=8) :: names(num_eos_basic_results)         
         call get_result_names(names)      
      end subroutine eos_result_names

      
      ! as a convenience
      
      elemental real(dp) function Radiation_Pressure(T)
         use const_def, only: crad
         real(dp), intent(in) :: T
         Radiation_Pressure = crad*T*T*T*T/3d0
      end function Radiation_Pressure
      
      
      
      ! eos evaluation
      
      ! you can call these routines after you've allocated a handle.
      ! NOTE: the information referenced via the handle is read-only,
      ! so you can do multiple evaluations in parallel using the same handle.
      
      
      
      ! The MESA EOS is based on the OPAL, SCVH, and HELM EOS's.
      ! The OPAL-SCVH values are used for approximately
      !     Z < 0.04, logRho < 3.3, and logT < 7.7
      ! HELM is used for all other cases.
      
      ! The OPAL-SCVH results are interpolated at runtime, 
      ! while the HELM results are computed as functions of Rho and T.

      ! There are multiple interfaces to the MESA EOS corresponding
      ! to different pairs of input arguments.  For each supported pair,
      ! there are tables for direct interpolation of the OPAL-SCVH values.
      ! For input pairs other then (Rho,T), the HELM values are computed
      ! by an iterative root find.  However, since HELM provides partial
      ! derivatives, this search usually takes only 3 or 4 iterations.
      
      ! The supported input pairs are
      
      !     (density, temperature) -- eosDT
      !     (gas pressure, temperature) -- eosPT
      !     (density, energy) -- eosDE
      
      
      

      subroutine eosDT_get( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               Rho, log10Rho, T, log10T,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_const_TRho, d_dzbar_const_TRho, ierr)

         use eos_def
         use eosDT_eval, only: Get_eosDT_Results

         ! INPUT
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: Rho, log10Rho ! the density
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
            ! "arg_not_provided" is defined in mesa const_def
            
         real(dp), intent(in) :: T, log10T ! the temperature
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
                     
         ! OUTPUT
         
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         ! partial derivatives of the basic results wrt lnd and lnT
         
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results) 
         ! d_dlnRho_const_T(i) = d(res(i))/dlnd|T,X where X = composition
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results) 
         ! d_dlnT(i) = d(res(i))/dlnT|Rho,X where X = composition

         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dabar(i) = d(res(i))/dabar|TRho,zbar
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dzbar(i) = d(res(i))/dzbar|TRho,abar
         
         integer, intent(out) :: ierr ! 0 means AOK.
         
         logical, parameter :: basic_flag = .false.
         type (EoS_General_Info), pointer :: rq
         call get_eos_ptr(handle,rq,ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for eos_get -- did you call alloc_eos_handle?'
            return
         end if
         call Get_eosDT_Results( &
               rq, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               Rho, log10Rho, T, log10T,  &
               basic_flag, res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_const_TRho, d_dzbar_const_TRho, ierr)
         
      end subroutine eosDT_get


      subroutine eosDT_HELMEOS_get( &
               handle, Z, X, abar, zbar, &
               species, chem_id, net_iso, xa, &
               Rho, log10Rho, T, log10T,  &
               include_radiation, always_skip_elec_pos, &
               res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_const_TRho, d_dzbar_const_TRho, helm_res, ierr)

         use eos_def
         use eosDT_eval, only: get_eosDT_HELMEOS_Results

         ! INPUT
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: Rho, log10Rho ! the density
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
            ! "arg_not_provided" is defined in mesa const_def
            
         real(dp), intent(in) :: T, log10T ! the temperature
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
         
         logical, intent(in) :: include_radiation, always_skip_elec_pos
         
         ! OUTPUT
         
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         ! partial derivatives of the basic results wrt lnd and lnT
         
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results) 
         ! d_dlnRho_const_T(i) = d(res(i))/dlnd|T,X where X = composition
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results) 
         ! d_dlnT(i) = d(res(i))/dlnT|Rho,X where X = composition

         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dabar(i) = d(res(i))/dabar|TRho,zbar
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dzbar(i) = d(res(i))/dzbar|TRho,abar

         real(dp), intent(out) :: helm_res(:) ! (num_helm_results)
         
         integer, intent(out) :: ierr ! 0 means AOK.
         
         logical, parameter :: basic_flag = .false.
         type (EoS_General_Info), pointer :: rq
         
         call get_eos_ptr(handle,rq,ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for eos -- did you call alloc_eos_handle?'
            return
         end if
         call get_eosDT_HELMEOS_Results( &
            rq, Z, X, abar, zbar, Rho, log10Rho, T, log10T, &
            include_radiation, always_skip_elec_pos, basic_flag,  &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_const_TRho, d_dzbar_const_TRho, helm_res, ierr)      
               
      end subroutine eosDT_HELMEOS_get


      subroutine eosDT_ideal_gas_get( &
               handle, Z, X, abar, zbar, &
               species, chem_id, net_iso, xa, &
               Rho, log10Rho, T, log10T,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_const_TRho, d_dzbar_const_TRho, ierr)

         use eos_def
         use eosDT_eval, only: get_eosDT_HELMEOS_Results

         ! INPUT
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: Rho, log10Rho ! the density
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
            ! "arg_not_provided" is defined in mesa const_def
            
         real(dp), intent(in) :: T, log10T ! the temperature
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
         
         ! OUTPUT
         
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         ! partial derivatives of the basic results wrt lnd and lnT
         
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results) 
         ! d_dlnRho_const_T(i) = d(res(i))/dlnd|T,X where X = composition
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results) 
         ! d_dlnT(i) = d(res(i))/dlnT|Rho,X where X = composition

         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dabar(i) = d(res(i))/dabar|TRho,zbar
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dzbar(i) = d(res(i))/dzbar|TRho,abar
         
         integer, intent(out) :: ierr ! 0 means AOK.
         
         logical, parameter :: basic_flag = .false., &
            include_radiation = .false., always_skip_elec_pos = .true.
         real(dp) :: helm_res(num_helm_results)
         type (EoS_General_Info), pointer :: rq
         
         call get_eos_ptr(handle,rq,ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for eos -- did you call alloc_eos_handle?'
            return
         end if
         call get_eosDT_HELMEOS_Results( &
            rq, Z, X, abar, zbar, Rho, log10Rho, T, log10T, &
            include_radiation, always_skip_elec_pos, basic_flag,  &
            res, d_dlnRho_const_T, d_dlnT_const_Rho, &
            d_dabar_const_TRho, d_dzbar_const_TRho, helm_res, ierr)      
               
      end subroutine eosDT_ideal_gas_get
      
      
      ! the following routine uses gas pressure and temperature as input variables
      subroutine eosPT_get( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               Pgas, log10Pgas, T, log10T,  &
               Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_const_TRho, d_dzbar_const_TRho, ierr)

         use eos_def
         use eosPT_eval, only: Get_eosPT_Results

         ! INPUT
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: Pgas, log10Pgas ! the gas pressure
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
            ! "arg_not_provided" is defined in mesa const_def
            
         real(dp), intent(in) :: T, log10T ! the temperature
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
                     
         ! OUTPUT
         
         real(dp), intent(out) :: Rho, log10Rho ! density
         real(dp), intent(out) :: dlnRho_dlnPgas_const_T
         real(dp), intent(out) :: dlnRho_dlnT_const_Pgas
         
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         
         ! partial derivatives of the basic results
         
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results) 
         ! d_dlnRho_const_T(i) = d(res(i))/dlnd|T,X where X = composition
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results) 
         ! d_dlnT(i) = d(res(i))/dlnT|Rho,X where X = composition

         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dabar(i) = d(res(i))/dabar|TRho,zbar
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dzbar(i) = d(res(i))/dzbar|TRho,abar
         
         integer, intent(out) :: ierr ! 0 means AOK.
         
         type (EoS_General_Info), pointer :: rq
         call get_eos_ptr(handle,rq,ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for eos -- did you call alloc_eos_handle?'
            return
         end if
         call Get_eosPT_Results( &
                  rq, Z, X, abar, zbar,  &
                  species, chem_id, net_iso, xa, &
                  Pgas, log10Pgas, T, log10T,  &
                  Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,  &
                  res, d_dlnRho_const_T, d_dlnT_const_Rho, &
                  d_dabar_const_TRho, d_dzbar_const_TRho, ierr)
         
      end subroutine eosPT_get
      
      
      ! the following routine uses density and internal energy as input variables
      subroutine eosDE_get( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               energy, log10E, rho, log10Rho, log10T_guess, &
               T, log10T, res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_const_TRho, d_dzbar_const_TRho, &
               dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
               dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
               ierr)

         use eos_def
         use eosDE_eval, only: Get_eosDE_Results

         ! INPUT
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: energy, log10E ! the internal energy
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
            ! "arg_not_provided" is defined in mesa const_def
            
         real(dp), intent(in) :: Rho, log10Rho ! the density
            ! provide both if you have them.  else pass one and set the other to arg_not_provided
            
         real(dp), intent(in) :: log10T_guess ! guess for logT to use if off table
                     
         ! OUTPUT
         
         real(dp), intent(out) :: T, log10T
         
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         
         ! partial derivatives of the basic results
         
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results) 
         ! d_dlnRho_const_T(i) = d(res(i))/dlnd|T,X where X = composition
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results) 
         ! d_dlnT(i) = d(res(i))/dlnT|Rho,X where X = composition

         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dabar(i) = d(res(i))/dabar|TRho,zbar
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 
         ! d_dzbar(i) = d(res(i))/dzbar|TRho,abar

         real(dp), intent(out) :: & 
            ! when possible, these come directly from table interpolation results
            dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
            dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E
         
         integer, intent(out) :: ierr ! 0 means AOK.

         ! NOTE: when converting partials for f = f(lnd,lnT(lnd,lnE)), 
         ! df_dlnE_const_Rho = df_dlnT_const_Rho*dlnT_dlnE_const_Rho
         !     dlnT_dlnE_const_Rho = E/(Cv*T)
         ! df_dlnd_const_E = df_dlnd_const_T + df_dlnT_const_Rho*dlnT_dlnd_const_E
         !     dlnT_dlnd_const_E = -Rho*dE_dRho/(Cv*T)
         
         type (EoS_General_Info), pointer :: rq
         call get_eos_ptr(handle,rq,ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for eos -- did you call alloc_eos_handle?'
            return
         end if
         call Get_eosDE_Results( &
               rq, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               energy, log10E, rho, log10Rho, log10T_guess, &
               T, log10T, res, d_dlnRho_const_T, d_dlnT_const_Rho, &
               d_dabar_const_TRho, d_dzbar_const_TRho, &
               dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
               dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
               ierr)
         
      end subroutine eosDE_get



      ! misc
      

      real(dp) function eos_theta_e(eta, d_theta_e_deta)
         ! theta_e is used in the Graboske et al screening method.
         ! for non-degenerate electrons, theta_e goes to 1.
         ! for significantly degenerate electrons, it goes to 0.
         use eosDT_eval, only:eval_theta_e
         real(dp), intent(in) :: eta ! degeneracy parameter
         real(dp), intent(out) :: d_theta_e_deta
         eos_theta_e = eval_theta_e(eta, d_theta_e_deta)
      end function eos_theta_e


      real(dp) function Plasma_Coupling_Parameter(T, den, abar, zbar)
         ! Plasma Coupling Parameter is also called Gamma (e.g., in eos_set_PC_parameters)
         use const_def
         use crlibm_lib
         ! just as a convenience, here's the routine to calculate this in the usual manner.
         real(dp), intent(in) :: T, den, abar, zbar
         real(dp) :: xni         
         xni = avo * den / abar
         Plasma_Coupling_Parameter =  &
               (zbar*qe)*(zbar*qe) * pow_cr((4.0d0/3.0d0)*pi*xni, one_third) / (kerg*T)         
      end function Plasma_Coupling_Parameter


      subroutine eos_fermi_dirac_integral(dk, eta, theta, fd, fdeta, fdtheta)
         !..from Frank Timmes' site, http://www.cococubed.com/code_pages/fermi_dirac.shtml
         !..this routine computes the fermi-dirac integrals of 
         !..index dk, with degeneracy parameter eta and relativity parameter theta.
         !..input is dk the real(dp) index of the fermi-dirac function,
         !..eta the degeneracy parameter, and theta the relativity parameter.
         !..theta = (k * T)/(mass_electron * c^2), k = Boltzmann const.
         !..the output is fd is computed by applying three 10-point 
         !..gauss-legendre and one 10-point gauss-laguerre rules over
         !..four appropriate subintervals. the derivative with respect to eta is
         !..output in fdeta, and the derivative with respct to theta is in fdtheta.
         !..within each subinterval the fd kernel.
         !..
         !..this routine delivers 14 significant figure accuracy
         !..
         !..reference: j.m. aparicio, apjs 117, 632 1998
         real(dp), intent(in) :: dk
         real(dp), intent(in) :: eta
         real(dp), intent(in) :: theta
         real(dp), intent(out) :: fd
         real(dp), intent(out) :: fdeta
         real(dp), intent(out) :: fdtheta
         call dfermi(dk, eta, theta, fd, fdeta, fdtheta)
      end subroutine eos_fermi_dirac_integral
      

      subroutine eos_get_helm_results( &
               X, abar, zbar, Rho, log10Rho, T, log10T, &
               include_radiation, always_skip_elec_pos, res, ierr)
         ! direct call to the helm eos.  
         ! returns much more info than the standard

         use eos_def
         use eosDT_eval, only: Get_HELM_Results

         ! INPUT

         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         real(dp), intent(in) :: Rho, log10Rho ! the density
            ! provide both if you have them.  
            ! else pass one and set the other to arg_not_provided
            
         real(dp), intent(in) :: T, log10T ! the temperature
            ! provide both if you have them.  
            ! else pass one and set the other to arg_not_provided

         logical, intent(in) :: include_radiation, always_skip_elec_pos
         
         ! OUTPUT
         
         real(dp), intent(out) :: res(:) ! (num_helm_results)
            ! array to hold the results
         integer, intent(out) :: ierr ! 0 means AOK.     

         call Get_HELM_Results( &
            X, abar, zbar, Rho, log10Rho, T, log10T, &
            include_radiation, always_skip_elec_pos, res, ierr)
              
      end subroutine eos_get_helm_results
      
      
      subroutine eos_convert_helm_results( &
            helm_res, Z, X, abar, zbar, Rho, T, basic_flag, res,  &
            d_dlnRho_const_T, d_dlnT_const_Rho,  &
            d_dabar_const_TRho, d_dzbar_const_TRho, ierr)      
         use eos_def
         use eosDT_eval, only: do_convert_helm_results
         real(dp), intent(in) :: helm_res(:) ! (num_helm_results)
         real(dp), intent(in) :: Z, X, abar, zbar, Rho, T
         logical, intent(in) :: basic_flag ! if true, then only want basic results
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 
         integer, intent(out) :: ierr      
         call do_convert_helm_results( &
                  helm_res, Z, X, abar, zbar, Rho, T, basic_flag,  &
                  res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
                  d_dabar_const_TRho, d_dzbar_const_TRho, ierr)      
      end subroutine eos_convert_helm_results

      
      
      subroutine eos_eval_PC( & ! Potekhin-Chabrier eos
            handle, Z, X, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            Rho, logRho, T, logT,  &
            res, d_dlnRho_c_T, d_dlnT_c_Rho, ierr)
         use eos_def
         use eosDT_eval, only: Get_PC_Results
         integer, intent(in) :: handle
         real(dp), intent(in) :: Z, X, abar, zbar
         integer, intent(in) :: species
         integer, pointer :: chem_id(:), net_iso(:)
         real(dp), intent(in) :: xa(:)
         real(dp), intent(in) :: Rho, logRho, T, logT
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results)
         integer, intent(out) :: ierr
         type (EoS_General_Info), pointer :: rq
         call get_eos_ptr(handle,rq,ierr)
         if (ierr /= 0) then
            write(*,*) 'invalid handle for eos_get -- did you call alloc_eos_handle?'
            return
         end if
         call Get_PC_Results( &
            rq, Z, X, abar, zbar,  &
            species, chem_id, net_iso, xa, &
            Rho, logRho, T, logT, .false.,  &
            res, d_dlnRho_c_T, d_dlnT_c_Rho, ierr)
      end subroutine eos_eval_PC
      
      
      ! eosDT search routines.  these use num_lib safe_root to find T or Rho.
      
      subroutine eosDT_get_T( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logRho, which_other, other_value, &
               logT_tol, other_tol, max_iter, logT_guess,  &
               logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
               logT_result, res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, eos_calls, ierr)
     
         ! finds log10 T given values for density and 'other', and initial guess for temperature.
         ! does up to max_iter attempts until logT changes by less than tol.
         
         ! 'other' can be any of the basic result variables for the eos
         ! specify 'which_other' by means of the definitions in eos_def (e.g., i_lnE)

         use eos_def
         use eosDT_eval, only : get_T
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logRho ! log10 of density
         integer, intent(in) :: which_other ! from eos_def.  e.g., i_lnE
         real(dp), intent(in) :: other_value ! desired value for the other variable
         real(dp), intent(in) :: other_tol
         
         real(dp), intent(in) :: logT_tol
         integer, intent(in) :: max_iter ! max number of iterations        

         real(dp), intent(in) :: logT_guess ! log10 of temperature
         real(dp), intent(in) :: logT_bnd1, logT_bnd2 ! bounds for logT
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: other_at_bnd1, other_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logT_result ! log10 of temperature
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 
         
         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         call get_T( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logRho, which_other, other_value, &
               logT_tol, other_tol, max_iter, logT_guess,  &
               logT_bnd1, logT_bnd2,  other_at_bnd1, other_at_bnd2, &
               logT_result, res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, eos_calls, ierr)

      end subroutine eosDT_get_T
      

      subroutine eosDT_get_Rho( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logT, which_other, other_value, &
               logRho_tol, other_tol, max_iter, logRho_guess,  &
               logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, &
               logRho_result, res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, eos_calls, ierr)
     
         ! finds log10 Rho given values for temperature and 'other', and initial guess for density.
         ! does up to max_iter attempts until logRho changes by less than tol.
         
         ! 'other' can be any of the basic result variables for the eos
         ! specify 'which_other' by means of the definitions in eos_def (e.g., i_lnE)

         use eos_def
         use eosDT_eval, only : get_Rho
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logT ! log10 of temperature

         integer, intent(in) :: which_other ! from eos_def.  e.g., i_lnE
         real(dp), intent(in) :: other_value ! desired value for the other variable
         real(dp), intent(in) :: other_tol
         
         real(dp), intent(in) :: logRho_tol

         integer, intent(in) :: max_iter ! max number of Newton iterations        

         real(dp), intent(in) :: logRho_guess ! log10 of density
         real(dp), intent(in) :: logRho_bnd1, logRho_bnd2 ! bounds for logRho
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: other_at_bnd1, other_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logRho_result ! log10 of density

         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 

         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         call get_Rho( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logT, which_other, other_value, &
               logRho_tol, other_tol, max_iter, logRho_guess,  &
               logRho_bnd1, logRho_bnd2, other_at_bnd1, other_at_bnd2, &
               logRho_result, res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, eos_calls, ierr)
      
      end subroutine eosDT_get_Rho

      
      ! eosPT search routines.  these use num_lib safe_root to find T or Pgas.
      
      subroutine eosPT_get_T( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logPgas, which_other, other_value, &
               logT_tol, other_tol, max_iter, logT_guess,  &
               logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
               logT_result, Rho, log10Rho,  &
               dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, &
               eos_calls, ierr)
     
         ! finds log10 T given values for gas pressure and 'other',
         ! and initial guess for temperature.
         ! does up to max_iter attempts until logT changes by less than tol.
         
         ! 'other' can be any of the basic result variables for the eos
         ! specify 'which_other' by means of the definitions in eos_def (e.g., i_lnE)

         use eos_def
         use eosPT_eval, only : get_T
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logPgas ! log10 of gas pressure
         integer, intent(in) :: which_other ! from eos_def.  e.g., i_lnE
         real(dp), intent(in) :: other_value ! desired value for the other variable
         real(dp), intent(in) :: other_tol
         
         real(dp), intent(in) :: logT_tol
         integer, intent(in) :: max_iter ! max number of iterations        

         real(dp), intent(in) :: logT_guess ! log10 of temperature
         real(dp), intent(in) :: logT_bnd1, logT_bnd2 ! bounds for logT
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: other_at_bnd1, other_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logT_result ! log10 of temperature
         real(dp), intent(out) :: Rho, log10Rho ! density
         real(dp), intent(out) :: dlnRho_dlnPgas_const_T
         real(dp), intent(out) :: dlnRho_dlnT_const_Pgas

         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 
         
         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         call get_T( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logPgas, which_other, other_value, &
               logT_tol, other_tol, max_iter, logT_guess,  &
               logT_bnd1, logT_bnd2,  other_at_bnd1, other_at_bnd2, &
               logT_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, &
               eos_calls, ierr)

      end subroutine eosPT_get_T
      

      subroutine eosPT_get_Pgas( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logT, which_other, other_value, &
               logPgas_tol, other_tol, max_iter, logPgas_guess,  &
               logPgas_bnd1, logPgas_bnd2, other_at_bnd1, other_at_bnd2, &
               logPgas_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, &
               eos_calls, ierr)
     
         ! finds log10 Pgas given values for temperature and 'other', and initial guess for gas pressure.
         ! does up to max_iter attempts until logPgas changes by less than tol.
         
         ! 'other' can be any of the basic result variables for the eos
         ! specify 'which_other' by means of the definitions in eos_def (e.g., i_lnE)

         use eos_def
         use eosPT_eval, only : get_Pgas
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logT ! log10 of temperature

         integer, intent(in) :: which_other ! from eos_def.  e.g., i_lnE
         real(dp), intent(in) :: other_value ! desired value for the other variable
         real(dp), intent(in) :: other_tol
         
         real(dp), intent(in) :: logPgas_tol

         integer, intent(in) :: max_iter ! max number of Newton iterations        

         real(dp), intent(in) :: logPgas_guess ! log10 of gas pressure
         real(dp), intent(in) :: logPgas_bnd1, logPgas_bnd2 ! bounds for logPgas
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: other_at_bnd1, other_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logPgas_result ! log10 of gas pressure
         real(dp), intent(out) :: Rho, log10Rho ! density
         real(dp), intent(out) :: dlnRho_dlnPgas_const_T
         real(dp), intent(out) :: dlnRho_dlnT_const_Pgas

         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 

         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         call get_Pgas( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logT, which_other, other_value, &
               logPgas_tol, other_tol, max_iter, logPgas_guess,  &
               logPgas_bnd1, logPgas_bnd2, other_at_bnd1, other_at_bnd2, &
               logPgas_result, Rho, log10Rho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, &
               eos_calls, ierr)
      
      end subroutine eosPT_get_Pgas
      

      subroutine eosPT_get_Pgas_for_Rho( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logT, logRho_want, &
               logPgas_tol, logRho_tol, max_iter, logPgas_guess,  &
               logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2, &
               logPgas_result, Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, &
               eos_calls, ierr)
     
         ! finds log10 Pgas given values for temperature and density, and initial guess for gas pressure.
         ! does up to max_iter attempts until logPgas changes by less than tol.
         
         use eos_def
         use eosPT_eval, only : get_Pgas_for_Rho
         
         integer, intent(in) :: handle

         real(dp), intent(in) :: Z ! the metals mass fraction
         real(dp), intent(in) :: X ! the hydrogen mass fraction
            
         real(dp), intent(in) :: abar
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: logT ! log10 of temperature

         real(dp), intent(in) :: logRho_want ! log10 of desired density
         real(dp), intent(in) :: logRho_tol
         
         real(dp), intent(in) :: logPgas_tol

         integer, intent(in) :: max_iter ! max number of Newton iterations        

         real(dp), intent(in) :: logPgas_guess ! log10 of gas pressure
         real(dp), intent(in) :: logPgas_bnd1, logPgas_bnd2 ! bounds for logPgas
            ! if don't know bounds, just set to arg_not_provided (defined in const_def)
         real(dp), intent(in) :: logRho_at_bnd1, logRho_at_bnd2 ! values at bounds
            ! if don't know these values, just set to arg_not_provided (defined in const_def)

         real(dp), intent(out) :: logPgas_result ! log10 of gas pressure
         real(dp), intent(out) :: Rho, logRho ! density corresponding to logPgas_result
         real(dp), intent(out) :: dlnRho_dlnPgas_const_T
         real(dp), intent(out) :: dlnRho_dlnT_const_Pgas

         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_const_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_const_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_const_TRho(:) ! (num_eos_basic_results) 
         real(dp), intent(out) :: d_dzbar_const_TRho(:) ! (num_eos_basic_results) 

         integer, intent(out) :: eos_calls
         integer, intent(out) :: ierr ! 0 means AOK.
         
         call get_Pgas_for_Rho( &
               handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logT, logRho_want, &
               logPgas_tol, logRho_tol, max_iter, logPgas_guess,  &
               logPgas_bnd1, logPgas_bnd2, logRho_at_bnd1, logRho_at_bnd2, &
               logPgas_result, Rho, logRho, dlnRho_dlnPgas_const_T, dlnRho_dlnT_const_Pgas,  &
               res, d_dlnRho_const_T, d_dlnT_const_Rho,  &
               d_dabar_const_TRho, d_dzbar_const_TRho, &
               eos_calls, ierr)
      
      end subroutine eosPT_get_Pgas_for_Rho

#ifdef offload
      !dir$ end options
#endif

      end module eos_lib
