!!
!! This version uses ln gas pressure as the independent variable
!!
!  implicit none
!  character (len=8) :: phase
!  integer (kind=4) :: icount,ipgasl,it,n
!  integer (kind=4), parameter :: ncomp=14
!  real (kind=8) :: coul,cf,cp,ct,dq,fl,gamma,gq,gqq,grada,nh,nh2,nhplus,p,pf,pgas,pl,plam,psi,pt,q,rf,rho,rl,rt, &
!    s,sf,st,t,th,tl,u,uf,ut,cv,vt,pgasl,pgasl10
!  real (kind=8), dimension (ncomp) :: x,gqx,px,rx,sx,ux,pip,pipf,pipt,fcs,fcsf,fcst,fneut,fneutf,fneutt
!  real (kind=8), dimension (ncomp,ncomp) :: pipx,fcsx,fneutx
!  open (1,file='p_eos.txt')
!  open (2,file='p_eos_debug.txt')
!!
!! for test purposes use equal mass fractions for all species
!!
!  x=1.0d0/14.0d0
!  tl=15.00d0
!  fl=-0.1d0
!  q=-0.1d0
!  icount=0
!  do ipgasl=165,190
!    pgasl10=0.1d0*float(ipgasl)
!    pgasl=pgasl10*2.302585d0
!    call statef (pgasl,fl,tl,q,x, &
!      pl,rl,u,p,rho,t,gamma,grada,cp,cv,vt,th,pgas,psi,s,gq,pf,pt,rf,rt,uf,ut,sf,st,px,rx,sx,ux, &
!      coul,plam,gqq,pip,pipf,pipt,pipx,fcs,fcsf,fcst,fcsx,fneut,fneutf,fneutt,fneutx,phase)
!!
!! tests for thermodynamic consistency
!!
!    cf=abs(uf-t*sf-p*rf/rho)/(abs(uf)+abs(t*sf)+abs(p*rf/rho))
!    ct=abs(ut-t*st-p*rt/rho)/(abs(ut)+abs(t*st)+abs(p*rt/rho))
!    write (1,fmt='(1x,1p,19e13.4)') q,fl,log10(t),log10(rho),log10(p),pipx(1,:)
!    if (icount.eq.0) then
!      write (2,*) q
!      icount=1
!    endif
!  enddo
!  stop
!  end
!
! Equation of state routine. Includes fluid and solid phases with phase transition smoothed by interpolation
! with respect to Gibbs free energy
!
! input arguments are:
!
!   pgasl = ln pgas
!
!   guess for fl = ln f (see EFF = Eggleton, Flannery and Faulkner 1973, AA 23, 325 for definition)
!
!   tl = ln t where t is temperature in K
!
!   xin is a vector array of mass fractions. ncomp = 14 isotopes are used. These are:
!      H1, H2, He3, He4. Li7, Be7, C12, C13, N14, O16, Ne20, Mg24, Si28, Fe56
!
!   guess for q, an additional independent variable used in the treatment of pressure ionization
!   It is related to the hard sphere volume filling factor eta by
!      q = ln (eta/(1-eta))
!
!   for given pgasl, tl, Newton_Raphson iteration is used to find fl and q
!
!   Output quantities are:
!
!     pl = ln p (p is pressure in units of dyne/cm^2)
!     rl = ln rho (rho is density in units of g/cm^3)
!     u is the internal energy per unit mass in units of erg/g
!        Note that energies of bound states are taken relative to the continuum. Hence U can be negative.
!     p
!     rho 
!     t
!     gamma is Gamma_1 = partial derivative of pressure with respect to density at constant entropy
!     grada is the adiabatic gradient = partial derivative of temperatire with respect to pressure at constant entropy
!     cp is the specific heat at constant pressure
!     cv is the specific heat at constant volume  
!     vt is a thermal expansion coefficient. Specifically it is the derivative of ln V with respect to ln T at constant pressure. Here V = 1/rho
!     th is the f'/f factor that appears in Salpeter's theory for weak screening of reaction rates to take into account electron degeneracy,
!         except that relativistic effects are included
!     pgas is the gas pressure (total pressure - radiation pressure)
!     psi is the degeneracy parameter (see EFF)
!     s is the entropy in cgs units
!     gq as descibed above
!
!     pf is the partial derivative of ln p with respect to ln f 
!     pt is the partial derivative of ln p with respect to ln t
!     rf is the partial derivative of ln rho with respect to ln f
!     rt is the partial derivative of ln rho with respect to ln t
!     uf is the partial derivative of u with respect to ln f
!     ut is the partial derivative of u with respect to ln t
!     sf is the partial derivative of s with respect to ln f
!     st is the partial derivative of s with respect to ln t
!     px, rx, sx, and ux are arrays of length ncomp that contain the partial derivatives
!      of ln p, ln rho, s and u with respect to the mass fractions.
!      NOTE: I do not use the derivatives with respect to mass fractions. I have done some testing by comparing to numerical derivatives
!      but recommend further testing.
!
!     pip is an array containing the partial pressures
!     fcs is an array containing the mean charge for each species
!     fneut is an array containing the fraction of particles of each species that are neutral
!
  subroutine statef (pgasl,fl,tl,q,x, &
    pl,rl,u,p,rho,t,gamma,grada,cp,cv,vt,th,pgas,psi,s,gq,pf,pt,rf,rt,uf,ut,sf,st,px,rx,sx,ux, &
    coul,plam,gqq,pip,pipf,pipt,pipx,fcs,fcsf,fcst,fcsx,fneut,fneutf,fneutt,fneutx,phase)
  use crlibm_lib, only: log_cr, exp_cr
  implicit none
  character (len=8) :: phase
  integer (kind=4) :: it,n
  integer (kind=4), parameter :: ncomp=14
  real (kind=8) :: fl,tl,q,pl,rl,u,p,rho,t,gamma,grada,cp,th,pgas,psi,s,gq,pf,pt,rf,rt,uf,ut,sf,st,coul,plam,gqq, &
    coulsol,plamsol,cv,vt,pgasl,pgasl_out,pgasf,pgasq,dfl,dq,delta_fl,delta_q
  real (kind=8) :: plflu,rlflu,uflu,pflu,rhoflu,tflu,gammaflu,gradaflu,cpflu,thflu,pgasflu,psiflu,sflu,gqflu, &
    pfflu,ptflu,rfflu,rtflu,ufflu,utflu,sfflu,stflu,coulflu,plamflu,gqqflu,gibbsflu,pf_qt,pq_ft,gqf_qt,gqq_ft
  real (kind=8) :: flsol,qsol,dflsol,gqqsol,plsol,pqsol,gqsol,pfsol,gqfsol,dqsol,gibbssol,eps,ceps  
  real (kind=8), dimension(ncomp) :: x,px,pxflu,rx,rxflu,sx,sxflu,ux,uxflu,pipflu,pipfflu,piptflu,fcsflu,fcsfflu,fcstflu, &
    fneutflu,fneutfflu,fneuttflu,pip,pipf,pipt,fcs,fcsf,fcst,fneut,fneutf,fneutt
  real (kind=8), dimension(ncomp,ncomp) :: pipxflu,fcsxflu,fneutxflu,pipx,fcsx,fneutx
!
! iterate to find fl and q for fluid phase
!
  phase='fluid'
  do it=1,10
    call statefx_fluid (fl,tl,q,x, &
      pl,rl,u,p,rho,t,gamma,grada,cp,cv,vt,th,pgas,psi,s,gq,gqq,pf,pt,rf,rt,uf,ut,sf,st,px,rx,sx,ux,pip,pipf,pipt,pipx, &
      fcs,fcsf,fcst,fcsx,fneut,fneutf,fneutt,fneutx,coul,plam,pf_qt,pq_ft,gqf_qt,gqq_ft)
      pgasl_out=log_cr(pgas)
      pgasf=p*pf_qt/pgas
      pgasq=p*pq_ft/pgas
      dfl=(pgasl*gqq_ft-pgasl_out*gqq_ft+pgasq*gq)/(pgasf*gqq_ft-pgasq*gqf_qt)
      dq=-(gq+gqf_qt*dfl)/gqq_ft
      fl=fl+dfl
      q =q +dq
      delta_fl=abs(dfl)/(1.0d0+abs(fl))
      delta_q =abs(dq )/(1.0d0+abs(q ))
!      write (2,*) dfl,dq
      if (delta_q.le.1.0d-08.and.delta_fl.le.1.0d-08) go to 100
  enddo
!
! iteration did not converge. Therefore stop
!
  write (2,fmt='(1x,"iteration to find fl and q did not converge")')
  stop
100 continue
  gibbsflu=u-t*s+p/rho
!
! note that the phase transition does not occur at constant Coulomb Gamma
! However there is a minimum Coulomb Gamma at which phase transition can occur
! (Gamma_m is about 178)
! Can safely assume fluid phase for Gamma less than 170
!
  if (coul.le.170.0d0) return
!
! Otherwise determine phase by comparing Gibbs free energy
!
! first store eos quantities for fluid state
!
  plflu=pl
  rlflu=rl
  uflu=u
  pflu=p
  rhoflu=rho
  tflu=t
  gammaflu=gamma
  gradaflu=grada
  cpflu=cp
  thflu=th
  pgasflu=pgas
  psiflu=psi
  sflu=s
  gqflu=gq
  pfflu=pf
  ptflu=pt
  rfflu=rf
  rtflu=rt
  ufflu=uf
  utflu=ut
  sfflu=sf
  stflu=st
  pxflu=px
  rxflu=rx
  sxflu=sx
  uxflu=ux
  pipflu =pip
  pipfflu=pipf
  piptflu=pipt
  fcsflu =fcs 
  fcsfflu=fcsf
  fcstflu=fcst
  fneutflu =fneut 
  fneutfflu=fneutf
  fneuttflu=fneutt
  pipxflu=pipx
  fcsxflu=fcsx
  fneutxflu=fneutx
  coulflu=coul
  plamflu=plam
!
! iterate to find fl and q for the solid phase at the same temperature and pressure
!
!
! 3 iterations appear sufficient to reach machine precision
!
  do it=1,3
    call statefx_solid (fl,tl,q,x, &
      pl,rl,u,p,rho,t,gamma,grada,cp,cv,vt,th,pgas,psi,s,gq,gqq,pf,pt,rf,rt,uf,ut,sf,st,px,rx,sx,ux,pip,pipf,pipt, &
      pipx,fcs,fcsf,fcst,fcsx,fneut,fneutf,fneutt,fneutx,coul,plam,pf_qt,pq_ft,gqf_qt,gqq_ft)
      pgasl_out=log_cr(pgas)
      pgasf=p*pf_qt/pgas
      pgasq=p*pq_ft/pgas
      dfl=(pgasl*gqq_ft-pgasl_out*gqq_ft+pgasq*gq)/(pgasf*gqq_ft-pgasq*gqf_qt)
      dq=-(gq+gqf_qt*dfl)/gqq_ft
      fl=fl+dfl
      q =q +dq
  enddo
  call statefx_solid (fl,tl,q,x, &
    pl,rl,u,p,rho,t,gamma,grada,cp,cv,vt,th,pgas,psi,s,gq,gqq,pf,pt,rf,rt,uf,ut,sf,st,px,rx,sx,ux,pip,pipf,pipt,pipx, &
    fcs,fcsf,fcst,fcsx,fneut,fneutf,fneutt,fneutx,coul,plam,pf_qt,pq_ft,gqf_qt,gqq_ft)
  gibbssol=u-t*s+p/rho
!
! value of eps determines phase
!
! eps less than zero gives fluid phase
! eps greater than zero gives solid phase
! However phase transition is smoothed by interpolation for 0 < eps < 1
!
  eps=5.0d+04*(gibbsflu-gibbssol)/gibbsflu
  phase='solid'
  if (eps.le.0.0d0) then
    phase='fluid'
!
! restore fluid values
!
    pl=plflu
    rl=rlflu
    u=uflu
    p=pflu
    rho=rhoflu
    gamma=gammaflu
    grada=gradaflu
    cp=cpflu
    th=thflu
    pgas=pgasflu
    psi=psiflu
    s=sflu
    gq=gqflu
    pf=pfflu
    pt=ptflu
    rf=rfflu
    rt=rtflu
    uf=ufflu
    ut=utflu
    sf=sfflu
    st=stflu
    px=pxflu
    rx=rxflu
    sx=sxflu
    ux=uxflu
    pip=pipflu
    pipf=pipfflu
    pipt=piptflu
    fcs =fcsflu 
    fcsf=fcsfflu
    fcst=fcstflu
    fneut =fneutflu 
    fneutf=fneutfflu
    fneutt=fneuttflu
    pipx=pipxflu
    fcsx=fcsxflu
    fneutx=fneutxflu
    coul=coulflu
    plam=plamflu
  else
    if (eps.lt.1.0d0) then
      phase='interpol'
!
! interpolate
!
      eps=eps**2*(3.0d0-2.0d0*eps)
      ceps=1.0d0-eps
      pl   =ceps*plflu   +eps*pl
      rl   =ceps*rlflu   +eps*rl
      u    =ceps*uflu    +eps*u
      p    =exp_cr(pl)
      rho  =exp_cr(rl)
      gamma=ceps*gammaflu+eps*gamma
      grada=ceps*gradaflu+eps*grada
      cp   =ceps*cpflu   +eps*cp
      th   =ceps*thflu   +eps*th
      pgas =ceps*pgasflu +eps*pgas
      psi  =ceps*psiflu  +eps*psi
      s    =ceps*sflu    +eps*s
      pf   =ceps*pfflu   +eps*pf
      pt   =ceps*ptflu   +eps*pt
      rf   =ceps*rfflu   +eps*rf
      rt   =ceps*rtflu   +eps*rt
      uf   =ceps*ufflu   +eps*uf
      ut   =ceps*utflu   +eps*ut
      sf   =ceps*sfflu   +eps*sf
      st   =ceps*stflu   +eps*st
      px=ceps*pxflu+eps*px
      rx=ceps*rxflu+eps*rx
      sx=ceps*sxflu+eps*sx
      ux=ceps*uxflu+eps*ux
      pip=ceps*pipflu+eps*pip
      pipf=ceps*pipfflu+eps*pipf
      pipt=ceps*piptflu+eps*pipt
      fcs =ceps*fcsflu +eps*fcs 
      fcsf=ceps*fcsfflu+eps*fcsf
      fcst=ceps*fcstflu+eps*fcst
      fneut =ceps*fneutflu +eps*fneut 
      fneutf=ceps*fneutfflu+eps*fneutf
      fneutt=ceps*fneuttflu+eps*fneutt
      pipx=ceps*pipxflu+eps*pipx
      fcsx=ceps*fcsxflu+eps*fcsx
      fneutx=ceps*fneutxflu+eps*fneutx
      coul=ceps*coulflu+eps*coul
      plam=ceps*plamflu+eps*plam
    endif
  endif
  return
  end
  subroutine eff (f,t,fl,tl,elnod,ref,ret,pe,pef,pet,se,sef,set,ue,uef,uet,wf,psi)
!
! EFF routine used to find electron or positron thermodynamical quantities
!
  use crlibm_lib, only: log_cr, exp_cr
  implicit none 
  integer (kind=4) :: i,j,k,l
  real (kind=8) :: d(3,3),ff(4),gg(4),c(48)
  real (kind=8) :: fl,tl,f,t,wf,psi,g,cb,cd,ct,vf,vg,uf,ug,fdf,vw,vx,vy,vz,re,ret,ref,pe,pet,pef,qe,qet,qef, &
    se,set,sef,ponrt,ue,uet,uef,elnod
  data c/ 2.315472d0, 7.128660d0, 7.504998d0, 2.665350d0, 7.837752d0, &
         23.507934d0,23.311317d0, 7.987465d0, 9.215560d0,26.834068d0, &
         25.082745d0, 8.020509d0, 3.693280d0,10.333176d0, 9.168960d0, &
          2.668248d0, 2.315472d0, 6.748104d0, 6.564912d0, 2.132280d0, &
          7.837752d0,21.439740d0,19.080088d0, 5.478100d0, 9.215560d0, &
         23.551504d0,19.015888d0, 4.679944d0, 3.693280d0, 8.859868d0, &
          6.500712d0, 1.334124d0, 1.157736d0, 3.770676d0, 4.015224d0, &
          1.402284d0, 8.283420d0,26.184486d0,28.211372d0,10.310306d0, &
         14.755480d0,45.031658d0,46.909420d0,16.633242d0, 7.386560d0, &
         22.159680d0,22.438048d0, 7.664928d0/
  data ff(1),gg(1),cd,cb,ct /1.0d0,1.0d0,2.9218d+06,1.4406d+24,1.6863d-10/
  wf=sqrt(1.0d0+f)
  psi=fl+2.0d0*(wf-log_cr(1.0d0+wf))
  g=ct*t*wf
  vf=1.0d0/(1.0d0+f)
  vg=1.0d0/(1.0d0+g)
  uf=f*vf
  ug=g*vg
  fdf=uf*(g*(1.0d0+g))**1.5d0
  do i=2,4
    ff(i)=f*ff(i-1)
    gg(i)=g*gg(i-1)
    fdf=fdf*vf*vg
  enddo
  l=1
  do i=1,3
    vx=0.0d0
    vy=0.0d0
    vz=0.0d0
    do j=1,4
      do k=1,4
        vw=c(l)*gg(j)*ff(k)
        vx=vx+vw
        vy=vy+(j-1)*vw
        vz=vz+(k-1)*vw
        l=l+1
      enddo
    enddo
    d(i,1)=fdf*vx
    d(i,2)=vy/vx+1.5d0-(1.5d0)*ug
    d(i,3)=vz/vx+1.0d0-(4.0d0-0.5d0*d(i,2))*uf
  enddo
  re=d(1,1)
  ret=d(1,2)
  ref=d(1,3)
  pe=d(2,1)
  pet=d(2,2)
  pef=d(2,3)
  qe=d(3,1)
  qet=d(3,2)
  qef=d(3,3)
  pe=g*pe
  pet=pet+1.0d0
  pef=pef+0.5d0*uf
  qe=qe/(re*wf)
!
! se is the entropy per electron divided by Boltzmann's constant.
!
  se=qe+2.0d0*wf-psi
  sef=qe*(qef-ref-0.5d0*uf)-1.0d0/wf
  set=qe*(qet-ret)
  ponrt=pe/(re*ct*t)
!
! ue is the internal energy per electron divided by kT.
!
  ue=se+psi-ponrt
  uef=sef+wf-ponrt*(pef-ref)
  uet=set-ponrt*(pet-ret-1.0d0)
!
! elnod is the electron number density divided by Avogradro's number.
!
  elnod =cd*re
!
! pe is the electron pressure.
!
  pe =cb*pe
  return
  end
  function debfun(x)
  use crlibm_lib, only: log_cr, exp_cr
  implicit none
  real (kind=8) :: c1,c2,c3,c4,c5,debfun,e1,e2,e3,t1,t2,t3,x,x2,x3
!
! calculates debye function for n=3.
!
  parameter (c1=1.0d0/20.0d0,c2=-1.0d0/1680.0d0,c3=1.0d0/90720.0d0,c4=-1.0d0/4435200.0d0,c5=1.0d0/207567360.0d0)
  if (x.le.2.75d0) then
!
! for small x use first seven terms of series expansion.
!
    x2=x**2
    debfun=1.0d0-0.375d0*x+x2*(c1+x2*(c2+x2*(c3+x2*(c4+x2*c5))))
  else
!
! for large x use first four terms of asymptotic expansion.
!
    e1=exp_cr(-x)
    x2=x**2
    x3=x*x2
    t1=e1*(x3+3.0d0*x2+6.0d0*x+6.0d0)
    e2=e1**2
    t2=e2*(0.5d0*x3+0.75d0*x2+0.75d0*x+0.375d0)
    e3=e1*e2
    t3=e3*(x3/3.0d0+x2/3.0d0+x/4.5d0+1.0d0/13.5d0)
    debfun=3.0d0*(6.493939404d0-(t1+t2+t3))/x3
  endif
  return
  end

!
! subroutine to find equation of state for fluid phase
!
  subroutine statefx_fluid (fl,tl,q,xin, &
    pl,rl,u,p,rho,t,gamma,grada,cp,cv,vt,th,pgas,psi,s,gq,gqq,pf,pt,rf,rt,uf,ut,sf,st,px,rx,sx,ux,pip,pipf,pipt,pipx, &
    fcs,fcsf,fcst,fcsx,fneut,fneutf,fneutt,fneutx,coul,plam,pf_qt,pq_ft,gqf_qt,gqq_ft)
  use crlibm_lib, only: log_cr, exp_cr
  implicit none
  integer (kind=4) :: i,ics,iion,imax,is,j,k,l,ispec,it,n
  integer (kind=4), parameter :: ncomp=14,kionmax=26
  integer (kind=4), dimension(ncomp) :: izmax
  real (kind=8) :: fl,tl,q,pl,rl,u,p,rho,t,gamma,grada,cp,th,pgas,psi,s,gq,pf,pt,rf,rt,uf,ut,sf,st,coul,plam,gqq, &
    nh2,nh,nhplus,coulsol,plamsol,cr,cd,cb,ct,cat,ni,ne0,zmean,f,t4,wf,g,vf,vg,ug,fdf,vx,vy,vz,vw,re,ret,ref,pe, &
    pet,pef,qe,qet,qef,se,sef,set,ponrt,ue,uef,uet,ti,da,db,dc,dcf,dct,dcq,zmeanx,dmupol,dmupolf,dmupolt,dmupolq,dv, &
    dvf,dvt,dvq,eq,hpi,dpi,shpi,hpiq,dpiq,snmax,sdv,sdvf,sdvt,sdvq,vsmax,sumsir,vzsf0,vzst0,vzsq0, &
    vzsl,vzsl1,sisxa,vis,vissq,ff1,ff1c,ff1f,ff1t,ff1q,dh2kt,zetah2,zetah2t,zetah2tt,omegah2,omegah2t,omegah2tt, &
    omegah2l,f2,f2f,f2t,f2q,ne,nef,net,neq,x1ona1,aaa,aaaf,aaat,aaaq,bbb,bbbf,bbbt,bbbq,ccc,cccf,ccct,cccq,bbbb,eee, &
    ddd,hhh,dhbot,hhhf,hhht,hhhq,nhf,nht,nhq,nhplusf,nhplust,nhplusq,nh2f,nh2t,nh2q,nhtot,nhtotf,nhtott,nhtotq, &
    nhtotxn,next,nextf,nextt,nextq,nion,nionf,niont,nionq,nz53,nz53f,nz53t,nz53q,nzsqa,nzsqaf,nzsqat,nzsqaq, &
    psiplus,gpos,sspos,fpos,dpos,fplus,flplus,posnod,posnodf,posnodt,ppos,pposf,ppost,spos,sposf,spost,upos,uposf, &
    upost,dpsiminus,dpsiplus,dflplusdflminus,dflplusdtl,en,elnod,rq,coulf,coult,coulq,coull,plams,plaml,plamf,plamt, &
    plamq,coul32,hcoul,dcoul,ddcoul,coul13,x1,x2,ex1,ex2,debx1,debx2,vlx1,vlx2,dlx1,dlx2,ddlx1,ddlx2,quj,dquj,ddquj, &
    hcoulf,hcoult,hcoulq,dcoulf,dcoult,dcoulq,qujf,qujt,qujq,dqujf,dqujt,dqujq,anhcor,anhcorf,anhcort,anhcorq,crt,p0, &
    p0f,p0t,p0q,pi,pion,pif,pit,piq,pr,pcoul,pcoulf,pcoult,pcoulq,pqu,pquf,pqut,pquq,panhcor,panhcorf,panhcort, &
    panhcorq,ppi,ppif,ppit,ppiq,ne2bnd,ne2bndf,ne2bndt,ne2bndq,ppola,ppol,ppolf,ppolt,ppolq,pq,pionf,piont,nimnh2, &
    x1mnh2,pionq,nimnh2xn,xionai,b,bf,bt,bq,sel,self,selt,selq,sions,sionsf,sionst,sionsq,spi,spif,spit,spiq, &
    scoul,scoulf,scoult,scoulq,squ,squf,squt,squq,sanhcor,sq,spol,spolf,spolt,spolq,spairs1,spairs,spairsf,spairst, &
    spairsq,uion,uionf,uiont,uionq,ucoul,ucoulf,ucoult,ucoulq,uqu,uquf,uqut,uquq,uanhcor,uanhcorf,uanhcort,uanhcorq, &
    uq,upol1,upol,upolf,upolt,upolq,upol2,upairs1,upairs2,upairs,upairsf,upairst,upairsq,fx1,fx2,gq1,gq1f,gq1t,gq1q, &
    gq2,gq2f,gq2t,gq2q,gqf,gqt,qf,qt,pqflu,gqfflu,gqtflu,qq,cv,vt,debfun,wfpos,psipos,pf_qt,pq_ft,gqf_qt,gqq_ft
  real (kind=8), dimension(ncomp) :: x,xin,px,pxflu,rx,rxflu,sx,sxflu,ux,uxflu,pipflu,pipfflu,piptflu,fcsflu,fcsfflu, &
    fcstflu,fneutflu,fneutfflu,fneuttflu,pip,pipf,pipt,fcs,fcsf,fcst,fneut,fneutf,fneutt,a,zmax,fx1coef,dax,dbx,dcx, &
    dvx,sdvx,vzsx0,ff1x,nex,aaax,bbbx,cccx,hhhx,nhx,nhplusx,nh2x,nextx,nionx,nz53x,nzsqax,pix,coulx,plamx,hcoulx, &
    dcoulx,qujx,dqujx,pcoulx,pqux,ppix,ne2bndx,ppolx,sionsx,spix,scoulx,squx,uionx,ucoulx,uqux,upolx,gqx,qx,gq1x, &
    gq2x,anhcorx,panhcorx,uanhcorx,select,selectf,selectt,selectq,sext,sextf,sextt,sextq,sion,sionf,siont,sionq,sis, &
    sisf,sist,sisq,suions,suionsf,suionst,suionsq,sz53,sz53f,sz53t,sz53q,szsq,szsqf,szsqt,szsqq,pipq,fcsq,fneutq, &
    spolx,bx,pionx
  real (kind=8), dimension(kionmax) :: z53,z2
  real (kind=8), dimension(ncomp,ncomp) :: pipx,fcsx,fneutx,selectx,sextx,sionx,sisx,suionsx,sz53x,szsqx
  real (kind=8), parameter :: third=1.0d0/3.0d0,small=1.0d-50 
  real (kind=8) :: d(3,3),ff(4),gg(4),c(48)
  real (kind=8), dimension (kionmax) :: vsf,vst,vsq
  real (kind=8), dimension (0:kionmax) :: vs,sir,spin,spinf,spint,spinq,vzs
  real (kind=8), dimension (kionmax,ncomp) :: vsx,sip
  real (kind=8), dimension (0:kionmax,ncomp) :: spinx,sgsw
!
! approximate atomic weights
!
!      data a/ 1.0d0,2.0d0,3.0d0,4.0d0,7.0d0,7.0d0,12.0d0,13.0d0,14.0d0,
!     1       16.0d0,20.0d0,24.0d0,28.0d0,56.0d0/
!
! accurate atomic weights from NIST
!
  data a/ 1.007825d0,2.014102d0,3.016029d0,4.002603d0,7.016005d0,7.016930d0,12.00000d0,13.00335d0,14.00307d0, &
          15.99491d0,19.99244d0,23.98504d0,27.97693d0,55.93494d0/
  data c/ 2.315472d0, 7.128660d0, 7.504998d0, 2.665350d0, 7.837752d0, &
         23.507934d0,23.311317d0, 7.987465d0, 9.215560d0,26.834068d0, &
         25.082745d0, 8.020509d0, 3.693280d0,10.333176d0, 9.168960d0, &
          2.668248d0, 2.315472d0, 6.748104d0, 6.564912d0, 2.132280d0, &
          7.837752d0,21.439740d0,19.080088d0, 5.478100d0, 9.215560d0, &
         23.551504d0,19.015888d0, 4.679944d0, 3.693280d0, 8.859868d0, &
          6.500712d0, 1.334124d0, 1.157736d0, 3.770676d0, 4.015224d0, &
          1.402284d0, 8.283420d0,26.184486d0,28.211372d0,10.310306d0, &
         14.755480d0,45.031658d0,46.909420d0,16.633242d0, 7.386560d0, &
         22.159680d0,22.438048d0, 7.664928d0/
  data ff(1),gg(1),cr,cd,cb,ct,cat /1.0d0,1.0d0,8.3143d+07,2.9218d+06,1.4406d+24,1.6863d-10,2.5215d-15/
  data fx1coef /0.86d0,0.43d0,0.12d0,0.088d0,0.49d0,0.22d0,0.095d0,0.088d0,0.056d0,0.054d0,0.022d0,0.085d0,0.066d0, &
    0.035d0/
  data izmax /1,1,2,2,3,4,6,6,7,8,10,12,14,26/
!
! ionization potentials in eV, measured from ground state of neutral atom
!
  data sip /  13.598d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              13.598d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              24.587d0,  79.003d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              24.587d0,  79.003d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
               5.392d0,  81.030d0, 203.481d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
               9.322d0,  27.533d0, 181.426d0, 399.139d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              11.260d0,  35.643d0,  83.530d0, 148.022d0, 540.102d0,1030.082d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              11.260d0,  35.643d0,  83.530d0, 148.022d0, 540.102d0,1030.082d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              14.534d0,  44.135d0,  91.583d0, 169.055d0,  266.95d0,  819.01d0, 1486.04d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              13.618d0,  48.734d0,  103.67d0,  181.08d0,  294.98d0,  433.09d0,  1172.4d0,  2043.8d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              21.565d0,  62.528d0,  125.98d0,  223.10d0,  349.32d0,  507.25d0,  714.52d0,  953.62d0, &
              2149.4d0,  3511.6d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
               7.646d0,  22.681d0, 102.824d0,  212.13d0,  353.40d0,  539.91d0,  764.86d0, 1030.78d0, &
              1358.8d0,  1726.3d0,  3488.1d0,  5451.0d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
            8.151683d0,24.49753d0,57.99053d0,103.1323d0,269.8997d0,475.1667d0,721.6473d0,1025.185d0, &
            1376.309d0,1777.683d0,2254.042d0,2777.463d0,5215.091d0,7888.272d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              7.9024d0, 24.1081d0, 54.7595d0,109.5605d0,184.5709d0,283.6343d0,408.6104d0,559.6703d0, &
            793.2565d0,1055.359d0,1345.606d0,1676.396d0,2037.438d0,2429.600d0,2886.606d0,3375.882d0, &
            4638.041d0,6000.007d0,7469.220d0,9044.811d0,10733.48d0,12532.49d0,14491.06d0,16536.80d0, &
            25364.85d0,34642.54d0/
!
! ground state statistical weights for atomic and ionic species
!
  data sgsw /2.0d0,1.0d0,25*0.0d0, &
             2.0d0,1.0d0,25*0.0d0, &
             1.0d0,2.0d0,1.0d0,24*0.0d0, &
             1.0d0,2.0d0,1.0d0,24*0.0d0, &
             2.0d0,1.0d0,2.0d0,1.0d0,23*0.0d0, &
             1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,22*0.0d0, &
             9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,20*0.0d0, &
             9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,20*0.0d0, &
             4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,19*0.0d0, &
             9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,18*0.0d0, &
             1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,16*0.0d0, &
             1.0d0,2.0d0,1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,14*0.0d0, &
             9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,12*0.0d0, &
            25.0d0,30.0d0,25.0d0,6.0d0,25.0d0,28.0d0,21.0d0,10.0d0,1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0, &
             1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0/
  data zmax /1.0d0,1.0d0,2.0d0,2.0d0,3.0d0,4.0d0,6.0d0,6.0d0,7.0d0,8.0d0,10.0d0,12.0d0,14.0d0,26.0d0/
!
! the following numbers are Z**(5/3) and Z**2 for Z = 1 to 26 for use in the
! Coulomb corrections and quantum corrections respectively.
!
  data z53 / 1.000d0,3.175d0,6.240d0,10.08d0,14.62d0,19.81d0,25.62d0,32.00d0,38.94d0,46.42d0,54.41d0,62.90d0, &
             71.87d0,81.32d0,91.23d0,101.6d0,112.4d0,123.6d0,135.3d0,147.4d0,159.8d0,172.7d0,186.0d0,199.7d0, &
             213.7d0,228.2d0/
  data z2  / 1.000d0,4.000d0,9.000d0,16.00d0,25.00d0,36.00d0,49.00d0,64.00d0,81.00d0,100.0d0,121.0d0,144.0d0, &
             169.0d0,196.0d0,225.0d0,256.0d0,289.0d0,324.0d0,361.0d0,400.0d0,441.0d0,484.0d0,529.0d0,576.0d0, &
             625.0d0,676.0d0/
!  do ispec=1,ncomp
!    x(ispec)=max(xin(ispec),0.0d0)
!  enddo
  x(:)=max(xin(:),0.0d0)
  ni=0.0d0
  ne0=0.0d0
  do ispec=1,ncomp
    ni=ni+x(ispec)/a(ispec)
    ne0=ne0+x(ispec)*zmax(ispec)/a(ispec)
  enddo
  zmean=ne0/ni
!
! EFF section to find electron number density, etc
!
  f=exp_cr(fl)
  t=exp_cr(tl)
  t4=t**4
  call eff (f,t,fl,tl,elnod,ref,ret,pe,pef,pet,se,sef,set,ue,uef,uet,wf,psi)
!
! th used in weak screening
!
  th=ref/wf
!
! inverse of temperature in eV
!
  ti=11605.0d0/t
!
! modified EFF non-ideal contribution related to 'polarization' i.e. electric microfield
!
  da=0.0862d0/zmean**3
  db=da*270.0d0*zmean**2
  dc=da+db*ti
  dcf=0.0d0
  dct=-db*ti/dc
  dcq=0.0d0
  do n=1,ncomp
    zmeanx=(zmax(n)/ne0-1.0d0/ni)/a(n)
    dax(n)=-3.0d0*da*zmeanx
    dbx(n)=-      db*zmeanx
    dcx(n)=dax(n)+ti*dbx(n)
  enddo
  dmupol =dc*elnod
  dmupolf=dmupol*(ref+dcf)
  dmupolt=dmupol*(ret+dct)
  dmupolq=0.0d0
  dv=-psi+dmupol
  dvf=-wf+dmupolf
  dvt=dmupolt
  dvq=0.0d0
  dvx=dcx*elnod
!
! factors needed for pressure ionization. This is determined from a simplified hard sphere free energy term
! HS free energy is based on that in Graboske, Harwood & Rogers 1969 Phys. Rev., 186, 210
!
  eq=exp_cr(q)
  hpi=log_cr(1.0d0+eq)+eq*(3.0d0+1.5d0*eq)
  dpi=eq*(4.0d0+eq*(6.0d0+eq*3.0d0))
  shpi=eq**2*(10.0d0+eq*(18.0d0+eq*9.0d0))
  hpiq=dpi/(1.0d0+eq)
  dpiq=(dpi+shpi)/(1.0d0+eq)
!
! ionization of all species, except H which is treated separately because of H2 molecule
!
  do ispec=2,ncomp
    imax=izmax(ispec)
    snmax=x(ispec)/a(ispec)
    sdv=0.0d0
    sdvf=0.0d0
    sdvt=0.0d0
    sdvq=0.0d0
    sdvx=0.0d0
    vs(0)=0.0d0
    do is=1,imax
       sdv=sdv+dv
       sdvf=sdvf+dvf
       sdvt=sdvt+dvt
       sdvq=sdvq+dvq
       sdvx=sdvx+dvx
       vs (is)=sdv -sip(is,ispec)*ti
       vsf(is)=sdvf
       vst(is)=sdvt+sip(is,ispec)*ti
       vsq(is)=sdvq
       vsx(is,:)=sdvx(:)
     enddo
     vs (imax)=vs (imax)+hpi +dpi
     vsq(imax)=vsq(imax)+hpiq+dpiq
!        
! find commonest ion.
!
     vsmax=vs(0)
     ics=0
     do is=1,imax
       if (vs(is).gt.vsmax) then
         vsmax=vs(is)
         ics=is
       endif
     enddo
!
! calculate ionic ratios relative to commonest ion.
!
     sumsir=0.0d0
     do is=0,imax
       sir(is)=exp_cr(vs(is)-vs(ics))*sgsw(is,ispec)/sgsw(ics,ispec)
       sumsir=sumsir+sir(is)
     enddo
!
! vzs contains fraction in each ionization state
!
     do is=0,imax
       vzs(is)=sir(is)/sumsir
       spin(is)=vzs(is)*snmax
     enddo
!
! derivatives of ln vzs(0)
!
     vzsf0=0.0d0
     vzst0=0.0d0
     vzsq0=0.0d0
     vzsx0=0.0d0
     do iion=1,imax
       vzsf0=vzsf0-vsf(iion)*vzs(iion)
       vzst0=vzst0-vst(iion)*vzs(iion)
       vzsq0=vzsq0-vsq(iion)*vzs(iion)
       do n=1,ncomp
         vzsx0(n)=vzsx0(n)-vsx(iion,n)*vzs(iion)
       enddo
     enddo
!
! derivatives of spin(0)
!
     spinf(0)=spin(0)*vzsf0
     spint(0)=spin(0)*vzst0
     spinq(0)=spin(0)*vzsq0
     do n=1,ncomp
        spinx(0,n)=spin(0)*vzsx0(n)
     enddo
     spinx(0,ispec)=spinx(0,ispec)+vzs(0)/a(ispec)
!
! calculate quantities needed to find density, pressure, entropy, internal energy
! and their derivatives
!
     vzsl=log_cr(vzs(0)/sgsw(0,ispec)+small)
     vzsl1=vzsl+vzs(0)/(vzs(0)+sgsw(0,ispec)*small)
!
! sum over ionization states for each species
!
!
! select is used to find number of free electrons from each species
!
     select(ispec)=0.0d0
     selectf(ispec)=0.0d0
     selectt(ispec)=0.0d0
     selectq(ispec)=0.0d0
!
! sion is used to find number of ions (excludes neutrals)
!
     sion (ispec)=0.0d0
     sionf(ispec)=0.0d0
     siont(ispec)=0.0d0
     sionq(ispec)=0.0d0
!
! suions is used to find contribution of bound states to the internal energy density
!
     suions (ispec)=spin (0)*(-sip(imax,ispec))
     suionsf(ispec)=spinf(0)*(-sip(imax,ispec))
     suionst(ispec)=spint(0)*(-sip(imax,ispec))
     suionsq(ispec)=spinq(0)*(-sip(imax,ispec))
!
! sz53 and szs are sums of z^5/3 and Z^2 needed to find Coulomb Gamma and the plasma parameter
! respectively
!
     sz53 (ispec)=0.0d0
     sz53f(ispec)=0.0d0
     sz53t(ispec)=0.0d0
     sz53q(ispec)=0.0d0
     szsq (ispec)=0.0d0
     szsqf(ispec)=0.0d0
     szsqt(ispec)=0.0d0
     szsqq(ispec)=0.0d0
!
! sis is needed to find entropy
!
     sis (ispec)=snmax*(1.5d0*log_cr(a(ispec))-log_cr(snmax+small))-spin (0)*vzsl
     sisf(ispec)=-spinf(0)*vzsl1
     sist(ispec)=-spint(0)*vzsl1
     sisq(ispec)=-spinq(0)*vzsl1
     sisxa=spin(0)*vzs(0)/(vzs(0)+small*sgsw(0,ispec))
     do n=1,ncomp
       selectx(ispec,n)=0.0d0
       sionx(ispec,n)=0.0d0
       suionsx(ispec,n)=spinx(0,n)*(-sip(imax,ispec))
       sz53x(ispec,n)=0.0d0
       szsqx(ispec,n)=0.0d0
       sisx(ispec,n)=-spinx(0,n)*vzsl-vzsx0(n)*sisxa
     enddo
     sisx(ispec,ispec)=sisx(ispec,ispec)+(1.5d0*log_cr(a(ispec))-log_cr(snmax+small)-snmax/(snmax+small))/a(ispec)
     vis=0.0d0
     do is=1,imax
       vis=vis+1.0d0
       vissq=vis**2
       spinf(is)=spin(is)*(vzsf0+vsf(is))
       spint(is)=spin(is)*(vzst0+vst(is))
       spinq(is)=spin(is)*(vzsq0+vsq(is))
       select(ispec)=select(ispec)+vis*spin (is)
       selectf(ispec)=selectf(ispec)+vis*spinf(is)
       selectt(ispec)=selectt(ispec)+vis*spint(is)
       selectq(ispec)=selectq(ispec)+vis*spinq(is)
       sion (ispec)=sion (ispec)+spin (is)
       sionf(ispec)=sionf(ispec)+spinf(is)
       siont(ispec)=siont(ispec)+spint(is)
       sionq(ispec)=sionq(ispec)+spinq(is)
       vzsl=log_cr(vzs(is)/sgsw(is,ispec)+small)
       vzsl1=vzsl+vzs(is)/(vzs(is)+sgsw(is,ispec)*small)
       sis (ispec)=sis (ispec)-spin (is)*vzsl
       sisf(ispec)=sisf(ispec)-spinf(is)*vzsl1
       sist(ispec)=sist(ispec)-spint(is)*vzsl1
       sisq(ispec)=sisq(ispec)-spinq(is)*vzsl1
       sz53 (ispec)=sz53 (ispec)+z53(is)*spin (is)
       sz53f(ispec)=sz53f(ispec)+z53(is)*spinf(is)
       sz53t(ispec)=sz53t(ispec)+z53(is)*spint(is)
       sz53q(ispec)=sz53q(ispec)+z53(is)*spinq(is)
       szsq (ispec)=szsq (ispec)+z2(is)*spin (is)
       szsqf(ispec)=szsqf(ispec)+z2(is)*spinf(is)
       szsqt(ispec)=szsqt(ispec)+z2(is)*spint(is)
       szsqq(ispec)=szsqq(ispec)+z2(is)*spinq(is)
       suions (ispec)=suions (ispec)+spin (is)*(sip(is,ispec)-sip(imax,ispec))
       suionsf(ispec)=suionsf(ispec)+spinf(is)*(sip(is,ispec)-sip(imax,ispec))
       suionst(ispec)=suionst(ispec)+spint(is)*(sip(is,ispec)-sip(imax,ispec))
       suionsq(ispec)=suionsq(ispec)+spinq(is)*(sip(is,ispec)-sip(imax,ispec))
       sisxa=spin(is)*vzs(is)/(vzs(is)+small*sgsw(is,ispec))
       do n=1,ncomp
         spinx(is,n)=spin(is)*(vzsx0(n)+vsx(is,n))
         if (n.eq.ispec) spinx(is,n)=spinx(is,n)+vzs(is)/a(n)
         selectx(ispec,n)=selectx(ispec,n)+vis*spinx(is,n)
         sionx(ispec,n)=sionx(ispec,n)+spinx(is,n)
         sisx(ispec,n)=sisx(ispec,n)-spinx(is,n)*vzsl-sisxa*(vsx(is,n)+vzsx0(n))
         sz53x(ispec,n)=sz53x(ispec,n)+z53(is)*spinx(is,n)
         szsqx(ispec,n)=szsqx(ispec,n)+z2(is)*spinx(is,n)
         suionsx(ispec,n)=suionsx(ispec,n)+spinx(is,n)*(sip(is,ispec)-sip(imax,ispec))
       enddo
     enddo
!
! sext is used to find number of extended particles (i.e. not electrons or bare nuclei)
!
     sext (ispec)=0.0d0
     sextf(ispec)=0.0d0
     sextt(ispec)=0.0d0
     sextq(ispec)=0.0d0
     do n=1,ncomp
       sextx(ispec,n)=0.0d0
     enddo
     do is=0,imax-1
       sext (ispec)=sext (ispec)+spin (is)
       sextf(ispec)=sextf(ispec)+spinf(is)
       sextt(ispec)=sextt(ispec)+spint(is)
       sextq(ispec)=sextq(ispec)+spinq(is)
       do n=1,ncomp
         sextx(ispec,n)=sextx(ispec,n)+spinx(is,n)
       enddo
     enddo
!
! fcs = select / X_i/A_i is the mean charge of all particles of an individual species
!
     fcs (ispec)=select (ispec)/snmax
     fcsf(ispec)=selectf(ispec)/snmax
     fcst(ispec)=selectt(ispec)/snmax
     fcsq(ispec)=selectq(ispec)/snmax
     do n=1,ncomp
       fcsx(ispec,n)=selectx(ispec,n)/snmax
     enddo
!
! fneut = fraction of particles of an individual species that are neutral
!
     fneut (ispec)=vzs(0)
     fneutf(ispec)=vzs(0)*vzsf0
     fneutt(ispec)=vzs(0)*vzst0
     fneutq(ispec)=vzs(0)*vzsq0
     do n=1,ncomp
       fneutx(ispec,n)=vzs(0)*vzsx0(n)
     enddo
   enddo
!
! treatment of H to include H2 molecules
!
   snmax=x(1)/a(1)
   vs(0)=0.0d0
   vs (1)=dv -sip(1,1)*ti
   vsf(1)=dvf
   vst(1)=dvt+sip(1,1)*ti
   vsq(1)=dvq
   vs (1)=vs (1)+hpi +dpi
   vsq(1)=vsq(1)+hpiq+dpiq
   do n=1,ncomp
     vsx(1,n)=dvx(n)
   enddo
!        
! find commonest ion.
!
   vsmax=vs(0)
   ics=0
   if (vs(1).gt.vsmax) then
     vsmax=vs(1)
     ics=1
   endif
!
! calculate ionic ratios relative to commonest ion.
!
   sumsir=0.0d0
   do is=0,1
     sir(is)=exp_cr(vs(is)-vs(ics))*sgsw(is,1)/sgsw(ics,1)
     sumsir=sumsir+sir(is)
   enddo
   do is=0,1
     vzs(is)=sir(is)/sumsir
     spin(is)=vzs(is)*snmax
   enddo
   vzsf0=-vsf(1)*vzs(1)
   vzst0=-vst(1)*vzs(1)
   vzsq0=-vsq(1)*vzs(1)
   spinf(0)=spin(0)*vzsf0
   spint(0)=spin(0)*vzst0
   spinq(0)=spin(0)*vzsq0
   spinf(1)=spin(1)*(vzsf0+vsf(1))
   spint(1)=spin(1)*(vzst0+vst(1))
   spinq(1)=spin(1)*(vzsq0+vsq(1))
   do n=1,ncomp
     vzsx0(n)=-vsx(1,n)*vzs(1)
     spinx(0,n)=spin(0)*vzsx0(n)
     spinx(1,n)=spin(1)*(vzsx0(n)+vsx(1,n))
   enddo
   spinx(0,1)=spinx(0,1)+vzs(0)/a(1)
   spinx(1,1)=spinx(1,1)+vzs(1)/a(1)
!
! ff1 = NH_plus/(NH_plus+NH)
!
   ff1 =spin(1)/(spin(0)+spin(1))
   ff1c=spin(0)/(spin(0)+spin(1))
   ff1f=ff1c*ff1*vsf(1)
   ff1t=ff1c*ff1*vst(1)
   ff1q=ff1c*ff1*vsq(1)
   do n=1,ncomp
     ff1x(n)=ff1c*ff1*vsx(1,n)
   enddo
!
! H2 partition function from Pols et al (1995), Webbink (1975)
! with modification at low temperature
!
   dh2kt=4.48d0*ti
   zetah2=1.0d0-(1.0d0+dh2kt)*exp_cr(-dh2kt)
   zetah2t=-dh2kt**2*exp_cr(-dh2kt)/zetah2
   zetah2tt=zetah2t*(dh2kt-2.0d0-zetah2t)
   if (ti.le.10.0d0) then
     omegah2=6608.8d0*zetah2*exp_cr(ti*(0.448d0+ti*(-0.02440d0+ti*6.163d-04)))/dh2kt**2.5d0
     omegah2t=2.5d0-ti*(0.448d0+ti*(-0.04880d0+ti*1.849d-03))+zetah2t
     omegah2tt=ti*(0.448d0+ti*(-0.0976d0+ti*5.547d-03))+zetah2tt
   else
     omegah2l =(33.4682d0+(-190.382d0+503.94d0      /ti)/ti)/ti
     omegah2  =exp_cr(omegah2l)
     omegah2t =(33.4682d0+(-190.382d0*2.0d0+503.94d0*3.0d0/ti)/ti)/ti
     omegah2tt=(33.4682d0+(-190.382d0*4.0d0+503.94d0*9.0d0/ti)/ti)/ti
   endif
!
! f2 = NH_2*Ne/NH^2
!
  f2 =exp_cr(dh2kt+7.726d0-1.5d0*tl+log_cr(omegah2*elnod))
  f2f=f2*ref
  f2t=f2*(-dh2kt-1.5d0+omegah2t+ret)
  f2q=0.0d0
!
! ne is the inverse of the molecular weight per electron, excluding contribution from H
! Contributions from H are added later
!
  ne =0.0d0
  nef=0.0d0
  net=0.0d0
  neq=0.0d0
  nex=0.0d0
  do ispec=2,ncomp
    ne =ne +select(ispec)
    nef=nef+selectf(ispec)
    net=net+selectt(ispec)
    neq=neq+selectq(ispec)
    do n=1,ncomp
      nex(n)=nex(n)+selectx(ispec,n)
    enddo
  enddo
!
! coefficients of quadratic equation for h and their derivatives
! h = NH_plus/(ff1*X(1)/A(1)) where X(1) is H mass fraction
!
  x1ona1=x(1)/a(1)
  aaa =(ff1 +2.0d0*f2 *ff1c**2                   )*x1ona1
  aaaf=(ff1f+2.0d0*f2f*ff1c**2-4.0d0*f2*ff1c*ff1f)*x1ona1
  aaat=(ff1t+2.0d0*f2t*ff1c**2-4.0d0*f2*ff1c*ff1t)*x1ona1
  aaaq=(ff1q+2.0d0*f2q*ff1c**2-4.0d0*f2*ff1c*ff1q)*x1ona1
  aaax=(ff1x                  -4.0d0*f2*ff1c*ff1x)*x1ona1
  aaax(1)=aaax(1)+(ff1+2.0d0*f2*ff1c**2)/a(1)
  bbb =ne -x1ona1*ff1
  bbbf=nef-x1ona1*ff1f
  bbbt=net-x1ona1*ff1t
  bbbq=neq-x1ona1*ff1q
  bbbx=nex-x1ona1*ff1x
  bbbx(1)=bbbx(1)-ff1/a(1)
  ccc =-ne
  cccf=-nef
  ccct=-net
  cccq=-neq
  cccx=-nex
  bbbb=0.5d0*bbb
  eee=bbbb/aaa
  ddd=sqrt(eee**2-ccc/aaa)
  if (bbbb.le.0.0d0) then
    hhh=ddd-eee
  else
    hhh=(-ccc/aaa)/(ddd+eee)
  endif
  dhbot=2.0d0*hhh*aaa+bbb
  hhhf=-(hhh**2*aaaf+hhh*bbbf+cccf)/dhbot
  hhht=-(hhh**2*aaat+hhh*bbbt+ccct)/dhbot
  hhhq=-(hhh**2*aaaq+hhh*bbbq+cccq)/dhbot
  hhhx=-(hhh**2*aaax+hhh*bbbx+cccx)/dhbot
!
! nh is number of hydrogen atoms per unit mass divided by Avogadro's number (NA)
! nhplus is number of protons per unit mass divided by NA
! nh2 is number of hydrogen molecules per unit mass divided by NA
!
  nh = hhh *ff1c          *x1ona1
  nhf=(hhhf*ff1c-hhh*ff1f)*x1ona1
  nht=(hhht*ff1c-hhh*ff1t)*x1ona1
  nhq=(hhhq*ff1c-hhh*ff1q)*x1ona1
  nhx=(hhhx*ff1c-hhh*ff1x)*x1ona1
  nhplus = hhh *ff1          *x1ona1
  nhplusf=(hhhf*ff1+hhh*ff1f)*x1ona1
  nhplust=(hhht*ff1+hhh*ff1t)*x1ona1
  nhplusq=(hhhq*ff1+hhh*ff1q)*x1ona1
  nhplusx=(hhhx*ff1+hhh*ff1x)*x1ona1
  nh2=nh**2*f2/(nhplus+ne)
  nh2f=-0.5d0*(nhf+nhplusf)
  nh2t=-0.5d0*(nht+nhplust)
  nh2q=-0.5d0*(nhq+nhplusq)
  nh2x=-0.5d0*(nhx+nhplusx)
  nhx(1)=nhx(1)+hhh*ff1c/a(1)
  nhplusx(1)=nhplusx(1)+hhh*ff1/a(1)
  nh2x(1)=0.5d0/a(1)-0.5d0*(nhx(1)+nhplusx(1))
!
! mean Z and neutral fraction for H
!
  nhtot =nh2 +nh +nhplus
  nhtotf=nh2f+nhf+nhplusf
  nhtott=nh2t+nht+nhplust
  nhtotq=nh2q+nhq+nhplusq
  fcs (1)=nhplus /nhtot
  fcsf(1)=(nhplusf-nhplus*nhtotf/nhtot)/nhtot
  fcst(1)=(nhplust-nhplus*nhtott/nhtot)/nhtot
  fcsq(1)=(nhplusq-nhplus*nhtotq/nhtot)/nhtot
  fneut (1)=(nh2 +nh )/nhtot
  fneutf(1)=(nh2f+nhf-(nh2+nh)*nhtotf/nhtot)/nhtot
  fneutt(1)=(nh2t+nht-(nh2+nh)*nhtott/nhtot)/nhtot
  fneutq(1)=(nh2q+nhq-(nh2+nh)*nhtotq/nhtot)/nhtot
  do n=1,ncomp
    nhtotxn=nh2x(n)+nhx(n)+nhplusx(n)
    fcsx  (1,n)=(nhplusx(n)    -nhplus  *nhtotxn/nhtot)/nhtot
    fneutx(1,n)=(nh2x(n)+nhx(n)-(nh2+nh)*nhtotxn/nhtot)/nhtot
  enddo
  select(1)=nhplus
  selectf(1)=nhplusf
  selectt(1)=nhplust
  selectq(1)=nhplusq
  sext (1)=nh +nh2
  sextf(1)=nhf+nh2f
  sextt(1)=nht+nh2t
  sextq(1)=nhq+nh2q
  sion (1)=nhplus
  sionf(1)=nhplusf
  siont(1)=nhplust
  sionq(1)=nhplusq
  sz53 (1)=nhplus
  sz53f(1)=nhplusf
  sz53t(1)=nhplust
  sz53q(1)=nhplusq
  szsq (1)=nhplus
  szsqf(1)=nhplusf
  szsqt(1)=nhplust
  szsqq(1)=nhplusq
  suions (1)=-nh2 *31.676d0-nh *13.598d0
  suionsf(1)=-nh2f*31.676d0-nhf*13.598d0
  suionst(1)=-nh2t*31.676d0-nht*13.598d0
  suionsq(1)=-nh2q*31.676d0-nhq*13.598d0
  sis (1)=-nh2*log_cr(0.35355d0*nh2/omegah2+small)-nh*log_cr(nh/sgsw(0,1)+small)-nhplus*log_cr(nhplus/sgsw(1,1)+small) &
    +1.5d0*(nh+nhplus+nh2)*log_cr(a(1))
  sisf(1)=-nh2f*(log_cr(0.35355d0*nh2/omegah2+small)+nh2/(nh2+2.82845d0*small*omegah2))-nhf*(log_cr(nh/sgsw(0,1)+small) &
    +nh/(nh+small*sgsw(0,1)))-nhplusf*(log_cr(nhplus/sgsw(1,1)+small)+nhplus/(nhplus+small*sgsw(1,1))) &
    +1.5d0*(nhf+nhplusf+nh2f)*log_cr(a(1))
  sist(1)=-nh2t*(log_cr(0.35355d0*nh2/omegah2+small)+nh2/(nh2+2.82845d0*small*omegah2))-nht*(log_cr(nh/sgsw(0,1)+small) &
    +nh/(nh+small*sgsw(0,1)))-nhplust*(log_cr(nhplus/sgsw(1,1)+small)+nhplus/(nhplus+small*sgsw(1,1))) &
    +nh2*(nh2/(nh2+2.82845d0*small*omegah2))*omegah2t+1.5d0*(nht+nhplust+nh2t)*log_cr(a(1))
  sisq(1)=-nh2q*(log_cr(0.35355d0*nh2/omegah2+small)+nh2/(nh2+2.82845d0*small*omegah2))-nhq*(log_cr(nh/sgsw(0,1)+small) &
    +nh/(nh+small*sgsw(0,1)))-nhplusq*(log_cr(nhplus/sgsw(1,1)+small)+nhplus/(nhplus+small*sgsw(1,1))) &
    +1.5d0*(nhq+nhplusq+nh2q)*log_cr(a(1))
  do n=1,ncomp
    selectx(1,n)=nhplusx(n)
    sextx(1,n)=nhx(n)+nh2x(n)
    sionx(1,n)=nhplusx(n)
    sz53x(1,n)=nhplusx(n)
    szsqx(1,n)=nhplusx(n)
    suionsx(1,n)=-nh2x(n)*31.676d0-nhx(n)*13.598d0
    sisx(1,n)=-nh2x(n)*(log_cr(0.35355d0*nh2/omegah2+small)+nh2/(nh2+2.82845d0*small*omegah2)) &
      -nhx(n)*(log_cr(nh/sgsw(0,1)+small)+nh/(nh+small*sgsw(0,1)))-nhplusx(n)*(log_cr(nhplus/sgsw(1,1)+small) &
      +nhplus/(nhplus+small*sgsw(1,1)))+1.5d0*(nhx(n)+nhplusx(n)+nh2x(n))*log_cr(a(1))
  enddo
!
! sums over species
!
  ne =0.0d0
  nef=0.0d0
  net=0.0d0
  neq=0.0d0
  next =0.0d0
  nextf=0.0d0
  nextt=0.0d0
  nextq=0.0d0
  nion =0.0d0
  nionf=0.0d0
  niont=0.0d0
  nionq=0.0d0
  nz53 =0.0d0
  nz53f=0.0d0
  nz53t=0.0d0
  nz53q=0.0d0
  nzsqa =0.0d0
  nzsqaf=0.0d0
  nzsqat=0.0d0
  nzsqaq=0.0d0
  nex   =0.0d0
  nextx =0.0d0
  nionx =0.0d0
  nz53x =0.0d0
  nzsqax=0.0d0
  do ispec=1,ncomp
    ne =ne +select(ispec)
    nef=nef+selectf(ispec)
    net=net+selectt(ispec)
    neq=neq+selectq(ispec)
    next =next +sext (ispec)
    nextf=nextf+sextf(ispec)
    nextt=nextt+sextt(ispec)
    nextq=nextq+sextq(ispec)
    nion =nion +sion (ispec)
    nionf=nionf+sionf(ispec)
    niont=niont+siont(ispec)
    nionq=nionq+sionq(ispec)
    nz53 =nz53 +sz53 (ispec)
    nz53f=nz53f+sz53f(ispec)
    nz53t=nz53t+sz53t(ispec)
    nz53q=nz53q+sz53q(ispec)
    nzsqa =nzsqa +szsq (ispec)/a(ispec)
    nzsqaf=nzsqaf+szsqf(ispec)/a(ispec)
    nzsqat=nzsqat+szsqt(ispec)/a(ispec)
    nzsqaq=nzsqaq+szsqq(ispec)/a(ispec)
    do n=1,ncomp
      nex(n)   =nex(n)   +selectx(ispec,n)
      nextx(n) =nextx(n) +sextx(ispec,n)
      nionx(n) =nionx(n) +sionx(ispec,n)
      nz53x(n) =nz53x(n) +sz53x(ispec,n)
      nzsqax(n)=nzsqax(n)+szsqx(ispec,n)/a(ispec)
    enddo
  enddo
!
! contributions from electron - positron pairs
!
!
! first iterate to find flplus = fl for the positrons
!
  psiplus=-psi-2.0d0/(ct*t)
  if (tl.ge.19.0d0.and.psiplus.ge.-170.0d0) then
    gpos=exp_cr(psiplus-2.0d0)
    if (gpos.le.4.33d-09) then
      sspos=gpos*(2.0d0-4.0d0*gpos)
    else
      sspos=log_cr(1.0d0+4.0d0*gpos)/2.0d0
    endif
    do it=1,4
      fpos=2.0d0+2.0d0*sspos+log_cr(sspos/(2.0d0+sspos))-psiplus
      dpos=2.0d0+2.0d0/(sspos*(2.0d0+sspos))
      sspos=sspos-fpos/dpos
    enddo
    fplus=sspos*(2.0d0+sspos)
    flplus=log_cr(fplus)
!
! use EFF routine to find the equivalent of ne, pe, etc for the positrons
!
    call eff (fplus,t,flplus,tl,posnod,posnodf,posnodt,ppos,pposf,ppost,spos,sposf,spost,upos,uposf,upost,wfpos,psipos)
!
! find derivatives of positron number density, etc. with respect to fl (minus) at constant t
! and with respect to t at constant fl 
!
    dpsiminus=1.0d0/f    +1.0d0/(1.0d0+sqrt(f    ))
    dpsiplus =1.0d0/fplus+1.0d0/(1.0d0+sqrt(fplus))
    dflplusdflminus=-f*dpsiminus/(fplus*dpsiplus)
    dflplusdtl=2.0d0/(ct*t*fplus*dpsiplus)
    posnodt=posnodt+posnodf*dflplusdtl
    posnodf=posnodf*dflplusdflminus
    ppost=ppost+pposf*dflplusdtl
    pposf=pposf*dflplusdflminus
    spost=spost+sposf*dflplusdtl
    sposf=sposf*dflplusdflminus
    upost=upost+uposf*dflplusdtl
    uposf=uposf*dflplusdflminus
  else
    posnod =0.0d0
    posnodf=0.0d0
    posnodt=0.0d0
    ppos =0.0d0
    pposf=0.0d0
    ppost=0.0d0
    spos =0.0d0
    sposf=0.0d0
    spost=0.0d0
    upos =0.0d0
    uposf=0.0d0
    upost=0.0d0
  endif
!
! en is mean molecular weight per electron from ionization
! elnod is electron number density divided by NA, includes those from pairs
!
  en=1.0d0/ne
!
! subtract electrons from pairs to find density
!
  rho=(elnod-posnod)*en
  rl=log_cr(rho)
  rf=(elnod*ref-posnod*posnodf)/(elnod-posnod)-en*nef
  rt=(elnod*ret-posnod*posnodt)/(elnod-posnod)-en*net
  rq=                                         -en*neq
  rx=                                         -en*nex
!
! include coulomb and quantum corrections to equation of state by assuming 
! they do not alter the ionisation balance. Follows the prescription of the 
! eos in Iben, Fujimoto and MacDonald, 1992 ApJ, 388, 521. Updated to use 
! results of Stringfellow, G.S., DeWitt, H.E. & Slattery, W.L. 1990 Phys. Rev. 
! A, 41, 1105.
!
  coul=2.275d+05*nz53*(ne*rho)**third/(t*nion)
!
! derivatives of ln coul.
!
  coulf=nz53f/nz53-nionf/nion+ref/3.0d0
  coult=nz53t/nz53-niont/nion+ret/3.0d0-1.0d0
  coulq=nz53q/nz53-nionq/nion
  coulx=nz53x/nz53-nionx/nion
  coull=log_cr(coul)
  plam =7.833d+03*sqrt(rho*nzsqa)/t
  plams=plam**2
  plaml=log_cr(plam)
!
! derivatives of ln plam
!
  plamf=0.5d0*(rf+nzsqaf/nzsqa)
  plamt=0.5d0*(rt+nzsqat/nzsqa)-1.0d0
  plamq=0.5d0*(rq+nzsqaq/nzsqa)
  plamx=0.5d0*(rx+nzsqax/nzsqa)
!
! fluid phase
!
  if (coul.le.1.0d0) then
!
! gas phase.
!
    coul32=sqrt(coul)**3
    hcoul =coul32*(0.577350d0+coul32*(-0.103489d0+0.173515d0*coull-coul32*0.034819d0))
    dcoul =coul32*(0.866025d0+coul32*(-0.136952d0+0.520545d0*coull-coul32*0.156687d0))
    ddcoul=coul32*(1.299038d0+coul32*( 0.109689d0+1.561635d0*coull-coul32*0.705091d0))
  else
!
! liquid phase.
!
    if (coul.le.200.0d0) then
      coul13=coul**third
      hcoul =0.899375d0*coul-1.707999d0*coul13+0.224470d0*coull+1.301291d0-0.053625d0/coul13
      dcoul =0.899375d0*coul-0.569333d0*coul13+0.224470d0                 +0.017875d0/coul13
      ddcoul=0.899375d0*coul-0.189778d0*coul13                            -0.005958d0/coul13
    else
      hcoul =0.895929d0*coul-2.834398d0*coull+8.621387d0-84.32908d0/coul
      dcoul =0.895929d0*coul-2.834398d0                 +84.32908d0/coul
      ddcoul=0.895929d0*coul                            -84.32908d0/coul
    endif
 endif
!
! quantum corrections for fluid phase
!
!      quc=1.0d0/sqrt(1.0d0+2.697d-03*plams)
!      qucs=quc**2
!      quj=plams*quc*(2.0d0+quc)/(36.0d0*(1.0d0+quc))
!      dquj=plams*quc*(2.0d0+qucs)/36.0d0
!      ddquj=plams*quc*(2.0d0+qucs*(1.0d0+3.0d0*qucs))/36.0d0
!
! An alternative is to use the same form as for the solid phase
! To use this option uncomment the statements below
!
! also if anhcor (see below) is set to zero then fluid - solid phase transition should occur
! at constant Coulomb gamma
!
  x1=plam/1.0643d0
  x2=plam/2.9438d0
  ex1=exp_cr(-x1)
  ex2=exp_cr(-x2)
  debx1=debfun(x1)
  debx2=debfun(x2)
  vlx1=1.125d0*x1+3.0d0*log_cr(1.0d0-ex1)-debx1
  vlx2=1.125d0*x2+3.0d0*log_cr(1.0d0-ex2)-debx2
  dlx1=1.125d0*x1+3.0d0*debx1
  dlx2=1.125d0*x2+3.0d0*debx2
  ddlx1=1.125d0*x1+9.0d0*x1*ex1/(1.0d0-ex1)-9.0d0*debx1
  ddlx2=1.125d0*x2+9.0d0*x2*ex2/(1.0d0-ex2)-9.0d0*debx2
  quj  =0.5711d0*vlx1 +0.4289d0*vlx2-3.0d0*plaml+1.17251d0+1.32351d0
  dquj =0.5711d0*dlx1 +0.4289d0*dlx2-3.0d0
  ddquj=0.5711d0*ddlx1+0.4289d0*ddlx2
  hcoulf= dcoul*coulf
  hcoult= dcoul*coult
  hcoulq= dcoul*coulq
  hcoulx= dcoul*coulx
  dcoulf=ddcoul*coulf
  dcoult=ddcoul*coult
  dcoulq=ddcoul*coulq
  dcoulx=ddcoul*coulx
  qujf =dquj *plamf
  qujt =dquj *plamt
  qujq =dquj *plamq
  qujx =dquj *plamx
  dqujf=ddquj*plamf
  dqujt=ddquj*plamt
  dqujq=ddquj*plamq
  dqujx=ddquj*plamx
!
! add zero-temperature anharmonic corrections for fluid phase
! Purpose is to have breakup of solid due to quantum vibrations when density
! becomes very large (about 2 10^13 g cm^-3)
!
!  anhcor=2.22027d0*plams/coul
  anhcor=0.0d0
  anhcorf=anhcor*(2.0d0*plamf-coulf)
  anhcort=anhcor*(2.0d0*plamt-coult)
  anhcorq=anhcor*(2.0d0*plamq-coulq)
  anhcorx=anhcor*(2.0d0*plamx-coulx)
!
! the various contributions to pressure
!
  crt=cr*t
  p0=crt*rho
  p0f=p0*rf
  p0t=p0*(rt+1.0d0)
  p0q=p0*rq
!
! electron pressure is pe
!
!
! ion pressure
!
  pi=(ni-nh2)*p0
  pion=pi
  pif=-nh2f*p0+(ni-nh2)*p0f
  pit=-nh2t*p0+(ni-nh2)*p0t
  piq=-nh2q*p0+(ni-nh2)*p0q
  pix=(1.0d0/a-nh2x)*p0+(ni-nh2)*p0*rx
!
! radiation pressure
!
  pr=cat*t4
!
! coulomb interaction terms
!
  pcoul =-p0*dcoul*nion/3.0d0
  pcoulf=pcoul*(      rf+dcoulf/dcoul+nionf/nion)
  pcoult=pcoul*(1.0d0+rt+dcoult/dcoul+niont/nion)
  pcoulq=pcoul*(      rq+dcoulq/dcoul+nionq/nion)
  pcoulx=pcoul*(      rx+dcoulx/dcoul+nionx/nion)
!
! quantum terms
!
  pqu =0.5d0*nion*p0*dquj
  pquf=pqu*(nionf/nion+rf      +dqujf/dquj)
  pqut=pqu*(niont/nion+rt+1.0d0+dqujt/dquj)
  pquq=pqu*(nionq/nion+rq      +dqujq/dquj)
  pqux=pqu*(nionx/nion+rx      +dqujx/dquj)
!
! anharmonic terms
!
  panhcor =-nion *p0*anhcor/1.5d0
  panhcorf=-nionf*p0*anhcor/1.5d0+panhcor*       rf -nion*p0*anhcorf/1.5d0
  panhcort=-niont*p0*anhcor/1.5d0+panhcor*(1.0d0+rt)-nion*p0*anhcort/1.5d0
  panhcorq=-nionq*p0*anhcor/1.5d0+panhcor*       rq -nion*p0*anhcorq/1.5d0
  panhcorx=-nionx*p0*anhcor/1.5d0+panhcor*       rx -nion*p0*anhcorx/1.5d0
!
! hard sphere terms
!
  ppi =p0*dpi* next
  ppif=p0*dpi*(nextf+ rf       *next)
  ppit=p0*dpi*(nextt+(rt+1.0d0)*next)
  ppiq=p0*dpi*(nextq+ rq       *next)+next*p0*dpiq
  ppix=p0*dpi*(nextx+ rx       *next)
!
! polarization terms
!
  ne2bnd =(ne0+ne)*(ne0-ne)
  ne2bndf=-2.0d0* ne*nef
  ne2bndt=-2.0d0* ne*net
  ne2bndq=-2.0d0* ne*neq
  ne2bndx=-2.0d0*(ne*nex-ne0*zmax/a)
  ppola=0.5d0*dc*p0*rho
  ppol =ne2bnd*ppola
  ppolf=ppol*(dcf+2.0d0*rf      )+ppola*ne2bndf
  ppolt=ppol*(dct+2.0d0*rt+1.0d0)+ppola*ne2bndt
  ppolq=ppol*(dcq+2.0d0*rq      )+ppola*ne2bndq
  ppolx=2.0d0*ppol*rx+0.5d0*p0*rho*(ne2bndx*dc+ne2bnd*dcx)
!
! add contributions to pressure
!
  pgas=pe+ppos+pi+pcoul+ppi+pqu+ppol+panhcor
  p=pgas+pr
  pl=log_cr(p)
!
! partial derivatives of ln p
!
  pf=(pe*pef+ppos*pposf+pif         +pcoulf+ppif+pquf+ppolf+panhcorf)/p
  pt=(pe*pet+ppos*ppost+pit+pr*4.0d0+pcoult+ppit+pqut+ppolt+panhcort)/p
  pq=(                  piq         +pcoulq+ppiq+pquq+ppolq+panhcorq)/p
  px=(                  pix         +pcoulx+ppix+pqux+ppolx+panhcorx)/p
!
! calculate partial pressures for diffusion equations, in linear approximation
!
  pion =pi +ppi +pqu +ppol
  pionf=pif+ppif+pquf+ppolf
  piont=pit+ppit+pqut+ppolt
  pionq=piq+ppiq+pquq+ppolq
  pionx=pix+ppix+pqux+ppolx
  nimnh2 =ni-nh2
  x1mnh2 =x(1)/a(1)-nh2
  pip (1)=x1mnh2*pion /nimnh2
  pipf(1)=(x1mnh2*pionf-nh2f*pion+x1mnh2*pion*nh2f/nimnh2)/nimnh2
  pipt(1)=(x1mnh2*piont-nh2t*pion+x1mnh2*pion*nh2t/nimnh2)/nimnh2
  pipq(1)=(x1mnh2*pionq-nh2q*pion+x1mnh2*pion*nh2q/nimnh2)/nimnh2
  do n=1,ncomp
    nimnh2xn=1.0d0/a(n)-nh2x(n)
    pipx(1,n)=(x1mnh2*pionx(n)-nh2x(n)*pion-x1mnh2*pion*nimnh2xn/nimnh2)/nimnh2
  enddo
  pipx(1,1)=pipx(1,1)+pion/(nimnh2*a(1))
  do ispec=2,ncomp
    xionai=x(ispec)/a(ispec)
    pip (ispec)=xionai*pion /nimnh2
    pipf(ispec)=xionai*(pionf+pion*nh2f/nimnh2)/nimnh2
    pipt(ispec)=xionai*(piont+pion*nh2t/nimnh2)/nimnh2
    pipq(ispec)=xionai*(pionq+pion*nh2q/nimnh2)/nimnh2
    do n=1,ncomp
      nimnh2xn=1.0d0/a(n)-nh2x(n)
      pipx(ispec,n)=xionai*(pionx(n)-pion*nimnh2xn/nimnh2)/nimnh2
    enddo
    pipx(ispec,ispec)=pipx(ispec,ispec)+pion/(nimnh2*a(ispec))
  enddo
!
! calculate the entropy per unit mass of gas.
!
!
! radiation
!
  b =4.0d0*pr/p0
  bf=-b*rf
  bt=-b*(rt-3.0d0)
  bq=-b*rq
  bx=-b*rx
!
! electrons, excluding those from pair production
!
  sel =ne*se
  self=nef*se+ne*sef
  selt=net*se+ne*set
  selq=neq*se
!
! molecules, atoms and ions
!
  sions =(ni-nh2)*(1.5d0*tl-rl-5.57243429d0)                                          +nh2 *omegah2t
  sionsf=(ni-nh2)*(        -rf             )         -nh2f *(1.5d0*tl-rl-5.57243429d0)+nh2f*omegah2t
  sionst=(ni-nh2)*(1.5d0   -rt             )         -nh2t *(1.5d0*tl-rl-5.57243429d0)+nh2t*omegah2t+nh2*omegah2tt
  sionsq=(ni-nh2)*(        -rq             )         -nh2q *(1.5d0*tl-rl-5.57243429d0)+nh2q*omegah2t
  sionsx=(ni-nh2)*(        -rx             )+(1.0d0/a-nh2x)*(1.5d0*tl-rl-5.57243429d0)+nh2x*omegah2t
  do ispec=1,ncomp
    sions =sions +sis (ispec)
    sionsf=sionsf+sisf(ispec)
    sionst=sionst+sist(ispec)
    sionsq=sionsq+sisq(ispec)
    do n=1,ncomp
      sionsx(n)=sionsx(n)+sisx(ispec,n)
    enddo
  enddo
!
! hard sphere interactions
!
  spi =-next *hpi
  spif=-nextf*hpi
  spit=-nextt*hpi
  spiq=-nextq*hpi-next*hpiq
  spix=-nextx*hpi
!
! Coulomb interactions
!
  scoul=nion*(hcoul-dcoul)
  scoulf=nionf*(hcoul-dcoul)+nion*(hcoulf-dcoulf)
  scoult=niont*(hcoul-dcoul)+nion*(hcoult-dcoult)
  scoulq=nionq*(hcoul-dcoul)+nion*(hcoulq-dcoulq)
  scoulx=nionx*(hcoul-dcoul)+nion*(hcoulx-dcoulx)
!
! quantum corrections
!
  squ =-nion *(quj-dquj)
  squf=-nionf*(quj-dquj)-nion*(qujf-dqujf)
  squt=-niont*(quj-dquj)-nion*(qujt-dqujt)
  squq=-nionq*(quj-dquj)-nion*(qujq-dqujq)
  squx=-nionx*(quj-dquj)-nion*(qujx-dqujx)
!
! anharmonic term has zero entropy because it is independent of temperature
! 
  sanhcor=0.0d0
!
! add some of the contributions to entropy
!
  s =cr*(sel   +sions +b +scoul +spi +squ )
  sf=cr*(self  +sionsf+bf+scoulf+spif+squf)
  st=cr*(selt  +sionst+bt+scoult+spit+squt)
  sq=cr*(selq  +sionsq+bq+scoulq+spiq+squq)
  sx=cr*(se*nex+sionsx+bx+scoulx+spix+squx)
!
! polarization 
!
  spol =-da*ppol/(dc*rho*t)
  spolf=-spol*(rf+dcf      )-da*ppolf/(dc*rho*t)
  spolt=-spol*(rt+dct+1.0d0)-da*ppolt/(dc*rho*t)
  spolq=-spol*(rq+dcq      )-da*ppolq/(dc*rho*t)
  spolx=-spol*(rx+dcx/dc   )-da*ppolx/(dc*rho*t)-ppol*dax/(dc*rho*t)
  s =s +spol
  sf=sf+spolf
  st=st+spolt
  sq=sq+spolq
  sx=sx+spolx
!
! add contribution from electron-positron pairs to entropy per unit mass
!
  spairs1=cr*posnod/rho
  spairs =spairs1*(se+spos)
  spairsf=spairs*(posnodf-rf)+spairs1*(sef+sposf)
  spairst=spairs*(posnodt-rt)+spairs1*(set+spost)
  spairsq=spairs*(       -rq)
  s =s +spairs
  sf=sf+spairsf
  st=st+spairst
  sq=sq+spairsq
  sx=sx-spairs*rx
!
! internal energy
!
! bound state contribution, excluding H2 molecule
!
  uion =0.0d0
  uionf=0.0d0
  uiont=0.0d0
  uionq=0.0d0
  uionx=0.0d0
  do ispec=1,ncomp
    uion =uion +suions (ispec)
    uionf=uionf+suionsf(ispec)
    uiont=uiont+suionst(ispec)
    uionq=uionq+suionsq(ispec)
    do n=1,ncomp
      uionx(n)=uionx(n)+suionsx(ispec,n)
    enddo
  enddo
!
! Coulomb terms
!
  ucoul =-dcoul *nion
  ucoulf=-dcoulf*nion-dcoul*nionf
  ucoult=-dcoult*nion-dcoul*niont
  ucoulq=-dcoulq*nion-dcoul*nionq
  ucoulx=-dcoulx*nion-dcoul*nionx
!
! quantum terms
!
  uqu =nion *dquj
  uquf=nionf*dquj+nion*dqujf
  uqut=niont*dquj+nion*dqujt
  uquq=nionq*dquj+nion*dqujq
  uqux=nionx*dquj+nion*dqujx
!
! anharmonic terms
!
  uanhcor =-anhcor *nion
  uanhcorf=-anhcorf*nion-anhcor*nionf
  uanhcort=-anhcort*nion-anhcor*niont
  uanhcorq=-anhcorq*nion-anhcor*nionq
  uanhcorx=-anhcorx*nion-anhcor*nionx
!
! add terms
!
  u =  crt*(ue *ne       +1.5d0*(ni-nh2)      +0.75d0*b +ti* uion       +ucoul +uqu +uanhcor )
  uf=  crt*(uef*ne+ue*nef-1.5d0*nh2f          +0.75d0*bf+ti* uionf      +ucoulf+uquf+uanhcorf)
  ut=u+crt*(uet*ne+ue*net-1.5d0*nh2t          +0.75d0*bt+ti*(uiont-uion)+ucoult+uqut+uanhcort)
  uq=  crt*(       ue*neq-1.5d0*nh2q          +0.75d0*bq+ti* uionq      +ucoulq+uquq+uanhcorq)
  ux=  crt*(ue*nex       +1.5d0*(1.0d0/a-nh2x)+0.75d0*bx+ti* uionx      +ucoulx+uqux+uanhcorx)
!
! add H2 molecule
!
  u =u +crt* nh2      *omegah2t
  uf=uf+crt* nh2f     *omegah2t
  ut=ut+crt*(nh2+nh2t)*omegah2t+crt*nh2*omegah2tt
  uq=uq+crt* nh2q     *omegah2t
  ux=ux+crt* nh2x     *omegah2t
!
! polarization term
!
  upol1=db*ti/(dc*rho)
  upol =upol1*ppol
  upolf=upol1*ppolf-upol*(dcf+rf      )
  upolt=upol1*ppolt-upol*(dct+rt+1.0d0)
  upolq=upol1*ppolq-upol*(dcq+rq      )
  upol2=0.5d0*rho*ti*crt
  upolx=upol*rx+upol2*(ne2bnd*dbx+ne2bndx*db)
  u =u +upol
  uf=uf+upolf
  ut=ut+upolt
  uq=uq+upolq
  ux=ux+upolx
!
! add contribution to internal energy per unit mass from electron-positron pairs
!
  upairs1=crt*posnod/rho
  upairs2=ue+upos+2.0d0/(ct*t)
  upairs =upairs1*upairs2
  upairsf=upairs1*((posnodf-rf)*upairs2+uef+uposf        )
  upairst=upairs1*((posnodt-rt)*upairs2+uet+upost+ue+upos)
  upairsq=upairs1*((       -rq)*upairs2                  )
  u =u +upairs
  uf=uf+upairsf
  ut=ut+upairst
  uq=uq+upairsq
  ux=ux-upairs*rx
!
! pressure ionization equation
! fx1 is (4*pi/3)*Na*sum (Xi*Rhsi**3/Ai)
! In hydrogenic approximation, hard sphere radii are given by rhsi/rhs(H)=sqrt(IP(H)/IPi)
!
  fx1=0.0d0
  do n=1,ncomp
    fx1=fx1+fx1coef(n)*x(n)
  enddo
  fx2=ni-nh2
  gq1=rho*fx1/fx2
!
! derivatives of ln gq1
!
  gq1f=rf+nh2f/fx2
  gq1t=rt+nh2t/fx2
  gq1q=rq+nh2q/fx2
  gq1x=rx+fx1coef/fx1-(1.0d0/a-nh2x)/fx2
  gq2=gq1*next+small
!
! derivatives of ln gq2
!
  gq2f=(gq1/gq2)*(gq1f*next+nextf)
  gq2t=(gq1/gq2)*(gq1t*next+nextt)
  gq2q=(gq1/gq2)*(gq1q*next+nextq)
  gq2x=(gq1/gq2)*(gq1x*next+nextx)
  gq=log_cr(gq2*(1.0d0+eq))-q
!
! derivatives of gq
!
  gqf=gq2f
  gqt=gq2t
  gqq=gq2q-1.0d0/(1.0d0+eq)
  gqx=gq2x
  qf=-gqf/gqq
  qt=-gqt/gqq
  qx=-gqx/gqq
!  
! terms needed to find q, fl given pressure as function of fl, tl and q
!
  pf_qt=pf
  pq_ft=pq
  gqf_qt=gqf
  gqq_ft=gqq
!
! here partial derivatives are calculated, removing dependence on q
!
  pf=pf+qf*pq
  pt=pt+qt*pq
  rf=rf+qf*rq
  rt=rt+qt*rq
  sf=sf+qf*sq
  st=st+qt*sq
  uf=uf+qf*uq
  ut=ut+qt*uq
  px=px+qx*pq
  rx=rx+qx*rq
  sx=sx+qx*sq
  ux=ux+qx*uq
  pipf=pipf+qf*pipq
  pipt=pipt+qt*pipq
  fcsf=fcsf+qf*fcsq
  fcst=fcst+qt*fcsq
  fneutf=fneutf+qf*fneutq
  fneutt=fneutt+qt*fneutq
  do n=1,ncomp
    pipx(:,n)=pipx(:,n)+qx(n)*pipq(:)
    fcsx(:,n)=fcsx(:,n)+qx(n)*fcsq(:)
    fneutx(:,n)=fneutx(:,n)+qx(n)*fneutq(:)
  enddo
  qq=pt*sf-pf*st
!
! specific heat at constant pressure
!
  cp=-qq/pf
!
! adiabatic gradient
!
  grada=sf/qq
!
! Gamma_1
!
  gamma=qq/(rt*sf-rf*st)
!
! cv is the specific heat at constant volume
!
  cv=(rf*st-rt*sf)/rf
!
! vt is partial derivative of ln V with respect to ln T at constant pressure (V = 1/rho)
!
  vt=-(rt*pf-rf*pt)/pf
  return
  end
!
! subroutine to find equation of state for solid phase
! Much of this routine is the same as for the fluid phase
!
  subroutine statefx_solid (fl,tl,q,xin, &
    pl,rl,u,p,rho,t,gamma,grada,cp,cv,vt,th,pgas,psi,s,gq,gqq,pf,pt,rf,rt,uf,ut,sf,st,px,rx,sx,ux,pip,pipf,pipt,pipx, &
    fcs,fcsf,fcst,fcsx,fneut,fneutf,fneutt,fneutx,coul,plam,pf_qt,pq_ft,gqf_qt,gqq_ft)
  use crlibm_lib, only: log_cr, exp_cr
  implicit none
  integer (kind=4) :: i,ics,iion,imax,is,j,k,l,ispec,it,n
  integer (kind=4), parameter :: ncomp=14,kionmax=26
  integer (kind=4), dimension(ncomp) :: izmax
  real (kind=8) :: fl,tl,q,pl,rl,u,p,rho,t,gamma,grada,cp,th,pgas,psi,s,gq,pf,pt,rf,rt,uf,ut,sf,st,coul,plam,gqq, &
    nh2,nh,nhplus,cr,cd,cb,ct,cat,ni,ne0,zmean,f,t4,wf,g,vf,vg,ug,fdf,vx,vy,vz,vw,re,ret,ref,pe, &
    pet,pef,qe,qet,qef,se,sef,set,ponrt,ue,uef,uet,ti,da,db,dc,dcf,dct,dcq,zmeanx,dmupol,dmupolf,dmupolt,dmupolq,dv, &
    dvf,dvt,dvq,eq,hpi,dpi,shpi,hpiq,dpiq,snmax,sdv,sdvf,sdvt,sdvq,vsmax,sumsir,vzsf0,vzst0,vzsq0, &
    vzsl,vzsl1,sisxa,vis,vissq,ff1,ff1c,ff1f,ff1t,ff1q,dh2kt,zetah2,zetah2t,zetah2tt,omegah2,omegah2t,omegah2tt, &
    omegah2l,f2,f2f,f2t,f2q,ne,nef,net,neq,x1ona1,aaa,aaaf,aaat,aaaq,bbb,bbbf,bbbt,bbbq,ccc,cccf,ccct,cccq,bbbb,eee, &
    ddd,hhh,dhbot,hhhf,hhht,hhhq,nhf,nht,nhq,nhplusf,nhplust,nhplusq,nh2f,nh2t,nh2q,nhtot,nhtotf,nhtott,nhtotq, &
    nhtotxn,next,nextf,nextt,nextq,nion,nionf,niont,nionq,nz53,nz53f,nz53t,nz53q,nzsqa,nzsqaf,nzsqat,nzsqaq, &
    psiplus,gpos,sspos,fpos,dpos,fplus,flplus,posnod,posnodf,posnodt,ppos,pposf,ppost,spos,sposf,spost,upos,uposf, &
    upost,dpsiminus,dpsiplus,dflplusdflminus,dflplusdtl,en,elnod,rq,coulf,coult,coulq,coull,plams,plaml,plamf,plamt, &
    plamq,coul32,hcoul,dcoul,ddcoul,coul13,x1,x2,ex1,ex2,debx1,debx2,vlx1,vlx2,dlx1,dlx2,ddlx1,ddlx2,quj,dquj,ddquj, &
    hcoulf,hcoult,hcoulq,dcoulf,dcoult,dcoulq,qujf,qujt,qujq,dqujf,dqujt,dqujq,anhcor,anhcorf,anhcort,anhcorq,crt,p0, &
    p0f,p0t,p0q,pi,pion,pif,pit,piq,pr,pcoul,pcoulf,pcoult,pcoulq,pqu,pquf,pqut,pquq,panhcor,panhcorf,panhcort, &
    panhcorq,ppi,ppif,ppit,ppiq,ne2bnd,ne2bndf,ne2bndt,ne2bndq,ppola,ppol,ppolf,ppolt,ppolq,pq,pionf,piont,nimnh2, &
    x1mnh2,pionq,nimnh2xn,xionai,b,bf,bt,bq,sel,self,selt,selq,sions,sionsf,sionst,sionsq,spi,spif,spit,spiq, &
    scoul,scoulf,scoult,scoulq,squ,squf,squt,squq,sanhcor,sq,spol,spolf,spolt,spolq,spairs1,spairs,spairsf,spairst, &
    spairsq,uion,uionf,uiont,uionq,ucoul,ucoulf,ucoult,ucoulq,uqu,uquf,uqut,uquq,uanhcor,uanhcorf,uanhcort,uanhcorq, &
    uq,upol1,upol,upolf,upolt,upolq,upol2,upairs1,upairs2,upairs,upairsf,upairst,upairsq,fx1,fx2,gq1,gq1f,gq1t,gq1q, &
    gq2,gq2f,gq2t,gq2q,gqf,gqt,qf,qt,qq,cv,vt,debfun,couls,scor,scorf,scorq,scort,gqtsol,wfpos, &
    psipos,pf_qt,pq_ft,gqf_qt,gqq_ft
  real (kind=8), dimension(ncomp) :: x,xin,px,rx,sx,ux,pip,pipf,pipt,fcs,fcsf,fcst,fneut,fneutf,fneutt,a,zmax, &
    fx1coef,dax,dbx,dcx,dvx,sdvx,vzsx0,ff1x,nex,aaax,bbbx,cccx,hhhx,nhx,nhplusx,nh2x,nextx,nionx,nz53x,nzsqax,pix, &
    coulx,plamx,hcoulx,dcoulx,qujx,dqujx,pcoulx,pqux,ppix,ne2bndx,ppolx,sionsx,spix,scoulx,squx,uionx,ucoulx,uqux, &
    upolx,gqx,qx,gq1x,gq2x,anhcorx,panhcorx,uanhcorx,select,selectf,selectt,selectq,sext,sextf,sextt,sextq,sion, &
    sionf,siont,sionq,sis,sisf,sist,sisq,suions,suionsf,suionst,suionsq,sz53,sz53f,sz53t,sz53q,szsq,szsqf,szsqt, &
    szsqq,pipq,fcsq,fneutq,scorx,bx,spolx,pionx
  real (kind=8), dimension(kionmax) :: z53,z2
  real (kind=8), dimension(ncomp,ncomp) :: pipx,fcsx,fneutx,selectx,sextx,sionx,sisx,suionsx,sz53x,szsqx
  real (kind=8), parameter :: third=1.0d0/3.0d0,small=1.0d-50 
  real (kind=8) :: d(3,3),ff(4),gg(4),c(48)
  real (kind=8), dimension (kionmax) :: vsf,vst,vsq
  real (kind=8), dimension (0:kionmax) :: vs,sir,spin,spinf,spint,spinq,vzs
  real (kind=8), dimension (kionmax,ncomp) :: vsx,sip
  real (kind=8), dimension (0:kionmax,ncomp) :: spinx,sgsw
!
! approximate atomic weights
!
!      data a/ 1.0d0,2.0d0,3.0d0,4.0d0,7.0d0,7.0d0,12.0d0,13.0d0,14.0d0,
!     1       16.0d0,20.0d0,24.0d0,28.0d0,56.0d0/
!
! accurate atomic weights from NIST
!
  data a/ 1.007825d0,2.014102d0,3.016029d0,4.002603d0,7.016005d0,7.016930d0,12.00000d0,13.00335d0,14.00307d0, &
          15.99491d0,19.99244d0,23.98504d0,27.97693d0,55.93494d0/
  data c/ 2.315472d0, 7.128660d0, 7.504998d0, 2.665350d0, 7.837752d0, &
         23.507934d0,23.311317d0, 7.987465d0, 9.215560d0,26.834068d0, &
         25.082745d0, 8.020509d0, 3.693280d0,10.333176d0, 9.168960d0, &
          2.668248d0, 2.315472d0, 6.748104d0, 6.564912d0, 2.132280d0, &
          7.837752d0,21.439740d0,19.080088d0, 5.478100d0, 9.215560d0, &
         23.551504d0,19.015888d0, 4.679944d0, 3.693280d0, 8.859868d0, &
          6.500712d0, 1.334124d0, 1.157736d0, 3.770676d0, 4.015224d0, &
          1.402284d0, 8.283420d0,26.184486d0,28.211372d0,10.310306d0, &
         14.755480d0,45.031658d0,46.909420d0,16.633242d0, 7.386560d0, &
         22.159680d0,22.438048d0, 7.664928d0/
  data ff(1),gg(1),cr,cd,cb,ct,cat /1.0d0,1.0d0,8.3143d+07,2.9218d+06,1.4406d+24,1.6863d-10,2.5215d-15/
  data fx1coef /0.86d0,0.43d0,0.12d0,0.088d0,0.49d0,0.22d0,0.095d0,0.088d0,0.056d0,0.054d0,0.022d0,0.085d0,0.066d0, &
    0.035d0/
  data izmax /1,1,2,2,3,4,6,6,7,8,10,12,14,26/
!
! ionization potentials in eV, measured from ground state of neutral atom
!
  data sip /  13.598d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              13.598d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              24.587d0,  79.003d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              24.587d0,  79.003d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
               5.392d0,  81.030d0, 203.481d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
               9.322d0,  27.533d0, 181.426d0, 399.139d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              11.260d0,  35.643d0,  83.530d0, 148.022d0, 540.102d0,1030.082d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              11.260d0,  35.643d0,  83.530d0, 148.022d0, 540.102d0,1030.082d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              14.534d0,  44.135d0,  91.583d0, 169.055d0,  266.95d0,  819.01d0, 1486.04d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              13.618d0,  48.734d0,  103.67d0,  181.08d0,  294.98d0,  433.09d0,  1172.4d0,  2043.8d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              21.565d0,  62.528d0,  125.98d0,  223.10d0,  349.32d0,  507.25d0,  714.52d0,  953.62d0, &
              2149.4d0,  3511.6d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
               7.646d0,  22.681d0, 102.824d0,  212.13d0,  353.40d0,  539.91d0,  764.86d0, 1030.78d0, &
              1358.8d0,  1726.3d0,  3488.1d0,  5451.0d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
            8.151683d0,24.49753d0,57.99053d0,103.1323d0,269.8997d0,475.1667d0,721.6473d0,1025.185d0, &
            1376.309d0,1777.683d0,2254.042d0,2777.463d0,5215.091d0,7888.272d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0,   0.000d0, &
               0.000d0,   0.000d0, &
              7.9024d0, 24.1081d0, 54.7595d0,109.5605d0,184.5709d0,283.6343d0,408.6104d0,559.6703d0, &
            793.2565d0,1055.359d0,1345.606d0,1676.396d0,2037.438d0,2429.600d0,2886.606d0,3375.882d0, &
            4638.041d0,6000.007d0,7469.220d0,9044.811d0,10733.48d0,12532.49d0,14491.06d0,16536.80d0, &
            25364.85d0,34642.54d0/
!
! ground state statistical weights for atomic and ionic species
!
  data sgsw /2.0d0,1.0d0,25*0.0d0, &
             2.0d0,1.0d0,25*0.0d0, &
             1.0d0,2.0d0,1.0d0,24*0.0d0, &
             1.0d0,2.0d0,1.0d0,24*0.0d0, &
             2.0d0,1.0d0,2.0d0,1.0d0,23*0.0d0, &
             1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,22*0.0d0, &
             9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,20*0.0d0, &
             9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,20*0.0d0, &
             4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,19*0.0d0, &
             9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,18*0.0d0, &
             1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,16*0.0d0, &
             1.0d0,2.0d0,1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,14*0.0d0, &
             9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0,12*0.0d0, &
            25.0d0,30.0d0,25.0d0,6.0d0,25.0d0,28.0d0,21.0d0,10.0d0,1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0, &
             1.0d0,6.0d0,9.0d0,4.0d0,9.0d0,6.0d0,1.0d0,2.0d0,1.0d0,2.0d0,1.0d0/
  data zmax /1.0d0,1.0d0,2.0d0,2.0d0,3.0d0,4.0d0,6.0d0,6.0d0,7.0d0,8.0d0,10.0d0,12.0d0,14.0d0,26.0d0/
!
! the following numbers are Z**(5/3) and Z**2 for Z = 1 to 26 for use in the
! Coulomb corrections and quantum corrections respectively.
!
  data z53 / 1.000d0,3.175d0,6.240d0,10.08d0,14.62d0,19.81d0,25.62d0,32.00d0,38.94d0,46.42d0,54.41d0,62.90d0, &
             71.87d0,81.32d0,91.23d0,101.6d0,112.4d0,123.6d0,135.3d0,147.4d0,159.8d0,172.7d0,186.0d0,199.7d0, &
             213.7d0,228.2d0/
  data z2  / 1.000d0,4.000d0,9.000d0,16.00d0,25.00d0,36.00d0,49.00d0,64.00d0,81.00d0,100.0d0,121.0d0,144.0d0, &
             169.0d0,196.0d0,225.0d0,256.0d0,289.0d0,324.0d0,361.0d0,400.0d0,441.0d0,484.0d0,529.0d0,576.0d0, &
             625.0d0,676.0d0/
  do ispec=1,ncomp
    x(ispec)=max(xin(ispec),0.0d0)
  enddo
  ni=0.0d0
  ne0=0.0d0
  do ispec=1,ncomp
    ni=ni+x(ispec)/a(ispec)
    ne0=ne0+x(ispec)*zmax(ispec)/a(ispec)
  enddo
  zmean=ne0/ni
!
! EFF section to find electron number density, etc
!
  f=exp_cr(fl)
  t=exp_cr(tl)
  t4=t**4
  call eff (f,t,fl,tl,elnod,ref,ret,pe,pef,pet,se,sef,set,ue,uef,uet,wf,psi)
!
! th used in weak screening
!
  th=ref/wf
!
! inverse of temperature in eV
!
  ti=11605.0d0/t
!
! modified EFF non-ideal contribution related to 'polarization' i.e. electric microfield
!
  da=0.0862d0/zmean**3
  db=da*270.0d0*zmean**2
  dc=da+db*ti
  dcf=0.0d0
  dct=-db*ti/dc
  dcq=0.0d0
  do n=1,ncomp
    zmeanx=(zmax(n)/ne0-1.0d0/ni)/a(n)
    dax(n)=-3.0d0*da*zmeanx
    dbx(n)=-      db*zmeanx
    dcx(n)=dax(n)+ti*dbx(n)
  enddo
  dmupol =dc*elnod
  dmupolf=dmupol*(ref+dcf)
  dmupolt=dmupol*(ret+dct)
  dmupolq=0.0d0
  dv=-psi+dmupol
  dvf=-wf+dmupolf
  dvt=dmupolt
  dvq=0.0d0
  dvx=dcx*elnod
!
! factors needed for pressure ionization. This is determined from a simplified hard sphere free energy term
! HS free energy is based on that in Graboske, Harwood & Rogers 1969 Phys. Rev., 186, 210
!
  eq=exp_cr(q)
  hpi=log_cr(1.0d0+eq)+eq*(3.0d0+1.5d0*eq)
  dpi=eq*(4.0d0+eq*(6.0d0+eq*3.0d0))
  shpi=eq**2*(10.0d0+eq*(18.0d0+eq*9.0d0))
  hpiq=dpi/(1.0d0+eq)
  dpiq=(dpi+shpi)/(1.0d0+eq)
!
! ionization of all species, except H which is treated separately because of H2 molecule
!
  do ispec=2,ncomp
    imax=izmax(ispec)
    snmax=x(ispec)/a(ispec)
    sdv=0.0d0
    sdvf=0.0d0
    sdvt=0.0d0
    sdvq=0.0d0
    sdvx=0.0d0
    vs(0)=0.0d0
    do is=1,imax
      sdv=sdv+dv
      sdvf=sdvf+dvf
      sdvt=sdvt+dvt
      sdvq=sdvq+dvq
      sdvx=sdvx+dvx
      vs (is)=sdv -sip(is,ispec)*ti
      vsf(is)=sdvf
      vst(is)=sdvt+sip(is,ispec)*ti
      vsq(is)=sdvq
      do n=1,ncomp
        vsx(is,n)=sdvx(n)
      enddo
    enddo
    vs (imax)=vs (imax)+hpi +dpi
    vsq(imax)=vsq(imax)+hpiq+dpiq
!        
! find commonest ion.
!
    vsmax=vs(0)
    ics=0
    do is=1,imax
      if (vs(is).gt.vsmax) then
        vsmax=vs(is)
        ics=is
      endif
    enddo
!
! calculate ionic ratios relative to commonest ion.
!
    sumsir=0.0d0
    do is=0,imax
      sir(is)=exp_cr(vs(is)-vs(ics))*sgsw(is,ispec)/sgsw(ics,ispec)
      sumsir=sumsir+sir(is)
   enddo
!
! vzs contains fraction in each ionization state
!
    do is=0,imax
      vzs(is)=sir(is)/sumsir
      spin(is)=vzs(is)*snmax
    enddo
!
! derivatives of ln vzs(0)
!
    vzsf0=0.0d0
    vzst0=0.0d0
    vzsq0=0.0d0
    do n=1,ncomp
      vzsx0(n)=0.0d0
    enddo
    do iion=1,imax
      vzsf0=vzsf0-vsf(iion)*vzs(iion)
      vzst0=vzst0-vst(iion)*vzs(iion)
      vzsq0=vzsq0-vsq(iion)*vzs(iion)
      do n=1,ncomp
        vzsx0(n)=vzsx0(n)-vsx(iion,n)*vzs(iion)
      enddo
    enddo
!
! derivatives of spin(0)
!
    spinf(0)=spin(0)*vzsf0
    spint(0)=spin(0)*vzst0
    spinq(0)=spin(0)*vzsq0
    do n=1,ncomp
      spinx(0,n)=spin(0)*vzsx0(n)
    enddo
    spinx(0,ispec)=spinx(0,ispec)+vzs(0)/a(ispec)
!
! calculate quantities needed to find density, pressure, entropy, internal energy
! and their derivatives
!
    vzsl=log_cr(vzs(0)/sgsw(0,ispec)+small)
    vzsl1=vzsl+vzs(0)/(vzs(0)+sgsw(0,ispec)*small)
!
! sum over ionization states for each species
!
!
! select is used to find number of free electrons from each species
!
    select(ispec)=0.0d0
    selectf(ispec)=0.0d0
    selectt(ispec)=0.0d0
    selectq(ispec)=0.0d0
!
! sion is used to find number of ions (excludes neutrals)
!
    sion (ispec)=0.0d0
    sionf(ispec)=0.0d0
    siont(ispec)=0.0d0
    sionq(ispec)=0.0d0
!
! suions is used to find contribution of bound states to the internal energy density
!
    suions (ispec)=spin (0)*(-sip(imax,ispec))
    suionsf(ispec)=spinf(0)*(-sip(imax,ispec))
    suionst(ispec)=spint(0)*(-sip(imax,ispec))
    suionsq(ispec)=spinq(0)*(-sip(imax,ispec))
!
! sz53 and szs are sums of z^5/3 and Z^2 needed to find Coulomb Gamma and the plasma parameter
! respectively
!
    sz53 (ispec)=0.0d0
    sz53f(ispec)=0.0d0
    sz53t(ispec)=0.0d0
    sz53q(ispec)=0.0d0
    szsq (ispec)=0.0d0
    szsqf(ispec)=0.0d0
    szsqt(ispec)=0.0d0
    szsqq(ispec)=0.0d0
!
! sis is needed to find entropy
!
    sis (ispec)=snmax*(1.5d0*log_cr(a(ispec))-log_cr(snmax+small))-spin (0)*vzsl
    sisf(ispec)=-spinf(0)*vzsl1
    sist(ispec)=-spint(0)*vzsl1
    sisq(ispec)=-spinq(0)*vzsl1
    sisxa=spin(0)*vzs(0)/(vzs(0)+small*sgsw(0,ispec))
    do n=1,ncomp
      selectx(ispec,n)=0.0d0
      sionx(ispec,n)=0.0d0
      suionsx(ispec,n)=spinx(0,n)*(-sip(imax,ispec))
      sz53x(ispec,n)=0.0d0
      szsqx(ispec,n)=0.0d0
      sisx(ispec,n)=-spinx(0,n)*vzsl-vzsx0(n)*sisxa
    enddo
    sisx(ispec,ispec)=sisx(ispec,ispec)+(1.5d0*log_cr(a(ispec))-log_cr(snmax+small)-snmax/(snmax+small))/a(ispec)
    vis=0.0d0
    do is=1,imax
      vis=vis+1.0d0
      vissq=vis**2
      spinf(is)=spin(is)*(vzsf0+vsf(is))
      spint(is)=spin(is)*(vzst0+vst(is))
      spinq(is)=spin(is)*(vzsq0+vsq(is))
      select(ispec)=select(ispec)+vis*spin (is)
      selectf(ispec)=selectf(ispec)+vis*spinf(is)
      selectt(ispec)=selectt(ispec)+vis*spint(is)
      selectq(ispec)=selectq(ispec)+vis*spinq(is)
      sion (ispec)=sion (ispec)+spin (is)
      sionf(ispec)=sionf(ispec)+spinf(is)
      siont(ispec)=siont(ispec)+spint(is)
      sionq(ispec)=sionq(ispec)+spinq(is)
      vzsl=log_cr(vzs(is)/sgsw(is,ispec)+small)
      vzsl1=vzsl+vzs(is)/(vzs(is)+sgsw(is,ispec)*small)
      sis (ispec)=sis (ispec)-spin (is)*vzsl
      sisf(ispec)=sisf(ispec)-spinf(is)*vzsl1
      sist(ispec)=sist(ispec)-spint(is)*vzsl1
      sisq(ispec)=sisq(ispec)-spinq(is)*vzsl1
      sz53 (ispec)=sz53 (ispec)+z53(is)*spin (is)
      sz53f(ispec)=sz53f(ispec)+z53(is)*spinf(is)
      sz53t(ispec)=sz53t(ispec)+z53(is)*spint(is)
      sz53q(ispec)=sz53q(ispec)+z53(is)*spinq(is)
      szsq (ispec)=szsq (ispec)+z2(is)*spin (is)
      szsqf(ispec)=szsqf(ispec)+z2(is)*spinf(is)
      szsqt(ispec)=szsqt(ispec)+z2(is)*spint(is)
      szsqq(ispec)=szsqq(ispec)+z2(is)*spinq(is)
      suions (ispec)=suions (ispec)+spin (is)*(sip(is,ispec)-sip(imax,ispec))
      suionsf(ispec)=suionsf(ispec)+spinf(is)*(sip(is,ispec)-sip(imax,ispec))
      suionst(ispec)=suionst(ispec)+spint(is)*(sip(is,ispec)-sip(imax,ispec))
      suionsq(ispec)=suionsq(ispec)+spinq(is)*(sip(is,ispec)-sip(imax,ispec))
      sisxa=spin(is)*vzs(is)/(vzs(is)+small*sgsw(is,ispec))
      do n=1,ncomp
        spinx(is,n)=spin(is)*(vzsx0(n)+vsx(is,n))
        if (n.eq.ispec) spinx(is,n)=spinx(is,n)+vzs(is)/a(n)
        selectx(ispec,n)=selectx(ispec,n)+vis*spinx(is,n)
        sionx(ispec,n)=sionx(ispec,n)+spinx(is,n)
        sisx(ispec,n)=sisx(ispec,n)-spinx(is,n)*vzsl-sisxa*(vsx(is,n)+vzsx0(n))
        sz53x(ispec,n)=sz53x(ispec,n)+z53(is)*spinx(is,n)
        szsqx(ispec,n)=szsqx(ispec,n)+z2(is)*spinx(is,n)
        suionsx(ispec,n)=suionsx(ispec,n)+spinx(is,n)*(sip(is,ispec)-sip(imax,ispec))
      enddo
    enddo
!
! sext is used to find number of extended particles (i.e. not electrons or bare nuclei)
!
    sext (ispec)=0.0d0
    sextf(ispec)=0.0d0
    sextt(ispec)=0.0d0
    sextq(ispec)=0.0d0
    do n=1,ncomp
      sextx(ispec,n)=0.0d0
    enddo
    do is=0,imax-1
      sext (ispec)=sext (ispec)+spin (is)
      sextf(ispec)=sextf(ispec)+spinf(is)
      sextt(ispec)=sextt(ispec)+spint(is)
      sextq(ispec)=sextq(ispec)+spinq(is)
      do n=1,ncomp
        sextx(ispec,n)=sextx(ispec,n)+spinx(is,n)
      enddo
    enddo
!
! fcs = select / X_i/A_i is the mean charge of all particles of an individual species
!
    fcs (ispec)=select (ispec)/snmax
    fcsf(ispec)=selectf(ispec)/snmax
    fcst(ispec)=selectt(ispec)/snmax
    fcsq(ispec)=selectq(ispec)/snmax
    do n=1,ncomp
      fcsx(ispec,n)=selectx(ispec,n)/snmax
    enddo
!
! fneut = fraction of particles of an individual species that are neutral
!
    fneut (ispec)=vzs(0)
    fneutf(ispec)=vzs(0)*vzsf0
    fneutt(ispec)=vzs(0)*vzst0
    fneutq(ispec)=vzs(0)*vzsq0
    do n=1,ncomp
      fneutx(ispec,n)=vzs(0)*vzsx0(n)
    enddo
  enddo
!
! treatment of H to include H2 molecules
!
  snmax=x(1)/a(1)
  vs(0)=0.0d0
  vs (1)=dv -sip(1,1)*ti
  vsf(1)=dvf
  vst(1)=dvt+sip(1,1)*ti
  vsq(1)=dvq
  vs (1)=vs (1)+hpi +dpi
  vsq(1)=vsq(1)+hpiq+dpiq
  do n=1,ncomp
    vsx(1,n)=dvx(n)
  enddo
!        
! find commonest ion.
!
  vsmax=vs(0)
  ics=0
  if (vs(1).gt.vsmax) then
    vsmax=vs(1)
    ics=1
  endif
!
! calculate ionic ratios relative to commonest ion.
!
  sumsir=0.0d0
  do is=0,1
    sir(is)=exp_cr(vs(is)-vs(ics))*sgsw(is,1)/sgsw(ics,1)
    sumsir=sumsir+sir(is)
  enddo
  do is=0,1
    vzs(is)=sir(is)/sumsir
    spin(is)=vzs(is)*snmax
  enddo
  vzsf0=-vsf(1)*vzs(1)
  vzst0=-vst(1)*vzs(1)
  vzsq0=-vsq(1)*vzs(1)
  spinf(0)=spin(0)*vzsf0
  spint(0)=spin(0)*vzst0
  spinq(0)=spin(0)*vzsq0
  spinf(1)=spin(1)*(vzsf0+vsf(1))
  spint(1)=spin(1)*(vzst0+vst(1))
  spinq(1)=spin(1)*(vzsq0+vsq(1))
  do n=1,ncomp
    vzsx0(n)=-vsx(1,n)*vzs(1)
    spinx(0,n)=spin(0)*vzsx0(n)
    spinx(1,n)=spin(1)*(vzsx0(n)+vsx(1,n))
  enddo
  spinx(0,1)=spinx(0,1)+vzs(0)/a(1)
  spinx(1,1)=spinx(1,1)+vzs(1)/a(1)
!
! ff1 = NH_plus/(NH_plus+NH)
!
  ff1 =spin(1)/(spin(0)+spin(1))
  ff1c=spin(0)/(spin(0)+spin(1))
  ff1f=ff1c*ff1*vsf(1)
  ff1t=ff1c*ff1*vst(1)
  ff1q=ff1c*ff1*vsq(1)
  do n=1,ncomp
    ff1x(n)=ff1c*ff1*vsx(1,n)
  enddo
!
! H2 partition function from Pols et al (1995), Webbink (1975)
! with modification at low temperature
!
  dh2kt=4.48d0*ti
  zetah2=1.0d0-(1.0d0+dh2kt)*exp_cr(-dh2kt)
  zetah2t=-dh2kt**2*exp_cr(-dh2kt)/zetah2
  zetah2tt=zetah2t*(dh2kt-2.0d0-zetah2t)
  if (ti.le.10.0d0) then
    omegah2  =6608.8d0*zetah2*exp_cr(ti*(0.448d0+ti*(-0.02440d0+ti*6.163d-04)))/dh2kt**2.5d0
    omegah2t =2.5d0-ti*(0.448d0+ti*(-0.04880d0+ti*1.849d-03))+zetah2t
    omegah2tt=ti*(0.448d0+ti*(-0.0976d0+ti*5.547d-03))+zetah2tt
  else
    omegah2l =(33.4682d0+(-190.382d0      +503.94d0      /ti)/ti)/ti
    omegah2  =exp_cr(omegah2l)
    omegah2t =(33.4682d0+(-190.382d0*2.0d0+503.94d0*3.0d0/ti)/ti)/ti
    omegah2tt=(33.4682d0+(-190.382d0*4.0d0+503.94d0*9.0d0/ti)/ti)/ti
  endif
!
! f2 = NH_2*Ne/NH^2
!
  f2=exp_cr(dh2kt+7.726d0-1.5d0*tl+log_cr(omegah2*elnod))
  f2f=f2*ref
  f2t=f2*(-dh2kt-1.5d0+omegah2t+ret)
  f2q=0.0d0
!
! ne is the inverse of the molecular weight per electron, excluding contribution from H
! Contributions from H are added later
!
  ne=0.0d0
  nef=0.0d0
  net=0.0d0
  neq=0.0d0
  nex=0.0d0
  do ispec=2,ncomp
    ne =ne +select(ispec)
    nef=nef+selectf(ispec)
    net=net+selectt(ispec)
    neq=neq+selectq(ispec)
    do n=1,ncomp
      nex(n)=nex(n)+selectx(ispec,n)
    enddo
  enddo
!
! coefficients of quadratic equation for h and their derivatives
! h = NH_plus/(ff1*X(1)/A(1)) where X(1) is H mass fraction
!
  x1ona1=x(1)/a(1)
  aaa =(ff1 +2.0d0*f2* ff1c**2                   )*x1ona1
  aaaf=(ff1f+2.0d0*f2f*ff1c**2-4.0d0*f2*ff1c*ff1f)*x1ona1
  aaat=(ff1t+2.0d0*f2t*ff1c**2-4.0d0*f2*ff1c*ff1t)*x1ona1
  aaaq=(ff1q+2.0d0*f2q*ff1c**2-4.0d0*f2*ff1c*ff1q)*x1ona1
  aaax=ff1x*(1.0d0-4.0d0*f2*ff1c)*x1ona1
  bbb =ne -x1ona1*ff1
  bbbf=nef-x1ona1*ff1f
  bbbt=net-x1ona1*ff1t
  bbbq=neq-x1ona1*ff1q
  bbbx=nex-x1ona1*ff1x
  ccc =-ne
  cccf=-nef
  ccct=-net
  cccq=-neq
  cccx=-nex
  aaax(1)=aaax(1)+(ff1+2.0d0*f2*ff1c**2)/a(1)
  bbbx(1)=bbbx(1)-ff1/a(1)
  bbbb=0.5d0*bbb
  eee=bbbb/aaa
  ddd=sqrt(eee**2-ccc/aaa)
  if (bbbb.le.0.0d0) then
    hhh=ddd-eee
  else
    hhh=(-ccc/aaa)/(ddd+eee)
  endif
  dhbot=2.0d0*hhh*aaa+bbb
  hhhf=-(hhh**2*aaaf+hhh*bbbf+cccf)/dhbot
  hhht=-(hhh**2*aaat+hhh*bbbt+ccct)/dhbot
  hhhq=-(hhh**2*aaaq+hhh*bbbq+cccq)/dhbot
  hhhx=-(hhh**2*aaax+hhh*bbbx+cccx)/dhbot
!
! nh is number of hydrogen atoms per unit mass divided by Avogadro's number (NA)
! nhplus is number of protons per unit mass divided by NA
! nh2 is number of hydrogen molecules per unit mass divided by NA
!
  nh = hhh *ff1c          *x1ona1
  nhf=(hhhf*ff1c-hhh*ff1f)*x1ona1
  nht=(hhht*ff1c-hhh*ff1t)*x1ona1
  nhq=(hhhq*ff1c-hhh*ff1q)*x1ona1
  nhplus = hhh *ff1          *x1ona1
  nhplusf=(hhhf*ff1+hhh*ff1f)*x1ona1
  nhplust=(hhht*ff1+hhh*ff1t)*x1ona1
  nhplusq=(hhhq*ff1+hhh*ff1q)*x1ona1
  nh2=nh**2*f2/(nhplus+ne)
  nh2f=-0.5d0*(nhf+nhplusf)
  nh2t=-0.5d0*(nht+nhplust)
  nh2q=-0.5d0*(nhq+nhplusq)
  do n=1,ncomp
    nhx(n)=(hhhx(n)*ff1c-hhh*ff1x(n))*x1ona1
    nhplusx(n)=(hhhx(n)*ff1+hhh*ff1x(n))*x1ona1
    nh2x(n)=-0.5d0*(nhx(n)+nhplusx(n))
  enddo
  nhx(1)=nhx(1)+hhh*ff1c/a(1)
  nhplusx(1)=nhplusx(1)+hhh*ff1/a(1)
  nh2x(1)=0.5d0/a(1)-0.5d0*(nhx(1)+nhplusx(1))
!
! mean Z and neutral fraction for H
!
  nhtot =nh2 +nh +nhplus
  nhtotf=nh2f+nhf+nhplusf
  nhtott=nh2t+nht+nhplust
  nhtotq=nh2q+nhq+nhplusq
  fcs (1)=nhplus /nhtot
  fcsf(1)=(nhplusf-nhplus*nhtotf/nhtot)/nhtot
  fcst(1)=(nhplust-nhplus*nhtott/nhtot)/nhtot
  fcsq(1)=(nhplusq-nhplus*nhtotq/nhtot)/nhtot
  fneut (1)=(nh2 +nh )/nhtot
  fneutf(1)=(nh2f+nhf-(nh2+nh)*nhtotf/nhtot)/nhtot
  fneutt(1)=(nh2t+nht-(nh2+nh)*nhtott/nhtot)/nhtot
  fneutq(1)=(nh2q+nhq-(nh2+nh)*nhtotq/nhtot)/nhtot
  do n=1,ncomp
    nhtotxn=nh2x(n)+nhx(n)+nhplusx(n)
    fcsx  (1,n)=(nhplusx(n)    -nhplus  *nhtotxn/nhtot)/nhtot
    fneutx(1,n)=(nh2x(n)+nhx(n)-(nh2+nh)*nhtotxn/nhtot)/nhtot
  enddo
  select(1)=nhplus
  selectf(1)=nhplusf
  selectt(1)=nhplust
  selectq(1)=nhplusq
  sext (1)=nh +nh2
  sextf(1)=nhf+nh2f
  sextt(1)=nht+nh2t
  sextq(1)=nhq+nh2q
  sion (1)=nhplus
  sionf(1)=nhplusf
  siont(1)=nhplust
  sionq(1)=nhplusq
  sz53 (1)=nhplus
  sz53f(1)=nhplusf
  sz53t(1)=nhplust
  sz53q(1)=nhplusq
  szsq (1)=nhplus
  szsqf(1)=nhplusf
  szsqt(1)=nhplust
  szsqq(1)=nhplusq
  suions (1)=-nh2 *31.676d0-nh *13.598d0
  suionsf(1)=-nh2f*31.676d0-nhf*13.598d0
  suionst(1)=-nh2t*31.676d0-nht*13.598d0
  suionsq(1)=-nh2q*31.676d0-nhq*13.598d0
  sis (1)=-nh2*log_cr(0.35355d0*nh2/omegah2+small)-nh*log_cr(nh/sgsw(0,1)+small)-nhplus*log_cr(nhplus/sgsw(1,1)+small) &
    +1.5d0*(nh+nhplus+nh2)*log_cr(a(1))
  sisf(1)=-nh2f   *(log_cr(0.35355d0*nh2/omegah2  +small)+nh2   /(nh2   +2.82845d0*small*omegah2  )) &
          -nhf    *(log_cr(          nh /sgsw(0,1)+small)+nh    /(nh    +          small*sgsw(0,1))) &
          -nhplusf*(log_cr(       nhplus/sgsw(1,1)+small)+nhplus/(nhplus+          small*sgsw(1,1))) &
          +1.5d0*(nhf+nhplusf+nh2f)*log_cr(a(1))
  sist(1)=-nh2t   *(log_cr(0.35355d0*nh2/omegah2  +small)+nh2   /(nh2   +2.82845d0*small*omegah2  )) &
          -nht    *(log_cr(          nh /sgsw(0,1)+small)+nh    /(nh    +          small*sgsw(0,1))) &
          -nhplust*(log_cr(       nhplus/sgsw(1,1)+small)+nhplus/(nhplus+          small*sgsw(1,1))) &
          +nh2*(nh2/(nh2+2.82845d0*small*omegah2))*omegah2t+1.5d0*(nht+nhplust+nh2t)*log_cr(a(1))
  sisq(1)=-nh2q   *(log_cr(0.35355d0*nh2/omegah2  +small)+nh2   /(nh2   +2.82845d0*small*omegah2  )) &
          -nhq    *(log_cr(          nh /sgsw(0,1)+small)+nh    /(nh    +          small*sgsw(0,1))) &
          -nhplusq*(log_cr(       nhplus/sgsw(1,1)+small)+nhplus/(nhplus+          small*sgsw(1,1))) &
          +1.5d0*(nhq+nhplusq+nh2q)*log_cr(a(1))
  do n=1,ncomp
    selectx(1,n)=nhplusx(n)
    sextx(1,n)=nhx(n)+nh2x(n)
    sionx(1,n)=nhplusx(n)
    sz53x(1,n)=nhplusx(n)
    szsqx(1,n)=nhplusx(n)
    suionsx(1,n)=-nh2x(n)*31.676d0-nhx(n)*13.598d0
    sisx(1,n)=-nh2x(n)   *(log_cr(0.35355d0*nh2/omegah2  +small)+nh2   /(nh2   +2.82845d0*small*omegah2  )) &
              -nhx(n)    *(log_cr(          nh /sgsw(0,1)+small)+nh    /(nh    +          small*sgsw(0,1))) &
              -nhplusx(n)*(log_cr(       nhplus/sgsw(1,1)+small)+nhplus/(nhplus+          small*sgsw(1,1))) &
              +1.5d0*(nhx(n)+nhplusx(n)+nh2x(n))*log_cr(a(1))
  enddo
!
! sums over species
!
  ne =0.0d0
  nef=0.0d0
  net=0.0d0
  neq=0.0d0
  next =0.0d0
  nextf=0.0d0
  nextt=0.0d0
  nextq=0.0d0
  nion =0.0d0
  nionf=0.0d0
  niont=0.0d0
  nionq=0.0d0
  nz53 =0.0d0
  nz53f=0.0d0
  nz53t=0.0d0
  nz53q=0.0d0
  nzsqa =0.0d0
  nzsqaf=0.0d0
  nzsqat=0.0d0
  nzsqaq=0.0d0
  do n=1,ncomp
    nex(n)   =0.0d0
    nextx(n) =0.0d0
    nionx(n) =0.0d0
    nz53x(n) =0.0d0
    nzsqax(n)=0.0d0
  enddo
  do ispec=1,ncomp
    ne =ne +select(ispec)
    nef=nef+selectf(ispec)
    net=net+selectt(ispec)
    neq=neq+selectq(ispec)
    next =next +sext (ispec)
    nextf=nextf+sextf(ispec)
    nextt=nextt+sextt(ispec)
    nextq=nextq+sextq(ispec)
    nion =nion +sion (ispec)
    nionf=nionf+sionf(ispec)
    niont=niont+siont(ispec)
    nionq=nionq+sionq(ispec)
    nz53 =nz53 +sz53 (ispec)
    nz53f=nz53f+sz53f(ispec)
    nz53t=nz53t+sz53t(ispec)
    nz53q=nz53q+sz53q(ispec)
    nzsqa =nzsqa +szsq (ispec)/a(ispec)
    nzsqaf=nzsqaf+szsqf(ispec)/a(ispec)
    nzsqat=nzsqat+szsqt(ispec)/a(ispec)
    nzsqaq=nzsqaq+szsqq(ispec)/a(ispec)
    do n=1,ncomp
      nex(n)   =nex(n)   +selectx(ispec,n)
      nextx(n) =nextx(n) +sextx(ispec,n)
      nionx(n) =nionx(n) +sionx(ispec,n)
      nz53x(n) =nz53x(n) +sz53x(ispec,n)
      nzsqax(n)=nzsqax(n)+szsqx(ispec,n)/a(ispec)
    enddo
  enddo
!
! contributions from electron - positron pairs
!
!
! first iterate to find flplus = fl for the positrons
!
  psiplus=-psi-2.0d0/(ct*t)
  if (tl.ge.19.0d0.and.psiplus.ge.-170.0d0) then
    gpos=exp_cr(psiplus-2.0d0)
    if (gpos.le.4.33d-09) then
      sspos=gpos*(2.0d0-4.0d0*gpos)
    else
      sspos=log_cr(1.0d0+4.0d0*gpos)/2.0d0
    endif
    do it=1,4
      fpos=2.0d0+2.0d0*sspos+log_cr(sspos/(2.0d0+sspos))-psiplus
      dpos=2.0d0+2.0d0/(sspos*(2.0d0+sspos))
      sspos=sspos-fpos/dpos
    enddo
    fplus=sspos*(2.0d0+sspos)
    flplus=log_cr(fplus)
!
! use EFF routine to find the equivalent of ne, pe, etc for the positrons
!
    call eff (fplus,t,flplus,tl,posnod,posnodf,posnodt,ppos,pposf,ppost,spos,sposf,spost,upos,uposf,upost,wfpos,psipos)
!
! find derivatives of positron number density, etc. with respect to fl (minus) at constant t
! and with respect to t at constant fl 
!
    dpsiminus=1.0d0/f    +1.0d0/(1.0d0+sqrt(f    ))
    dpsiplus =1.0d0/fplus+1.0d0/(1.0d0+sqrt(fplus))
    dflplusdflminus=-f*dpsiminus/(fplus*dpsiplus)
    dflplusdtl=2.0d0/(ct*t*fplus*dpsiplus)
    posnodt=posnodt+posnodf*dflplusdtl
    posnodf=posnodf*dflplusdflminus
    ppost=ppost+pposf*dflplusdtl
    pposf=pposf*dflplusdflminus
    spost=spost+sposf*dflplusdtl
    sposf=sposf*dflplusdflminus
    upost=upost+uposf*dflplusdtl
    uposf=uposf*dflplusdflminus
  else
    posnod =0.0d0
    posnodf=0.0d0
    posnodt=0.0d0
    ppos =0.0d0
    pposf=0.0d0
    ppost=0.0d0
    spos =0.0d0
    sposf=0.0d0
    spost=0.0d0
    upos =0.0d0
    uposf=0.0d0
    upost=0.0d0
  endif
!
! en is mean molecular weight per electron from ionization
! elnod is electron number density divided by NA, includes those from pairs
!
  en=1.0d0/ne
!
! subtract electrons from pairs to find density
!
   rho=(elnod-posnod)*en
   rl=log_cr(rho)
   rf=(elnod*ref-posnod*posnodf)/(elnod-posnod)-en*nef
   rt=(elnod*ret-posnod*posnodt)/(elnod-posnod)-en*net
   rq=                                         -en*neq
   rx=                                         -en*nex
!
! include coulomb and quantum corrections to equation of state by assuming 
! they do not alter the ionisation balance. Follows the prescription of the 
! eos in Iben, Fujimoto and MacDonald, 1992 ApJ, 388, 521. Updated to use 
! results of Stringfellow, G.S., DeWitt, H.E. & Slattery, W.L. 1990 Phys. Rev. 
! A, 41, 1105.
!
  coul=2.275d+05*nz53*(ne*rho)**third/(t*nion)
!
! derivatives of ln coul.
!
  coulf=nz53f/nz53-nionf/nion+ref/3.0d0
  coult=nz53t/nz53-niont/nion+ret/3.0d0-1.0d0
  coulq=nz53q/nz53-nionq/nion
  coulx=nz53x/nz53-nionx/nion
  coull=log_cr(coul)
  plam =7.833d+03*sqrt(rho*nzsqa)/t
  plams=plam**2
  plaml=log_cr(plam)
!
! derivatives of ln plam
!
  plamf=0.5d0*(rf+nzsqaf/nzsqa)
  plamt=0.5d0*(rt+nzsqat/nzsqa)-1.0d0
  plamq=0.5d0*(rq+nzsqaq/nzsqa)
  plamx=0.5d0*(rx+nzsqax/nzsqa)
!
! solid phase - different from fluid phase
!
  x1=plam/1.0643d0
  x2=plam/2.9438d0
  ex1=exp_cr(-x1)
  ex2=exp_cr(-x2)
  debx1=debfun(x1)
  debx2=debfun(x2)
  vlx1=-1.875d0*x1+3.0d0*log_cr(1.0d0-ex1)-debx1+3.0d0*x1
  vlx2=-1.875d0*x2+3.0d0*log_cr(1.0d0-ex2)-debx2+3.0d0*x2
  dlx1=1.125d0*x1+3.0d0*debx1
  dlx2=1.125d0*x2+3.0d0*debx2
  ddlx1=1.125d0*x1+9.0d0*x1*ex1/(1.0d0-ex1)-9.0d0*debx1
  ddlx2=1.125d0*x2+9.0d0*x2*ex2/(1.0d0-ex2)-9.0d0*debx2
  quj  =0.5711d0*vlx1 +0.4289d0*vlx2
  dquj =0.5711d0*dlx1 +0.4289d0*dlx2
  ddquj=0.5711d0*ddlx1+0.4289d0*ddlx2
  couls=coul**2
  hcoul =0.895929d0*coul+1606.0d0/couls
  dcoul =0.895929d0*coul-3212.0d0/couls
  ddcoul=0.895929d0*coul+6424.0d0/couls
  hcoulf= dcoul*coulf
  hcoult= dcoul*coult
  hcoulq= dcoul*coulq
  hcoulx= dcoul*coulx
  dcoulf=ddcoul*coulf
  dcoult=ddcoul*coult
  dcoulq=ddcoul*coulq
  dcoulx=ddcoul*coulx
  qujf=dquj*plamf
  qujt=dquj*plamt
  qujq=dquj*plamq
  qujx=dquj*plamx
  dqujf=ddquj*plamf
  dqujt=ddquj*plamt
  dqujq=ddquj*plamq
  dqujx=ddquj*plamx
  crt=cr*t
  p0=crt*rho
  p0f=p0*rf
  p0t=p0*(rt+1.0d0)
  p0q=p0*rq
!
! electron pressure is pe
!
!
! ion pressure
!
  pi=(ni-nh2)*p0
  pion=pi
  pif=        -nh2f *p0+(ni-nh2)*p0f
  pit=        -nh2t *p0+(ni-nh2)*p0t
  piq=        -nh2q *p0+(ni-nh2)*p0q
  pix=(1.0d0/a-nh2x)*p0+(ni-nh2)*p0*rx
!
! radiation pressure
!
  pr=cat*t4
!
! ion contribution to pressure is included in coulomb and quantum terms
!
  pcoul=-p0*dcoul*nion/3.0d0
  pcoulf=pcoul*(      rf+dcoulf/dcoul+nionf/nion)
  pcoult=pcoul*(1.0d0+rt+dcoult/dcoul+niont/nion)
  pcoulq=pcoul*(      rq+dcoulq/dcoul+nionq/nion)
  pcoulx=pcoul*(      rx+dcoulx/dcoul+nionx/nion)
!
! quantum terms
!
  pqu=0.5d0*nion*p0*dquj
  pquf=pqu*(nionf/nion+rf      +dqujf/dquj)
  pqut=pqu*(niont/nion+rt+1.0d0+dqujt/dquj)
  pquq=pqu*(nionq/nion+rq      +dqujq/dquj)
  pqux=pqu*(nionx/nion+rx      +dqujx/dquj)
  ppi=next*p0*dpi
  ppif=p0*dpi*(nextf+ rf*next       )
  ppit=p0*dpi*(nextt+(rt+1.0d0)*next)
  ppiq=p0*dpi*nextq+next*p0q*dpi+next*p0*dpiq
  ppix=p0*dpi*(nextx+ rx*next       )
!
! polarization terms
!
  ne2bnd =(ne0+ne)*(ne0-ne)
  ne2bndf=-2.0d0*ne*nef
  ne2bndt=-2.0d0*ne*net
  ne2bndq=-2.0d0*ne*neq
  ne2bndx=2.0d0*(ne0*zmax/a-ne*nex)
  ppola=0.5d0*dc*p0*rho
  ppol =ne2bnd*ppola
  ppolf=ppol*(dcf+2.0d0*rf      )+ppola*ne2bndf
  ppolt=ppol*(dct+2.0d0*rt+1.0d0)+ppola*ne2bndt
  ppolq=ppol*(dcq+2.0d0*rq      )+ppola*ne2bndq
  ppolx=2.0d0*ppol*rx+0.5d0*p0*rho*(ne2bndx*dc+ne2bnd*dcx)
  pgas=pe+ppos+pcoul+ppi+pqu+ppol
  p=pgas+pr
  pl=log_cr(p)
  pf=(pe*pef+ppos*pposf         +pcoulf+ppif+pquf+ppolf)/p
  pt=(pe*pet+ppos*ppost+pr*4.0d0+pcoult+ppit+pqut+ppolt)/p
  pq=(                           pcoulq+ppiq+pquq+ppolq)/p
  px=(                           pcoulx+ppix+pqux+ppolx)/p
!
! calculate partial pressures for diffusion equations, in linear approximation
!
  pion =pi +ppi +pqu +ppol
  pionf=pif+ppif+pquf+ppolf
  piont=pit+ppit+pqut+ppolt
  pionq=piq+ppiq+pquq+ppolq
  pionx=pix+ppix+pqux+ppolx
  nimnh2 =ni-nh2
  x1mnh2 =x(1)/a(1)-nh2
  pip (1)=x1mnh2*pion /nimnh2
  pipf(1)=(x1mnh2*pionf-nh2f*pion+x1mnh2*pion*nh2f/nimnh2)/nimnh2
  pipt(1)=(x1mnh2*piont-nh2t*pion+x1mnh2*pion*nh2t/nimnh2)/nimnh2
  pipq(1)=(x1mnh2*pionq-nh2q*pion+x1mnh2*pion*nh2q/nimnh2)/nimnh2
  do n=1,ncomp
    nimnh2xn=1.0d0/a(n)-nh2x(n)
    pipx(1,n)=(x1mnh2*pionx(n)-nh2x(n)*pion-x1mnh2*pion*nimnh2xn/nimnh2)/nimnh2
  enddo
  pipx(1,1)=pipx(1,1)+pion/(nimnh2*a(1))
  do ispec=2,ncomp
    xionai=x(ispec)/a(ispec)
    pip (ispec)=xionai*pion /nimnh2
    pipf(ispec)=xionai*(pionf+pion*nh2f/nimnh2)/nimnh2
    pipt(ispec)=xionai*(piont+pion*nh2t/nimnh2)/nimnh2
    pipq(ispec)=xionai*(pionq+pion*nh2q/nimnh2)/nimnh2
    do n=1,ncomp
      nimnh2xn=1.0d0/a(n)-nh2x(n)
      pipx(ispec,n)=xionai*(pionx(n)-pion*nimnh2xn/nimnh2)/nimnh2
    enddo
    pipx(ispec,ispec)=pipx(ispec,ispec)+pion/(nimnh2*a(ispec))
  enddo
!
! calculate the entropy per unit mass of gas.
!
!
! radiation
!
  b=4.0d0*pr/p0
  bf=-b* rf
  bt=-b*(rt-3.0d0)
  bq=-b* rq
  bx=-b* rx
!
! electrons, excluding those from pair production
!
  sel =ne*se
  self=nef*se+ne*sef
  selt=net*se+ne*set
  selq=neq*se
!
! molecules, atoms and ions
!
  sions =(ni-nh2)*(1.5d0*tl-rl-5.57243429d0)
  sionsf=(ni-nh2)*(        -rf             )         -nh2f *(1.5d0*tl-rl-5.57243429d0)
  sionst=(ni-nh2)*(1.5d0   -rt             )         -nh2t *(1.5d0*tl-rl-5.57243429d0)
  sionsq=(ni-nh2)*(        -rq             )         -nh2q *(1.5d0*tl-rl-5.57243429d0)
  sionsx=(ni-nh2)*(        -rx             )+(1.0d0/a-nh2x)*(1.5d0*tl-rl-5.57243429d0)
  sions =sions +nh2 *omegah2t
  sionsf=sionsf+nh2f*omegah2t
  sionst=sionst+nh2t*omegah2t+nh2*omegah2tt
  sionsq=sionsq+nh2q*omegah2t
  sionsx=sionsx+nh2x*omegah2t
  do ispec=1,ncomp
     sions =sions +sis (ispec)
     sionsf=sionsf+sisf(ispec)
     sionst=sionst+sist(ispec)
     sionsq=sionsq+sisq(ispec)
     do n=1,ncomp
        sionsx(n)=sionsx(n)+sisx(ispec,n)
     enddo
  enddo
!
! hard sphere interactions
!
  spi=-next*hpi
  spif=-nextf*hpi
  spit=-nextt*hpi
  spiq=-nextq*hpi-next*hpiq
  spix=-nextx*hpi
!
! Coulomb interactions
!
  scoul =nion *(hcoul-dcoul)
  scoulf=nionf*(hcoul-dcoul)+nion*(hcoulf-dcoulf)
  scoult=niont*(hcoul-dcoul)+nion*(hcoult-dcoult)
  scoulq=nionq*(hcoul-dcoul)+nion*(hcoulq-dcoulq)
  scoulx=nionx*(hcoul-dcoul)+nion*(hcoulx-dcoulx)
!
! quantum corrections
!
  squ =-nion *(quj-dquj)
  squf=-nionf*(quj-dquj)-nion*(qujf-dqujf)
  squt=-niont*(quj-dquj)-nion*(qujt-dqujt)
  squq=-nionq*(quj-dquj)-nion*(qujq-dqujq)
  squx=-nionx*(quj-dquj)-nion*(qujx-dqujx)
!
! entropy correction due to ion contribution being included in Coulomb and quantum terms
! determined by comparing Gibbs free energies at phase transition in low density limit
!
  scor =-1.5d0*nion *log_cr((nz53*ne**third)/(nion*nzsqa)) &
    -(ni-nh2)*(1.5d0*tl-rl-5.57243429d0)
  scorf=-1.5d0*nionf*log_cr((nz53*ne**third)/(nion*nzsqa))-nion*(1.5d0*coulf-3.0d0*plamf+rf      ) &
    -(ni-nh2)*(     -rf)+         nh2f *(1.5d0*tl-rl-5.57243429d0)
  scort=-1.5d0*niont*log_cr((nz53*ne**third)/(nion*nzsqa))-nion*(1.5d0*coult-3.0d0*plamt+rt-1.5d0) &
    -(ni-nh2)*(1.5d0-rt)+         nh2t *(1.5d0*tl-rl-5.57243429d0)
  scorq=-1.5d0*nionq*log_cr((nz53*ne**third)/(nion*nzsqa))-nion*(1.5d0*coulq-3.0d0*plamq+rq      ) &
    -(ni-nh2)*(     -rq)+         nh2q *(1.5d0*tl-rl-5.57243429d0)
  scorx=-1.5d0*nionx*log_cr((nz53*ne**third)/(nion*nzsqa))-nion*(1.5d0*coulx-3.0d0*plamx+rx      ) &
    -(ni-nh2)*(     -rx)-(1.0d0/a-nh2x)*(1.5d0*tl-rl-5.57243429d0)
  s =cr*(sel   +b +scoul +spi +squ +sions +scor )
  sf=cr*(self  +bf+scoulf+spif+squf+sionsf+scorf)
  st=cr*(selt  +bt+scoult+spit+squt+sionst+scort)
  sq=cr*(selq  +bq+scoulq+spiq+squq+sionsq+scorq)
  sx=cr*(se*nex+bx+scoulx+spix+squx+sionsx+scorx)
!
! polarization 
!
  spol=-da*ppol/(dc*rho*t)
  spolf=-spol*(rf+dcf      )- da*ppolf          /(dc*rho*t)
  spolt=-spol*(rt+dct+1.0d0)- da*ppolt          /(dc*rho*t)
  spolq=-spol*(rq+dcq      )- da*ppolq          /(dc*rho*t)
  spolx=-spol*(rx+dcx/dc   )-(da*ppolx+dax*ppol)/(dc*rho*t)
  s =s +spol
  sf=sf+spolf
  st=st+spolt
  sq=sq+spolq
  sx=sx+spolx
!
! add contribution from electron-positron pairs to entropy per unit mass
!
  spairs1=cr*posnod/rho
  spairs =spairs1*(se+spos)
  spairsf=spairs*(posnodf-rf)+spairs1*(sef+sposf)
  spairst=spairs*(posnodt-rt)+spairs1*(set+spost)
  spairsq=spairs*(       -rq)
  s =s +spairs
  sf=sf+spairsf
  st=st+spairst
  sq=sq+spairsq
  sx=sx-spairs*rx
!
! internal energy
!
! bound state contribution, excluding H2 molecule
!
  uion =0.0d0
  uionf=0.0d0
  uiont=0.0d0
  uionq=0.0d0
  uionx=0.0d0
  do ispec=1,ncomp
    uion =uion +suions (ispec)
    uionf=uionf+suionsf(ispec)
    uiont=uiont+suionst(ispec)
    uionq=uionq+suionsq(ispec)
    do n=1,ncomp
      uionx(n)=uionx(n)+suionsx(ispec,n)
    enddo
  enddo
!
! Coulomb terms
!
  ucoul=-dcoul*nion
  ucoulf=-dcoulf*nion-dcoul*nionf
  ucoult=-dcoult*nion-dcoul*niont
  ucoulq=-dcoulq*nion-dcoul*nionq
  ucoulx=-dcoulx*nion-dcoul*nionx
!
! quantum terms
!
  uqu=nion*dquj
  uquf=nionf*dquj+nion*dqujf
  uqut=niont*dquj+nion*dqujt
  uquq=nionq*dquj+nion*dqujq
  uqux=nionx*dquj+nion*dqujx
  u=   crt*(ue *ne                            +0.75d0*b +ti* uion       +ucoul +uqu )
  uf=  crt*(uef*ne+ue*nef                     +0.75d0*bf+ti* uionf      +ucoulf+uquf)
  ut=u+crt*(uet*ne+ue*net                     +0.75d0*bt+ti*(uiont-uion)+ucoult+uqut)
  uq=  crt*(       ue*neq                     +0.75d0*bq+ti* uionq      +ucoulq+uquq)
  ux=  crt*(ue*nex       +1.5d0*(1.0d0/a-nh2x)+0.75d0*bx+ti* uionx      +ucoulx+uqux)
  u =u +crt* nh2      *omegah2t
  uf=uf+crt* nh2f     *omegah2t
  ut=ut+crt*(nh2+nh2t)*omegah2t+crt*nh2*omegah2tt
  uq=uq+crt* nh2q     *omegah2t
  ux=ux+crt* nh2x     *omegah2t
!
! polarization term
!
  upol1=db*ti/(dc*rho)
  upol =upol1*ppol
  upolf=upol1*ppolf-upol*(dcf+rf      )
  upolt=upol1*ppolt-upol*(dct+rt+1.0d0)
  upolq=upol1*ppolq-upol*(dcq+rq      )
  upol2=0.5d0*rho*ti*crt
  upolx=upol*rx+upol2*(ne2bnd*dbx+ne2bndx*db)
  u =u +upol
  uf=uf+upolf
  ut=ut+upolt
  uq=uq+upolq
  ux=ux+upolx
!
! add contribution to internal energy per unit mass from electron-positron pairs
!
  upairs1=crt*posnod/rho
  upairs2=ue+upos+2.0d0/(ct*t)
  upairs =upairs1*upairs2
  upairsf=upairs1*((posnodf-rf)*upairs2+uef+uposf        )
  upairst=upairs1*((posnodt-rt)*upairs2+uet+upost+ue+upos)
  upairsq=upairs1*((       -rq)*upairs2                  )
  u =u +upairs
  uf=uf+upairsf
  ut=ut+upairst
  uq=uq+upairsq
  ux=ux-upairs*rx
!
! pressure ionization equation
! fx1 is (4*pi/3)*Na*sum (Xi*Rhsi**3/Ai)
! In hydrogenic approximation, hard sphere radii are given by rhsi/rhs(H)=sqrt(IP(H)/IPi)
!
  fx1=0.0d0
  do n=1,ncomp
    fx1=fx1+fx1coef(n)*x(n)
  enddo
  fx2=ni-nh2
  gq1=rho*fx1/fx2
!
! derivatives of ln gq1
!
 gq1f=rf+nh2f/fx2
 gq1t=rt+nh2t/fx2
 gq1q=rq+nh2q/fx2
 gq1x=rx+fx1coef/fx1-(1.0d0/a-nh2x)/fx2
 gq2=gq1*next+small
!
! derivatives of ln gq2
!
  gq2f=(gq1/gq2)*(gq1f*next+nextf)
  gq2t=(gq1/gq2)*(gq1t*next+nextt)
  gq2q=(gq1/gq2)*(gq1q*next+nextq)
  gq2x=(gq1/gq2)*(gq1x*next+nextx)
  gq=log_cr(gq2*(1.0d0+eq))-q
!
! derivatives of gq
!
  gqf=gq2f
  gqt=gq2t
  gqq=gq2q-1.0d0/(1.0d0+eq)
  gqx=gq2x
  qf=-gqf/gqq
  qt=-gqt/gqq
  qx=-gqx/gqq
!
! need derivatives of ln pressure and gq when considered as functions
! of fl, tl and q
!
  pf_qt=pf
  pq_ft=pq
  gqf_qt=gqf
  gqq_ft=gqq
!
!
! here partial derivatives are calculated, removing dependence on q
!
  pf=pf+qf*pq
  pt=pt+qt*pq
  rf=rf+qf*rq
  rt=rt+qt*rq
  sf=sf+qf*sq
  st=st+qt*sq
  uf=uf+qf*uq
  ut=ut+qt*uq
  px=px+qx*pq
  rx=rx+qx*rq
  sx=sx+qx*sq
  ux=ux+qx*uq
  do ispec=1,ncomp
    pipf(ispec)=pipf(ispec)+qf*pipq(ispec)
    pipt(ispec)=pipt(ispec)+qt*pipq(ispec)
    fcsf(ispec)=fcsf(ispec)+qf*fcsq(ispec)
    fcst(ispec)=fcst(ispec)+qt*fcsq(ispec)
    fneutf(ispec)=fneutf(ispec)+qf*fneutq(ispec)
    fneutt(ispec)=fneutt(ispec)+qt*fneutq(ispec)
    do n=1,ncomp
      pipx(ispec,n)=pipx(ispec,n)+qx(n)*pipq(ispec)
      fcsx(ispec,n)=fcsx(ispec,n)+qx(n)*fcsq(ispec)
      fneutx(ispec,n)=fneutx(ispec,n)+qx(n)*fneutq(ispec)
    enddo
  enddo
  qq=pt*sf-pf*st
!
! specific heat at constant pressure
!
  cp=-qq/pf
!
! adiabatic gradient
!
  grada=sf/qq
!
! Gamma_1
!
  gamma=qq/(rt*sf-rf*st)
!
! cv is the specific heat at constant volume
!
  cv=(rf*st-rt*sf)/rf
!
! vt is partial derivative of ln V with respect to ln T at constant pressure (V = 1/rho)
!
  vt=-(rt*pf-rf*pt)/pf
  return
  end
