! ***********************************************************************
!
!   Copyright (C) 2014  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module eosDE_load_tables
      use eos_def
      use const_def
      use crlibm_lib
      
      implicit none
      
      ! the file eosDE data
      integer, parameter :: jlogT = 1
      integer, parameter :: jlogPgas = 2
      integer, parameter :: jlogS = 3
      integer, parameter :: jchiRho = 4
      integer, parameter :: jchiT = 5
      integer, parameter :: jCp = 6
      integer, parameter :: jCv = 7
      integer, parameter :: jdE_dRho = 8
      integer, parameter :: jdS_dT = 9
      integer, parameter :: jdS_dRho = 10
      integer, parameter :: jmu = 11
      integer, parameter :: jlogfree_e = 12
      integer, parameter :: jgamma1 = 13
      integer, parameter :: jgamma3 = 14
      integer, parameter :: jgrad_ad = 15
      integer, parameter :: jeta = 16
      integer, parameter :: num_eosDE_file_vals = 16

      integer, parameter :: file_max_num_logVs = 1000
      

      contains
      
      
      subroutine check_for_error_in_eosDE_data(ierr, fname)
         integer, intent(in) :: ierr
         character (len=*) :: fname
         if (ierr == 0) return
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*)
         write(*,*) 'load eos tables ' // trim(fname)
         write(*,*)
         write(*,*)
         write(*,'(a)') 'FATAL ERROR: missing or bad eos data.'
         write(*,'(a)') 'Please update by removing the directories ' &
            // 'mesa/data/eos*_data' &
            // 'and rerunning the mesa ./install script.'
         write(*,*)
         stop 1
      end subroutine check_for_error_in_eosDE_data
      
      
      subroutine request_reinstall_eosDE_data
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*)
            write(*,*) 'NOTICE: you need to install a new verion of the eos data.'
            write(*,*) 'Please update by removing the directory mesa/data/eosDE_data,'
            write(*,*) 'and rerunning the mesa ./install script.'
            write(*,*)
            write(*,*)
            stop 1
      end subroutine request_reinstall_eosDE_data
      
      
      subroutine Load_eosDE_Table(ierr)
         use utils_lib
         integer, intent(out) :: ierr
         
         integer :: iz, ix, i
         
         ierr = 0
         if (eosDE_is_initialized) return
         
!$OMP CRITICAL
         call do_read
         if (ierr == 0) eosDE_is_initialized = .true.
!$OMP END CRITICAL
         
         contains
         
         subroutine do_read
            if (eosDE_is_initialized) return
            do iz = 1, num_eosDE_Zs
               do ix = 1, num_eos_Xs
                  if (eos_Zs(iz) + eos_Xs(ix) > 1.0000001d0) cycle
                  call read_one(ix,iz,ierr)
                  if (ierr /= 0) return
               end do
            end do
         end subroutine do_read
               
         subroutine read_one(ix,iz,ierr)
            use const_def, only: mesa_data_dir
            integer, intent(in) :: ix, iz
            integer, intent(out) :: ierr
            character (len=256) :: fname, cache_filename
            integer :: iounit1, iounit2            
            iounit1 = alloc_iounit(ierr); if (ierr /= 0) return
            iounit2 = alloc_iounit(ierr); if (ierr /= 0) return
            call Get_eosDE_Table_Filenames( &
               eos_Zs(iz), eos_Xs(ix), mesa_data_dir, fname, cache_filename)
            call Load1_eosDE_Table( &
               ix, iz, fname, cache_filename, iounit1, iounit2, use_cache_for_eos, ierr)
            if (ierr /= 0) then
               write(*,*) 'Load_eosDE_Table ierr', ierr, ix, iz, eos_Xs(ix), eos_Zs(iz)
            end if
            call free_iounit(iounit2)
            call free_iounit(iounit1)        
         end subroutine read_one
         
      end subroutine Load_eosDE_Table
      
      
      subroutine Get_eosDE_Table_Filenames(Z, X, data_dir, fname, cache_filename)
            
         real(dp), intent(in) :: Z, X
         character (*), intent(in) :: data_dir
         character (len=*), intent(out) :: fname, cache_filename
         character (len=256) :: Zstr, Xstr, suffix
         
         call setstr(Z,Zstr)
         call setstr(X,Xstr)
         suffix = ''
         
         fname = trim(data_dir) //  &
               '/eosDE_data/' // trim(eosDE_file_prefix) // '-eosDE_' // &
               trim(Zstr) // 'z' // trim(Xstr) // 'x' // trim(suffix) // '.data'
         cache_filename = trim(eosDE_cache_dir) //  &
               '/' // trim(eosDE_file_prefix) // '-eosDE_' // &
               trim(Zstr) // 'z' // trim(Xstr) // 'x' // trim(suffix) // '.bin'

         contains
         
         subroutine setstr(v,str)
            real(dp), intent(in) :: v
            character (len=*) :: str
            if (v > 0.99999) then
               str = '100'
            else if (v > 0.09999) then
               write(str, '(i2)') floor(100d0 * v + 0.5)
            else
               write(str, '(a,i1)') '0', floor(100d0 * v + 0.5)
            end if
         end subroutine setstr

      end subroutine Get_eosDE_Table_Filenames
      
      
      subroutine Load1_eosDE_Table( &
            ix, iz, filename, cache_filename, io_unit, cache_io_unit, use_cache, info)
         integer, intent(in) :: ix, iz
         character (*), intent(in) :: filename, cache_filename
         integer, intent(in) :: io_unit, cache_io_unit
         logical, intent(in) :: use_cache
         integer, intent(out) :: info
         
         real(dp) :: X, Z
         real :: X_in, Z_in, logV, logE
         integer :: j, i, k, iV, ios, status
         character (len=1000) :: message
         real(dp), parameter :: tiny = 1e-6
         type (EosDE_XZ_Info), pointer :: ep
         real, pointer :: tbl(:,:,:,:) ! => ep% tbl1
         real, pointer :: tbl2_1(:), tbl2(:,:,:)
         real(dp), target :: vec_ary(50)
         real(dp), pointer :: vec(:)
         integer :: n

         info = 0            
         vec => vec_ary
         
         X = eos_Xs(ix)
         Z = eos_Zs(iz)   
         
         ep => eosDE_XZ_data(ix,iz)     
         
         write(message,*) 'open ', trim(filename)
         
         open(UNIT=io_unit, FILE=trim(filename), ACTION='READ', STATUS='OLD', IOSTAT=ios)
         call check_for_error_in_eosDE_data(ios, filename)

         read(io_unit,*,iostat=info)
         if (info /= 0) return

         read(io_unit,'(a)',iostat=info) message
         if (info == 0) call str_to_vector(message, vec, n, info)
         if (info /= 0 .or. n < 11) then
            write(*,'(a)') 'failed while reading ' // trim(filename)
            close(19)
            return
         end if
         ep% version = int(vec(1))
         X_in = vec(2)
         Z_in = vec(3)
         ep% num_logEs = int(vec(4))
         ep% logE_min = vec(5)
         ep% logE_max = vec(6)
         ep% del_logE = vec(7)
         ep% num_logVs = int(vec(8))
         ep% logV_min = vec(9)
         ep% logV_max = vec(10)
         ep% del_logV = vec(11)

         read(io_unit,*,iostat=info)
         if (info /= 0) return

         if (abs(X-X_in) > tiny .or. abs(Z-Z_in) > tiny) then
            write(*,*) 'bad header info in ' // trim(filename)
            info = -1
            close(io_unit)
            if (abs(X-X_in) > tiny) then
               write(*,'(a50,l)') 'abs(X-X_in) > tiny', abs(X-X_in) > tiny
            end if
            if (abs(Z-Z_in) > tiny) then
               write(*,'(a50,l)') 'abs(Z-Z_in) > tiny', abs(Z-Z_in) > tiny
            end if
            write(*,*)
            call request_reinstall_eosDE_data
            return
         end if

         allocate(ep% tbl1(sz_per_eos_point*num_eos_vals*ep% num_logVs*ep% num_logEs),  &
                  ep% logVs(ep% num_logVs), ep% logEs(ep% num_logEs),   &
                  STAT=info)
         if (info /= 0) return
         
         tbl(1:sz_per_eos_point,1:num_eos_vals,1:ep% num_logVs,1:ep% num_logEs) =>  &
               ep% tbl1(1:sz_per_eos_point*num_eos_vals*ep% num_logVs*ep% num_logEs)

         ep% logVs(1) = ep% logV_min
         do i = 2, ep% num_logVs-1
            ep% logVs(i) = ep% logVs(i-1) + ep% del_logV
         end do
         ep% logVs(ep% num_logVs) = ep% logV_max
         
         ep% logEs(1) = ep% logE_min
         do i = 2, ep% num_logEs-1
            ep% logEs(i) = ep% logEs(i-1) + ep% del_logE
         end do
         ep% logEs(ep% num_logEs) = ep% logE_max

         if (use_cache) then
            call Read_eosDE_Cache(X, Z, ep, cache_filename, cache_io_unit, ios)
            if (ios == 0) then
               close(io_unit)
               return
            end if
         end if
         
         status = 0
         allocate(tbl2_1(num_eosDE_file_vals*ep% num_logVs*ep% num_logEs), STAT=status)
         if (status .ne. 0) then
            info = -1
            return
         end if
         
         tbl2(1:num_eosDE_file_vals,1:ep% num_logVs,1:ep% num_logEs) =>  &
               tbl2_1(1:num_eosDE_file_vals*ep% num_logVs*ep% num_logEs)
         
         do iV=1,ep% num_logVs
         
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return
            
            read(io_unit,'(a)',iostat=info) message
            if (info == 0) call str_to_double(message, vec(1), info)
            if (failed('read logV')) return
            logV = vec(1)
            
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return 
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return
            
            do i=1,ep% num_logEs

               read(io_unit,'(a)',iostat=info) message
               if (failed('read line')) then
                  write(*,'(a)') trim(message)
                  write(*,*) trim(filename)
                  write(*,*) 'iV, i', iV, i
                  write(*,*) 'logV', logV
                  write(*,*) 'bad input line?'
                  stop 1
               end if
               
               call str_to_vector(message, vec, n, info)
               if (info /= 0 .or. n < 1+num_eosDE_file_vals) then
                  write(*,'(a)') trim(message)
                  write(*,*) trim(filename)
                  write(*,*) 'iV, i', iV, i
                  write(*,*) 'logV', logV
                  write(*,*) 'bad input line?'
                  stop 1
               end if
               logE = vec(1)
               do j=1,num_eosDE_file_vals
                  tbl2(j,iV,i) = vec(1+j)
               end do
               
            enddo
            
            if (iV == ep% num_logVs) exit
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return 
            read(io_unit,*,iostat=info)
            if (failed('skip line')) return 
            
         end do
            
         close(io_unit)
         
         call Make_XeosDE_Interpolation_Data(ep, tbl2_1, info)
         deallocate(tbl2_1)
         if (info /= 0) return
         
         call Check_XeosDE_Interpolation_Data(ep)
         
         if (.not. use_cache) return

         open(unit=cache_io_unit, file=trim(cache_filename), iostat=ios,  &
               action='write', form='unformatted')

         if (ios == 0) then
            write(*,'(a)') 'write ' // trim(cache_filename)
            write(cache_io_unit)  &
               X_in, Z_in, ep% num_logEs, ep% logE_min, ep% logE_max, ep% del_logE,  &
               ep% num_logVs, ep% logV_min, ep% logV_max, ep% del_logV, ep% version
            write(cache_io_unit)  &
               ep% tbl1(1:sz_per_eos_point*num_eos_vals*ep% num_logVs*ep% num_logEs)
            close(cache_io_unit)
         end if
         
         contains
         
         subroutine Check_XeosDE_Interpolation_Data(ep)
            use utils_lib,only:is_bad_real
            type (EosDE_XZ_Info), pointer :: ep
            ! for logT > 6.8 and logRho < -10, splines can get bogus higher order terms
            ! replace NaN's and Infinities with 0
            integer :: i, j, iV, jE
            do i = 1, sz_per_eos_point
               do j = 1, num_eos_vals
                  do iV = 1, ep% num_logVs
                     do jE = 1, ep% num_logEs
                        if (is_bad_real(tbl(i,j,iV,jE))) then
                           tbl(i,j,iV,jE) = 0
                        end if
                     end do
                  end do
               end do
            end do
         end subroutine Check_XeosDE_Interpolation_Data
         
         
          logical function failed(str)
            character (len=*), intent(in) :: str
            failed = (info /= 0)
            if (failed) write(*,*) 'Load1_eosDE_Table failed: ' // trim(str)
         end function failed


      end subroutine Load1_eosDE_Table
      
      
      subroutine Make_XeosDE_Interpolation_Data(ep, tbl2_1, info)
         use interp_2d_lib_sg

         type (EosDE_XZ_Info), pointer :: ep
         real, pointer :: tbl2_1(:) ! =(num_eos_file_vals, ep% num_logVs, ep% num_logEs)
         integer, intent(out) :: info

         real :: logVs(ep% num_logVs)              ! x vector, strict ascending
         real :: logEs(ep% num_logEs)              ! y vector, strict ascending
         real, target :: f1_ary(sz_per_eos_point * ep% num_logVs * ep% num_logEs)              
            ! data & spline coefficients
         real, pointer :: f1(:), f(:,:,:), ep_tbl(:,:,:,:), tbl2(:,:,:)
         integer :: ibcxmin                   ! bc flag for x=xmin
         real :: bcxmin(ep% num_logEs)    ! bc data vs. y at x=xmin
         integer :: ibcxmax                   ! bc flag for x=xmax
         real :: bcxmax(ep% num_logEs)     ! bc data vs. y at x=xmax
         integer :: ibcymin                   ! bc flag for y=ymin
         real :: bcymin(ep% num_logVs)   ! bc data vs. x at y=ymin
         integer :: ibcymax                   ! bc flag for y=ymax
         real :: bcymax(ep% num_logVs)   ! bc data vs. x at y=ymax
         integer :: ili_logVs    ! =1: logRho grid is "nearly" equally spaced
         integer :: ili_logEs      ! =1: logT grid is "nearly" equally spaced
         integer :: ier            ! =0 on exit if there is no error.         
         integer :: iV, jE, ilogE, ilogV, num_logVs, num_logEs
         real :: fval(num_eosDE_file_vals), df_dx(num_eosDE_file_vals), df_dy(num_eosDE_file_vals)
         
         integer :: v, vlist(3), var, i, j
         character (len=256) :: message

         info = 0
         
         num_logVs = ep% num_logVs
         num_logEs = ep% num_logEs
         
         ep_tbl(1:sz_per_eos_point,1:num_eos_vals,1:num_logVs,1:num_logEs) =>  &
               ep% tbl1(1:sz_per_eos_point*num_eos_vals*num_logVs*num_logEs)

         tbl2(1:num_eosDE_file_vals,1:num_logVs,1:num_logEs) =>  &
               tbl2_1(1:num_eosDE_file_vals*num_logVs*num_logEs)
         
         f1 => f1_ary
         f(1:sz_per_eos_point,1:num_logVs,1:num_logEs) => &
               f1_ary(1:sz_per_eos_point*num_logVs*num_logEs)
                  
         do iV = 1, num_logVs
            logVs(iV) = ep% logV_min + (iV-1) * ep% del_logV
         end do

         do jE = 1, num_logEs
            logEs(jE) = ep% logE_min + (jE-1) * ep% del_logE
         end do
         
         ! copy file eos variables to internal eos interpolation tables
         do j=1,num_logEs
            do i=1,num_logVs
               ep_tbl(1,eosDE_ilnT,i,j) = tbl2(jlogT,i,j)*ln10
               ep_tbl(1,eosDE_ilnPgas,i,j) = tbl2(jlogPgas,i,j)*ln10
               ep_tbl(1,eos_ilnS,i,j) = tbl2(jlogS,i,j)*ln10
               ep_tbl(1,eos_igrad_ad,i,j) = tbl2(jgrad_ad,i,j)
               ep_tbl(1,eos_ichiRho,i,j) = tbl2(jchiRho,i,j)
               ep_tbl(1,eos_ichiT,i,j) = tbl2(jchiT,i,j)
               ep_tbl(1,eos_iCp,i,j) = tbl2(jCp,i,j)
               ep_tbl(1,eos_iCv,i,j) = tbl2(jCv,i,j)
               ep_tbl(1,eos_idE_dRho,i,j) = tbl2(jdE_dRho,i,j)
               ep_tbl(1,eos_idS_dT,i,j) = tbl2(jdS_dT,i,j)
               ep_tbl(1,eos_idS_dRho,i,j) = tbl2(jdS_dRho,i,j)
               ep_tbl(1,eos_imu,i,j) = tbl2(jmu,i,j)
               ep_tbl(1,eos_ilnfree_e,i,j) = tbl2(jlogfree_e,i,j)*ln10
               ep_tbl(1,eos_igamma1,i,j) = tbl2(jgamma1,i,j)
               ep_tbl(1,eos_igamma3,i,j) = tbl2(jgamma3,i,j)
               ep_tbl(1,eos_ieta,i,j) = tbl2(jeta,i,j)
            end do             
         end do

         ! just use "not a knot" bc's at edges of tables
         ibcxmin = 0; bcxmin(:) = 0
         ibcxmax = 0; bcxmax(:) = 0
         ibcymin = 0; bcymin(:) = 0
         ibcymax = 0; bcymax(:) = 0

         ! create tables for bicubic spline interpolation         
         do v = 1, num_eos_vals
            do i=1,ep% num_logVs
               do j=1,ep% num_logEs
                  f(1,i,j) = ep_tbl(1,v,i,j)
               end do
            end do
            call interp_mkbicub_sg( &
               logVs,ep% num_logVs,logEs,ep% num_logEs,f1,ep% num_logVs, &
               ibcxmin,bcxmin,ibcxmax,bcxmax, &
               ibcymin,bcymin,ibcymax,bcymax, &
               ili_logVs,ili_logEs,ier)
            if (ier /= 0) then
               write(*,*) 'interp_mkbicub_sg error happened for ep% value', v
               info = 3
               return
            end if
            do i=1,ep% num_logVs
               do j=1,ep% num_logEs
                  ep_tbl(2,v,i,j) = f(2,i,j)
                  ep_tbl(3,v,i,j) = f(3,i,j)
                  ep_tbl(4,v,i,j) = f(4,i,j)
               end do
            end do
         end do
         
      end subroutine Make_XeosDE_Interpolation_Data
      
      
      subroutine Read_eosDE_Cache(X, Z, ep, cache_filename, io_unit, ios)
         real(dp), intent(in) :: X, Z
         type (EosDE_XZ_Info), pointer :: ep
         character (*), intent(in) :: cache_filename
         integer, intent(in) :: io_unit ! use this for file access
         integer, intent(out) :: ios

         real :: X_in, Z_in, logE_min_in, logE_max_in, del_logE_in,  &
               logV_min_in, logV_max_in, del_logV_in
         integer :: num_logVs_in, num_logEs_in, version_in, i, j
         real, parameter :: tiny = 1e-6
         
         ios = 0
         open(unit=io_unit,file=trim(cache_filename),action='read', &
               status='old',iostat=ios,form='unformatted')
         if (ios /= 0) return
         
         read(io_unit, iostat=ios)  &
               X_in, Z_in, num_logEs_in, logE_min_in, logE_max_in, del_logE_in,  &
               num_logVs_in, logV_min_in, logV_max_in, del_logV_in, version_in
         if (ios /= 0) return
         
         if (ep% version /= version_in) then
            ios = 1
            write(*,*) 'read cache failed for version_in'
         end if
         if (ep% num_logVs /= num_logVs_in) then
            ios = 1
            write(*,*) 'read cache failed for ep% num_logVs'
         end if 
         if (ep% num_logEs /= num_logEs_in) then
            ios = 1
            write(*,*) 'read cache failed for ep% num_logEs'
         end if
         if (abs(X-X_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for X_in'
         end if
         if (abs(Z-Z_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for Z_in'
         end if
         if (abs(ep% logE_min-logE_min_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for ep% logE_min'
         end if    
         if (abs(ep% logE_max-logE_max_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for ep% logE_max'
         end if    
         if (abs(ep% del_logE-del_logE_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for ep% del_logE'
         end if    
         if (abs(ep% logV_min-logV_min_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for ep% logV_min'
         end if    
         if (abs(ep% logV_max-logV_max_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for ep% logV_max'
         end if
         if (abs(ep% del_logV-del_logV_in) > tiny) then
            ios = 1
            write(*,*) 'read cache failed for ep% del_logV'
         end if
         
         if (ios /= 0) then
            close(io_unit); return
         end if
         
         read(io_unit, iostat=ios)  &
            ep% tbl1(1:sz_per_eos_point*num_eos_vals*ep% num_logVs*ep% num_logEs)
         if (ios /= 0) then
            close(io_unit); return
         end if
         
         close(io_unit)

      end subroutine Read_eosDE_Cache

      
      end module eosDE_load_tables
