! ***********************************************************************
!
!   Copyright (C) 2014  Bill Paxton
!
!   MESA is free software; you can use it and/or modify
!   it under the combined terms and restrictions of the MESA MANIFESTO
!   and the GNU General Library Public License as published
!   by the Free Software Foundation; either version 2 of the License,
!   or (at your option) any later version.
!
!   You should have received a copy of the MESA MANIFESTO along with
!   this software; if not, it is available at the mesa website:
!   http://mesa.sourceforge.net/
!
!   MESA is distributed in the hope that it will be useful,
!   but WITHOUT ANY WARRANTY; without even the implied warranty of
!   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
!   See the GNU Library General Public License for more details.
!
!   You should have received a copy of the GNU Library General Public License
!   along with this software; if not, write to the Free Software
!   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
!
!
! ***********************************************************************

      module eosDE_eval
      use eos_def
      use const_def
      use crlibm_lib

      implicit none


      contains


      subroutine Get_eosDE_Results(rq, &
               Z_in, X_in, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               aE, alogE, aRho, alogRho, logT_guess, &
               T, logT, res, d_dlnRho_c_T, d_dlnT_c_Rho,  &
               d_dabar_c_TRho, d_dzbar_c_TRho, &
               dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
               dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
               ierr)

         use utils_lib, only: is_bad_num
         use eosDE_load_tables, only: Load_eosDE_Table
         
         ! INPUT
         
         type (EoS_General_Info), pointer :: rq ! general information about the request

         real(dp), intent(in) :: Z_in ! the desired Z
         real(dp), intent(in) :: X_in ! the desired X
            
         real(dp), intent(in) :: abar 
            ! mean atomic number (nucleons per nucleus; grams per mole)
         real(dp), intent(in) :: zbar 
            ! mean charge per nucleus
         
         integer, intent(in) :: species
         integer, pointer :: chem_id(:) ! maps species to chem id
            ! index from 1 to species
            ! value is between 1 and num_chem_isos         
         integer, pointer :: net_iso(:) ! maps chem id to species number
            ! index from 1 to num_chem_isos (defined in chem_def)
            ! value is 0 if the iso is not in the current net
            ! else is value between 1 and number of species in current net
         real(dp), intent(in) :: xa(:) ! mass fractions
         
         real(dp), intent(in) :: aE, alogE
            
         real(dp), intent(in) :: aRho, alogRho
            
         real(dp), intent(in) :: logT_guess

         
         ! OUTPUT    
              
         real(dp), intent(out) :: T, logT
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: &
            dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
            dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E
         integer, intent(out) :: ierr
         
         ! LOCALS
         
         real(dp), dimension(num_eos_basic_results) ::  &
            res1, d_dlnRho_c_T1, d_dlnT_c_Rho1, d_dabar_c_TRho1, d_dzbar_c_TRho1
         real(dp), dimension(num_eos_basic_results) ::  &
            res2, d_dlnRho_c_T2, d_dlnT_c_Rho2
         real(dp) :: X, Z, &
               dlnT_dlnE_c_Rho1, dlnT_dlnd_c_E1, &
               dlnPgas_dlnE_c_Rho1, dlnPgas_dlnd_c_E1, &
               dlnT_dlnE_c_Rho2, dlnT_dlnd_c_E2, &
               dlnPgas_dlnE_c_Rho2, dlnPgas_dlnd_c_E2
         integer :: iregion
         character (len=256) :: message
   
         integer, parameter :: pure_helm = 1
         integer, parameter :: pure_opal_scvh = 2
         integer, parameter :: blend_in_x = 3
         integer, parameter :: blend_in_y = 4
         integer, parameter :: blend_corner_out = 5
         
         real(dp), parameter :: tiny = 1d-20
         real(dp) :: Rho, logRho, energy, logE, logV, alfa, beta, &
               T1, logT1, T2, logT2
         
         logical, parameter :: dbg = .false.

         include 'formats.dek'
         
         ierr = 0
         
         if (.not. eosDE_is_initialized) then
            call Load_eosDE_Table(ierr)
            if (ierr /= 0) return
         end if
         
         if (is_bad_num(X_in) .or. is_bad_num(Z_in)) then
            ierr = -1
            if (dbg) write(*,*) 'error from is_bad_num'
            return
         end if
         
         X = X_in; Z = Z_in
         if (X < tiny) X = 0
         if (Z < tiny) Z = 0
         
         call get_DE_args( &
            aE, alogE, aRho, alogRho, energy, logE, Rho, logRho, ierr)
         if (ierr /= 0) then
            if (dbg) write(*,*) 'error from get_DE_args'
            return
         end if
         
         logV = logRho - 0.7d0*logE + 20d0
         
         call get_alfa_beta(ierr)
         if (ierr /= 0) then
            if (dbg) write(*,*) 'error from get_alfa_beta'
            return
         end if
         
         if (beta /= 0D0) then
            ! before calling OPAL_SCVH, check that Z isn't too large. 
            if (Z >= rq% Z_all_HELM .or. Z > eos_Zs(num_eosDE_Zs)) then
               beta = 0d0
            end if
            alfa = 1d0 - beta
         end if
         
         if (beta > 0.9999) then
            beta = 1; alfa = 0
         else if (alfa > 0.9999) then
            alfa = 1; beta = 0
         end if

         if (dbg) write(*,1) 'alfa', alfa
         if (dbg) write(*,1) 'beta', beta
               
         if (beta /= 0d0) then ! table lookup
            call Get_DE_OPAL_SCVH_Results( &
               rq, Z, X, abar, zbar, energy, logE, Rho, logRho, &
               T2, logT2, res2, d_dlnRho_c_T2, d_dlnT_c_Rho2, &
               dlnT_dlnE_c_Rho2, dlnT_dlnd_c_E2, &
               dlnPgas_dlnE_c_Rho2, dlnPgas_dlnd_c_E2, &
               ierr)
            if (ierr /= 0) then
               ierr = 0
               beta = 0
               alfa = 1
            end if
         end if
      
         if (alfa /= 0D0) then
            !write(*,1) 'Get_DE_Results_using_DT: alfa logV, logE, logRho, logT_guess', &
            !   alfa, logV, logE, logRho, logT_guess
            call Get_DE_Results_using_DT( &
               rq, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               energy, logE, Rho, logRho, logT_guess, &
               T1, logT1, res1, d_dlnRho_c_T1, d_dlnT_c_Rho1,  &
               d_dabar_c_TRho1, d_dzbar_c_TRho1, &
               dlnT_dlnE_c_Rho1, dlnT_dlnd_c_E1, &
               dlnPgas_dlnE_c_Rho1, dlnPgas_dlnd_c_E1, &
               ierr)
            if (ierr /= 0) then
               if (dbg) then
                  write(*,*) 'error from Get_DE_Results_using_DT'
                  stop
               end if
               return
            end if
         end if
      
         if (beta == 0d0) then
            res = res1
            T = T1
            logT = logT1
            dlnT_dlnE_c_Rho = dlnT_dlnE_c_Rho1
            dlnT_dlnd_c_E = dlnT_dlnd_c_E1
            dlnPgas_dlnE_c_Rho = dlnPgas_dlnE_c_Rho1
            dlnPgas_dlnd_c_E = dlnPgas_dlnd_c_E1
            d_dlnRho_c_T = d_dlnRho_c_T1
            d_dlnT_c_Rho = d_dlnT_c_Rho1
            d_dabar_c_TRho = d_dabar_c_TRho1
            d_dzbar_c_TRho = d_dzbar_c_TRho1
         else if (alfa == 0d0) then
            res = res2
            T = T2
            logT = logT2
            dlnT_dlnE_c_Rho = dlnT_dlnE_c_Rho2
            dlnT_dlnd_c_E = dlnT_dlnd_c_E2
            dlnPgas_dlnE_c_Rho = dlnPgas_dlnE_c_Rho2
            dlnPgas_dlnd_c_E = dlnPgas_dlnd_c_E2
            d_dlnRho_c_T = d_dlnRho_c_T2
            d_dlnT_c_Rho = d_dlnT_c_Rho2
            d_dabar_c_TRho = 0d0
            d_dzbar_c_TRho = 0d0
         else
            !alfa = (1 - cospi_cr(alfa))/2 ! smooth the transitions
            beta = 1 - alfa
            res = alfa*res1 + beta*res2
            logT = alfa*logT1 + beta*logT2
            T = exp10_cr(logT)            
            dlnT_dlnE_c_Rho = alfa*dlnT_dlnE_c_Rho1 + beta*dlnT_dlnE_c_Rho2
            dlnT_dlnd_c_E = alfa*dlnT_dlnd_c_E1 + beta*dlnT_dlnd_c_E2
            dlnPgas_dlnE_c_Rho = alfa*dlnPgas_dlnE_c_Rho1 + beta*dlnPgas_dlnE_c_Rho2
            dlnPgas_dlnd_c_E = alfa*dlnPgas_dlnd_c_E1 + beta*dlnPgas_dlnd_c_E2
            d_dlnRho_c_T = alfa*d_dlnRho_c_T1 + beta*d_dlnRho_c_T2
            d_dlnT_c_Rho = alfa*d_dlnT_c_Rho1 + beta*d_dlnT_c_Rho2
            d_dabar_c_TRho = alfa*d_dabar_c_TRho1
            d_dzbar_c_TRho = alfa*d_dzbar_c_TRho1
         end if


         contains
         
         subroutine get_alfa_beta(ierr)
            integer, intent(out) :: ierr
            
            real(dp) :: logV_margin, logE_margin, &
               logV1, logV2, logV3, logV4, logE1, logE2, logE3, logE4, c_dx, c_dy
            type (EosDE_XZ_Info), pointer :: ep
            
            include 'formats.dek'
            
            c_dx = 0
            c_dy = 0
            
            ep => eosDE_XZ_data(1,1) ! use ix=1, iz=1 to set alfa    
            logV_margin = 0.3d0
            logE_margin = 0.1d0
         
            logV1 = ep% logV_max
            logV2 = ep% logV_max - logV_margin
            logV3 = ep% logV_min + logV_margin
            logV4 = ep% logV_min
         
            logE1 = rq% logE_all_HELM
            logE2 = rq% logE_all_OPAL
            logE3 = ep% logE_min + logE_margin
            logE4 = ep% logE_min
            
            if (dbg) write(*,1) 'logV', logV
            if (dbg) write(*,1) 'logV1', logV1
            if (dbg) write(*,1) 'logV2', logV2
            if (dbg) write(*,1) 'logV3', logV3
            if (dbg) write(*,1) 'logV4', logV4
            if (dbg) write(*,1) 'logE', logE
            if (dbg) write(*,1) 'logE1', logE1
            if (dbg) write(*,1) 'logE2', logE2
            if (dbg) write(*,1) 'logE3', logE3
            if (dbg) write(*,1) 'logE4', logE4
         
            if (logV <= logV4 .or. logV >= logV1 &
               .or. logE <= logE4 .or. logE >= logE1) then
               iregion = pure_helm
               if (dbg) write(*,*) 'case 1'
               if (dbg) write(*,*) 'logV <= logV4', logV <= logV4, logV, logV4
               if (dbg) write(*,*) 'logV >= logV1', logV >= logV1
               if (dbg) write(*,*) 'logE <= logE4', logE <= logE4
               if (dbg) write(*,*) 'logE >= logE1', logE >= logE1
            else if (logE > logE2) then
               c_dy = (logE - logE2) / (logE1 - logE2)
               if (logV > logV2) then
                  c_dx = (logV - logV2) / (logV1 - logV2)
                  iregion = blend_corner_out
                  if (dbg) write(*,*) 'case 2'
               else if (logV > logV3) then
                  iregion = blend_in_y
                  if (dbg) write(*,*) 'case 3'
               else ! logV > logV4
                  c_dx = (logV - logV3) / (logV4 - logV3)
                  iregion = blend_corner_out
                  if (dbg) write(*,*) 'case 4'
               end if
            else if (logE > logE3) then
               if (logV > logV2) then
                  c_dx = (logV - logV2) / (logV1 - logV2)
                  iregion = blend_in_x
                  if (dbg) write(*,*) 'case 5'
               else if (logV > logV3) then
                  iregion = pure_opal_scvh
                  if (dbg) write(*,*) 'case 6'
               else ! logV > logV4
                  c_dx = (logV - logV3) / (logV4 - logV3)
                  iregion = blend_in_x
                  if (dbg) write(*,*) 'case 7'
               end if
            else ! logE > logE4
               c_dy = (logE - logE3) / (logE4 - logE3)
               if (logV > logV2) then
                  c_dx = (logV - logV2) / (logV1 - logV2)
                  iregion = blend_corner_out
                  if (dbg) write(*,*) 'case 8'
               else if (logV > logV3) then
                  iregion = blend_in_y
                  if (dbg) write(*,*) 'case 9'
               else ! logV > logV4
                  c_dx = (logV - logV3) / (logV4 - logV3)
                  iregion = blend_corner_out
                  if (dbg) write(*,*) 'case 10'
               end if
            end if
         
            if (iregion == pure_helm) then
               alfa = 1
               beta = 0
            else if (iregion == pure_opal_scvh) then
               alfa = 0
               beta = 1
            else if (iregion == blend_in_y) then
               alfa = c_dy
               beta = 1 - alfa
            else if (iregion == blend_in_x) then
               alfa = c_dx
               beta = 1 - alfa
            else if (iregion == blend_corner_out) then
               alfa = min(1d0, sqrt(c_dx*c_dx + c_dy*c_dy))
               beta = 1 - alfa
            !else if (iregion == blend_corner_in) then
            !   beta = min(1d0, sqrt(c_dx*c_dx + c_dy*c_dy))
            !   alfa = 1 - beta
            else
               ierr = -1
               return
            end if
         end subroutine get_alfa_beta
         
      end subroutine Get_eosDE_Results
      
      
      subroutine get_DE_args( &
            aE, alogE, aRho, alogRho, energy, logE, Rho, logRho, ierr)       
         real(dp), intent(in) :: aE, alogE, aRho, alogRho
         real(dp), intent(out) :: energy, logE, Rho, logRho
         integer, intent(out) :: ierr
         include 'formats.dek'
         ierr = 0
         energy = aE; logE = alogE
         if (aE == arg_not_provided .and. alogE == arg_not_provided) then
            ierr = -2; return
         end if
         if (alogE == arg_not_provided) logE = log10_cr(energy)
         if (aE == arg_not_provided) energy = exp10_cr(logE)
         if (energy <= 0) then
            ierr = -1
            return
         end if
         Rho = aRho; logRho = alogRho
         if (Rho == arg_not_provided .and. logRho == arg_not_provided) then
            ierr = -3; return
         end if
         if (logRho == arg_not_provided) logRho = log10_cr(Rho)
         if (Rho == arg_not_provided) Rho = exp10_cr(logRho)
         if (Rho <= 0) then
            ierr = -1
            return
         end if
      end subroutine get_DE_args

      
      subroutine Get_DE_OPAL_SCVH_Results(  &
            rq, Z, X, abar, zbar, energy, logE, Rho, logRho,  &
            T, logT, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
            dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
            dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
            ierr)
         use chem_def
         type (EoS_General_Info), pointer :: rq ! general information about the request
         real(dp), intent(in) :: Z, X, abar, zbar, energy, logE, Rho, logRho
         real(dp), intent(out) :: T, logT
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: &
            dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
            dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E
         integer, intent(out) :: ierr

         real(dp), dimension(num_eos_basic_results, num_eosDE_Zs) :: &
            res_zx, d_res_dX, d_dlnRho_c_T_zx, d_dlnT_c_Rho_zx
         real(dp), dimension(num_eos_basic_results) :: d_dX, d_dZ
         real(dp), dimension(3) :: c, logT_z, &
            dlnT_dlnE_c_Rho_z, dlnT_dlnd_c_E_z, &
            dlnPgas_dlnE_c_Rho_z, dlnPgas_dlnd_c_E_z
         real(dp) :: d_res_dZ(num_eos_basic_results), dZ, denom
         real(dp), parameter :: tiny = 1d-6

         character (len=256) :: message
         integer :: iz, j, ci
         logical, parameter :: OPAL_SCVH_dbg = .false.
         
         include 'formats.dek'

         ierr = 0
         
         if (num_eosDE_Zs /= 3) then
            write(*, *) 'error: Get_DE_OPAL_SCVH_Results assumes num_eosDE_Zs = 3'
            stop 1
         end if
         
         if (eos_Zs(1) /= 0) then
            write(*, *) 'error: Get_DE_OPAL_SCVH_Results assumes eos_Zs(1) == 0'
            stop 1
         end if
         
         if (abs(eos_Zs(1) - 2*eos_Zs(2) + eos_Zs(3)) > tiny) then
            write(*, *) 'error: Get_DE_OPAL_SCVH_Results assumes equal spaced Zs(1:3)'
            stop 1
         end if
         
         if (Z < 1d-20) then
            call Get_DE_OPAL_SCVH_for_X( &
               rq, 1, X, abar, zbar, energy, logE, Rho, logRho,  &
               T, logT, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
               dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
               dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
               ierr)
            return
         end if
         
         if (Z > eos_Zs(3)) then ! Z too large for DE tables
            ierr = -1
            return
         end if
         
         do iz = 1, 3
            call Get_DE_OPAL_SCVH_for_X(  &
               rq, iz, X, abar, zbar, energy, logE, Rho, logRho, &
               T, logT_z(iz), res_zx(:, iz), &
               d_dlnRho_c_T_zx(:, iz), d_dlnT_c_Rho_zx(:, iz), &
               dlnT_dlnE_c_Rho_z(iz), dlnT_dlnd_c_E_z(iz), &
               dlnPgas_dlnE_c_Rho_z(iz), dlnPgas_dlnd_c_E_z(iz), &
               ierr)
            if (ierr /= 0) return
         end do

         dZ = eos_Zs(2) - eos_Zs(1)
         denom = 2*dZ*dZ
         c(1) = (2*dZ*dZ - 3*dZ*Z + Z*Z)/denom
         c(2) = 2*(2*dZ-Z)*Z/denom
         c(3) = Z*(Z-dZ)/denom
         
         res(:) = c(1)*res_zx(:, 1) + c(2)*res_zx(:, 2) + c(3)*res_zx(:, 3)
         
         logT = dot_product(c(:), logT_z(:))
         T = exp10_cr(logT)

         dlnT_dlnE_c_Rho = dot_product(c(:), dlnT_dlnE_c_Rho_z(:))
         dlnT_dlnd_c_E = dot_product(c(:), dlnT_dlnd_c_E_z(:))
         dlnPgas_dlnE_c_Rho = dot_product(c(:), dlnPgas_dlnE_c_Rho_z(:))
         dlnPgas_dlnd_c_E = dot_product(c(:), dlnPgas_dlnd_c_E_z(:))
         
         d_dlnRho_c_T(:) = &
           c(1)*d_dlnRho_c_T_zx(:,1) + &
           c(2)*d_dlnRho_c_T_zx(:,2) + &
           c(3)*d_dlnRho_c_T_zx(:,3)
     
         d_dlnT_c_Rho(:) =  &
           c(1)*d_dlnT_c_Rho_zx(:,1) + &
           c(2)*d_dlnT_c_Rho_zx(:,2) + &
           c(3)*d_dlnT_c_Rho_zx(:,3)
         
      end subroutine Get_DE_OPAL_SCVH_Results

      
      subroutine Get_DE_OPAL_SCVH_for_X(  &
               rq, iz, X, abar, zbar, energy, logE, Rho, logRho, &
               T, logT, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
               dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
               dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
               ierr)
         type (EoS_General_Info), pointer :: rq ! general information about the request
         integer, intent(in) :: iz ! the index in eos_Zs
         real(dp), intent(in) :: X, abar, zbar, energy, logE, Rho, logRho
         real(dp), intent(out) :: T, logT
         real(dp), dimension(:,:), pointer :: d_dx ! (num_eos_basic_results, species)
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: &
            dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
            dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E
         integer, intent(out) :: ierr

         real(dp), dimension(num_eos_basic_results, 4) :: &
            res_zx, d_dlnRho_c_T_zx, d_dlnT_c_Rho_zx
         real(dp), dimension(4) :: c, logT_zx, &
            dlnT_dlnE_c_Rho_zx, dlnT_dlnd_c_E_zx, &
            dlnPgas_dlnE_c_Rho_zx, dlnPgas_dlnd_c_E_zx
         real(dp) :: dX, denom, delX, coef, dlnP_dX, dmu_dX, dlnmu_dX
         character (len=256) :: message
         integer :: ix, ix_lo, ix_hi, j, num_Xs
         real(dp), parameter :: tiny = 1d-6
         logical, parameter :: dbg_for_X = .false.
         
         include 'formats.dek'

         ierr = 0
         
         if (num_eos_Xs /= 6) then
            write(*, *) 'error: Get_DE_OPAL_SCVH_for_X assumes num_eos_Xs == 6'
            stop 1
         end if
         
         if (eos_Xs(1) /= 0) then
            write(*, *) 'error: Get_DE_OPAL_SCVH_for_X assumes eos_Xs(1) == 0'
            stop 1
         end if
         
         num_Xs = num_eos_Xs_for_Z(iz)
         
         if (X < tiny .or. num_Xs == 1) then
            call Get_eosDE_XTable_Results(rq, 1, iz, energy, logE, Rho, logRho, &
               T, logT, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
               dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
               dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
               ierr)
            if (dbg_for_X) write(*,1) 'dbg_for_X logT', logT
            return
         end if
         
         dX = eos_Xs(2)-eos_Xs(1)
         
         do ix = 3, num_Xs
            if (abs(dX - (eos_Xs(ix) - eos_Xs(ix-1))) > tiny) then
               write(*, *) 'error: Get_DE_OPAL_SCVH_for_X assumes equal spaced Xs'
               stop 1
            end if
         end do
         
         ix_hi = -1
         if (X <= eos_Xs(2)) then
            ix_lo = 1; ix_hi = 3
         else if (X >= eos_Xs(num_Xs-1)) then
            ix_lo = num_Xs-2; ix_hi = num_Xs
         else
            do ix = 3, num_Xs-1
               if (X <= eos_Xs(ix)) then
                  ix_lo = ix-2; ix_hi = ix+1; exit
               end if
            end do
         end if
         
         if (ix_hi < 0) then
            write(*, *) 'error: Get_DE_OPAL_SCVH_for_X logic bug'
            stop 1
         end if
         
         if (dbg_for_X) then
            write(*, *) 'X', X
            write(*, *) 'ix_lo', ix_lo
            write(*, *) 'ix_hi', ix_hi
         end if
         
         do ix=ix_lo, ix_hi
            j = ix-ix_lo+1
            call Get_eosDE_XTable_Results(rq, ix, iz, energy, logE, Rho, logRho, &
               T, logT_zx(j), res_zx(:, j), &
               d_dlnRho_c_T_zx(:, j), d_dlnT_c_Rho_zx(:, j), &
               dlnT_dlnE_c_Rho_zx(j), dlnT_dlnd_c_E_zx(j), &
               dlnPgas_dlnE_c_Rho_zx(j), dlnPgas_dlnd_c_E_zx(j), &
               ierr)
            if (ierr /= 0) return
         end do
         
         delX = X - eos_Xs(ix_lo)
         if (ix_hi-ix_lo==2) then
         
            denom = 2*dX*dX
            c(1) = (2*dX*dX - 3*dX*delX + delX*delX)/denom
            c(2) = 2*(2*dX-delX)*delX/denom
            c(3) = delX*(delX-dX)/denom
            res(:) = c(1)*res_zx(:, 1) + c(2)*res_zx(:, 2) + c(3)*res_zx(:, 3)
            
            logT = dot_product(c(1:3), logT_zx(1:3))
            T = exp10_cr(logT)

            dlnT_dlnE_c_Rho = dot_product(c(1:3), dlnT_dlnE_c_Rho_zx(1:3))
            dlnT_dlnd_c_E = dot_product(c(1:3), dlnT_dlnd_c_E_zx(1:3))
            dlnPgas_dlnE_c_Rho = dot_product(c(1:3),dlnPgas_dlnE_c_Rho_zx(1:3))
            dlnPgas_dlnd_c_E = dot_product(c(1:3), dlnPgas_dlnd_c_E_zx(1:3))
            
            d_dlnRho_c_T(:) = &
               c(1)*d_dlnRho_c_T_zx(:, 1) + &
               c(2)*d_dlnRho_c_T_zx(:, 2) + &
               c(3)*d_dlnRho_c_T_zx(:, 3)
            d_dlnT_c_Rho(:) =  &
               c(1)*d_dlnT_c_Rho_zx(:, 1) + &
               c(2)*d_dlnT_c_Rho_zx(:, 2) + &
               c(3)*d_dlnT_c_Rho_zx(:, 3)
     
            c(1) = (2*delX-3*dX)/denom
            c(2) = 4*(dX-delX)/denom
            c(3) = (2*delX-dX)/denom
            
            dlnP_dX = c(1)*res_zx(i_lnPgas, 1) &
                  + (c(2)*res_zx(i_lnPgas, 2) + c(3)*res_zx(i_lnPgas, 3))
            dmu_dX = c(1)*res_zx(i_mu, 1) &
                  + (c(2)*res_zx(i_mu, 2) + c(3)*res_zx(i_mu, 3))
            
         else
         
            coef = (X-eos_Xs(ix_lo+1))/dX 
            ! coef = fractional location of X between 2nd and 3rd X's for fit.
            ! coef is weight for the quadratic based on points 2, 3, 4 of fit.
            ! (1-coef) is weight for quadratic based on points 1, 2, 3 of fit.
            if (coef < 0 .or. coef > 1) then
               ierr = -1
               return
            end if
            
            c(1) = -coef*(coef-1)*(coef-1)/2
            c(2) = (2 - coef*coef*(5 - 3*coef))/2
            c(3) = coef*(1 + coef*(4 - 3*coef))/2
            c(4) = coef*coef*(coef-1)/2
            res(:) = c(1)*res_zx(:, 1) + &
                        (c(2)*res_zx(:, 2) + &
                           (c(3)*res_zx(:, 3) + &
                              c(4)*res_zx(:, 4)))
     
            logT = dot_product(c(1:4), logT_zx(1:4))
            T = exp10_cr(logT)

            dlnT_dlnE_c_Rho = dot_product(c(1:4), dlnT_dlnE_c_Rho_zx(1:4))
            dlnT_dlnd_c_E = dot_product(c(1:4), dlnT_dlnd_c_E_zx(1:4))
            dlnPgas_dlnE_c_Rho = dot_product(c(1:4),dlnPgas_dlnE_c_Rho_zx(1:4))
            dlnPgas_dlnd_c_E = dot_product(c(1:4), dlnPgas_dlnd_c_E_zx(1:4))
            
            d_dlnRho_c_T(:) =  &
               c(1)*d_dlnRho_c_T_zx(:, 1) +  &
                  (c(2)*d_dlnRho_c_T_zx(:, 2) +  &
                     (c(3)*d_dlnRho_c_T_zx(:, 3) +  &
                           c(4)*d_dlnRho_c_T_zx(:, 4)))
            d_dlnT_c_Rho(:) = &
               c(1)*d_dlnT_c_Rho_zx(:, 1) +  &
                  (c(2)*d_dlnT_c_Rho_zx(:, 2) +  &
                     (c(3)*d_dlnT_c_Rho_zx(:, 3) +  &
                           c(4)*d_dlnT_c_Rho_zx(:, 4)))
     
            denom = 2*dX
            c(1) = (-1 + coef*(3 - 2*coef))/denom
            c(2) = coef*(-7 + 6*coef)/denom
            c(3) = (1 + coef*(5 - 6*coef))/denom
            c(4) = coef*(-1 + 2*coef)/denom
            
            dlnP_dX = &
               c(1)*res_zx(i_lnPgas, 1) + &
                  (c(2)*res_zx(i_lnPgas, 2) + &
                     (c(3)*res_zx(i_lnPgas, 3) + &
                        c(4)*res_zx(i_lnPgas, 4)))
            dmu_dX = &
               c(1)*res_zx(i_mu, 1) + &
                  (c(2)*res_zx(i_mu, 2) + &
                     (c(3)*res_zx(i_mu, 3) + &
                        c(4)*res_zx(i_mu, 4)))
     
         end if
         
      end subroutine Get_DE_OPAL_SCVH_for_X
         
      
      subroutine Get_eosDE_XTable_Results( &
               rq, ix, iz, energy, logE_in, Rho, logRho_in, &
               T, logT, res, d_dlnRho_c_T, d_dlnT_c_Rho, &
               dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
               dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
               ierr)
         use eosDT_eval, only: Do_EoS_Interpolations
         type (EoS_General_Info), pointer :: rq
         integer, intent(in) :: ix, iz
         real(dp), intent(in) :: energy, logE_in, Rho, logRho_in
         real(dp), intent(out) :: T, logT
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results)
            ! only does partials wrt lnRho and lnT
         real(dp), intent(out) :: &
            dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
            dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E
         integer, intent(out) :: ierr
         
         integer, parameter :: nv = num_eos_vals
         real, dimension(nv) :: f, df_dlogV_c_E, df_dlogE_c_V, &
            df_dlogRho_c_T, df_dlogT_c_Rho
         real :: logV0, logV1, logE0, logE1, &
            logE, logRho, logV, &
            dlogV_dlnE_c_Rho, dlogV_dlnd_c_E, &
            dlogE_dlogT_c_Rho, dlogE_dlogRho_c_T, &
            dlogV_dlogRho_c_T, dlogV_dlogT_c_Rho 
         integer :: iV, jE, k
         type (EosDE_XZ_Info), pointer :: ep
         
         character (len=256) :: message
         
         include 'formats.dek'

         logE = logE_in
         logRho = logRho_in
         logV = logRho - 0.7*logE + 20

         ierr = 0
         ep => eosDE_XZ_data(ix,iz)     
         
         call Locate_logV(rq, ep, logV, iV, logV0, logV1, ierr)
         if (ierr /= 0) return
         
         call Locate_logE(rq, ep, logE, jE, logE0, logE1, ierr)
         if (ierr /= 0) return

         call Do_EoS_Interpolations(  &
            nv, ep% num_logVs, ep% logVs, ep% num_logEs, ep% logEs,  &
            ep% tbl1, iV, jE, logV0, logV, logV1, logE0, logE, logE1, &
            f, df_dlogV_c_E, df_dlogE_c_V, ierr)
         if (ierr /= 0) return
            
         logT = f(eosDE_ilnT)/ln10
         T = exp10_cr(logT)
         
         dlogE_dlogT_c_Rho = f(eos_iCv)*T/energy
         dlogE_dlogRho_c_T = f(eos_idE_dRho)*Rho/energy
         
         ! logV = logRho - 0.7*logE + 20
         dlogV_dlogRho_c_T = 1.0 - 0.7*dlogE_dlogRho_c_T
         dlogV_dlogT_c_Rho = -0.7*dlogE_dlogT_c_Rho
         
         dlogV_dlnE_c_Rho = -0.7d0/ln10
         dlogV_dlnd_c_E = 1d0/ln10
         
         dlnT_dlnd_c_E = df_dlogV_c_E(eosDE_ilnT)*dlogV_dlnd_c_E
         dlnPgas_dlnd_c_E = df_dlogV_c_E(eosDE_ilnPgas)*dlogV_dlnd_c_E

         dlnT_dlnE_c_Rho = &
            df_dlogE_c_V(eosDE_ilnT)/ln10 + &
            df_dlogV_c_E(eosDE_ilnT)*dlogV_dlnE_c_Rho
         dlnPgas_dlnE_c_Rho = &
            df_dlogE_c_V(eosDE_ilnPgas)/ln10 + &
            df_dlogV_c_E(eosDE_ilnPgas)*dlogV_dlnE_c_Rho
         
         ! now get partials of f = f(logV(logRho,logT),logE(logRho,logT))
         do k=1,nv
            df_dlogT_c_Rho(k) = &
               df_dlogV_c_E(k)*dlogV_dlogT_c_Rho + df_dlogE_c_V(k)*dlogE_dlogT_c_Rho
            df_dlogRho_c_T(k) = &
               df_dlogV_c_E(k)*dlogV_dlogRho_c_T + df_dlogE_c_V(k)*dlogE_dlogRho_c_T
         end do

         res(i_lnPgas) = f(eosDE_ilnPgas)
         res(i_lnE) = logE*ln10
         res(i_lnS) = f(eos_ilnS)
         res(i_grad_ad) = f(eos_igrad_ad)
         res(i_chiRho) = f(eos_ichiRho)
         res(i_chiT) = f(eos_ichiT)
         res(i_Cp) = f(eos_iCp)
         res(i_Cv) = f(eos_iCv)
         res(i_dE_dRho) = f(eos_idE_dRho)
         res(i_dS_dT) = f(eos_idS_dT)
         res(i_dS_dRho) = f(eos_idS_dRho)
         res(i_mu) = f(eos_imu)
         res(i_lnfree_e) = f(eos_ilnfree_e)
         res(i_gamma1) = f(eos_igamma1)
         res(i_gamma3) = f(eos_igamma3)
         res(i_eta) = f(eos_ieta)
         
         d_dlnRho_c_T(i_lnPgas) = df_dlogRho_c_T(eosDE_ilnPgas)/ln10
         d_dlnRho_c_T(i_lnE) = dlogE_dlogRho_c_T
         d_dlnRho_c_T(i_lnS) = df_dlogRho_c_T(eos_ilnS)/ln10
         d_dlnRho_c_T(i_grad_ad) = df_dlogRho_c_T(eos_igrad_ad)/ln10
         d_dlnRho_c_T(i_chiRho) = df_dlogRho_c_T(eos_ichiRho)/ln10
         d_dlnRho_c_T(i_chiT) = df_dlogRho_c_T(eos_ichiT)/ln10
         d_dlnRho_c_T(i_Cp) = df_dlogRho_c_T(eos_iCp)/ln10
         d_dlnRho_c_T(i_Cv) = df_dlogRho_c_T(eos_iCv)/ln10
         d_dlnRho_c_T(i_dE_dRho) = df_dlogRho_c_T(eos_idE_dRho)/ln10
         d_dlnRho_c_T(i_dS_dT) = df_dlogRho_c_T(eos_idS_dT)/ln10
         d_dlnRho_c_T(i_dS_dRho) = df_dlogRho_c_T(eos_idS_dRho)/ln10
         d_dlnRho_c_T(i_mu) = df_dlogRho_c_T(eos_imu)/ln10
         d_dlnRho_c_T(i_lnfree_e) = df_dlogRho_c_T(eos_ilnfree_e)/ln10
         d_dlnRho_c_T(i_gamma1) = df_dlogRho_c_T(eos_igamma1)/ln10
         d_dlnRho_c_T(i_gamma3) = df_dlogRho_c_T(eos_igamma3)/ln10
         d_dlnRho_c_T(i_eta) = df_dlogRho_c_T(eos_ieta)/ln10
      
         d_dlnT_c_Rho(i_lnPgas) = df_dlogT_c_Rho(eosDE_ilnPgas)/ln10
         d_dlnT_c_Rho(i_lnE) = dlogE_dlogT_c_Rho
         d_dlnT_c_Rho(i_lnS) = df_dlogT_c_Rho(eos_ilnS)/ln10
         d_dlnT_c_Rho(i_grad_ad) = df_dlogT_c_Rho(eos_igrad_ad)/ln10
         d_dlnT_c_Rho(i_chiRho) = df_dlogT_c_Rho(eos_ichiRho)/ln10
         d_dlnT_c_Rho(i_chiT) = df_dlogT_c_Rho(eos_ichiT)/ln10
         d_dlnT_c_Rho(i_Cp) = df_dlogT_c_Rho(eos_iCp)/ln10
         d_dlnT_c_Rho(i_Cv) = df_dlogT_c_Rho(eos_iCv)/ln10
         d_dlnT_c_Rho(i_dE_dRho) = df_dlogT_c_Rho(eos_idE_dRho)/ln10
         d_dlnT_c_Rho(i_dS_dT) = df_dlogT_c_Rho(eos_idS_dT)/ln10
         d_dlnT_c_Rho(i_dS_dRho) = df_dlogT_c_Rho(eos_idS_dRho)/ln10
         d_dlnT_c_Rho(i_mu) = df_dlogT_c_Rho(eos_imu)/ln10
         d_dlnT_c_Rho(i_lnfree_e) = df_dlogT_c_Rho(eos_ilnfree_e)/ln10
         d_dlnT_c_Rho(i_gamma1) = df_dlogT_c_Rho(eos_igamma1)/ln10
         d_dlnT_c_Rho(i_gamma3) = df_dlogT_c_Rho(eos_igamma3)/ln10
         d_dlnT_c_Rho(i_eta) = df_dlogT_c_Rho(eos_ieta)/ln10

      end subroutine Get_eosDE_XTable_Results
      
      
      subroutine Locate_logV(rq, ep, logV, iW, logV0, logV1, ierr)
         type (EoS_General_Info), pointer :: rq
         type (EosDE_XZ_Info), pointer :: ep
         real, intent(inout) :: logV
         integer, intent(out) :: iW
         real, intent(out) :: logV0, logV1
         integer, intent(out) :: ierr
         
         logical, parameter :: dbg = .false.

         include 'formats.dek'
      
         ierr = 0
         iW = int((logV - ep% logV_min) / ep% del_logV + 1d-4) + 1
         
         if (iW < 1 .or. iW >= ep% num_logVs) then
            
            if (iW < 1) then
               iW = 1
               logV0 = ep% logV_min
               logV1 = logV0 + ep% del_logV
               logV = logV0
            else
               iW = ep% num_logVs-1
               logV0 = ep% logV_min + (iW-1) * ep% del_logV
               logV1 = logV0 + ep% del_logV
               logV = logV1
            end if
            
         else
         
            logV0 = ep% logV_min + (iW-1) * ep% del_logV
            logV1 = logV0 + ep% del_logV

         end if
         
         if (dbg) then
            write(*,*) 'Locate_logV iW', iW
            write(*,1) 'logV0', logV0
            write(*,1) 'logV', logV
            write(*,1) 'logV1', logV1
            write(*,1) 'ep% logV_min', ep% logV_min
            write(*,1) 'ep% del_logV', ep% del_logV
            write(*,*)
         end if

      end subroutine Locate_logV
      
      
      subroutine Locate_logE(rq, ep, logE, iE, logE0, logE1, ierr)
         type (EoS_General_Info), pointer :: rq
         type (EosDE_XZ_Info), pointer :: ep
         real, intent(inout) :: logE
         integer, intent(out) :: iE
         real, intent(out) :: logE0, logE1
         integer, intent(out) :: ierr
         
         logical, parameter :: dbg = .false.

         include 'formats.dek'
      
         ierr = 0
         iE = int((logE - ep% logE_min) / ep% del_logE + 1d-4) + 1
         
         if (iE < 1 .or. iE >= ep% num_logEs) then
            
            if (iE < 1) then
               iE = 1
               logE0 = ep% logE_min
               logE1 = logE0 + ep% del_logE
               logE = logE0
            else
               iE = ep% num_logEs-1
               logE0 = ep% logE_min + (iE-1) * ep% del_logE
               logE1 = logE0 + ep% del_logE
               logE = logE1
            end if
            
         else
         
            logE0 = ep% logE_min + (iE-1) * ep% del_logE
            logE1 = logE0 + ep% del_logE

         end if
         
         if (dbg) then
            write(*,*) 'Locate_logE iE', iE
            write(*,1) 'logE0', logE0
            write(*,1) 'logE', logE
            write(*,1) 'logE1', logE1
            write(*,1) 'ep% logE_min', ep% logE_min
            write(*,1) 'ep% del_logE', ep% del_logE
            write(*,*)
         end if

      end subroutine Locate_logE


      subroutine Get_DE_Results_using_DT( &
               rq, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               energy, logE, Rho, logRho, logT_guess, &
               T, logT, res, d_dlnRho_c_T, d_dlnT_c_Rho,  &
               d_dabar_c_TRho, d_dzbar_c_TRho, &
               dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
               dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E, &
               ierr)
         use eosDT_eval, only: get_T
         use utils_lib, only: is_bad_num
         
         type (EoS_General_Info), pointer :: rq ! general information about the request
         real(dp), intent(in) :: Z, X, abar, zbar
         integer, intent(in) :: species
         integer, pointer :: chem_id(:)    
         integer, pointer :: net_iso(:)
         real(dp), intent(in) :: xa(:)
         real(dp), intent(in) :: energy, logE, Rho, logRho, logT_guess
         real(dp), intent(out) :: T, logT
         real(dp), intent(out) :: res(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnRho_c_T(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dlnT_c_Rho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dabar_c_TRho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: d_dzbar_c_TRho(:) ! (num_eos_basic_results)
         real(dp), intent(out) :: &
            dlnT_dlnE_c_Rho, dlnT_dlnd_c_E, &
            dlnPgas_dlnE_c_Rho, dlnPgas_dlnd_c_E
         integer, intent(out) :: ierr
         
         logical, parameter :: basic_flag = .false.
         integer:: i, eos_calls, max_iter, which_other
         real(dp) ::  &
            other, other_tol, logT_tol, &
            logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, logT_result
         
         logical, parameter :: dbg = .false.
         
         include 'formats.dek'
         
         ierr = 0

         which_other = i_lnE
         other = logE*ln10
         other_tol = 1d-8
         logT_tol = 1d-8
               
         logT_bnd1 = arg_not_provided
         logT_bnd2 = arg_not_provided
         other_at_bnd1 = arg_not_provided
         other_at_bnd2 = arg_not_provided

         max_iter = 20
         eos_calls = 0
         
         if (dbg) write(*,1) 'logRho', logRho
         if (dbg) write(*,1) 'logE', logE
         if (dbg) write(*,1) 'logT_guess', logT_guess
         if (dbg) write(*,1) 'T_guess', exp10_cr(logT_guess)
         if (dbg) write(*,1) 'T_guess_gas', 2*energy*abar*mp/(3*kerg*(1+zbar)) ! ideal gas
         if (dbg) write(*,1) 'T_guess_rad', pow_cr(energy/crad,0.25d0)
         
         call get_T( &
               rq% handle, Z, X, abar, zbar,  &
               species, chem_id, net_iso, xa, &
               logRho, which_other, other, &
               logT_tol, other_tol, max_iter, logT_guess,  &
               logT_bnd1, logT_bnd2, other_at_bnd1, other_at_bnd2, &
               logT_result, res, d_dlnRho_c_T, d_dlnT_c_Rho,  &
               d_dabar_c_TRho, d_dzbar_c_TRho, eos_calls, ierr)
         if (ierr /= 0 .or. (dbg .and. abs(logT_result) > 20)) then
            if (dbg) then
               write(*,*) 'failed in get_T for Get_DE_Results_using_DT'
               write(*,1) 'Z = ', Z
               write(*,1) 'X = ', X
               write(*,1) 'abar = ', abar
               write(*,1) 'zbar = ', zbar
               write(*,1) 'logRho = ', logRho
               write(*,1) 'logE = ', logE
               write(*,1) 'logT_guess = ', logT_guess
               write(*,*)
               stop 'Get_DE_Results_using_DT'
            end if
            return
         end if
         
         logT = logT_result
         T = exp10_cr(logT)

         dlnT_dlnd_c_E = -rho*res(i_dE_dRho)/(T*res(i_Cv))
         dlnPgas_dlnd_c_E = &
            d_dlnRho_c_T(i_lnPgas) + d_dlnT_c_Rho(i_lnPgas)*dlnT_dlnd_c_E

         dlnT_dlnE_c_Rho = energy/(T*res(i_Cv))
         dlnPgas_dlnE_c_Rho = d_dlnT_c_Rho(i_lnPgas)*dlnT_dlnE_c_Rho

         if (dbg) write(*,1) 'logT', logT
         if (dbg) write(*,1) 'T', T

      end subroutine Get_DE_Results_using_DT


      end module eosDE_eval
      